using System;
using System.Collections.Generic;
using System.Text;

namespace Zensys.Framework.Console
{
    /// <summary>
    /// Implements command line parsing
    /// </summary>
    public abstract class CommandLine
    {
        /// <summary>
        /// Parses the passed command line arguments and returns the result
        /// in a CommandArgs object.
        /// </summary>
        /// Arguments are defined as key/value pairs. The argument key must begin
        /// with a '-', or '/'.  Between the argument and the value must be at
        /// least one space or a single '='.  Extra spaces are ignored.  Arguments MAY
        /// be followed by a value or, if no value supplied, the string 'true' is used.
        /// You must enclose argument values in quotes if they contain a space, otherwise
        /// they will not parse correctly.
        /// 
        /// <param name="args">array of command line arguments</param>
        /// <returns>CommandArgs object containing the parsed command line</returns>
        public CommandArgs Parse(string[] arguments)
        {
            char[] kEqual = new char[] { '=' };
            char[] kArgStart = new char[] { '-', '/' };

            CommandArgs ca = new CommandArgs();
            int ii = -1;
            string token = NextToken(arguments, ref ii);
            while (token != null)
            {
                if (IsArg(token))
                {
                    string arg = token.TrimStart(kArgStart).TrimEnd(kEqual);

                    string value = null;

                    if (arg.Contains("="))
                    {
                        string[] r = arg.Split(kEqual, 2);
                        if (r.Length == 2 && r[1] != string.Empty)
                        {
                            arg = r[0];
                            value = r[1];
                        }
                    }

                    while (value == null)
                    {
                        string next = NextToken(arguments, ref ii);
                        if (next != null)
                        {
                            if (IsArg(next))
                            {
                                ii--;
                                value = "true";
                            }
                            else if (next != "=")
                            {
                                value = next.TrimStart(kEqual);
                            }
                        }
                        else
                            break;
                    }
                    ca.ArgPairs.Add(arg, value);
                }
                else if (token != string.Empty)
                {
                    ca.Params.Add(token);
                }

                token = NextToken(arguments, ref ii);
            }
            return ca;
        }

        /// <summary>
        /// Validates the arguments.
        /// </summary>
        /// <param name="CommandArguments">The command arguments.</param>
        /// <returns></returns>
        public abstract bool ValidateArguments(CommandArgs CommandArguments);

        /// <summary>
        /// Returns True if the passed string is an argument (starts with '-', or '/'.)
        /// </summary>
        /// <param name="arg">the string token to test</param>
        /// <returns>true if the passed string is an argument, else false if a parameter</returns>
        private bool IsArg(string arg)
        {
            return (arg.StartsWith("-") || arg.StartsWith("/"));
        }

        /// <summary>
        /// Returns the next string token in the argument list
        /// </summary>
        /// <param name="args">list of string tokens</param>
        /// <param name="ii">index of the current token in the array</param>
        /// <returns>the next string token, or null if no more tokens in array</returns>
        private string NextToken(string[] args, ref int ii)
        {
            ii++;
            while (ii < args.Length)
            {
                string cur = args[ii].Trim();
                if (cur != string.Empty)
                {
                    return cur;
                }
                ii++;
                if (ii > 100)
                    break;
            }
            return null;
        }

        protected abstract  List<string> GetUsageMessage();

        public void ShowUsage()
        {
            List<string> sb = GetUsageMessage();
            if (sb != null)
                foreach (string str in sb)
                {
                    System.Console.WriteLine(str);
                }
        }
    }
}
