using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Devices;
using Zensys.ZWave.Enums;
using Zensys.ZWave.Application;

namespace Zensys.ZWave.ZWaveHAL.Actions
{
    public class CommandClassViewActions : ActionsHAL
    {
        public delegate void AddQueueCommandDelegate(byte nodeId, QueueAction action);
        private AddQueueCommandDelegate mAddQueueCommandCallback;
        public AddQueueCommandDelegate AddQueueCommandCallback
        {
            get { return mAddQueueCommandCallback; }
            set { mAddQueueCommandCallback = value; }
        }
        public CommandClassViewActions(BaseEntryPointHAL entryPoint)
            : base(entryPoint)
        {
        }

        public void SendCommand(IController controller, IDeviceInfo device, CommandClass cmdClass, Command cmd, List<ParamValue> parameters, string msgSendCommand)
        {
            QueueAction action = null;
            try
            {
                byte[] data = mXmlDataManager.FillParameters(parameters);
                if (device.IsListening || device.IsCommandQueueOverrided || (device.Security & 0x20) != 0 || (device.Security & 0x40) != 0)
                {
                    SendCommand(cmdClass, cmd, device.Id, controller, new List<byte>(data));
                }
                else
                {
                    action = new QueueAction(
                         new SendCommandDelegate(SendCommand),
                         new object[] { cmdClass, cmd, device.Id, controller, new List<byte>(data) },
                         msgSendCommand, false, (byte)CommandTypes.CmdZWaveSendData, "Send Command: " + cmd.Text, 1000);
                    if (mAddQueueCommandCallback != null)
                    {
                        mAddQueueCommandCallback.DynamicInvoke(new object[] { device.Id, action });
                        mLogManager.LogMessage(string.Format("{0} added to command queue for Node {1}.", cmd.Name, device.Id));
                    }
                }
            }
            catch (Exception ex)
            {
                mExceptionManager.RegisterException(ex);
            }
        }
        private delegate void SendCommandDelegate(CommandClass cmdClass, Command cmd, byte deviceId, IController controller, List<byte> data);
        public virtual void SendCommand(CommandClass cmdClass, Command cmd, byte deviceId, IController controller, List<byte> data)
        {
            TransmitStatuses ret = TransmitStatuses.ResMissing;
            if (data.Count > 0)
            {
                List<byte> tmp = new List<byte>(data);
                tmp.Insert(0, cmd.KeyId);
                tmp.Insert(0, cmdClass.KeyId);
                ret = controller.SendData(deviceId, tmp.ToArray(), TransmitOptions.TransmitOptionAutoRoute | TransmitOptions.TransmitOptionAcknowledge | TransmitOptions.TransmitOptionExplore);
            }
            else
            {
                ret = controller.SendData(deviceId, new byte[] { cmdClass.KeyId, cmd.KeyId }, TransmitOptions.TransmitOptionAutoRoute | TransmitOptions.TransmitOptionAcknowledge | TransmitOptions.TransmitOptionExplore);
            }
            if (ret != TransmitStatuses.CompleteOk)
            {
                mLogManager.LogOperationFailure(string.Format("{0} failed. {1} received", cmd.Name, ret));
            }
            else
            {
                mLogManager.LogMessage(string.Format("{0} succeeded. {1} received", cmd.Name, ret));
            }
        }

        public void SendMetaCommand(IController controller, IDeviceInfo device, CommandClass cmdClass, Command cmd, List<ParamValue> parameters, string msgSendCommand)
        {
            QueueAction action = null;
            try
            {
                byte[] data = mXmlDataManager.FillParameters(parameters);
                if (device.IsListening || device.IsCommandQueueOverrided || (device.Security & 0x20) != 0 || (device.Security & 0x40) != 0)
                {
                    SendMetaCommand(cmdClass, cmd, device.Id, controller, new List<byte>(data));
                }
                else
                {
                    action = new QueueAction(
                         new SendMetaCommandDelegate(SendMetaCommand),
                         new object[] { cmdClass, cmd, device.Id, controller, new List<byte>(data) },
                         msgSendCommand, false, (byte)CommandTypes.CmdZWaveSendDataMeta, "Send Meta Command: " + cmd.Text, 1000);
                    if (mAddQueueCommandCallback != null)
                    {
                        mAddQueueCommandCallback.DynamicInvoke(new object[] { device.Id, action });
                        mLogManager.LogMessage(string.Format("{0} added to command queue for Node {1}.", cmd.Name, device.Id));
                    }
                }
            }
            catch (Exception ex)
            {
                mExceptionManager.RegisterException(ex);
            }
        }
        private delegate void SendMetaCommandDelegate(CommandClass cmdClass, Command cmd, byte deviceId, IController controller, List<byte> data);
        public virtual void SendMetaCommand(CommandClass cmdClass, Command cmd, byte deviceId, IController controller, List<byte> data)
        {
            TransmitStatuses ret = TransmitStatuses.ResMissing;
            if (data.Count > 0)
            {
                List<byte> tmp = new List<byte>(data);
                tmp.Insert(0, cmd.KeyId);
                tmp.Insert(0, cmdClass.KeyId);
                ret = controller.SendDataMeta(deviceId, tmp.ToArray(), TransmitOptions.TransmitOptionAutoRoute | TransmitOptions.TransmitOptionAcknowledge | TransmitOptions.TransmitOptionExplore);
            }
            else
            {
                ret = controller.SendDataMeta(deviceId, new byte[] { cmdClass.KeyId, cmd.KeyId }, TransmitOptions.TransmitOptionAutoRoute | TransmitOptions.TransmitOptionAcknowledge | TransmitOptions.TransmitOptionExplore);
            }
            if (ret != TransmitStatuses.CompleteOk)
            {
                mLogManager.LogOperationFailure(string.Format("{0} failed. {1} received", cmd.Name, ret));
            }
            else
            {
                mLogManager.LogMessage(string.Format("{0} succeeded. {1} received", cmd.Name, ret));
            }
        }

        public void SendCRC16Command(IController controller, IDeviceInfo device, CommandClass cmdClass, Command cmd, List<ParamValue> parameters, string msgSendCommand)
        {
            QueueAction action = null;
            try
            {
                byte[] data = mXmlDataManager.FillParameters(parameters);
                if (device.IsListening || device.IsCommandQueueOverrided || (device.Security & 0x20) != 0 || (device.Security & 0x40) != 0)
                {
                    SendCRC16Command(cmdClass, cmd, device.Id, controller, new List<byte>(data));
                }
                else
                {
                    action = new QueueAction(
                         new SendCRC16CommandDelegate(SendCRC16Command),
                         new object[] { cmdClass, cmd, device.Id, controller, new List<byte>(data) },
                         msgSendCommand, false, (byte)CommandTypes.CmdZWaveSendData, "Send CRC16 Command: " + cmd.Text, 1000);
                    if (mAddQueueCommandCallback != null)
                    {
                        mAddQueueCommandCallback.DynamicInvoke(new object[] { device.Id, action });
                        mLogManager.LogMessage(string.Format("{0} added to command queue for Node {1}.", cmd.Name, device.Id));
                    }
                }
            }
            catch (Exception ex)
            {
                mExceptionManager.RegisterException(ex);
            }
        }
        private delegate void SendCRC16CommandDelegate(CommandClass cmdClass, Command cmd, byte deviceId, IController controller, List<byte> data);
        public virtual void SendCRC16Command(CommandClass cmdClass, Command cmd, byte deviceId, IController controller, List<byte> data)
        {
            TransmitStatuses ret = TransmitStatuses.ResMissing;
            byte cmdClassCRC16 = mXmlDataManager.GetCommandClassKey("COMMAND_CLASS_CRC_16_ENCAP");
            byte cmdCRC16 = mXmlDataManager.GetCommandKey("COMMAND_CLASS_CRC_16_ENCAP", "CRC_16_ENCAP");
            List<byte> tmp = new List<byte>();
            if (data.Count > 0)
            {
                tmp = new List<byte>(data);
            }
            tmp.Insert(0, cmd.KeyId);
            tmp.Insert(0, cmdClass.KeyId);
            tmp.Insert(0, cmdCRC16);
            tmp.Insert(0, cmdClassCRC16);
            ushort crc = ZW_CreateCrc16(null, 0, tmp.ToArray(), (byte)tmp.Count);
            tmp.Add((byte)(crc >> 8));
            tmp.Add((byte)(crc));

            ret = controller.SendData(deviceId, tmp.ToArray(), TransmitOptions.TransmitOptionAutoRoute | TransmitOptions.TransmitOptionAcknowledge | TransmitOptions.TransmitOptionExplore);
            if (ret != TransmitStatuses.CompleteOk)
            {
                mLogManager.LogOperationFailure(string.Format("{0} failed. {1} received", cmd.Name, ret));
            }
            else
            {
                mLogManager.LogMessage(string.Format("{0} succeeded. {1} received", cmd.Name, ret));
            }
        }

        private ushort ZW_CheckCrc16(ushort crc, byte[] pDataAddr, ushort bDataLen)
        {
            ushort POLY = 0x1021;
            byte WorkData;
            byte bitMask;
            bool NewBit;
            byte i = 0;
            while ((bDataLen--) > 0)
            {
                WorkData = pDataAddr[i++];
                for (bitMask = 0x80; bitMask != 0; bitMask >>= 1)
                {
                    /* Align test bit with next bit of the message byte, starting with msb.
                    */
                    NewBit = ((WorkData & bitMask) != 0) ^ ((crc & 0x8000) != 0);
                    crc <<= 1;
                    if (NewBit)
                    {
                        crc ^= POLY;
                    }
                } /* for (bitMask = 0x80; bitMask != 0; bitMask >>= 1) */
            }
            return crc;
        }

        private ushort ZW_CreateCrc16(byte[] pHeaderAddr, byte bHeaderLen, byte[] pPayloadAddr, byte bPayloadLen)
        {
            ushort crc;
            crc = 0x1D0F;
            if (pHeaderAddr != null && bHeaderLen > 0)
            {
                crc = ZW_CheckCrc16(crc, pHeaderAddr, bHeaderLen);
            }
            crc = ZW_CheckCrc16(crc, pPayloadAddr, bPayloadLen);
            return crc;
        }

    }
}
