using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Enums;
using Zensys.Framework;

namespace Zensys.ZWave.Logging
{
    public static class SerialApiParser
    {
        internal static string GetParameterString(byte applicationByte, byte[] payload, bool isHostToZW, bool isReq)
        {
            StringBuilder ret = new StringBuilder();

            if (payload != null && payload.Length == 1 && payload[0] == applicationByte)
            {
            }
            else if (payload != null)
            {
                try
                {
                    CommandTypes type = (CommandTypes)applicationByte;
                    switch (type)
                    {
                        case CommandTypes.None:
                            break;
                        case CommandTypes.CmdSerialApiGetInitData:
                            #region
                            if (!isHostToZW)
                            {
                                ret.Append(string.Format("ver={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("capabilities={0}; ", Tools.ToHexString(payload[1])));
                                byte nodeIdx = 0;
                                if (payload[2] > 0)
                                {
                                    ret.Append(string.Format("nodes="));
                                    for (int i = 0; i < payload[2]; i++)
                                    {
                                        byte availabilityMask = payload[3 + i];
                                        for (byte bit = 0; bit < 8; bit++)
                                        {
                                            nodeIdx++;
                                            if ((availabilityMask & (1 << bit)) > 0)
                                            {
                                                ret.Append(string.Format("{0} ", nodeIdx.ToString("000")));
                                            }
                                        }
                                    }
                                    ret.Append(string.Format("; "));
                                }
                                ret.Append(string.Format("chip_type={0}; ", Tools.ToHexString(payload[3 + payload[2]])));
                                ret.Append(string.Format("chip_version={0}; ", Tools.ToHexString(payload[4 + payload[2]])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdSerialApiApplNodeInformation:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("deviceOptionsMask={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("generic={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("specific={0}; ", Tools.ToHexString(payload[2])));
                                if (payload[3] > 0)
                                {
                                    ret.Append(string.Format("nodeParm="));
                                    for (int i = 0; i < payload[3]; i++)
                                    {
                                        ret.Append(string.Format("{0} ", Tools.ToHexString(payload[4 + i])));
                                    }
                                    ret.Append(string.Format("; "));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdApplicationCommandHandler:
                            #region
                            if (!isHostToZW)
                            {
                                ret.Append(string.Format("rxStatus={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("sourceNode={0}; ", Tools.ToHexString(payload[1])));
                                if (payload[2] > 0)
                                {
                                    ret.Append(string.Format("pCmd="));
                                    for (int i = 0; i < payload[2]; i++)
                                    {
                                        ret.Append(string.Format("{0} ", Tools.ToHexString(payload[3 + i])));
                                    }
                                    ret.Append(string.Format("; "));
                                }
                                if (payload.Length > 3 + payload[2])
                                {
                                    ret.Append(string.Format("destNode={0}; ", Tools.ToHexString(payload[3 + payload[2]])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveGetControllerCapabilities:
                            #region
                            if (!isHostToZW)
                            {
                                ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdSerialApiSetTimeouts:
                            break;
                        case CommandTypes.CmdSerialApiGetCapabilities:
                            #region
                            if (!isHostToZW)
                            {
                                ret.Append(string.Format("AppVersion={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("AppRevision={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("ManufacturerId={0} {1}; ", Tools.ToHexString(payload[2]), Tools.ToHexString(payload[3])));
                                ret.Append(string.Format("ManufacturerProductType={0} {1}; ", Tools.ToHexString(payload[4]), Tools.ToHexString(payload[5])));
                                ret.Append(string.Format("ManufacturerProductId={0} {1}; ", Tools.ToHexString(payload[6]), Tools.ToHexString(payload[7])));
                                byte funcIdx = 0;
                                ret.Append(string.Format("SupportedFuncIds="));
                                for (int n = 0; n < payload.Length - 8; n++)
                                {
                                    byte availabilityMask = payload[n + 8];
                                    for (byte bit = 0; bit < 8; bit++)
                                    {
                                        funcIdx++;
                                        if ((availabilityMask & (1 << bit)) > 0)
                                        {
                                            ret.Append(string.Format("{0} ", Tools.ToHexString(funcIdx)));
                                        }
                                    }
                                }
                                ret.Append(string.Format("; "));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdSerialApiSoftReset:
                            break;
                        case CommandTypes.CmdZWaveSetRFReceiveMode:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("mode={0}; ", Tools.ToHexString(payload[0])));
                            }
                            else
                            {
                                ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSetSleepMode:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("mode={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("intEnable={0}; ", Tools.ToHexString(payload[1])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSendNodeInformation:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("destNode={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("txOptions={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[2])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSendData:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("nodeID={0}; ", Tools.ToHexString(payload[0])));
                                if (payload[1] > 0)
                                {
                                    ret.Append(string.Format("pData="));
                                    for (int i = 0; i < payload[1]; i++)
                                    {
                                        ret.Append(string.Format("{0} ", Tools.ToHexString(payload[2 + i])));
                                    }
                                    ret.Append(string.Format("; "));
                                }
                                ret.Append(string.Format("txOptions={0}; ", Tools.ToHexString(payload[2 + payload[1]])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[3 + payload[1]])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSendDataMulti:
                            #region
                            if (isHostToZW)
                            {
                                if (payload[0] > 0)
                                {
                                    ret.Append(string.Format("pNodeIDList="));
                                    for (int i = 0; i < payload[0]; i++)
                                    {
                                        ret.Append(string.Format("{0} ", payload[1 + i].ToString("000")));
                                    }
                                    ret.Append(string.Format("; "));


                                    if (payload[1 + payload[0]] > 0)
                                    {
                                        ret.Append(string.Format("pData="));
                                        for (int i = 0; i < payload[1 + payload[0]]; i++)
                                        {
                                            ret.Append(string.Format("{0} ", Tools.ToHexString(payload[2 + payload[0] + i])));
                                        }
                                        ret.Append(string.Format("; "));
                                    }
                                    ret.Append(string.Format("txOptions={0}; ", Tools.ToHexString(payload[2 + payload[0] + payload[1]])));
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[3 + payload[0] + payload[1]])));
                                }
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveGetVersion:
                            #region
                            if (!isHostToZW)
                            {
                                System.Text.UTF7Encoding utf = new System.Text.UTF7Encoding();
                                if (payload.Length > 12)
                                {
                                    ret.Append(string.Format("ver={0}; ", utf.GetString(payload, 0, 12)));
                                    ret.Append(string.Format("library_type={0}; ", Tools.ToHexString(payload[12])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSendDataAbort:
                            break;
                        case CommandTypes.CmdZWaveRFPowerLevelSet:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("powerLevel={0}; ", Tools.ToHexString(payload[0])));
                            }
                            else
                            {
                                ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSendDataMeta:
                            break;
                        case CommandTypes.CmdZWaveSendDataMR:
                            break;
                        case CommandTypes.CmdZWaveSendDataMetaMR:
                            break;
                        case CommandTypes.CmdZWaveSetRoutingInfo:
                            break;
                        case CommandTypes.CmdZWaveSendTestFrame:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("nodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("powerLevel={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[2])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdMemoryGetId:
                            #region
                            if (!isHostToZW)
                            {
                                ret.Append(string.Format("HomeID={0} {1} {2} {3}; ", Tools.ToHexString(payload[0]), Tools.ToHexString(payload[1]), Tools.ToHexString(payload[2]), Tools.ToHexString(payload[3])));
                                ret.Append(string.Format("NodeID={0}; ", payload[4].ToString("000")));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdMemoryGetByte:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("offset={0} {1}; ", Tools.ToHexString(payload[0]), Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdMemoryPutByte:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("offset={0} {1}; ", Tools.ToHexString(payload[0]), Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdMemoryGetBuffer:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("offset={0} {1}; ", Tools.ToHexString(payload[0]), Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("length={0}; ", Tools.ToHexString(payload[2])));
                            }
                            else
                            {
                                ret.Append(string.Format("buffer="));
                                for (int i = 0; i < payload.Length; i++)
                                {
                                    ret.Append(string.Format("{0} ", Tools.ToHexString(payload[i])));
                                }
                                ret.Append(string.Format("; "));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdMemoryPutBuffer:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("offset={0} {1}; ", Tools.ToHexString(payload[0]), Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("length={0} {1}; ", Tools.ToHexString(payload[2]), Tools.ToHexString(payload[3])));
                                ret.Append(string.Format("buffer="));
                                for (int i = 0; i < payload.Length - 5; i++)
                                {
                                    ret.Append(string.Format("{0} ", Tools.ToHexString(payload[i + 4])));
                                }
                                ret.Append(string.Format("; "));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[payload.Length - 1])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdClockSet:
                            break;
                        case CommandTypes.CmdClockGet:
                            break;
                        case CommandTypes.CmdClockCompare:
                            break;
                        case CommandTypes.CmdRtcTimerCreate:
                            break;
                        case CommandTypes.CmdRtcTimerRead:
                            break;
                        case CommandTypes.CmdRtcTimerDelete:
                            break;
                        case CommandTypes.CmdRtcTimerCall:
                            break;
                        case CommandTypes.CmdZWaveGetNodeProtocolInfo:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("bNodeID={0}; ", Tools.ToHexString(payload[0])));
                            }
                            else
                            {
                                ret.Append(string.Format("Capability={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("Security={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("Basic={0}; ", Tools.ToHexString(payload[3])));
                                ret.Append(string.Format("Generic={0}; ", Tools.ToHexString(payload[4])));
                                ret.Append(string.Format("Specific={0}; ", Tools.ToHexString(payload[5])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSetDefault:
                            #region
                            ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveReplicationCommandComplete:
                            break;
                        case CommandTypes.CmdZWaveReplicationSendData:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("destNodeID={0}; ", Tools.ToHexString(payload[0])));
                                if (payload[1] > 0)
                                {
                                    ret.Append(string.Format("pData="));
                                    for (int i = 0; i < payload[1]; i++)
                                    {
                                        ret.Append(string.Format("{0} ", Tools.ToHexString(payload[2 + i])));
                                    }
                                    ret.Append(string.Format("; "));
                                }
                                ret.Append(string.Format("txOptions={0}; ", Tools.ToHexString(payload[2 + payload[1]])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[3 + payload[1]])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveAssignReturnRoute:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("bSrcNodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("bDstNodeID={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[2])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveDeleteReturnRoute:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("nodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveRequestNodeNeighborUpdate:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("nodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdApplicationControllerUpdate:
                            #region
                            if (!isHostToZW)
                            {
                                if (payload.Length > 4)
                                {
                                    ret.Append(string.Format("bStatus={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("bNodeID={0}; ", Tools.ToHexString(payload[1])));
                                    ret.Append(string.Format("basic={0}; ", Tools.ToHexString(payload[3])));
                                    ret.Append(string.Format("generic={0}; ", Tools.ToHexString(payload[4])));
                                    ret.Append(string.Format("specific={0}; ", Tools.ToHexString(payload[5])));
                                    ret.Append(string.Format("commandclasses="));
                                    for (int i = 0; i < payload[2] - 3; i++)
                                    {
                                        ret.Append(string.Format("{0} ", Tools.ToHexString(payload[6 + i])));
                                    }
                                    ret.Append(string.Format("; "));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveAddNodeToNetwork:
                            #region
                            if (isHostToZW)
                            {
                                if (payload.Length > 1)
                                {
                                    ret.Append(string.Format("mode={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            else
                            {
                                if (payload.Length > 4)
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("bStatus={0}; ", Tools.ToHexString(payload[1])));
                                    ret.Append(string.Format("bSource={0}; ", Tools.ToHexString(payload[2])));
                                    ret.Append(string.Format("basic={0}; ", Tools.ToHexString(payload[4])));
                                    ret.Append(string.Format("generic={0}; ", Tools.ToHexString(payload[5])));
                                    ret.Append(string.Format("specific={0}; ", Tools.ToHexString(payload[6])));
                                    ret.Append(string.Format("commandclasses="));
                                    for (int i = 0; i < payload[3] - 3; i++)
                                    {
                                        ret.Append(string.Format("{0} ", Tools.ToHexString(payload[7 + i])));
                                    }
                                    ret.Append(string.Format("; "));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveRemoveNodeFromNetwork:
                            #region
                            if (isHostToZW)
                            {
                                if (payload.Length > 1)
                                {
                                    ret.Append(string.Format("mode={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            else
                            {
                                if (payload.Length > 4)
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("bStatus={0}; ", Tools.ToHexString(payload[1])));
                                    ret.Append(string.Format("bSource={0}; ", Tools.ToHexString(payload[2])));
                                    ret.Append(string.Format("basic={0}; ", Tools.ToHexString(payload[4])));
                                    ret.Append(string.Format("generic={0}; ", Tools.ToHexString(payload[5])));
                                    ret.Append(string.Format("specific={0}; ", Tools.ToHexString(payload[6])));
                                    ret.Append(string.Format("commandclasses="));
                                    for (int i = 0; i < payload[3] - 3; i++)
                                    {
                                        ret.Append(string.Format("{0} ", Tools.ToHexString(payload[7 + i])));
                                    }
                                    ret.Append(string.Format("; "));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveCreateNewPrimary:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("mode={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("bStatus={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("bSource={0}; ", Tools.ToHexString(payload[2])));
                                ret.Append(string.Format("basic={0}; ", Tools.ToHexString(payload[4])));
                                ret.Append(string.Format("generic={0}; ", Tools.ToHexString(payload[5])));
                                ret.Append(string.Format("specific={0}; ", Tools.ToHexString(payload[6])));
                                ret.Append(string.Format("commandclasses="));
                                for (int i = 0; i < payload[3] - 3; i++)
                                {
                                    ret.Append(string.Format("{0} ", Tools.ToHexString(payload[7 + i])));
                                }
                                ret.Append(string.Format("; "));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveControllerChange:
                            #region
                            if (isHostToZW)
                            {
                                if (payload.Length > 0)
                                    ret.Append(string.Format("mode={0}; ", Tools.ToHexString(payload[0])));
                                if (payload.Length > 1)
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                if (payload.Length > 0)
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                if (payload.Length > 1)
                                    ret.Append(string.Format("bStatus={0}; ", Tools.ToHexString(payload[1])));
                                if (payload.Length > 2)
                                    ret.Append(string.Format("bSource={0}; ", Tools.ToHexString(payload[2])));
                                if (payload.Length > 4)
                                    ret.Append(string.Format("basic={0}; ", Tools.ToHexString(payload[4])));
                                if (payload.Length > 5)
                                    ret.Append(string.Format("generic={0}; ", Tools.ToHexString(payload[5])));
                                if (payload.Length > 6)
                                    ret.Append(string.Format("specific={0}; ", Tools.ToHexString(payload[6])));
                                ret.Append(string.Format("commandclasses="));
                                if (payload.Length > 2)
                                {
                                    for (int i = 0; i < payload[3] - 3; i++)
                                    {
                                        ret.Append(string.Format("{0} ", Tools.ToHexString(payload[7 + i])));
                                    }
                                }
                                ret.Append(string.Format("; "));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveRequestNodeNeighborUpdateMR:
                            break;
                        case CommandTypes.CmdZWaveAssignReturnRouteMR:
                            break;
                        case CommandTypes.CmdZWaveSetLearnMode:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("mode={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("bStatus={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("bSource={0}; ", Tools.ToHexString(payload[2])));
                                ret.Append(string.Format("pCmd="));
                                for (int i = 0; i < payload[3] - 3; i++)
                                {
                                    ret.Append(string.Format("{0} ", Tools.ToHexString(payload[7 + i])));
                                }
                                ret.Append(string.Format("; "));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveAssignSucReturnRoute:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("bSrcNodeId={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[2])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveEnableSuc:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("state={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("capabilities={0}; ", Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveRequestNetworkUpdate:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSetSucNodeId:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("nodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("SUCState={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("bTxOption={0}; ", Tools.ToHexString(payload[2])));
                                ret.Append(string.Format("capabilities={0}; ", Tools.ToHexString(payload[3])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[4])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveDeleteSucReturnRoute:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("nodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveGetSucNodeId:
                            #region
                            if (!isHostToZW)
                            {
                                ret.Append(string.Format("SUCNodeID={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSendSucId:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("node={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("txOption={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[2])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("RetVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveAssignSucReturnRouteMR:
                            break;
                        case CommandTypes.CmdZWaveRediscoveryNeeded:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("bNodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveRequestNodeInfo:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("NodeID={0}; ", Tools.ToHexString(payload[0])));
                            }
                            else
                            {
                                ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveRemoveFailedNodeId:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("nodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveIsFailedNode:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("nodeID={0}; ", Tools.ToHexString(payload[0])));
                            }
                            else
                            {
                                ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveReplaceFailedNode:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("nodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[1])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdTimerStart:
                            break;
                        case CommandTypes.CmdTimerRestart:
                            break;
                        case CommandTypes.CmdTimerCancel:
                            break;
                        case CommandTypes.CmdTimerCall:
                            break;
                        case CommandTypes.CmdGetRoutingTableLine:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("bNodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("bRemoveBad={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("bRemoveNonReps={0}; ", Tools.ToHexString(payload[2])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[3])));
                            }
                            else
                            {
                                byte funcIdx = 0;
                                ret.Append(string.Format("Nodes="));
                                for (int n = 0; n < payload.Length; n++)
                                {
                                    byte availabilityMask = payload[n];
                                    for (byte bit = 0; bit < 8; bit++)
                                    {
                                        funcIdx++;
                                        if ((availabilityMask & (1 << bit)) > 0)
                                        {
                                            ret.Append(string.Format("{0} ", funcIdx.ToString("000")));
                                        }
                                    }
                                }
                                ret.Append(string.Format("; "));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdGetTXCounter:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("counter={0}; ", Tools.ToHexString(payload[0])));
                            }
                            else
                            {
                                ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdResetTXCounter:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("counter={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdStoreNodeInfo:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("bNodeID={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("capability={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("security={0}; ", Tools.ToHexString(payload[2])));
                                ret.Append(string.Format("basic={0}; ", Tools.ToHexString(payload[4])));
                                ret.Append(string.Format("generic={0}; ", Tools.ToHexString(payload[5])));
                                ret.Append(string.Format("specific={0}; ", Tools.ToHexString(payload[6])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[7])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdStoreHomeId:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("pHomeID="));
                                for (int i = 0; i < 4; i++)
                                {
                                    ret.Append(string.Format("{0} ", Tools.ToHexString(payload[i])));
                                }
                                ret.Append(string.Format("; "));
                                ret.Append(string.Format("bNodeID={0}; ", Tools.ToHexString(payload[4])));
                            }
                            #endregion
                            break;
                        case CommandTypes.CmdLockRouteResponse:
                            break;
                        case CommandTypes.CmdZWaveSendDataRouteDemo:
                            break;
                        case CommandTypes.CmdSerialApiTest:
                            break;
                        case CommandTypes.CmdSerialApiSlaveNodeInfo:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("destNode={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("listening={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("genericType={0}; ", Tools.ToHexString(payload[2])));
                                ret.Append(string.Format("specificType={0}; ", Tools.ToHexString(payload[3])));

                                ret.Append(string.Format("nodeParm="));
                                for (int i = 0; i < payload[4]; i++)
                                {
                                    ret.Append(string.Format("{0} ", Tools.ToHexString(payload[i + 5])));
                                }
                                ret.Append(string.Format("; "));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdApplicationSlaveCommandHandler:
                            break;
                        case CommandTypes.CmdZWaveSendSlaveNodeInfo:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("srcNode={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("destNode={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("txOptions={0}; ", Tools.ToHexString(payload[2])));
                                ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[3])));
                            }
                            else
                            {
                                if (!isReq)
                                {
                                    ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                                }
                                else
                                {
                                    ret.Append(string.Format("funcID={0}; ", Tools.ToHexString(payload[0])));
                                    ret.Append(string.Format("txStatus={0}; ", Tools.ToHexString(payload[1])));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSendSlaveData:
                            break;
                        case CommandTypes.CmdZWaveSetSlaveLearnMode:
                            break;
                        case CommandTypes.CmdZWaveGetVirtualNodes:
                            #region
                            if (!isHostToZW)
                            {
                                byte funcIdx = 0;
                                ret.Append(string.Format("Nodes="));
                                for (int n = 0; n < payload.Length; n++)
                                {
                                    byte availabilityMask = payload[n];
                                    for (byte bit = 0; bit < 8; bit++)
                                    {
                                        funcIdx++;
                                        if ((availabilityMask & (1 << bit)) > 0)
                                        {
                                            ret.Append(string.Format("{0} ", funcIdx.ToString("000")));
                                        }
                                    }
                                }
                                ret.Append(string.Format("; "));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveIsVirtualNode:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("nodeID={0}; ", Tools.ToHexString(payload[0])));
                            }
                            else
                            {
                                ret.Append(string.Format("retVal={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveReservedSSD:
                            break;
                        case CommandTypes.CmdApplicationCommandHandler_Bridge:
                            #region
                            if (!isHostToZW)
                            {
                                ret.Append(string.Format("rxStatus={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("destNodeID={0}; ", Tools.ToHexString(payload[1])));
                                ret.Append(string.Format("srcNodeID={0}; ", Tools.ToHexString(payload[2])));
                                if (payload[3] > 0)
                                {
                                    ret.Append(string.Format("pCmd="));
                                    for (int i = 0; i < payload[3]; i++)
                                    {
                                        ret.Append(string.Format("{0} ", Tools.ToHexString(payload[4 + i])));
                                    }
                                    ret.Append(string.Format("; "));
                                }
                                if (payload.Length > 4 + payload[3])
                                {
                                    byte funcIdx = 0;
                                    ret.Append(string.Format("multiDestsNodes="));
                                    for (int n = 0; n < payload[4 + payload[3]]; n++)
                                    {
                                        byte availabilityMask = payload[5 + payload[3]];
                                        for (byte bit = 0; bit < 8; bit++)
                                        {
                                            funcIdx++;
                                            if ((availabilityMask & (1 << bit)) > 0)
                                            {
                                                ret.Append(string.Format("{0} ", funcIdx.ToString("000")));
                                            }
                                        }
                                    }
                                    ret.Append(string.Format("; "));
                                }
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveGetRandom:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("noRandomBytes={0}; ", Tools.ToHexString(payload[0])));
                            }
                            else
                            {
                                ret.Append(string.Format("randomGenerationSuccess={0}; ", Tools.ToHexString(payload[0])));
                                ret.Append(string.Format("noRandomGenerated="));
                                for (int i = 0; i < payload[1]; i++)
                                {
                                    ret.Append(string.Format("{0} ", Tools.ToHexString(payload[i + 1])));
                                }
                                ret.Append(string.Format("; "));
                            }
                            break;
                            #endregion
                        case CommandTypes.CmdZWaveSetPromiscuousMode:
                            #region
                            if (isHostToZW)
                            {
                                ret.Append(string.Format("state={0}; ", Tools.ToHexString(payload[0])));
                            }
                            break;
                            #endregion
                        default:
                            break;
                    }
                }
                catch (Exception ex)
                {
                }
            }
            return ret.ToString();
        }

        internal static string GetApplicationString(byte applicationByte, byte[] payload, Type enumType)
        {
            if (payload != null && payload.Length == 1 && payload[0] == applicationByte)
            {
                HeaderTypes type = (HeaderTypes)applicationByte;
                return type.ToString();
            }
            else
            {
                string ret = Tools.ToHexString(applicationByte);
                try
                {
                    ret = Enum.GetName(enumType, applicationByte);
                }
                catch (ArgumentNullException)
                {
                }
                catch (ArgumentException)
                {
                }
                return ret;
            }
        }
    }
}
