using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Events;

namespace Zensys.ZWave.Devices
{
    public class CommandClassesStore
    {
        private IController mController;
        private const uint EEPROM_START = 0x100;
        private const uint EEP_MAX_CLASSES = 32;
        private const uint EEP_CLASS_START = EEPROM_START + EEP_MAX_CLASSES;
        private const byte EEP_BITMASK_LENGTH = 29;
        private byte[] storedClasses = new byte[EEP_MAX_CLASSES];
        private NodeBitmask[] NodeMask;

        public event GetExternalStorageBufferEventHandler GetExternalStorageBuffer;

        public event PutExternalStorageBufferEventHandler PutExternalStorageBuffer;


        public void Load()
        {
            NodeMask = new NodeBitmask[EEP_MAX_CLASSES];
            for (int i = 0; i < EEP_MAX_CLASSES; i++)
            {
                NodeMask[i] = new NodeBitmask(EEP_BITMASK_LENGTH);
            }
            GetStoredClassTable();
            for (byte i = 0; i < storedClasses.Length; i++)
            {
                if (storedClasses[i] != 0)
                {
                    GetStoredMask(i);
                }
            }
        }

        private CommandClassesStore()
        {
        }

        public CommandClassesStore(IController controller, bool useExternalStorage)
        {
            mController = controller;
            mUsingExternalStorage = useExternalStorage;
        }

        /// <summary>
        /// Stores a command class for a given NodeID
        /// </summary>
        /// <param name="nodeId"></param>
        /// <param name="cmdClass"></param>
        /// <returns>true if succesfull false if not</returns>
        public bool Store(byte nodeId, byte cmdClass)
        {

            uint i = 0;
            uint firstZero = EEP_MAX_CLASSES;
            for (i = 0; i < EEP_MAX_CLASSES; i++)
            {
                if ((storedClasses[i] == 0) && (firstZero == EEP_MAX_CLASSES))
                {
                    firstZero = i;
                }
                if (storedClasses[i] == cmdClass)
                {
                    break;
                }
            }
            if (i < EEP_MAX_CLASSES)
            {
                //EEPROM already have class update and store this nodeID
                StoreNodeMaskByte(nodeId, (byte)i, true);
            }
            else
            {
                //New Cmdclass store it if we have room
                if (firstZero < EEP_MAX_CLASSES)
                {
                    StoreClassIndex((byte)firstZero, cmdClass);
                    NodeMask[firstZero].ZWaveNodeMaskClear();
                    StoreNodeMaskByte(nodeId, (byte)firstZero, true);
                }
                else
                {
                    //No more room for Cmd Classes
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// Stores a class in the index and EEPROM
        /// </summary>
        /// <param name="index"></param>
        /// <param name="cmdClass"></param>
        private void StoreClassIndex(byte index, byte cmdClass)
        {
            storedClasses[index] = cmdClass;
            try
            {
                if (!UsingExternalStorage)
                {
                    mController.Memory.PutByte(EEPROM_START + index, cmdClass);
                }
                else
                {
                    if (PutExternalStorageBuffer != null)
                    {
                        PutExternalStorageBuffer(EEPROM_START + index, new byte[] { cmdClass }, 1);
                    }
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e.Message.ToString());
            }
        }

        /// <summary>
        /// Stores a single mask byte in external EEPROM and the RAM table
        /// </summary>
        /// <param name="nodeId">nodeId</param>
        /// <param name="index">index of mask</param>
        /// <param name="val">true if to write 1 and false to write 0</param>
        private void StoreNodeMaskByte(byte nodeId, byte index, bool val)
        {
            uint adr = EEP_CLASS_START + (uint)(index * EEP_BITMASK_LENGTH);
            uint offset = (uint)nodeId >> 3;
            if (val)
            {
                NodeMask[index].ZWaveNodeMaskSetBit(nodeId);
            }
            else
            {
                NodeMask[index].ZWaveNodeMaskClearBit(nodeId);
            }
            byte maskbyte = NodeMask[index].Get(nodeId);
            if (!UsingExternalStorage)
            {
                mController.Memory.PutByte(adr + offset, maskbyte);
            }
            else
            {
                if (PutExternalStorageBuffer != null)
                {
                    PutExternalStorageBuffer(adr + offset, new byte[] { maskbyte }, 1);
                }
            }
        }

        /// <summary>
        /// Retrieves a stored nodemask from the EEPROM
        /// </summary>
        /// <param name="maskNo">zero based offset of mask to retrieve</param>
        private void GetStoredMask(byte maskNo)
        {
            try
            {
                byte[] arr = null;

                if (!UsingExternalStorage)
                {
                    arr = mController.Memory.GetBuffer(EEP_CLASS_START + (uint)(maskNo * EEP_BITMASK_LENGTH), (byte)EEP_BITMASK_LENGTH);
                }
                else
                {
                    if (GetExternalStorageBuffer != null)
                    {
                        arr = GetExternalStorageBuffer(EEP_CLASS_START + (uint)(maskNo * EEP_BITMASK_LENGTH), (byte)EEP_BITMASK_LENGTH);
                    }
                }

                if (NodeMask.Length > maskNo)
                {
                    NodeMask[maskNo].Store(arr);
                }
                else
                {

                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e.Message.ToString());
            }
        }

        /// <summary>
        /// Retrieves all the Stored Class table from the EEPROM
        /// </summary>
        /// <returns></returns>
        private void GetStoredClassTable()
        {
            try
            {
                byte[] arr = null;
                if (!UsingExternalStorage)
                {
                    arr = mController.Memory.GetBuffer(EEPROM_START, (byte)EEP_MAX_CLASSES);
                }
                else
                {
                    if (GetExternalStorageBuffer != null)
                    {
                        arr = GetExternalStorageBuffer(EEPROM_START, (byte)EEP_MAX_CLASSES);
                    }
                }
                if (arr.Length == storedClasses.Length)
                {
                    storedClasses = arr;
                }
            }
            catch (Exception e)
            {

            }
        }

        /// <summary>
        /// Returns an array with the supported commandclasses for this nodeID.
        /// </summary>
        /// <param name="nodeID"></param>
        /// <returns>an array with the supported commandclasses for a given NodeID. null if no supported classes</returns>
        public byte[] CmdClasses(byte nodeId)
        {
            byte[] temp = new byte[EEP_MAX_CLASSES];
            int n = 0;
            for (uint i = 0; i < EEP_MAX_CLASSES; i++)
            {
                if (storedClasses[i] != 0)
                {
                    if (NodeMask[i].ZWaveNodeMaskNodeIn(nodeId))
                    {
                        temp[n++] = storedClasses[i];
                    }
                }
            }
            if (n != 0)
            {
                byte[] ret = new byte[n];
                while (n > 0)
                {
                    ret[n - 1] = temp[n - 1];
                    n--;
                }
                return ret;
            }
            else
            {
                return new byte[] { };
            }
        }

        /// <summary>
        /// Returns wheter a specific commandclass is supported or not
        /// </summary>
        /// <param name="nodeId"></param>
        /// <param name="cmdClass"></param>
        /// <returns></returns>
        public bool IsSupported(byte nodeId, byte cmdClass)
        {
            byte i = CmdClassToIndex(cmdClass);
            if (i == 0xFF)
            {
                return false;
            }
            return NodeMask[i].ZWaveNodeMaskNodeIn(nodeId);
        }

        /// <summary>
        /// Returns Index a given commandclass is stored at
        /// </summary>
        /// <param name="cmdClass">command class to search for</param>
        /// <returns>index or 0xFF if not found</returns>
        private byte CmdClassToIndex(byte cmdClass)
        {
            byte i = 0;
            for (i = 0; i < EEP_MAX_CLASSES; i++)
            {
                if (cmdClass == storedClasses[i])
                {
                    break;
                }
            }
            if (i == EEP_MAX_CLASSES)
            {
                i = 0xFF;
            }
            return i;
        }




        public NodeBitmask[] Items
        {
            get { return NodeMask; }

        }

        //public List<byte> GetCommandClassesByNodeId(byte nodeId)
        //{
        //    List<byte> result = new List<byte>();
        //    foreach (CommandClassesStoreItem item in Items)
        //    {
        //        if (NodeInMask(nodeId, item.NodeMask))
        //        {
        //            result.Add(item.CommandClass);
        //        }
        //    }
        //    return result;
        //}

        public static bool NodeInMask(byte nodeId, byte[] mask)
        {
            if (nodeId < 1)
            {
                return false;
            }
            nodeId--;
            if ((((mask[(nodeId >> 3)]) >> (byte)(nodeId & 7)) & (byte)0x01) != 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        //public CommandClassesStoreItem GetCommandClassItem(byte cmdClass)
        //{
        //    CommandClassesStoreItem result = null;
        //    foreach (CommandClassesStoreItem item in Items)
        //    {
        //        if (item.CommandClass == cmdClass)
        //        {
        //            result = item;
        //            break;
        //        }
        //    }
        //    return result;
        //}

        public static void NodeMaskSetBit(byte nodeId, byte[] mask)
        {
            mask[nodeId >> 3] |= (byte)(0x1 << (nodeId & 7));
        }


        public void ClearEEPROMData()
        {
            if (NodeMask != null)
            {
                for (int resetCount = 0; resetCount < EEP_MAX_CLASSES; resetCount++)
                {
                    //Clear bitmasks for each Command Class

                    NodeMask[resetCount].ZWaveNodeMaskClear();

                    if (!UsingExternalStorage)
                    {
                        mController.Memory.PutBuffer(EEP_CLASS_START + (uint)(resetCount * EEP_BITMASK_LENGTH), NodeMask[resetCount].Get(), EEP_BITMASK_LENGTH);
                    }
                    else
                    {
                        if (PutExternalStorageBuffer != null)
                        {
                            PutExternalStorageBuffer(EEP_CLASS_START + (uint)(resetCount * EEP_BITMASK_LENGTH), NodeMask[resetCount].Get(), EEP_BITMASK_LENGTH);
                        }
                    }
                }
            }
            byte[] clearBuf = new byte[EEP_MAX_CLASSES];
            // Clear the stored command class values
            if (!UsingExternalStorage)
            {
                mController.Memory.PutBuffer(EEPROM_START, clearBuf, EEP_MAX_CLASSES);
            }
            else
            {
                if (PutExternalStorageBuffer != null)
                {
                    PutExternalStorageBuffer(EEPROM_START, clearBuf, EEP_MAX_CLASSES);
                }
            }
        }

        private bool mUsingExternalStorage;
        public bool UsingExternalStorage
        {
            get { return mUsingExternalStorage; }
        }

        public void LoadAllFromDeviceMemory()
        {
            bool prev = UsingExternalStorage;
            mUsingExternalStorage = false;
            Load();
            mUsingExternalStorage = prev;
        }

        public void StoreAllToDeviceMemory()
        {
            bool prev = UsingExternalStorage;
            mUsingExternalStorage = false;
            for (byte index = 0; index < this.storedClasses.Length; index++)
            {
                StoreClassIndex(index, storedClasses[index]);
            }
            for (int resetCount = 0; resetCount < EEP_MAX_CLASSES; resetCount++)
            {
                mController.Memory.PutBuffer(EEP_CLASS_START + (uint)(resetCount * EEP_BITMASK_LENGTH), NodeMask[resetCount].Get(), EEP_BITMASK_LENGTH);
            }
            mUsingExternalStorage = prev;
        }
    }
}
