using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Enums;

namespace Zensys.ZWave.SerialPortFrame
{
    internal class DataFrame : IDataFrame
    {
        /// <summary>
        /// If true ToString() returns DataFrame command as the defined Enum
        /// If false the numerical value is returned
        /// </summary>
        public bool ToStringShowCmdAsEnum = true;
        private const int BUFFER_SIZE = 100;

        private bool mIsRemoteSerialApi;
        public bool IsRemoteSerialApi
        {
            get { return mIsRemoteSerialApi; }
            set { mIsRemoteSerialApi = value; }
        }
        /// <summary>
        /// DataFrame()
        /// </summary>
        public DataFrame(bool isRemoteSerialApi)
        {
            mIsRemoteSerialApi = isRemoteSerialApi;
            this.timestamp = DateTime.Now;
            this.payloadBuffer = new byte[BUFFER_SIZE];
        }

        /// <summary>
        /// DataFrame(byte payloadSize)
        /// </summary>
        /// <param name="payloadSize"></param>
        public DataFrame(byte payloadSize, bool isRemoteSerialApi)
        {
            mIsRemoteSerialApi = isRemoteSerialApi;
            this.timestamp = DateTime.Now;
            this.payloadBuffer = new byte[payloadSize];
        }

        /// <summary>
        /// AddPayload(byte data)
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        public bool AddPayload(byte data)
        {
            if (payloadIdx > payloadBuffer.Length - 1)
                return false;
            payloadBuffer[payloadIdx++] = data;
            return true;
        }

        /// <summary>
        /// AddPayload(byte[] data)
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        public bool AddPayload(byte[] data)
        {
            if (data == null)
            {
                throw new ArgumentNullException("data");
            }
            if ((payloadIdx + data.Length) > payloadBuffer.Length - 1)
                return false;
            data.CopyTo(payloadBuffer, payloadIdx);
            payloadIdx += (byte)data.Length;
            return true;
        }

        /// <summary>
        /// GetPayloadBuffer()
        /// </summary>
        /// <returns></returns>
        public byte[] GetPayloadBuffer()
        {
            return payloadBuffer;
        }

        /// <summary>
        /// Command
        /// </summary>
        public byte Command
        {
            set
            {
                this.cmd = value;
            }
            get
            {
                return cmd;
            }
        }

        /// <summary>
        /// Type
        /// </summary>
        public FrameTypes Type
        {
            set
            {
                this.type = value;
            }
            get
            {
                return type;
            }
        }

        /// <summary>
        ///  isPayloadFull()
        /// </summary>
        /// <returns></returns>
        public bool IsPayloadFull()
        {
            return payloadIdx > payloadBuffer.Length - 1;
        }

        /// <summary>
        /// isChecksumValid(byte checksum)
        /// </summary>
        /// <param name="checksum"></param>
        /// <returns></returns>
        public bool IsChecksumValid(byte checksum)
        {
            if (IsRemoteSerialApi)
                return true;
            else
                return calculateChecksum() == checksum;
        }

        /// <summary>
        /// String
        /// </summary>
        /// <returns></returns>
        public override String ToString()
        {
            StringBuilder sb = new StringBuilder(100);

            if (!IsRemoteSerialApi)
            {
                // Header...
                sb.Append((byte)HeaderTypes.StartOfFrame);
                sb.Append(' ');
            }
            // Length...
            sb.Append((byte)(payloadIdx + 3));
            sb.Append(' ');

            // Type...
            sb.Append((byte)type);
            sb.Append(' ');

            // Command...
            if (ToStringShowCmdAsEnum)
            {
                sb.Append(cmd.ToString());
            }
            else
            {
                sb.Append((byte)cmd);
            }
            sb.Append(' ');

            // Data payload...
            for (int i = 0; i < payloadIdx; i++)
            {
                sb.Append(payloadBuffer[i]);
                sb.Append(' ');
            }
            if (!IsRemoteSerialApi)
            {
                // Checksum...
                sb.Append(calculateChecksum());
            }
            return sb.ToString();
        }

        /// <summary>
        /// GetFrameBuffer()
        /// </summary>
        /// <returns></returns>
        public byte[] GetFrameBuffer()
        {
            byte[] buffer;
            if (IsRemoteSerialApi)
            {
                buffer = new byte[payloadIdx + 3];
                buffer[IDX_LENGTH - 1] = (byte)(payloadIdx + 3);
                buffer[IDX_TYPE - 1] = (byte)type;
                buffer[IDX_CMD - 1] = (byte)cmd;
                Array.Copy(payloadBuffer, 0, buffer, IDX_DATA - 1, payloadIdx);
            }
            else
            {
                buffer = new byte[payloadIdx + 5];
                buffer[IDX_HEADER] = (byte)HeaderTypes.StartOfFrame;
                buffer[IDX_LENGTH] = (byte)(payloadIdx + 3);
                buffer[IDX_TYPE] = (byte)type;
                buffer[IDX_CMD] = (byte)cmd;
                Array.Copy(payloadBuffer, 0, buffer, IDX_DATA, payloadIdx);
                buffer[buffer.Length - 1] = calculateChecksum();
            }
            return buffer;
        }

        private byte calculateChecksum()
        {
            byte calcChksum = 0xFF;
            calcChksum ^= (byte)(payloadIdx + 3); // Length
            calcChksum ^= (byte)type;     // Type
            calcChksum ^= (byte)cmd;      // Command
            for (int i = 0; i < payloadIdx; i++)
                calcChksum ^= payloadBuffer[i];      // Data
            return calcChksum;
        }

        /// <summary>
        /// TimeStamp
        /// </summary>
        public DateTime Timestamp
        {
            get
            {
                return timestamp;
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="df1"></param>
        /// <param name="df2"></param>
        /// <returns></returns>
        public static bool operator ==(DataFrame df1, DataFrame df2)
        {
            if (!(df1 is DataFrame) || !(df2 is DataFrame))
                return false;
            if (df1.Type != df2.Type)
                return false;
            if (df1.Command != df2.Command)
                return false;
            if (df1.payloadIdx != df2.payloadIdx)
                return false;
            for (int i = 0; i < df1.payloadIdx; i++)
                if (df1.payloadBuffer[i] != df2.payloadBuffer[i])
                    return false;
            return true;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="df1"></param>
        /// <param name="df2"></param>
        /// <returns></returns>
        public static bool operator !=(DataFrame df1, DataFrame df2)
        {
            return !(df1 == df2);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            if (!(obj is DataFrame))
                return false;
            return (this == (DataFrame)obj);
        }


        /// <summary>
        /// GetHashCode()
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }

        private byte[] payloadBuffer;
        private byte payloadIdx;
        private DateTime timestamp;
        private FrameTypes type;
        private byte cmd;



        private const int IDX_HEADER = 0;
        private const int IDX_LENGTH = 1;
        private const int IDX_TYPE = 2;
        private const int IDX_CMD = 3;
        private const int IDX_DATA = 4;





        #region IDataFrame Members


        public byte[] GetPayload()
        {
            if (payloadIdx > 0)
            {
                byte[] buffer = new byte[payloadIdx];
                Array.Copy(payloadBuffer, 0, buffer, 0, payloadIdx);
                return buffer;
            }
            else
                return null;
            // return payloadBuffer;
        }

        #endregion
    }
}
