using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Devices;
using Zensys.ZWave.Enums;
using Zensys.Framework;
using Zensys.ZWave.Logging;
using Zensys.ZWave.Events;
using Zensys.ZWave.Exceptions;

namespace Zensys.ZWave.SerialPortApplication.Devices
{
    public class DeviceMemory : IDeviceMemory
    {
        public event ProgressChangedEventHandler ProgressChanged;

        private ISessionLayer mSessionLayer;

        public DeviceMemory(ISessionLayer sessionLayer)
        {
            mSessionLayer = sessionLayer;
        }

        #region IDeviceMemory Members

        private IDevice mDevice;
        public IDevice Device
        {
            get
            {
                return mDevice;
            }
            set
            {
                mDevice = value;
            }
        }
        private byte[] mBuffer;
        public byte[] Buffer
        {
            get { return mBuffer; }
            set { mBuffer = value; }
        }
        public void GetId()
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession("DeviceMemory.GetId"));
            byte[] response = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdMemoryGetId, null);
            if (response.Length == 5)
            {
                this.Device.HomeId = new byte[] { response[0], response[1], response[2], response[3] };
                this.Device.Id = response[4];
            }
        }

        public byte GetByte(uint offset)
        {
            byte[] response = null;
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(string.Format("DeviceMemory.GetByte (offset={0})", offset)));
            response = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdMemoryGetByte, new byte[] { (byte)(offset >> 8), (byte)(offset & 0xFF) });
            return response[0];
        }

        public TransmitStatuses PutByte(uint offset, byte value)
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(string.Format("DeviceMemory.PutByte (offset={0}, value={1})", offset, value)));
            byte[] response = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdMemoryPutByte, new byte[] { (byte)(offset >> 8), (byte)(offset & 0xFF), value });
            if (response != null && response.Length > 0 && response[0] == 0x01)
                return TransmitStatuses.CompleteOk;
            else
                return TransmitStatuses.CompleteFail;

        }

        public byte[] GetBuffer(uint offset, byte length)
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(string.Format("DeviceMemory.GetBuffer (offset={0}, length={1})", offset, length)));
            byte[] response = null;
            if (this.Device != null)
            {
                if (this.Device.IsSupportedSerialApiCommand((byte)CommandTypes.CmdMemoryGetBuffer))
                {
                    byte[] request = new byte[3];
                    request[0] = ((byte)(offset >> 8));
                    request[1] = ((byte)(offset & 0xFF));
                    request[2] = (length);
                    response = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdMemoryGetBuffer, request);
                }
                else if (this.Device.IsSupportedSerialApiCommand((byte)CommandTypes.CmdMemoryGetByte, true))
                {
                    byte[] buffer = new byte[length];
                    for (uint i = 0; i < buffer.Length; i++)
                    {
                        buffer[i] = this.GetByte(offset + i);
                    }
                    response = buffer;
                }
            }
            return response;
        }

        public TransmitStatuses PutBuffer(uint offset, byte[] buffer, uint length)
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(string.Format("DeviceMemory.PutBuffer (offset={0}, length={1})", offset, length)));
            TransmitStatuses tsret = TransmitStatuses.CompleteFail;
            if (length > 66) length = 66;
            byte[] request = new byte[buffer.Length + 4 + 1];
            request[0] = (byte)(offset >> 8);
            request[1] = (byte)(offset & 0xFF);
            request[2] = (byte)(length >> 8);
            request[3] = (byte)(length & 0xFF);
            for (int i = 0; i < buffer.Length; i++)
            {
                request[i + 4] = buffer[i];
            }
            request[request.Length - 1] = mSessionLayer.SequenceNumber;
            if (this.Device.IsSupportedSerialApiCommand((byte)CommandTypes.CmdMemoryPutBuffer))
            {
                List<byte[]> responses = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdMemoryPutBuffer, 2, (int)mSessionLayer.RequestTimeout.TotalMilliseconds, request);
                if (responses.Count == 2 && responses[1].Length > 1)
                {
                    tsret = TransmitStatuses.CompleteOk;
                }
                else if (responses.Count > 0 && responses[0].Length > 1 && responses[0][0] == 0x01)
                {
                    tsret = TransmitStatuses.ResMissing;
                }
            }
            else if (this.Device.IsSupportedSerialApiCommand((byte)CommandTypes.CmdMemoryPutByte, true))
            {
                for (int i = 0; i < buffer.Length; i++)
                {
                    tsret = PutByte((uint)(offset + i), buffer[i]);
                    if (tsret != TransmitStatuses.CompleteOk)
                        break;
                }
            }
            return tsret;
        }

        public NVMTypes SwitchEEPROM(bool state)
        {
            NVMTypes nvm_type = NVMTypes.NVMInvalid;
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0} ({1})", "Switch EEPROM", state.ToString())));
            byte seqNum = mSessionLayer.SequenceNumber;
            byte[] response = null;
            if (state)
            {
                try
                {
                    response = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_TOGGLE_EEPROM_IF,
                        new byte[] { 0x01, this.Device.ChipType, seqNum });
                }
                catch (RequestTimeoutException)
                {
                    //Old programmer firmwares are supported only Serial EEPROM and does not supprt returnting the responces.
                    nvm_type = NVMTypes.NVMSerialEEPROM;
                }
                catch
                {
                    nvm_type = NVMTypes.NVMInvalid;
                }
            }
            else
            {
                try
                {
                    response = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_TOGGLE_EEPROM_IF,
                        new byte[] { 0x00, this.Device.ChipType, seqNum });
                }
                catch (RequestTimeoutException)
                {
                    nvm_type = NVMTypes.NVMUnknown;
                }
                catch
                {
                    nvm_type = NVMTypes.NVMInvalid;
                }
            }
            if (response != null)
            {
                if (response.Length == 5)
                {
                    if (response[4] == seqNum)
                    {
                        nvm_type = (NVMTypes)
                            ((((uint)response[0]) << (int)NVMTypes.NVMTypePos)
                            | (((uint)response[1]) << (int)NVMTypes.NVMManufacturerPos)
                            | (((uint)response[2]) << (int)NVMTypes.NVMDevicePos)
                            | (((uint)response[3]) << (int)NVMTypes.NVMSizePos));
                    }
                }
            }
            return nvm_type;
        }

        public void ClearEEPROM(byte[] eepromDataRaw)
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0}", "Clear EEPROM")));
            WriteEEPROM(0, eepromDataRaw, Constants.EEPROM_PAGE_SIZE);
        }

        public void ReadEEPROM()
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0}", "Read EEPROM")));
            List<byte> tempBuffer = new List<byte>();
            for (int i = 0; i < (int)this.EepromSize; i += (int)Constants.EEP_MAX_CLASSES)
            {
                byte[] request = new byte[5];
                request[0] = (byte)(i >> 16);
                request[1] = (byte)(i >> 8);
                request[2] = (byte)(i & 0xFF);
                request[3] = (byte)Constants.EEP_MAX_CLASSES;
                request[4] = mSessionLayer.SequenceNumber;
                System.Diagnostics.Debug.WriteLine("!!!!!!!!!!!!!!!!!!!" + request[0].ToString("X2") +
                    request[1].ToString("X2") +
                    request[2].ToString("X2") +
                    request[3].ToString("X2") +
                    request[4].ToString("X2"));
                byte[] buffer = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_MEMORY_GET_BUFFER, request);//(ushort)i, (byte)Constants.EEP_MAX_CLASSES);
                if (buffer != null)
                {
                    tempBuffer.AddRange(buffer);
                    tempBuffer.RemoveAt(tempBuffer.Count - 1); //Remove checksum
                    if (tempBuffer[i] == (byte)(i >> 16)
                        && tempBuffer[i + 1] == (byte)(i >> 8)
                        && tempBuffer[i + 2] == (byte)(i & 0xFF))
                    {
                        tempBuffer.RemoveAt(i); //Remove address
                        tempBuffer.RemoveAt(i);
                        tempBuffer.RemoveAt(i);
                    }
                }
            }
            if (tempBuffer.Count > 0)
            {
                this.Buffer = tempBuffer.ToArray();
            }
            else
            {
                this.Buffer = null;
            }
        }

        public bool Compare(byte[] eepromDataRaw)
        {
            bool result = true;
            if (this.Buffer.Length == eepromDataRaw.Length)
            {
                for (int i = 0; i < this.Buffer.Length; i++)
                {
                    if (i < 8 || i > 11)
                    {
                        if (this.Buffer[i] != eepromDataRaw[i])
                        {
                            result = false;
                            break;
                        }
                    }
                }
            }
            else
            {
                result = false;
            }
            return result;
        }

        public uint SramSize
        {
            get
            {
                if (this.Device.ChipType == (byte)ChipTypes.ZW040x)
                    return Constants.SRAM_SIZE;
                else if (this.Device.ChipType == (byte)ChipTypes.ZW050x)
                    return 4 * 1024;

                else
                    return 0;
            }
        }

        public bool WriteSRAM(byte[] sramDataRaw, bool verify)
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0}", "Write SRAM")));
            return WriteSRAMByPages(sramDataRaw,
                (int)(this.SramSize / Constants.SRAM_PAGE_SIZE), (int)Constants.SRAM_PAGE_SIZE, verify);
        }

        private bool WriteSRAMByPages(byte[] sramDataRaw, int pagesCount, int pageSize, bool verify)
        {
            verify = false;
            bool result = true;
            for (int pageIndex = 0; pageIndex < pagesCount; pageIndex++)
            {
                //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0} ({1})", "SRAM Write Page", pageIndex.ToString())));

                byte[] pageBuffer = new byte[pageSize];
                for (int j = 0; j < pageSize; j++)
                {
                    pageBuffer[j] = sramDataRaw[pageIndex * pageSize + j];
                }

                if (this.Device.ChipType == (byte)ChipTypes.ZW050x)
                {
                    //verify = false;
                    byte[] buffer1 = new byte[pageBuffer.Length / 2];
                    Array.Copy(pageBuffer, 0, buffer1, 0, buffer1.Length);
                    byte[] buffer2 = new byte[pageBuffer.Length / 2];
                    Array.Copy(pageBuffer, buffer1.Length, buffer2, 0, buffer2.Length);

                    result = WriteSRAMPage(pageIndex * 2, buffer1, verify);
                    result &= WriteSRAMPage(pageIndex * 2 + 1, buffer2, verify);
                }
                else
                {
                    result = WriteSRAMPage(pageIndex, pageBuffer, verify);
                }

            }
            return result;
        }

        private bool WriteSRAMPage(int pageIndex, byte[] pageBuffer, bool verify)
        {
            bool result = false;
            ProgrammerCommandTypes cmd;
            byte[] request = new byte[4 + pageBuffer.Length];
            int i = 0;
            cmd = ProgrammerCommandTypes.FUNC_ID_ZW0x0x_SRAM_WRITE_PAGE;
            request[i++] = (byte)this.Device.ChipType;
            request[i++] = (byte)pageIndex;
            request[i++] = (byte)(verify ? 1 : 0);
            for (int j = 0; j < pageBuffer.Length; j++)
                request[i++] = pageBuffer[j];
            byte seqNum;
            request[i++] = seqNum = mSessionLayer.SequenceNumber;
            byte[] response = mSessionLayer.ExecuteRequest((byte)cmd, request);
            if (response != null)
            {
                if (response.Length == 2)
                {
                    result = (response[0] == (byte)FlashProgrammingStatuses.SUCCESS && response[1] == seqNum);
                }
            }
            return result;
        }

        public bool ReadSRAM()
        {
            bool res = true;
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0}", "Read SRAM")));
            List<byte> tempBuffer = new List<byte>();

            for (int paramPageIndex = 0; paramPageIndex < this.SramSize / Constants.SRAM_PAGE_SIZE; paramPageIndex++)
            {
                if (this.Device.ChipType == (byte)ChipTypes.ZW050x)
                {
                    byte[] page = ReadSRAMPage(paramPageIndex * 2);
                    if (page != null)
                    {
                        tempBuffer.AddRange(page);
                    }
                    else
                    {
                        res = false;
                        break;
                    }
                    page = ReadSRAMPage(paramPageIndex * 2 + 1);
                    if (page != null)
                    {
                        tempBuffer.AddRange(page);
                    }
                    else
                    {
                        res = false;
                        break;
                    }
                }
                else
                {
                    byte[] page = ReadSRAMPage(paramPageIndex);
                    if (page != null)
                    {
                        tempBuffer.AddRange(page);
                    }
                    else
                    {
                        res = false;
                        break;
                    }
                }
            }

            if (tempBuffer.Count > 0 && res)
            {
                this.Buffer = tempBuffer.ToArray();
            }
            else
            {
                this.Buffer = null;
            }
            return res;
        }

        public byte[] ReadSRAMPage(int pageIndex)
        {
            byte[] result = null;
            ProgrammerCommandTypes cmd;
            byte[] request = new byte[3];
            int i = 0;
            cmd = ProgrammerCommandTypes.FUNC_ID_ZW0x0x_SRAM_READ_PAGE;
            request[i++] = (byte)this.Device.ChipType;
            request[i++] = (byte)pageIndex;
            byte seqNum;
            request[i++] = seqNum = mSessionLayer.SequenceNumber;
            byte[] response = mSessionLayer.ExecuteRequest((byte)cmd, request);
            if (response != null)
            {
                //if (response.Length == Constants.SRAM_PAGE_SIZE + 1)
                //{
                //	if (response[Constants.SRAM_PAGE_SIZE] == seqNum)
                //	{
                List<byte> tempBuffer = new List<byte>();
                tempBuffer.AddRange(response);
                tempBuffer.RemoveAt(tempBuffer.Count - 1);
                result = tempBuffer.ToArray();
                //	}
                //}
            }
            return result;
        }


        public IFlashSettings ReadSRAMRfOptions()
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession("Read SRAM Options"));
            IFlashSettings result = null;
            int pageIndex = (int)(this.SramSize / (uint)Constants.SRAM_PAGE_SIZE - 1);	//RF Settings are in last page.
            uint offset = (uint)this.Device.Flash.FlashSize - (uint)Constants.SRAM_PAGE_SIZE;	//we need Absolute OTP
            // address here, because SRAM emulates the part of OTP in Development mode.
            byte[] page = this.ReadSRAMPage(pageIndex);
            if (page != null)
            {
                result = new FlashSettings();
                result.ParseBuffer(this.Device.ChipType, offset, page);
            }
            return result;
        }

        public WriteRfOptionsStatuses WriteSRAMRfOptions(IFlashSettings rfSettings)
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession("Write SRAM Options"));
            WriteRfOptionsStatuses result = WriteRfOptionsStatuses.NoErrors;
            int pageIndex = (int)(this.SramSize / (uint)Constants.SRAM_PAGE_SIZE - 1);	//RF Settings are in last page.
            uint offset = (uint)this.Device.Flash.FlashSize - (uint)Constants.SRAM_PAGE_SIZE;	//we need Absolute OTP
            // address here, because SRAM emulates the part of OTP in Development mode.
            if (rfSettings != null)
            {
                byte[] page = this.ReadSRAMPage(pageIndex);
                if (page != null)
                {
                    rfSettings.StoreToBuffer(this.Device.ChipType, offset, page);
                    if (!WriteSRAMPage(pageIndex, page, true))
                    {
                        result = WriteRfOptionsStatuses.CantWriteAppRfSettings;
                    }
                }
                else
                {
                    result = WriteRfOptionsStatuses.CantReadAppRfSettings;
                }
            }
            else
            {
                result = WriteRfOptionsStatuses.UndefinedRfSettings;
            }
            return result;
        }

        public bool WriteEEPROM(byte[] eepromDataRaw)
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0}", "Write EEPROM")));
            return WriteEEPROM(0, eepromDataRaw, Constants.EEPROM_PAGE_SIZE);
        }

        public bool WriteEEPROM(int startAddress, byte[] eepromDataRaw, ushort pageSize)
        {
            ushort index;
            byte[] data = new byte[pageSize];
            if (pageSize == 1)
            {
                while (startAddress < eepromDataRaw.Length)
                {
                    if (!WriteEEPByte(startAddress, eepromDataRaw[startAddress++]))
                        return false;
                }
            }
            while (startAddress < eepromDataRaw.Length)
            {
                for (index = 0; index < pageSize; index++)
                {
                    if (startAddress + index >= eepromDataRaw.Length)
                        break;
                    data[index] = eepromDataRaw[startAddress + index];
                }
                if (index < pageSize)
                {
                    byte[] wData = new byte[index];
                    for (int i = 0; i < wData.Length; i++)
                        wData[i] = data[i];
                    if (!WriteEEP(startAddress, wData))
                    {
                        return false;
                    }
                    else
                    {
                        if (ProgressChanged != null)
                        {
                            ProgressChanged(new ProgressChangedEventArgs(ProgressStatuses.Write, wData.Length * (startAddress / pageSize), eepromDataRaw.Length));
                        }
                    }
                }
                else
                {
                    if (!WriteEEP(startAddress, data))
                    {
                        return false;
                    }
                    else
                    {
                        if (ProgressChanged != null)
                        {
                            ProgressChanged(new ProgressChangedEventArgs(ProgressStatuses.Write, data.Length * (startAddress / pageSize), eepromDataRaw.Length));
                        }
                    }
                }

                startAddress += pageSize;
            }
            return true;
        }

        private bool WriteEEP(int startAddress, byte[] wData)
        {
            bool result = false;
            try
            {
                byte[] request = new byte[wData.Length + 5];
                int i = 0;
                byte seqNum = mSessionLayer.SequenceNumber;
                request[i++] = (byte)((startAddress >> 16) & 0xFF);
                request[i++] = (byte)((startAddress >> 8) & 0xFF);
                request[i++] = (byte)(startAddress & 0xFF);
                request[i++] = (byte)(wData.Length & 0xFF);

                for (int j = 0; j < wData.Length; j++)
                    request[i++] = wData[j];
                request[i++] = seqNum;

                byte[] resp = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_MEMORY_PUT_BUFFER, request);
                if (resp.Length < 2)
                    return false;
                return (resp[0] == (byte)FlashProgrammingStatuses.DONE
                    && resp[1] == seqNum);
            }
            catch
            {
                result = false;
            }
            return result;
        }

        private bool WriteEEPByte(int startAddress, byte byteData)
        {
            bool result = true;
            try
            {
                mSessionLayer.ExecuteNonRequest((byte)ProgrammerCommandTypes.FUNC_ID_MEMORY_PUT_BYTE, new byte[] { 0x00, (byte)(startAddress >> 8), (byte)(startAddress & 0xFF), byteData, mSessionLayer.SequenceNumber });
            }
            catch
            {
                result = false;
            }
            return result;
        }
        private uint mEepromSize = (uint)Zensys.ZWave.Enums.EepromSize.EEPROM_CS_SIZE_ZERO;
        public uint EepromSize
        {
            get
            {
                if (mEepromSize == (uint)Zensys.ZWave.Enums.EepromSize.EEPROM_CS_SIZE_ZERO)
                {
                    if (this.Device.ChipType != (byte)ChipTypes.ZW040x)
                        return (uint)Zensys.ZWave.Enums.EepromSize.EEPROM_CS_SIZE_ZW010x_ZW030x;
                    else
                        return (uint)Zensys.ZWave.Enums.EepromSize.EEPROM_CS_SIZE_ZW040x;
                }
                else
                {
                    return mEepromSize;
                }
            }
            set
            {
                mEepromSize = value;
            }
        }

        public byte[] ReadHomeId()
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0}", "Read Home Id")));
            byte[] result = null;
            byte[] response = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_MEMORY_GET_ID, new byte[] { 0, mSessionLayer.SequenceNumber });
            if (response.Length == 5)
            {
                List<byte> tmp = new List<byte>(response);
                tmp.RemoveAt(tmp.Count - 1);
                result = tmp.ToArray();
            }
            return result;
        }

        public bool WriteHomeId(byte[] homeId)
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0}", "Write Home Id")));
            bool result = true;
            List<byte> request = new List<byte>();
            request.Add(0);
            for (int i = (int)homeId.Length - (int)4; i < (int)homeId.Length; i++)
            {
                if (i < 0)
                {
                    request.Add(0);
                }
                else
                {
                    request.Add(homeId[i]);
                }
            }
            request.Add(mSessionLayer.SequenceNumber);
            byte[] response = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_STORE_HOMEID, request.ToArray());
            return result;
        }




        public uint MtpSize
        {
            get
            {
                if (this.Device.ChipType != (byte)ChipTypes.ZW040x)
                    return 0;
                else
                    return 64;			//TODO:  get from Constants. ??
            }
        }

        public bool SwitchMTP(bool state)
        {
            bool result = false;
            try
            {
                //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0} ({1})", "Switch MTP", state.ToString())));
                byte[] response;
                byte seqNum = mSessionLayer.SequenceNumber;
                if (state)
                {
                    response = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_TOGGLE_MTP_IF,
                        new byte[] { 0x01, this.Device.ChipType, seqNum });
                }
                else
                {
                    response = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_TOGGLE_MTP_IF,
                        new byte[] { 0x00, this.Device.ChipType, seqNum });
                }

                if (response.Length == 2)
                {
                    result = (response[0] == (byte)FlashProgrammingStatuses.SUCCESS && response[1] == seqNum);
                }
            }
            catch
            {
                result = false;
            }
            return result;
        }

        public bool FillMTP(byte fillValue)
        {
            bool result = false;
            try
            {
                //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0} ({1})", "Fill MTP with value", fillValue.ToString())));
                byte seqNum = mSessionLayer.SequenceNumber;
                byte[] response = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_MTP_FILL,
                    new byte[] { this.Device.ChipType, fillValue, seqNum });
                if (response.Length == 2)
                {
                    result = (response[0] == (byte)FlashProgrammingStatuses.SUCCESS && response[1] == seqNum);
                }
            }
            catch
            {
                result = false;
            }
            return result;
        }

        public bool ReadMTP()
        {
            bool res = true;
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0}", "Read MTP")));
            List<byte> tempBuffer = new List<byte>();

            for (int paramPageIndex = 0; paramPageIndex < this.MtpSize / Constants.MTP_PAGE_SIZE; paramPageIndex++)
            {
                byte seqNum = mSessionLayer.SequenceNumber;
                byte[] response = mSessionLayer.ExecuteRequest(
                    (byte)ProgrammerCommandTypes.FUNC_ID_MTP_READ_PAGE,
                    new byte[] { (byte)this.Device.ChipType, (byte)paramPageIndex, seqNum });
                if (response != null)
                {
                    if (response.Length == Constants.MTP_PAGE_SIZE + 1)
                    {
                        if (response[Constants.MTP_PAGE_SIZE] == seqNum)
                        {
                            tempBuffer.AddRange(response);
                            tempBuffer.RemoveAt(tempBuffer.Count - 1);
                        }
                        else
                        {
                            res = false;
                            break;
                        }
                    }
                    else
                    {
                        res = false;
                        break;
                    }
                }
                else
                {
                    res = false;
                    break;
                }
            }

            if (tempBuffer.Count > 0 && res)
            {
                this.Buffer = tempBuffer.ToArray();
            }
            else
            {
                this.Buffer = null;
            }
            return res;
        }

        public bool WriteMTP(byte[] mtpDataRaw, bool verify)
        {
            //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0}", "Write MTP")));
            return WriteMTPByPages(mtpDataRaw,
                (int)(this.MtpSize / Constants.MTP_PAGE_SIZE), (int)Constants.MTP_PAGE_SIZE, verify);
        }

        private bool WriteMTPByPages(byte[] mtpDataRaw, int pagesCount, int pageSize, bool verify)
        {
            bool result = true;
            for (int pageIndex = 0; pageIndex < pagesCount; pageIndex++)
            {
                //mSessionLayer.LogDataSource.AddTopSession(new LogSession(String.Format("{0} ({1})", "MTP Write Page", pageIndex.ToString())));
                byte[] pageBuffer = new byte[pageSize];
                for (int j = 0; j < pageSize; j++)
                {
                    pageBuffer[j] = mtpDataRaw[pageIndex * pageSize + j];
                }
                if (!WriteMTPPage(pageIndex, pageBuffer, verify))
                {
                    result = false;
                }
            }
            return result;
        }

        private bool WriteMTPPage(int pageIndex, byte[] pageBuffer, bool verify)
        {
            bool result = false;
            ProgrammerCommandTypes cmd;
            byte[] request = new byte[4 + pageBuffer.Length];
            int i = 0;
            cmd = ProgrammerCommandTypes.FUNC_ID_MTP_WRITE_PAGE;
            request[i++] = (byte)this.Device.ChipType;
            request[i++] = (byte)pageIndex;
            request[i++] = (byte)(verify ? 1 : 0);
            for (int j = 0; j < pageBuffer.Length; j++)
                request[i++] = pageBuffer[j];
            byte seqNum;
            request[i++] = seqNum = mSessionLayer.SequenceNumber;
            byte[] response = mSessionLayer.ExecuteRequest((byte)cmd, request);
            if (response.Length == 2)
                result = (response[0] == (byte)FlashProgrammingStatuses.SUCCESS && response[1] == seqNum);
            return result;
        }

        #endregion
    }
}
