using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Devices;
using Zensys.ZWave.Enums;
using System.Threading;
using Zensys.Framework;
using Zensys.ZWave.Logging;
using Zensys.ZWave.Events;
using Zensys.ZWave.Exceptions;

namespace Zensys.ZWave.SerialPortApplication.Devices
{
    public class Controller : Device, IController
    {
        public event UnsolititedFrameReceivedEventHandler UnsolititedFrameReceived;
        public event ControllerUpdatedEventHandler ControllerUpdated;
        public event NodeStatusChangedEventHandler NodeStatusChanged;
        public event FailedNodeStatusChangedEventHandler FailedNodeStatusChanged;
        public event LearnModeStatusChangedEventHandler LearnModeStatusChanged;

        protected Controller()
        {
        }

        public Controller(ISessionLayer sessionLayer, bool useExternalCommandClassesStorage)
            : base(sessionLayer)
        {
            if (sessionLayer != null)
            {
                //mAssociations = new Association(this);
                mCommandClassesStore = new CommandClassesStore(this, useExternalCommandClassesStorage);
            }
        }

        #region IController Members

        #region Properties

        private byte mSucNodeId;
        public byte SucNodeId
        {
            get
            {
                return mSucNodeId;
            }
            set
            {
                mSucNodeId = value;
            }
        }

        private List<IDeviceInfo> mIncludedNodes = new List<IDeviceInfo>();
        public List<IDeviceInfo> IncludedNodes
        {
            get
            {
                return mIncludedNodes;
            }
            set
            {
                mIncludedNodes = value;
            }
        }

        //private IAssociation mAssociations;
        //public IAssociation Associations
        //{
        //    get
        //    {
        //        return mAssociations;
        //    }
        //    set
        //    {
        //        mAssociations = value;
        //    }
        //}

        private CommandClassesStore mCommandClassesStore;
        public CommandClassesStore CommandClassesStore
        {
            get
            {
                return mCommandClassesStore;
            }
            set
            {
                mCommandClassesStore = value;
            }
        }

        public ControllerRoles NetworkRole
        {
            get
            {
                ControllerRoles result = ControllerRoles.None;
                if ((Capability & (byte)ControllerCapabilities.IS_SECONDARY) != 0)
                {
                    result |= ControllerRoles.Secondary;
                }
                if ((Capability & (byte)ControllerCapabilities.IS_SUC) != 0)
                {
                    result |= ControllerRoles.SUC;
                }
                if ((Capability & (byte)ControllerCapabilities.ON_OTHER_NETWORK) != 0)
                {
                    result |= ControllerRoles.OtherNetwork;
                }
                if ((Capability & (byte)ControllerCapabilities.IS_REAL_PRIMARY) != 0)
                {
                    result |= ControllerRoles.RealPrimary;
                }
                if ((Capability & (byte)ControllerCapabilities.NODEID_SERVER_PRESENT) != 0)
                {
                    result |= ControllerRoles.NodeIdServerPresent;
                }
                if ((result & ControllerRoles.SUC) != 0 && (result & ControllerRoles.NodeIdServerPresent) != 0)
                {
                    result |= ControllerRoles.SIS;
                }
                if ((result & ControllerRoles.SIS) == 0 && (result & ControllerRoles.SUC) == 0 && (result & ControllerRoles.NodeIdServerPresent) != 0)
                {
                    result |= ControllerRoles.Inclusion;
                }
                return result;
            }
        }

        #endregion

        /// <summary>
        /// Gets the protocol info.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="checkIfVirtual">if set to <c>true</c> [check if virtual].</param>
        /// <returns></returns>
        /// <remarks>
        /// Request:
        /// SOF|Length|Command|FrameType|NodeId|CheckSum
        /// 01  04     41      00        xx     xx
        /// Request:
        /// SOF|Length|Command|FrameType|NodeIInfo|CheckSum
        /// 01  xx     41      01        xx        xx
        /// </remarks>
        public IDeviceInfo GetProtocolInfo(byte nodeId, bool checkIfVirtual)
        {
            LogTopSession();

            IDeviceInfo result = null;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveGetNodeProtocolInfo, true))
            {
                result = new DeviceInfo();
                byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveGetNodeProtocolInfo, new byte[] { nodeId });
                if (response != null)
                {
                    if (response.Length > 5)
                    {
                        result.Id = nodeId;
                        result.Capability = response[0];
                        result.Security = response[1];
                        result.IsSlaveApi = (result.Security & 0x02) == 0;
                        result.Reserved = response[2];
                        result.Basic = response[3];
                        result.Generic = response[4];
                        result.Specific = response[5];
                    }
                    result.HomeId = this.HomeId;
                    if (checkIfVirtual)
                    {
                        if (this.Version.Library == Libraries.ControllerBridgeLib)
                        {
                            result.IsVirtual = this.IsVirtualNode(nodeId);
                        }
                        else
                        {
                            result.IsVirtual = false;
                        }

                    }
                }
            }
            return result;
        }

        public bool IsVirtualNode(byte nodeId)
        {
            LogTopSession();
            bool ret = false;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveIsVirtualNode, true))
            {
                byte[] res = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveIsVirtualNode, new byte[] { nodeId });
                ret = res[0] != 0x00;
            }
            return ret;
        }

        public IDeviceInfo AddNodeToNetworkWide()
        {
            LogTopSession();
            addedNode = null;
            waitForEvent = true;
            ResetSignalResponseReceived();
            if (this.IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveAddNodeToNetwork, true))
            {
                byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveAddNodeToNetwork, new byte[] { (byte)Mode.NodeAny | (byte)Mode.NodeOptionHighPower | (byte)Mode.NodeOptionNetworkWide, SessionLayer.SequenceNumber });
            }
            WaitOneSignalResponseReceived();
            waitForEvent = false;
            return addedNode;
        }

        public IDeviceInfo AddNodeToNetwork()
        {
            return AddNodeToNetwork(Mode.NodeAny);
        }

        public IDeviceInfo AddNodeToNetwork(Mode mode)
        {
            LogTopSession();
            addedNode = null;
            waitForEvent = true;
            ResetSignalResponseReceived();
            if (this.IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveAddNodeToNetwork, true))
            {
                byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveAddNodeToNetwork, new byte[] { (byte)(mode | Mode.NodeOptionHighPower), SessionLayer.SequenceNumber });
            }
            WaitOneSignalResponseReceived();
            waitForEvent = false;
            return addedNode;
        }

        public IDeviceInfo RemoveNodeFromNetwork()
        {
            LogTopSession();
            waitForEvent = true;
            ResetSignalResponseReceived();
            if (this.IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveRemoveNodeFromNetwork, true))
            {
                byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveRemoveNodeFromNetwork, new byte[] { (byte)Mode.NodeAny, SessionLayer.SequenceNumber });
            }
            WaitOneSignalResponseReceived();
            waitForEvent = false;
            return removedNode;
        }
        private bool isNodeInfoRequested = false;
        public IDeviceInfo RequestNodeInfo(byte nodeId)
        {
            LogTopSession();
            waitForEvent = true;
            isNodeInfoRequested = true;
            ResetSignalResponseReceived();
            if (this.IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveRequestNodeInfo, true))
            {
                byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveRequestNodeInfo, new byte[] { nodeId });
            }
            WaitOneSignalResponseReceived();
            waitForEvent = false;
            return requestedNode;
        }
        public void ControllerChange(ControllerChangeMode controllerChangeMode)
        {
            LogTopSession();
            waitForEvent = true;
            ResetSignalResponseReceived();
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveControllerChange, true))
            {
                byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveControllerChange, new byte[] { (byte)controllerChangeMode, SessionLayer.SequenceNumber });
            }
            WaitOneSignalResponseReceived();
            waitForEvent = false;
        }
        protected LearnMode learnModeResult = LearnMode.Unknown;
        protected SlaveLearnMode learnModeSlaveResult = SlaveLearnMode.VirtualSlaveLearnModeDisable;
        override public LearnMode SetLearnMode(bool learnMode)
        {
            LogTopSession();
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSetLearnMode, true))
            {
                if (learnMode)
                {
                    waitForEvent = true;
                    ResetSignalResponseReceived();
                    SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveSetLearnMode, new byte[] { Convert.ToByte(learnMode), SessionLayer.SequenceNumber });
                    WaitOneSignalResponseReceived();
                }
                else
                {
                    SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveSetLearnMode, new byte[] { Convert.ToByte(learnMode) });
                }
            }

            waitForEvent = false;
            return learnModeResult;
        }

        public void GetControllerCapabilities()
        {
            LogTopSession();
            byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveGetControllerCapabilities, null);
            this.Capability = response[0];
        }
        /// <summary>
        /// Replications the send.
        /// </summary>
        /// <param name="destNodeID">The dest node ID.</param>
        /// <param name="data">The data.</param>
        /// <param name="txOptions">The tx options.</param>
        /// <param name="responseCount">The response count.</param>
        /// <returns></returns>
        public TransmitStatuses ReplicationSend(byte destNodeID, byte[] data, TransmitOptions txOptions)
        {
            LogTopSession();
            IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveReplicationSendData, true);

            byte[] request = new byte[data.Length + 4];
            request[0] = destNodeID;
            request[1] = (byte)data.Length;
            for (int i = 0; i < data.Length; i++)
            {
                request[i + 2] = data[i];
            }
            request[request.Length - 2] = (byte)txOptions;
            request[request.Length - 1] = SessionLayer.SequenceNumber;

            List<byte[]> responses = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveReplicationSendData, 2, (int)SessionLayer.RequestTimeout.TotalMilliseconds, request);

            if (responses.Count == 2 && responses[1].Length > 1)
            {
                return (TransmitStatuses)responses[1][1];
            }
            else
            {
                return TransmitStatuses.ResMissing;
            }
        }

        public void ReplicationReceiveComplete()
        {
            LogTopSession();
            SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveReplicationCommandComplete);
        }

        public TransmitStatuses AssignReturnRoute(byte srcNodeId, byte destNodeId)
        {
            LogTopSession();
            TransmitStatuses result;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveAssignReturnRoute, true))
            {
                List<byte[]> responses = SessionLayer.ExecuteRequest(
                    (byte)CommandTypes.CmdZWaveAssignReturnRoute,
                    2,
                    30000, //TO# 03150 "Assign Return Route failed to node replaced 5 or 4 hops away"
                    new byte[] { srcNodeId, destNodeId, SessionLayer.SequenceNumber });
                if (responses.Count == 2 && responses[1] != null && responses[1].Length > 1)
                {
                    result = (TransmitStatuses)responses[1][1];
                }
                else
                {
                    throw new InvalidResponseSerialApiException();
                }
            }
            else
            {
                throw new NotSupportedCommandSerialApiException();
            }
            return result;
        }

        public void CreateNewPrimaryCtrl(CreateNewPrimaryControllerMode mode)
        {
            LogTopSession();
            waitForEvent = true;
            ResetSignalResponseReceived();
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveCreateNewPrimary, true))
            {
                SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveCreateNewPrimary, new byte[] { (byte)mode, SessionLayer.SequenceNumber });
            }
            WaitOneSignalResponseReceived();
            waitForEvent = false;
        }

        public TransmitStatuses DeleteReturnRoute(byte srcNodeId)
        {
            LogTopSession();
            TransmitStatuses result;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveDeleteReturnRoute, true))
            {
                List<byte[]> responses = SessionLayer.ExecuteRequest(
                    (byte)CommandTypes.CmdZWaveDeleteReturnRoute,
                    2,
                    30000, //TO# 03150 "Assign Return Route failed to node replaced 5 or 4 hops away"
                    new byte[] { srcNodeId, SessionLayer.SequenceNumber });
                if (responses.Count == 2 && responses[1] != null && responses[1].Length > 1)
                {
                    result = (TransmitStatuses)responses[1][1];
                }
                else
                {
                    throw new InvalidResponseSerialApiException();
                }
            }
            else
            {
                throw new NotSupportedCommandSerialApiException();
            }
            return result;
        }

        public FailedNodeStatus RemoveFailedNodeID(byte nodeId)
        {
            LogTopSession();
            FailedNodeStatus result = FailedNodeStatus.FailedNodeNotRemoved;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveRemoveFailedNodeId, true))
            {
                List<byte[]> responses = SessionLayer.ExecuteRequest(
                    (byte)CommandTypes.CmdZWaveRemoveFailedNodeId,
                    2,
                    (int)SessionLayer.RequestTimeout.TotalMilliseconds,
                    new byte[] { nodeId, SessionLayer.SequenceNumber });
                if (responses.Count == 2 && responses[1] != null && responses[1].Length > 1)
                {
                    result = (FailedNodeStatus)responses[1][1];
                }
            }
            return result;
        }
        public bool IsFailedNode(byte nodeId)
        {
            LogTopSession();
            bool result = false;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveIsFailedNode, true))
            {
                byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveIsFailedNode, new byte[] { nodeId });
                if (response != null && response.Length > 0)
                {
                    if (response[0] == 0x01)
                    {
                        result = true;
                    }
                }
            }
            return result;
        }
        public void IsPrimaryController()
        {
            throw new Exception("The method or operation is not implemented.");
        }
        public IDeviceInfo ReplaceFailedNode(byte nodeId)
        {
            LogTopSession();
            waitForEvent = true;
            addedNode = new DeviceInfo();
            addedNode.Id = nodeId;
            ResetSignalResponseReceived();
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveReplaceFailedNode, true))
            {
                byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveReplaceFailedNode, new byte[] { nodeId, SessionLayer.SequenceNumber });
            }
            WaitOneSignalResponseReceived();
            waitForEvent = false;
            return addedNode;
        }
        public void GetNeighborCount()
        {
            throw new Exception("The method or operation is not implemented.");
        }
        public void AreNodesNeighbours()
        {
            throw new Exception("The method or operation is not implemented.");
        }
        private RequestNeighborUpdateStatuses mRequestNodeNeighborUpdateResult = RequestNeighborUpdateStatuses.None;
        public RequestNeighborUpdateStatuses RequestNodeNeighborUpdate(byte nodeId)
        {
            LogTopSession();
            ResetSignalResponseReceived();
            byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveRequestNodeNeighborUpdate, new byte[] { nodeId, SessionLayer.SequenceNumber });
            WaitOneSignalResponseReceived();
            return mRequestNodeNeighborUpdateResult;
        }
        public byte[] RequestNodeNeighborUpdateMR(byte nodeId, byte[] nodeMask)
        {
            LogTopSession();
            byte[] result = new byte[29];
            byte[] request = new byte[31];
            request[0] = nodeId;
            for (int i = 1; i < request.Length - 1; i++)
            {
                if (i <= nodeMask.Length)
                {
                    request[i] = nodeMask[i - 1];
                }
                else //Make sure that nodeMask is 29 bytes long
                {
                    request[i] = 0;
                }
            }
            request[30] = SessionLayer.SequenceNumber;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveRequestNodeNeighborUpdateMR, true))
            {
                List<byte[]> responses = SessionLayer.ExecuteRequest(
                        (byte)CommandTypes.CmdZWaveRequestNodeNeighborUpdateMR,
                        3,
                        700 * 232,
                        request);
                if (responses.Count == 3 && responses[2] != null && responses[2].Length > 1)
                {
                    if ((RequestNeighborUpdateStatuses)responses[2][0] == RequestNeighborUpdateStatuses.RequestNeighborUpdateDone)
                    {
                        byte[] returnedFromModule = responses[2];
                        for (int i = 0; i < result.Length; i++)
                        {
                            if (i < (returnedFromModule.Length - 1))
                            {
                                result[i] = returnedFromModule[i + 1];
                            }
                        }
                    }
                }
                else
                {
                    throw new InvalidResponseSerialApiException();
                }
            }
            else
            {
                throw new NotSupportedCommandSerialApiException();
            }
            return result;
        }
        public byte[] GetRoutingInfo(byte nodeId, byte removeBad, byte removeNonReps)
        {
            LogTopSession();
            byte[] response = null;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdGetRoutingTableLine, true))
            {
                response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdGetRoutingTableLine, new byte[] { nodeId, removeBad, removeNonReps, SessionLayer.SequenceNumber });
            }
            else
            {
                throw new NotSupportedCommandSerialApiException();
            }
            return response;
        }

        public bool SetRoutingInfo(byte nodeId, byte[] nodeMask)
        {
            LogTopSession();
            bool ret = false;
            byte[] request = new byte[30];
            request[0] = nodeId;
            for (int i = 1; i < request.Length; i++)
            {
                if (i <= nodeMask.Length)
                {
                    request[i] = nodeMask[i - 1];
                }
                else //Make sure that nodeMask is 29 bytes long
                {
                    request[i] = 0;
                }
            }
            byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSetRoutingInfo, request);
            if (response.Length > 0)
            {
                if ((TransmitStatuses)response[0] == TransmitStatuses.CompleteOk)
                {
                    ret = true;
                }
            }
            return ret;
        }

        public void GetSUCNodeID()
        {
            LogTopSession();
            byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveGetSucNodeId, null);
            this.SucNodeId = response[0];
        }
        public SetSucReturnValue SetSUCNodeID(byte nodeId, bool sucState, TransmitOptions txOptions, byte capabilities)
        {
            LogTopSession();
            SetSucReturnValue result = SetSucReturnValue.SucUndefined;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSetSucNodeId, true))
            {
                if (this.Id != nodeId)
                {
                    List<byte[]> responses = SessionLayer.ExecuteRequest(
                        (byte)CommandTypes.CmdZWaveSetSucNodeId,
                        2,
                        (int)SessionLayer.RequestTimeout.TotalMilliseconds,
                        new byte[] { nodeId, Convert.ToByte(sucState), (byte)txOptions, capabilities, SessionLayer.SequenceNumber });
                    if (responses.Count == 2 && responses[1] != null && responses[1].Length > 1)
                    {
                        result = (SetSucReturnValue)responses[1][1];
                    }
                }
                else
                {
                    byte[] response = SessionLayer.ExecuteRequest(
                            (byte)CommandTypes.CmdZWaveSetSucNodeId,
                            new byte[] { nodeId, Convert.ToByte(sucState), (byte)txOptions, capabilities, SessionLayer.SequenceNumber });
                    if (response != null && response.Length > 0 && response[0] == 0x01)
                    {
                        result = SetSucReturnValue.SucSetSucceeded;
                    }
                }
            }
            return result;
        }
        public TransmitStatuses SendSUCID(byte nodeId, TransmitOptions txOptions)
        {
            LogTopSession();
            IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendSucId, true);
            List<byte[]> responses = SessionLayer.ExecuteRequest(
                (byte)CommandTypes.CmdZWaveSendSucId,
                2,
                (int)SessionLayer.RequestTimeout.TotalMilliseconds,
                new byte[] { nodeId, (byte)txOptions, SessionLayer.SequenceNumber });
            if (responses.Count == 2 && responses[1].Length > 1)
            {
                return (TransmitStatuses)responses[1][1];
            }
            else
            {
                return TransmitStatuses.ResMissing;
            }
        }
        public TransmitStatuses AssignSUCReturnRoute(byte srcNodeId)
        {
            LogTopSession();
            IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveAssignSucReturnRoute, true);
            List<byte[]> responses = SessionLayer.ExecuteRequest(
                (byte)CommandTypes.CmdZWaveAssignSucReturnRoute,
                2,
                30000, //TO# 03150 "Assign Return Route failed to node replaced 5 or 4 hops away"
                new byte[] { srcNodeId, SessionLayer.SequenceNumber });
            if (responses.Count == 2 && responses[1].Length > 1)
            {
                return (TransmitStatuses)responses[1][1];
            }
            else
            {
                return TransmitStatuses.ResMissing;
            }
        }

        public TransmitStatuses DeleteSUCReturnRoute(byte nodeId)
        {
            LogTopSession();
            IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveDeleteSucReturnRoute, true);
            List<byte[]> responses = SessionLayer.ExecuteRequest(
                (byte)CommandTypes.CmdZWaveDeleteSucReturnRoute,
                2,
                30000, //TO# 03150 "Assign Return Route failed to node replaced 5 or 4 hops away"
                new byte[] { nodeId, SessionLayer.SequenceNumber });
            if (responses.Count == 2 && responses[1].Length > 1)
            {
                return (TransmitStatuses)responses[1][1];
            }
            else
            {
                return TransmitStatuses.ResMissing;
            }
        }
        public List<IDeviceInfo> GetNodes()
        {
            this.IncludedNodes = new List<IDeviceInfo>();
            byte[] initDataResponse = this.SerialApiInitData();
            if (!this.IsSlaveApi)
            {
                NodeMask virtualNodeMask = GetVirtualNodes();
                byte nodeIdx = 0;
                byte length = initDataResponse[2];

                for (int i = 0; i < length; i++)
                {
                    byte availabilityMask = initDataResponse[3 + i];
                    for (byte bit = 0; bit < 8; bit++)
                    {
                        nodeIdx++;
                        if ((availabilityMask & (1 << bit)) > 0)
                        {
                            IDeviceInfo node = GetProtocolInfo(nodeIdx, false);
                            node.SupportedCommandClasses = this.CommandClassesStore.CmdClasses(node.Id);
                            node.IsVirtual = virtualNodeMask.Get(nodeIdx);
                            IncludedNodes.Add(node);
                        }
                    }
                }

            }
            return this.IncludedNodes;
        }

        private NodeMask GetVirtualNodes()
        {
            LogTopSession();
            NodeMask nodeMask;
            if (Version.Library == Libraries.ControllerBridgeLib)
            {
                byte[] initDataResponse = null;
                if (this.IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveGetVirtualNodes, true))
                {
                    initDataResponse = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveGetVirtualNodes, null);
                }
                nodeMask = new NodeMask(initDataResponse);
            }
            else
            {
                nodeMask = new NodeMask();
            }
            return nodeMask;
        }
        public void StopRequest(byte commandType)
        {
            LogTopSession();
            SetSignalResponseReceived();
            switch (commandType)
            {
                case (byte)CommandTypes.CmdZWaveSetLearnMode:
                case (byte)CommandTypes.CmdZWaveControllerChange:
                    {
                        SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveControllerChange, new byte[] { (byte)Mode.NodeStop, 0x00 });
                        break;
                    }
                case (byte)CommandTypes.CmdZWaveAddNodeToNetwork:
                    {
                        SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveAddNodeToNetwork, new byte[] { (byte)Mode.NodeStop, 0x00 });
                        break;
                    }
                case (byte)CommandTypes.CmdZWaveRemoveNodeFromNetwork:
                    {
                        SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveRemoveNodeFromNetwork, new byte[] { (byte)Mode.NodeStop, 0x00 });
                        break;
                    }
                case (byte)CommandTypes.CmdZWaveCreateNewPrimary:
                    {
                        SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveCreateNewPrimary, new byte[] { (byte)Mode.NodeStop, 0x00 });
                        break;
                    }
                case (byte)CommandTypes.CmdZWaveSendData:
                    {
                        SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveSendDataAbort);
                        break;
                    }
            }
        }
        public override void SetDefault()
        {
            LogTopSession();
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSetDefault, true))
            {
                try
                {
                    SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSetDefault, new byte[] { SessionLayer.SequenceNumber });
                }
                catch (Exception ex)
                {
                    Tools._writeDebugDiagnosticMessage(ex.Message, true, true);
                }
                this.IncludedNodes.Clear();
            }
        }

        public bool SetSelfAsSUC(bool sucState, byte capabilities)
        {
            LogTopSession();
            bool result = false;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSetSucNodeId, true))
            {
                if (this.SetSUCNodeID(this.Id, sucState, 0, capabilities) == SetSucReturnValue.SucSetSucceeded)
                {
                    this.GetControllerCapabilities();
                    result = true;
                }
            }
            return result;

        }
        public bool EnableSUC(bool enable, byte capabilities)
        {
            LogTopSession();
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveEnableSuc, true))
            {
                byte[] response = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveEnableSuc, new byte[] { Convert.ToByte(enable), capabilities });
                if (response.Length > 0)
                {
                    if (response[0] != 0)
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        protected long NodeStopSendCounter = 0;
        protected bool waitForEvent = false;
        private IDeviceInfo addedNode = null;
        private IDeviceInfo requestedNode = null;
        private IDeviceInfo removedNode = null;
        protected byte[] slaveLearnModeResult = null;
        private bool isNodeAlreadyIncluded = false;

        protected override void OnSessionLayerResponseReceivedCustomActions(ResponseReceivedEventArgs args)
        {
            switch (args.CommandType)
            {
                #region CmdZWaveControllerChange | CmdZWaveCreateNewPrimary | CmdZWaveAddNodeToNetwork
                case ((byte)CommandTypes.CmdZWaveControllerChange):
                    {
                        byte nid = args.Data[2];
                        NodeStatuses nodeStatus = (NodeStatuses)args.Data[1];
                        if (NodeStatusChanged != null)
                        {
                            NodeStatusChanged(new NodeStatusChangedEventArgs(args.Data, nid, (CommandTypes)args.CommandType, nodeStatus, addedNode));
                        }
                        if (nodeStatus == NodeStatuses.ProtocolDone)
                        {
                            SetSignalResponseReceived();
                        }
                    } break;
                case ((byte)CommandTypes.CmdZWaveCreateNewPrimary):
                case ((byte)CommandTypes.CmdZWaveAddNodeToNetwork):
                    {
                        byte nid = args.Data[2];
                        NodeStatuses nodeStatus = (NodeStatuses)args.Data[1];
                        if (NodeStatusChanged != null)
                        {
                            NodeStatusChanged(new NodeStatusChangedEventArgs(args.Data, nid, (CommandTypes)args.CommandType, nodeStatus, addedNode));
                        }
                        switch (nodeStatus)
                        {
                            #region NodeFound
                            case NodeStatuses.NodeFound:
                                {
                                    if (nid != 0) isNodeAlreadyIncluded = true;
                                    else isNodeAlreadyIncluded = false;
                                } break;
                            #endregion
                            #region AddingRemovingSlave | AddingRemovingController
                            case NodeStatuses.AddingRemovingSlave:
                            case NodeStatuses.AddingRemovingController:
                                {
                                    if (!isNodeAlreadyIncluded)
                                    {
                                        if (args.Data.Length >= 4)
                                        {
                                            if (args.Data[3] >= 3)// verify len parameter in payload
                                            {
                                                addedNode = new DeviceInfo();
                                                addedNode.Id = nid;
                                                addedNode.Capability = 0;
                                                addedNode.Security = 0;
                                                addedNode.Reserved = 0;
                                                addedNode.Basic = args.Data[4];
                                                addedNode.Generic = args.Data[5];
                                                addedNode.Specific = args.Data[6];

                                                if (nodeStatus == NodeStatuses.AddingRemovingSlave)
                                                {
                                                    addedNode.IsSlaveApi = true;
                                                }
                                                else
                                                {
                                                    addedNode.IsSlaveApi = false;
                                                }

                                                if ((args.Data.Length - 7) > 0)
                                                {
                                                    addedNode.SupportedCommandClasses = new byte[args.Data.Length - 7];
                                                    for (int i = 0; i < addedNode.SupportedCommandClasses.Length; i++)
                                                    {
                                                        addedNode.SupportedCommandClasses[i] = args.Data[i + 7];
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                                break;
                            #endregion
                            #region Failed
                            case NodeStatuses.Failed:
                                {
                                    NodeAddedDelegate asyncThread = new NodeAddedDelegate(NodeAddedFailed);
                                    asyncThread.BeginInvoke(NodeAddedFailedCallback, true);
                                }
                                break;
                            #endregion
                            #region Done | ProtocolDone
                            case NodeStatuses.Done:
                                {
                                    if (waitForEvent)
                                    {
                                        waitForEvent = false;
                                        NodeAddedDelegate asyncThread = new NodeAddedDelegate(NodeAddedDone);
                                        IAsyncResult res = asyncThread.BeginInvoke(NodeAddedDoneCallback, true);
                                    }
                                }
                                break;
                            case NodeStatuses.ProtocolDone:
                                {
                                    SetSignalResponseReceived();
                                    //NodeAddedDelegate asyncThread = new NodeAddedDelegate(NodeAddedProtocolDone);
                                    //IAsyncResult res = asyncThread.BeginInvoke(NodeAddedProtocolDoneCallback, true);
                                }
                                break;
                            #endregion
                        }
                    }
                    break;
                #endregion
                #region CmdZWaveRemoveNodeFromNetwork
                case ((byte)CommandTypes.CmdZWaveRemoveNodeFromNetwork):
                    {
                        NodeStatuses nodeStatus = (NodeStatuses)args.Data[1];
                        if (NodeStatusChanged != null)
                        {
                            NodeStatusChanged(new NodeStatusChangedEventArgs(args.Data, 0, (CommandTypes)args.CommandType, nodeStatus, removedNode));
                        }
                        switch (nodeStatus)
                        {
                            #region AddingRemovingController | AddingRemovingSlave | LearnReady | NodeFound
                            case NodeStatuses.AddingRemovingController:
                            case NodeStatuses.AddingRemovingSlave:
                            case NodeStatuses.LearnReady:
                            case NodeStatuses.NodeFound:
                                {
                                    NodeStopSendCounter = 0;
                                    byte nid = args.Data[2];
                                    removedNode = new DeviceInfo();
                                    removedNode.Id = nid;
                                    break;
                                }
                            #endregion
                            #region Done
                            case NodeStatuses.Done:
                                {
                                    if (NodeStopSendCounter < 2)
                                    {
                                        NodeStopSendCounter++;
                                        NodeRemovedDelegate asyncThread = new NodeRemovedDelegate(NodeRemovedDone);
                                        IAsyncResult result = asyncThread.BeginInvoke(null, true);
                                        //asyncThread.EndInvoke(result);
                                        SetSignalResponseReceived();
                                    }
                                }
                                break;
                            #endregion
                            #region Failed
                            case NodeStatuses.Failed:
                                {
                                    if (NodeStopSendCounter < 2)
                                    {
                                        NodeStopSendCounter++;
                                        NodeRemovedDelegate asyncThread = new NodeRemovedDelegate(NodeRemovedFailed);
                                        IAsyncResult result = asyncThread.BeginInvoke(null, true);
                                        //asyncThread.EndInvoke(result);
                                        SetSignalResponseReceived();
                                    }
                                }
                                break;
                            #endregion

                        }
                    }
                    break;
                #endregion
                #region CmdApplicationControllerUpdate
                case ((byte)CommandTypes.CmdApplicationControllerUpdate):
                    {
                        byte nId = args.Data[1];
                        ApplicationControllerUpdateStatuses status = (ApplicationControllerUpdateStatuses)args.Data[0];
                        switch (status)
                        {
                            #region ADD_DONE | DELETE_DONE | SUC_ID | NODE_INFO_RECEIVED
                            case ApplicationControllerUpdateStatuses.ADD_DONE:
                                {
                                    IDeviceInfo changedNode = new DeviceInfo();
                                    changedNode.Id = nId;
                                    if (args.Data.Length >= 3)
                                    {
                                        {
                                            changedNode.Capability = 0;
                                            changedNode.Security = 0;
                                            changedNode.Reserved = 0;
                                            changedNode.Basic = args.Data[3];
                                            changedNode.Generic = args.Data[4];
                                            changedNode.Specific = args.Data[5];
                                            if ((args.Data.Length - 6) > 0)
                                            {
                                                changedNode.SupportedCommandClasses = new byte[args.Data.Length - 6];
                                                for (int i = 0; i < changedNode.SupportedCommandClasses.Length; i++)
                                                {
                                                    changedNode.SupportedCommandClasses[i] = args.Data[i + 6];
                                                }
                                            }
                                        }
                                    }
                                    if (ControllerUpdated != null)
                                    {
                                        ControllerUpdated(new ControllerUpdatedEventArgs(changedNode, ApplicationControllerUpdateStatuses.ADD_DONE));
                                    }
                                } break;
                            case ApplicationControllerUpdateStatuses.DELETE_DONE:
                                {
                                    IDeviceInfo changedNode = new DeviceInfo();
                                    changedNode.Id = nId;
                                    if (ControllerUpdated != null)
                                    {
                                        ControllerUpdated(new ControllerUpdatedEventArgs(changedNode, ApplicationControllerUpdateStatuses.DELETE_DONE));
                                    }
                                } break;
                            case ApplicationControllerUpdateStatuses.SUC_ID:
                                {
                                    this.SucNodeId = args.Data[1];
                                    IDeviceInfo changedNode = new DeviceInfo();
                                    changedNode.Id = nId;
                                    if (ControllerUpdated != null)
                                    {
                                        ControllerUpdated(new ControllerUpdatedEventArgs(changedNode, ApplicationControllerUpdateStatuses.SUC_ID));
                                    }
                                } break;
                            case ApplicationControllerUpdateStatuses.NODE_INFO_RECEIVED:
                                {
                                    requestedNode = new DeviceInfo();
                                    if (args.Data.Length >= 4)
                                    {
                                        requestedNode.Id = nId;
                                        requestedNode.Basic = args.Data[3];
                                        requestedNode.Generic = args.Data[4];
                                        requestedNode.Specific = args.Data[5];
                                        requestedNode.SupportedCommandClasses = new byte[] { };
                                        if ((args.Data.Length - 6) > 0)
                                        {
                                            requestedNode.SupportedCommandClasses = new byte[args.Data.Length - 6];
                                            for (int i = 0; i < (args.Data.Length - 6); i++)
                                            {
                                                requestedNode.SupportedCommandClasses[i] = args.Data[i + 6];
                                            }
                                        }
                                    }
                                    if (isNodeInfoRequested)
                                    {
                                        isNodeInfoRequested = false;
                                        SetSignalResponseReceived();
                                    }
                                } break;
                            #endregion
                            #region NODE_INFO_REQ_DONE
                            case ApplicationControllerUpdateStatuses.NODE_INFO_REQ_DONE:
                                {
                                    SetSignalResponseReceived();

                                } break;
                            #endregion
                            #region NODE_INFO_REQ_FAILED
                            case ApplicationControllerUpdateStatuses.NODE_INFO_REQ_FAILED:
                                {
                                    if (isNodeInfoRequested)
                                    {
                                        isNodeInfoRequested = false;
                                        requestedNode = null;
                                        SetSignalResponseReceived();
                                    }
                                } break;
                            #endregion
                            #region ROUTING_PENDING
                            case ApplicationControllerUpdateStatuses.ROUTING_PENDING: { } break;
                            #endregion
                        }
                        #region Old Code
                        //byte nid = payload[1];
                        //switch ((AppCtrlUpdateStatus)payload[0])
                        //{
                        //    // Node requested removed from remote control...
                        //    case AppCtrlUpdateStatus.DELETE_DONE:
                        //        {
                        //            if (nodeTable.contains(nid))
                        //            {
                        //                Node node = nodeTable.get(nid);
                        //                nodeTable.remove(nid);
                        //                RemoveNodeEventArgs e = new RemoveNodeEventArgs(node, cmd);
                        //                if (RemoveNodeEvent != null) RemoveNodeEvent(this, e);
                        //            }
                        //        }
                        //        break;

                        //    case AppCtrlUpdateStatus.ADD_DONE:
                        //        {
                        //            Node node;
                        //            if (libraryType.lib == Library.ControllerBridgeLib)
                        //            {
                        //                node = ZWaveGetNodeProtocolInfo(nid, true);
                        //            }
                        //            else
                        //            {
                        //                node = ZWaveGetNodeProtocolInfo(nid);
                        //            }
                        //            if (payload.Length - 6 > 0)
                        //            {
                        //                node.SupportedCmdClasses = new byte[payload.Length - 6];
                        //                for (int i = 0; i < node.SupportedCmdClasses.Length; i++)
                        //                {
                        //                    node.SupportedCmdClasses[i] = payload[i + 6];
                        //                }
                        //            }
                        //            nodeTable.add(node);
                        //            AddNodeEventArgs e = new AddNodeEventArgs(node, cmd, NodeStatus.Done);
                        //            if (AddNodeEvent != null) AddNodeEvent(this, e);
                        //        }
                        //        break;

                        //    case AppCtrlUpdateStatus.SUC_ID:
                        //        {
                        //            UpdateEventArgs e = new UpdateEventArgs(payload[0], payload[1]);
                        //            if (UpdateEvent != null) UpdateEvent(this, e);
                        //        }
                        //        break;

                        //    case AppCtrlUpdateStatus.NODE_INFO_REQ_FAILED:
                        //        {
                        //            waitForNodeInfocallbackHandler(null);
                        //        }
                        //        break;

                        //    // NODE_INFO_REQ_DONE only used with Serial API 2.16 and below.
                        //    case AppCtrlUpdateStatus.NODE_INFO_REQ_DONE:
                        //        {
                        //            RequestNodeInfo(payload, nid);
                        //        }
                        //        break;

                        //    case AppCtrlUpdateStatus.NODE_INFO_RECEIVED:
                        //        {
                        //            if (_requestNodeInfo)
                        //            {
                        //                RequestNodeInfo(payload, nid);
                        //            }
                        //        }
                        //        break;

                        //    case AppCtrlUpdateStatus.ROUTING_PENDING:
                        //        break;

                        //    default:
                        //        {
                        //            UnknownCommandEventArgs e = new UnknownCommandEventArgs(packet);
                        //            if (UnknownCommandEvent != null) UnknownCommandEvent(this, e);
                        //        }
                        //        break;
                        //}//Case Status


                        #endregion
                    }
                    break;
                #endregion
                #region CmdZWaveRequestNodeInfo
                case ((byte)CommandTypes.CmdZWaveRequestNodeInfo):
                    {

                    }
                    break;
                #endregion
                #region CmdZWaveRequestNodeNeighborUpdate
                case ((byte)CommandTypes.CmdZWaveRequestNodeNeighborUpdate):
                    {
                        if (args.Data.Length == 2)
                        {
                            if (args.Data[1] == (byte)RequestNeighborUpdateStatuses.RequestNeighborUpdateDone)
                            {
                                mRequestNodeNeighborUpdateResult = RequestNeighborUpdateStatuses.RequestNeighborUpdateDone;
                                SetSignalResponseReceived();
                            }
                            if (args.Data[1] == (byte)RequestNeighborUpdateStatuses.RequestNeighborUpdateFailed)
                            {
                                mRequestNodeNeighborUpdateResult = RequestNeighborUpdateStatuses.RequestNeighborUpdateFailed;
                                SetSignalResponseReceived();
                            }
                        }
                        else
                        {
                            SetSignalResponseReceived();
                        }
                    }
                    break;
                #endregion
                #region CmdZWaveSetLearnMode
                case ((byte)CommandTypes.CmdZWaveSetLearnMode):
                    {
                        if (args.Data.Length > 2)
                        {
                            byte nId = args.Data[2];
                            learnModeResult = (LearnMode)args.Data[1];
                            if (LearnModeStatusChanged != null)
                            {
                                LearnModeStatusChanged(new LearnModeStatusChangedEventArgs(args.Data, nId, (CommandTypes)args.CommandType, learnModeResult));
                            }
                            switch (learnModeResult)
                            {
                                //case LearnMode.Started:
                                case LearnMode.Failed:
                                case LearnMode.Done:
                                    {
                                        SetSignalResponseReceived();
                                    }
                                    break;
                            }
                        }
                    }
                    break;
                #endregion

                #region CmdZWaveReplaceFailedNode
                case ((byte)CommandTypes.CmdZWaveReplaceFailedNode):
                    {
                        FailedNodeStatus replaceModeResult;
                        if (args.Data.Length > 1)
                        {
                            replaceModeResult = (FailedNodeStatus)args.Data[1];
                            if (FailedNodeStatusChanged != null)
                            {
                                FailedNodeStatusChanged(new FailedNodeStatusChangedEventArgs(args.Data, 0, (CommandTypes)args.CommandType, replaceModeResult, false));
                            }
                        }
                        else
                        {
                            replaceModeResult = (FailedNodeStatus)args.Data[0];
                            if (FailedNodeStatusChanged != null)
                            {
                                FailedNodeStatusChanged(new FailedNodeStatusChangedEventArgs(args.Data, 0, (CommandTypes)args.CommandType, replaceModeResult, true));
                            }
                        }

                        switch (replaceModeResult)
                        {
                            case FailedNodeStatus.FailedNodeReplaceDone:
                            case FailedNodeStatus.FailedNodeReplaceFailed:
                                {
                                    SetSignalResponseReceived();
                                }
                                break;
                        }
                    } break;
                #endregion
                #region CmdZWaveReplicationSendData
                case ((byte)CommandTypes.CmdZWaveReplicationSendData):
                    {
                    }
                    break;
                #endregion
                #region CmdSerialApiTest
                case ((byte)CommandTypes.CmdSerialApiTest):
                    {
                        #region Old Code
                        //// testcmd, state, nodeid, status, runnr.
                        //if (TestEvent != null)
                        //{
                        //    if (payload.Length >= 6)
                        //    {
                        //        TestEventArgs e = new TestEventArgs(payload[1], payload[2], payload[3], payload[4], payload[5]);
                        //        TestEvent(this, e);
                        //    }
                        //    else
                        //    {
                        //        TestEvent(this, null);
                        //    }
                        //}
                        #endregion
                    }
                    break;
                #endregion
                #region default
                default:
                    {
                        if (UnsolititedFrameReceived != null)
                        {
                            UnsolititedFrameReceived(new UnsolititedFrameReceivedArgs(args.FrameBuffer));
                        }
                    }
                    break;
                #endregion
            }
        }


        delegate void NodeRemovedDelegate();
        private void NodeRemovedDone()
        {
            SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveRemoveNodeFromNetwork, new byte[] { (byte)Mode.NodeStop });
        }

        private void NodeRemovedFailed()
        {
            SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveRemoveNodeFromNetwork, new byte[] { (byte)Mode.NodeStop });
        }

        delegate void NodeAddedDelegate();
        private void NodeAddedDone()
        {
            SessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveAddNodeToNetwork, new byte[] { (byte)Mode.NodeStop, 0x00 });
        }
        private void NodeAddedDoneCallback(IAsyncResult asyncResult)
        {
            if (asyncResult.IsCompleted && (bool)asyncResult.AsyncState == true)
            {
                SetSignalResponseReceived();
            }

        }
        public void NodeAddedProtocolDone()
        {
            ResetSignalResponseReceived();
            waitForEvent = true;
            SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveAddNodeToNetwork, new byte[] { (byte)Mode.NodeStop, SessionLayer.SequenceNumber });
            WaitOneSignalResponseReceived();
        }
        private void NodeAddedProtocolDoneCallback(IAsyncResult asyncResult)
        {
            if (asyncResult.IsCompleted && (bool)asyncResult.AsyncState == true)
            {

            }
        }
        private void NodeAddedFailed()
        {
            SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveAddNodeToNetwork, new byte[] { (byte)Mode.NodeStopFailed });
        }
        private void NodeAddedFailedCallback(IAsyncResult asyncResult)
        {
            if (asyncResult.IsCompleted && (bool)asyncResult.AsyncState == true)
            {
                Thread.Sleep(200); // give controller some time
                SetSignalResponseReceived();
            }
        }

        //public virtual bool VerifyNodeInfo()
        //{
        //    bool ret = false;
        //    Memory.GetId();
        //    foreach (IDevice device in IncludedNodes)
        //    {
        //        if (device.Id == Id &&
        //            device.Generic == GenericDevices.Instance.GENERIC_TYPE_STATIC_CONTROLLER.Key &&
        //            device.Specific <= GenericDevices.Instance.GENERIC_TYPE_STATIC_CONTROLLER.
        //            FindSpecificByName(GenericDevices.SpecificDeviceTypes.SPECIFIC_TYPE_PC_CONTROLLER).Key)
        //        {
        //            ret = true;
        //            break;
        //        }
        //    }
        //    return ret;
        //}
        #endregion
    }
}
