using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Events;
using System.Collections;
using Zensys.ZWave.Enums;
using System.Threading;
using Zensys.ZWave.Logging;
using System.ComponentModel;
using Zensys.Framework;

namespace Zensys.ZWave.ProgrammerFrame
{
    class FrameLayer : IFrameLayer
    {
        private const int ACK_WAIT_TIME = 2000; // How long in ms to wait for an ack
        private const int MAX_RETRANSMISSION = 3;
        private const ushort MAX_FRAME_SIZE = 512*8;
        private const byte MIN_FRAME_SIZE = 3;

        private Stack mRetransmissionStack = new Stack();
        private Timer mRetransmissionTimeoutTimer;
        private ITransportLayer mTransportLayer = null;
        private BindingList<LogPacket> mLogDataSource;
        private FrameReceiveState mParserState;
        private DataFrame mCurrentDataFrame;
        private ushort frameLength = 0;


        #region IFrameLayer Members

        public event FrameReceivedEventHandler FrameReceived;
        public event FrameReceivedEventHandler AcknowledgeReceived;
        public event ExceptionReceivedEventHandler ExceptionReceived;

        public void Init(ITransportLayer transportLayer, BindingList<LogPacket> logDataSource)
        {
            mTransportLayer = transportLayer;
            mLogDataSource = logDataSource;
            mRetransmissionStack = Stack.Synchronized(mRetransmissionStack);
            TimerCallback tc = new TimerCallback(this.ReTransmissionTimeOutCallbackHandler);
            mRetransmissionTimeoutTimer = new Timer(tc, this, System.Threading.Timeout.Infinite, System.Threading.Timeout.Infinite);
        }

        private bool mRetransmissionQueueLocked;
        /// <summary>
        /// Gets or sets a value indicating whether [retransmission queue locked].
        /// </summary>
        /// <value>
        /// 	<c>true</c> if [retransmission queue locked]; otherwise, <c>false</c>.
        /// </value>
        public bool RetransmissionQueueLocked
        {
            get { return mRetransmissionQueueLocked; }
            set { mRetransmissionQueueLocked = value; }
        }

        private void ReTransmissionTimeOutCallbackHandler(Object handler)
        {
            CheckRetransmission(true);
        }

        public void Close()
        {
            mTransportLayer.DataReceived -= new DataReceivedEventHandler(mTransportLayer_DataReceived);
            mTransportLayer.Close();
            mRetransmissionTimeoutTimer.Change(System.Threading.Timeout.Infinite, System.Threading.Timeout.Infinite);
        }

        public int Write(byte[] data)
        {
            int ret = -1;
            try
            {
                ret = mTransportLayer.Write(data);
            }
            catch (Exception ex)
            {
                if (ExceptionReceived != null)
                {
                    ExceptionReceived(new ExceptionReceivedEventArgs(data, ex));
                }
            }
            return ret;
        }

        public IDataFrame CreateDataFrame(FrameTypes frameType, byte commandType, params byte[] parameters)
        {
            DataFrame frame = new DataFrame();
            frame.Command = commandType;
            frame.Type = frameType;
            if (parameters != null)
                frame.AddPayload(parameters);
            return frame;
        }

        public int Write(IDataFrame frame)
        {
            AddLogPacket(frame, true, null);
            TransmittedDataFrame tdf = new TransmittedDataFrame((DataFrame)frame);
            mRetransmissionStack.Push(tdf);
            int ret = Write(((DataFrame)frame).GetFrameBuffer());
            mRetransmissionTimeoutTimer.Change(ACK_WAIT_TIME, System.Threading.Timeout.Infinite);
            return ret;
        }

        private void AddLogPacket(IDataFrame frame, bool isPut, string comment)
        {
            //string app = GetApplicationString(frame.Command);
            //if (!string.IsNullOrEmpty(comment))
            //{
            //    app += " " + comment;
            //}

            mLogDataSource.Add(
                new LogPacket(
                frame.GetPayload(),
                isPut,
                frame.Type == FrameTypes.Request,
                Tools.CurrentDateTime,
                frame.Command,
                typeof(ProgrammerCommandTypes)));
        }

        private string GetApplicationString(byte commandKey)
        {
            ProgrammerCommandTypes type = (ProgrammerCommandTypes)commandKey;
            return type.ToString();
        }

        public void Open(string portName)
        {
            mTransportLayer.Open(portName);
            mTransportLayer.DataReceived += new DataReceivedEventHandler(mTransportLayer_DataReceived);
        }

        public bool IsRemoteSerialApi
        {
            get { return false; }
            set { }
        }

        #endregion

        private void mTransportLayer_DataReceived(DataReceivedEventArgs e)
        {
            //mLogDataSource.Add(new LogPacket(e.Data, false, Tools.CurrentDateTime, ""));
            foreach (byte var in e.Data)
            {
                ParseRawData(var);
            }
        }
        private bool ParseRawData(byte buffer)
        {

            switch (mParserState)
            {
                case FrameReceiveState.FRS_SOF_HUNT:
                    if (HeaderTypes.StartOfFrame == (HeaderTypes)buffer)
                    {
                        mParserState = FrameReceiveState.FRS_LENGTH_H;
                    }
                    // Acknowledge received from peer
                    else if (HeaderTypes.Acknowledge == (HeaderTypes)buffer)
                    {
                        mLogDataSource.Add(new LogPacket(new byte[] { buffer }, false, false, Tools.CurrentDateTime, buffer, typeof(HeaderTypes)));
                        // Remove the last frame from the retransmission stack
                        lock (this) { if (mRetransmissionStack.Count > 0) mRetransmissionStack.Pop(); }
                        if (AcknowledgeReceived != null)
                        {
                            AcknowledgeReceived(new FrameReceivedEventArgs(new byte[] { (byte)HeaderTypes.Acknowledge }, false, 0x00, new byte[] { 0x00 }));
                        }
                    }
                    // Not Acknowledge received from peer
                    else if (HeaderTypes.NotAcknowledged == (HeaderTypes)buffer)
                    {
                        mLogDataSource.Add(new LogPacket(new byte[] { buffer }, false, false, Tools.CurrentDateTime, buffer, typeof(HeaderTypes)));
                        CheckRetransmission(true);
                    }
                    // CAN frame received - peer dropped a data frame transmitted by us
                    else if (HeaderTypes.Can == (HeaderTypes)buffer)
                    {
                        mLogDataSource.Add(new LogPacket(new byte[] { buffer }, false, false, Tools.CurrentDateTime, buffer, typeof(HeaderTypes)));
                        // Do noting... just wait for the retransmission timer to kick-in
                    }
                    break;
                case FrameReceiveState.FRS_LENGTH_H:
                    {
                        frameLength = buffer;
                        frameLength <<= 8;
                        mParserState = FrameReceiveState.FRS_LENGTH_L;
                    }
                    break;
                case FrameReceiveState.FRS_LENGTH_L:
                    frameLength |= buffer;
                    if (frameLength < MIN_FRAME_SIZE || frameLength > MAX_FRAME_SIZE)
                    {
                        mParserState = FrameReceiveState.FRS_SOF_HUNT;
                    }
                    else
                    {
                        mCurrentDataFrame = new DataFrame((ushort)(frameLength - 4)); // Payload size is excluding len, type & cmd field
                        mParserState = FrameReceiveState.FRS_TYPE;
                    }
                    break;

                case FrameReceiveState.FRS_TYPE:
                    mCurrentDataFrame.Type = (FrameTypes)buffer;
                    if (mCurrentDataFrame.Type == FrameTypes.Request ||
                      mCurrentDataFrame.Type == FrameTypes.Response)
                    {
                        mParserState = FrameReceiveState.FRS_COMMAND;
                    }
                    else
                    {
                        mParserState = FrameReceiveState.FRS_SOF_HUNT;
                    }
                    break;

                case FrameReceiveState.FRS_COMMAND:
                    mCurrentDataFrame.Command = buffer;
                    if (mCurrentDataFrame.IsPayloadFull())
                        mParserState = FrameReceiveState.FRS_CHECKSUM;
                    else
                        mParserState = FrameReceiveState.FRS_DATA;
                    break;

                case FrameReceiveState.FRS_DATA:
                    if (!mCurrentDataFrame.AddPayload(buffer))
                    {
                        mParserState = FrameReceiveState.FRS_SOF_HUNT;
                    }
                    else if (mCurrentDataFrame.IsPayloadFull())
                    {
                        mParserState = FrameReceiveState.FRS_CHECKSUM;
                    }
                    break;

                case FrameReceiveState.FRS_CHECKSUM:
                    // Frame received successfully -> Send acknowledge (ACK)
                    if (mCurrentDataFrame.IsChecksumValid(buffer))
                    {

                        AddLogPacket(mCurrentDataFrame, false, null);
                        TransmitACK();
                        CheckRetransmission(true);
                        if (FrameReceived != null)
                        {
                            FrameReceived(new FrameReceivedEventArgs(mCurrentDataFrame.GetPayloadBuffer(), false, mCurrentDataFrame.Command, mCurrentDataFrame.GetFrameBuffer()));
                        }

                    }
                    // Frame receive failed -> Send NAK
                    else
                    {
                        TransmitNAK();
                    }

                    mParserState = FrameReceiveState.FRS_SOF_HUNT;
                    break;

                case FrameReceiveState.FRS_RX_TIMEOUT:
                default:
                    mParserState = FrameReceiveState.FRS_SOF_HUNT;
                    break;

            }

            return true;
        }

        private void TransmitACK()
        {
            mLogDataSource.Add(new LogPacket(new byte[] { (byte)HeaderTypes.Acknowledge }, true, false, Tools.CurrentDateTime, (byte)HeaderTypes.Acknowledge, typeof(HeaderTypes)));
            mTransportLayer.Write(new byte[] { (byte)HeaderTypes.Acknowledge });
        }

        private void TransmitNAK()
        {
            mLogDataSource.Add(new LogPacket(new byte[] { (byte)HeaderTypes.NotAcknowledged }, true, false, Tools.CurrentDateTime, (byte)HeaderTypes.NotAcknowledged, typeof(HeaderTypes)));
            mTransportLayer.Write(new byte[] { (byte)HeaderTypes.NotAcknowledged });
        }

        private bool CheckRetransmission(bool isretry)
        {
            if (!RetransmissionQueueLocked)
            {
                try
                {
                    lock (this)
                    {
                        if (mRetransmissionStack.Count > 0)
                        {
                            TransmittedDataFrame tdf = (TransmittedDataFrame)mRetransmissionStack.Peek();
                            // Transmit the frame to the peer...
                            AddLogPacket(tdf.frame, true, "retransmition");
                            mTransportLayer.Write(tdf.frame.GetFrameBuffer());
                            if (isretry)
                            {
                                // Drop the frame if retried to many times
                                if (++tdf.retries >= MAX_RETRANSMISSION)
                                {
                                    mRetransmissionStack.Pop();
                                }
                            }
                        }

                        if (mRetransmissionStack.Count > 0)
                        {
                            mRetransmissionTimeoutTimer.Change(ACK_WAIT_TIME, System.Threading.Timeout.Infinite);
                        }
                        else
                        {
                            mRetransmissionTimeoutTimer.Change(System.Threading.Timeout.Infinite, System.Threading.Timeout.Infinite);
                        }
                    }
                }
                catch
                {
                    throw;
                }
            }
            return true;
        }

    }
}
