﻿using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Application;
using System.Xml.Serialization;
using System.Xml;
using System.IO;
using System.Collections.ObjectModel;
using System.Security.Permissions;
using System.Security;
using System.Collections;
using Zensys.Framework;

namespace Zensys.ZWave
{
    public class DefinitionConverter
    {
        public const string MSG_LENGTH = "MSG_LENGTH";
        public const string SPECIFIC_TYPE_NOT_USED = "SPECIFIC_TYPE_NOT_USED";
        public const string VG = "vg";

        private ZWaveDefinition mZWaveDefinition;
        public ZWaveDefinition ZWaveDefinition
        {
            get { return mZWaveDefinition; }
            set { mZWaveDefinition = value; }
        }

        private ZWaveDefinition mZWaveTempDefinition;
        public ZWaveDefinition ZWaveTempDefinition
        {
            get { return mZWaveTempDefinition; }
            set { mZWaveTempDefinition = value; }
        }

        private zw_classes mZXmlDefinition;

        public zw_classes ZXmlDefinition
        {
            get { return mZXmlDefinition; }
            set { mZXmlDefinition = value; }
        }

        private string mZWaveDefinitionFile;

        public string ZWaveDefinitionFile
        {
            get { return mZWaveDefinitionFile; }
            set { mZWaveDefinitionFile = value; }
        }
        private string mZXmlDefinitionFile;

        public string ZXmlDefinitionFile
        {
            get { return mZXmlDefinitionFile; }
            set { mZXmlDefinitionFile = value; }
        }


        public DefinitionConverter(string zxmlDefinitionFile, string zwaveDefinitionFile)
        {
            ZXmlDefinitionFile = zxmlDefinitionFile;
            ZWaveDefinitionFile = zwaveDefinitionFile;
            if (File.Exists(zxmlDefinitionFile))
            {
                LoadZXmlDefinition(zxmlDefinitionFile);
            }
            else
            {
                ZXmlDefinition = new zw_classes();
            }
            if (File.Exists(zwaveDefinitionFile))
            {
                LoadZWaveDefinition(zwaveDefinitionFile);
            }
            else
            {
                ZWaveDefinition = new ZWaveDefinition();
                ZWaveDefinition.BasicDevices = new List<BasicDevice>();
                ZWaveDefinition.CommandClasses = new List<CommandClass>();
                ZWaveDefinition.GenericDevices = new List<GenericDevice>();
            }
            ZWaveTempDefinition = new ZWaveDefinition();
            ZWaveTempDefinition.BasicDevices = new List<BasicDevice>();
            ZWaveTempDefinition.CommandClasses = new List<CommandClass>();
            ZWaveTempDefinition.GenericDevices = new List<GenericDevice>();
        }

        XmlSerializer zwXmlNewSerializer = new XmlSerializer(typeof(ZWaveDefinition));
        private bool LoadZWaveDefinition(string zWaveDefinitionFileName)
        {
            bool ret = false;

            XmlReader reader = XmlReader.Create(zWaveDefinitionFileName);
            try
            {
                ZWaveDefinition = (ZWaveDefinition)zwXmlNewSerializer.Deserialize(reader);
                AssignParentProperties();
                ret = true;
            }
            catch (InvalidOperationException)
            {

            }
            finally
            {
                reader.Close();
            }
            return ret;
        }

        private bool SaveZWaveDefinition(string zWaveDefinitionFileName)
        {
            bool ret = false;
            XmlWriterSettings sett = new XmlWriterSettings();
            sett.Indent = true;
            sett.NewLineChars = Environment.NewLine;
            XmlSerializerNamespaces names = new XmlSerializerNamespaces();
            names.Add(String.Empty, string.Empty);
            XmlWriter writer = XmlWriter.Create(zWaveDefinitionFileName, sett);
            try
            {
                zwXmlNewSerializer.Serialize(writer, ZWaveDefinition, names);
                ret = true;
            }
            catch (InvalidOperationException)
            {
            }
            finally
            {
                writer.Close();
            }
            return ret;
        }

        private bool LoadZXmlDefinition(string zXmlDefinitionFileName)
        {
            bool ret = false;
            XmlReader reader = XmlReader.Create(zXmlDefinitionFileName);
            try
            {
                ZXmlDefinition = (zw_classes)zwXmlOldSerializer.Deserialize(reader);
                AssignReferenceProperties();
                ret = true;
            }
            catch (InvalidOperationException iex)
            {
                Tools._writeDebugDiagnosticExceptionMessage(iex.Message);
            }
            finally
            {
                reader.Close();
            }
            return ret;
        }

        private void AssignParentProperties()
        {
            foreach (GenericDevice generic in ZWaveDefinition.GenericDevices)
            {
                if (generic.SpecificDevice != null)
                    foreach (SpecificDevice specific in generic.SpecificDevice)
                    {
                        specific.Parent = generic;
                    }
            }
            foreach (CommandClass item in ZWaveDefinition.CommandClasses)
            {
                if (item.Command != null)
                    foreach (Command cmd in item.Command)
                    {
                        cmd.Parent = item;
                        if (cmd.Param != null)
                            foreach (Param par in cmd.Param)
                            {
                                par.ParentCmd = cmd;
                                if (par.Param1 != null)
                                    foreach (Param par1 in par.Param1)
                                    {
                                        par1.ParentParam = par;
                                        par1.ParentCmd = cmd;
                                    }
                            }
                    }
            }
        }

        private void AssignReferenceProperties()
        {
            foreach (object item in ZXmlDefinition.Items)
            {
                if (item as cmd_class == null)
                    continue;
                cmd_class cmdClass = (cmd_class)item;
                if (cmdClass.cmd != null)
                {
                    foreach (cmd cmdItem in cmdClass.cmd)
                    {
                        if (cmdItem.Items != null)
                        {
                            MakeUniqueParameterNames(cmdItem.Items);
                            AssignReferencePropertiesInner(cmdItem.Items, cmdItem.Items);
                        }
                    }
                }
            }
        }

        private void MakeUniqueParameterNames(List<object> parameters)
        {
            SortedList<string, byte> duplicatesList = new SortedList<string, byte>();
            List<string> tmpList = new List<string>();
            foreach (object item in parameters)
            {
                PrepareDuplicateList(ref duplicatesList, ref tmpList, item);
            }
            foreach (object item in parameters)
            {
                MakeUnique(item, duplicatesList);
            }
        }

        private void PrepareDuplicateList(ref SortedList<string, byte> duplicatesList, ref List<string> tmpList, object parameter)
        {
            string name = null;
            if (parameter as param != null)
            {
                param itemP = (param)parameter;
                if (itemP.type == zwXmlParamType.STRUCT_BYTE)
                {
                    foreach (object parItem in itemP.Items)
                    {
                        if (parItem as bitfield != null)
                        {
                            name = ((bitfield)parItem).fieldname;
                        }
                        else if (parItem as bitflag != null)
                        {
                            name = ((bitflag)parItem).flagname;
                        }
                        else if (parItem as fieldenum != null)
                        {
                            name = ((fieldenum)parItem).fieldname;
                        }
                        string groupedName = itemP.name != null ? itemP.name + name : name;
                        if (tmpList.Contains(groupedName) && !duplicatesList.Keys.Contains(groupedName))
                            duplicatesList.Add(groupedName, 0);
                        else
                            tmpList.Add(groupedName);
                    }
                }
                else
                {
                    name = itemP.name;
                    if (tmpList.Contains(name) && !duplicatesList.Keys.Contains(name))
                        duplicatesList.Add(name, 0);
                    else
                        tmpList.Add(name);
                }
            }
            else if (parameter as variant_group != null)
            {
                variant_group itemVG = (variant_group)parameter;
                name = string.IsNullOrEmpty(itemVG.name) ? "vg" : itemVG.name;
                if (itemVG.Items != null)
                {
                    foreach (param item in itemVG.Items)
                    {
                        PrepareDuplicateList(ref duplicatesList, ref tmpList, item);
                    }
                }

                if (tmpList.Contains(name) && !duplicatesList.Keys.Contains(name))
                    duplicatesList.Add(name, 0);
                else
                    tmpList.Add(name);
            }
        }

        private void MakeUnique(object parameter, SortedList<string, byte> duplicatesList)
        {
            if (parameter as param != null)
            {
                param itemP = (param)parameter;
                if (itemP.type == zwXmlParamType.STRUCT_BYTE)
                {
                    foreach (object parItem in itemP.Items)
                    {
                        if (parItem as bitfield != null)
                        {
                            bitfield parItemField = (bitfield)parItem;
                            string groupedName = itemP.name != null ? itemP.name + parItemField.fieldname : parItemField.fieldname;
                            if (duplicatesList.Keys.Contains(groupedName))
                            {
                                duplicatesList[groupedName]++;
                                parItemField.fieldname = string.Format("{0}{1}", parItemField.fieldname, duplicatesList[groupedName]);
                            }
                        }
                        else if (parItem as bitflag != null)
                        {
                            bitflag parItemFlag = (bitflag)parItem;
                            string groupedName = itemP.name != null ? itemP.name + parItemFlag.flagname : parItemFlag.flagname;
                            if (duplicatesList.Keys.Contains(groupedName))
                            {
                                duplicatesList[groupedName]++;
                                parItemFlag.flagname = string.Format("{0}{1}", parItemFlag.flagname, duplicatesList[groupedName]);
                            }
                        }
                        else if (parItem as fieldenum != null)
                        {
                            fieldenum parItemFieldenum = (fieldenum)parItem;
                            string groupedName = itemP.name != null ? itemP.name + parItemFieldenum.fieldname : parItemFieldenum.fieldname;
                            if (duplicatesList.Keys.Contains(groupedName))
                            {
                                duplicatesList[groupedName]++;
                                parItemFieldenum.fieldname = string.Format("{0}{1}", parItemFieldenum.fieldname, duplicatesList[groupedName]);
                            }
                        }
                    }
                }
                else if (duplicatesList.Keys.Contains(itemP.name))
                {
                    duplicatesList[itemP.name]++;
                    itemP.name = string.Format("{0}{1}", itemP.name, duplicatesList[itemP.name]);
                }

            }
            else if (parameter as variant_group != null)
            {
                variant_group itemVG = (variant_group)parameter;
                string name = string.IsNullOrEmpty(itemVG.name) ? "vg" : itemVG.name;
                if (duplicatesList.Keys.Contains(name))
                {
                    duplicatesList[name]++;
                    itemVG.name = string.Format("{0}{1}", name, duplicatesList[name]);
                }
                else
                    itemVG.name = name;
                if (itemVG.Items != null)
                {
                    foreach (param item in itemVG.Items)
                    {
                        MakeUnique(item, duplicatesList);
                    }
                }
            }

        }

        private void AssignReferencePropertiesInner(IList items, IList outerSearchScope)
        {
            foreach (object paramItem in items)
            {
                if (paramItem as param != null)
                {
                    param par = (param)paramItem;
                    if (!string.IsNullOrEmpty(par.optionaloffs))
                    {
                        par.OptionalReference = GetReference(items, outerSearchScope, Tools.GetByte(par.optionaloffs), Tools.GetByte(par.optionalmask));
                    }
                    if (par.type == zwXmlParamType.VARIANT)
                    {
                        variant parVariant = (variant)par.Items[0];
                        par.SizeReference = GetReference(items, outerSearchScope, Tools.GetByte(parVariant.paramoffs), Tools.GetByte(parVariant.sizemask));
                    }
                    if (par.type == zwXmlParamType.ARRAY)
                    {
                        arraylen parLen = null;
                        foreach (object item in par.Items)
                        {
                            if (item as arraylen != null)
                                parLen = (arraylen)item;
                        }
                        if (parLen != null)
                        {
                            par.SizeReference = GetReference(items, outerSearchScope, (byte)parLen.paramoffs, Tools.GetByte(parLen.lenmask));
                        }
                    }
                    if (par.type == zwXmlParamType.BITMASK)
                    {
                        bitmask parBitmask = (bitmask)par.Items[0];
                        par.SizeReference = GetReference(items, outerSearchScope, Tools.GetByte(parBitmask.paramoffs), Tools.GetByte(parBitmask.lenmask));
                    }
                }
                else if (paramItem as variant_group != null)
                {
                    variant_group vg = (variant_group)paramItem;
                    if (!string.IsNullOrEmpty(vg.sizemask))
                    {
                        vg.SizeReference = GetReference(items, outerSearchScope, Tools.GetByte(vg.paramOffs), Tools.GetByte(vg.sizemask));
                    }
                    AssignReferencePropertiesInner(vg.Items, items);
                }
            }
        }

        private string GetReference(IList searchScope, IList outerSearchScope, byte parameterNumber, byte mask)
        {
            string ret = null;
            if (parameterNumber == 0xFF)
                return MSG_LENGTH;
            param paramStructByte = null;
            if (parameterNumber > 0x7F) // parameter number placed outside the variant group
            {
                paramStructByte = (param)outerSearchScope[parameterNumber - 0x80];
            }
            else
            {
                paramStructByte = (param)searchScope[parameterNumber];
            }

            foreach (object item in paramStructByte.Items)
            {
                if (item as valueattrib != null || item as word != null || item as dword != null || item as @const != null)
                {
                    if (ret == null)
                    {
                        ret = Tools.MakeLegalMixCaseIdentifier(paramStructByte.name);
                    }
                    else
                    {
                        throw new ApplicationException("duplicate reference found");
                    }
                }
                if (item as bitfield != null)
                {
                    bitfield itemF = (bitfield)item;
                    if (Tools.GetByte(itemF.fieldmask) == mask)
                    {
                        if (ret == null)
                        {
                            ret = Tools.MakeLegalMixCaseIdentifier(itemF.fieldname);
                        }
                        else
                        {
                            throw new ApplicationException("duplicate reference found");
                        }
                    }
                }
                else if (item as bitflag != null)
                {
                    bitflag itemF = (bitflag)item;
                    if (Tools.GetByte(itemF.flagmask) == mask)
                    {
                        if (ret == null)
                        {
                            ret = Tools.MakeLegalMixCaseIdentifier(itemF.flagname);
                        }
                        else
                        {
                            throw new ApplicationException("duplicate reference found");
                        }
                    }
                }
                else if (item as fieldenum != null)
                {
                    fieldenum itemF = (fieldenum)item;
                    if (Tools.GetByte(itemF.fieldmask) == mask)
                    {
                        if (ret == null)
                        {
                            ret = Tools.MakeLegalMixCaseIdentifier(itemF.fieldname);
                        }
                        else
                        {
                            throw new ApplicationException("duplicate reference found");
                        }
                    }
                }
            }
            return ret;
        }

        public bool SaveZXmlDefinition(string zXmlDefinitionFileName)
        {
            FileIOPermission permission2 = new FileIOPermission(FileIOPermissionAccess.AllAccess, Path.GetFullPath(zXmlDefinitionFileName));
            try
            {
                permission2.Demand();
            }
            catch (SecurityException s)
            {
                //logger("AC: SecurityException when temp file requested (init)");
                Tools._writeDebugDiagnosticExceptionMessage(s.Message);
            }

            bool ret = false;
            XmlWriterSettings sett = new XmlWriterSettings();
            sett.Indent = true;
            sett.NewLineChars = Environment.NewLine;
            XmlSerializerNamespaces names = new XmlSerializerNamespaces();
            names.Add(String.Empty, string.Empty);
            XmlWriter writer = XmlWriter.Create(zXmlDefinitionFileName, sett);
            try
            {
                zwXmlOldSerializer.Serialize(writer, ZXmlDefinition, names);
                ret = true;
            }
            catch (InvalidOperationException)
            {
            }
            finally
            {
                writer.Close();
            }
            return ret;
        }

        public void UpgradeConvert(bool isSaveConverted)
        {
            ZWaveDefinition = new ZWaveDefinition();
            ZWaveDefinition.BasicDevices = new List<BasicDevice>();
            ZWaveDefinition.CommandClasses = new List<CommandClass>();
            ZWaveDefinition.GenericDevices = new List<GenericDevice>();
            foreach (object item in ZXmlDefinition.Items)
            {
                if (item is bas_dev)
                {
                    ZWaveDefinition.BasicDevices.Add(UpgradeBasicDevice(item));
                }
                if (item is gen_dev)
                {
                    ZWaveDefinition.GenericDevices.Add(UpgradeGenericDevice(item));
                }
                if (item is cmd_class)
                {
                    ZWaveDefinition.CommandClasses.Add(UpgradeCommandClass(item));
                }
            }
            AssignParentProperties();
            if (isSaveConverted)
            {
                FileIOPermission permission2 = new FileIOPermission(FileIOPermissionAccess.AllAccess, Path.GetFullPath(ZWaveDefinitionFile));
                try
                {
                    permission2.Demand();
                }
                catch (SecurityException s)
                {
                    //logger("AC: SecurityException when temp file requested (init)");
                    Tools._writeDebugDiagnosticExceptionMessage(s.Message);
                }
                SaveZWaveDefinition(ZWaveDefinitionFile);
            }
        }

        public void UpgradeTempConvert()
        {
            AssignReferenceProperties();
            ZWaveTempDefinition = new ZWaveDefinition();
            ZWaveTempDefinition.BasicDevices = new List<BasicDevice>();
            ZWaveTempDefinition.CommandClasses = new List<CommandClass>();
            ZWaveTempDefinition.GenericDevices = new List<GenericDevice>();
            foreach (object item in ZXmlDefinition.Items)
            {
                if (item is bas_dev)
                {
                    ZWaveTempDefinition.BasicDevices.Add(UpgradeBasicDevice(item));
                }
                if (item is gen_dev)
                {
                    ZWaveTempDefinition.GenericDevices.Add(UpgradeGenericDevice(item));
                }
                if (item is cmd_class)
                {
                    ZWaveTempDefinition.CommandClasses.Add(UpgradeCommandClass(item));
                }
            }
            AssignParentProperties();
        }

        private CommandClass UpgradeCommandClass(object commandClass)
        {
            if (commandClass as cmd_class == null)
                return null;
            else
            {
                cmd_class val = (cmd_class)commandClass;
                CommandClass ret = new CommandClass();
                ret.Comment = val.comment;
                ret.Key = val.key;
                ret.Name = val.name;
                ret.Text = val.help;
                ret.Version = Byte.Parse(val.version);
                List<DefineSet> defineSetList = new List<DefineSet>();
                int definesCounter = 0;
                if (val.cmd != null)
                {
                    List<Command> commands = new List<Command>();
                    foreach (cmd item in val.cmd)
                    {
                        commands.Add(UpgradeCommand(item, ref defineSetList, ref definesCounter));
                    }
                    if (commands.Count > 0)
                    {
                        ret.Command = new List<Command>(commands);
                    }
                }
                if (defineSetList.Count > 0)
                {
                    ret.DefineSet = new List<DefineSet>(defineSetList);
                }
                return ret;
            }
        }

        private GenericDevice UpgradeGenericDevice(object genericDevice)
        {
            if (genericDevice as gen_dev == null)
                return null;
            else
            {
                gen_dev val = (gen_dev)genericDevice;
                GenericDevice ret = new GenericDevice();
                ret.Comment = val.comment;
                ret.Key = val.key;
                ret.Name = val.name;
                ret.Text = val.help;
                if (val.spec_dev != null)
                {
                    List<SpecificDevice> specificDevices = new List<SpecificDevice>();
                    foreach (spec_dev item in val.spec_dev)
                    {
                        SpecificDevice sd = UpgradeSpecificDevice(item, ret);
                        if (sd != null)
                        {
                            specificDevices.Add(sd);
                        }
                    }
                    if (specificDevices.Count > 0)
                        ret.SpecificDevice = new List<SpecificDevice>(specificDevices);
                }
                return ret;
            }
        }

        private BasicDevice UpgradeBasicDevice(object basicDevice)
        {
            if (basicDevice as bas_dev == null)
                return null;
            else
            {
                bas_dev val = (bas_dev)basicDevice;
                BasicDevice ret = new BasicDevice();
                ret.Comment = val.comment;
                ret.Key = val.key;
                ret.Name = val.name;
                ret.Text = val.help;
                return ret;
            }
        }

        private SpecificDevice UpgradeSpecificDevice(object specificDevice, GenericDevice parentGeneric)
        {
            if (specificDevice as spec_dev == null)
                return null;
            else
            {

                spec_dev val = (spec_dev)specificDevice;
                SpecificDevice ret = new SpecificDevice();
                ret.Comment = val.comment;
                ret.Key = val.key;
                ret.Name = val.name;
                ret.Text = val.help;
                if (val.name == SPECIFIC_TYPE_NOT_USED)
                {
                    ret.ScopeKeyId = parentGeneric.KeyId;
                }
                return ret;
            }
        }

        private Command UpgradeCommand(object command, ref List<DefineSet> defineSetList, ref int definesCounter)
        {
            if (command as cmd == null)
                return null;
            else
            {
                cmd val = (cmd)command;
                Command ret = new Command();
                ret.Comment = val.comment;
                if (val.cmd_mask != null)
                {
                    ret.Bits = Tools.GetBitsFromMask(Tools.GetByte(val.cmd_mask));
                }
                ret.Key = val.key;
                ret.Name = val.name;
                ret.Text = val.help;
                if (val.Items != null)
                {
                    byte order = 0;
                    List<Param> pars = new List<Param>();
                    foreach (object item in val.Items)
                    {
                        #region exceptions
                        if (ret.Name == "NODE_INFO" || ret.Name == "NEW_NODE_REGISTERED")
                        {
                            if (item as param != null && ((param)item).name == "Basic Device Class")
                            {
                                continue;
                            }
                        }

                        #endregion
                        foreach (Param itemParam in UpgradeParam(item, ref order, ref defineSetList, ref definesCounter))
                        {
                            pars.Add(itemParam);
                        }
                    }
                    if (pars.Count > 0)
                        ret.Param = new List<Param>(pars);

                }
                return ret;
            }
        }

        private List<Param> UpgradeParam(object param, ref byte order, ref List<DefineSet> defineSetList, ref int definesCounter)
        {
            List<Param> retList = new List<Param>();
            if (param as param != null)
            {
                param val = (param)param;
                //try
                {
                    switch (val.type)
                    {
                        case zwXmlParamType.ARRAY:
                            retList.Add(UgradeParameterTypeArray(val, ref order));
                            break;
                        case zwXmlParamType.BIT_24:
                            retList.Add(UgradeParameterTypeBit24(val, ref order));
                            break;
                        case zwXmlParamType.BITMASK:
                            retList.Add(UgradeParameterTypeBitmask(val, ref order));
                            break;
                        case zwXmlParamType.BYTE:
                            List<Define> defines1 = new List<Define>();
                            Param parameter1 = UgradeParameterTypeByte(val, ref order, ref defines1);
                            if (defines1.Count > 0)
                            {
                                DefineSet defineSet1 = new DefineSet();
                                defineSet1.Type = zwDefineSetType.Unknown;
                                defineSet1.Define = new List<Define>(defines1);
                                MergeDefineSet(ref defineSetList, defineSet1, ref parameter1, ref definesCounter);
                            }
                            retList.Add(parameter1);
                            break;
                        case zwXmlParamType.CONST:

                            List<Define> defines2 = new List<Define>();
                            Param parameter2 = UgradeParameterTypeConst(val, ref order, ref defines2);
                            if (defines2.Count > 0)
                            {
                                DefineSet defineSet2 = new DefineSet();
                                defineSet2.Type = zwDefineSetType.Full;
                                defineSet2.Define = new List<Define>(defines2);
                                MergeDefineSet(ref defineSetList, defineSet2, ref parameter2, ref definesCounter);
                            }
                            retList.Add(parameter2);
                            break;
                        case zwXmlParamType.DWORD:
                            retList.Add(UgradeParameterTypeDWord(val, ref order));
                            break;
                        case zwXmlParamType.ENUM:
                            retList.Add(UgradeParameterTypeEnum(val, ref order));
                            break;
                        case zwXmlParamType.ENUM_ARRAY:
                            retList.Add(UgradeParameterTypeEnumArray(val, ref order));
                            break;
                        case zwXmlParamType.MARKER:
                            retList.Add(UgradeParameterTypeMarker(val, ref order));
                            break;
                        case zwXmlParamType.MULTI_ARRAY:
                            retList.Add(UgradeParameterTypeMultiArray(val, ref order));
                            break;
                        case zwXmlParamType.STRUCT_BYTE:
                            foreach (Param item in (UgradeParameterTypeStructByte(val, ref order, ref defineSetList, ref definesCounter)))
                            {
                                retList.Add(item);
                            }
                            break;
                        case zwXmlParamType.VARIANT:
                            retList.Add(UgradeParameterTypeVariant(val, ref order));
                            break;
                        case zwXmlParamType.WORD:
                            retList.Add(UgradeParameterTypeWord(val, ref order));
                            break;
                        default:
                            retList.Add(UpgradeParameterCommon(val));
                            break;
                    }
                }
                //catch (Exception ex)
                //{
                //    Tools._writeDebugDiagnosticExceptionMessage(ex.Message);
                //}
            }
            else if (param as variant_group != null)
            {
                retList.Add(UpgradeVariantGroup(param, ref order, ref defineSetList, ref definesCounter));
            }
            return retList;
        }

        private void MergeDefineSet(ref List<DefineSet> defineSetList, DefineSet defineSet, ref Param parameter, ref int counter)
        {
            string str = ContainDefineSet(defineSetList, defineSet);
            if (str == null)
            {
                str = ContainDefineSetName(defineSetList, parameter.Name) ? parameter.Name + ++counter : parameter.Name;
                defineSet.Name = str;
                defineSetList.Add(defineSet);
            }
            parameter.Defines = str;
        }

        private bool ContainDefineSetName(List<DefineSet> defineSetList, string name)
        {
            bool ret = false;
            foreach (DefineSet item in defineSetList)
            {
                if (item.Name == name)
                {
                    ret = true;
                    break;
                }
            }
            return ret;
        }

        /// <summary>
        /// Looks for similar define set.
        /// </summary>
        /// <param name="defineSetList">The define set list.</param>
        /// <param name="defineSet">The define set.</param>
        /// <returns>name of the define set if found</returns>
        private string ContainDefineSet(List<DefineSet> defineSetList, DefineSet defineSet)
        {
            string ret = null;
            foreach (DefineSet item in defineSetList)
            {
                int mismatches = defineSet.Define.Count;
                foreach (Define defItem1 in item.Define)
                {
                    foreach (Define defItem2 in defineSet.Define)
                    {
                        if (defItem1.Name == defItem2.Name && defItem1.Key == defItem2.Key)
                        {
                            mismatches--;
                        }
                    }
                }
                if (mismatches == 0)
                {
                    ret = item.Name;
                    break;
                }
            }
            return ret;
        }

        private bool IsZwBoolean(bool isSpecified, zwBoolean zwBooleanValue)
        {
            bool ret = false;
            if (isSpecified)
            {
                ret = zwBooleanValue == zwBoolean.True || zwBooleanValue == zwBoolean.@true;
            }
            return ret;
        }



        private Param UpgradeParameterCommon(param val)
        {
            Param ret = new Param();
            ret.Type = zwParamType.HEX;
            if (IsZwBoolean(val.encapsulatedSpecified, val.encapsulated))
            {
                ret.Type = zwParamType.ENCAP_CMD;
            }
            ret.Comment = val.comment;
            ret.Name = Tools.MakeLegalMixCaseIdentifier(val.name);
            ret.Text = val.name;
            ret.SizeReference = val.SizeReference;
            ret.OptionalReference = val.OptionalReference;

            return ret;
        }

        private Param UgradeParameterTypeWord(param val, ref byte order)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 16;
            ret.Order = order;
            order++;
            if (val.Items != null && val.Items.Count == 1)
            {
                word parAttrib = (word)val.Items[0];
                if (IsZwBoolean(parAttrib.showhexSpecified, parAttrib.showhex))
                {
                    ret.Type = zwParamType.HEX;
                }
                else
                {
                    ret.Type = zwParamType.NUMBER;
                }
            }
            return ret;
        }

        private Param UgradeParameterTypeVariant(param val, ref byte order)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 8;
            ret.Order = order;
            order++;
            if (ret.Type != zwParamType.ENCAP_CMD)
            {
                foreach (object item in val.Items)
                {
                    if (item as variant != null)
                    {
                        variant itemAttrib = (variant)item;
                        if (IsZwBoolean(itemAttrib.showhexSpecified, itemAttrib.showhex))
                        {
                            ret.Type = zwParamType.HEX;
                        }
                        else
                        {
                            ret.Type = zwParamType.NUMBER;
                        }
                    }
                }
            }
            return ret;
        }

        private IList<Param> UgradeParameterTypeStructByte(param val, ref byte order, ref List<DefineSet> defineSetList, ref int definesCounter)
        {
            SortedList<byte, Param> ret = new SortedList<byte, Param>();
            byte sumBits = 0;
            if (val.cmd_mask != null)
            {
                sumBits = (byte)(8 - Tools.GetBitsFromMask(Tools.GetByte(val.cmd_mask)));
            }
            foreach (object item in val.Items)
            {
                Param p = new Param();
                p.Type = zwParamType.NUMBER;
                p.GroupName = val.name;
                p.Order = order;
                order++;
                if (item as bitfield != null)
                {
                    bitfield itemF = (bitfield)item;
                    p.Bits = Tools.GetBitsFromMask(Tools.GetByte(itemF.fieldmask));
                    sumBits += p.Bits;
                    p.Name = Tools.MakeLegalMixCaseIdentifier(itemF.fieldname);
                    p.Text = itemF.fieldname;
                    ret.Add(Tools.GetByte(itemF.fieldmask), p);
                }
                else if (item as bitflag != null)
                {
                    bitflag itemF = (bitflag)item;
                    p.Bits = 1;
                    sumBits += p.Bits;
                    p.Type = zwParamType.BOOLEAN;
                    p.Name = Tools.MakeLegalMixCaseIdentifier(itemF.flagname);
                    p.Text = itemF.flagname;
                    ret.Add(Tools.GetByte(itemF.flagmask), p);
                }
                else if (item as fieldenum != null)
                {
                    fieldenum itemF = (fieldenum)item;
                    p.Bits = Tools.GetBitsFromMask(Tools.GetByte(itemF.fieldmask));
                    sumBits += p.Bits;
                    p.Name = Tools.MakeLegalMixCaseIdentifier(itemF.fieldname);
                    p.Text = itemF.fieldname;
                    if (itemF.fieldenum1 != null)
                    {
                        byte counter = 0;
                        List<Define> defines = new List<Define>();
                        byte innerCounter = 0;
                        foreach (fieldenum itemEnum in itemF.fieldenum1)
                        {
                            if (itemEnum as fieldenum != null)
                            {
                                fieldenum parAttrib = (fieldenum)itemEnum;
                                Define d = new Define();
                                d.Key = Tools.GetHex(counter, true);
                                d.Name = GetUnique(Tools.MakeLegalMixCaseIdentifier(parAttrib.value), defines, ref innerCounter);
                                d.Text = parAttrib.value;
                                defines.Add(d);
                                counter++;
                            }
                        }
                        if (defines.Count > 0)
                        {
                            DefineSet defineSet = new DefineSet();
                            defineSet.Type = zwDefineSetType.Partial;
                            defineSet.Define = new List<Define>(defines);
                            MergeDefineSet(ref defineSetList, defineSet, ref p, ref definesCounter);
                        }
                    }
                    ret.Add(Tools.GetByte(itemF.fieldmask), p);
                }
                else
                    throw new ApplicationException("Invalid parameter");

            }
            if (sumBits < 8)
            {
                Param p = new Param();
                p.Type = zwParamType.NUMBER;
                p.GroupName = val.name;
                p.Order = order;
                order++;

                p.Bits = (byte)(8 - sumBits);
                p.Name = Tools.MakeLegalMixCaseIdentifier("Reserved");
                p.Text = "Reserved";
                ret.Add(Tools.GetMaskFromBits(p.Bits, sumBits), p);
            }
            else if (sumBits > 8)
            {
                throw new ApplicationException("Invalid STRUCT_BYTE parameter");
            }
            return ret.Values;
        }

        private string GetUnique(string name, List<Define> defines, ref byte uniqueCounter)
        {
            foreach (Define item in defines)
            {
                if (item.Name == name)
                {
                    uniqueCounter++;
                    return string.Format("{0}{1}", name, uniqueCounter);
                }
            }
            return name;
        }

        private Param UgradeParameterTypeMultiArray(param val, ref byte order)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 8;
            ret.Order = order;
            order++;
            ret.Type = zwParamType.SPEC_DEV_REF;
            return ret;
        }

        private Param UgradeParameterTypeMarker(param val, ref byte order)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 8;
            ret.Order = order;
            order++;
            ret.Type = zwParamType.MARKER;
            if (val.Items != null)
            {
                List<byte> tmp = new List<byte>();
                foreach (object item in val.Items)
                {
                    if (item is @const)
                    {
                        tmp.Add(Tools.GetByte(((@const)item).flagmask));
                    }
                }
                ret.DefaultValue = tmp.ToArray();
                ret.Bits = (byte)(ret.DefaultValue.Length * 8);
            }
            return ret;
        }

        private Param UgradeParameterTypeEnumArray(param val, ref byte order)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 8;
            ret.Order = order;
            order++;
            ret.SizeReference = MSG_LENGTH;
            ret.Type = zwParamType.CMD_CLASS_REF;
            return ret;
        }

        private Param UgradeParameterTypeEnum(param val, ref byte order)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 8;
            ret.Order = order;
            order++;
            if (val.name.StartsWith("Basic", StringComparison.InvariantCultureIgnoreCase))
            {
                ret.Type = zwParamType.BAS_DEV_REF;
            }
            else if (val.name.StartsWith("Generic", StringComparison.InvariantCultureIgnoreCase))
            {
                ret.Type = zwParamType.GEN_DEV_REF;
            }
            return ret;
        }

        private Param UgradeParameterTypeDWord(param val, ref byte order)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 32;
            ret.Order = order;
            order++;
            if (val.Items != null && val.Items.Count == 1)
            {
                dword parAttrib = (dword)val.Items[0];
                if (IsZwBoolean(parAttrib.showhexSpecified, parAttrib.showhex))
                {
                    ret.Type = zwParamType.HEX;
                }
                else
                {
                    ret.Type = zwParamType.NUMBER;
                }
            }
            return ret;
        }

        private Param UgradeParameterTypeConst(param val, ref byte order, ref List<Define> defines)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 8;
            ret.Order = order;
            order++;
            if (val.Items != null)
            {
                byte innerCounter = 0;
                foreach (object item in val.Items)
                {
                    @const parAttrib = (@const)item;
                    Define d = new Define();
                    d.Key = Tools.GetHex(Tools.GetByte(parAttrib.flagmask), true);
                    d.Name = GetUnique(Tools.MakeLegalMixCaseIdentifier(parAttrib.flagname), defines, ref innerCounter);
                    d.Text = parAttrib.flagname;
                    defines.Add(d);
                }
            }
            ret.Type = zwParamType.NUMBER;
            return ret;
        }

        private Param UgradeParameterTypeByte(param val, ref byte order, ref List<Define> defines)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 8;
            ret.Order = order;
            order++;
            if (val.Items != null)
            {
                byte innerCounter = 0;
                foreach (object item in val.Items)
                {
                    if (item as valueattrib != null)
                    {
                        valueattrib parAttrib = (valueattrib)item;
                        if (IsZwBoolean(parAttrib.showhexSpecified, parAttrib.showhex))
                        {
                            ret.Type = zwParamType.HEX;
                        }
                        else
                        {
                            ret.Type = zwParamType.NUMBER;
                        }
                    }
                    else if (item as bitflag != null)
                    {
                        bitflag parAttrib = (bitflag)item;
                        Define d = new Define();
                        d.Key = Tools.GetHex(Tools.GetByte(parAttrib.flagmask), true);
                        d.Name = GetUnique(Tools.MakeLegalMixCaseIdentifier(parAttrib.flagname), defines, ref innerCounter);
                        d.Text = parAttrib.flagname;
                        defines.Add(d);
                    }
                }

            }
            return ret;
        }

        private Param UgradeParameterTypeBitmask(param val, ref byte order)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 8;
            ret.Order = order;
            order++;
            ret.Type = zwParamType.BITMASK;
            return ret;
        }

        private Param UgradeParameterTypeBit24(param val, ref byte order)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 24;
            ret.Order = order;
            order++;
            if (val.Items != null && val.Items.Count == 1)
            {
                bit_24 parAttrib = (bit_24)val.Items[0];
                if (IsZwBoolean(parAttrib.showhexSpecified, parAttrib.showhex))
                {
                    ret.Type = zwParamType.HEX;
                }
                else
                {
                    ret.Type = zwParamType.NUMBER;
                }
            }
            return ret;
        }

        private Param UgradeParameterTypeArray(param val, ref byte order)
        {
            Param ret = UpgradeParameterCommon(val);
            ret.Bits = 8;
            ret.Order = order;
            order++;
            bool sizeCalculated = false;
            foreach (object item in val.Items)
            {
                if (item as arraylen != null)
                {
                    arraylen itemLen = (arraylen)item;
                    sizeCalculated = true;
                }
                if (item as arrayattrib != null)
                {
                    arrayattrib itemAttrib = (arrayattrib)item;
                    if (!sizeCalculated)
                    {

                        ret.Size = Tools.GetByte(itemAttrib.len);
                        if (ret.Size < 255)
                            ret.SizeSpecified = true;
                        else
                        {
                            ret.Size = 0;
                            ret.SizeSpecified = false;
                            ret.SizeReference = MSG_LENGTH;
                            ret.Type = zwParamType.HEX;
                        }
                    }
                    if (itemAttrib.is_asciiSpecified)
                    {
                        if (itemAttrib.is_ascii == zwBoolean.True || itemAttrib.is_ascii == zwBoolean.@true)
                            ret.Type = zwParamType.CHAR;
                        else
                            ret.Type = zwParamType.NUMBER;
                    }
                }
            }
            return ret;
        }

        private Param UpgradeVariantGroup(object variantGroup, ref byte order, ref List<DefineSet> defineSetList, ref int definesCounter)
        {
            if (variantGroup as variant_group == null)
                return null;
            else
            {
                variant_group val = (variant_group)variantGroup;
                Param ret = new Param();
                ret.Type = zwParamType.VARIANT_GROUP;
                ret.Name = Tools.MakeLegalMixCaseIdentifier(val.name);
                ret.Order = order;
                order++;
                ret.Bits = 8;
                ret.Comment = val.comment;
                ret.SizeReference = val.SizeReference;
                ret.Text = val.name;
                if (val.Items != null)
                {
                    byte innerorder = 0;
                    List<Param> pars = new List<Param>();
                    foreach (param item in val.Items)
                    {
                        foreach (Param itemParam in UpgradeParam(item, ref innerorder, ref defineSetList, ref definesCounter))
                        {
                            pars.Add(itemParam);
                        }
                    }
                    if (pars.Count > 0)
                        ret.Param1 = new List<Param>(pars);

                }
                return ret;
            }
        }

        private void CalculateReference(string name, Param parameter, out byte reference, out byte mask, out byte shifter)
        {
            Param referenceParameter = null;
            if (name != MSG_LENGTH)
            {
                byte referenceLevel = 0;
                if (parameter.ParentParam != null) // parameter in variant group
                {
                    referenceParameter = FindParameter(parameter.ParentParam.Param1, name);
                }
                if (referenceParameter == null)
                {
                    referenceParameter = FindParameter(parameter.ParentCmd.Param, name);
                    if (parameter.ParentParam != null)
                    {
                        referenceLevel = 0x80;// parameter in variant group but name not found in parent param
                    }
                }

                if (referenceParameter.ParentParam != null)
                {
                    CalculateReferenceInner(referenceParameter, referenceParameter.ParentParam.Param1, out reference, out mask, out shifter, 0);
                }
                else
                {
                    CalculateReferenceInner(referenceParameter, referenceParameter.ParentCmd.Param, out reference, out mask, out shifter, referenceParameter.ParentCmd.Bits);
                }
                reference += referenceLevel;
            }
            else
            {
                reference = 255;
                mask = 0;
                shifter = 0;
            }
        }

        private Param FindParameter(List<Param> list, string name)
        {
            foreach (Param var in list)
            {
                if (var.Name == name)
                {
                    return var;
                }
            }
            return null;
        }
        private void CalculateReferenceInner(Param parameter, List<Param> parameters, out byte reference, out byte mask, out byte shifter, byte cmdBits)
        {
            reference = 0;
            mask = 0;
            shifter = 0;
            if (cmdBits > 0 && cmdBits < 8)
            {
                shifter = cmdBits;
            }
            foreach (Param item in parameters)
            {
                if (item.Name == parameter.Name)
                {
                    break;
                }
                if (item.Bits % 8 == 0)
                {
                    reference += 1;
                }
                else
                {
                    shifter += item.Bits;
                    if (shifter == 8)
                    {
                        shifter = 0;
                        reference += 1;
                    }
                    else if (shifter > 8)
                        throw new ApplicationException("Invalid STRUCT_BYTE parameter");
                }
            }
            mask = Tools.GetMaskFromBits(parameter.Bits, shifter);
        }


     

        XmlSerializer zwXmlOldSerializer = new XmlSerializer(typeof(zw_classes));

        private string DefinitionToString(XmlSerializer serializer, object definition)
        {
            StringBuilder ret = new StringBuilder();
            XmlWriterSettings sett = new XmlWriterSettings();
            sett.Indent = true;
            sett.NewLineChars = Environment.NewLine;
            XmlSerializerNamespaces names = new XmlSerializerNamespaces();
            names.Add(String.Empty, string.Empty);
            XmlWriter writer = XmlWriter.Create(ret, sett);
            try
            {
                serializer.Serialize(writer, definition, names);
            }
            catch (InvalidOperationException)
            {
            }
            finally
            {
                writer.Close();
            }
            return ret.ToString();
        }

        public XmlDocument ZXmlDefinitionToDocument(string node)
        {
            XmlDocument ret = new XmlDocument();
            using (StringReader reader = new StringReader(DefinitionToString(zwXmlOldSerializer, ZXmlDefinition)))
            {
                ret.Load(reader);
            }
            if (node != null)
            {
                XmlNode xn = ret.SelectSingleNode(node);
                ret.RemoveAll();
                ret.AppendChild(xn);
            }
            return ret;
        }

        public XmlDocument ZWaveDefinitionToDocument(string node)
        {
            XmlDocument ret = new XmlDocument();
            using (StringReader reader = new StringReader(DefinitionToString(zwXmlNewSerializer, ZWaveTempDefinition)))
            {
                ret.Load(reader);
            }
            if (node != null)
            {
                XmlNode xn = ret.SelectSingleNode(node);
                ret.RemoveAll();
                ret.AppendChild(xn);
            }
            return ret;
        }
    }
}
