/******************************* p_button.c  *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Button module for development kit controller board.
 *
 * Author:   Oleg Zadorozhnyy
 *
 * Last Changed By:  $Author: oza $
 * Revision:         $Revision: 1.1 $
 * Last Changed:     $Date: 2007/08/31 11:39:20 $
 *
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include "TYPES.H"      // Standard types
#include "p_button.h"
#ifdef __GNUC__
#include <avr/io.h>
#else
#include "iom128.h"
#endif
#include "LED.h"

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
/*Timing used when checking button status*/
#define PB_DEBOUNCE_TIME 2      /*10-20ms debounce should be enough*/
#define PB_LONG_TIME 60         /*600ms before we assume the key is held down*/
#define PB_REPEAT_WAIT_TIME   20 /* Wait 200ms before repeating */

#define PIN_GET(val)  (PINE & (1 << PINE##val))
#define SET_PB_STATUS_PB(val,val1)  if(PIN_GET(val)==0) bPressed |= DOWN_PB##val1;
#define SET_PB_STATUS_PB7(val,val1)  if(PIN_GET(val)==0) bPressed7 |= DOWN_PB##val1;
/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
BYTE bPressed = FALSE;
BYTE bOldPressed = FALSE;
BYTE readyCount = 0;

BYTE bPressed7 = FALSE;
BYTE readyCount7 = 0;
BYTE bPressed7DEBOUNCE = FALSE;

/*============================   TimerCheckButtons   ======================
**    Function description
**    This function checks and updates the status of the Push buttons.
**    It is run every 10ms
**    Side effects:
**
**--------------------------------------------------------------------------*/
void TimerCheckButtons(void)
{
  if(bPressed) bOldPressed = bPressed&PB_MASK; //Store old values if a key is down

  bPressed = FALSE;
  bPressed7 = FALSE;
  /*Call macro to set value of PBs*/
  SET_PB_STATUS_PB(2,0);
  SET_PB_STATUS_PB(3,1);
  SET_PB_STATUS_PB(4,2);
  SET_PB_STATUS_PB(5,3);
  SET_PB_STATUS_PB(6,4);
  SET_PB_STATUS_PB7(7,0);

  /*Make sure that MOSI is output after we checked the key. The rest is ok as inputs*/

  if(bPressed7)
  {
    if (readyCount7<5) readyCount7++;
    if (readyCount7==4) bPressed7DEBOUNCE = 1;
  }
  if(bPressed||bPressed7)
  {
    readyCount++; /*Increment readyCount if PB is down*/
    LED_ON(6);
  }
  else
  {
    /*A key has been released, increment debounce*/
    if(bOldPressed&PB_RELEASE)
    readyCount++;
    else  if((bOldPressed&PB_MASK))
          {
            /*If last detection resulted in a value. Restart DEBOUNCE*/
            readyCount = 0;
            bPressed = 0;    /*Zero bPressed*/
            bOldPressed |= PB_RELEASE;         /*Set bit, release detect ongoing*/
          }
    bPressed7 = 0;
    readyCount7 = 0;
    bPressed7DEBOUNCE = 0;
    LED_OFF(6);
  }
  if(readyCount == 0xFF)
    readyCount = 0xFE;
  if(readyCount>PB_DEBOUNCE_TIME)
    bOldPressed = FALSE;              /*Zero old values. No need when PB_DEBOUNCE expires*/
  if(readyCount>PB_REPEAT_WAIT_TIME)
    bPressed |= PB_REPEAT;
  if(readyCount>PB_LONG_TIME)
    bPressed |= PB_HELD;
}

/*============================   ButtonPressed   ============================
**    Function description
**     Returns the status of the Push buttons
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE ButtonPressed(void)
{
 if(readyCount>PB_DEBOUNCE_TIME)
   return bPressed;
 else
   return bOldPressed;
}


