/******************************* p_button.c  *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Button module for development kit controller board.
 *
 * Author:   Henrik Holm
 *
 * Last Changed By:  $Author: efh $
 * Revision:         $Revision: 22531 $
 * Last Changed:     $Date: 2012-03-20 16:19:42 +0100 (Tue, 20 Mar 2012) $
 *
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <p_button.h>
#include <ZW_controller_api.h>
#include <ZW_pindefs.h>
#include <ZW_uart_api.h>

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

/*Timing used when checking button status*/
#define PB_DEBOUNCE_TIME 2      /*10-20ms debounce should be enough*/
#define PB_LONG_TIME 60         /*600ms before we assume the key is held down*/

#define PB_REPEAT_WAIT_TIME   20 /* Wait 200ms before repeating */

/*If the button is pressed set bit to 1*/
#define SET_PB_STATUS(val) if(PIN_GET(val##)==0) bPressed |= DOWN_##val

#define IS_DOWN_REPEAT(val) ((bPressed&DOWN_##val)&&(bPressed&PB_REPEAT))
#define IS_DOWN_HELD(val) ((bPressed&DOWN_##val)&&(bPressed&PB_HELD))

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
BYTE bPressed = FALSE;
BYTE bOldPressed = FALSE;
BYTE readyCount = 0;

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/
/*============================   TimerCheckButtons   ======================
**    Function description
**    This function checks and updates the status of the Push buttons.
**    It is run every 10ms
**    Side effects:
**
**--------------------------------------------------------------------------*/
void TimerCheckButtons(void)
{
  if(PIN_GET(EEPCS)!=0) /*Only check buttons if EEPROM is not selected*/
  {
    if(bPressed)
      bOldPressed = bPressed&PB_MASK; //Store old values if a key is down

    bPressed = FALSE;
  /*Call macro to set value of PBs*/
    PIN_IN(PB0,1);
    PIN_IN(PB1,1);
    PIN_IN(PB2,1);
    PIN_IN(PB3,1);
#ifndef ZW_DEBUG
#ifndef ZW_DEBUG_CMD
#ifndef ZW_DEBUG_PROTOCOL
    PIN_IN(PB4,1);
#endif /*ZW_DEBUG_PROTOCOL*/
#endif
#endif /*ZW_DEBUG*/
    SET_PB_STATUS(PB0);
    SET_PB_STATUS(PB1);
    SET_PB_STATUS(PB2);
    SET_PB_STATUS(PB3);
#ifndef ZW_DEBUG
#ifndef ZW_DEBUG_CMD
#ifndef ZW_DEBUG_PROTOCOL
    SET_PB_STATUS(PB4);
#endif /*ZW_DEBUG_PROTOCOL*/
#endif
#endif /*ZW_DEBUG*/
  }
  /*Make sure that MOSI is output after we checked the key. The rest is ok as inputs*/
  PIN_OUT(MOSI);

  if(bPressed)
    readyCount++; /*Increment readyCount if PB is down*/
  else
  {
    /*A key has been released, increment debounce*/
    if(bOldPressed&PB_RELEASE)
      readyCount++;
    else if((bOldPressed&PB_MASK))
    {
      /*If last detection resulted in a value. Restart DEBOUNCE*/
      readyCount = 0;
      bPressed = 0;    /*Zero bPressed*/
      bOldPressed |= PB_RELEASE;         /*Set bit, release detect ongoing*/
    }

  }
  if(readyCount == 0xFF)
    readyCount = 0xFE;

  if(readyCount>PB_DEBOUNCE_TIME)
    bOldPressed = FALSE;              /*Zero old values. No need when PB_DEBOUNCE expires*/
  if(readyCount>PB_REPEAT_WAIT_TIME)
    bPressed |= PB_REPEAT;
  if(readyCount>PB_LONG_TIME)
    bPressed |= PB_HELD;
}
/*============================   ButtonPressed   ============================
**    Function description
**     Returns the status of the Push buttons
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE ButtonPressed(void)
{
 if(readyCount>PB_DEBOUNCE_TIME)
   return bPressed;
 else
   return bOldPressed;
}
