/*******************************  SERIALAPPL.C  *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description:       Serial API implementation for Enhanced Z-Wave module
 *
 * Author:            Kim Jakobsen
 *
 * Last Changed By:  $Author: efh $
 * Revision:         $Revision: 22520 $
 * Last Changed:     $Date: 2012-03-20 16:06:58 +0100 (Tue, 20 Mar 2012) $
 *
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <config_app.h>
#include <ZW_basis_api.h>
#ifdef ZW_CONTROLLER
#ifdef ZW_INSTALLER
#include <ZW_controller_installer_api.h>
#else
#ifdef ZW_CONTROLLER_STATIC
#ifdef ZW_CONTROLLER_BRIDGE
#include <ZW_controller_bridge_api.h>
#else
#include <ZW_controller_static_api.h>
#endif /* ZW_CONTROLLER_BRIDGE */
#else
#include <ZW_controller_api.h>
#endif /* ZW_CONTROLLER_STATIC */
#endif /* ZW_INSTALLER */
#endif /* ZW_CONTROLLER */

#ifdef ZW_SLAVE_32
#include <ZW_slave_32_api.h>
#endif

#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>

#ifdef ZW_SLAVE
#ifdef ZW_SLAVE_ROUTING
#include <ZW_slave_routing_api.h>
#else
#include <ZW_slave_api.h>
#endif
#endif

#include <ZW_uart_api.h>
#include <ZW_SerialAPI.h>
#include <UART_buf_io.h>
#include <conhandle.h>
#include <ZW_power_api.h>

#include <ZW_uart_api.h>

/* Include serialappl header file - containing version and */
/* SerialAPI functionality support definitions */
#include <serialappl.h>

/* Basic level definitions */
#define BASIC_ON                                        0xFF
#define BASIC_OFF                                       0x00

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/

/* State vars for ApplicationPoll */
#ifdef ZW_CONTROLLER
static IBYTE state = stateIdle;
#ifdef ZW_CONTROLLER_STATIC
IBYTE retry;
#else
BYTE retry;
#endif
#else
static IIBYTE state = stateIdle, retry = 0;
#endif
/* variable indicating if a connection has been established with an outside host (PC-Controller...) */
//BOOL connected = FALSE;

/* Work variable */
#ifdef ZW_CONTROLLER
IBYTE i;
#else
IIBYTE i;
#endif

static XBYTE retVal;

#define BUF_SIZE_RX 74
#define BUF_SIZE_TX 60
static BYTE workbuf[BUF_SIZE_RX];          /* Used for frames received from remote side */
static XBYTE compl_workbuf[BUF_SIZE_TX];   /* Used for frames send to remote side. */

#ifdef ZW_CONTROLLER
static BYTE groupMask[MAX_NODEMASK_LENGTH]; /* Used for sending nodelist in a multicast */
#endif

/* Queue for frames transmitted to PC - callback, ApplicationCommandHandler, ApplicationControllerUpdate... */
#ifdef ZW_CONTROLLER_BRIDGE
#define MAX_CALLBACK_QUEUE 2
#else
#ifdef ZW_CONTROLLER_SINGLE
#define MAX_CALLBACK_QUEUE 2
#else
#define MAX_CALLBACK_QUEUE 3
#endif
#endif
static struct _callback_element_
{
  BYTE      wCmd;
  BYTE      wLen;
  BYTE      wBuf[BUF_SIZE_TX];
} callbackQueue[MAX_CALLBACK_QUEUE];

static IBYTE callbackOut = 0;
static IBYTE callbackIn = 0;
static IBYTE callbackCnt = 0;

/* Set listening, generic, specific and nodeparm defaults */
#ifdef ZW_CONTROLLER
#ifdef ZW_CONTROLLER_STATIC
static XBYTE applNodeInfo_deviceOptionsMask = APPLICATION_NODEINFO_LISTENING;
#else
static IBYTE applNodeInfo_deviceOptionsMask = APPLICATION_NODEINFO_NOT_LISTENING;
#endif
#else
static IBYTE applNodeInfo_deviceOptionsMask = APPLICATION_NODEINFO_LISTENING;
#endif  /* ZW_CONTROLLER */

#ifdef ZW_CONTROLLER

#ifdef ZW_CONTROLLER_STATIC
static PBYTE applNodeInfo_nodeType_generic = GENERIC_TYPE_STATIC_CONTROLLER;
static PBYTE applNodeInfo_nodeType_specific = SPECIFIC_TYPE_PC_CONTROLLER;
#else
static PBYTE applNodeInfo_nodeType_generic = GENERIC_TYPE_GENERIC_CONTROLLER;
static PBYTE applNodeInfo_nodeType_specific = SPECIFIC_TYPE_PORTABLE_REMOTE_CONTROLLER;
#endif

#else
static PBYTE applNodeInfo_nodeType_generic = GENERIC_TYPE_SWITCH_MULTILEVEL;
static PBYTE applNodeInfo_nodeType_specific = SPECIFIC_TYPE_NOT_USED;
#endif  /* ZW_CONTROLLER */

static PBYTE applNodeInfo_parmLength = 0;
static PBYTE applNodeInfo_nodeParm[APPL_NODEPARM_MAX];

#ifdef ZW_CONTROLLER_BRIDGE
static BOOL applSlaveNodeInfo_deviceOptionsMask = APPLICATION_NODEINFO_LISTENING;
static IBYTE applSlaveNodeInfo_nodeType_generic = GENERIC_TYPE_SWITCH_MULTILEVEL;
static IBYTE applSlaveNodeInfo_nodeType_specific = SPECIFIC_TYPE_NOT_USED;
static IBYTE applSlaveNodeInfo_parmLength = 0;
static PBYTE applSlaveNodeInfo_nodeParm[APPL_SLAVENODEPARM_MAX];

#endif  /* ZW_CONTROLLER_BRIDGE */

/* params used for storing Callback function IDs returned to remote side */
#ifdef ZW_CONTROLLER

#if SUPPORT_ZW_SEND_NODE_INFORMATION
#ifdef ZW_CONTROLLER_STATIC
static XBYTE funcID_ComplHandler_ZW_SendNodeInformation;
#else
static IBYTE funcID_ComplHandler_ZW_SendNodeInformation;
#endif
#endif

#if SUPPORT_ZW_SEND_DATA || SUPPORT_ZW_SEND_DATA_MR || SUPPORT_ZW_SEND_DATA_BRIDGE
static IBYTE funcID_ComplHandler_ZW_SendData;
#endif

#if SUPPORT_ZW_SEND_DATA_META || SUPPORT_ZW_SEND_DATA_META_MR || SUPPORT_ZW_SEND_DATA_META_BRIDGE
static IBYTE funcID_ComplHandler_ZW_SendDataMeta;
#endif

#else /* ZW_CONTROLLER */

#if SUPPORT_ZW_SEND_NODE_INFORMATION
static IIBYTE funcID_ComplHandler_ZW_SendNodeInformation;
#endif

#if SUPPORT_ZW_SEND_DATA || SUPPORT_ZW_SEND_DATA_MR
static IIBYTE funcID_ComplHandler_ZW_SendData;
#endif

#if SUPPORT_ZW_SEND_DATA_META || SUPPORT_ZW_SEND_DATA_META_MR
static IIBYTE funcID_ComplHandler_ZW_SendDataMeta;
#endif

#endif  /* ZW_CONTROLLER */

#if SUPPORT_ZW_REDISCOVERY_NEEDED
static BYTE funcID_ComplHandler_ZW_RediscoveryNeeded;
#endif

#if SUPPORT_ZW_REQUEST_NEW_ROUTE_DESTINATIONS
static BYTE funcID_ComplHandler_ZW_RequestNewRouteDestinations;

#ifdef ZW_SLAVE_32
static BYTE returnRouteDests[5];
#else
static BYTE returnRouteDests[ZW_MAX_RETURN_ROUTE_DESTINATIONS];
#endif
#endif

#if SUPPORT_ZW_SET_SUC_NODE_ID
static BYTE funcID_ComplHandler_ZW_SetSUCNodeID;
#endif

#if SUPPORT_ZW_SEND_DATA_MULTI || SUPPORT_ZW_SEND_DATA_MULTI_BRIDGE
static IBYTE funcID_ComplHandler_ZW_SendDataMulti;
#endif

#if SUPPORT_ZW_SET_LEARN_NODE_STATE
static BYTE funcID_ComplHandler_ZW_SetLearnNodeState;
#endif

#if SUPPORT_ZW_REQUEST_NODE_NEIGHBOR_UPDATE
static BYTE funcID_ComplHandler_ZW_RequestNodeNeighborUpdate;
#endif

#if SUPPORT_ZW_SET_DEFAULT
static BYTE funcID_ComplHandler_ZW_SetDefault;
#endif

#if SUPPORT_ZW_NEW_CONTROLLER
static BYTE funcID_ComplHandler_ZW_NewController;
#else
static BYTE funcID_ComplHandler_ZW_NodeManagement;
static BYTE nodeManagement_Func_ID;
#endif

#if SUPPORT_ZW_REPLICATION_SEND_DATA
static BYTE funcID_ComplHandler_ZW_ReplicationSendData;
#endif

#if SUPPORT_ZW_ASSIGN_RETURN_ROUTE
static BYTE funcID_ComplHandler_ZW_AssignReturnRoute;
#endif

#if SUPPORT_ZW_DELETE_RETURN_ROUTE
static BYTE funcID_ComplHandler_ZW_DeleteReturnRoute;
#endif

#if SUPPORT_ZW_REMOVE_FAILED_NODE_ID
static BYTE funcID_ComplHandler_ZW_RemoveFailedNodeID;
#endif

#if SUPPORT_ZW_REPLACE_FAILED_NODE
static BYTE funcID_ComplHandler_ZW_ReplaceFailedNode;
#endif

#if SUPPORT_ZW_SEND_SUC_ID
static BYTE funcID_ComplHandler_ZW_SendSUC_ID;
#endif

#if SUPPORT_STORE_NODEINFO
static BYTE funcID_ComplHandler_ZW_StoreNodeInfo;
#endif

#if SUPPORT_ZW_SET_LEARN_MODE
#ifdef ZW_SLAVE
static BYTE funcID_ComplHandler_ZW_SetLearnMode;
#endif
#endif

#if SUPPORT_SERIAL_API_APPL_SLAVE_NODE_INFORMATION
static BYTE funcID_ComplHandler_ZW_SendSlaveNodeInformation;
#endif

#if SUPPORT_ZW_SEND_SLAVE_DATA
static BYTE funcID_ComplHandler_ZW_SendSlaveData;
#endif

#if SUPPORT_ZW_SET_SLAVE_LEARN_MODE
static BYTE funcID_ComplHandler_ZW_SetSlaveLearnMode;
#endif

#if SUPPORT_ZW_SEND_TEST_FRAME
static BYTE funcID_ComplHandler_ZW_SendTestFrame;
#endif

#ifdef ZW_CONTROLLER
static IBYTE funcID_ComplHandler_netWork_Management;
static IBYTE management_Func_ID;
#else
static IIBYTE funcID_ComplHandler_netWork_Management;
static IIBYTE management_Func_ID;
#endif

#if SUPPORT_MEMORY_PUT_BUFFER
static BYTE funcID_ComplHandler_MemoryPutBuffer;
#endif

#if SUPPORT_ZW_SEND_DATA_MR || SUPPORT_ZW_SEND_DATA_BRIDGE || SUPPORT_ZW_SEND_DATA_META_MR || SUPPORT_ZW_SEND_DATA_META_BRIDGE
BYTE pRouteBuffer[4];
#endif

BYTE appHomeId[HOMEID_LENGTH];
BYTE appNodeId;

#if SUPPORT_ZW_WATCHDOG_START | SUPPORT_ZW_WATCHDOG_STOP
BYTE bWatchdogStarted;
#endif

BYTE dataLength;

#ifdef ZW_CONTROLLER_SINGLE

#if SUPPORT_SERIAL_API_TEST
ZW_APPLICATION_META_TX_BUFFER txBuffer;

static BYTE funcID_ComplHandler_Serial_API_Test;
XBYTE testnodemask[ZW_MAX_NODES / 8];
XBYTE testresultnodemask[ZW_MAX_NODES / 8];

IBYTE testnodemasklen = 0;
IBYTE timerTestHandle = 0xff;
BYTE testCmd;
BOOL setON = TRUE;
BYTE testNodeID = 0;
WORD testDelay;
WORD testCurrentDelay;
BYTE testPayloadLen;
IWORD testCount;
WORD testSuccessCount;
WORD testFailedCount;
BYTE testState = 0;
BYTE testTxOptions;
#endif  /* SUPPORT_SERIAL_API_TEST */

#endif  /* ZW_CONTROLLER_SINGLE */


/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

/*===============================   PostRequest   ===========================
**    Queues post request to be transmitted to remote side
**
**--------------------------------------------------------------------------*/
void              /*RET  Nothing                  */
PostRequest(
  XBYTE *pData,   /*IN   pointer to data          */
  BYTE len)       /*IN   Length of data.          */
{
  register BYTE callbackIndex = callbackIn - 1;
  if (callbackIndex >= MAX_CALLBACK_QUEUE)
  {
    callbackIndex = MAX_CALLBACK_QUEUE - 1;
  }
  /*Check data-length not exceed buffer size*/
  if (((callbackQueue[callbackIndex].wLen + len) < BUF_SIZE_TX) && (pData != NULL))
  {
    /* TO#3532 fix - only copy len bytes */
    for(i = 0; i < len; i++)
    {
      callbackQueue[callbackIndex].wBuf[callbackQueue[callbackIndex].wLen+i] = BYTE_IN_AR(pData, i);
    }
    callbackQueue[callbackIndex].wLen += len;
  }
}


/*===============================   PreRequestPar1   ==========================
**    Queues request 2 parameters to be transmitted to remote side
**
**--------------------------------------------------------------------------*/
BOOL              /*RET  status                  */
PreRequestPar1(
  BYTE cmd,       /*IN   Command                  */
  BYTE par0)      /*IN   Parameter 0              */
{
  if (callbackCnt < MAX_CALLBACK_QUEUE)
  {
    callbackCnt++;
    callbackQueue[callbackIn].wCmd = cmd;
    callbackQueue[callbackIn].wBuf[0] = par0;
    callbackQueue[callbackIn].wLen = 1;
    if (++callbackIn >= MAX_CALLBACK_QUEUE)
    {
      callbackIn = 0;
    }
    return TRUE;
  }
  return FALSE;
}

/*===============================   PreRequestPar2  ==========================
**    Queues request 2 parameters to be transmitted to remote side
**
**--------------------------------------------------------------------------*/
BOOL              /*RET  status                  */
PreRequestPar2(
  BYTE cmd,       /*IN   Command                  */
  BYTE par0,      /*IN   Parameter 0              */
  BYTE par1)      /*IN   Parameter 1              */
{
  if (callbackCnt < MAX_CALLBACK_QUEUE)
  {
    callbackCnt++;
    callbackQueue[callbackIn].wCmd = cmd;
    callbackQueue[callbackIn].wBuf[0] = par0;
    callbackQueue[callbackIn].wBuf[1] = par1;
    callbackQueue[callbackIn].wLen = 2;
    if (++callbackIn >= MAX_CALLBACK_QUEUE)
    {
      callbackIn = 0;
    }
    return TRUE;
  }
  return FALSE;
}

/*===============================   PreRequestPar3  ==========================
**    Queues request 3 parameters to be transmitted to remote side
**
**--------------------------------------------------------------------------*/
BOOL              /*RET  status                  */
PreRequestPar3(
  BYTE cmd,       /*IN   Command                  */
  BYTE par0,      /*IN   Parameter 0              */
  BYTE par1,      /*IN   Parameter 1              */
  BYTE par2)      /*IN   Parameter 2              */
{
  if ( TRUE == PreRequestPar2(cmd, par0, par1))
  {
    PostRequest(&par2, 1);
    return TRUE;
  }
  return FALSE;
}


/*===============================   Respond   ===============================
**    Send immediate respons to remote side
**
**    Side effects: Sets state variable to stateTxSerial (wait for ack)
**
**--------------------------------------------------------------------------*/
void              /*RET  Nothing                 */
Respond(
  BYTE cmd,       /* IN   Command                  */
  XBYTE *pData,   /* IN   pointer to data          */
  BYTE len)       /* IN   Length of data           */
{
  /* If there are no data; pData == NULL and len == 0 we must set the data pointer */
  /* to some dummy data. ConTxFrame interprets NULL pointer as retransmit indication */
  if (len == 0)
  {
    pData = (XBYTE *)0x1; /* Just something is not used anyway */
  }
  ConTxFrame(cmd, RESPONSE, pData, len);

  state = stateTxSerial;  /* We want ACK/NAK...*/
}


#if SUPPORT_ZW_SEND_NODE_INFORMATION
/*=====================   ComplHandler_ZW_SendNodeInformation   =============
**    Completion handler for ZW_SendNodeInformation
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendNodeInformation(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
  PreRequestPar2(FUNC_ID_ZW_SEND_NODE_INFORMATION, funcID_ComplHandler_ZW_SendNodeInformation, txStatus);
}
#endif


#if SUPPORT_ZW_SEND_DATA
/*======================   ComplHandler_ZW_SendData   ========================
**    Completion handler for ZW_SendData
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendData(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
  /* NOTE - In devkit 4.5x more Transmit completion status has been introduced, */
  /* TRANSMIT_COMPLETE_HOP_0_FAIL, TRANSMIT_COMPLETE_HOP_1_FAIL, */
  /* TRANSMIT_COMPLETE_HOP_2_FAIL, TRANSMIT_COMPLETE_HOP_3_FAIL and */
  /* TRANSMIT_COMPLETE_HOP_4_FAIL these must be converted to TRANSMIT_COMPLETE_FAIL */
  /* to simulate old ZW_SendData transmit completion more correctly */
  PreRequestPar2(FUNC_ID_ZW_SEND_DATA,
                funcID_ComplHandler_ZW_SendData,
                (txStatus < TRANSMIT_COMPLETE_HOP_0_FAIL) ? txStatus : TRANSMIT_COMPLETE_NO_ACK);
}
#endif  /* SUPPORT_ZW_SEND_DATA */


#if SUPPORT_ZW_SEND_DATA_META
/*====================   ComplHandler_ZW_SendDataMeta   ======================
**    Completion handler for ZW_SendDataMeta
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendDataMeta(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
  /* NOTE - In devkit 4.5x more Transmit completion status has been introduced, */
  /* TRANSMIT_COMPLETE_HOP_0_FAIL, TRANSMIT_COMPLETE_HOP_1_FAIL, */
  /* TRANSMIT_COMPLETE_HOP_2_FAIL, TRANSMIT_COMPLETE_HOP_3_FAIL and */
  /* TRANSMIT_COMPLETE_HOP_4_FAIL these must be converted to TRANSMIT_COMPLETE_NO_ACK */
  /* to simulate old ZW_SendData transmit completion more correctly */
  PreRequestPar2(FUNC_ID_ZW_SEND_DATA_META,
                funcID_ComplHandler_ZW_SendDataMeta,
                (txStatus < TRANSMIT_COMPLETE_HOP_0_FAIL) ? txStatus : TRANSMIT_COMPLETE_NO_ACK);
}
#endif


#if SUPPORT_ZW_SEND_DATA_MULTI
/*=====================   ComplHandler_ZW_SendDataMulti   ====================
**    Completion handler for ZW_SendDataMulti
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendDataMulti(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
  /* NOTE - In devkit 4.5x more Transmit completion status has been introduced, */
  /* TRANSMIT_COMPLETE_HOP_0_FAIL, TRANSMIT_COMPLETE_HOP_1_FAIL, */
  /* TRANSMIT_COMPLETE_HOP_2_FAIL, TRANSMIT_COMPLETE_HOP_3_FAIL and */
  /* TRANSMIT_COMPLETE_HOP_4_FAIL these must be converted to TRANSMIT_COMPLETE_FAIL */
  /* to simulate old ZW_SendData transmit completion more correctly */
  PreRequestPar2(FUNC_ID_ZW_SEND_DATA_MULTI,
                funcID_ComplHandler_ZW_SendDataMulti,
                (txStatus < TRANSMIT_COMPLETE_HOP_0_FAIL) ? txStatus : TRANSMIT_COMPLETE_NO_ACK);
}
#endif


#if SUPPORT_ZW_SEND_DATA_MR
/*=====================   ComplHandler_ZW_SendDataMR   =======================
**    Completion handler for ZW_SendDataMR
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendDataMR(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
    PreRequestPar2(FUNC_ID_ZW_SEND_DATA_MR, funcID_ComplHandler_ZW_SendData, txStatus);
}
#endif


#if SUPPORT_ZW_SEND_DATA_META_MR
/*===================   ComplHandler_ZW_SendDataMetaMR   =====================
**    Completion handler for ZW_SendDataMeta
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendDataMetaMR(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
  PreRequestPar2(FUNC_ID_ZW_SEND_DATA_META_MR, funcID_ComplHandler_ZW_SendDataMeta, txStatus);
}
#endif


#if SUPPORT_ZW_SEND_DATA_BRIDGE
/*=================   ComplHandler_ZW_SendData_Bridge   ======================
**    Completion handler for ZW_SendData_Bridge
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendData_Bridge(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
  PreRequestPar2(FUNC_ID_ZW_SEND_DATA_BRIDGE, funcID_ComplHandler_ZW_SendData, txStatus);
}
#endif


#if SUPPORT_ZW_SEND_DATA_META_BRIDGE
/*===============   ComplHandler_ZW_SendDataMeta_Bridge   ====================
**    Completion handler for ZW_SendDataMeta_Bridge
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendDataMeta_Bridge(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
  PreRequestPar2(FUNC_ID_ZW_SEND_DATA_META_BRIDGE, funcID_ComplHandler_ZW_SendDataMeta, txStatus);
}
#endif


#if SUPPORT_ZW_SEND_DATA_MULTI_BRIDGE
/*================   ComplHandler_ZW_SendDataMulti_Bridge   ==================
**    Completion handler for ZW_SendDataMulti
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendDataMulti_Bridge(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
  PreRequestPar2(FUNC_ID_ZW_SEND_DATA_MULTI_BRIDGE,funcID_ComplHandler_ZW_SendDataMulti, txStatus);
}
#endif


#if SUPPORT_STORE_NODEINFO
/*===============   ComplHandler_ZW_StoreNodeInfo  ===============
**    Completion handler for ZW_STORE_NODEINFO
**
**--------------------------------------------------------------------------*/
static void
ComplHandler_ZW_StoreNodeInfo(void)
{
  PreRequestPar1(FUNC_ID_STORE_NODEINFO,funcID_ComplHandler_ZW_StoreNodeInfo);
}
#endif /* SUPPORT_STORE_NODEINFO */


#if SUPPORT_ZW_SET_SUC_NODE_ID
/*====================   ComplHandler_ZW_SetSUCNodeID   ======================
**    Function description
**    ZW_SUC_SET_SUCCEEDED
**    Side effects:
**
**--------------------------------------------------------------------------*/
static void
ComplHandler_ZW_SetSUCNodeID(
  BYTE txStatus)          /*IN   Completion status*/
{
  PreRequestPar2(FUNC_ID_ZW_SET_SUC_NODE_ID, funcID_ComplHandler_ZW_SetSUCNodeID, txStatus);
}
#endif


#if SUPPORT_ZW_REQUEST_NODE_INFO
/*====================== ComplHandler_ZW_RequestNodeInfo =====================
**    Completion handler for ZW_RequestNodeInfo
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_RequestNodeInfo(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
  /* */
#ifdef ZW_CONTROLLER
  if (txStatus != TRANSMIT_COMPLETE_OK)
  {
    ApplicationControllerUpdate(UPDATE_STATE_NODE_INFO_REQ_FAILED, 0, NULL, 0);
  }
#else
  if (txStatus != TRANSMIT_COMPLETE_OK)
  {
    ApplicationSlaveUpdate(UPDATE_STATE_NODE_INFO_REQ_FAILED, 0, NULL, 0);
  }
#endif
}
#endif


#if SUPPORT_ZW_SET_LEARN_NODE_STATE
/*=====================   ComplHandler_ZW_SetLearnNodeState   ================
**    Completion handler for ZW_SetLearnNodeState
**
**--------------------------------------------------------------------------*/
static void                         /* RET  Nothing                         */
ComplHandler_ZW_SetLearnNodeState(
  LEARN_INFO *learnNodeInfo)
{
  if(TRUE == PreRequestPar3(FUNC_ID_ZW_SET_LEARN_NODE_STATE,
                            funcID_ComplHandler_ZW_SetLearnNodeState,
                            (*learnNodeInfo).bStatus,
                            (*learnNodeInfo).bSource)
  {
    PostRequest(&(learnNodeInfo->bLen), 1);
    PostRequest(learnNodeInfo->pCmd, (*learnNodeInfo).bLen));
  }
}
#endif


#if SUPPORT_ZW_REQUEST_NODE_NEIGHBOR_UPDATE
/*===============   ComplHandler_ZW_RequestNodeNeighborUpdate  ===============
**    Completion handler for ZW_REQUEST_NODE_NEIGHBOR_UPDATE
**
**--------------------------------------------------------------------------*/
static void                                 /* RET  Nothing                    */
ComplHandler_ZW_RequestNodeNeighborUpdate(
  BYTE txStatus)                              /* IN   Transmit completion status */
{
  PreRequestPar2(FUNC_ID_ZW_REQUEST_NODE_NEIGHBOR_UPDATE, funcID_ComplHandler_ZW_RequestNodeNeighborUpdate, txStatus);
}
#endif


#if SUPPORT_ZW_SET_DEFAULT
/*=====================   ComplHandler_ZW_SetDefault   =============
**    Completion handler for ZW_SetDefault
**
**--------------------------------------------------------------------------*/
static void                            /* RET  Nothing */
ComplHandler_ZW_SetDefault(void)    /* IN   Nothing */
{
  PreRequestPar1(FUNC_ID_ZW_SET_DEFAULT,funcID_ComplHandler_ZW_SetDefault);
}
#endif  /* SUPPORT_ZW_SET_DEFAULT */


#ifdef ZW_CONTROLLER
/*=======================   ComplHandler_ZW_NodeManagement   =================
**    Completion handler for ZW_AddNodeToNetwork, ZW_RemoveNodeFromNetwork
**    ZW_CreateNewPrimary, ZW_ControllerChange and ZW_SetLearnMode
**
**--------------------------------------------------------------------------*/
static void                                /* RET  Nothing */
ComplHandler_ZW_NodeManagement(
  LEARN_INFO *learnNodeInfo)
{
  if(TRUE == PreRequestPar3(nodeManagement_Func_ID,
                            funcID_ComplHandler_ZW_NodeManagement,
                            (*learnNodeInfo).bStatus,
                            (*learnNodeInfo).bSource))
  {
    PostRequest(&(learnNodeInfo->bLen),1);
    PostRequest(learnNodeInfo->pCmd, (*learnNodeInfo).bLen);
  }
}
#endif /* ZW_CONTROLLER */


#if SUPPORT_ZW_REPLICATION_SEND_DATA
/*=====================   ComplHandler_ZW_ReplicationSendData   =============
**    Completion handler for ZW_ReplicationSendData
**
**--------------------------------------------------------------------------*/
static void                                    /* RET  Nothing */
ComplHandler_ZW_ReplicationSendData(
  BYTE txStatus)                          /* IN   Transmit completion status */
{
  PreRequestPar2(FUNC_ID_ZW_REPLICATION_SEND_DATA, funcID_ComplHandler_ZW_ReplicationSendData, txStatus);
}
#endif


#if SUPPORT_ZW_ASSIGN_RETURN_ROUTE
/*=====================   ComplHandler_ZW_AssignReturnRoute   =============
**    Completion handler for ZW_AssignReturnRoute
**
**--------------------------------------------------------------------------*/
void                                    /* RET  Nothing                     */
ComplHandler_ZW_AssignReturnRoute(
  BYTE bStatus)                          /* IN   Transmit completion status  */
{
  PreRequestPar2(FUNC_ID_ZW_ASSIGN_RETURN_ROUTE, funcID_ComplHandler_ZW_AssignReturnRoute, bStatus);
}
#endif


#if SUPPORT_ZW_SEND_SUC_ID
void
ComplHandler_ZW_SendSUC_ID(
  BYTE bStatus)
{
  PreRequestPar2(FUNC_ID_ZW_SEND_SUC_ID, funcID_ComplHandler_ZW_SendSUC_ID, bStatus);
}
#endif


#if SUPPORT_ZW_DELETE_RETURN_ROUTE
/*=====================   ComplHandler_ZW_DeleteReturnRoute   =============
**    Completion handler for ZW_DeleteReturnRoute
**
**--------------------------------------------------------------------------*/
void                                    /* RET  Nothing                     */
ComplHandler_ZW_DeleteReturnRoute(
  BYTE bStatus)                          /* IN   Transmit completion status  */
{
  PreRequestPar2(FUNC_ID_ZW_DELETE_RETURN_ROUTE, funcID_ComplHandler_ZW_DeleteReturnRoute, bStatus);
}
#endif


#if SUPPORT_ZW_REMOVE_FAILED_NODE_ID
/*=====================   ComplHandler_ZW_RemoveFailedNodeID   ==============
**    Completion handler for ZW_RemoveFailedNodeID
**
**--------------------------------------------------------------------------*/
void                                    /* RET  Nothing                     */
ComplHandler_ZW_RemoveFailedNodeID(
  BYTE bStatus)                          /* IN   Transmit completion status  */
{
  PreRequestPar2(FUNC_ID_ZW_REMOVE_FAILED_NODE_ID, funcID_ComplHandler_ZW_RemoveFailedNodeID, bStatus);
}
#endif


#if SUPPORT_ZW_REPLACE_FAILED_NODE
/*=====================   ComplHandler_ZW_RemoveFailedNodeID   ==============
**    Completion handler for ZW_RemoveFailedNodeID
**
**--------------------------------------------------------------------------*/
void                                    /* RET  Nothing                     */
ComplHandler_ZW_ReplaceFailedNode(
  BYTE bStatus)                          /* IN   Transmit completion status  */
{
  PreRequestPar2(FUNC_ID_ZW_REPLACE_FAILED_NODE, funcID_ComplHandler_ZW_ReplaceFailedNode, bStatus);
}
#endif


/*=====================   ComplHandler_ZW_netWork_Management   ===============
**    Completion handler for the network management functionality
**
**--------------------------------------------------------------------------*/
#if defined(ZW_SLAVE_32) || defined(ZW_CONTROLLER) || defined(ZW_SLAVE_ROUTING)
void                                    /* RET  Nothing                     */
ComplHandler_ZW_netWork_Management(
  BYTE bStatus)                          /* IN   Transmit completion status  */
{
  PreRequestPar2(management_Func_ID, funcID_ComplHandler_netWork_Management, bStatus);
}
#endif /*ZW_SLAVE_32 ZW_CONTROLLER*/


#ifdef ZW_SLAVE

#if SUPPORT_ZW_SET_LEARN_MODE
/*=========================   ComplHandler_ZW_SetLearnMode   =================
**    Completion handler for ZW_SetLearnMode
**
**--------------------------------------------------------------------------*/
static void                             /*RET Nothing                       */
ComplHandler_ZW_SetLearnMode(
  BYTE bStatus,                         /*IN  ZW_SetLearnMode status        */
  BYTE bNodeID)                         /*IN  Node ID                       */
{
  PreRequestPar3(FUNC_ID_ZW_SET_LEARN_MODE, funcID_ComplHandler_ZW_SetLearnMode, bStatus, bNodeID);
}
#endif  /* SUPPORT_ZW_SET_LEARN_MODE */


#if SUPPORT_ZW_REQUEST_NEW_ROUTE_DESTINATIONS
#ifdef ZW_SLAVE_ROUTING
/*===============   ComplHandler_ZW_RequestNewRouteDestinations   ============
**    Completion handler for ZW_RequestNewRouteDestinations
**
**--------------------------------------------------------------------------*/
void                                    /* RET  Nothing                     */
ComplHandler_ZW_RequestNewRouteDestinations(
  BYTE bStatus)                          /* IN  Function status */
{
  PreRequestPar2(FUNC_ID_ZW_REQUEST_NEW_ROUTE_DESTINATIONS, funcID_ComplHandler_ZW_RequestNewRouteDestinations, bStatus);
}
#endif  /* ZW_SLAVE_ROUTING */
#endif  /* SUPPORT_ZW_REQUEST_NEW_ROUTE_DESTINATIONS */

#endif /* ZW_SLAVE */


#if SUPPORT_ZW_SET_SLAVE_LEARN_MODE
/*=================   ComplHandler_ZW_SetSlaveLearnMode   ====================
**    Completion handler for ZW_SetSlaveLearnMode
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SetSlaveLearnMode(
  BYTE bStatus,
  BYTE orgID,
  BYTE newID)                           /*  IN  Node ID                     */
{
  if(TRUE == PreRequestPar3(FUNC_ID_ZW_SET_SLAVE_LEARN_MODE,
                  funcID_ComplHandler_ZW_SetSlaveLearnMode,
                  bStatus,
                  orgID))
  {
    PostRequest(&newID,1);
  }
}
#endif


#if SUPPORT_ZW_SEND_TEST_FRAME
/*=================    ComplHandler_ZW_SendTestFrame    ====================
**    Completion handler for ZW_SendTestFrame
**
**--------------------------------------------------------------------------*/
static void
ComplHandler_ZW_SendTestFrame(
  BYTE txStatus)
{
  PreRequestPar2(FUNC_ID_ZW_SEND_TEST_FRAME, funcID_ComplHandler_ZW_SendTestFrame, txStatus);
}
#endif


#if SUPPORT_SERIAL_API_APPL_SLAVE_NODE_INFORMATION
/*=================   ComplHandler_ZW_SendSlaveNodeInformation   ============
**    Completion handler for ZW_SendSlaveNodeInformation
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendSlaveNodeInformation(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
    PreRequestPar2(FUNC_ID_ZW_SEND_SLAVE_NODE_INFORMATION, funcID_ComplHandler_ZW_SendSlaveNodeInformation, txStatus);
}
#endif


#if SUPPORT_ZW_SEND_SLAVE_DATA
/*===================   ComplHandler_ZW_SendSlaveData   ======================
**    Completion handler for ZW_SendData
**
**--------------------------------------------------------------------------*/
static void                             /* RET  Nothing                     */
ComplHandler_ZW_SendSlaveData(
  BYTE txStatus)                        /* IN   Transmit completion status  */
{
  PreRequestPar2(FUNC_ID_ZW_SEND_SLAVE_DATA, funcID_ComplHandler_ZW_SendSlaveData, txStatus);
}
#endif


#if SUPPORT_MEMORY_PUT_BUFFER
/*=====================   ComplHandler_MemoryPutBuffer   =============
**    Completion handler for MemoryPutBuffer
**
**--------------------------------------------------------------------------*/
static void                                /* RET  Nothing */
ComplHandler_MemoryPutBuffer(void)  /* IN   Nothing */
{
  PreRequestPar1(FUNC_ID_MEMORY_PUT_BUFFER, funcID_ComplHandler_MemoryPutBuffer);
}
#endif


void
DoRespond( void )
{
  Respond(serFrameCmd, &retVal, 1);
}


void
DoRespond_workbuf(
  BYTE cnt)
{
  Respond(serFrameCmd, compl_workbuf, cnt);
}

#if SUPPORT_ZW_REDISCOVERY_NEEDED
void
ComplHandler_ZW_RediscoveryNeeded(
  BYTE status)
{
  PreRequestPar2(FUNC_ID_ZW_REDISCOVERY_NEEDED, funcID_ComplHandler_ZW_RediscoveryNeeded, status);
}
#endif


#ifdef ZW_CONTROLLER
/*===========================  SetupNodeManagement  ==========================
**    Function description
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
SetupNodeManagement(void)
{
  nodeManagement_Func_ID = serFrameCmd;
  funcID_ComplHandler_ZW_NodeManagement = serFrameDataPtr[1];
  state = stateIdle;
}
#endif


#ifdef ZW_CONTROLLER_SINGLE
#if SUPPORT_SERIAL_API_TEST
void
TestStartRound(void);

void
TestSend(void);


/*==============================  SendTestReport  ============================
**    Function description
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
SendTestReport(
  BYTE txStatus)
{
  if (funcID_ComplHandler_Serial_API_Test != 0)
  {
    if(TRUE == PreRequestPar3(FUNC_ID_SERIAL_API_TEST,
                              funcID_ComplHandler_Serial_API_Test,
                              testCmd,
                              testState))
    {
      PostRequest(&testNodeID,1);
      PostRequest(&txStatus, 1);
      retVal = BYTE_GET_LOW_BYTE_IN_WORD(testCount);
      PostRequest(&retVal, 1);
    }
  }
}


/*===========================  SendTestRoundReport  ==========================
**    Function description
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
SendTestRoundReport(
  BYTE txStatus)
{
  if (funcID_ComplHandler_Serial_API_Test != 0)
  {
    if(TRUE == PreRequestPar3(FUNC_ID_SERIAL_API_TEST,
                              funcID_ComplHandler_Serial_API_Test,
                              testCmd,
                              testState))
    {
      retVal = BYTE_GET_HIGH_BYTE_IN_WORD(testCount);
      PostRequest(&retVal, 1);
      retVal = BYTE_GET_LOW_BYTE_IN_WORD(testCount);
      PostRequest(&retVal, 1);
      PostRequest(&testnodemasklen, 1);
      /* Initialy we assume every node acked, so we assume no nodemask is to be send */
      if (ZW_NODE_MASK_BITS_IN(testresultnodemask, testnodemasklen))
      {
        PostRequest(testresultnodemask, testnodemasklen);
      }
    }
  }
}


/*=============================  TestDelayTimeout  ===========================
**    Function description
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
TestDelayTimeout(void)
{
  if (--testCurrentDelay == 0)
  {
    if (timerTestHandle != 0xff)
    {
      TimerCancel(timerTestHandle);
    }
    timerTestHandle = 0xff;
    setON = !setON;
    TestStartRound();
  }
}


/*=============================  TestFindNextNode  ===========================
**    Function description
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL
TestFindNextNode(void)
{
  do
  {
    if (ZW_NodeMaskNodeIn(testnodemask, ++testNodeID))
    {
      return TRUE;
    }
  } while (testNodeID < ZW_MAX_NODES);
  return FALSE;
}


/*=============================  TestSendComplete  ===========================
**    Function description
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
TestSendComplete(
  BYTE bStatus)
{
  ZW_DEBUG_SEND_BYTE('C');
  ZW_DEBUG_SEND_NUM(bStatus);

  if (bStatus == TRANSMIT_COMPLETE_OK)
  {
    testSuccessCount++;
  }
  else
  {
    /* Set bit indicating that node failed to acknowledge */
    ZW_NODE_MASK_SET_BIT(testresultnodemask, testNodeID);
    testFailedCount++;
  }
  /* Should we transmit result (to host) after every transmit or do we send one frame after every round? */
  if (testCmd < 0x05)
  {
    /* One result frame after every transmit */
    SendTestReport(bStatus);
  }
  if (testState == POWERLEVEL_TEST_NODE_REPORT_ZW_TEST_INPROGRESS)
  {
    if (TestFindNextNode())
    {
      TestSend();
    }
    else
    {
      if (testCmd >= 0x05)
      {
        /* One result frame after every round */
        SendTestRoundReport(0);
      }
      /* No more nodes in this round - delay (if any delay to be done) before starting next round */
      if (testCount && (--testCount != 0))
      {
        testCurrentDelay = testDelay;
        if (timerTestHandle != 0xff)
        {
          TimerCancel(timerTestHandle);
        }
        if (!testCurrentDelay )
        {
          testCurrentDelay++;
          TestDelayTimeout();
        }
        else
        {
          timerTestHandle = TimerStart(TestDelayTimeout, 1, TIMER_FOREVER);
        }
      }
      else
      {
        testState = POWERLEVEL_TEST_NODE_REPORT_ZW_TEST_SUCCES;
        SendTestReport(0);
      }
    }
  }
  else
  {
    if (timerTestHandle != 0xff)
    {
      TimerCancel(timerTestHandle);
    }
    SendTestReport(0);
  }
}


/*=================================  TestSend  ===============================
**    Function description
**        Send the next data frame
**
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
TestSend(void)
{
  register BYTE payLoadLen;
  ZW_DEBUG_SEND_BYTE('N');

  if ((testCmd == 0x03) || (testCmd == 0x04) || (testCmd == 0x07) || (testCmd == 0x08))
  {
    txBuffer.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
    txBuffer.ZW_BasicSetFrame.cmd      = BASIC_SET;
    txBuffer.ZW_BasicSetFrame.value    = setON ? BASIC_ON : BASIC_OFF;
    payLoadLen = sizeof(txBuffer.ZW_BasicSetFrame);
  }
  else
  {
    /* TO#1672 temporary fix - Manufacturer Specific command class has disappear from ZW_classcmd.h */
    txBuffer.ZW_ManufacturerSpecificReportFrame.cmdClass = COMMAND_CLASS_MANUFACTURER_PROPRIETARY;
    txBuffer.ZW_ManufacturerSpecificReportFrame.manufacturerId1 = 0x00;
    txBuffer.ZW_ManufacturerSpecificReportFrame.manufacturerId2 = 0x00;
    txBuffer.ZW_ManufacturerSpecificReportFrame.productId1 = 0x04;
    txBuffer.ZW_ManufacturerSpecificReportFrame.productId2 = 0x81;
    *(&(txBuffer.ZW_ManufacturerSpecificReportFrame.productId2) + 1) = 0x00;
    *(&(txBuffer.ZW_ManufacturerSpecificReportFrame.productId2) + 2) = BYTE_GET_HIGH_BYTE_IN_WORD(testCount); //data1
    *(&(txBuffer.ZW_ManufacturerSpecificReportFrame.productId2) + 3) = BYTE_GET_LOW_BYTE_IN_WORD(testCount);  //data1+1
    payLoadLen = sizeof(txBuffer.ZW_ManufacturerSpecificReportFrame);
  }
  /* Send meta data frame */
#if SUPPORT_ZW_SEND_DATA_META
  /* Do we use ZW_SendDataMeta */
  if ((testCmd & 0x01) == 0)
  {
    if (!ZW_SEND_DATA_META(testNodeID,
                           (BYTE*)&txBuffer,
                           (testPayloadLen > payLoadLen) ? testPayloadLen : payLoadLen,
                           testTxOptions,
                           TestSendComplete))
    {
      TestSendComplete(TRANSMIT_COMPLETE_NO_ACK);
    }
  }
  else
#endif
  {
    if (!ZW_SEND_DATA(testNodeID,
                      (BYTE*)&txBuffer,
                      (testPayloadLen > payLoadLen) ? testPayloadLen : payLoadLen,
                      testTxOptions,
                      TestSendComplete))
    {
      TestSendComplete(TRANSMIT_COMPLETE_NO_ACK);
    }
  }
}


/*============================   TestStartRound   ============================
**    Start a Test round
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
TestStartRound(void)
{
  register BYTE bTemp;

  ZW_NODE_MASK_CLEAR(testresultnodemask, ZW_MAX_NODES / 8);
  /* Fill the meta data frame with data */
  /* TO#1672 temporary fix - Manufacturer Specific command class has disappear from ZW_classcmd.h */
  for (bTemp = 0; bTemp < (META_DATA_MAX_DATA_SIZE - sizeof(ZW_MANUFACTURER_SPECIFIC_REPORT_FRAME)); bTemp++)
  {
      *(&(txBuffer.ZW_ManufacturerSpecificReportFrame.productId2) + 1 + bTemp) = bTemp + 1; // was .ZW_MetaDataReportFrame.data1
  }
  testNodeID = 0;
  /* Find first node to transmit to */
  if ((testState == POWERLEVEL_TEST_NODE_REPORT_ZW_TEST_INPROGRESS) && TestFindNextNode())
  {
    /* Found a node */
    TestSend();
  }
  else
  {
    testState = POWERLEVEL_TEST_NODE_REPORT_ZW_TEST_FAILED;
    SendTestReport(0);
  }
}
#endif
#endif


/*======================   SaveApplicationSettings   =========================
**    Save Application specifics
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
SaveApplicationSettings(void)
{
  /* First make NV-mem app. data invalid */
//  ZW_MEM_PUT_BYTE(EEOFFSET_MAGIC, 0);
#ifdef ZW_SLAVE
  ZW_MEM_PUT_BYTE(EEOFFSET_LISTENING, applNodeInfo_deviceOptionsMask);
  ZW_MEM_PUT_BYTE(EEOFFSET_GENERIC, applNodeInfo_nodeType_generic);
  ZW_MEM_PUT_BYTE(EEOFFSET_SPECIFIC, applNodeInfo_nodeType_specific);
#endif
  if (applNodeInfo_parmLength > APPL_NODEPARM_MAX)
  {
    applNodeInfo_parmLength = APPL_NODEPARM_MAX;
  }
  ZW_MEM_PUT_BYTE(EEOFFSET_CMDCLASSS_LEN, applNodeInfo_parmLength);
  for (i = 0; i < applNodeInfo_parmLength; i++)
  {
    ZW_MEM_PUT_BYTE(EEOFFSET_CMDCLASS + i, applNodeInfo_nodeParm[i]);
  }
#if SUPPORT_ZW_WATCHDOG_START || SUPPORT_ZW_WATCHDOG_STOP
  ZW_MEM_PUT_BYTE(EEOFFSET_WATCHDOG_STARTED, bWatchdogStarted);
#endif
  /* Now app. data is valid! */
  ZW_MEM_PUT_BYTE(EEOFFSET_MAGIC, MAGIC_VALUE);
}


void
PopCallBackQueue(void);


/*===============================   ApplicationPoll   =======================
**    Application poll function, handling the receiving and transmitting
**    communication with the PC.
**
**--------------------------------------------------------------------------*/
void           /*RET  Nothing                  */
ApplicationPoll( void )  /*IN  Nothing                   */
{
  IBYTE conVal;
#if SUPPORT_ZW_GET_RANDOM
  register BYTE noRndBytes;
  register WORD rndWord;
#endif

  /* ApplicationPoll is controlled by a statemachine with the four states:
      stateIdle, stateFrameParse, stateTxSerial, stateCbTxSerial.

      stateIdle: If there is anything to transmit do so. -> stateCbTxSerial
                 If not, check if anything is received. -> stateFrameParse
                 If neither, stay in the state
                 Note: frames received while we are transmitting are lost
                 and must be retransmitted by PC

      stateFrameParse: Parse received frame.
                 If the request has no response -> stateIdle
                 If there is an immediate response send it. -> stateTxSerial

      stateTxSerial:  Waits for ack on responses send in stateFrameParse.
                 Retransmit frame as needed.
                 -> stateIdle

      stateCbTxSerial:  Waits for ack on requests send in stateIdle
                  (callback, ApplicationCommandHandler etc).
                 Retransmit frame as needed and remove from callbackqueue when done.
                 -> stateIdle

      The serialapi sample code uses the LEDs as indicator for state (if ENABLE_LEDS is defined)
      and the following are defined:
      LED1 ON, LED2 OFF, LED3 ON - Idle (waiting for action) - stateIdle.
      LED1 OFF, LED2 ON, LED3 OFF - A command has been received from HOST
                and this is being executed (a Response frame will also be send if needed) - stateFrameParse.
      LED1 OFF, LED2 OFF, LED3 ON - Waiting for ACK for Response frame transmitted - stateTxSerial.
      LED1 ON, LED2 ON, LED3 OFF - A Request is transmitted to HOST (next state is
                Waiting for ACK for Request - in stateIdle if any Requests in queue)
      LED1 OFF, LED2 ON, LED3 ON - Waiting for ACK for Request frame transmitted - stateCbTxSerial.

  */

#if SUPPORT_ZW_WATCHDOG_START || SUPPORT_ZW_WATCHDOG_STOP
  if (bWatchdogStarted)
  {
    /* If watchdog enabled - kick it... */
    ZW_WATCHDOG_KICK;
  }
#endif

  switch (state)
  {
    case stateIdle :
    {
      /* Check if there is anything to transmit. If so do it */
      if (callbackCnt)
      {
#ifdef ENABLE_LEDS
        LED_ON(1);
        LED_ON(2);
        LED_OFF(3);
#endif
        ConTxFrame(callbackQueue[callbackOut].wCmd, REQUEST, (XBYTE*)callbackQueue[callbackOut].wBuf, callbackQueue[callbackOut].wLen);
        state = stateCbTxSerial;
        /* callbackCnt decremented when frame is acknowledged from PC - or timed out after retries */
      }
      else
      {
#ifdef ENABLE_LEDS
        LED_ON(1);
        LED_OFF(2);
        LED_ON(3);
#endif
        /* Nothing to transmit. Check if we received anything */
        if (ConUpdate(TRUE) == conFrameReceived)
        {
          /* We got a frame... */
          state = stateFrameParse;
        }
      }
    }
    break;

    case stateFrameParse :
    {
#ifdef ENABLE_LEDS
      LED_OFF(1);
      LED_ON(2);
      LED_OFF(3);
#endif
      /* Parse received frame */
      switch (serFrameCmd)
      {
#if SUPPORT_ZW_EXPLORE_REQUEST_INCLUSION
        case FUNC_ID_ZW_EXPLORE_REQUEST_INCLUSION:
#if 0
          /* TO#3355 fix - Return retval to HOST */
          retVal = ZW_ExploreRequestInclusion();
          DoRespond();
#else
          ZW_ExploreRequestInclusion();
          state = stateIdle;
#endif
          break;
#endif

#if SUPPORT_ZW_SET_RF_RECEIVE_MODE
        case FUNC_ID_ZW_SET_RF_RECEIVE_MODE:
          /* HOST->ZW: mode */
          retVal = ZW_SetRFReceiveMode(serFrameDataPtr[0]);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_SET_RF_RECEIVE_MODE */

#if SUPPORT_ZW_REDISCOVERY_NEEDED
        case FUNC_ID_ZW_REDISCOVERY_NEEDED:
          /* HOST->ZW: destNode | funcId */
          /* TO#1734 - Fix: ZW_RediscoveryNeeded() returns wrong callbacks upon completion due to missing "Transfer End" frame */
          funcID_ComplHandler_ZW_RediscoveryNeeded = serFrameDataPtr[1];
          retVal = ZW_REDISCOVERY_NEEDED(serFrameDataPtr[0],
                                        (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_RediscoveryNeeded : NULL);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_REDISCOVERY_NEEDED */

#if SUPPORT_ZW_REQUEST_NEW_ROUTE_DESTINATIONS
        case FUNC_ID_ZW_REQUEST_NEW_ROUTE_DESTINATIONS:
          /* HOST->ZW: destList[5] | funcID */
          /* destList zero terminated if less than ZW_MAX_RETURN_ROUTE_DESTINATIONS */
          funcID_ComplHandler_ZW_RequestNewRouteDestinations = serFrameDataPtr[5];
          for (i = 0; i < 5; i++)
          {
            if (serFrameDataPtr[i])
            {
              returnRouteDests[i] = serFrameDataPtr[i];
            }
            else
            {
              break;
            }
          }
          retVal = ZW_REQUEST_NEW_ROUTE_DESTINATIONS(returnRouteDests, i,
                                                     serFrameDataPtr[5] ? ComplHandler_ZW_RequestNewRouteDestinations : NULL);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_REQUEST_NEW_ROUTE_DESTINATIONS */

#if SUPPORT_ZW_IS_NODE_WITHIN_DIRECT_RANGE
        case FUNC_ID_ZW_IS_NODE_WITHIN_DIRECT_RANGE:
          /* HOST->ZW: bNodeID */
          retVal = ZW_IS_NODE_WITHIN_DIRECT_RANGE(serFrameDataPtr[0]);
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_SEND_NODE_INFORMATION
        case FUNC_ID_ZW_SEND_NODE_INFORMATION:
          /* HOST->ZW: destNode | txOptions | funcID */
          funcID_ComplHandler_ZW_SendNodeInformation = serFrameDataPtr[2];
          /* TO#1888 fix - ZW_SendNodeInformation return value is not returned to HOST */
          retVal = ZW_SendNodeInformation(serFrameDataPtr[0], serFrameDataPtr[1],
                                          (serFrameDataPtr[2] != 0) ? ComplHandler_ZW_SendNodeInformation : NULL);
          DoRespond();
          break;
#endif /* SUPPORT_ZW_SEND_NODE_INFORMATION */

#if SUPPORT_ZW_SEND_DATA
        case FUNC_ID_ZW_SEND_DATA:
          /* HOST->ZW: nodeID | dataLength | pData[] | txOptions | funcID */
          {
            dataLength = serFrameDataPtr[1];

            for (i = 0; i < dataLength; i++)
            {
              workbuf[i] = serFrameDataPtr[2 + i];
            }
            funcID_ComplHandler_ZW_SendData = serFrameDataPtr[3+dataLength];
            retVal = ZW_SEND_DATA(serFrameDataPtr[0], workbuf, dataLength, serFrameDataPtr[2 + dataLength],
                                  (serFrameDataPtr[3+dataLength] != 0) ? ComplHandler_ZW_SendData : NULL);
            DoRespond();
          }
          break;
#endif

#if SUPPORT_ZW_SEND_DATA_META
        case FUNC_ID_ZW_SEND_DATA_META:
          /* HOST->ZW: nodeID | dataLength | pData[] | txOptions | funcID */
          {
            dataLength = serFrameDataPtr[1];
            for (i = 0; i < dataLength; i++)
            {
              workbuf[i] = serFrameDataPtr[2 + i];
            }
            funcID_ComplHandler_ZW_SendDataMeta = serFrameDataPtr[3 + dataLength];
            retVal = ZW_SEND_DATA_META(serFrameDataPtr[0], workbuf, dataLength, serFrameDataPtr[2 + dataLength],
                                       (serFrameDataPtr[3 + dataLength] != 0) ? ComplHandler_ZW_SendDataMeta : NULL);
            DoRespond();
          }
          break;
#endif

#if SUPPORT_ZW_SEND_DATA_MULTI
        case FUNC_ID_ZW_SEND_DATA_MULTI:
          /* HOST->ZW: numberNodes | pNodeIDList[] | dataLength | pData[] | txOptions | funcId */
          {
            BYTE numberNodes, txOptions;

            numberNodes = serFrameDataPtr[0];
            /* clear the destination node mask */
            ZW_memset(groupMask, 0, MAX_NODEMASK_LENGTH);

            /* Set the destination node mask bits */
            for (i = 0; (i < numberNodes) && (i < MAX_GROUP_NODES); i++)
            {
              if((!serFrameDataPtr[1+i]) || (ZW_MAX_NODES >= serFrameDataPtr[1+i]))
              {
                ZW_NodeMaskSetBit(groupMask, serFrameDataPtr[1+i]);
              }
            }
            dataLength = serFrameDataPtr[1 + numberNodes];
            for (i = 0; i < dataLength; i++)
            {
              workbuf[i] = serFrameDataPtr[2 + numberNodes + i];
            }
            txOptions = serFrameDataPtr[2 + numberNodes + dataLength];

            funcID_ComplHandler_ZW_SendDataMulti = serFrameDataPtr[3 + numberNodes + dataLength];
            retVal = ZW_SEND_DATA_MULTI(groupMask, workbuf, dataLength, txOptions,
                                        (serFrameDataPtr[3 + numberNodes + dataLength] != 0) ? ComplHandler_ZW_SendDataMulti : NULL);
            DoRespond();
          }
          break;
#endif

#if SUPPORT_ZW_SEND_DATA_MR
        case FUNC_ID_ZW_SEND_DATA_MR:
          /* HOST->ZW: nodeID | dataLength | pData[dataLength] | txOptions | pRoute[4] | funcID */
          {
            dataLength = serFrameDataPtr[1];
            funcID_ComplHandler_ZW_SendData = serFrameDataPtr[3+4+dataLength];

            for (i = 0; i < dataLength; i++)
            {
              workbuf[i] = serFrameDataPtr[2 + i];
            }
            for (i = 0; i < 4; i++)
            {
              pRouteBuffer[i] = serFrameDataPtr[3 + dataLength + i];
            }
            retVal = ZW_SEND_DATA_GENERIC(serFrameDataPtr[0], workbuf, dataLength, serFrameDataPtr[2 + dataLength],
                                          (pRouteBuffer[0] != 0) ? pRouteBuffer : NULL,
                                          (serFrameDataPtr[3+4+dataLength] != 0) ? ComplHandler_ZW_SendDataMR : NULL);
            DoRespond();
          }
        break;
#endif

#if SUPPORT_ZW_SEND_DATA_META_MR
        case FUNC_ID_ZW_SEND_DATA_META_MR:
          /* HOST->ZW: nodeID | dataLength | pData[dataLength] | txOptions | pRoute[4] | funcID */
          {
            dataLength = serFrameDataPtr[1];
            funcID_ComplHandler_ZW_SendDataMeta = serFrameDataPtr[3+4+dataLength];

            for (i = 0; i < dataLength; i++)
            {
                workbuf[i] = serFrameDataPtr[2 + i];
            }
            for(i=0;i<4;i++)
            {
              pRouteBuffer[i] = serFrameDataPtr[3 + dataLength + i];
            }
            retVal = ZW_SEND_DATA_META_GENERIC(serFrameDataPtr[0], workbuf, dataLength, serFrameDataPtr[2 + dataLength],
                                               (pRouteBuffer[0] != 0) ? pRouteBuffer : NULL,
                                               (serFrameDataPtr[3+4+dataLength] != 0) ? ComplHandler_ZW_SendDataMetaMR : NULL);
            DoRespond();
          }
          break;
#endif

#if SUPPORT_ZW_SEND_DATA_BRIDGE
        case FUNC_ID_ZW_SEND_DATA_BRIDGE:
          /* HOST->ZW: srcNodeID | destNodeID | dataLength | pData[] | txOptions | pRoute[4] | funcID */
          {
            dataLength = serFrameDataPtr[2];
            for (i = 0; i < dataLength; i++)
            {
              workbuf[i] = serFrameDataPtr[3 + i];
            }
            for (i = 0; i < 4; i++)
            {
              pRouteBuffer[i] = serFrameDataPtr[3 + 1 + dataLength + i];
            }
            funcID_ComplHandler_ZW_SendData = serFrameDataPtr[3 + 1 + 4 + dataLength];
            retVal = ZW_SEND_DATA_BRIDGE(serFrameDataPtr[0], serFrameDataPtr[1], workbuf, dataLength,
                                         serFrameDataPtr[3 + dataLength], (pRouteBuffer[0] != 0) ? pRouteBuffer : NULL,
                                         (serFrameDataPtr[3 + 1 + 4 + dataLength] != 0) ? ComplHandler_ZW_SendData_Bridge : NULL);
            DoRespond();
          }
          break;
#endif

#if SUPPORT_ZW_SEND_DATA_META_BRIDGE
        case FUNC_ID_ZW_SEND_DATA_META_BRIDGE:
          /* HOST->ZW: srcNodeID | destNodeID | dataLength | pData[] | txOptions | pRoute[4] | funcID */
          {
            dataLength = serFrameDataPtr[2];
            for (i = 0; i < dataLength; i++)
            {
              workbuf[i] = serFrameDataPtr[3 + i];
            }
            for (i = 0; i < 4; i++)
            {
              pRouteBuffer[i] = serFrameDataPtr[3 + 1 + dataLength + i];
            }
            funcID_ComplHandler_ZW_SendDataMeta = serFrameDataPtr[3 + 1 + 4 + dataLength];
            retVal = ZW_SEND_DATA_META_BRIDGE(serFrameDataPtr[0], serFrameDataPtr[1], workbuf, dataLength,
                                              serFrameDataPtr[3 + dataLength], (pRouteBuffer[0] != 0) ? pRouteBuffer : NULL,
                                              (serFrameDataPtr[3 + 1 + 4 + dataLength] != 0) ? ComplHandler_ZW_SendDataMeta_Bridge : NULL);
            DoRespond();
          }
          break;
#endif

#if SUPPORT_ZW_SEND_DATA_MULTI_BRIDGE
        case FUNC_ID_ZW_SEND_DATA_MULTI_BRIDGE:
          /* HOST->ZW: srcNodeID | numberNodes | pNodeIDList[] | dataLength | pData[] | txOptions | funcId */
          {
            BYTE numberNodes, txOptions;

            numberNodes = serFrameDataPtr[1];
            /* clear the destination node mask */
            ZW_memset(groupMask, 0, MAX_NODEMASK_LENGTH);
            /* Set the destination node mask bits */
            for (i = 0; (i < numberNodes) && (i < MAX_GROUP_NODES); i++)
            {
              if((!serFrameDataPtr[2+i]) || (232 > serFrameDataPtr[2+i]))
              {
                ZW_NodeMaskSetBit(groupMask, serFrameDataPtr[2 + i]);
              }
            }
            dataLength = serFrameDataPtr[2 + numberNodes];
            for (i = 0; i < dataLength; i++)
            {
              workbuf[i] = serFrameDataPtr[2 + 1 + numberNodes + i];
            }
            txOptions = serFrameDataPtr[2 + 1 + numberNodes + dataLength];

            funcID_ComplHandler_ZW_SendDataMulti = serFrameDataPtr[2 + 1 + 1 + numberNodes + dataLength];
            retVal = ZW_SEND_DATA_MULTI_BRIDGE(serFrameDataPtr[0], groupMask, workbuf, dataLength, txOptions,
                                               (serFrameDataPtr[2 + 1 + 1 + numberNodes + dataLength] != 0) ?
                                                 ComplHandler_ZW_SendDataMulti_Bridge : NULL);
            DoRespond();
          }
          break;
#endif

#if SUPPORT_ZW_SEND_DATA_ABORT
        case FUNC_ID_ZW_SEND_DATA_ABORT:
          /* HOST->ZW: No params defined */
          ZW_SendDataAbort(); /* If we are in middle of transmitting an application frame then STOP that. */
          state = stateIdle;
          break;
#endif

#if SUPPORT_MEMORY_GET_ID
        case FUNC_ID_MEMORY_GET_ID:
          /* HOST->ZW: No params defined */
          ZW_MEMORY_GET_ID(compl_workbuf, compl_workbuf + 4);
          DoRespond_workbuf(5);
          break;
#endif

#if SUPPORT_MEMORY_GET_BYTE
        case FUNC_ID_MEMORY_GET_BYTE:
          /* HOST->ZW: offset (MSB) | offset (LSB) */
          retVal = MemoryGetByte((((WORD)serFrameDataPtr[0] << 8) + serFrameDataPtr[1]));
          DoRespond();
          break;
#endif

#if SUPPORT_MEMORY_PUT_BYTE
        case FUNC_ID_MEMORY_PUT_BYTE:
          /* HOST->ZW: offset(MSB) | offset (LSB) | data */
          retVal = MemoryPutByte((((WORD)serFrameDataPtr[0] << 8) + serFrameDataPtr[1]), serFrameDataPtr[2]);
          DoRespond();
          break;
#endif

#if SUPPORT_MEMORY_GET_BUFFER
        case FUNC_ID_MEMORY_GET_BUFFER:
          /* HOST->ZW: offset(MSB) | offset (LSB) | length */
          {
            dataLength = serFrameDataPtr[2];
            /* Make sure the length isn't larger than the available buffer size */
            if (dataLength > BUF_SIZE_TX)
            {
              dataLength = BUF_SIZE_TX;
            }
            MemoryGetBuffer((((WORD)serFrameDataPtr[0] << 8) + serFrameDataPtr[1]), compl_workbuf, dataLength);
            DoRespond_workbuf(dataLength);
          }
          break;
#endif

#if SUPPORT_MEMORY_PUT_BUFFER
        case FUNC_ID_MEMORY_PUT_BUFFER:
          /* HOST->ZW: offset(MSB) | offset (LSB) | length(MSB) | length (LSB) | buffer[] | funcID */
          {
            WORD length = ((WORD)(serFrameDataPtr[2] << 8)) + serFrameDataPtr[3];
            if (length > BUF_SIZE_RX)
            {
              retVal = FALSE; /* ignore request if length is larger than available buffer */
            }
            else
            {
              for (i = 0; i <length; i++)
              {
                workbuf[i] = serFrameDataPtr[4+i];
              }
              funcID_ComplHandler_MemoryPutBuffer = serFrameDataPtr[4+length];
              retVal = ZW_MEM_PUT_BUFFER((((WORD)serFrameDataPtr[0] << 8) + serFrameDataPtr[1]),
                                       workbuf, length,
                                       (serFrameDataPtr[4 + length] != 0) ? ComplHandler_MemoryPutBuffer : NULL);
            }
            DoRespond();
          }
          break;
#endif

#if SUPPORT_SERIAL_API_GET_APPL_HOST_MEMORY_OFFSET
        case FUNC_ID_SERIAL_API_GET_APPL_HOST_MEMORY_OFFSET:
          /* HOST->ZW: No params defined */
          retVal =
          break;
#endif

#if SUPPORT_ZW_REQUEST_NETWORK_UPDATE
        case FUNC_ID_ZW_REQUEST_NETWORK_UPDATE:
          /* HOST->ZW: funcID */
          funcID_ComplHandler_netWork_Management = serFrameDataPtr[0];
          management_Func_ID = serFrameCmd;
          retVal = ZW_REQUEST_NETWORK_UPDATE((serFrameDataPtr[0] != 0) ? ComplHandler_ZW_netWork_Management : NULL);
          DoRespond();
          break;
#endif /* SUPPORT_ZW_REQUEST_NETWORK_UPDATE */

#if SUPPORT_ZW_ENABLE_SUC
        case FUNC_ID_ZW_ENABLE_SUC:
          /* HOST->ZW: state | capabilities */
          retVal = ZW_ENABLE_SUC(serFrameDataPtr[0], serFrameDataPtr[1]);
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_REQUEST_NODE_NEIGHBOR_UPDATE
        case FUNC_ID_ZW_REQUEST_NODE_NEIGHBOR_UPDATE:
          /* HOST->ZW: nodeID | funcID */
          funcID_ComplHandler_ZW_RequestNodeNeighborUpdate = serFrameDataPtr[1];
          if (!ZW_REQUEST_NODE_NEIGHBOR_UPDATE(serFrameDataPtr[0],
               serFrameDataPtr[1] ? ComplHandler_ZW_RequestNodeNeighborUpdate : NULL))
          {
            /* It did not kick off, no nodes to ask? */
            if (serFrameDataPtr[1])
            {
              ComplHandler_ZW_RequestNodeNeighborUpdate(REQUEST_NEIGHBOR_UPDATE_FAILED);
            }
          }
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_GET_NEIGHBOR_COUNT
        case FUNC_ID_ZW_GET_NEIGHBOR_COUNT:
          /* HOST->ZW: nodeID */
          retVal = ZW_GET_NEIGHBOR_COUNT(serFrameDataPtr[0]);
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_ARE_NODES_NEIGHBOURS
        case FUNC_ID_ZW_ARE_NODES_NEIGHBOURS:
          /* HOST->ZW: nodeIDa | nodeIDb */
          retVal = ZW_ARE_NODES_NEIGHBOURS(serFrameDataPtr[0], serFrameDataPtr[1]);
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_SET_LEARN_NODE_STATE
        case FUNC_ID_ZW_SET_LEARN_NODE_STATE:
          /* HOST->ZW: mode | funcID */
          funcID_ComplHandler_ZW_SetLearnNodeState = serFrameDataPtr[1];
          ZW_SetLearnNodeState(serFrameDataPtr[0], (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_SetLearnNodeState : NULL);
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_GET_NODE_PROTOCOL_INFO
        case FUNC_ID_ZW_GET_NODE_PROTOCOL_INFO:
          /* HOST->ZW: bNodeID */
          ZW_GET_NODE_STATE(serFrameDataPtr[0], (NODEINFO *) compl_workbuf);
          DoRespond_workbuf(6);
          break;
#endif

#if SUPPORT_ZW_SET_DEFAULT
        case FUNC_ID_ZW_SET_DEFAULT:
          /* HOST->ZW: funcID */
          funcID_ComplHandler_ZW_SetDefault = serFrameDataPtr[0];

#ifdef ZW_CONTROLLER
          ZW_SET_DEFAULT((serFrameDataPtr[0] != 0) ? ComplHandler_ZW_SetDefault : NULL);
#else
          ZW_SET_DEFAULT();
          if (serFrameDataPtr[0] != 0)
          {
            ComplHandler_ZW_SetDefault(); //To make sure that controllers and slaves behave identically
          }
#endif
          state = stateIdle;
          break;
#endif  /* SUPPORT_ZW_SET_DEFAULT */

#if SUPPORT_ZW_NEW_CONTROLLER
        case FUNC_ID_ZW_NEW_CONTROLLER:
          /* HOST->ZW: state | funcID */
          funcID_ComplHandler_ZW_NewController = serFrameDataPtr[1];
          ZW_NEW_CONTROLLER( serFrameDataPtr[0], (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_NewController : NULL);
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_ADD_NODE_TO_NETWORK
        case FUNC_ID_ZW_ADD_NODE_TO_NETWORK:
          /* HOST->ZW: mode | funcID */
          SetupNodeManagement();
          ZW_ADD_NODE_TO_NETWORK(serFrameDataPtr[0], (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_NodeManagement : NULL);
          break;
#endif

#if SUPPORT_ZW_REMOVE_NODE_FROM_NETWORK
        case FUNC_ID_ZW_REMOVE_NODE_FROM_NETWORK:
          /* HOST->ZW: mode | funcID */
          SetupNodeManagement();
          ZW_REMOVE_NODE_FROM_NETWORK(serFrameDataPtr[0], (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_NodeManagement : NULL);
          break;
#endif

#if SUPPORT_ZW_CREATE_NEW_PRIMARY
        case FUNC_ID_ZW_CREATE_NEW_PRIMARY:
          /* HOST->ZW: mode | funcID */
          SetupNodeManagement();
          ZW_CREATE_NEW_PRIMARY_CTRL(serFrameDataPtr[0], (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_NodeManagement : NULL);
          break;
#endif

#if SUPPORT_ZW_CONTROLLER_CHANGE
        case FUNC_ID_ZW_CONTROLLER_CHANGE:
          /* HOST->ZW: mode | funcID */
          SetupNodeManagement();
          ZW_CONTROLLER_CHANGE(serFrameDataPtr[0], (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_NodeManagement : NULL);
          break;
#endif

#if SUPPORT_ZW_SET_LEARN_MODE
        case FUNC_ID_ZW_SET_LEARN_MODE:
          /* HOST->ZW: mode | funcID */
#ifdef ZW_CONTROLLER
          SetupNodeManagement();
          ZW_SET_LEARN_MODE(serFrameDataPtr[0], (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_NodeManagement : NULL);
#else
          funcID_ComplHandler_ZW_SetLearnMode = serFrameDataPtr[1];
          ZW_SetLearnMode(serFrameDataPtr[0], (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_SetLearnMode : NULL);
          state = stateIdle;
#endif /* ZW_CONTROLLER */
          break;
#endif

#if SUPPORT_ZW_SUPPORT9600_ONLY
        case FUNC_ID_ZW_SUPPORT9600_ONLY:
          /* HOST->ZW: mode */
          retVal = ZW_SUPPORT9600_ONLY(serFrameDataPtr[0]);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_SUPPORT9600_ONLY */

#if SUPPORT_ZW_REPLICATION_COMMAND_COMPLETE
        case FUNC_ID_ZW_REPLICATION_COMMAND_COMPLETE:
          /* HOST->ZW: No params defined */
          ZW_REPLICATION_COMMAND_COMPLETE();
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_REPLICATION_SEND_DATA
        case FUNC_ID_ZW_REPLICATION_SEND_DATA:
          /* HOST->ZW: nodeID | dataLength | pData[] | txOptions | funcID */
          {
            dataLength = serFrameDataPtr[1];
            for (i = 0; i < dataLength; i++)
            {
              workbuf[i] = serFrameDataPtr[2+i];
            }
            funcID_ComplHandler_ZW_ReplicationSendData = serFrameDataPtr[3 + dataLength];
            retVal = ZW_REPLICATION_SEND_DATA(serFrameDataPtr[0], workbuf, dataLength, serFrameDataPtr[2 + dataLength],
                                              (serFrameDataPtr[3 + dataLength] != 0) ? ComplHandler_ZW_ReplicationSendData : NULL);
            DoRespond();
          }
          break;
#endif  /* SUPPORT_ZW_REPLICATION_SEND_DATA */

#if SUPPORT_ZW_ASSIGN_RETURN_ROUTE
        case FUNC_ID_ZW_ASSIGN_RETURN_ROUTE:
          /* HOST->ZW: srcNodeID | destNodeID | funcID */
          funcID_ComplHandler_ZW_AssignReturnRoute = serFrameDataPtr[2];
          retVal = ZW_ASSIGN_RETURN_ROUTE(serFrameDataPtr[0], serFrameDataPtr[1],
                                          (serFrameDataPtr[2] != 0) ? ComplHandler_ZW_AssignReturnRoute : NULL);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_ASSIGN_RETURN_ROUTE */

#if SUPPORT_ZW_DELETE_RETURN_ROUTE
        case FUNC_ID_ZW_DELETE_RETURN_ROUTE:
          /* HOST->ZW: nodeID | funcID */
          funcID_ComplHandler_ZW_DeleteReturnRoute = serFrameDataPtr[1];
          retVal = ZW_DELETE_RETURN_ROUTE(serFrameDataPtr[0],
                                          (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_DeleteReturnRoute : NULL);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_DELETE_RETURN_ROUTE */


#if SUPPORT_ZW_ASSIGN_SUC_RETURN_ROUTE
        case FUNC_ID_ZW_ASSIGN_SUC_RETURN_ROUTE:
          /* HOST->ZW: srcNodeID | funcID */
          funcID_ComplHandler_netWork_Management = serFrameDataPtr[1];
          retVal = ZW_ASSIGN_SUC_RETURN_ROUTE(serFrameDataPtr[0],
                                              (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_netWork_Management : NULL);
          management_Func_ID = serFrameCmd;
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_ASSIGN_SUC_RETURN_ROUTE */

#if SUPPORT_ZW_DELETE_SUC_RETURN_ROUTE
        case FUNC_ID_ZW_DELETE_SUC_RETURN_ROUTE:
          /* HOST->ZW: srcNodeID | funcID */
          funcID_ComplHandler_netWork_Management = serFrameDataPtr[1];
          retVal = ZW_DELETE_SUC_RETURN_ROUTE(serFrameDataPtr[0],
                                              (serFrameDataPtr[1] != 0) ? ComplHandler_ZW_netWork_Management : NULL);
          management_Func_ID = serFrameCmd;
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_DELETE_SUC_RETURN_ROUTE */

#if SUPPORT_ZW_SEND_SUC_ID
        case FUNC_ID_ZW_SEND_SUC_ID:
          /* HOST->ZW: destNodeID | txOptions | funcID */
          funcID_ComplHandler_ZW_SendSUC_ID = serFrameDataPtr[2];
          retVal = ZW_SEND_SUC_ID(serFrameDataPtr[0], serFrameDataPtr[1], (serFrameDataPtr[2] != 0) ? ComplHandler_ZW_SendSUC_ID : NULL);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_SEND_SUC_ID */

#if SUPPORT_ZW_SET_SUC_NODE_ID
        case FUNC_ID_ZW_SET_SUC_NODE_ID:
          /* HOST->ZW: nodeID | SUCState | txOptions | capabilities | funcID */
          funcID_ComplHandler_ZW_SetSUCNodeID = serFrameDataPtr[4];
          retVal = ZW_SET_SUC_NODEID(serFrameDataPtr[0],
                                     serFrameDataPtr[1],
                                     serFrameDataPtr[2],
                                     serFrameDataPtr[3],
                                     (serFrameDataPtr[4] != 0) ? ComplHandler_ZW_SetSUCNodeID : NULL);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_SET_SUC_NODE_ID */

#if SUPPORT_ZW_GET_SUC_NODE_ID
        case FUNC_ID_ZW_GET_SUC_NODE_ID:
          /* HOST->ZW: No params defined */
          retVal = ZW_GET_SUC_NODEID();
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_GET_SUC_NODE_ID */

#if SUPPORT_ZW_REMOVE_FAILED_NODE_ID
        case FUNC_ID_ZW_REMOVE_FAILED_NODE_ID:
          /* HOST->ZW: nodeID | funcID */
          funcID_ComplHandler_ZW_RemoveFailedNodeID = serFrameDataPtr[1];
          retVal = ZW_REMOVE_FAILED_NODE_ID(serFrameDataPtr[0],
                                            (serFrameDataPtr[1] != 0) ?
                                            ComplHandler_ZW_RemoveFailedNodeID : NULL);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_REMOVE_FAILED_NODE_ID */

#if SUPPORT_ZW_IS_FAILED_NODE_ID
        case FUNC_ID_ZW_IS_FAILED_NODE_ID:
          /* HOST->ZW: nodeID */
          retVal = ZW_IS_FAILED_NODE_ID(serFrameDataPtr[0]);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_IS_FAILED_NODE_ID */

#if SUPPORT_ZW_REPLACE_FAILED_NODE
        case FUNC_ID_ZW_REPLACE_FAILED_NODE:
          /* HOST->ZW: nodeID | funcID */
          funcID_ComplHandler_ZW_ReplaceFailedNode = serFrameDataPtr[1];
          retVal = ZW_REPLACE_FAILED_NODE(serFrameDataPtr[0],
                                          /* Use NormalPower for including the replacement */
                                          TRUE,
                                          (serFrameDataPtr[1] != 0) ?
                                          ComplHandler_ZW_ReplaceFailedNode : NULL);
          DoRespond();
          break;
#endif  /* SUPPORT_ZW_REPLACE_FAILED_NODE */

#if SUPPORT_GET_ROUTING_TABLE_LINE
        case FUNC_ID_GET_ROUTING_TABLE_LINE:
          /* HOST->ZW: bLine | bRemoveBad | bRemoveNonReps | funcID */
          ZW_GET_ROUTING_INFO(serFrameDataPtr[0], compl_workbuf,
                              ((serFrameDataPtr[1]) ? GET_ROUTING_INFO_REMOVE_BAD : 0) |
                              ((serFrameDataPtr[2]) ? GET_ROUTING_INFO_REMOVE_NON_REPS : 0));
          DoRespond_workbuf(MAX_NODEMASK_LENGTH);
          break;
#endif  /* SUPPORT_GET_ROUTING_TABLE_LINE */

#if SUPPORT_ZW_SET_ROUTING_INFO
        case FUNC_ID_ZW_SET_ROUTING_INFO:
          /* HOST->ZW: nodeId | NodeMask[29] */
          retVal = ZW_SetRoutingInfo(serFrameDataPtr[0], MAX_NODEMASK_LENGTH, &serFrameDataPtr[1]);
          DoRespond();
          break;
#endif

#if SUPPORT_GET_TX_COUNTER
        case FUNC_ID_GET_TX_COUNTER:
          /* Get the transmit counter */
          /* HOST->ZW: No params defined */
          retVal = ZW_TX_COUNTER;
          DoRespond();
          break;
#endif  /* SUPPORT_GET_TX_COUNTER */

#if SUPPORT_RESET_TX_COUNTER
       case FUNC_ID_RESET_TX_COUNTER:
          /* Reset the transmit counter */
          /* HOST->ZW: No params defined */
          ZW_TX_COUNTER = 0;
          state = stateIdle;
          break;
#endif  /* SUPPORT_RESET_TX_COUNTER */

#if SUPPORT_STORE_NODEINFO
       case FUNC_ID_STORE_NODEINFO:
          /*              0         1                   2                   3                 4               5               6             7
          /* HOST->ZW: nodeID | nodeInfo.capability|nodeInfo.security|nodeInfo.reserved|nodeInfo.basic|nodeInfo.generic|nodeInfo.specific|funcId */
          funcID_ComplHandler_ZW_StoreNodeInfo = serFrameDataPtr[7];

          retVal = ZW_STORE_NODE_INFO(serFrameDataPtr[0], &serFrameDataPtr[1],
                                      (serFrameDataPtr[7] != 0) ? ComplHandler_ZW_StoreNodeInfo : NULL);
          DoRespond();
          break;
#endif  /* SUPPORT_STORE_NODEINFO */

#if SUPPORT_STORE_HOMEID
       case FUNC_ID_STORE_HOMEID:
          /* Store homeID and Node ID. */
          /* HOST->ZW: homeID1 | homeID2 | homeID3 | homeID4 | nodeID */
          ZW_STORE_HOME_ID(&serFrameDataPtr[0], serFrameDataPtr[4]);
          state = stateIdle;
          break;
#endif /* SUPPORT_STORE_HOMEID */

#if SUPPORT_LOCK_ROUTE_RESPONSE
       case FUNC_ID_LOCK_ROUTE_RESPONSE:
          /* Lock the response routes (nodeID or 0x00 to free routes */
          /* HOST->ZW: No params defined */
          ZW_LOCK_RESPONSE_ROUTE(serFrameDataPtr[0]);
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_GET_VERSION
        case FUNC_ID_ZW_GET_VERSION:
          /* HOST->ZW: No params defined */
          BYTE_IN_AR(compl_workbuf, 12) = ZW_Version((BYTE *)compl_workbuf);
          DoRespond_workbuf(13);
          break;
#endif

#if SUPPORT_SERIAL_API_APPL_NODE_INFORMATION
        case FUNC_ID_SERIAL_API_APPL_NODE_INFORMATION:
          /* HOST->ZW: listening | generic | specific | parmLength | nodeParms[] */
          applNodeInfo_deviceOptionsMask = serFrameDataPtr[0];
          applNodeInfo_nodeType_generic = serFrameDataPtr[1];
          applNodeInfo_nodeType_specific = serFrameDataPtr[2];
          applNodeInfo_parmLength = serFrameDataPtr[3];
          if (applNodeInfo_parmLength > APPL_NODEPARM_MAX)
          {
            applNodeInfo_parmLength = APPL_NODEPARM_MAX;
          }
          for (i = 0; i < applNodeInfo_parmLength; i++)
          {
            applNodeInfo_nodeParm[i] = serFrameDataPtr[4+i];
          }
          SaveApplicationSettings();
          state = stateIdle;
          break;
#endif

#if SUPPORT_SERIAL_API_APPL_SLAVE_NODE_INFORMATION
        case FUNC_ID_ZW_SEND_SLAVE_NODE_INFORMATION:
          /* HOST->ZW: srcNode | destNode | txOptions | funcID */
          funcID_ComplHandler_ZW_SendSlaveNodeInformation = serFrameDataPtr[3];
          retVal = ZW_SendSlaveNodeInformation(serFrameDataPtr[0], serFrameDataPtr[1], serFrameDataPtr[2],
                                               (serFrameDataPtr[3] != 0) ? ComplHandler_ZW_SendSlaveNodeInformation : NULL);
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_SEND_SLAVE_DATA
        case FUNC_ID_ZW_SEND_SLAVE_DATA:
          /* HOST->ZW: srcID | destID | dataLength | pData[] | txOptions | funcID */
          {
            dataLength = serFrameDataPtr[2];
            for (i = 0; i < dataLength; i++)
            {
              workbuf[i] = serFrameDataPtr[3+i];
            }
            funcID_ComplHandler_ZW_SendSlaveData = serFrameDataPtr[4+dataLength];
            retVal = ZW_SEND_SLAVE_DATA(serFrameDataPtr[0], serFrameDataPtr[1],
                                        workbuf, dataLength, serFrameDataPtr[3 + dataLength],
                                        (serFrameDataPtr[4 + dataLength] != 0) ? ComplHandler_ZW_SendSlaveData : NULL);
            DoRespond();
          }
          break;
#endif

#if SUPPORT_ZW_SET_SLAVE_LEARN_MODE
        case FUNC_ID_ZW_SET_SLAVE_LEARN_MODE:
          /* HOST->ZW: node | mode | funcID */
          funcID_ComplHandler_ZW_SetSlaveLearnMode = serFrameDataPtr[2];
          retVal = ZW_SET_SLAVE_LEARN_MODE(serFrameDataPtr[0], serFrameDataPtr[1],
                                           (serFrameDataPtr[2] != 0) ? ComplHandler_ZW_SetSlaveLearnMode : NULL);
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_SEND_TEST_FRAME
        case FUNC_ID_ZW_SEND_TEST_FRAME:
          /* HOST->ZW: node | powerlevel | funcID */
          funcID_ComplHandler_ZW_SendTestFrame = serFrameDataPtr[2];
          retVal = ZW_SEND_TEST_FRAME(serFrameDataPtr[0], serFrameDataPtr[1],
                                       (serFrameDataPtr[2] != 0) ? ComplHandler_ZW_SendTestFrame : NULL);
          DoRespond();
          break;
#endif

#if SUPPORT_SERIAL_API_APPL_SLAVE_NODE_INFORMATION
        case FUNC_ID_SERIAL_API_APPL_SLAVE_NODE_INFORMATION:
          /* HOST->ZW: srcNodeID | listening | generic | specific | parmLength | nodeParms[] */
          /* TODO - Here its possible to do something intelligent using the srcNodeID */
          /* serFrameDataPtr[0] = srcNodeID */
          applSlaveNodeInfo_deviceOptionsMask = serFrameDataPtr[1];
          applSlaveNodeInfo_nodeType_generic = serFrameDataPtr[2];
          applSlaveNodeInfo_nodeType_specific = serFrameDataPtr[3];
          applSlaveNodeInfo_parmLength = serFrameDataPtr[4];
          for (i = 0; (i < applSlaveNodeInfo_parmLength) && (i < APPL_SLAVENODEPARM_MAX); i++)
          {
            applSlaveNodeInfo_nodeParm[i] = serFrameDataPtr[5 + i];
          }
          for (; i < APPL_SLAVENODEPARM_MAX; i++)
          {
            applSlaveNodeInfo_nodeParm[i] = 0;
          }
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_IS_VIRTUAL_NODE
        case FUNC_ID_ZW_IS_VIRTUAL_NODE:
          /* HOST->ZW: No params defined */
          retVal = ZW_IS_VIRTUAL_NODE(serFrameDataPtr[0]);
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_GET_VIRTUAL_NODES
        case FUNC_ID_ZW_GET_VIRTUAL_NODES:
          /* HOST->ZW: No params defined */
          ZW_GET_VIRTUAL_NODES(compl_workbuf);
          DoRespond_workbuf(ZW_MAX_NODES / 8);
          break;
#endif

#if SUPPORT_SERIAL_API_GET_INIT_DATA
        case FUNC_ID_SERIAL_API_GET_INIT_DATA:
          /* HOST->ZW: No params defined */
          BYTE_IN_AR(compl_workbuf, 0) = SERIAL_API_VER;
          BYTE_IN_AR(compl_workbuf, 1) = 0; /* Flag byte - default: controller api, no timer support, no primary, no SUC */
#ifdef ZW_CONTROLLER
          if (!ZW_PRIMARYCTRL())
          {
            BYTE_IN_AR(compl_workbuf, 1) |= GET_INIT_DATA_FLAG_SECONDARY_CTRL; /* Set Primary/secondary bit */
          }
#ifdef ZW_CONTROLLER_STATIC
          if (ZW_GET_CONTROLLER_CAPABILITIES() & CONTROLLER_IS_SUC)    /* if (ZW_IS_SUC_ACTIVE()) */
          {
            BYTE_IN_AR(compl_workbuf, 1) |= GET_INIT_DATA_FLAG_IS_SUC; /* Set SUC bit if active */
          }
#endif /*ZW_CONTROLLER_STATIC*/
          {
            NODEINFO nodeInfo;

            /* compl_workbuf[1] is already set to controller api*/
            BYTE_IN_AR(compl_workbuf, 2) = ZW_MAX_NODES / 8;     /* node bitmask length */
            /* Remember to clear the buffer!! */
            for (i = 3; i < (ZW_MAX_NODES / 8) + 3; i++)
            {
              BYTE_IN_AR(compl_workbuf, i) = 0;
            }
            /* Next ZW_MAX_NODES/8 = 29 bytes of compl_workbuf reserved for node bitmask */
            for (i = 1; i <= ZW_MAX_NODES; i++)
            {
              ZW_GetNodeProtocolInfo(i, &nodeInfo);
              if (nodeInfo.nodeType.generic)
              {
                ZW_NodeMaskSetBit(compl_workbuf + 3, i);
              }
              ZW_POLL();
            }
          }
          BYTE_IN_AR(compl_workbuf, 3 + (ZW_MAX_NODES/8)) = ZW_CHIP_TYPE;
          BYTE_IN_AR(compl_workbuf, 3 + (ZW_MAX_NODES/8) + 1) = ZW_CHIP_REVISION;
          DoRespond_workbuf(3 + (ZW_MAX_NODES/8) + 2);
#endif
          /* TO#1967 fix. */
#ifdef ZW_SLAVE
          BYTE_IN_AR(compl_workbuf, 1) |= GET_INIT_DATA_FLAG_SLAVE_API;  /* Flag byte */
          BYTE_IN_AR(compl_workbuf, 2) = 0;     /* node bitmask length */
          BYTE_IN_AR(compl_workbuf, 3) = ZW_CHIP_TYPE;
          BYTE_IN_AR(compl_workbuf, 4) = ZW_CHIP_REVISION;
          DoRespond_workbuf(3 + 2);
#endif /* ZW_SLAVE */
          break;
#endif

#if SUPPORT_ZW_GET_CONTROLLER_CAPABILITIES
        case FUNC_ID_ZW_GET_CONTROLLER_CAPABILITIES:
          /* HOST->ZW: No params defined */
          retVal = ZW_GET_CONTROLLER_CAPABILITIES();
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_REQUEST_NODE_INFO
        case FUNC_ID_ZW_REQUEST_NODE_INFO:
          /* HOST->ZW: nodeID */
          retVal = ZW_RequestNodeInfo(serFrameDataPtr[0], ComplHandler_ZW_RequestNodeInfo);
          DoRespond();
          break;
#endif

#if SUPPORT_SERIAL_API_SET_TIMEOUTS
        case FUNC_ID_SERIAL_API_SET_TIMEOUTS:
          /* HOST->ZW: RX_ACK_timeout | RX_BYTE_timeout */
          BYTE_IN_AR(compl_workbuf, 0) = timeOutRX_ACK;     /* Respond with the old timeout settings */
          BYTE_IN_AR(compl_workbuf, 1) = timeOutRX_BYTE;
          timeOutRX_ACK = serFrameDataPtr[0];   /* Max time to wait for ACK after frame transmission in 10ms ticks */
          timeOutRX_BYTE = serFrameDataPtr[1];  /* Max time to wait for next byte when collecting a new frame in 10ms ticks */
          /* Respond with the old timeout settings */
          DoRespond_workbuf(2);
          break;
#endif

#if SUPPORT_SERIAL_API_SOFT_RESET
        case FUNC_ID_SERIAL_API_SOFT_RESET:
          /* Do soft reset the system */
          /* HOST->ZW: No params defined */
          /* TO#2145 - ZW_SoftReset changed to use watchdog instead of jumping to reset vector */
          ZW_WATCHDOG_ENABLE;
          while (1)
          {
            ;
          }
          break;
#endif

#if SUPPORT_SERIAL_API_GET_CAPABILITIES
        case FUNC_ID_SERIAL_API_GET_CAPABILITIES:
          /* HOST->ZW: No params defined */
          /* ZW->HOST: RES | 0x07 | */
          /* SERIAL_APPL_VERSION | SERIAL_APPL_REVISION | SERIALAPI_MANUFACTURER_ID1 | SERIALAPI_MANUFACTURER_ID2 | */
          /* SERIALAPI_MANUFACTURER_PRODUCT_TYPE1 | SERIALAPI_MANUFACTURER_PRODUCT_TYPE2 | */
          /* SERIALAPI_MANUFACTURER_PRODUCT_ID1 | SERIALAPI_MANUFACTURER_PRODUCT_ID2 | FUNCID_SUPPORTED_BITMASK[] */
          Respond(serFrameCmd, SERIALAPI_CAPABILITIES, sizeof(SERIALAPI_CAPABILITIES));
          /* TODO - Maybe we should transfer the current listening flag setting, generic */
          /* and specific device types and the supported command classes here! Or should we */
          /* try to catch the "Request nodeinfo" to the controller itself and inject the frame? */
          break;
#endif

#if SUPPORT_ZW_RF_POWER_LEVEL_REDISCOVERY_SET
        case FUNC_ID_ZW_RF_POWER_LEVEL_REDISCOVERY_SET:
          /* HOST->ZW: powerLevel */
          ZW_RF_POWER_LEVEL_REDISCOVERY_SET(serFrameDataPtr[0]);
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_RF_POWER_LEVEL_SET
        case FUNC_ID_ZW_RF_POWER_LEVEL_SET:
          /* HOST->ZW: powerLevel */
          retVal = ZW_RF_POWERLEVEL_SET(serFrameDataPtr[0]);
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_RF_POWER_LEVEL_GET
        case FUNC_ID_ZW_RF_POWER_LEVEL_GET:
          /* return powerlevel */
          /* HOST->ZW: No params defined */
          retVal = ZW_RF_POWERLEVEL_GET();
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_TYPE_LIBRARY
        case FUNC_ID_ZW_TYPE_LIBRARY:
          /* HOST->ZW: No params defined */
          retVal = ZW_TYPE_LIBRARY();
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_SET_EXT_INT_LEVEL
        case FUNC_ID_ZW_SET_EXT_INT_LEVEL:
          /* HOST->ZW: No params defined */
          ZW_SET_EXT_INT_LEVEL(serFrameDataPtr[0], serFrameDataPtr[1]);
          state = stateIdle;
          break;
#endif

#if SUPPORT_PWR_SETSTOPMODE
        case FUNC_ID_PWR_SETSTOPMODE:
          /* HOST->ZW: No params defined */
          ZW_PWR_SET_STOP_MODE;
          state = stateIdle;
          break;
#endif

#if SUPPORT_PWR_CLK_PD
        case FUNC_ID_PWR_CLK_PD:
          ZW_PWR_CLK_POWERDOWN(serFrameDataPtr[0]);
          state = stateIdle;
          break;
#endif

#if SUPPORT_PWR_CLK_PUP
        case FUNC_ID_PWR_CLK_PUP:
          ZW_PWR_CLK_POWERUP(serFrameDataPtr[0]);
          state = stateIdle;
          break;
#endif

#if SUPPORT_PWR_SELECT_CLK
        case FUNC_ID_PWR_SELECT_CLK:
          ZW_PWR_SELECT_CLK(serFrameDataPtr[0]);
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_SET_WUT_TIMEOUT
        case FUNC_ID_ZW_SET_WUT_TIMEOUT:
          ZW_SET_WUT_TIMEOUT(serFrameDataPtr[0]);
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_IS_WUT_KICKED
        case FUNC_ID_ZW_IS_WUT_KICKED:
          retVal = ZW_IS_WUT_KICKED();
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_GET_PROTOCOL_STATUS
        case FUNC_ID_ZW_GET_PROTOCOL_STATUS:
          retVal = ZW_GET_PROTOCOL_STATUS();
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_WATCHDOG_ENABLE
        case FUNC_ID_ZW_WATCHDOG_ENABLE:
          ZW_WATCHDOG_ENABLE;
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_WATCHDOG_DISABLE
        case FUNC_ID_ZW_WATCHDOG_DISABLE:
          ZW_WATCHDOG_DISABLE;
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_WATCHDOG_KICK
        case FUNC_ID_ZW_WATCHDOG_KICK:
          ZW_WATCHDOG_KICK;
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_WATCHDOG_START
        case FUNC_ID_ZW_WATCHDOG_START:
          ZW_WATCHDOG_ENABLE;
          bWatchdogStarted = TRUE;
          ZW_MEM_PUT_BYTE(EEOFFSET_WATCHDOG_STARTED, bWatchdogStarted);
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_WATCHDOG_STOP
        case FUNC_ID_ZW_WATCHDOG_STOP:
          ZW_WATCHDOG_DISABLE;
          bWatchdogStarted = FALSE;
          ZW_MEM_PUT_BYTE(EEOFFSET_WATCHDOG_STARTED, bWatchdogStarted);
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_SET_PROMISCUOUS_MODE
        case FUNC_ID_ZW_SET_PROMISCUOUS_MODE:
          ZW_SET_PROMISCUOUS_MODE(serFrameDataPtr[0]);
          state = stateIdle;
          break;
#endif

#ifdef ZW_CONTROLLER_SINGLE
#if SUPPORT_SERIAL_API_TEST
        case FUNC_ID_SERIAL_API_TEST:
          /* testCmd | testDelay(MSB) | testDelay(LSB) | testPayloadLen | */
          /* testCount(MSB) | testCount(LSB) | testTxOptions | nodemasklen | testnodemask[] | funcID */
          /* testCmd = 0x01 - use sendData */
          /* testCmd = 0x02 - use sendDataMeta */
          /* testCmd = 0x03 - use sendData with Basic Set toggle ON/OFF between rounds */
          /* testCmd = 0x04 - use sendDataMeta with Basic Set toggle ON/OFF between rounds */
          /* testCmd = 0x05 - use sendData with one group result frame (serial) at every round end */
          /* testCmd = 0x06 - use sendDataMeta with one group result frame (serial) at every round end */
          /* testCmd = 0x07 - use sendData with Basic Set toggle ON/OFF and one group result frame (serial) between rounds */
          /* testCmd = 0x08 - use sendDataMeta with Basic Set toggle ON/OFF and one group result frame (serial) between rounds */
          retVal = FALSE;
          testCmd = serFrameDataPtr[0];
          if (testCmd && (testState != POWERLEVEL_TEST_NODE_REPORT_ZW_TEST_INPROGRESS))
          {
            WORD_SET_HIGH_LOW_BYTES(testDelay, serFrameDataPtr[1], serFrameDataPtr[2]);
            testPayloadLen = serFrameDataPtr[3];
            WORD_SET_HIGH_LOW_BYTES(testCount, serFrameDataPtr[4], serFrameDataPtr[5]);
            testTxOptions = serFrameDataPtr[6];
            testnodemasklen = serFrameDataPtr[7];
            ZW_NODE_MASK_CLEAR(testnodemask, MAX_NODEMASK_LENGTH);
            for (i = 0; i < testnodemasklen; i++)
            {
              BYTE_IN_AR(testnodemask, i) = serFrameDataPtr[i + 8];
            }
            funcID_ComplHandler_Serial_API_Test = serFrameDataPtr[i + 8];
            if (testCount != 0)
            {
              testFailedCount = 0;
              testSuccessCount = 0;
              testState = POWERLEVEL_TEST_NODE_REPORT_ZW_TEST_INPROGRESS;
              TestStartRound();
              retVal = TRUE;
            }
            else
            {
              testState = POWERLEVEL_TEST_NODE_REPORT_ZW_TEST_FAILED;
              SendTestReport(0);
              break;
            }
          }
          else
          {
            /* STOP test ??? */
            if (!testCmd)
            {
              testState = POWERLEVEL_TEST_NODE_REPORT_ZW_TEST_FAILED;
              SendTestReport(0);
              retVal = TRUE;
            }
          }
          DoRespond();
          break;
#endif
#endif

#if SUPPORT_ZW_SET_SLEEP_MODE
        /* TO#1468 - SetSleepMode added */
        case FUNC_ID_ZW_SET_SLEEP_MODE:
          /* mode | intEnable */
          ZW_SET_SLEEP_MODE(serFrameDataPtr[0], serFrameDataPtr[1]);
          state = stateIdle;
          break;
#endif

#if SUPPORT_ZW_SET_ROUTING_MAX
        case FUNC_ID_ZW_SET_ROUTING_MAX:
          /* HOST->ZW: maxRouteTries */
          /* ZW->HOST: */
          retVal = ZW_SetRoutingMAX(serFrameDataPtr[0]);
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_GET_ROUTING_MAX
        case FUNC_ID_ZW_GET_ROUTING_MAX:
          /* HOST->ZW - No parameters */
          /* ZW->HOST: RES | currentMaxRouteTries */
          retVal = ZW_GetRoutingMAX();
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_RANDOM
        case FUNC_ID_ZW_RANDOM:
          /* HOST->ZW - No parameters */
          /* Get pseudo random number */
          retVal = ZW_Random();
          DoRespond();
          break;
#endif

#if SUPPORT_ZW_GET_RANDOM
        case FUNC_ID_ZW_GET_RANDOM:
          /* HOST->ZW - noRandomBytes - Optional if not present or equal ZERO then  */
          /*                            2 random bytes are returned.                */
          /*                            Range 1..32 random bytes are supported      */
          /* ZW->HOST - randomGenerationSuccess | noRandomBytesGenerated | noRandomGenerated[] */
          noRndBytes = serFrameDataPtr[0];
          if ((serFrameLen > FRAME_LENGTH_MIN) && (noRndBytes != 0))
          {
            if (noRndBytes > 32)
            {
              noRndBytes = 32;
            }
          }
          else
          {
            noRndBytes = 2;
          }
          i = 0;
          if (ZW_SetRFReceiveMode(FALSE))
          {
            do
            {
              if (!(ZW_GET_RANDOM_WORD(&rndWord, FALSE)))
              {
                break;
              }
              BYTE_IN_AR(compl_workbuf, 2 + i++) = BYTE_GET_HIGH_BYTE_IN_WORD(rndWord);
              BYTE_IN_AR(compl_workbuf, 2 + i++) = BYTE_GET_LOW_BYTE_IN_WORD(rndWord);
            } while (noRndBytes > i);
          }
          /* Number of random bytes returned */
          if (i > noRndBytes)
          {
            /* We made one Random Byte more than needed */
            i--;
          }
          /* If any random numbers have been generated then return TRUE */
          BYTE_IN_AR(compl_workbuf, 0) = (i != 0);
          /* This is the number of Random Numbers successfully generated */
          BYTE_IN_AR(compl_workbuf, 1) = i;
          /* Reinitialize the RF */
          ZW_GET_RANDOM_WORD(&rndWord, TRUE);
          /* Check if RF should be set in Receive mode. */
          ZW_SetRFReceiveMode((applNodeInfo_deviceOptionsMask & APPLICATION_NODEINFO_LISTENING) ? TRUE : FALSE);
          /* We always return 2 byte telling if any random numbers was generated */
          DoRespond_workbuf(2 + i);
          break;
#endif

        default :
          /* TODO - send a "Not Supported" respond frame */
          /* UNKNOWN - just drop it */
          state = stateIdle;
          break;
      }
    }
    break;

    case stateTxSerial :
    {
      /* Wait for ACK on send respond. Retransmit as needed */
#ifdef ENABLE_LEDS
      LED_OFF(1);
      LED_OFF(2);
      LED_ON(3);
#endif
      if ((conVal = ConUpdate(FALSE)) == conFrameSent)
      {
        /* One more RES transmitted succesfully */
        retry = 0;
        state = stateIdle;
      }
      else if (conVal == conTxTimeout)
      {
        /* Either a NAK has been received or we timed out waiting for ACK */
        if (retry++ < MAX_SERIAL_RETRY)
        {
          ConTxFrame(0, REQUEST, NULL, 0);  /* Retry... */
        }
        else
        {
          /* Drop RES as HOST could not be reached */
          retry = 0;
          state = stateIdle;
        }
      }
      /* All other states are ignored, as for now the only thing we are looking for is ACK/NAK! */
    }
    break;

    case stateCbTxSerial:
    {
      /* Wait for ack on unsolicited event (callback, ApplicationCommandHandler etc.) */
      /* Retransmit as needed. Remove frame from callbackqueue when done */
#ifdef ENABLE_LEDS
      LED_OFF(1);
      LED_ON(2);
      LED_ON(3);
#endif
      if ((conVal = ConUpdate(FALSE)) == conFrameSent)
      {
        /* One more REQ transmitted succesfully */
        PopCallBackQueue();
      }
      else if (conVal == conTxTimeout)
      {
        /* Either a NAK has been received or we timed out waiting for ACK */
        if (retry++ < MAX_SERIAL_RETRY)
        {
          ConTxFrame(0, REQUEST, NULL, 0);  /* Retry... */
        }
        else
        {
          /* Drop REQ as HOST could not be reached */
          PopCallBackQueue();
        }
      }
      /* All other states are ignored, as for now the only thing we are looking for is ACK/NAK! */
    }
    break;

    default:
      state = stateIdle;
      break;
  }
}


void
PopCallBackQueue(void)
{
  if (callbackCnt)
  {
    callbackCnt--;
    if (++callbackOut >= MAX_CALLBACK_QUEUE)
    {
      callbackOut = 0;
    }
  }
  else
  {
    callbackOut = callbackIn;
  }
  retry = 0;
  state = stateIdle;
}


/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*============================   ApplicationTestPoll   =======================
**    Function called instead of ApplicationPoll if ApplicationInitHW
**    returns false
**
**--------------------------------------------------------------------------*/
void
ApplicationTestPoll(void)
{
  /* If ApplicationInitHW returns FALSE this function will be called in place of
     ApplicationPoll */
}


/*==============================   ApplicationInitHW   ======================
**    Init UART and setup port pins for LEDs
**
**--------------------------------------------------------------------------*/
BYTE                   /*RET TRUE - If normal operation, FALSE - if production test*/
ApplicationInitHW(
  BYTE bWakeupReason)  /* IN Reason for wakeup - ZW_WAKEUP_RESET (RESET)   */
                       /*                      - ZW_WAKEUP_WUT (WUT)       */
                       /*                      - ZW_WAKEUP_SENSOR (SENSOR) */
{
  /* Productiontest pin not connected to a LED - hopefully */
#ifdef ENABLE_LEDS
  PIN_OUT(LED1);
  PIN_OUT(LED2);
  PIN_OUT(LED3);

  LED_OFF(3);
#endif /* ENABLE_LEDS */
  PIN_IN(Button, 1);
  ZW_UART_INIT(1152);   /* 115200 baud */

  return(TRUE); /* Return FALSE to enter production test mode */
}


/*==============================   ApplicationInitSW   ======================
**    Initialization of the Application Software
**
**--------------------------------------------------------------------------*/
BYTE                        /*RET  TRUE       */
ApplicationInitSW(  void )  /*IN Nothing */
{
  InitSerialIf();
  ConInit();

  ZW_MEMORY_GET_ID(appHomeId, &appNodeId);
  if (ZW_MEM_GET_BYTE(EEOFFSET_MAGIC) == MAGIC_VALUE)
  {
#ifdef ZW_SLAVE
    applNodeInfo_nodeType_generic = ZW_MEM_GET_BYTE(EEOFFSET_GENERIC);
    applNodeInfo_nodeType_specific = ZW_MEM_GET_BYTE(EEOFFSET_SPECIFIC);
#endif
#ifdef ZW_CONTROLLER
    ZW_GET_NODE_STATE(appNodeId, (NODEINFO *)compl_workbuf);
    /* Update application level nodetype variables */
    applNodeInfo_deviceOptionsMask = ((((NODEINFO *)compl_workbuf)->capability & NODEINFO_LISTENING_SUPPORT) ? APPLICATION_NODEINFO_LISTENING : 0)|
                                      ((((NODEINFO *)compl_workbuf)->security & NODEINFO_OPTIONAL_FUNC_SUPPORT ) ? APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY : 0);
    applNodeInfo_nodeType_generic = ((NODEINFO *)compl_workbuf)->nodeType.generic;
    applNodeInfo_nodeType_specific = ((NODEINFO *)compl_workbuf)->nodeType.specific;
#endif
    /* Update command class membership */
    applNodeInfo_parmLength = ZW_MEM_GET_BYTE(EEOFFSET_CMDCLASSS_LEN);
    if (applNodeInfo_parmLength > APPL_NODEPARM_MAX)
    {
      applNodeInfo_parmLength = APPL_NODEPARM_MAX;
    }
    for (i = 0; i < applNodeInfo_parmLength; i++)
    {
      applNodeInfo_nodeParm[i] = ZW_MEM_GET_BYTE(EEOFFSET_CMDCLASS + i);
    }
#if SUPPORT_ZW_WATCHDOG_START || SUPPORT_ZW_WATCHDOG_STOP
    /* Check if host has started the watchdog, and if it has then enable */
    /* the watchdog again */
    bWatchdogStarted = ZW_MEM_GET_BYTE(EEOFFSET_WATCHDOG_STARTED);
    if (bWatchdogStarted)
    {
      ZW_WATCHDOG_ENABLE;
    }
#endif
  }
  else
  {
    /* Probably first time we start after a reset */
    applNodeInfo_parmLength = 1;
#ifdef ZW_CONTROLLER
    /* Controllers are members of the COMMAND_CLASS_CONTROLLER_REPLICATION command class */
    applNodeInfo_nodeParm[0] = COMMAND_CLASS_CONTROLLER_REPLICATION;
#else
    /* Slaves are members of the COMMAND_CLASS_SWITCH_MULTILEVEL command class */
    applNodeInfo_nodeParm[0] = COMMAND_CLASS_SWITCH_MULTILEVEL;
#endif
    SaveApplicationSettings();
  }
  return TRUE;
}


#ifndef ZW_CONTROLLER_BRIDGE
/*==========================   ApplicationCommandHandler   ==================
**    Handling of received application commands and requests
**
**--------------------------------------------------------------------------*/
void                              /*RET Nothing                  */
ApplicationCommandHandler(
  BYTE  rxStatus,                 /* IN Frame header info */
#if defined(ZW_CONTROLLER) && !defined(ZW_CONTROLLER_STATIC)
  /* TO#1692 */
  BYTE  destNode,                 /* IN  Frame destination ID, only valid when frame is not Multicast */
#endif
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, */
                                  /*    the command is the very first byte */
  BYTE cmdLength)                 /* IN Number of command bytes including the command */
{
  /* ZW->PC: REQ | 0x04 | rxStatus | sourceNode | cmdLength | pCmd[] */

  /* TODO - should use SUPPORT_PROMISCUOUS define */
  /* TO#1802 fix */
  /* Syntax when a promiscuous frame is received (i.e. RECEIVE_STATUS_FOREIGN_FRAME is set): */
  /* ZW->PC: REQ | 0xD1 | rxStatus | sourceNode | cmdLength | pCmd[] | destNode */
#if defined(ZW_CONTROLLER) && !defined(ZW_CONTROLLER_STATIC)
/* For libraries supporting promiscuous mode... */
  if (TRUE == PreRequestPar3((rxStatus & RECEIVE_STATUS_FOREIGN_FRAME) ?
                              FUNC_ID_PROMISCUOUS_APPLICATION_COMMAND_HANDLER :
                              FUNC_ID_APPLICATION_COMMAND_HANDLER, rxStatus, sourceNode, cmdLength))
  {
    PostRequest(pCmd, cmdLength);
    if (rxStatus & RECEIVE_STATUS_FOREIGN_FRAME)
    {
      PostRequest(&destNode, 1);
    }
  }
#else
  /* Less code space-consuming version for libraries without promiscuous support */
  if (TRUE == PreRequestPar3(FUNC_ID_APPLICATION_COMMAND_HANDLER, rxStatus, sourceNode, cmdLength))
  {
    PostRequest(pCmd, cmdLength);
  }
#endif
}
#endif


#ifdef ZW_CONTROLLER_BRIDGE
/*======================   ApplicationCommandHandler_Bridge   ================
**    Handling of received application commands and requests
**
**--------------------------------------------------------------------------*/
void                              /*RET Nothing                  */
ApplicationCommandHandler_Bridge(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  destNode,                 /* IN Frame destination ID, only valid when frame is not Multicast */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_MULTI_DEST *multi,           /* IN multicast structure - only valid if multicast frame */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, */
                                  /*    the command is the very first byte */
  BYTE cmdLength)                 /* IN Number of command bytes including the command */
{
#if SUPPORT_APPLICATION_COMMAND_HANDLER_BRIDGE
  /* ZW->PC: REQ | 0xA8 | rxStatus | destNode | sourceNode | cmdLength | pCmd[] | multiDestsOffset_NodeMaskLen | multiDestsNodeMask[] */
  /* Unified Application Command Handler for Bridge and Virtual nodes */
  if (TRUE == PreRequestPar3(FUNC_ID_APPLICATION_COMMAND_HANDLER_BRIDGE, rxStatus, destNode, sourceNode))
  {
    /* cmdLength */
    PostRequest(&cmdLength, 1);
    /* pCmd[] */
    PostRequest(pCmd, cmdLength);
    if (multi)
    {
      retVal = (multi->multiDestsOffset_NodeMaskLen & MULTI_DEST_MASK_LEN_MASK) + 1;
      /* multiDestsOffset_NodeMaskLen */
      PostRequest(&retVal, 1);
      /* multiDestsNodeMask[] */
      PostRequest(multi, retVal);
    }
    else
    {
      retVal = 0;
      /* multiDestsOffset_NodeMaskLen = 0 */
      PostRequest(&retVal, 1);
    }
  }
#else
  /* Simulate old split Application Command Handlers */
  if (!ZW_IsVirtualNode(destNode))
  {
    /* ZW->PC: REQ | 0x04 | rxStatus | sourceNode | cmdLength | pCmd[] */
    if (TRUE == PreRequestPar3(FUNC_ID_APPLICATION_COMMAND_HANDLER, rxStatus, sourceNode, cmdLength))
    {
      PostRequest(pCmd, cmdLength);
    }
  }
  else
  {
    /* ZW->PC: REQ | 0xA1 | rxStatus | destNode | sourceNode | cmdLength | pCmd[] */
    if (TRUE == PreRequestPar3(FUNC_ID_APPLICATION_SLAVE_COMMAND_HANDLER, rxStatus, destNode, sourceNode))
    {
      PostRequest(&cmdLength, 1);
      PostRequest(pCmd, cmdLength);
    }
  }
#endif
}
#endif


/*==========================   ApplicationNodeInformation   =================
**    Request Application Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**--------------------------------------------------------------------------*/
void               /*RET Nothing */
ApplicationNodeInformation(
  BYTE   *deviceOptionsMask,     /*OUT Bitmask with application options     */
  APPL_NODE_TYPE *nodeType, /*OUT  Device type Generic and Specific   */
  BYTE      **nodeParm,     /*OUT  Device parameter buffer pointer    */
  BYTE      *parmLength )   /*OUT  Number of Device parameter bytes   */
{
  *deviceOptionsMask = applNodeInfo_deviceOptionsMask;
  (*nodeType).generic = applNodeInfo_nodeType_generic;  /* Generic Device Type */
  (*nodeType).specific = applNodeInfo_nodeType_specific;  /* Specific Device Type */
  *nodeParm = applNodeInfo_nodeParm;
  *parmLength = applNodeInfo_parmLength;
}


#if SUPPORT_APPLICATION_SLAVE_COMMAND_HANDLER
/*======================   ApplicationSlaveCommandHandler   ==================
**    Handling of a received application slave commands and requests
**
**--------------------------------------------------------------------------*/
void                  /*RET  Nothing                  */
ApplicationSlaveCommandHandler(
  BYTE  rxStatus,     /*IN  Frame header info */
  BYTE  destNode,     /* To whom it might concern - which node is to receive the frame */
  BYTE  sourceNode,   /*IN  Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER  *pCmd, /*IN  Payload from the received frame,
                                         the command is the very first byte */
  BYTE   cmdLength)   /*IN  Number of command bytes including the command */
{
  /* ZW->PC: REQ | 0xA1 | rxStatus | destNode | sourceNode | cmdLength | pCmd[] */
  if (TRUE == PreRequestPar3(FUNC_ID_APPLICATION_SLAVE_COMMAND_HANDLER, rxStatus, destNode, sourceNode))
  {
    PostRequest(&cmdLength, 1);
    PostRequest(pCmd, cmdLength);
  }
}
#endif


#ifdef ZW_CONTROLLER_BRIDGE
/* TODO - Bridge work, need to determine if we want more slave nodetypes */
/* and if this should be handled by sending the nodeinformation request to the */
/* PC, so it is up to the PC to set the nodeinformation when needed */
/* Or it is OK to set nodeinformation from PC side and then start the learning */
/*======================   ApplicationSlaveNodeInformation   =================
**    Request Application Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**--------------------------------------------------------------------------*/
void               /*RET Nothing */
ApplicationSlaveNodeInformation(
  BYTE      destNode,       /* IN Which node do we want the nodeinfo on */
  BYTE   *deviceOptionsMask,     /*OUT Bitmask with application options     */
  APPL_NODE_TYPE *nodeType, /*OUT  Device type Generic and Specific   */
  BYTE      **nodeParm,     /*OUT  Device parameter buffer pointer    */
  BYTE      *parmLength )   /*OUT  Number of Device parameter bytes   */
{
  *deviceOptionsMask = applSlaveNodeInfo_deviceOptionsMask;
  (*nodeType).generic = applSlaveNodeInfo_nodeType_generic;  /* Generic Device Type */
  (*nodeType).specific = applSlaveNodeInfo_nodeType_specific;  /* Specific Device Type */
  *nodeParm = applSlaveNodeInfo_nodeParm;
  *parmLength = applSlaveNodeInfo_parmLength;
}
#endif


#ifdef ZW_CONTROLLER
/*=====================   ApplicationControllerUpdate   =====================
**    Inform the static controller of node information update done through
**   the network managment.
**
**--------------------------------------------------------------------------*/
void                                /* RET  Nothing                         */
ApplicationControllerUpdate(
  BYTE bStatus,                     /* IN   Status of learn mode            */
  BYTE bNodeID,                     /* IN   Node id of node sending nodeinfo*/
  BYTE *pCmd,                       /* IN   Pointer to appl. node info      */
  BYTE bLen)                        /* IN   Node info length                */
{
  if (TRUE == PreRequestPar3(FUNC_ID_ZW_APPLICATION_UPDATE, bStatus, bNodeID, bLen))
  {
    PostRequest(pCmd, bLen);
  }
}
#endif /* ZW_CONTROLLER */


#ifdef ZW_SLAVE
/*======================   ApplicationSlaveUpdate   =========================
**    Inform the slave of node information received
**
**--------------------------------------------------------------------------*/
void                                /* RET  Nothing                         */
ApplicationSlaveUpdate(
  BYTE bStatus,                     /* IN   Status of learn mode            */
  BYTE bNodeID,                     /* IN   Node id of node sending nodeinfo*/
  BYTE *pCmd,                       /* IN   Pointer to appl. node info      */
  BYTE bLen)                        /* IN   Node info length                */
{
  if (TRUE == PreRequestPar3(FUNC_ID_ZW_APPLICATION_UPDATE, bStatus, bNodeID, bLen))
  {
    PostRequest(pCmd, bLen);
  }
}
#endif /* ZW_SLAVE */
