/**************************  PRODUCTIONTEST.C  ******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: This program starts by setting pin 0 on PORTA (now LED0) HIGH
 *  and if the button is pressed it sends 10 NOP frames while clearing LED0
 *  and setting pin 1 on PORTA (now LED1) HIGH, if ACK is received for all
 *  frames, LED1 continues to be HIGH, if not then LED1 is set LOW and LED0
 *  is set HIGH. By pressing button the sequence is started again...
 *
 * Author:   Johann Sigfredsson
 *
 * Last Changed By:  $Author: efh $
 * Revision:         $Revision: 22551 $
 * Last Changed:     $Date: 2012-03-20 16:27:45 +0100 (Tue, 20 Mar 2012) $
 *
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <config_app.h>
#include <ZW_sysdefs.h>
#include <ZW_typedefs.h>
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>
#include <ZW_classcmd.h>
#include <ZW_basis_api.h>
#include <ZW_slave_api.h>
#include <ZW_uart_api.h>
#if defined(ZW020x)
#include <ZW_RF020x.h>
#else
#include <ZW_RF030x.h>
#endif

extern IBYTE RF_freqno;
/*===============================   RFDriverInit   ==========================
**    Starts the sample interrupt and the generation of halfbit values,
**    and system time ticks.
**
**--------------------------------------------------------------------------*/
extern  void                /*RET Nothing           */
RFDriverInit( void ); /*IN  Nothing */

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/

#define ON  TRUE
#define OFF FALSE
#define TEST_COUNT 10     /*Initial test count*/
#define TEST_INTERVAL 2   /* 20ms */

#define TESTHOMEID  0x00000000
#define TESTDESTID  0x00

#define TRANSMIT_FAIL               0x04  /* A single_multicast/route frame transmit failed */

static BYTE homeID[4];
static BYTE destID;		 /* destination NodeID. */
static BYTE keyWasDepressed = 0, timerTestReady = 1;
static BYTE commandBuf[4] = {0, 0, 0, 0};
static BYTE count = 0, success = 0;

static BYTE test_count = TEST_COUNT;

#if defined(ZW020x) || defined(ZW030x)
/*This is defined in ZW_SLAVE_PRODTEST_ZW020XS.LIB.
 If TRUE NOP is sent with 40kbit/sec Default is FALSE*/
extern BOOL send40Kbit;
#endif
/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

static void                   /*RET  Nothing                  */
TimerTestHandler( void ); /*IN  Nothing                   */

#ifdef ZW030x
/*===========================   ApplicationRfNotify   ===========================
**    Notify the application when the radio switch state
**    Called from the Z-Wave PROTOCOL When radio switch from Rx to Tx or from Tx to Rx
**    or when the modulator PA (Power Amplifier) turn on/off
**---------------------------------------------------------------------------------*/
void          /*RET Nothing */
ApplicationRfNotify(
  BYTE rfState)          /* IN state of the RF, the available values is as follow:
                               ZW_RF_TX_MODE: The RF switch from the Rx to Tx mode, the modualtor is started and PA is on
                               ZW_RF_PA_ON: The RF in the Tx mode, the modualtor PA is turned on
                               ZW_RF_PA_OFF: the Rf in the Tx mode, the modulator PA is turned off
                               ZW_RF_RX_MODE: The RF switch from Tx to Rx mode, the demodulator is started.*/
{
  ; /* Do nothing */
}
#endif

void ToggleRED(void)
{
#if defined(ZW020x) || defined(ZW030x)
#else
  LED_TOGGLE(1);
#endif
#ifdef __ICCAVR__
  PORTA ^= 0x02;
  if (PORTA & 0x02)
  {
    LED_ON(1);
  }
  else
  {
    LED_OFF(1);
  }
#endif
}


void SetGREEN(BYTE on)
{
  if (on)
  {
    LED_ON(2);
#ifdef __ICCAVR__
    PORTA |= 0x01;
#endif
  }
  else
  {
    LED_OFF(2);
#ifdef __ICCAVR__
    PORTA &= ~(0x01);
#endif
  }
}


void SetRED(BYTE on)
{
  if (on)
  {
    LED_ON(1);
#ifdef __ICCAVR__
    PORTA |= 0x02;
#endif
  }
  else
  {
    LED_OFF(1);
#ifdef __ICCAVR__
    PORTA &= ~(0x02);
#endif
  }
}


/*==========================   ApplictionSlaveUpdate   =======================
**   Inform a slave application that a node information is received.
**   Called from the slave command handler when a node information frame
**   is received and the Z-Wave protocol is not in a state where it is needed.
**
**--------------------------------------------------------------------------*/
void
ApplicationSlaveUpdate(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen)       /*IN  Node info length                        */
{
}


/*==========================   SendCompletedHandler   =======================
**    Handling of a completed send command.
**    This function is called when the Z-Wave modul has completed sending a frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
static void       /*RET  Nothing                  */
SendCompleteHandler(
BYTE  commandCode) reentrant  /*IN  Send completed code  */
{
  if (commandCode == TRANSMIT_COMPLETE_NO_ACK)
  {
    /*Not received*/
#ifdef ZW010x
    LED_TOGGLE(4);
#endif

  }
  else
  {
    /*Received*/
    LED_TOGGLE(3);
    success++;
  }
  if (++count >= test_count)
  {
    if (success < test_count)
    {
      SetGREEN(OFF);
      SetRED(ON);
      ZW_UART_SEND_BYTE('-');
    }
    else
    {
      SetGREEN(ON);
      SetRED(OFF);
      ZW_UART_SEND_BYTE('+');
    }
    ZW_UART_SEND_NUM(success);
    count = success = 0;
    timerTestReady = 1;
  }
  else
  {
    ZW_TIMER_START(TimerTestHandler, TEST_INTERVAL, 1);
  }
}


/*=======================   TimerTestHandler   ========================
**
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void                   /*RET  Nothing                  */
TimerTestHandler( void ) /*IN  Nothing                   */
{
  ToggleRED();
  /* Send CMD_NOP frame */
  /* the transmit option 0x40 is TRANSMIT_OPTION_SINGLE, which isn't known */
  /* to the application-layer */
  if(!ZW_SEND_DATA(destID, commandBuf, sizeof(BYTE)*2, TRANSMIT_OPTION_ACK | 0x40, SendCompleteHandler))
  {
    SendCompleteHandler(TRANSMIT_COMPLETE_NO_ACK);
  }
}

/*=============================   ApplicationPoll   ========================
**    Application poll function
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void           /*RET  Nothing                  */
ApplicationPoll( void )  /* IN  Nothing                  */
{
  if (BUTTON_PRESSED())
  {
    if (timerTestReady)
    {
      keyWasDepressed = 1;
      timerTestReady = 0;
    }
  }
  else
  {
    /*No key is pressed*/
    if (keyWasDepressed)
    {
      keyWasDepressed = 0;
      SetGREEN(OFF);
      ToggleRED();
      ZW_TIMER_START(TimerTestHandler, TEST_INTERVAL, 1);
    }
    if(ZW_UART_REC_STATUS)
    {
      BYTE recData = ZW_UART_REC_BYTE;
      ZW_UART_SEND_BYTE(' ');
      switch (recData)
      {
        case 'U':
          {
            RF_freqno = RF_US;
            RFDriverInit();
            ZW_UART_SEND_BYTE('U');
            ZW_UART_SEND_BYTE('S');
          }
          break;
        case 'E':
          {
            RF_freqno = RF_EU;
            RFDriverInit();
            ZW_UART_SEND_BYTE('E');
            ZW_UART_SEND_BYTE('U');
          }
          break;
#ifndef ZW010x
        case 'A':
        	{
            RF_freqno = RF_ANZ;
            RFDriverInit();
            ZW_UART_SEND_BYTE('A');
            ZW_UART_SEND_BYTE('N');
          	break;
          }
        case 'H':
					{
            RF_freqno = RF_HK;
            RFDriverInit();
            ZW_UART_SEND_BYTE('H');
            ZW_UART_SEND_BYTE('K');
          	break;
          }
#endif
        case 'S':
          {
            if (timerTestReady)
            {
              keyWasDepressed = 1;
              timerTestReady = 0;
            }
            ZW_UART_SEND_BYTE('S');
            ZW_UART_SEND_BYTE('T');
          }
          break;
        case 'C':
          {
            test_count = ZW_UART_REC_BYTE;
            ZW_UART_SEND_BYTE('C');
            ZW_UART_SEND_BYTE('O');
          }
          break;
        case 'N':
          {
            destID = ZW_UART_REC_BYTE;
            ZW_UART_SEND_BYTE('N');
            ZW_UART_SEND_BYTE('I');
          }
        break;
        case 'R':
          {
            /*Reset the module*/
            ZW_UART_SEND_BYTE('R');
            ZW_UART_SEND_BYTE('S');
            while(ZW_UART_SEND_STATUS);
            ZW_WATCHDOG_ENABLE; /*reset asic*/
            while(1);
          }
        break;
#if defined(ZW020x) || defined(ZW030x)
				case 'B':
				if(ZW_UART_REC_BYTE=='4')
				{
					send40Kbit = TRUE;
				}
				else
				{
					send40Kbit = FALSE;
				}
				ZW_UART_SEND_BYTE('B');
        ZW_UART_SEND_BYTE(':');
        if(send40Kbit)
        {
        	ZW_UART_SEND_NUM(0x40);
        }
        else
        {
        	ZW_UART_SEND_BYTE('9');
        	ZW_UART_SEND_BYTE('.');
					ZW_UART_SEND_BYTE('6');
				}
				ZW_UART_SEND_BYTE('k');
				break;
#endif
        default:
          {
            ZW_UART_SEND_BYTE('!');
            ZW_UART_SEND_BYTE(recData);
          }
        break;

      }
    }
  }
}


/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*==============================   InitApplicationSW   ===========================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
#ifdef ZW010x
BYTE                       /*RET  TRUE        */
ApplicationInitHW( void )  /* IN  Nothing     */
#else
BYTE
ApplicationInitHW(
  BYTE bWakeupReason       /* Reason for the powerup of the chip */
)
#endif /* *ZW010x */
{
  /* hardware initialization */
  PIN_IN(Button, 1);

#ifdef __ICCAVR__
  PORTA = 0x00;
  DDRA |= 0x03; /* PORTA 0 og 1 now output ports */
#endif
  PIN_OUT(LED1);
  PIN_OUT(LED2);
  PIN_OUT(LED3);
#ifdef ZW010x
  PIN_OUT(LED4);
#endif
#ifdef __ICCAVR__
  PIN_OUT(LED5);
#endif
  return(TRUE);
}
/* Version string protocol   123456789012 (12 char max)*/
BYTE api_version[] 				= "Unknown    ";
BYTE code appl_version[] = {'\n','\r','V','e','r','s','i','o','n',' ',
                            ((APP_VERSION/10) ? (APP_VERSION/10)+0x30 : ' '),
                            (APP_VERSION%10)+0x30, '.',
                            (APP_REVISION/10)+0x30,
                            (APP_REVISION%10)+0x30, ' ',
                            0x00
                           };
//BYTE code appl_version[] = "\n\rVersion ";
/*===========================   InitApplicationSW   =========================
**    Initialization of the Application Software
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                   /*RET  TRUE       */
ApplicationInitSW(  void )  /*IN Nothing */
{
  *((DWORD *)&homeID[0]) = TESTHOMEID;

  ZW_DEBUG_INIT(1152);
  SetRED(ON);
  SetGREEN(OFF);

  LED_OFF(3);
#ifdef ZW010x
  LED_OFF(4);
#endif
#ifdef __ICCAVR__
  LED_OFF(5);
#endif
  destID = TESTDESTID;
  ZW_DEBUG_SEND_BYTE('@');
  ZW_UART_INIT(1152);
  ZW_UART_SEND_STRING(appl_version);
  ZW_Version(api_version);
	ZW_UART_SEND_STRING(api_version);
  return(TRUE);
}


/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the slave enters test mode.
**    Side effects:
**       Code will not exit until it is reset
**--------------------------------------------------------------------------*/
void ApplicationTestPoll(void)
{
  ApplicationPoll();
}


/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void       /*RET  Nothing                  */
ApplicationCommandHandler(
BYTE  rxStatus,   /*IN  Frame header info */
BYTE  sourceNode, /*IN  Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, the union */
                                  /*    should be used to access the fields */
BYTE   cmdLength) /*IN  Number of command bytes including the command */
{
}

/*==========================   GetNodeInformation   =========================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern  void               /*RET Nothing */
ApplicationNodeInformation(
BYTE   *deviceOptionsMask,     /*OUT Bitmask with application options     */
  APPL_NODE_TYPE  *nodeType,  /*OUT  Device type Generic and Specific   */
BYTE  **nodeParm,     /*OUT  Device parameter buffer pointer      */
BYTE  *parmLength )   /*OUT  Number of Device parameter bytes   */
{
  BYTE dimLevel = 0;

*deviceOptionsMask = APPLICATION_NODEINFO_LISTENING;             /* this is a powered node */
  nodeType->generic = GENERIC_TYPE_SWITCH_MULTILEVEL; /* Generic device type */
  nodeType->specific = 0;             /* We have no Specific class */
  *nodeParm = (BYTE *)&dimLevel;
  *parmLength = sizeof(dimLevel);
}
