/******************************* pushbutton.c  *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Meta data controller main file
 *
 * Author:   Peter Shorty
 *
 * Last Changed By:  $Author: efh $
 * Revision:         $Revision: 22535 $
 * Last Changed:     $Date: 2012-03-20 16:21:12 +0100 (Tue, 20 Mar 2012) $
 *
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <ZW_controller_api.h>
#include <ZW_uart_api.h>
#include <ZW_classcmd.h>
#include <ZW_debug_api.h>
#include <ZW_power_api.h>
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>
#include <one_button.h>

#define META_CTRL_VERSION    1
#define META_CTRL_REVISION   0

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

/*mainState states*/
#define STATE_IDLE                  0
#define STATE_WAITING_FOR_ADD       1
#define STATE_ADDING_DONE           2
#define META_DATA_GETTING           3
#define STATE_POWER_UP              4
#define STATE_WAITING_FOR_TIMEOUT   5
#define STATE_SENDING_WAKEUP        6
#define STATE_EXECUTING             10

#define SLEEP_TIMEOUT               100
#define SLEEP_TIMEOUT_REPEAT        4   /* Timeout is 10ms*SLEEP_TIMEOUT*SLEEP_TIMEOUT_REPEAT */
#define LONG_SLEEP_TIMEOUT          250
#define WAKE_TX_TIMEOUT             100

/* Function prototypes */
void StartSendingWakeup();
void StartSleepTimeout();

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
static BYTE bMainState;                /*State variable for the main loop*/
BYTE bSleepTimeout; /* Timer handler for sleep timer */
BYTE bSleepTimeoutCount;
BYTE  bMyNodeID;
BYTE receivingRemote;
BYTE bWakeupTxTimeout = 0;

BYTE bBellID = 0;

#define CLASS_MEMBERSHIP_COUNT  2

typedef struct s_nodeInfo_
{
   BYTE memberClass[CLASS_MEMBERSHIP_COUNT];  /* Command class membership */
} t_nodeInfo;

/* A list of the known command classes. Except the basic class which allways */
/* should be supported. Used when node info is send */
t_nodeInfo supportedCmdClasses = {COMMAND_CLASS_MANUFACTURER_SPECIFIC,
                                  COMMAND_CLASS_VERSION};

/* Buffer used to hold commands */
ZW_APPLICATION_TX_BUFFER txBuffer;

/****************************************************************************/
/*                              PRIVATE FUNCTIONS                           */
/****************************************************************************/

/*================================   GotoSleep   ============================
**    Power down the ZW0201 chip
**
**
**
**--------------------------------------------------------------------------*/
void GotoSleep()
{
  ZW_DEBUG_SEND_BYTE('S');

  /* Stop learn mode */
  ZW_AddNodeToNetwork(ADD_NODE_STOP, NULL);

  LED_OFF(1);
  LED_OFF(2);
  LED_OFF(3);

#ifdef ZW_DEBUG
  /* Empty debug buffer */
  while(UART_SendStatus())
    ZW_Poll();
#endif

  EX1 = 1;  /* enable int1 (button) before power down */

  /* Goto sleep */
  ZW_SetSleepMode(ZW_STOP_MODE, ZW_INT_MASK_EXT1);
}


/*=============================  SleepTimeout   =========================
**    Timeout function for going back to sleep
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void SleepTimeout()
{
  ZW_DEBUG_SEND_BYTE('T');
  ZW_DEBUG_SEND_NUM(bSleepTimeoutCount);

  /* Decrement timeout counter */
  bSleepTimeoutCount--;

  /* Check for timeout*/
  if (bSleepTimeoutCount==0)
  {
    /* Reset timer handler */
    bSleepTimeout = 0;

    /* If button is still held then do a reset of the controller */
    if (OneButtonLastAction() == BUTTON_IS_HELD)
    {
      ZW_DEBUG_SEND_BYTE('R');
      LED_ON(1);
      LED_ON(2);
      LED_ON(3);

      ZW_SetDefault(GotoSleep);
    }
    else
      GotoSleep();
  }
}

/*=============================  StartSleepTimeout   =========================
**    Start the timeout timer for going back to sleep
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void StartSleepTimeout()
{
  bSleepTimeoutCount = SLEEP_TIMEOUT_REPEAT;

  if (!bSleepTimeout)
    bSleepTimeout = TimerStart(SleepTimeout, LONG_SLEEP_TIMEOUT, SLEEP_TIMEOUT_REPEAT);
  else
    TimerRestart(bSleepTimeout);
}

/*=============================  StopSleepTimeout   =========================
**    Stop the timeout timer for going back to sleep
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void StopSleepTimeout()
{
  if (bSleepTimeout)
    TimerCancel(bSleepTimeout);
  bSleepTimeout = 0;
}

/*================================  SaveBellID   ============================
**    Save the node ID of the bell
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void SaveBellID(BYTE bNodeID)
{
  bBellID = bNodeID;
  MemoryPutByte(1, bNodeID);
}

/*================================  GetBellID   ============================
**    Save the node ID of the bell
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE GetBellID()
{
  register bTemp;
  bTemp = MemoryGetByte(1);
  if (bTemp == 0xff)
    bTemp = 0;

  return bTemp;
}


/*=========================   LearnModeCompleted =======================
**    Function description
**        Callback function used when adding devices to the controller
**          bStatus BYTE,     IN  Status of learn process
**          bSource BYTE,     IN  Node id of the node that send node info
**          BYTE* pCmd,       IN  Pointer to Node information
**          BYTE bLen));      IN  Node info length
**
**    Side effects: Shuts down RF receive mode when completed
**--------------------------------------------------------------------------*/
void
LearnModeCompleted(
LEARN_INFO *learnNodeInfo)
{
  ZW_DEBUG_SEND_BYTE('L');
  ZW_DEBUG_SEND_NUM(learnNodeInfo->bStatus);

  if (learnNodeInfo->bStatus == ADD_NODE_STATUS_DONE)
  {
    LED_OFF(2);
    LED_OFF(3);
    /*We have added a new device to the controller*/
    ZW_AddNodeToNetwork(ADD_NODE_STOP, NULL);
    bMainState = STATE_WAITING_FOR_TIMEOUT;
  }
  else if (learnNodeInfo->bStatus == ADD_NODE_STATUS_FAILED)
  {
    LED_OFF(3);
    /*We have added a new device to the controller*/
    ZW_AddNodeToNetwork(ADD_NODE_STOP, NULL);
    bMainState = STATE_WAITING_FOR_TIMEOUT;
  }
  else
  {
    StopSleepTimeout();
    if ((learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_CONTROLLER) ||
        (learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_SLAVE))
    {
      LED_ON(3);
      SaveBellID(learnNodeInfo->bSource);
    }
    bMainState = STATE_EXECUTING;
  }
}


/*==========================   ApplicationControllerUpdate ==================
**    Function description
**      This function is called when a Static Update Controller
**      command have been received by the protocol.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ApplicationControllerUpdate(
  BYTE bStatus,             /*IN  Status of learn process*/
  BYTE bSource,             /*IN  Node id of the node that send node info*/
  BYTE* pCmd,               /*IN  Pointer to application node information*/
  BYTE bLen)                /*IN  Node info length*/
{

}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*==============================   ApplicationInitHW  =======================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                   /*RET  TRUE       */
ApplicationInitHW( void )  /*IN Nothing */
{
  /* Disable external int 1 */
  EX1 = 0;

  /* hardware initialization */
  PIN_IN(Button, 1);

  /* Setup specifics for current dim module */
  PIN_OUT(LED1);
  PIN_OUT(LED2);
  PIN_OUT(LED3);


  return(TRUE);
}

/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software
**
**
**--------------------------------------------------------------------------*/
BYTE                   /*RET  TRUE       */
ApplicationInitSW( void )  /*IN Nothing */
{

  ZW_DEBUG_INIT(1152);

  ZW_DEBUG_SEND_BYTE('P');
  ZW_DEBUG_SEND_BYTE('U');
  ZW_DEBUG_SEND_BYTE('S');
  ZW_DEBUG_SEND_BYTE('H');
  ZW_DEBUG_SEND_BYTE('B');
  ZW_DEBUG_SEND_BYTE('U');
  ZW_DEBUG_SEND_BYTE('T');
  ZW_DEBUG_SEND_BYTE('T');
  ZW_DEBUG_SEND_BYTE('O');
  ZW_DEBUG_SEND_BYTE('N');
  ZW_DEBUG_SEND_BYTE(' ');

  /* Initialize button detection */
  OneButtonInit();

  /* Turn off all LED's */
  LED_OFF(1);
  LED_OFF(2);
  LED_OFF(3);

  /* Get our node ID */
  MemoryGetID(NULL, &bMyNodeID);

  ZW_DEBUG_SEND_BYTE('I');
  ZW_DEBUG_SEND_NUM(bMyNodeID);

  bBellID = GetBellID();

  bMainState = STATE_POWER_UP;

  return(TRUE);
}


/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the controller enters test mode.
**      It can be extended with the functionalities needed to verify the hardware
**      In this application it is kept very simple.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ApplicationTestPoll( void )
{

}

/*===========================  ApplicationPoll ===============================
**    Function description
**      Main poll routine. This routine should be registered as the application
**      poll routine.
**      This routine handles menu navigation and executes any functions
**      registered as internalfunctions.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing                  */
ApplicationPoll( void ) /* IN  Nothing                  */
{
  register BYTE bStatus;

  bStatus = OneButtonLastAction();

  switch (bMainState)
  {
    case STATE_POWER_UP:
      /* The push button only wakes up when the button is perssed, so check button */
      bMainState = STATE_IDLE;
      StartSleepTimeout();
      break;

    case STATE_IDLE:   /* Idle state check the button state */
      if (bStatus == BUTTON_WAS_PRESSED)
      {
        LED_ON(1);
        ZW_DEBUG_SEND_BYTE('s');
        StartSendingWakeup();
        bMainState = STATE_SENDING_WAKEUP;
      }
      else if (bStatus == BUTTON_IS_HELD)
      {
        LED_ON(2);
        ZW_DEBUG_SEND_BYTE('L');
        ZW_AddNodeToNetwork(ADD_NODE_ANY, LearnModeCompleted);
        bMainState = STATE_WAITING_FOR_ADD;
        StartSleepTimeout();
      }
      break;

    case STATE_WAITING_FOR_ADD:
      break;

    case STATE_ADDING_DONE:
      StopSleepTimeout();
      bMainState = STATE_IDLE;
      GotoSleep();
      break;

    case STATE_WAITING_FOR_TIMEOUT:
      StartSleepTimeout();
      bMainState = STATE_IDLE;
      break;
  }
}


/*========================  ApplicationCommandHandler  ======================
**    Function description
**        This callback function is called by the Z-Wave system when a
**        RF message that should be handled by the application is received
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ApplicationCommandHandler(        /*RET nothing*/
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame,
                                        the command is the very first byte */
  BYTE cmdLength)                 /* IN Number of command bytes including the command */
{
  switch(pCmd->ZW_Common.cmdClass)
  {

    case COMMAND_CLASS_CONTROLLER_REPLICATION:
      /* Handle replication frames */
      if (receivingRemote)
      {
        ZW_ReplicationReceiveComplete();
      }
      break;

    case COMMAND_CLASS_VERSION:
      if (pCmd->ZW_Common.cmd == VERSION_GET)
      {
        txBuffer.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuffer.ZW_VersionReportFrame.cmd = VERSION_REPORT;
        txBuffer.ZW_VersionReportFrame.libType = ZW_TYPE_LIBRARY();

        txBuffer.ZW_VersionReportFrame.proVersion = ZW_VERSION_MAJOR;
        txBuffer.ZW_VersionReportFrame.proSubVersion = ZW_VERSION_MINOR;

        txBuffer.ZW_VersionReportFrame.appVersion = META_CTRL_VERSION;
        txBuffer.ZW_VersionReportFrame.appSubVersion = META_CTRL_REVISION;
        ZW_SendData(sourceNode, (BYTE *)&txBuffer, sizeof(txBuffer.ZW_VersionReportFrame), TRANSMIT_OPTION_AUTO_ROUTE|TRANSMIT_OPTION_ACK, NULL);
      }
      break;

    default:
      /*Ignore any unknown commands*/
      break;
  }
}

/*======================ApplicationNodeInformation   =========================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern void                 /*RET Nothing */
ApplicationNodeInformation(
  BYTE  *listening,         /*OUT  TRUE if this node is always on air */
  APPL_NODE_TYPE *nodeType, /*OUT  Device type                        */
  BYTE  **nodeParm,         /*OUT  Device parameter buffer pointer    */
  BYTE  *parmLength )       /*OUT  Number of Device parameter bytes   */
{

  *listening = FALSE;             /* this is a battery powered node */
  nodeType->generic = GENERIC_TYPE_GENERIC_CONTROLLER;

  nodeType->specific = SPECIFIC_TYPE_PORTABLE_REMOTE_CONTROLLER;
  *nodeParm = (BYTE *)&supportedCmdClasses;
  *parmLength = sizeof(supportedCmdClasses);
}


/*============================= ActivateBell ===============================
**    Send on command to the bell
**
**
**
**--------------------------------------------------------------------------*/
void ActivateBell()
{
  txBuffer->ZW_BinarySwitchSetFrame.cmdClass  = COMMAND_CLASS_SWITCH_BINAR;
  txBuffer->ZW_BinarySwitchSetFrame.cmd       = WITCH_BINARY_SET;
  txBuffer->ZW_BinarySwitchSetFrame.value     = SWITCHED_ON;

  if (!ZW_SendData(bBellID, (BYTE *)&txBuffer,
              sizeof(txBuffer->ZW_BinarySwitchSetFrame),
              TRANSMIT_OPTION_ACK,
              ActivateBellComplete))
    ActivateBellComplete(TRANSMIT_COMPLETE_NO_ACK);
}


/*============================ StopSendingWakeup ==============================
**    Stop Sending Wakeup to the node
**
**
**
**--------------------------------------------------------------------------*/
void StopSendingWakeup()
{
  bWakeupTxTimeout = 0xFF;
}

/*============================ SendWakeupFrame ==============================
**    Start Sending Wakeup to the node
**
**
**
**--------------------------------------------------------------------------*/
void SendWakeupFrame(BYTE bStatus)
{
  if (bWakeupTxTimeout != 0xFF)
    ZW_SendWakeupFrame(bBellID, StartSendingWakeup);
  else
    ActivateBell();
}


/*========================= StartSendingWakeup ============================
**    Sending Wakeup to the node until the timer timies out
**
**
**
**--------------------------------------------------------------------------*/
void StartSendingWakeup(BYTE bStatus)
{
  bWakeupTxTimeout = TimerStart(StopSendingWakeup, WAKE_TX_TIMEOUT, 1);
  if (bWakeupTxTimeout == 0cFF)
    StopSendingWakeup();
  else
    SendWakeupFrame(TRUE);
}

