using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.UPnPBridge.Classes;
using Zensys.ZWave.UPnPBridge.Controllers;
using Zensys.ZWave.UPnPBridge.UI;
using Zensys.ZWave.UPnPBridge.Interfaces;
using WeifenLuo.WinFormsUI.Docking;
using System.IO;
using Zensys.ZWave.UPnPBridge.Properties;
using System.Windows.Forms;
using Zensys.ZWave.Devices;
using Zensys.ZWave.Enums;
using Zensys.ZWave.SerialPortApplication.Devices;
using System.Xml;

using Zensys.ZWave.Logging;
using Zensys.ZWave;
using Zensys.Framework;


namespace Zensys.ZWave.UPnPBridge.Actions
{
    public class MainFormActions : BaseAction
    {
        public MainFormActions(ControllerManager controller): base(controller){}
        
        #region Form
        public void OnFormLoad(object sender, EventArgs e)
        {
            ControllerManager.DocumentModel.Controllers.ListChanged += new System.ComponentModel.ListChangedEventHandler(ControllersListChanged);
            LayoutManager.OnLoad(ControllerManager, DeserializeDockContent);
            if (Settings.Default.CommandClassesStore != null)
            {
                Settings.Default.CommandClassesStore.CopyTo(ControllerManager.DocumentModel.ExternalCommandClassesStorage, 0);
            }
            if (!string.IsNullOrEmpty(Settings.Default.LastUsedDevice))
            {
                IBridgeController device = null;
                ControllerManager.DoAction(new EventHandler(delegate
                  {
                      device = DetectDevice(Settings.Default.LastUsedDevice);
                  }), null, Resources.MsgSettingsDetectInterface, false, (byte)CommandTypes.None);
                if (device != null && IsControllerLibrary(device.Version.Library))
                {
                    ControllerManager.Actions.ControllerFormActions.SetUpController(device);
                    ControllerManager.Actions.ControllerFormActions.SetUpController();
                }
            }
        }

        public void OnFormClosing(object sender, EventArgs e)
        {
            ControllerManager.ApplicationSettings.Synchronize();
            LayoutManager.OnClose(ControllerManager);
            if (ControllerManager.DocumentModel.Controller != null)
            {
                if (ControllerManager.DocumentModel.Controller.ConnectionStatus == Zensys.ZWave.Enums.ConnectionStatuses.Closed)
                {
                    ControllerManager.DocumentModel.Controller.Close();
                }
            }
            if (ControllerManager.DocumentModel.ExternalCommandClassesStorage != null)
            {
                Settings.Default.CommandClassesStore = ControllerManager.DocumentModel.ExternalCommandClassesStorage;
            }
            ControllerManager.Actions.ControllerFormActions.FinalizeEntryPointHAL();
            Settings.Default.Save();
        }
        

      
        #endregion

        #region Help
        public void OnAboutToolStripMenuItemClick(object sender, EventArgs e)
        {
            AboutBox about = new AboutBox();
            about.ShowDialog();
        }
        public void OnIndexToolStripMenuItemClick(object sender, EventArgs e)
        {

        }
        public void OnSearchToolStripMenuItemClick(object sender, EventArgs e)
        {

        }
        public void OnContentsToolStripMenuItemClick(object sender, EventArgs e)
        {

        }
        #endregion

        #region Views
        public void OnControllerToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(ControllerForm), ControllerManager.MainForm.ControllerToolStripMenuItem.Checked);
        }
        public void OnNodeToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(NodeForm), ControllerManager.MainForm.NodeToolStripMenuItem.Checked);
        }
        public void OnCmdClassToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(CommandClassForm), ControllerManager.MainForm.CmdClassToolStripMenuItem.Checked);
        }
        public void OnAssociationsToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(AssociationsForm), ControllerManager.MainForm.AssociationsToolStripMenuItem.Checked);
        }
        public void OnUPnPBinaryLightCPToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(UPnPBinaryLightCPForm), ControllerManager.MainForm.UPnPBinaryLightCPToolStripMenuItem.Checked);
        }
        public void OnUPnPBridgedDevicesToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(BridgedUPnPDevicesForm), ControllerManager.MainForm.UPnPBridgedDevicesToolStripMenuItem.Checked);
        }
        public void OnUPnPDeviceScannerToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(UPnPDeviceScannerForm), ControllerManager.MainForm.UPnPDeviceScannerToolStripMenuItem.Checked);
        }
        public void OnUPnPRendererCPToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(UPnPRendererCPForm), ControllerManager.MainForm.UPnPRendererCPToolStripMenuItem.Checked);
        }
        public void OnLogToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(LogForm), ControllerManager.MainForm.LogToolStripMenuItem.Checked);
        }
        #endregion

        #region Other

        private void ControllersListChanged(object sender, System.ComponentModel.ListChangedEventArgs e)
        {
            switch (e.ListChangedType)
            {
                case System.ComponentModel.ListChangedType.ItemAdded:
                    ControllerManager.MainForm.LabelStatusText = "";
                    ControllerManager.MainForm.LabelHomeIdText = Tools.ToHexString(ControllerManager.DocumentModel.Controller.HomeId);
                    ControllerManager.MainForm.LabelNodeIdText = Tools.ToHexString(ControllerManager.DocumentModel.Controller.Id);
                    break;
                case System.ComponentModel.ListChangedType.ItemChanged:
                    break;
                case System.ComponentModel.ListChangedType.ItemDeleted:
                    break;
                case System.ComponentModel.ListChangedType.ItemMoved:
                    break;
                case System.ComponentModel.ListChangedType.PropertyDescriptorAdded:
                    break;
                case System.ComponentModel.ListChangedType.PropertyDescriptorChanged:
                    break;
                case System.ComponentModel.ListChangedType.PropertyDescriptorDeleted:
                    break;
                case System.ComponentModel.ListChangedType.Reset:
                    ControllerManager.MainForm.LabelHomeIdText = "";
                    ControllerManager.MainForm.LabelNodeIdText = "";
                    break;
                default:
                    break;
            }
        }

        private IDockContent DeserializeDockContent(string stringId)
        {
            IDockContent ret = null;
            if (stringId == typeof(NodeForm).FullName)
            {
                ControllerManager.NodeForm = new NodeForm();
                ret = ControllerManager.NodeForm;
            } 
            else if (stringId == typeof(AssociationsForm).FullName)
            {
                ControllerManager.AssociationsForm = new AssociationsForm();
                ret = ControllerManager.AssociationsForm;
            }
            else if (stringId == typeof(CommandClassForm).FullName)
            {
                ControllerManager.CommandClassForm = new CommandClassForm();
                ret = ControllerManager.CommandClassForm;
            }
            else if (stringId == typeof(ControllerForm).FullName)
            {
                ControllerManager.ControllerForm = new ControllerForm();
                ret = ControllerManager.ControllerForm;
            }
            else if (stringId == typeof(LogForm).FullName)
            {
                ControllerManager.LogForm = new LogForm();
                ret = ControllerManager.LogForm;
            }
            else if (stringId == typeof(UPnPBinaryLightCPForm).FullName)
            {
                ControllerManager.UPnPBinaryLightCPForm = new UPnPBinaryLightCPForm();
                ret = ControllerManager.UPnPBinaryLightCPForm;
            }
            else if (stringId == typeof(BridgedUPnPDevicesForm).FullName)
            {
                ControllerManager.BridgedUPnPDevicesForm = new BridgedUPnPDevicesForm();
                ret = ControllerManager.BridgedUPnPDevicesForm;
            }
            else if (stringId == typeof(UPnPDeviceScannerForm).FullName)
            {
                ControllerManager.UPnPDeviceScannerForm = new UPnPDeviceScannerForm();
                ret = ControllerManager.UPnPDeviceScannerForm;
            }
            else if (stringId == typeof(UPnPRendererCPForm).FullName)
            {
                ControllerManager.UPnPRendererCPForm = new UPnPRendererCPForm();
                ret = ControllerManager.UPnPRendererCPForm;
            }
            else
            {
                ret = new DockContent();
            }
            return ret;
        }
        
        public void OnExitToolStripMenuItemClick(object sender, EventArgs e)
        {
            System.Windows.Forms.Application.Exit();
        }

        public IBridgeController DetectDevice(string deviceId)
        {
            bool isError = false;
            IBridgeController device = null;
            device = ControllerManager.ZWaveManager.ApplicationLayer.CreateBridgeController();
            try
            {
                device.Open(deviceId);
                device.GetVersion();
                if (device != null && IsControllerLibrary(device.Version.Library))
                {
                    //device.GetCapabilities();
                    //device.GetControllerCapabilities();
                    //device.CommandClassesStore.Load();
                    //device.Memory.GetId();
                    //device.GetSUCNodeID();
                    //device.GetNodes();
                }
            }
            catch (Exception)
            {
                isError = true;
            }
            finally
            {
                if (device != null)
                {
                    device.Close();
                }
            }
            return isError ? null : device;
        }

        public void OnSettingsToolStripMenuItemClick(object sender, EventArgs e)
        {
            IBridgeController device = null;
            SettingsForm settingsForm = new SettingsForm();
            settingsForm.OnLoadInterfaces += new EventHandler(SettingsOnLoadInterfaces);
            settingsForm.OnDetectInterface += new EventHandler(SettingsOnDetectInterface);
            if (settingsForm.ShowDialog(ControllerManager.MainForm) == DialogResult.OK)
            {

                if (settingsForm.SelectedDevice != null && IsControllerLibrary(settingsForm.SelectedDevice.Version.Library))
                {
                    ControllerManager.Actions.ControllerFormActions.SetUpController(settingsForm.SelectedDevice);
                    ControllerManager.Actions.ControllerFormActions.SetUpController();
                    SaveSettings(ControllerManager.DocumentModel.Controller.SerialPort);
                }
                else
                {
                    ControllerManager.DoAction(new EventHandler(delegate
                    {
                        try
                        {
                            if (ControllerManager.DocumentModel.Controller != null)
                            {
                                ControllerManager.DocumentModel.Controller.Close();
                            }
                            device = DetectDevice(settingsForm.SelectedDeviceId);
                            if (device != null)
                            {

                            }
                        }
                        catch (Exception ex)
                        {
                            throw ex;
                        }
                    }), null, Resources.MsgSettingsDetectInterface, false, (byte)CommandTypes.None);


                    if (device != null && IsControllerLibrary(device.Version.Library))
                    {
                        ControllerManager.Actions.ControllerFormActions.SetUpController(device);
                        ControllerManager.Actions.ControllerFormActions.SetUpController();
                        SaveSettings(device.SerialPort);
                    }
                    else if (device != null)
                    {
                        SaveSettings("");
                        MessageBox.Show(Resources.MsgNoControllerLibrary, Resources.CaptionZWaveController);
                    }
                    else
                    {
                        SaveSettings("");
                        MessageBox.Show(Resources.MsgNoControllerDevice, Resources.CaptionZWaveController);
                    }
                }
            }
        }

        private void SaveSettings(string serialPortName)
        {
            Settings.Default.LastUsedDevice = serialPortName;
            Settings.Default.Save();
        }

        private void SettingsOnDetectInterface(object sender, EventArgs e)
        {
            if (sender is SettingsForm)
            {
                SettingsForm form = sender as SettingsForm;
                IBridgeController device = null;
                SortedList<string, IBridgeController> detectedControllers = new SortedList<string, IBridgeController>();
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    try
                    {
                        if (ControllerManager.DocumentModel.Controller != null)
                        {
                            ControllerManager.DocumentModel.Controller.Close();
                        }
                        List<Win32PnPEntityClass> interfaces = ComputerSystemHardwareHelper.GetWin32PnPEntityClassSerialPortDevices();
                        foreach (Win32PnPEntityClass serialPortInfo in interfaces)
                        {

                            device = DetectDevice(serialPortInfo.DeviceID);
                            if (device != null)
                            {
                                detectedControllers.Add(serialPortInfo.DeviceID, device);
                            }

                        }
                    }
                    catch (Exception ex)
                    {
                        throw ex;
                    }

                }), null, Resources.MsgSettingsDetectInterface, false, (byte)CommandTypes.None);
                LoadInterfaces(form, detectedControllers);
            }
        }

        private void SettingsOnLoadInterfaces(object sender, EventArgs e)
        {
            if (sender is SettingsForm)
            {
                SettingsForm form = sender as SettingsForm;
                LoadInterfaces(form, new SortedList<string, IBridgeController>());
            }
        }

        private void LoadInterfaces(SettingsForm form, SortedList<string, IBridgeController> detectedControllers)
        {
            List<Win32PnPEntityClass> interfaces = ComputerSystemHardwareHelper.GetWin32PnPEntityClassSerialPortDevices();
            form.InterfacesListBox.Items.Clear();
            form.InterfacesListBox.Enabled = true;
            form.NoCommInterfacesLabel.Text = "";
            if (interfaces != null && interfaces.Count > 0)
            {
                foreach (Win32PnPEntityClass serialPortInfo in interfaces)
                {
                    IBridgeController device = null;
                    if (detectedControllers.ContainsKey(serialPortInfo.DeviceID))
                    {
                        device = detectedControllers[serialPortInfo.DeviceID];
                    }
                    int index = form.InterfacesListBox.Items.Add(new DeviceWrapper(serialPortInfo, device));
                    if (ControllerManager.DocumentModel.Controller != null &&
                        ControllerManager.DocumentModel.Controller.SerialPort == serialPortInfo.DeviceID)
                    {
                        form.InterfacesListBox.SelectedIndex = index;
                    }

                }
            }
            else
            {
                form.InterfacesListBox.Enabled = false;
                form.NoCommInterfacesLabel.Text = Resources.MsgNoCommInterfaces;
            }
        }
        #endregion
    }
}