
using System;
using System.Collections.Generic;
using System.Text;
using Zensys.PCController.Classes;
using Zensys.PCController.Controllers;
using Zensys.ZWave.Devices;
using Zensys.PCController.Properties;
using Zensys.ZWave.Enums;
using System.Windows.Forms;
using Zensys.ZWave.SerialPortApplication.Devices;
using System.Drawing;
using Zensys.Framework;
using Zensys.ZWave;
using Zensys.Framework.Helpers;
using System.Threading;
using Zensys.Framework.UI.Controls;
using Zensys.ZWave.ZWaveHAL.Actions;
using Zensys.ZWave.ZWaveHAL;
using Zensys.ZWave.Application;

namespace Zensys.PCController.Actions
{
    public class NodeFormActions : BaseAction
    {
        NodesViewActions NodesHAL;
        public NodeFormActions(ControllerManager controller)
            : base(controller)
        {
            NodesHAL = CustomLoader.CreateNodesViewActionsInstance();
            NodesHAL.Initialize(controller.LogManager, controller.ExceptionManager, controller.XmlDataManager);
            NodesHAL.AddQueueCommandCallback = controller.AddQueueCommand;
            NodesHAL.ExecuteQueueCommandCallback = controller.ExecuteQueueCommand;
            NodesHAL.ItemChanged += new EventHandler(NodesHAL_ItemChanged);
            NodesHAL.NodeAdded += new EventHandler<NodeEventArgs>(NodesHAL_NodeAdded);
            base.ActionsHAL = NodesHAL;
        }

        private void NodesHAL_NodeAdded(object sender, NodeEventArgs e)
        {
            AddControllerNode(e.Device);
        }

        void NodesHAL_ItemChanged(object sender, EventArgs e)
        {
            int position = -1;
            if (sender != null && sender is IDeviceInfo)
            {
                IDeviceInfo device = (IDeviceInfo)sender;
                if (device.SupportedCommandClasses != null &&
                    ControllerManager.DocumentModel.Controller.CommandClassesStore.IsSupported(device.Id, ControllerManager.XmlDataManager.GetCommandClassKey("COMMAND_CLASS_ASSOCIATION"))
                    //||
                    //CustomLoader.GetEntryPoint().CustomCommandClassesStore != null &&
                    //CustomLoader.GetEntryPoint().CustomCommandClassesStore.IsSupported(device.Id, ControllerManager.XmlDataManager.GetCommandClassKey("COMMAND_CLASS_ASSOCIATION"))
                    )
                {
                    ControllerManager.DocumentModel.RemoveAssociativeDevice(device.Id);
                    ControllerManager.DocumentModel.AssociativeDevices.Add(device);
                    ControllerManager.DocumentModel.SetDocumentModelStateChanged();
                }
                foreach (IDeviceInfo d in ControllerManager.DocumentModel.Devices)
                {
                    position++;
                    if (d.Id == ((IDeviceInfo)sender).Id)
                        break;
                }
                if (position > -1)
                {
                    DataGridViewSelectedRowCollection selectedRows = ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.SelectedRows;
                    List<int> selectedIndex = new List<int>();
                    foreach (DataGridViewRow r in selectedRows)
                    {
                        selectedIndex.Add(r.Index);
                    }

                    ControllerManager.DocumentModel.Devices.RemoveAt(position);
                    ControllerManager.DocumentModel.Devices.Insert(position, device);
                    ControllerManager.DocumentModel.DevicesCurrencyManager.Position = position;

                    foreach (DataGridViewRow r in ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.Rows)
                    {
                        r.Selected = selectedIndex.Contains(r.Index);
                    }
                    //ControllerManager.DocumentModel.Devices.ResetItem(position);
                }
            }
        }

        #region Form
        public void OnFormLoad(object sender, EventArgs e)
        {
            ControllerManager.NodeForm.nodeGridViewControl.btndSetSUC.Text = "Set SUC/SIS";

            DataGridViewColumn colNodeId = new DataGridViewTextBoxColumn();
            DataGridViewColumn colNodeType = new DataGridViewTextBoxColumn();
            DataGridViewBitImageColumn colIsListening = new DataGridViewBitImageColumn();
            DataGridViewCheckBoxColumn colCommandQueueOverrided = new DataGridViewCheckBoxColumn();
            DataGridViewBitImageColumn colIsVirtual = new DataGridViewBitImageColumn();

            colNodeId.Name = "colId";
            colNodeId.HeaderText = "Id";
            colNodeId.DataPropertyName = "Id";
            colNodeId.AutoSizeMode = DataGridViewAutoSizeColumnMode.AllCellsExceptHeader;

            colNodeType.Name = "colType";
            colNodeType.HeaderText = "Type";
            colNodeType.DataPropertyName = "Type";
            colNodeType.MinimumWidth = 100;
            colNodeType.FillWeight = 100;
            colNodeType.AutoSizeMode = DataGridViewAutoSizeColumnMode.Fill;

            colIsListening.Name = "colIsListening";
            colIsListening.HeaderText = "Listening";
            colIsListening.DataPropertyName = "IsListening";
            colIsListening.Images = ControllerManager.NodeForm.nodeGridViewControl.Images;
            colIsListening.TrueImageIndex = 0;
            colIsListening.AutoSizeMode = DataGridViewAutoSizeColumnMode.AllCellsExceptHeader;
            colIsListening.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleCenter;
            colIsListening.MinimumWidth = 30;

            colCommandQueueOverrided.Name = "colCommandQueueOverrided";
            colCommandQueueOverrided.HeaderText = "Queue Overrided";
            colCommandQueueOverrided.DataPropertyName = "CommandQueueOverrided";
            colCommandQueueOverrided.AutoSizeMode = DataGridViewAutoSizeColumnMode.AllCellsExceptHeader;
            colCommandQueueOverrided.HeaderCell.Style.WrapMode = DataGridViewTriState.False;
            colCommandQueueOverrided.ThreeState = true;
            colCommandQueueOverrided.MinimumWidth = 30;

            colIsVirtual.Name = "colIsVirtual";
            colIsVirtual.HeaderText = "Virtual";
            colIsVirtual.DataPropertyName = "IsVirtual";
            colIsVirtual.Images = ControllerManager.NodeForm.nodeGridViewControl.Images;
            colIsVirtual.TrueImageIndex = 1;
            colIsVirtual.AutoSizeMode = DataGridViewAutoSizeColumnMode.AllCellsExceptHeader;
            colIsVirtual.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleCenter;
            colIsVirtual.MinimumWidth = 30;

            ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill;
            ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.AutoGenerateColumns = false;
            ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.Columns.AddRange(new DataGridViewColumn[] { colNodeId, colNodeType, colIsListening, colCommandQueueOverrided, colIsVirtual });
            BindData();


            ControllerManager.NodeForm.vcNodeInfoTreeView.ImageList = ControllerManager.NodeForm.nodeGridViewControl.Images;
            ControllerManager.NodeForm.vcNodeInfoTreeView.DataSource = ControllerManager.DocumentModel.Devices;
            ControllerManager.NodeForm.vcNodeInfoTreeView.CreatePrefetchedObject = CreatePrefetchedObject;
            OnDocumentModelStateChanged();
        }

        public void CellContentClick(object sender, DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex == 3)
            {
                if (!ControllerManager.DocumentModel.CurrentDevice.IsListening)
                {
                    bool val = ControllerManager.DocumentModel.CurrentDevice.IsCommandQueueOverrided;
                    ControllerManager.DocumentModel.CurrentDevice.CommandQueueOverrided = !val;
                }
            }
        }

        private void BindData()
        {
            ControllerManager.DocumentModel.Devices.Ctx = SynchronizationContext.Current;
            ControllerManager.NodeForm.nodeGridViewControl.nodesGridView.DataSource = ControllerManager.DocumentModel.Devices;
        }

        private PrefetchedObject CreatePrefetchedObject(object currentObject)
        {
            IDeviceInfo device = (IDeviceInfo)currentObject;
            List<ParamValue> payloadValues = GetNodeInfoPayloadValues(device);
            PrefetchedObject poRoot = new PrefetchedObject();
            poRoot.Text = string.Format("Id: {0}; {1}", device.Id, device.Type);
            poRoot.ImageIndex = 2;
            foreach (ParamValue paramValue in payloadValues)
            {
                List<string> str = paramValue.TextValueList;
                PrefetchedObject po = new PrefetchedObject();
                po.ImageIndex = 3;
                if (paramValue.TextValueList.Count == 1)
                {
                    po.Text = string.Format("{0}: {1}", paramValue.ParamDefinition.Text, paramValue.TextValueList[0]);
                }
                else if (paramValue.TextValueList.Count > 1)
                {
                    po.Text = string.Format("{0}: {1}", paramValue.ParamDefinition.Text, Tools.GetHex(paramValue.ByteValueList.ToArray(), " ", false));
                    foreach (string var in paramValue.TextValueList)
                    {
                        PrefetchedObject subpo = new PrefetchedObject();
                        subpo.Text = var;
                        subpo.ImageIndex = 4;
                        po.Children.Add(subpo);
                    }
                }
                else
                {
                    po.Text = string.Format("{0}: {1}", paramValue.ParamDefinition.Text, Tools.GetHex(paramValue.ByteValueList.ToArray(), " ", false));
                }
                if (!String.IsNullOrEmpty(paramValue.ParamDefinition.GroupName))
                {
                    PrefetchedObject groupPo = FindPrefetchedObject(poRoot, paramValue.ParamDefinition.GroupName);
                    if (groupPo != null)
                    {
                        groupPo.Children.Add(po);
                    }
                    else
                    {
                        groupPo = new PrefetchedObject();
                        groupPo.ImageIndex = 3;
                        groupPo.Text = paramValue.ParamDefinition.GroupName;
                        groupPo.Children.Add(po);
                        poRoot.Children.Add(groupPo);
                    }
                }
                else
                {
                    poRoot.Children.Add(po);
                }
            }

            return poRoot;
        }

        private PrefetchedObject FindPrefetchedObject(PrefetchedObject node, string name)
        {
            PrefetchedObject result = null;
            if (node.Text == name) return node;
            if (node.Children != null && node.Children.Count > 0)
            {
                foreach (PrefetchedObject child in node.Children)
                {
                    result = FindPrefetchedObject(child, name);
                }
            }
            return result;
        }

        private List<ParamValue> GetNodeInfoPayloadValues(IDeviceInfo device)
        {
            CommandClass cmdClass = ControllerManager.XmlDataManager.FindCommandClass("ZWAVE_CMD_CLASS", 1);
            Command cmd = ControllerManager.XmlDataManager.FindCommand("ZWAVE_CMD_CLASS", 1, "NODE_INFO");
            List<byte> payload = new List<byte>();
            payload.Add(cmdClass.KeyId);
            payload.Add(cmd.KeyId);
            payload.AddRange(new byte[] { device.Capability, device.Security, device.Reserved, /*device.Basic,*/ device.Generic, device.Specific });
            if (device.SupportedCommandClasses != null)
            {
                payload.AddRange(device.SupportedCommandClasses);
            }
            CommandClassValue[] cmdClassValues = null;
            ControllerManager.XmlDataManager.ParseApplicationObject(payload.ToArray(), out cmdClassValues);
            if (cmdClassValues != null)
            {
                return cmdClassValues[0].CommandValue.ParamValues;
            }
            return null;
        }

        public void OnFormClosing(object sender, FormClosingEventArgs e)
        {
            ControllerManager.MainForm.NodeToolStripMenuItem.Checked = false;
            ControllerManager.NodeForm.Hide();
            e.Cancel = true;
        }
        #endregion

        #region DocumentModel
        public void OnDocumentModelStateChanged(object sender, EventArgs e)
        {
            OnDocumentModelStateChanged();
        }

        private delegate void OnDocumentModelStateChangedDelegate();
        private void OnDocumentModelStateChanged()
        {
            if (ControllerManager.NodeForm != null && !ControllerManager.NodeForm.IsDisposed)
            {
                if (ControllerManager.NodeForm.InvokeRequired)
                {
                    ControllerManager.NodeForm.Invoke(new OnDocumentModelStateChangedDelegate(OnDocumentModelStateChanged));
                }
                else
                {
                    ControllerManager.NodeForm.vcSetSucSisToolStripButton.Visible = true;

                    ControllerManager.NodeForm.vcAddNodeToolStripButton.Enabled =
                      ControllerManager.DocumentModel.Controller != null &&
                      ((ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Secondary) == 0 ||
                      (ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Inclusion) != 0);

                    ControllerManager.NodeForm.vcAddNodeNetworkWideToolStripButton.Enabled =
                      ControllerManager.DocumentModel.Controller != null &&
                      ((ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Secondary) == 0 ||
                      (ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Inclusion) != 0);

                    ControllerManager.NodeForm.vcRemoveNodeToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null &&
                      ((ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Secondary) == 0 ||
                      (ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Inclusion) != 0);

                    ControllerManager.NodeForm.vcNodesListView.Enabled =
                        ControllerManager.DocumentModel.Controller != null;

                    ControllerManager.NodeForm.vcNodeInfoTreeView.Enabled =
                        ControllerManager.DocumentModel.Controller != null;

                    ControllerManager.NodeForm.vcRemoveFailedNodeToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null &&
                      ((ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Secondary) == 0 ||
                      (ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Inclusion) != 0) &&
                        ControllerManager.DocumentModel.DevicesCurrencyManager.Position > -1;

                    ControllerManager.NodeForm.vcIsFailedNodeToolStripButton.Enabled =
                                            ControllerManager.DocumentModel.Controller != null &&
                                            ControllerManager.DocumentModel.DevicesCurrencyManager.Position > -1;

                    ControllerManager.NodeForm.vcReplaceFailedNodeToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null &&
                      ((ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Secondary) == 0 ||
                      (ControllerManager.DocumentModel.Controller.NetworkRole & ControllerRoles.Inclusion) != 0) &&
                        ControllerManager.DocumentModel.DevicesCurrencyManager.Position > -1;

                    ControllerManager.NodeForm.vcSetAllOffToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null;

                    ControllerManager.NodeForm.vcSetAllOnToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null;

                    ControllerManager.NodeForm.vcSetOffToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null &&
                        ControllerManager.DocumentModel.DevicesCurrencyManager.Position > -1;

                    ControllerManager.NodeForm.vcSetOnToolStripButton.Enabled =
                        ControllerManager.DocumentModel.Controller != null &&
                        ControllerManager.DocumentModel.DevicesCurrencyManager.Position > -1;

                    ControllerManager.NodeForm.vcToolStripMain.Enabled =
                        ControllerManager.DocumentModel.Controller != null;
                }
            }
        }

        #endregion

        #region Toolbar

        public void OnRequestNodeNeighborUpdateClick(object sender, EventArgs e)
        {
            ControllerManager.DoAction(new EventHandler(delegate
            {
                if (ControllerManager.DocumentModel.Controller != null)
                {
                    if (ControllerManager.DocumentModel.Controller.ConnectionStatus == Zensys.ZWave.Enums.ConnectionStatuses.Closed)
                    {
                        ControllerManager.DocumentModel.Controller.Open(ControllerManager.DocumentModel.Controller.SerialPort);
                    }
                    if (ControllerManager.DocumentModel.CurrentDevice != null)
                    {
                        RequestNeighborUpdateStatuses result = ControllerManager.DocumentModel.Controller.RequestNodeNeighborUpdate(ControllerManager.DocumentModel.CurrentDevice.Id);
                        switch (result)
                        {
                            case RequestNeighborUpdateStatuses.RequestNeighborUpdateDone:
                                {
                                    ControllerManager.ShowMessage("Request Neighbors Update Done", false);
                                } break;
                            case RequestNeighborUpdateStatuses.RequestNeighborUpdateFailed:
                                {
                                    ControllerManager.ShowMessage("Request Neighbor Update Failed", false);
                                } break;
                            case RequestNeighborUpdateStatuses.RequestNeighborUpdateStarted: { } break;
                            default:
                                {
                                    ControllerManager.ShowMessage("Request Neighbor Update Failed", false);
                                } break;
                        }
                    }
                }
            }), null, "Requesting Neighbor Update...", false, (byte)CommandTypes.None);
        }

        public void OnAddNodeClick(object sender, EventArgs e)
        {
            nodesAddedFailedList.Clear();
            IDeviceInfo addedNode = (IDeviceInfo)ControllerManager.DoAction(
                new FunctionCaller<IDeviceInfo, IController>(NodesHAL.AddNodeAction),
                new object[] { ControllerManager.DocumentModel.Controller },
                Resources.MsgAddNode, true, (byte)CommandTypes.CmdZWaveAddNodeToNetwork);
        }

        public void OnNetworkWideInclusionClick(object sender, EventArgs e)
        {
            ControllerManager.DoAction(
                new Action<IController>(OnNetworkWideInclusionInner),
                new object[] { ControllerManager.DocumentModel.Controller },
                Resources.MsgAddNodeNWI, true, (byte)CommandTypes.CmdZWaveAddNodeToNetwork);
        }

        public void OnNetworkWideInclusionInner(IController controller)
        {
            nodesAddedFailedList.Clear();
            IDeviceInfo addedNode = null;
            ControllerManager.NetworkWideInclusionCanceled = false;
            while (!ControllerManager.NetworkWideInclusionCanceled)
            {
                addedNode = NodesHAL.NetworkWideInclusionAction(ControllerManager.DocumentModel.Controller);
            }
        }

        public void OnRemoveNodeClick(object sender, EventArgs e)
        {
            byte removedNodeId = (byte)ControllerManager.DoAction(
                new FunctionCaller<byte, IController>(NodesHAL.RemoveNodeAction),
                new object[] { ControllerManager.DocumentModel.Controller },
                Resources.MsgRemoveNode, true, (byte)CommandTypes.CmdZWaveRemoveNodeFromNetwork);
            if (removedNodeId > 0)
            {
                RemoveControllerNodeInternal(removedNodeId);
            }
        }
        public void OnNodeInfoClick(object sender, EventArgs e)
        {

            IDeviceInfo device = ControllerManager.DocumentModel.CurrentDevice;
            QueueAction action = new QueueAction(
                            new FunctionCaller<IDeviceInfo, IController, byte, IDeviceInfo>(NodesHAL.NodeInfoAction),
                            new object[] 
                        { 
                    ControllerManager.DocumentModel.Controller, 
                    device.Id, 
                    device },
                         Resources.MsgGetNodeReport, true, (byte)CommandTypes.CmdZWaveRequestNodeInfo, "Request Node Information", 0);

            if (device.IsListening || device.IsCommandQueueOverrided || (device.Security & 0x20) != 0 || (device.Security & 0x40) != 0)
            {
                IDeviceInfo deviceInfo = (IDeviceInfo)ControllerManager.DoAction(action);
                device = deviceInfo;
                //ControllerManager.DocumentModel.Devices.Remove(device);
                //ControllerManager.DocumentModel.Devices.Add(deviceInfo);
            }
            else
            {
                ControllerManager.AddQueueCommand(device.Id, action);
                ControllerManager.LogManager.LogMessage(string.Format("{0} added to command queue for Node {1}.", "Request Node Information", device.Id));
            }

        }

        public void OnSetOnClick(object sender, EventArgs e)
        {
            SendBasicSetOnOff(true, GetSelectedDevices());
        }

        private List<IDeviceInfo> GetSelectedDevices()
        {
            return ControllerManager.Actions.CommonActions.GetSelectedDevices(false);
        }


        public void OnSetOffClick(object sender, EventArgs e)
        {
            SendBasicSetOnOff(false, GetSelectedDevices());
        }
        public void OnSetAllOnClick(object sender, EventArgs e)
        {
            SendSwitchOnOff(true);
        }
        public void OnSetAllOffClick(object sender, EventArgs e)
        {
            SendSwitchOnOff(false);
        }
        public void OnReplaceFailedNodeClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.DevicesCurrencyManager.Position > -1)
            {
                byte nodeId = ((IDeviceInfo)ControllerManager.DocumentModel.DevicesCurrencyManager.Current).Id;
                IDeviceInfo replacedNode = (IDeviceInfo)ControllerManager.DoAction(
                   new FunctionCaller<IDeviceInfo, IController, byte>(NodesHAL.ReplaceFailedNodeAction),
                   new object[] { ControllerManager.DocumentModel.Controller, nodeId },
                   Resources.MsgReplaceFailedNode, true, (byte)CommandTypes.None);
                if (replacedNode != null)
                {
                    ControllerManager.NodeForm.nodeGridViewControl.UnMarkFailedNode(replacedNode.Id);
                    AddControllerNode(replacedNode);
                }
            }
        }

        public void OnIsFailedNodeClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.DevicesCurrencyManager.Position > -1)
            {
                byte nodeId = ((IDeviceInfo)ControllerManager.DocumentModel.DevicesCurrencyManager.Current).Id;
                bool isFailed = (bool)ControllerManager.DoAction(
                new FunctionCaller<bool, IController, byte>(NodesHAL.IsFailedNodeAction),
                new object[] { ControllerManager.DocumentModel.Controller, nodeId },
                Resources.MsgIsFailedNode, true, (byte)CommandTypes.None);
                if (isFailed)
                {
                    ControllerManager.NodeForm.nodeGridViewControl.MarkFailedNode(nodeId);
                }
            }
        }

        public void OnAbortAction(byte commandType, IController device)
        {
            FunctionCaller<bool, IController, byte> caller = new FunctionCaller<bool, IController, byte>(NodesHAL.AbortAction);
            caller.DynamicInvoke(new object[] { device, commandType });
        }

        public void OnRemoveFailedNodeClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.DevicesCurrencyManager.Position > -1)
            {
                byte nodeId = ((IDeviceInfo)ControllerManager.DocumentModel.DevicesCurrencyManager.Current).Id;
                bool removedNode = (bool)ControllerManager.DoAction(
                   new FunctionCaller<bool, IController, byte>(NodesHAL.RemoveFailedNodeAction),
                   new object[] { ControllerManager.DocumentModel.Controller, nodeId },
                   Resources.MsgRemoveFailedNode, true, (byte)CommandTypes.CmdZWaveRemoveNodeFromNetwork);
                if (removedNode)
                {
                    RemoveControllerNodeInternal(nodeId);
                }
            }
        }

        public void OnSetWakeUpIntervalClick(object sender, EventArgs e)
        {
            int wakeUpInterval = int.Parse(ControllerManager.NodeForm.vcWakeUpIntervalToolStripTextBox.Text);
            if (wakeUpInterval > 0 && wakeUpInterval < 279620)
            {
                QueueAction action = (QueueAction)ControllerManager.DoAction(
                    new FunctionCaller<QueueAction, IController, IDeviceInfo, string, int>(NodesHAL.SetWakeUpIntervalAction),
                    new object[] { 
                        ControllerManager.DocumentModel.Controller,
                        ControllerManager.DocumentModel.CurrentDevice, 
                        Resources.MsgSetWakeUpInterval,
                        wakeUpInterval},
                   Resources.MsgSetWakeUpInterval, true, (byte)CommandTypes.CmdZWaveSendData);
                if (action != null)
                {
                    ControllerManager.AddQueueCommand(ControllerManager.DocumentModel.CurrentDevice.Id, action);
                    ControllerManager.LogManager.LogMessage(string.Format("{0} added to command queue for Node {1}.", "Set Wake Up Interval", ControllerManager.DocumentModel.CurrentDevice.Id));
                }
            }
        }

        public void OnBasicToggleToolStripButtonClick(object sender, EventArgs e)
        {
            List<IDeviceInfo> nodes = ControllerManager.Actions.CommonActions.GetSelectedDevices(true);
            NodesHAL.BasicGetAction(ControllerManager.DocumentModel.Controller,
                 nodes,
                  ControllerManager.NodeForm.vcBasicToggleToolStripButton.Checked);

            //ControllerManager.DoAction(
            //  new ProcedureCaller<IController, IList<IDeviceInfo>, bool>(NodesHAL.BasicGetAction),
            //  new object[]{
            //      ControllerManager.DocumentModel.Controller,
            //      nodes,
            //      ControllerManager.NodeForm.vcBasicToggleToolStripButton.Checked},
            //  Resources.MsgSetOn, true, (byte)CommandTypes.CmdZWaveSendData);
        }

        public void OnSetSucToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.DoAction(
              new ProcedureCaller<IController, IDeviceInfo, byte>(NodesHAL.SetSucNodeId),
              new object[]{
                  ControllerManager.DocumentModel.Controller,
                  ControllerManager.DocumentModel.CurrentDevice,
                  (byte)0x00},
              Resources.MsgEnableSucSis, true, (byte)CommandTypes.CmdZWaveSetSucNodeId);
            ControllerManager.Actions.CommonActions.SetUpController();
        }

        public void OnSetSisToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.DoAction(
              new ProcedureCaller<IController, IDeviceInfo, byte>(NodesHAL.SetSucNodeId),
              new object[]{
                  ControllerManager.DocumentModel.Controller,
                  ControllerManager.DocumentModel.CurrentDevice,
                  (byte)0x01},
              Resources.MsgEnableSucSis, true, (byte)CommandTypes.CmdZWaveSetSucNodeId);
            ControllerManager.Actions.CommonActions.SetUpController();
        }

        public void OnNopToolStripButtonClick(object sender, EventArgs e)
        {
            IDeviceInfo device = null;
            byte nodeId = 0;
            if (!byte.TryParse(ControllerManager.NodeForm.vcNopIDToolStripNumericTextBox.Text, out nodeId) || nodeId == 0)
            {
                if (ControllerManager.DocumentModel.DevicesCurrencyManager.Position > -1)
                {
                    nodeId = ControllerManager.DocumentModel.CurrentDevice.Id;
                    device = ControllerManager.DocumentModel.CurrentDevice;
                }
            }
            else
            {
                foreach (IDeviceInfo d in ControllerManager.DocumentModel.Devices)
                {
                    if (d.Id == nodeId)
                    {
                        device = d;
                        break;
                    }
                }
            }
            if (nodeId > 0)
            {
                QueueAction action = new QueueAction(
                    new ProcedureCaller<IController, byte, string>(NodesHAL.NopAction),
                    new object[] { ControllerManager.DocumentModel.Controller, nodeId, Resources.ErrNopSendFail },
                    Resources.MsgSendNop, true, (byte)CommandTypes.CmdZWaveSendData, "NOP", 0);

                if (device == null || device.IsListening || device.IsCommandQueueOverrided || (device.Security & 0x20) != 0 || (device.Security & 0x40) != 0)
                {
                    IDeviceInfo infoNode = (IDeviceInfo)ControllerManager.DoAction(action);
                }
                else
                {
                    ControllerManager.AddQueueCommand(device.Id, action);
                    ControllerManager.LogManager.LogMessage(string.Format("{0} added to command queue for Node {1}.", "NOP", device.Id));
                }
            }
        }

        public void RemoveControllerNode(byte removedNodeId)
        {
            RemoveControllerNodeInternal(removedNodeId);
        }

        private void RemoveControllerNodeInternal(byte removedNodeId)
        {
            if (removedNodeId > 0)
            {
                ControllerManager.DocumentModel.RemoveDevice(removedNodeId);
                if (ControllerManager.DocumentModel.CurrentAssociativeDevice != null && ControllerManager.DocumentModel.CurrentAssociativeDevice.Id == removedNodeId)
                {
                    ControllerManager.DocumentModel.AssociativeGroups.Clear();
                }
                ControllerManager.DocumentModel.RemoveAssociativeDevice(removedNodeId);
                ControllerManager.DocumentModel.RoutesCollection.RemoveSourceRoute(removedNodeId);
                ControllerManager.DocumentModel.RoutesCollection.RemoveDestinationRoute(removedNodeId);
            }
        }
        List<byte> nodesAddedFailedList = new List<byte>();
        public void AddControllerNode(IDeviceInfo addedNode)
        {
            if (addedNode != null)
            {
                ControllerManager.DocumentModel.RemoveDevice(addedNode.Id);
                if (!nodesAddedFailedList.Contains(addedNode.Id))
                {
                    GenericDevice gd = ControllerManager.XmlDataManager.FindGenericDevice(addedNode.Generic);
                    if (gd != null)
                    {
                        addedNode.Type = (String.IsNullOrEmpty(gd.Text)) ? gd.Name : gd.Text;
                    }
                    else
                    {
                        addedNode.Type = "Generic Device:" + Tools.ToHexString(addedNode.Generic);
                    }
                    ControllerManager.DocumentModel.Devices.Add(addedNode);
                    if (addedNode.SupportedCommandClasses != null &&
                        ControllerManager.DocumentModel.Controller.CommandClassesStore.IsSupported(addedNode.Id, 
                        ControllerManager.XmlDataManager.GetCommandClassKey("COMMAND_CLASS_ASSOCIATION")))
                    {
                        ControllerManager.DocumentModel.AssociativeDevices.Add(addedNode);
                    }
                    if (addedNode.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_SENSOR_BINARY") ||
                        addedNode.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_SENSOR_MULTILEVEL"))
                    {
                        ControllerManager.DocumentModel.RoutesCollection.AddSourceRoute(addedNode);
                    }
                    if (addedNode.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_SWITCH_BINARY") ||
                        addedNode.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_SWITCH_MULTILEVEL") ||
                        addedNode.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_STATIC_CONTROLLER") ||
                        addedNode.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_GENERIC_CONTROLLER") ||
                        addedNode.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_REPEATER_SLAVE") ||
                        addedNode.Generic == ControllerManager.XmlDataManager.GetGenericDeviceKey("GENERIC_TYPE_THERMOSTAT"))
                    {
                        ControllerManager.DocumentModel.RoutesCollection.AddDestinationRoute(addedNode);
                    }
                }
            }
            if (addedNode != null && !nodesAddedFailedList.Contains(addedNode.Id))
            {
                ControllerManager.LogManager.LogMessage("Node added to the network...");
            }
            else
            {
                ControllerManager.LogManager.LogMessage("Add Node to the network failed...");
            }
        }

        private void SendSwitchOnOff(bool isOn)
        {
            ControllerManager.DoAction(
              new ProcedureCaller<IController, IList<IDeviceInfo>, bool>(NodesHAL.SendSwitchOnOffAction),
              new object[]{
                  ControllerManager.DocumentModel.Controller,
                  ControllerManager.DocumentModel.Devices,
                  isOn},
              Resources.MsgSetOn, true, (byte)CommandTypes.CmdZWaveSendData);
        }

        private void SendBasicSetOnOff(bool isOn, List<IDeviceInfo> selectedNodes)
        {
            ControllerManager.DoAction(
             new ProcedureCaller<IController, IList<IDeviceInfo>, bool>(NodesHAL.SendBasicSetOnOffAction),
             new object[]{
                  ControllerManager.DocumentModel.Controller,
                  selectedNodes,
                  isOn},
             Resources.MsgSetOn, true, (byte)CommandTypes.CmdZWaveSendData);
        }

        #endregion
        public void OnControllerNodeStatusChanged(Zensys.ZWave.Events.NodeStatusChangedEventArgs args)
        {
            ControllerManager.MainForm.LabelStatusText = args.Status.ToString();
            switch (args.Status)
            {
                case NodeStatuses.AddingRemovingController:
                    break;
                case NodeStatuses.AddingRemovingSlave:
                    break;
                case NodeStatuses.Done:
                    break;
                case NodeStatuses.Failed:
                    ControllerManager.LogManager.LogOperationFailure(string.Format("NodesStatuses.Failed received Node:{0}...", args.SourceNodeId));
                    nodesAddedFailedList.Add(args.SourceNodeId);
                    break;
                case NodeStatuses.LearnReady:
                    break;
                case NodeStatuses.NodeFound:
                    break;
                case NodeStatuses.ProtocolDone:
                    break;
                case NodeStatuses.Unknown:
                    break;
                default:
                    break;
            }
        }
    }
}