using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Collections;
using Zensys.ZWave.Logging;
using Zensys.ZWave.InstallerTool.Models;
using Zensys.ZWave.Devices;
using Zensys.ZWave;
using Zensys.Framework;

namespace Zensys.ZWave.InstallerTool.Controls
{
    public class TreeNodeInfoDataView : TreeView
    {
        private DevicesCollection dataSource;
        public TreeNodeInfoDataView()
            : base()
        {
            
        }
        private void BuildTree()
        {
            lock (this)
            {
                BeginUpdate();
                SuspendLayout();
                this.Nodes.Clear();
                if (dataSource.SelectedDevice != null)
                {
                    CommandClasses.CommandClass cmdClass = CommandClasses.FindByName(CommandClasses.Types.ZWAVE_CMD_CLASS);
                    CommandClasses.Command cmd = cmdClass.FindCommandByName(CommandClasses.CommandTypes.NODE_INFO);
                    foreach (CommandClasses.Command.ParameterBase prm in cmd.Parameters)
                    {
                        TreeNode _node = Nodes.Add(prm.Name + ": ");
                        if (prm.Name == "Capability")
                        {
                            ProcessParameterValue(_node, prm, dataSource.SelectedDevice.Device.Capability);
                        }
                        if (prm.Name == "Security")
                        {
                            ProcessParameterValue(_node, prm, dataSource.SelectedDevice.Device.Security);
                        }
                        if (prm.Name == "Reserved")
                        {
                            ProcessParameterValue(_node, prm, dataSource.SelectedDevice.Device.Reserved);
                        }
                        if (prm.Name == "Basic Device Class")
                        {
                            ProcessParameterValue(_node, prm, dataSource.SelectedDevice.Device.Basic);
                        }
                        if (prm.Name == "Generic Device Class")
                        {
                            GenericDevices.GenericDevice genericDevice = GenericDevices.FindByKey(dataSource.SelectedDevice.Device.Generic);
                            if (genericDevice != null)
                            {
                                _node.Text += genericDevice.Name;
                            }
                        }
                        if (prm.Name == "Specific Device Class")
                        {
                            GenericDevices.GenericDevice genericDevice = GenericDevices.FindByKey(dataSource.SelectedDevice.Device.Generic);
                            if (genericDevice != null)
                            {
                                GenericDevices.SpecificDevice specificDevice = genericDevice.FindSpecificByKey(dataSource.SelectedDevice.Device.Specific);
                                if (specificDevice != null)
                                {
                                    _node.Text += specificDevice.Name;
                                }
                            }
                        }
                        if (prm.Name == "Command Classes")
                        {
                            if (dataSource.SelectedDevice.Device.SupportedCommandClasses != null)
                            {
                                foreach (byte cmdKey in dataSource.SelectedDevice.Device.SupportedCommandClasses)
                                {
                                    _node.Text += Tools.ToHexString(cmdKey) + " ";
                                    _node.Nodes.Add(((CommandClasses.Types)cmdKey).ToString());
                                }
                            }
                        }
                    }
                }
                ResumeLayout();
                EndUpdate();
            }
        }

        [Category("Data")]
        public DevicesCollection DataSource
        {
            get
            {
                return dataSource;
            }
            set
            {

                if (value != null)
                {
                    dataSource = value;
                    BuildTree();
                    dataSource.SelectedDeviceChanged += new EventHandler<EventArgs>(dataSource_SelectedDeviceChanged);
                }
            }
        }

        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
            if (dataSource != null)
            {
                dataSource.SelectedDeviceChanged -= new EventHandler<EventArgs>(dataSource_SelectedDeviceChanged);
            }
        }

        void dataSource_SelectedDeviceChanged(object sender, EventArgs e)
        {
            ReBuildTree();
        }

        private delegate void ReBuildTreeDelegate();
        private void ReBuildTree()
        {
            if (InvokeRequired)
            {
                Invoke(new ReBuildTreeDelegate(ReBuildTree));
            }
            else
            {
                BuildTree();
            }
        }

        private void ProcessParameterValue(TreeNode rootNode, CommandClasses.Command.ParameterBase prm, byte value)
        {

            switch (prm.Type)
            {
                case ("BYTE"):
                    {
                        rootNode.Text += Tools.ToHexString(value);
                    } break;
                case ("WORD"):
                    {
                    } break;
                case ("DWORD"):
                    {
                    } break;
                case ("BIT_24"):
                    {
                    } break;
                case ("ARRAY"):
                    {
                    } break;
                case ("BITMASK"):
                    {
                    } break;
                case ("STRUCT_BYTE"):
                    {
                        rootNode.Text += Tools.ToHexString(value);
                        CommandClasses.Command.ParameterStructByte parameter = (CommandClasses.Command.ParameterStructByte)prm;
                        if (parameter.BitFlags != null)
                        {
                            foreach (CommandClasses.Command.BitFlag bitFlag in parameter.BitFlags)
                            {
                                if ((value & bitFlag.FlagMask) != 0)
                                {
                                    rootNode.Nodes.Add(String.Format("{0}: {1}", bitFlag.FlagName, "True"));
                                }
                                else
                                {
                                    rootNode.Nodes.Add(string.Format("{0}: {1}", bitFlag.FlagName, "False"));
                                }
                            }
                        }
                        if (parameter.BitFields != null)
                        {
                            foreach (CommandClasses.Command.BitField bitField in parameter.BitFields)
                            {
                                //TODO
                            }
                        }
                        if (parameter.FieldEnums != null)
                        {
                            foreach (CommandClasses.Command.FieldEnum fieldEnum in parameter.FieldEnums)
                            {
                                byte index = (byte)((value & fieldEnum.FieldMask) >> fieldEnum.Shifter);
                                if (fieldEnum.Values != null && index < fieldEnum.Values.Count)
                                {
                                    rootNode.Nodes.Add(string.Format("{0}: {1}", fieldEnum.FieldName, fieldEnum.Values[index]));
                                }
                            }
                        }

                    } break;
                case ("ENUM"):
                case ("ENUM_ARRAY"):
                    {
                        CommandClasses.Command.ParameterEnum parameter = (CommandClasses.Command.ParameterEnum)prm;
                        if (parameter.EnumValues != null)
                        {
                            foreach (CommandClasses.Command.EnumValue enumValue in parameter.EnumValues)
                            {
                                if (enumValue.Key == value)
                                {
                                    rootNode.Text += enumValue.Name;
                                }
                            }
                        }
                    } break;
                case ("MULTI_ARRAY"):
                    {
                        CommandClasses.Command.ParameterMultiArray parameter = (CommandClasses.Command.ParameterMultiArray)prm;
                        //TODO
                    } break;
                case ("CONST"):
                    {
                    } break;
                case ("VARIANT"):
                    {
                    } break;
                case ("MARKER"):
                    {
                    } break;
            }

        }

    }
}
