using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.InstallerTool.Classes;
using Zensys.ZWave.InstallerTool.Controllers;
using WeifenLuo.WinFormsUI.Docking;
using Zensys.ZWave.InstallerTool.Layout;
using Zensys.ZWave.InstallerTool.UI;
using Zensys.ZWave.Devices;
using System.Windows.Forms;
using Zensys.ZWave.InstallerTool.Properties;
using Zensys.ZWave.Enums;
using Zensys.Framework;

namespace Zensys.ZWave.InstallerTool.Actions
{
    /// <summary>
    /// MainFormActions class. Contains Main Form functions.
    /// </summary>
    public class MainFormActions : BaseAction
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="MainFormActions"/> class.
        /// </summary>
        /// <param name="controller">The controller.</param>
        public MainFormActions(ControllerManager controller)
            : base(controller)
        {

        }

        public void OnFormLoad(object sender, EventArgs e)
        {
            LayoutManager.OnLoad(ControllerManager, DeserializeDockContent);
            if (Settings.Default.CommandClassesStore != null)
            {
                Settings.Default.CommandClassesStore.CopyTo(ControllerManager.DocumentModel.ExternalCommandClassesStorage, 0);
            }
            if (!string.IsNullOrEmpty(Settings.Default.LastUsedDevice))
            {
                IInstallerController device = null;
                ControllerManager.DoAction(new EventHandler(delegate
                  {
                      device = ControllerManager.Actions.CommonActions.DetectDevice(Settings.Default.LastUsedDevice);
                  }), null, Resources.MsgSettingsDetectInterface, false, (byte)CommandTypes.None);
                if (device != null && IsControllerLibrary(device.Version.Library))
                {
                    ControllerManager.DocumentModel.SetUpController(device);
                    ControllerManager.Actions.CommonActions.SetUpController();
                }
            }
        }
        public void OnFormClosing(object sender, EventArgs e)
        {
            ControllerManager.ApplicationSettings.Synchronize();
            LayoutManager.OnClose(ControllerManager);
            if (ControllerManager.DocumentModel.Controller != null)
            {
                if (ControllerManager.DocumentModel.Controller.ConnectionStatus == Zensys.ZWave.Enums.ConnectionStatuses.Closed)
                {
                    ControllerManager.DocumentModel.Controller.Close();
                }
            }
            if (ControllerManager.DocumentModel.ExternalCommandClassesStorage != null)
            {
                Settings.Default.CommandClassesStore = ControllerManager.DocumentModel.ExternalCommandClassesStorage;
            }
            ControllerManager.Actions.CommonActions.FinalizeEntryPointHAL();
            
            Settings.Default.Save();
        }

        private IDockContent DeserializeDockContent(string stringId)
        {
            IDockContent ret = null;
            if (stringId == typeof(NodeForm).FullName)
            {
                ControllerManager.NodeForm = new NodeForm();
                ret = ControllerManager.NodeForm;
                if (ret.DockHandler != null)
                {
                    ControllerManager.MainForm.NodeToolStripMenuItem.Checked = !ret.DockHandler.IsHidden;
                }
            }
            else if (stringId == typeof(ControllerForm).FullName)
            {
                ControllerManager.ControllerForm = new ControllerForm();
                ret = ControllerManager.ControllerForm;
                if (ret.DockHandler != null)
                {
                    ControllerManager.MainForm.ControllerToolStripMenuItem.Checked = !ret.DockHandler.IsHidden;
                }
            }
            else if (stringId == typeof(LogForm).FullName)
            {
                ControllerManager.LogForm = new LogForm();
                ret = ControllerManager.LogForm;
                if (ret.DockHandler != null)
                {
                    ControllerManager.MainForm.LogToolStripMenuItem.Checked = !ret.DockHandler.IsHidden;
                }
            }
            else if (stringId == typeof(RoutingTableForm).FullName)
            {
                ControllerManager.RoutingTableForm = new RoutingTableForm();
                ret = ControllerManager.RoutingTableForm;
                if (ret.DockHandler != null)
                {
                    ControllerManager.MainForm.TopologyMapToolStripMenuItem.Checked = !ret.DockHandler.IsHidden;
                }
            }
            else
            {
                ret = new DockContent();
            }
            return ret;
        }

        /// <summary>
        /// Called when Active Content changed.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnActiveContentChanged(object sender, EventArgs e)
        {

        }

        public void OnControllerToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(ControllerForm), ControllerManager.MainForm.ControllerToolStripMenuItem.Checked);
        }

        public void OnNodeToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(NodeForm), ControllerManager.MainForm.NodeToolStripMenuItem.Checked);
        }

        public void OnLogToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(LogForm), ControllerManager.MainForm.LogToolStripMenuItem.Checked);
        }
        public void OnTopologyMapToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.ShowView(typeof(RoutingTableForm), ControllerManager.MainForm.TopologyMapToolStripMenuItem.Checked);
        }

        public void OnExitToolStripMenuItemClick(object sender, EventArgs e)
        {
            ControllerManager.MainForm.Close();
            System.Windows.Forms.Application.Exit();
        }

        public void OnAboutToolStripMenuItemClick(object sender, EventArgs e)
        {
            AboutBox about = new AboutBox();
            about.ShowDialog();
        }
        public void OnSettingsToolStripMenuItemClick(object sender, EventArgs e)
        {
            SettingsForm settingsForm = new SettingsForm();
            settingsForm.OnLoadInterfaces += new EventHandler(SettingsOnLoadInterfaces);
            settingsForm.OnDetectInterface += new EventHandler(SettingsOnDetectInterface);
            if (settingsForm.ShowDialog(ControllerManager.MainForm) == DialogResult.OK)
            {
                if (settingsForm.SelectedDevice != null && CommonActions.IsControllerLibrary(settingsForm.SelectedDevice.Version.Library))
                {
                    ControllerManager.DocumentModel.SetUpController(settingsForm.SelectedDevice);
                    SaveSettings(ControllerManager.DocumentModel.Controller.SerialPort);
                }
                else
                {
                    IInstallerController device = null;
                    ControllerManager.DoAction(new EventHandler(delegate
                    {
                        try
                        {
                            if (ControllerManager.DocumentModel.Controller != null)
                            {
                                ControllerManager.DocumentModel.Controller.Close();
                            }
                            device = ControllerManager.Actions.CommonActions.DetectDevice(settingsForm.SelectedDeviceId);
                        }
                        catch (Exception ex)
                        {
                            throw ex;
                            //System.Diagnostics.Debug.WriteLine(ex.Message);
                        }
                    }), null, Resources.MsgSettingsDetectInterface, false, (byte)CommandTypes.None);


                    if (device != null && CommonActions.IsControllerLibrary(device.Version.Library))
                    {
                        ControllerManager.DocumentModel.SetUpController(device);
                        ControllerManager.Actions.CommonActions.SetUpController();
                        SaveSettings(device.SerialPort);
                    }
                    else if (device != null)
                    {
                        SaveSettings("");
                        MessageBox.Show(Resources.MsgNoControllerLibrary, Resources.CaptionZWaveInstallerTool);
                    }
                    else
                    {
                        SaveSettings("");
                        MessageBox.Show(Resources.MsgNoControllerDevice, Resources.CaptionZWaveInstallerTool);
                    }
                }
            }
        }

        private void SaveSettings(string serialPortName)
        {
            Settings.Default.LastUsedDevice = serialPortName;
            Settings.Default.Save();
        }

        private void SettingsOnDetectInterface(object sender, EventArgs e)
        {
            if (sender is SettingsForm)
            {
                SettingsForm form = sender as SettingsForm;
                IInstallerController device = null;
                SortedList<string, IInstallerController> detectedControllers = new SortedList<string, IInstallerController>();
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    try
                    {
                        if (ControllerManager.DocumentModel.Controller != null)
                        {
                            ControllerManager.DocumentModel.Controller.Close();
                        }
                        List<Win32PnPEntityClass> interfaces = ComputerSystemHardwareHelper.GetWin32PnPEntityClassSerialPortDevices();
                        foreach (Win32PnPEntityClass serialPortInfo in interfaces)
                        {

                            device = ControllerManager.Actions.CommonActions.DetectDevice(serialPortInfo.DeviceID);
                            if (device != null)
                            {
                                detectedControllers.Add(serialPortInfo.DeviceID, device);
                            }

                        }
                    }
                    catch (Exception ex)
                    {
                        throw ex;
                        //System.Diagnostics.Debug.WriteLine(ex.Message);
                    }

                }), null, Resources.MsgSettingsDetectInterface, false, (byte)CommandTypes.None);
                LoadInterfaces(form, detectedControllers);
            }
        }
        private void SettingsOnLoadInterfaces(object sender, EventArgs e)
        {
            if (sender is SettingsForm)
            {
                SettingsForm form = sender as SettingsForm;
                LoadInterfaces(form, new SortedList<string, IInstallerController>());
            }
        }
        private void LoadInterfaces(SettingsForm form, SortedList<string, IInstallerController> detectedControllers)
        {
            //List<Win32PnPEntityClass> interfaces = ComputerSystemHardwareHelper.GetWin32PnPEntityClassSerialPortDevices();
            //form.InterfacesListBox.Items.Clear();
            //foreach (Win32PnPEntityClass serialPortInfo in interfaces)
            //{
            //    IInstallerController device = null;
            //    if (detectedControllers.ContainsKey(serialPortInfo.DeviceID))
            //    {
            //        device = detectedControllers[serialPortInfo.DeviceID];
            //    }
            //    form.InterfacesListBox.Items.Add(new DeviceWrapper(serialPortInfo, device));
            //}
            List<Win32PnPEntityClass> interfaces = ComputerSystemHardwareHelper.GetWin32PnPEntityClassSerialPortDevices();
            form.InterfacesListBox.Items.Clear();
            form.InterfacesListBox.Enabled = true;
            form.NoCommInterfacesLabel.Text = "";
            if (interfaces != null && interfaces.Count > 0)
            {
                foreach (Win32PnPEntityClass serialPortInfo in interfaces)
                {
                    IInstallerController device = null;
                    if (detectedControllers.ContainsKey(serialPortInfo.DeviceID))
                    {
                        device = detectedControllers[serialPortInfo.DeviceID];
                    }
                    int index = form.InterfacesListBox.Items.Add(new DeviceWrapper(serialPortInfo, device));
                    if (ControllerManager.DocumentModel.Controller != null &&
                        ControllerManager.DocumentModel.Controller.SerialPort == serialPortInfo.DeviceID)
                    {
                        form.InterfacesListBox.SelectedIndex = index;
                    }
                }
            }
            else
            {
                form.InterfacesListBox.Enabled = false;
                form.NoCommInterfacesLabel.Text = Resources.MsgNoCommInterfaces;
            }
        }


    }
}
