using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace Zensys.Framework.UI.Controls.BitBox
{
    /// <summary>
    /// BitBox control
    /// </summary>
    public partial class BitBox : UserControl
    {
        /// <summary>
        /// Occurs when range changed.
        /// </summary>
        public event EventHandler RangeChanged;
        private bool[] innerbits = new bool[8];
        private bool[] lockedbits = new bool[8];
        /// <summary>
        /// Initializes a new instance of the <see cref="BitBox"/> class.
        /// </summary>
        public BitBox()
        {
            InitializeComponent();
            DoubleBuffered = true;
        }

        #region Properties
        /// <summary>
        /// Gets the bit range start.
        /// </summary>
        /// <value>The bit range start.</value>
        public int BitRangeStart
        {
            get
            {
                for (int i = 0; i < 8; i++)
                {
                    if (innerbits[i])
                        return i;
                }
                return -1;
            }
        }

        /// <summary>
        /// Gets the bit range end.
        /// </summary>
        /// <value>The bit range end.</value>
        public int BitRangeEnd
        {
            get
            {
                for (int i = 7; i > -1; i--)
                {
                    if (innerbits[i])
                        return i;
                }
                return -1;
            }
        }

        private Color mFilledBitColor = SystemColors.ControlDark;
        /// <summary>
        /// Gets or sets the color of the filled bit.
        /// </summary>
        /// <value>The color of the filled bit.</value>
        public Color FilledBitColor
        {
            get { return mFilledBitColor; }
            set { mFilledBitColor = value; }
        }

        private Color mEmptyBitColor = SystemColors.Window;
        /// <summary>
        /// Gets or sets the empty color of the empty bit.
        /// </summary>
        /// <value>The empty color of the empty bit.</value>
        public Color EmptyBitColor
        {
            get { return mEmptyBitColor; }
            set { mEmptyBitColor = value; }
        }

        private Color mLockedBitColor = SystemColors.ControlDarkDark;
        /// <summary>
        /// Gets or sets the locked color of the locked bit.
        /// </summary>
        /// <value>The locked color of the locked bit.</value>
        public Color LockedBitColor
        {
            get { return mLockedBitColor; }
            set { mLockedBitColor = value; }
        }

        private Color mSeparatorColor = SystemColors.ControlDarkDark;

        /// <summary>
        /// Gets or sets the color of the separator.
        /// </summary>
        /// <value>The color of the separator.</value>
        public Color SeparatorColor
        {
            get { return mSeparatorColor; }
            set { mSeparatorColor = value; }
        }

        private SeparatorWidth mSeparatorWidth;

        /// <summary>
        /// Gets or sets the width of the separator.
        /// </summary>
        /// <value>The width of the separator.</value>
        public SeparatorWidth SeparatorWidth
        {
            get { return mSeparatorWidth; }
            set { mSeparatorWidth = value; }
        }

        private bool mInversed;
        /// <summary>
        /// Gets or sets a value indicating whether this <see cref="BitBox"/> is inversed.
        /// </summary>
        /// <value><c>true</c> if inversed; otherwise, <c>false</c>.</value>
        public bool Inversed
        {
            get { return mInversed; }
            set { mInversed = value; }
        }

        private bool mShowBitCaptions;
        /// <summary>
        /// Gets or sets a value indicating whether show bit captions.
        /// </summary>
        /// <value><c>true</c> if show bit captions; otherwise, <c>false</c>.</value>
        public bool ShowBitCaptions
        {
            get { return mShowBitCaptions; }
            set { mShowBitCaptions = value; }
        }


        private SolidBrush mFilledBitBrush;
        private SolidBrush FilledBitBrush
        {
            get
            {
                if (mFilledBitBrush == null)
                {
                    mFilledBitBrush = new SolidBrush(mFilledBitColor);
                }
                else
                {
                    if (mFilledBitBrush.Color != mFilledBitColor)
                    {
                        mFilledBitBrush.Dispose();
                        mFilledBitBrush = null;
                        mFilledBitBrush = FilledBitBrush;
                    }
                }
                return mFilledBitBrush;
            }
        }

        private SolidBrush mEmptyBitBrush;
        private SolidBrush EmptyBitBrush
        {
            get
            {
                if (mEmptyBitBrush == null)
                {
                    mEmptyBitBrush = new SolidBrush(mEmptyBitColor);
                }
                else
                {
                    if (mEmptyBitBrush.Color != mEmptyBitColor)
                    {
                        mEmptyBitBrush.Dispose();
                        mEmptyBitBrush = null;
                        mEmptyBitBrush = EmptyBitBrush;
                    }
                }
                return mEmptyBitBrush;
            }
        }

        private SolidBrush mLockedBitBrush;
        private SolidBrush LockedBitBrush
        {
            get
            {
                if (mLockedBitBrush == null)
                {
                    mLockedBitBrush = new SolidBrush(mLockedBitColor);
                }
                else
                {
                    if (mLockedBitBrush.Color != mLockedBitColor)
                    {
                        mLockedBitBrush.Dispose();
                        mLockedBitBrush = null;
                        mLockedBitBrush = LockedBitBrush;
                    }
                }
                return mLockedBitBrush;
            }
        }

        private Pen mSeparatorPen;
        private Pen SeparatorPen
        {
            get
            {
                if (mSeparatorPen == null)
                {
                    mSeparatorPen = new Pen(mSeparatorColor);
                    mSeparatorPen.Width = (int)mSeparatorWidth;
                }
                else
                {
                    if (mSeparatorPen.Color != mSeparatorColor || mSeparatorPen.Width != (int)mSeparatorWidth)
                    {
                        mSeparatorPen.Dispose();
                        mSeparatorPen = null;
                        mSeparatorPen = SeparatorPen;
                    }
                }
                return mSeparatorPen;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether this instance has valid range.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance has valid range; otherwise, <c>false</c>.
        /// </value>
        public bool HasValidRange
        {
            get
            {
                return BitRangeStart <= BitRangeEnd && BitRangeStart >= 0 && BitRangeEnd < 8;
            }

        }

        #endregion

        /// <summary>
        /// Sets the bit range.
        /// </summary>
        /// <param name="startRange">The start range.</param>
        /// <param name="endRange">The end range.</param>
        public void SetBitRange(int startRange, int endRange)
        {
            if (startRange <= endRange &&
                startRange >= 0 &&
                startRange < 8 &&
                endRange >= 0 &&
                endRange < 8)
            {
                for (int i = startRange; i <= endRange; i++)
                {
                    innerbits[i] = true;
                }
                Invalidate();
                Refresh();
            }
            else
                throw new ApplicationException("The startRange value should be less than endRange value. Both values should be in range 0..7");

        }

        /// <summary>
        /// Locks the and clear current range.
        /// </summary>
        public void LockAndClearCurrentRange()
        {
            for (int i = 0; i < 8; i++)
            {
                if (innerbits[i])
                    lockedbits[i] = true;
            }
            ClearInnerBits();
            Invalidate();
            Refresh();
        }

        /// <summary>
        /// Unlocks range.
        /// </summary>
        /// <param name="start">The start.</param>
        /// <param name="end">The end.</param>
        public void UnLockRange(int start, int end)
        {
            for (int i = start; i <= end; i++)
            {
                lockedbits[i] = false;
            }
            Invalidate();
            Refresh();
        }
        /// <summary>
        /// Locks the range.
        /// </summary>
        /// <param name="start">The start.</param>
        /// <param name="end">The end.</param>
        public void LockRange(int start, int end)
        {
            for (int i = start; i <= end; i++)
            {
                lockedbits[i] = true;
            }
            Invalidate();
            Refresh();
        }

        /// <summary>
        /// Clears the locked bits.
        /// </summary>
        public void ClearLockedBits()
        {
            for (int i = 0; i < 8; i++)
            {
                lockedbits[i] = false;
            }
        }

        private void ClearInnerBits()
        {
            for (int i = 0; i < 8; i++)
            {
                innerbits[i] = false;
            }
        }

        /// <summary>
        /// Clears the bits.
        /// </summary>
        public void ClearBits()
        {
            ClearInnerBits();
            ClearLockedBits();
        }


        /// <summary>
        /// Raises the <see cref="E:System.Windows.Forms.Control.Paint"></see> event.
        /// </summary>
        /// <param name="e">A <see cref="T:System.Windows.Forms.PaintEventArgs"></see> that contains the event data.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
            for (int i = 0; i < 8; i++)
            {
                DrawBit(i, e);
            }
        }

        private void DrawBit(int bitNo, PaintEventArgs e)
        {
            if (bitNo < 0 || bitNo > 7)
                throw new ApplicationException("The bitNo value should be in range 0..7");
            Rectangle bitRect = Inversed ? GetBitRectangle(7 - bitNo, ClientRectangle) : GetBitRectangle(bitNo, ClientRectangle);
            if (innerbits[bitNo])
                e.Graphics.FillRectangle(FilledBitBrush, bitRect);
            else
                e.Graphics.FillRectangle(EmptyBitBrush, bitRect);
            if (lockedbits[bitNo])
                e.Graphics.FillRectangle(LockedBitBrush, bitRect);
            if (SeparatorPen.Width > 0)
                e.Graphics.DrawRectangle(SeparatorPen, bitRect);
            if (ShowBitCaptions)
            {
                if (lockedbits[bitNo])
                    e.Graphics.DrawString(bitNo.ToString(), this.Font, Brushes.Gray, bitRect);
                else
                    e.Graphics.DrawString(bitNo.ToString(), this.Font, Brushes.Black, bitRect);
            }
        }

        private Rectangle GetBitRectangle(int bitNo, Rectangle fromRectangle)
        {
            int bitWindth = (int)(fromRectangle.Width - 2 * (int)SeparatorWidth) / 8;
            Rectangle ret = new Rectangle(new Point(bitWindth * bitNo + (int)SeparatorWidth, ClientRectangle.Top + (int)SeparatorWidth), new Size(bitWindth, fromRectangle.Height - 2 * (int)SeparatorWidth));
            return ret;
        }

        private void BitBox_MouseMove(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                int temp = GetBitNo(e);
                if (startbit != -1 && temp != -1 && endbit != temp)
                {
                    endbit = temp;
                    ChangeRange();
                }
                this.Invalidate();
            }
        }


        private int GetBitNo(MouseEventArgs e)
        {
            int j;
            for (int i = 0; i < 8; i++)
            {
                j = Inversed ? 7 - i : i;
                if (!lockedbits[j] && GetBitRectangle(i, ClientRectangle).Contains(e.Location))
                {
                    return j;
                }
            }
            return -1;
        }

        private int endbit;
        private int startbit;
        private void BitBox_MouseDown(object sender, MouseEventArgs e)
        {
            startbit = GetBitNo(e);
            endbit = startbit;
            if (startbit != -1)
            {
                ClearInnerBits();
                ChangeRange();
                this.Invalidate();
            }
        }

        /// <summary>
        /// Changes the range.
        /// </summary>
        private void ChangeRange()
        {
            if (startbit <= endbit)
            {
                for (int i = startbit; i <= endbit; i++)
                {
                    if (lockedbits[i])
                        break;
                    innerbits[i] = true;
                }
            }
            else
            {
                for (int i = startbit; i >= endbit; i--)
                {
                    if (lockedbits[i])
                        break;
                    innerbits[i] = true;
                }

            }
            if (RangeChanged != null)
                RangeChanged(this, EventArgs.Empty);
        }
    }
    /// <summary>
    /// 
    /// </summary>
    public enum SeparatorWidth
    {
        /// <summary>
        /// 
        /// </summary>
        None,
        /// <summary>
        /// 
        /// </summary>
        Weak,
        /// <summary>
        /// 
        /// </summary>
        Normal,
        /// <summary>
        /// 
        /// </summary>
        Strong
    }
}
