using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Enums;

namespace Zensys.ZWave.Devices
{
    /// <summary>
    /// This interface provides methods that allow the client program using Z-Wave Device common Serial API.
    /// </summary>
    public interface IDevice : IDisposable, IComparable<IDevice>, IEquatable<IDevice>
    {
        /// <summary>
        /// Occurs when connection status changed.
        /// </summary>
        event Events.StatusChangedEventHandler ConnectionStatusChanged;
        /// <summary>
        /// Occurs when ApplicationCommandHandler command received.
        /// </summary>
        event Events.DeviceAppCommandHandlerEventHandler ApplicationCommandHandlerEvent;
        /// <summary>
        /// Occurs when ApplicationCommandHandler_Bridge command received
        /// </summary>
        event Events.DeviceAppCommandHandler_BridgeEventHandler ApplicationCommandHandler_BridgeEvent;
        /// <summary>
        /// Gets or sets the Id.
        /// </summary>
        /// <value>The Id.</value>
        byte Id { get; set; }
        /// <summary>
        /// Gets or sets the Home Id.
        /// </summary>
        /// <value>The Home Id.</value>
        byte[] HomeId { get; set; }
        /// <summary>
        /// Gets or sets the type of the chip.
        /// </summary>
        /// <value>The type of the chip.</value>
        byte ChipType { get; set; }
        /// <summary>
        /// Gets or sets the chip revision.
        /// </summary>
        /// <value>The chip revision.</value>
        byte ChipRevision { get; set; }
        /// <summary>
        /// Gets or sets the Serial API version.
        /// </summary>
        /// <value>The serial API version.</value>
        byte SerialApiVersion { get; set; }
        /// <summary>
        /// Gets or sets the Serial Port name.
        /// </summary>
        /// <value>The serial port.</value>
        string SerialPort { get; set; }
        /// <summary>
        /// Gets or sets the <see cref="IDeviceTimer"></see>.
        /// </summary>
        /// <value>The timer.</value>
        IDeviceTimer Timer { get; set; }
        /// <summary>
        /// Gets or sets the <see cref="IDeviceMemory"></see>.
        /// </summary>
        /// <value>The memory.</value>
        IDeviceMemory Memory { get; set; }
        /// <summary>
        /// Gets the <see cref="DeviceLedCollection"></see>.
        /// </summary>
        /// <value>The Device Led collection.</value>
        DeviceLedCollection Leds { get; }
        /// <summary>
        /// Gets or sets the <see cref="IDeviceFlash"/>.
        /// </summary>
        /// <value>The flash.</value>
        IDeviceFlash Flash { get; set; }
        /// <summary>
        /// Gets or sets the connection status.
        /// </summary>
        /// <value>The connection status.</value>
        ConnectionStatuses ConnectionStatus { get; set; }
        /// <summary>
        /// Gets or sets the <see cref="VersionInfo" />.
        /// </summary>
        /// <value>The version.</value>
        VersionInfo Version { get; set; }
        /// <summary>
        /// Gets or sets a value indicating whether this node has Slave API.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this node has Slave API; otherwise, <c>false</c>.
        /// </value>
        bool IsSlaveApi { get; set; }
        /// <summary>
        /// Gets or sets a value indicating whether this node is virtual.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if node is virtual; otherwise, <c>false</c>.
        /// </value>
        bool IsVirtual { get; set; }
        /// <summary>
        /// Gets or sets a value indicating whether this node is failed.
        /// </summary>
        /// <value><c>true</c> if this node is failed; otherwise, <c>false</c>.</value>
        bool IsFailed { get; set; }
        /// <summary>
        /// Gets or sets the capability.
        /// </summary>
        /// <value>The capability.</value>
        byte Capability { get; set; }
        /// <summary>
        /// Gets or sets the security.
        /// </summary>
        /// <value>The security.</value>
        byte Security { get; set; }
        /// <summary>
        /// Gets or sets the reserved.
        /// </summary>
        /// <value>The reserved.</value>
        byte Reserved { get; set; }
        /// <summary>
        /// Gets or sets the basic class key.
        /// </summary>
        /// <value>The basic.</value>
        byte Basic { get; set; }
        /// <summary>
        /// Gets or sets the generic class key.
        /// </summary>
        /// <value>The generic.</value>
        byte Generic { get; set; }
        /// <summary>
        /// Gets or sets the specific class key.
        /// </summary>
        /// <value>The specific.</value>
        byte Specific { get; set; }
        /// <summary>
        /// Gets a value indicating whether this node is in listening mode.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this node is in listening mode; otherwise, <c>false</c>.
        /// </value>
        bool IsListening { get; }
        /// <summary>
        /// Gets the syncronization count.
        /// </summary>
        /// <value>The syncronization count.</value>
        byte SyncCount { get; }
        /// <summary>
        /// Gets or sets the supported command classes array.
        /// </summary>
        /// <value>The supported command classes array.</value>
        byte[] SupportedCommandClasses { get; set; }
        /// <summary>
        /// Gets or sets the capability bitmask.
        /// </summary>
        /// <value>The capability bitmask.</value>
        NodeBitmask CapabilityBitmask { get; set; }
        /// <summary>
        /// Determines whether if the specified command is supported.
        /// </summary>
        /// <param name="commandId">The command id.</param>
        /// <returns>
        /// 	<c>true</c> if the specified command is supported; otherwise, <c>false</c>.
        /// </returns>
        bool IsSupportedSerialApiCommand(byte commandId);
        /// <summary>
        /// Sets the sleep mode.
        /// </summary>
        /// <param name="mode">The mode.</param>
        /// <param name="intEnable">The int enable.</param>
        void SetSleepMode(SleepMode mode, byte intEnable);
        /// <summary>
        /// Sets the timeouts.
        /// </summary>
        /// <param name="acknowledgeTimeout">The acknowledge timeout.</param>
        /// <param name="timeout">The timeout.</param>
        /// <returns></returns>
        byte[] SerialApiSetTimeouts(byte acknowledgeTimeout, byte timeout);
        /// <summary>
        /// Determines whether if the specified command is supported.
        /// </summary>
        /// <param name="commandId">The command id.</param>
        /// <param name="throwException">if set to <c>true</c> throw exception.</param>
        /// <returns>
        /// 	<c>true</c> if the specified command is supported; otherwise, <c>false</c>.
        /// </returns>
        bool IsSupportedSerialApiCommand(byte commandId, bool throwException);
        /// <summary>
        /// Resets this node.
        /// </summary>
        void Reset();
        /// <summary>
        /// Sends the data.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="data">The data.</param>
        /// <param name="txOptions">The Tx options.</param>
        /// <returns></returns>
        TransmitStatuses SendData(byte nodeId, byte[] data, TransmitOptions txOptions);
        /// <summary>
        /// Sends the ZWave test.
        /// </summary>
        /// <param name="testCmd">The test command.</param>
        /// <param name="testDelay">The test delay.</param>
        /// <param name="testPayloadLength">Length of the test payload.</param>
        /// <param name="testCount">The test count.</param>
        /// <param name="txOptions">The Trnsmit options.</param>
        /// <param name="maxLength">Max Length.</param>
        /// <param name="testNodeMask">The test node mask.</param>
        /// <returns></returns>
        TransmitStatuses SendZWaveTest(byte testCmd, ushort testDelay, byte testPayloadLength, int testCount, TransmitOptions txOptions, byte maxLength, byte[] testNodeMask);
        /// <summary>
        /// Sends the meta data.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="data">The data.</param>
        /// <param name="txOptions">The tx options.</param>
        /// <returns></returns>
        TransmitStatuses SendDataMeta(byte nodeId, byte[] data, TransmitOptions txOptions);
        /// <summary>
        /// Sends the data multi.
        /// </summary>
        /// <param name="nodeList">The node list.</param>
        /// <param name="data">The data.</param>
        /// <param name="txOptions">The Tx options.</param>
        /// <returns></returns>
        TransmitStatuses SendDataMulti(List<byte> nodeIdList, byte[] data, TransmitOptions txOptions);
        /// <summary>
        /// Sends the data MR.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="data">The data.</param>
        /// <param name="txOptions">The tx options.</param>
        /// <param name="route">The route.</param>
        /// <param name="timeout">The timeout.</param>
        /// <returns></returns>
        TransmitStatuses SendDataMR(byte nodeId, byte[] data, TransmitOptions txOptions, byte[] route, int timeout);
        /// <summary>
        /// Sets the programming mode.
        /// </summary>
        /// <param name="enable">if set to <c>true</c> [enable].</param>
        /// <returns></returns>
        bool SetProgrammingMode(bool enable);
        /// <summary>
        /// Erases the chip.
        /// </summary>
        /// <param name="isAsic">if set to <c>true</c> [is asic].</param>
        void EraseChip(bool isAsic);
        /// <summary>
        /// Reads the signature bits.
        /// </summary>
        void ReadSignatureBits();
        /// <summary>
        /// Reads the lock bits.
        /// </summary>
        /// <returns>Returns the ZW0x0x Lock Bits.</returns>
        byte ReadLockBits();
        /// <summary>
        /// Writes the lock bits.
        /// </summary>
        /// <returns>Readed ZW0x0x Lock Bits</returns>
        bool WriteLockBits(byte lockBits);
        /// <summary>
        /// Sets the default write cycle time.
        /// </summary>
        /// <returns>Returns <c>true</c>, if write thime was set succesfully, else <c>false</c>.</returns>
        bool SetWriteCycleTime();
        /// <summary>
        /// Sets the write cycle time.
        /// </summary>
        /// <param name="xtalFrequencyMHz">X-Tal Frequency of th Z-Wave mudule in MHz</param>
        /// <returns>Returns <c>true</c>, if write thime was set succesfully, else <c>false</c>.</returns>
        bool SetWriteCycleTime(float xtalFrequencyMHz);
        /// <summary>
        /// Calculate the default write cycle time c value.
        /// </summary>
        /// <returns>Returns c - is the value set by the Set Write Cycle Time instruction.</returns>
        byte CalculateWriteCycle();
        /// <summary>
        /// Calculate the write cycle time c value.
        /// </summary>
        /// <param name="xtalFrequencyMHz">X-Tal Frequency of th Z-Wave mudule in MHz</param>
        /// <returns>Returns c - is the value set by the Set Write Cycle Time instruction.</returns>
        byte CalculateWriteCycle(float xtalFrequencyMHz);
        /// <summary>
        /// Opens connection with the specified port name.
        /// </summary>
        /// <param name="portName">COM port name.</param>
        void Open(string portName);
        /// <summary>
        /// Closes the connection.
        /// </summary>
        void Close();
        /// <summary>
        /// Gets the version.
        /// </summary>
        VersionInfo GetVersion();
        /// <summary>
        /// Gets the capabilities.
        /// </summary>
        void GetCapabilities();
        /// <summary>
        /// Init the Serial API.
        /// </summary>
        /// <returns></returns>
        byte[] SerialApiInitData();
        /// <summary>
        /// Sets the working mode.See <see cref="WorkingModes"/>
        /// </summary>
        /// <param name="mode">The mode.</param>
        /// <returns></returns>
        bool SetMode(WorkingModes mode);
        /// <summary>
        /// Set the bit that identifies the Chip as a Modem device
        /// </summary>
        /// <returns><c>true</c>, if modem bits set ok.</returns>
        bool SetModemBits();
        /// <summary>
        /// Read status from the ZW040x chip.
        /// </summary>
        /// <returns>Readed state byte</returns>
        byte ReadState();
        /// <summary>
        /// Read the write OTP stats from the ZW040x chip.
        /// </summary>
        /// <returns>Number of excessive writes</returns>
        short ReadWriteOtpStats();
        /// <summary>
        /// Sets the boot loader mode.
        /// </summary>
        /// <param name="enable">if set to <c>true</c> enable.</param>
        /// <returns></returns>
        bool SetBootLoaderMode(bool enable);
        /// <summary>
        /// Gets the current firmware version.
        /// </summary>
        /// <param name="version">The version.</param>
        /// <param name="revision">The revision.</param>
        /// <returns></returns>
        bool GetCurrentFirmwareVersion(out int version, out int revision);

        /// <summary>
        /// Gets the random number.
        /// </summary>
        /// <param name="randomBytes">The 8 bytes of random numbers.</param>
        /// <returns></returns>
        byte[] GetRandomNumber(int randomBytes);

        /// <summary>
        /// Sends the node information.
        /// </summary>
        /// <param name="destination">The destination.</param>
        /// <param name="txOptions">The TransmitOptions.</param>
        /// <returns></returns>
        TransmitStatuses SendNodeInformation(byte destination, TransmitOptions txOptions);
        /// <summary>
        /// Sets the RF receive mode.
        /// </summary>
        /// <param name="mode">The mode.</param>
        /// <returns></returns>
        bool SetRFReceiveMode(byte mode);
        /// <summary>
        /// Sets the learn mode.
        /// </summary>
        /// <param name="learnMode">if set to <c>true</c> learn mode.</param>
        /// <returns></returns>
        LearnMode SetLearnMode(bool learnMode);
        /// <summary>
        /// Sets the node information.
        /// </summary>
        /// <param name="isListening">if set to <c>true</c> node is listening.</param>
        /// <param name="generic">The Generic class.</param>
        /// <param name="specific">The Specific class.</param>
        /// <param name="nodeParameter">The node parameters.</param>
        void SetNodeInformation(bool isListening, byte generic, byte specific, byte[] nodeParameter);
        /// <summary>
        /// Resets the Device.
        /// </summary>
        void SetDefault();
        /// <summary>
        /// Sends the test frame.
        /// </summary>
        /// <param name="nodeId">The node id.</param>
        /// <param name="powerLevel">The power level.</param>
        /// <returns></returns>
        TransmitStatuses SendTestFrame(byte nodeId, PowerLevels powerLevel);
        /// <summary>
        /// Requests the network update.
        /// </summary>
        RequestNetworkUpdateStatuses RequestNetworkUpdate();
        /// <summary>
        /// This function can be used to request a controller to update the nodes neighbors.
        /// </summary>
        /// <returns></returns>
        RediscoveryNeededReturnValues RediscoveryNeeded(byte nodeId);
        /// <summary>
        /// Start/stop calibration.
        /// </summary>
        /// <param name="enable">if set to <c>true</c> [enable].</param>
        /// <returns></returns>
        bool Calibration(bool enable);
    }
}
