using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave;
using System.IO.Ports;
using Zensys.ZWave.Exceptions;
using Zensys.ZWave.Events;
using Zensys.Framework;

namespace Zensys.ZWave.SerialPortTransport
{
    class TransportLayer : ITransportLayer
    {
        SerialPort mPort = null;
        SocketAsyncClient mClient = null;
        UsbPort mUsbPort = null;

        #region ITransportLayer Members
        public event DataReceivedEventHandler DataReceived;
        /// <summary>
        /// Close()
        /// </summary>
        public void Close()
        {
            if (mPort != null)
            {
                Tools._writeDebugDiagnosticMessage("", true, true, 1);
                mPort.Close();
                mPort = null;
            }
            if (mClient != null)
            {
                Tools._writeDebugDiagnosticMessage(mClient.ConnectionString, true, true, 1);
                mClient.Close();
                mClient = null;
            }
            if (mUsbPort != null)
            {
                Tools._writeDebugDiagnosticMessage("USB", true, true, 1);
                mUsbPort.Close();
                mUsbPort = null;

            }
        }

        /// <summary>
        /// Gets the device names.
        /// </summary>
        /// <returns></returns>
        public IList<string> GetDeviceNames()
        {
            List<string> ports = new List<string>(SerialPort.GetPortNames());
            ports.AddRange(UsbPort.GetPortNames());
            ports.Sort();
            return ports;
        }

        /// <summary>
        /// Opens the specified device name.
        /// </summary>
        /// <param name="deviceName">Name of the device.</param>
        public void Open(string deviceName)
        {
            if (deviceName.StartsWith("socket:///"))
            {
                if (mClient != null)
                {
                    mClient.Close();
                    mClient = null;

                }
                try
                {
                    Tools._writeDebugDiagnosticMessage("", true, true, 1);
                    mClient = new SocketAsyncClient();
                    mClient.DataReceived += new EventHandler<SocketDataReceivedEventArgs>(mClient_DataReceived);
                    mClient.Connect(deviceName.Replace("socket:///", ""));
                }
                catch (Exception ex)
                {
                    throw ex;
                }
            }
            if (deviceName.StartsWith(@"\\?\usb#vid_"))
            {
                if (mUsbPort != null)
                {
                    //mUsbPort.Close();
                    //mUsbPort = null;
                }
                try
                {
                    Tools._writeDebugDiagnosticMessage("USB", true, true, 1);
                    if (mUsbPort == null)
                    {
                        mUsbPort = new UsbPort(deviceName);
                        mUsbPort.DataReceived += new EventHandler<UsbDataReceivedEventArgs>(mUsbPort_DataReceived);
                    }                    
                    mUsbPort.Open();
                }
                catch (Exception ex)
                {
                    throw ex;
                }
            }
            else
            {
                if (mPort == null)
                {
                    Tools._writeDebugDiagnosticMessage("", true, true, 1);
                    mPort = new SerialPort(deviceName, 115200, Parity.None, 8, StopBits.One);
                    mPort.DataReceived += new SerialDataReceivedEventHandler(mPort_DataReceived);
                    mPort.Open();
                }
                else if (mPort.PortName != deviceName)
                {
                    throw new TransportLayerException("another port is already opened");
                }
                else if (mPort.IsOpen)
                {
                    throw new TransportLayerException("port is already opened");
                }
                else if (!mPort.IsOpen)
                {
                    mPort.Open();
                }
            }

        }

        void mUsbPort_DataReceived(object sender, UsbDataReceivedEventArgs e)
        {
            if (mUsbPort != null && mUsbPort.IsOpen)
            {
                try
                {
                    Tools._writeDebugDiagnosticMessage(mUsbPort.DevicePath+ " << " + Tools.ToHexString(e.Data, " "), true, false);
                    if (DataReceived != null)
                    {
                        DataReceived(new ZWave.Events.DataReceivedEventArgs(e.Data));
                    }
                }
                catch (Exception ex)
                {
                    Tools._writeDebugDiagnosticMessage(ex.Message, true, true);
                }
            }
        }

        void mClient_DataReceived(object sender, SocketDataReceivedEventArgs e)
        {
            if (mClient != null && mClient.Connected)
            {
                try
                {
                    Tools._writeDebugDiagnosticMessage(mClient.ConnectionString + " << " + Tools.ToHexString(e.Data, " "), true, false);
                    if (DataReceived != null)
                    {
                        DataReceived(new ZWave.Events.DataReceivedEventArgs(e.Data));
                    }
                }
                catch (Exception ex)
                {
                    Tools._writeDebugDiagnosticMessage(ex.Message, true, true);
                }
            }
        }

        void mPort_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {
            if (mPort != null && mPort.IsOpen)
            {
                try
                {
                    int bytesToRead = mPort.BytesToRead;
                    byte[] data = new byte[bytesToRead];
                    mPort.Read(data, 0, bytesToRead);
                    Tools._writeDebugDiagnosticMessage(mPort.PortName + " << " + Tools.ToHexString(data, " "), true, false);
                    if (DataReceived != null)
                    {
                        DataReceived(new DataReceivedEventArgs(data));
                    }
                }
                catch (Exception ex)
                {
                    Tools._writeDebugDiagnosticMessage(ex.Message, true, true);
                }
            }
        }


        /// <summary>
        /// Write to buffer
        /// </summary>
        /// <param name="buffer"></param>
        /// <returns></returns>
        public int Write(byte[] buffer)
        {
            int ret = buffer.Length;
            if (mPort != null && mPort.IsOpen)
            {
                try
                {
                    mPort.Write(buffer, 0, ret);
                    Tools._writeDebugDiagnosticMessage(mPort.PortName + " >> " + Tools.ToHexString(buffer, " "), true, false);
                }
                catch (Exception ex)
                {
                    Tools._writeDebugDiagnosticMessage(ex.Message, true, true);
                }
            }
            else if (mClient != null && mClient.Connected)
            {
                try
                {
                    mClient.Send(buffer);
                    Tools._writeDebugDiagnosticMessage(mClient.ConnectionString + " >> " + Tools.ToHexString(buffer, " "), true, false);
                }
                catch (Exception ex)
                {
                    Tools._writeDebugDiagnosticMessage(ex.Message, true, true);
                }
            }
            else if (mUsbPort != null && mUsbPort.IsOpen)
            {
                try
                {
                    mUsbPort.Write(buffer);
                    Tools._writeDebugDiagnosticMessage(mUsbPort.DevicePath + " >> " + Tools.ToHexString(buffer, " "), true, false);
                }
                catch (Exception ex)
                {
                    Tools._writeDebugDiagnosticMessage(ex.Message, true, true);
                }
            }
            else
                throw new TransportLayerException("port is not opened");
            return ret;

        }

        /// <summary>
        /// Is the port already Open ?
        /// </summary>
        /// <returns></returns>
        public bool IsOpen()
        {
            bool ret = false;
            if (mPort != null)
            {
                ret = mPort.IsOpen;
            }
            else if (mClient != null)
            {
                ret = mClient.Connected;
            }
            else if (mUsbPort != null)
            {
                ret = mUsbPort.IsOpen;
            }
            else
            {
                throw new TransportLayerException("port is not created");
            }
            return ret;
        }

        #endregion
    }
}
