﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Xml.Serialization;
using System.Xml;
using System.Globalization;
using Zensys.ZWave.Application;
using System.Xml.Schema;
using Zensys.Framework;


namespace Zensys.ZWave
{
    public class XmlDataManager
    {
        private ZWaveDefinition mZWaveDefinition;

        public ZWaveDefinition ZWaveDefinition
        {
            get { return mZWaveDefinition; }
            set { mZWaveDefinition = value; }
        }

        private void ValidationCallback(object sender, ValidationEventArgs e)
        {
            Tools._writeDebugDiagnosticMessage(e.Message, true, false);
        }

        private const uint POLY = 0x1021;          /* crc-ccitt mask */

        private byte GetBitCountFormMask(byte mask)
        {
            byte ret = 0;
            if ((mask & 0x01) > 0)
                ret++;
            if ((mask & 0x02) > 0)
                ret++;
            if ((mask & 0x04) > 0)
                ret++;
            if ((mask & 0x08) > 0)
                ret++;
            if ((mask & 0x10) > 0)
                ret++;
            if ((mask & 0x20) > 0)
                ret++;
            if ((mask & 0x40) > 0)
                ret++;
            if ((mask & 0x80) > 0)
                ret++;
            return ret;
        }

        public string ParseApplicationString(byte[] payload)
        {
            string ret = "PARSE_ERROR";
            if (payload != null && payload.Length > 0)
            {
                try
                {
                    foreach (CommandClass item in FindCommandClasses(payload[0]))
                    {
                        ret = item.Text;
                        if (payload.Length > 1)
                        {
                            ret = FindCommand(item, payload[1]).Text;
                        }
                    }
                }
                catch (Exception ex)
                {
                    throw ex;
                }
            }
            else
            {
                ret = "";
            }
            return ret;

        }

        public void ParseApplicationObject(byte[] data, out CommandClassValue[] commandClasses)
        {
            List<CommandClassValue> ret = new List<CommandClassValue>();

            if (data != null && data.Length > 1)
            {
                foreach (CommandClass itemCommandClass in FindCommandClasses(data[0]))
                {
                    if (itemCommandClass.Command != null)
                    {
                        Command itemCommand = FindCommand(itemCommandClass, data[1]);
                        {
                            CommandValue cv = new CommandValue();
                            cv.CommandDefinition = itemCommand;
                            if (itemCommand.Bits == 8 || itemCommand.Bits == 0)
                            {
                                if (data.Length > 2)
                                {
                                    byte[] payload = new byte[data.Length - 2];
                                    Array.Copy(data, 2, payload, 0, payload.Length);
                                    cv.ProcessParameters(payload);
                                }
                            }
                            else
                            {
                                // transform |C|C|C|C|C|P|P|P| to |P|P|P|0|0|0|0|0|
                                // where c - command bit, p - parameter bit
                                byte offset = (byte)(itemCommand.Bits);
                                byte[] payload = new byte[data.Length - 1];
                                payload[0] = (byte)(data[1] << offset);
                                if (data.Length > 2)
                                {
                                    Array.Copy(data, 2, payload, 1, payload.Length - 1);
                                }
                                cv.ProcessParameters(payload, offset);
                            }
                            ProcessParametersTextValue(cv.ParamValues, itemCommandClass);

                            CommandClassValue ccv = new CommandClassValue();
                            ccv.CommandClassDefinition = itemCommandClass;
                            ccv.CommandValue = cv;
                            ccv.Data = data;
                            ret.Add(ccv);
                        }
                    }
                }
            }
            commandClasses = ret.ToArray();
        }

        private void ProcessParametersTextValue(List<ParamValue> paramValues, CommandClass cmdClass)
        {
            foreach (ParamValue item in paramValues)
            {
                if (item.ParamDefinition.Defines != null)
                {
                    DefineSet dSet = FindDefineSet(cmdClass, item.ParamDefinition.Defines);
                    if (dSet != null)
                    {
                        foreach (byte byteItem in item.ByteValueList)
                        {
                            string str = null;
                            foreach (Define dItem in dSet.Define)
                            {
                                if (dItem.KeyId == byteItem)
                                {
                                    str = dItem.Text;
                                }
                            }
                            if (str != null)
                            {
                                item.TextValueList.Add(str);
                            }
                            else
                            {
                                item.TextValueList.AddRange(ParameterToString(new List<byte>(new byte[] { byteItem }), item.ParamDefinition.Type));
                            }
                        }
                    }
                    else
                    {
                        item.TextValueList.AddRange(ParameterToString(item.ByteValueList, item.ParamDefinition.Type));
                    }
                }
                else
                {
                    item.TextValueList.AddRange(ParameterToString(item.ByteValueList, item.ParamDefinition.Type));
                }
            }
        }

        GenericDevice genericDevice = null;

        private IEnumerable<string> ParameterToString(List<byte> list, zwParamType type)
        {
            List<string> ret = new List<string>();
            switch (type)
            {
                case zwParamType.CHAR:
                    ASCIIEncoding encoding = new ASCIIEncoding();
                    try
                    {
                        ret.Add(encoding.GetString(list.ToArray()));
                    }
                    catch (DecoderFallbackException dex)
                    {
                        Tools._writeDebugDiagnosticExceptionMessage(dex.Message);
                    }
                    break;
                case zwParamType.HEX:
                    ret.Add(Tools.GetHex(list.ToArray(), " ", false));
                    break;
                case zwParamType.NUMBER:
                    if (list.Count <= 4) //int
                    {
                        byte[] val = new byte[] { 0, 0, 0, 0 };
                        list.Reverse();
                        Array.Copy(list.ToArray(), val, list.Count);
                        ret.Add(BitConverter.ToInt32(val, 0).ToString());
                    }
                    else // as HEX
                    {
                        ret.Add(Tools.GetHex(list.ToArray(), " ", false));
                    }
                    break;
                case zwParamType.NODE_NUMBER:
                    foreach (byte item in list)
                    {
                        ret.Add(item.ToString("000"));
                    }
                    break;
                case zwParamType.BITMASK:
                    for (int i = 0; i < list.Count; i++)
                    {

                        byte maskByte = list[i];
                        if (maskByte == 0)
                        {
                            continue;
                        }
                        byte bitMask = 0x01;
                        byte bitOffset = 0x01;//nodes starting from 1 in mask bytes array
                        for (int j = 0; j < 8; j++)
                        {
                            if ((bitMask & maskByte) != 0)
                            {
                                byte nodeID = (byte)(((i * 8) + j) + bitOffset);
                                ret.Add(nodeID.ToString("000"));
                            }
                            bitMask <<= 1;
                        }
                    }
                    break;
                case zwParamType.VARIANT_GROUP:
                    ret.Add(Tools.GetHex(list.ToArray(), " ", false));
                    break;
                case zwParamType.BOOLEAN:
                    if (list.Count > 0)
                    {
                        ret.Add(list[0] > 0 ? "true" : "false");
                    }
                    else
                        ret.Add("false");
                    break;
                case zwParamType.MARKER:
                    ret.Add(Tools.GetHex(list.ToArray(), " ", false));
                    break;
                case zwParamType.ENCAP_CMD:
                    ret.Add(Tools.GetHex(list.ToArray(), " ", false));
                    break;
                case zwParamType.BAS_DEV_REF:
                    foreach (byte item in list)
                    {
                        BasicDevice bd = FindBasicDevice(item);
                        if (bd != null)
                        {
                            ret.Add(bd.Name);
                        }
                        else
                        {
                            ret.Add(Tools.GetHex(item, false));
                        }
                    }
                    break;
                case zwParamType.GEN_DEV_REF:
                    foreach (byte item in list)
                    {
                        genericDevice = FindGenericDevice(item);
                        if (genericDevice != null)
                        {
                            ret.Add(genericDevice.Name);
                        }
                        else
                        {
                            ret.Add(Tools.GetHex(item, false));
                        }
                    }
                    break;
                case zwParamType.SPEC_DEV_REF:
                    if (genericDevice != null)
                    {
                        foreach (byte item in list)
                        {
                            SpecificDevice bd = FindSpecificDevice(genericDevice, item);
                            if (bd != null)
                            {
                                ret.Add(bd.Name);
                            }
                            else
                            {
                                ret.Add(Tools.GetHex(item, false));
                            }
                        }
                    }
                    else
                    {
                        ret.Add(Tools.GetHex(list.ToArray(), " ", false));
                    }
                    break;
                case zwParamType.CMD_CLASS_REF:
                    foreach (byte item in list)
                    {
                        CommandClass bd = FindCommandClass(item);
                        if (bd != null)
                        {
                            ret.Add(bd.Name);
                        }
                        else
                        {
                            ret.Add(Tools.GetHex(item, false));
                        }
                    }
                    break;
                case zwParamType.CMD_REF:
                    break;
                default:
                    break;
            }
            return ret;
        }

        public byte GetCommandClassKey(string name)
        {
            byte ret = 0;
            bool found = false;
            foreach (CommandClass var in ZWaveDefinition.CommandClasses)
            {
                if (var.Name == name)
                {
                    ret = var.KeyId;
                    found = true;
                    break;
                }
            }
            if (!found)
            {
                throw new ApplicationException(string.Format("Command Class {0} not found in xml", name));
            }
            return ret;
        }

        public byte GetCommandKey(string cmdClassName, string cmdName)
        {
            foreach (CommandClass var in ZWaveDefinition.CommandClasses)
            {
                if (var.Name == cmdClassName)
                {
                    foreach (Command cmd in var.Command)
                    {
                        if (cmd.Name == cmdName)
                        {
                            return cmd.KeyId;
                        }
                    }
                }
            }
            throw new ApplicationException(string.Format("Command {0} not found in xml", cmdName));
        }

        public byte GetGenericDeviceKey(string name)
        {
            byte ret = 0;
            bool found = false;
            foreach (GenericDevice var in ZWaveDefinition.GenericDevices)
            {
                if (var.Name == name)
                {
                    ret = var.KeyId;
                    found = true;
                    break;
                }
            }
            if (!found)
            {
                throw new ApplicationException(string.Format("Generic Device {0} not found in xml", name));
            }
            return ret;
        }

        public byte GetBasicDeviceKey(string name)
        {
            byte ret = 0;
            bool found = false;
            foreach (BasicDevice var in ZWaveDefinition.BasicDevices)
            {
                if (var.Name == name)
                {
                    ret = var.KeyId;
                    found = true;
                    break;
                }
            }
            if (!found)
            {
                throw new ApplicationException(string.Format("Basic Device {0} not found in xml", name));
            }
            return ret;
        }

        public BasicDevice FindBasicDevice(byte key)
        {
            BasicDevice ret = null;
            foreach (BasicDevice var in ZWaveDefinition.BasicDevices)
            {
                if (var.KeyId == key)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public GenericDevice FindGenericDevice(byte key)
        {
            GenericDevice ret = null;
            foreach (GenericDevice var in ZWaveDefinition.GenericDevices)
            {
                if (var.KeyId == key)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public GenericDevice FindGenericDevice(string name)
        {
            GenericDevice ret = null;
            foreach (GenericDevice var in ZWaveDefinition.GenericDevices)
            {
                if (var.Name == name)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public SpecificDevice FindSpecificDevice(GenericDevice genericDevice, byte key)
        {
            SpecificDevice ret = null;
            foreach (SpecificDevice var in genericDevice.SpecificDevice)
            {
                if (var.KeyId == key)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public SpecificDevice FindSpecificDevice(GenericDevice genericDevice, string name)
        {
            SpecificDevice ret = null;
            foreach (SpecificDevice var in genericDevice.SpecificDevice)
            {
                if (var.Name == name)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public SpecificDevice FindSpecificDevice(string genericName, string specificName)
        {
            SpecificDevice ret = null;
            foreach (SpecificDevice var in FindGenericDevice(genericName).SpecificDevice)
            {
                if (var.Name == specificName)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public CommandClass FindCommandClass(byte key)
        {
            CommandClass ret = null;
            foreach (CommandClass var in ZWaveDefinition.CommandClasses)
            {
                if (var.KeyId == key)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public CommandClass FindCommandClass(string name, byte version)
        {
            CommandClass ret = null;
            foreach (CommandClass var in ZWaveDefinition.CommandClasses)
            {
                if (var.Name == name && var.Version == version)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public CommandClass FindCommandClass(byte key, byte version)
        {
            CommandClass ret = null;
            foreach (CommandClass var in ZWaveDefinition.CommandClasses)
            {
                if (var.KeyId == key && var.Version == version)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public List<CommandClass> FindCommandClasses(byte key)
        {
            List<CommandClass> ret = new List<CommandClass>();
            foreach (CommandClass var in ZWaveDefinition.CommandClasses)
            {
                if (var.KeyId == key)
                {
                    ret.Add(var);
                }
            }
            return ret;
        }

        public Command FindCommand(CommandClass commandClass, byte key)
        {
            Command ret = null;
            foreach (Command var in commandClass.Command)
            {
                if (var.Bits > 0 && var.Bits < 8)
                {
                    if (var.KeyId == (key & Tools.GetMaskFromBits(var.Bits, (byte)(8 - var.Bits))))
                    {
                        ret = var;
                        break;
                    }
                }
                else if (var.KeyId == key)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public Command FindCommand(CommandClass commandClass, string name)
        {
            Command ret = null;
            foreach (Command var in commandClass.Command)
            {
                if (var.Name == name)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public Command FindCommand(string commandClassName, byte commandClassVersion, string name)
        {
            CommandClass cmdClass = FindCommandClass(commandClassName, commandClassVersion);
            Command ret = null;
            foreach (Command var in cmdClass.Command)
            {
                if (var.Name == name)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public DefineSet FindDefineSet(CommandClass cmdClass, string name)
        {
            DefineSet ret = null;
            foreach (DefineSet var in cmdClass.DefineSet)
            {
                if (var.Name == name)
                {
                    ret = var;
                    break;
                }
            }
            return ret;
        }

        public byte[] FillCommand(string cmdClassName, string commandName, byte[] data)
        {
            List<byte> ret = new List<byte>();
            ret.Add(GetCommandClassKey(cmdClassName));
            ret.Add(GetCommandKey(cmdClassName, commandName));
            if (data != null && data.Length > 0)
            {
                ret.AddRange(data);
            }
            return ret.ToArray();
        }

        private void FillCommandInner(List<ParamValue> parameters, ref List<byte> data)
        {
            if (parameters != null && parameters.Count > 0)
            {
                byte structByte = 0;
                byte structByteBitCounter = 0;
                foreach (ParamValue var in parameters)
                {
                    if (var.ParamValues != null && var.ParamValues.Count > 0)
                    {
                        if (structByteBitCounter > 0 && structByteBitCounter < 8)
                        {
                            throw new ApplicationException("Invalid payload");
                        }
                        FillCommandInner(var.ParamValues, ref data);
                    }
                    else
                    {
                        if (var.ParamDefinition.Bits > 0 && var.ParamDefinition.Bits < 8)
                        {
                            byte tt = (byte)(var.ByteValueList[0] << structByteBitCounter);
                            structByteBitCounter += var.ParamDefinition.Bits;
                            structByte += tt;
                        }
                        else
                        {
                            data.AddRange(var.ByteValueList);
                        }

                        if (structByteBitCounter == 8)
                        {
                            data.Add(structByte);
                            structByte = 0;
                            structByteBitCounter = 0;
                        }
                        else if (structByteBitCounter > 8)
                        {
                            throw new ApplicationException("Invalid payload");
                        }
                    }
                }
            }
        }

        public byte[] FillParameters(List<ParamValue> parameters)
        {
            List<byte> data = new List<byte>();
            FillCommandInner(parameters, ref data);
            return data.ToArray();
        }

        public byte[] FillCommand(CommandClass cmdClass, Command cmd, List<ParamValue> parameters)
        {
            List<byte> data = new List<byte>();
            FillCommandInner(parameters, ref data);
            return FillCommand(cmdClass.KeyId, cmd.KeyId, data.ToArray());
        }

        public byte[] FillCommand(byte cmdClass, byte command, byte[] data)
        {
            List<byte> ret = new List<byte>();
            ret.Add(cmdClass);
            ret.Add(command);
            if (data != null && data.Length > 0)
            {
                ret.AddRange(data);
            }
            return ret.ToArray();
        }

    }
}
