/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : API functions (envelope only)
 *--------------------------------------------------------------------
 * MODULE        : Linux - UD
 * DEPENDENCIES  :
 ********************************************************************/

#include <string.h>
#include <stdio.h>

#include "cmapi.h"
#include "udconfig.h"


/*
  This file contains a formal implementation of UD functions.

  The functions are place holders that delegate the call to a sample
  implementation. User is supposed to change some of them
 */

/*
 *====================================================================
 * PURPOSE: Initialize the CIFS configuration
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NQ_SUCCESS on success or NQ_FAIL on error
 *
 * NOTES:   Initializes this module
 *====================================================================
 */

NQ_STATUS
udInit(
    void
    )

{
    return udDefInit();
}

/*
 *====================================================================
 * PURPOSE: Stop the CIFS configuration
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udStop(
    void
    )

{
    udDefStop();
}

/*
 *====================================================================
 * PURPOSE: reset server parameters
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: None
 *
 * NOTES: This function should be called during server restart
 *====================================================================
 */

void
udResetServerParams(
    void
    )
{
    udDefResetServerParams();
}

#ifdef UD_NQ_INCLUDECIFSSERVER
/*
 *====================================================================
 * PURPOSE: Signal to the user level of CIFS server start
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 * NOTES:   NQ calls this function after NQ Server is ready
 *          This implementation is a placeholder
 *====================================================================
 */
void
udCifsServerStarted(
    void
    )
{
    syPrintf("NQSERVER: server is ready");
#ifdef NQ_INTERNALTRACE
#ifdef UD_CS_INCLUDEPASSTHROUGH
    syPrintf(" (pass through authentication enabled)");
#endif /* UD_CS_INCLUDEPASSTHROUGH */
#endif /* NQ_INTERNALTRACE */
    syPrintf("\n");
}

/*
 *====================================================================
 * PURPOSE: Signal to the user level of CIFS server shutdown
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 * NOTES:   NQ calls this function after NQ Server is closed and may be restarted
 *          This implementation is a placeholder
 *====================================================================
 */
void
udCifsServerClosed(
    void
    )
{
    syPrintf("\n---- NQ Server was shut down ---\n");
}
#endif /* UD_NQ_INCLUDECIFSSERVER */

/*
 *====================================================================
 * PURPOSE: Signal to the user level of NetBios Daemon start
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 * NOTES:   NQ calls this function after NetBios Daemon is ready
 *          This implementation is a placeholder
 *====================================================================
 */

void
udNetBiosDaemonStarted(
    void
    )
{
}

/*
 *====================================================================
 * PURPOSE: Signal to the user level of NetBios Daemon shutdown
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 * NOTES:   NQ calls this function after NetBios Daemon is closed and may be restarted
 *          This implementation is a placeholder
 *====================================================================
 */

void
udNetBiosDaemonClosed(
    void
    )
{
}


/*
 *====================================================================
 * PURPOSE: get scope id
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the result
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udGetScopeID(
    NQ_WCHAR *buffer
    )
{
    if (NULL == buffer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    udDefGetScopeID(buffer);

Exit:
    return;
}

/*
 *=========================================================================
 * PURPOSE: get WINS addresses
 *--------------------------------------------------------------------
 * PARAMS:  OUT A semicolon\-delimited list of WINS servers
 *
 * RETURNS: None.
 *
 * NOTES:   returns WINS servers as supplied by User Defined level (for example cm_cfg.txt)
 *=========================================================================
 */

void
udGetWins(
    NQ_WCHAR *pServers
    )
{
    if (NULL == pServers)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    udDefGetWins(pServers);

Exit:
    return;
}

/*
 *====================================================================
 * PURPOSE: get domain name
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the result
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udGetDomain(
    NQ_WCHAR *buffer,
    NQ_BOOL  *isWorkgroup
    )
{
    udDefGetDomain(buffer, isWorkgroup);
}

#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)
/*
 *====================================================================
 * PURPOSE: get DNS initialization parameters (DNS addresses and domain)
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the default domain target belongs to
 *          OUT A semicolon\-delimited list of DNS servers
 *
 * RETURNS: None
 *
 * NOTES:   returns DNS servers as supplied by User Defined level (for example cm_cfg.txt)
 *====================================================================
 */

void
udGetDnsParams(
    NQ_WCHAR *domain,
    NQ_WCHAR *server
    )
{
    if (NULL == domain || NULL == server)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    udDefGetDnsParams(domain, server);

Exit:
    return;
}
#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */

/*
 *====================================================================
 * PURPOSE: get authentication parameters
 *--------------------------------------------------------------------
 * PARAMS:  IN: URI the CIFS client is about to connect to
 *          OUT buffer for user name
 *          OUT buffer for password
 *          OUT buffer for domain name
 *
 * RETURNS: TRUE - success
 *          FALSE - fail
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udGetCredentials(
    const void* resource,
    NQ_WCHAR* userName,
    NQ_WCHAR* password,
    NQ_WCHAR* domain
    )
{
    return udDefGetCredentials(resource, userName, password, domain);
}

/*
 *====================================================================
 * PURPOSE: Determine file system for the given share
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the share name
 *          IN pointer to the share path
 *          OUT buffer for the file system name
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udGetFileSystemName(
    const NQ_WCHAR* shareName,
    const NQ_WCHAR* sharePath,
    NQ_WCHAR* fileSystemName
    )
{
    syAnsiToUnicode(fileSystemName, UD_FS_FILESYSTEMNAME);
}

/*
 *====================================================================
 * PURPOSE: get next share in the list of shares for CIFS Server (extended)
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for share name
 *          OUT buffer for the local path
 *          OUT pointer to variable getting 0 for file system share and 1 for a print queue
 *          OUT buffer for the share description
 *          OUT pointer to variable getting 0 for regular share and 1 for encrypted share (SMB3)
 *
 * RETURNS: TRUE - next share read
 *          FALSE - no more shares
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
udGetNextShareEx(
    NQ_WCHAR* name,
    NQ_WCHAR* map,
    NQ_BOOL* isPrinter,
    NQ_WCHAR* description,
    NQ_BOOL* isEncrypted
    )
{
    return udDefGetNextShareEx(name, map, isPrinter, description, isEncrypted);
}

/*
 *====================================================================
 * PURPOSE: get next mount in the list of mounted volumes for CIFS
 *          Client
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for volume name
 *          OUT buffer for the map path
 *
 * RETURNS: TRUE - a mount read FALSE - no more mounts
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udGetNextMount(
    NQ_WCHAR* name,
    NQ_WCHAR* map
    )
{
    return udDefGetNextMount(name, map);
}

/*
 *====================================================================
 * PURPOSE: get server message signing policy
 *--------------------------------------------------------------------
 * PARAMS:  OUT pointer to parameter getting 0 - disabled, 1 - enabled, 2 - required
 *
 * RETURNS: TRUE - implemented
 *          FALSE - not implemented (use NQ default)
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
udGetMessageSigningPolicy(
        NQ_INT* policy
        )
{
    return udDefGetMessageSigningPolicy(policy);
}

/*
 *====================================================================
 * PURPOSE: get server global encryption
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE - global encryption
 *          FALSE - no global encryption
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
udGetGlobalEncryption(void)
{
    return udDefGetGlobalEncryption();
}


#ifdef UD_NQ_INCLUDESMBCAPTURE
/*
 *====================================================================
 * PURPOSE: whether to use internal capture
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE
 *          FALSE
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
udGetInternalCapture(void)
{
    return udDefGetInternalCapture();
}

/*
*====================================================================
* PURPOSE: get base folder path of internal capture file
*--------------------------------------------------------------------
* PARAMS:  buffer - buffer for the path
*          length - buffer size
*
* NOTES:
*====================================================================
*/
void
udGetCaptureFileBaseFolder(
    NQ_CHAR* buffer,
    NQ_UINT  size
    )
{
    udDefGetCaptureFileBaseFolder(buffer, size);
}
#endif /* UD_NQ_INCLUDESMBCAPTURE */

/*
 *====================================================================
 * PURPOSE: whether to use internal trace
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE
 *          FALSE
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
udGetInternalTrace(void)
{
    return udDefGetInternalTrace();
}

/*
 *====================================================================
 * PURPOSE: get transport priority
 *--------------------------------------------------------------------
 * PARAMS:  transport - the transport identifier
 *
 * RETURNS: transport priority
 *   0 - the transport isn't used
 *   1..3 - the bigger number is highest priority
 *   if more then one transport has same priority
 *   built in order is used: IPv6, IPv4, NetBIOS
 *
 * NOTES:
 *====================================================================
 */

NQ_INT
udGetTransportPriority(
    NQ_UINT transport
    )
{
    /* default implementation returns the same priority for all transports */

    return 1 ;

    /* for redefining one or more priorities:
        1) comment the previous operator
        2) uncomment switch statement
        3) leave cases for those transports that should be redefined
        4) place appropriate priorities (0,1,2,3) instead of ???
    */

    /*    switch (transport)
      {
          case NS_TRANSPORT_NETBIOS:
              return ???;

          case NS_TRANSPORT_IPV4
              return ???;

          case NS_TRANSPORT_IPV6
              return ???;

          default:
              return 0;
    }
    */
}

/*
 *====================================================================
 * PURPOSE: get task priority
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: task priority
 *
 * NOTES:
 *====================================================================
 */

NQ_INT
udGetTaskPriorities(
    void
    )
{
    return udDefGetTaskPriorities();
}

/*
 *====================================================================
 * PURPOSE: get server comment string
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the result
 *
 * RETURNS: None
 *
 * NOTES:
 *
 *====================================================================
 */

void
udGetServerComment(
    NQ_WCHAR *buffer
    )
{
    udDefGetServerComment(buffer);
}

/*
 *====================================================================
 * PURPOSE: get next user name and password from the list of passwords
 *--------------------------------------------------------------------
 * PARAMS:  IN login name
 *          OUT buffer for password
 *          OUT TRUE if the password is hashed, FALSE - otherwise
 *          OUT user number while administrative users have numbers < 0
 *
 * RETURNS: NQ_CS_PWDFOUND - user found equivalent to 3 (deprecated)
 *          NQ_CS_PWDNOAUTH - authentication is not required
 *          NQ_CS_PWDNOUSER - no such user
 *          NQ_CS_PWDLMHASH - user found and password is LM hash (*pwdIsHashed value has to
 *              be TRUE in this case)
 *          NQ_CS_PWDANY - user found and password is either LM and NTLM hash or plain
 *              text depending on the *pwdIsHashed value
 *
 * NOTES:   Opens the file, parses it and stores parameter values if
 *          those parameters were found. User number is returned as ID from
 *          the pwd file
 *====================================================================
 */

NQ_INT
udGetPassword(
    const NQ_WCHAR* userName,
    NQ_CHAR* password,
    NQ_BOOL* pwdIsHashed,
    NQ_UINT32* userNumber
    )
{
    return udDefGetPassword(userName, password, pwdIsHashed, userNumber);
}


/*
 *====================================================================
 * PURPOSE: return driver name
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the result
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udGetDriverName(
    NQ_CHAR *buffer
    )
{
    if (NULL == buffer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    udDefGetDriverName(buffer);

Exit:
    return;
}

/*
 *====================================================================
 * PURPOSE: project-level processing on incoming data to NetBios Daemon
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 * NOTES:   Include any project-level processing here
 *====================================================================
 */

void
udNetBiosDataIn(
    void
    )
{
}

/*
 *====================================================================
 * PURPOSE: project-level processing on incoming data to NQ Server
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 * NOTES:   Include any project-level processing here
 *====================================================================
 */

void
udServerDataIn(
    void
    )
{
}

/*
 *====================================================================
 * PURPOSE: project-level processing on client connection to a share
 *--------------------------------------------------------------------
 * PARAMS:  IN share name
 *
 * RETURNS: None
 *
 * NOTES:   Include any project-level processing here
 *====================================================================
 */

void
udServerShareConnect(
    const NQ_WCHAR* share
    )
{
}

/*
 *====================================================================
 * PURPOSE: project-level processing on client disconnect from a share
 *--------------------------------------------------------------------
 * PARAMS:  IN share name
 *
 * RETURNS: None
 *
 * NOTES:   Include any project-level processing here
 *====================================================================
 */

void
udServerShareDisconnect(
    const NQ_WCHAR* share
    )
{
}

/*
 *====================================================================
 * PURPOSE: allocate buffer in the user space
 *--------------------------------------------------------------------
 * PARAMS:  IN buffer index zero based
 *          IN total number of buffers
 *          IN buffer size in bytes
 *
 * RETURNS: pointer to the buffer
 *
 * NOTES:   Include any project-level processing here
 *====================================================================
 */

NQ_BYTE*
udAllocateBuffer(
    NQ_INT idx,
    NQ_COUNT numBufs,
    NQ_UINT bufferSize
    )
{
    return udDefAllocateBuffer(idx, numBufs, bufferSize);
}

/*
 *====================================================================
 * PURPOSE: release buffer in the user space
 *--------------------------------------------------------------------
 * PARAMS:  IN buffer index zero based
 *          IN total number of buffers to release
 *          IN buffer address
 *          IN buffer size in bytes
 *
 * RETURNS: None
 *
 * NOTES:   Include any project-level processing here
 *====================================================================
 */

void
udReleaseBuffer(
    NQ_INT idx,
    NQ_COUNT numBufs,
    NQ_BYTE* buffAddr,
    NQ_UINT bufferSize
    )
{
    /* do nothing for static buffers */
}

#ifdef UD_NQ_INCLUDECODEPAGE

/*
 *====================================================================
 * PURPOSE: get the current code page number
 *--------------------------------------------------------------------
 * PARAMS:
 *
 * RETURNS: code page as defined in udparams.h (UD_NQ_CODEPAGE<XXX>)
 *
 * NOTES:
 *====================================================================
 */

NQ_INT
udGetCodePage(
    void
    )
{
    return udDefGetCodePage();
}

/*
 *====================================================================
 * PURPOSE: set the current code page number
 *--------------------------------------------------------------------
 * PARAMS: code page number.
 *
 * RETURNS:
 *
 * NOTES:
 *====================================================================
 */

void
udSetCodePage(
    NQ_INT codePage
    )
{
    udDefSetCodePage(codePage);
}

#endif /* UD_NQ_INCLUDECODEPAGE */

/*
 *====================================================================
 * PURPOSE: validate port usage
 *--------------------------------------------------------------------
 * PARAMS:  IN suggested port number
 *
 * RETURNS: valid port number in HBO
 *
 * NOTES:   This function allows to redefine port numbers for those ports that
 *          NQ uses. For instance NQ NetBIOS component is capable of co-existence
 *          with a foreign NetBIOS component on the same server. In this case
 *          NQ NetBIOS should use alternate port numbers instead of well-known
 *          port numbers. Each one of three well-known ports may be alternated separately.
 *====================================================================
 */

NQ_PORT
udGetPort(
    NQ_PORT port
    )
{
    /* default implementation returns the same number */

    return port;

    /* for redefining one or more ports:
        1) comment the previous operator
        2) uncomment switch statement
        3) leave cases for those ports that should be redefined
        4) plave appropriate numbers instead of ???
    */

    /*    switch (port)
    {
    case 137:
        return ???;
    case 138:
        return ???;
    case 139:
        return ???;
    case UD_BR_INTERNALIPCPORT:
        return ???;
    case UD_BR_INTERNALDSPORT:
        return ???;
    case UD_BR_INTERNALNSPORT:
        return ???;
    default:
        return port;
    }
    */
}

#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)

/*
 *====================================================================
 * PURPOSE: get unique ID for the current machine
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer of 12 bytes length
 *
 * RETURNS: None
 *
 * NOTES:   The returned 12-byte value should be:
 *           - "statistically unique" for the given machine
 *           - persistently the same for each call
 *          Recommended methods are:
 *           - MAC address of the default adapter
 *           - product serial number when available
 *====================================================================
 */

void
udGetComputerId(
    NQ_BYTE* buf
    )
{
    if (NULL == buf)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    udDefGetComputerId(buf);

Exit:
    return;
}

/*
 *====================================================================
 * PURPOSE: Get persistent security descriptor for share
 *--------------------------------------------------------------------
 * PARAMS:  IN share name
 *          OUT buffer for SD
 *          IN buffer length
 *
 * RETURNS: SD length or zero on error
 *
 * NOTES:
 *====================================================================
 */

NQ_COUNT
udLoadShareSecurityDescriptor(
    const NQ_WCHAR* shareName,
    NQ_BYTE* buffer,
    NQ_COUNT bufferLen
    )
{
    return udDefLoadShareSecurityDescriptor(shareName, buffer, bufferLen);
}

/*
 *====================================================================
 * PURPOSE: Save persistent security descriptor for share
 *--------------------------------------------------------------------
 * PARAMS:  IN share name
 *          In pointer to SD
 *          IN SD length
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udSaveShareSecurityDescriptor(
    const NQ_WCHAR* shareName,
    const NQ_BYTE* sd,
    NQ_COUNT sdLen
    )
{
    udDefSaveShareSecurityDescriptor(shareName, sd, sdLen);
}

#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT

/*
 *====================================================================
 * PURPOSE: get number of local users
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: Number of local users
 *
 * NOTES:
 *====================================================================
 */

NQ_COUNT
udGetUserCount(
    void
    )
{
    return udDefGetUserCount();
}

/*
 *====================================================================
 * PURPOSE: get user ID by name
 *--------------------------------------------------------------------
 * PARAMS:  IN user name
 *          OUT buffer for user ID
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udGetUserRidByName(
    const NQ_WCHAR* name,
    NQ_UINT32* rid
    )
{
    return udDefGetUserRidByName(name, rid);
}

/*
 *====================================================================
 * PURPOSE: get user name by ID
 *--------------------------------------------------------------------
 * PARAMS:  IN user ID
 *          OUT buffer for user name
 *          OUT buffer for full user name
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udGetUserNameByRid(
    NQ_UINT32 rid,
    NQ_WCHAR* nameBuffer,
    NQ_WCHAR* fullNameBuffer
    )
{
    return udDefGetUserNameByRid(rid, nameBuffer, fullNameBuffer);
}

/*
 *====================================================================
 * PURPOSE: enumerate users
 *--------------------------------------------------------------------
 * PARAMS:  IN user index (zero based)
 *          OUT buffer for user id
 *          OUT buffer for user name (256 bytes)
 *          OUT buffer for user's full name (256 bytes)
 *          OUT buffer for user description (256 bytes)
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udGetUserInfo(
    NQ_UINT index,
    NQ_UINT32* rid,
    NQ_WCHAR* name,
    NQ_WCHAR* fullName,
    NQ_WCHAR* description
    )
{
    return udDefGetUserInfo(index, rid, name, fullName, description);
}

/*
 *====================================================================
 * PURPOSE: modify user
 *--------------------------------------------------------------------
 * PARAMS:  IN user RID
 *          IN user name
 *          IN full user name
 *          IN user description
 *          IN Unicode text password or NULL
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:   This function either creates a new user or modifies an existing one.
 *====================================================================
 */

NQ_BOOL
udSetUserInfo(
    NQ_UINT32 rid,
    const NQ_WCHAR* name,
    const NQ_WCHAR* fullName,
    const NQ_WCHAR* description,
    const NQ_WCHAR* password
    )
{
    return udDefSetUserInfo(rid, name, fullName, description, password);
}

/*
 *====================================================================
 * PURPOSE: add user
 *--------------------------------------------------------------------
 * PARAMS:  IN user name
 *          IN full user name
 *          IN user description
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:   This function either creates a new user or modifies an existing one.
 *====================================================================
 */

NQ_BOOL
udCreateUser(
    const NQ_WCHAR* name,
    const NQ_WCHAR* fullName,
    const NQ_WCHAR* description
    )
{
    return udDefCreateUser(name, fullName, description);
}

/*
 *====================================================================
 * PURPOSE: remove user
 *--------------------------------------------------------------------
 * PARAMS:  IN user RID
 *
 * RETURNS: TRUE when user was deleted
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udDeleteUserByRid(
    NQ_UINT32 rid
    )
{
    return udDefDeleteUserByRid(rid);
}

/*
 *====================================================================
 * PURPOSE: set user administrative rights
 *--------------------------------------------------------------------
 * PARAMS:  IN user RID
 *          IN TRUE to make user an administrator
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udSetUserAsAdministrator(
    NQ_UINT32 rid,
    NQ_BOOL    isAdmin
    )
{
    return udDefSetUserAsAdministrator(rid, isAdmin);
}

#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */

#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)*/

#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION

/*
 *====================================================================
 * PURPOSE: modify/create share information in a persistent store
 *--------------------------------------------------------------------
 * PARAMS:  IN share name to change or NULL for a new share
 *          IN share name
 *          IN share path
 *          IN share descriptor
 *
 * RETURNS: TRUE on success, FALSE on failure
 *
 * NOTES:   user-level should return TRUE in the following cases:
 *          1) new share was perisistently stored
 *          2) existing share was peristently modified
 *          3) new share was not persistently stored but it should
 *             be exposed until server shutdown
 *          4) share was not persistently modified but this modification
 *             should be exposed until server shutdown
 *          user-level should return FALSE when a new share should not \
 *          be created or an existing share should not be modified
 *====================================================================
 */

NQ_BOOL
udSaveShareInformation(
    const NQ_WCHAR* name,
    const NQ_WCHAR* newName,
    const NQ_WCHAR* newMap,
    const NQ_WCHAR* newDescription
    )
{
    return udDefSaveShareInformation(name, newName, newMap, newDescription);
}

/*
 *====================================================================
 * PURPOSE: remove share from the persistent store
 *--------------------------------------------------------------------
 * PARAMS:  IN share name
 *
 * RETURNS: TRUE on success, FALSE on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udRemoveShare(
    const NQ_WCHAR* name
    )
{
    return udDefRemoveShare(name);
}

#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */

#ifdef UD_NQ_INCLUDEEVENTLOG

/*
 *====================================================================
 * PURPOSE: event log function
 *--------------------------------------------------------------------
 * PARAMS:  IN code of NQ module that originated this event
 *          IN event class code
 *          IN event type
 *          IN pointer to the user name string
 *          IN IP address on the second side of the connection
 *          IN zero if the operation has succeeded or error code on failure
 *             for server event this code is the same that will be transmitted
 *             to the client
 *             for an NQ CIFS client event this value is the same that will be
 *             installed as system error
 *          IN pointer to a structure that is filled with event data
 *             actual structure depends on event type
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udEventLog (
    NQ_UINT module,
    NQ_UINT eventClass,
    NQ_UINT type,
    const NQ_WCHAR* userName,
    const NQ_IPADDRESS* pIp,
    NQ_UINT32 status,
    const NQ_BYTE* parameters
    )
{
    udDefEventLog(module, eventClass, type, userName, pIp, status, parameters);
}

#endif /* UD_NQ_INCLUDEEVENTLOG */

#ifdef UD_CC_INCLUDEDOMAINMEMBERSHIP

NQ_BOOL
udGetComputerSecret(
    NQ_BYTE **secret
    )
{
    return udDefGetComputerSecret(secret);
}

void
udSetComputerSecret(
    NQ_BYTE *secret
    )
{
    udDefSetComputerSecret(secret);
}

NQ_BOOL
udGetComputerSecretByDomain(
        NQ_BYTE *secret,
        const NQ_WCHAR *domainDNS,
        NQ_WCHAR *domainNB  /* OUT */
    )
{
    NQ_BOOL result = FALSE;

    if (NULL == secret || NULL == domainDNS)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    result = udDefGetComputerSecretByDomain(secret, domainDNS, domainNB);

Exit:
    return result;
}

void
udSetComputerSecretByDomain(
    const NQ_BYTE *secret,
    const NQ_WCHAR *domainDNS,
    const NQ_WCHAR *domainNB
    )
{
    if (NULL == secret || NULL == domainDNS)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    udDefSetComputerSecretByDomain(secret, domainDNS, domainNB);

Exit:
    return;
}

#endif /* UD_CC_INCLUDEDOMAINMEMBERSHIP */

void
udNetBiosError(
    NQ_STATUS cause,                /* error code */
    const NQ_CHAR* name             /* NetBIOS name that caused that error */
    )
{
}


void
udGetHostName(
    NQ_CHAR* buffer,
    NQ_UINT length
    )
{
    if (NULL == buffer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    udDefGetHostName(buffer, length);

Exit:
    return;
}


/*
*====================================================================
* PURPOSE: get base folder path of log file
*--------------------------------------------------------------------
* PARAMS:  buffer - buffer for the path
*          size   - buffer size
*
* NOTES:
*====================================================================
*/
void
udGetLogFileBaseFolder(
    NQ_CHAR* buffer,
    NQ_UINT  size
    )
{
    udDefGetLogFileBaseFolder(buffer, size);
}

/*
*====================================================================
* PURPOSE: get disk size limit
*--------------------------------------------------------------------
* PARAMS:  size - pointer to size limit
*
* NOTES:
*====================================================================
*/
void
udGetMaxDiskSize(
    NQ_UINT32 *size
    )
{
    udDefGetMaxDiskSize(size);
}
