/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : GSASL interface implementation
 *--------------------------------------------------------------------
 * MODULE        : Linux - SY
 * DEPENDENCIES  : None
 ********************************************************************/

#include "cmapi.h"
#include "sycompil.h" 

#if defined(UD_CC_INCLUDEEXTENDEDSECURITY) && defined(UD_CC_INCLUDEEXTENDEDSECURITY_KERBEROS)

#include <krb5.h>

/*#define SYSASL_DEBUG*/

/*#define MIT*/
#define HEIMDAL

/*
 * some differences between Heimdal and MIT
 */

#ifdef HEIMDAL

#define krb5_free_unparsed(_a, _b)                  krb5_xfree(_b)
#define krb5_auth_con_getlocalsubkey(_a, _b, _c)    krb5_auth_con_getlocalsubkey(_a, _b, _c)
#define krb5_set_default_tgs_ktypes(_a,_b)          krb5_set_default_in_tkt_etypes(_a, _b)
#define krb5_get_err_text(_c, _e)                   krb5_get_error_message(_c, _e) /* error_message(_e) */
#define KEYDATA(_k)                                 (_k)->keyvalue.data
#define KEYLEN(_k)                                  (_k)->keyvalue.length
#define krb5_free_data_contents(_c, _d)             krb5_data_free(_d)

#else /* HEIMDAL */
#ifdef MIT

#define krb5_free_unparsed(_a, _b)                  krb5_free_unparsed_name(_a, _b)
#ifndef KRB5_DEPRECATED
#define krb5_auth_con_getlocalsubkey(_a, _b, _c)    krb5_auth_con_getlocalsubkey(_a, _b, _c)
#else
#define krb5_auth_con_getlocalsubkey(_a, _b, _c)    krb5_auth_con_getsendsubkey(_a, _b, _c)
#endif /* KRB5_DEPRECATED */
#define krb5_set_default_tgs_ktypes(_a,_b)          krb5_set_default_tgs_enctypes(_a, _b)
#define krb5_get_err_text(_c, _e)                   krb5_get_error_message(_c, _e)/* error_message(_e) */
#define KEYDATA(_k)                                 (_k)->contents
#define KEYLEN(_k)                                  (_k)->length
#define krb5_free_data_contents(_c, _d)             krb5_free_data_contents(_c, _d)

#else /* MIT */

#error Neither HEIMDAL nor MIT is defined

#endif /* MIT */
#endif /* HEIMDAL */

/** Require mutual authentication */
#define SYSASL_REQUIRE_MUTUAL_AUTH
/*#undef SYSASL_REQUIRE_MUTUAL_AUTH*/

/** Number of retries to attempt to get a service ticket */
#define SYSASL_GET_TICKET_NUM_OF_RETRIES    3

/** Number of seconds considered still valid to use for service ticket approaching it's end of life */
#define SYSASL_TICKET_SECONDS_LEFT          10

/*
 * Static functions and data
 *
 */

typedef struct _context
{
    krb5_context krbCtx;            /* Kerberos context */
    krb5_auth_context authCtx;      /* authentication context */
    krb5_creds *creds;              /* credentials to use */
    krb5_data in;                   /* incoming data */
    krb5_data out;                  /* generated data */
}
Context;

/*
 *====================================================================
 * PURPOSE: Security mechanism ID
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: pointer to the mechanism name
 *
 * NOTES:
 *====================================================================
 */

const NQ_CHAR*
sySaslGetSecurityMechanism(
    void
    )
{
    return "GSSAPI";
}

/*
 *====================================================================
 * PURPOSE: start SASL client
 *--------------------------------------------------------------------
 * PARAMS:  IN callback
 *
 * RETURNS: TRUE on success, FALSE on error
 *
 * NOTES:   the callback calls sySaslSetCredentials
 *====================================================================
 */

NQ_BOOL
sySaslClientInit(
    void* cb
    )
{
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: stop SASL client
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: TRUE on success, FALSE on error
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
sySaslClientStop(
    void
    )
{
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: create SASL context client
 *--------------------------------------------------------------------
 * PARAMS:  IN principal name
 *          IN is smb2
 *          IN signing on/off
 *
 * RETURNS: SASL context or illegal handle on error
 *
 * NOTES:
 *====================================================================
 */

NQ_BYTE*
sySaslContextCreate(
    const NQ_CHAR* serverName,
    NQ_BOOL isSmb2,
    NQ_BOOL signingOn
    )
{
    Context* ctx = NULL;
    krb5_ccache cache = NULL;
    krb5_principal server = NULL;
    krb5_creds creds;
    int i, ret;
    NQ_BOOL success = FALSE;
    NQ_CHAR principal[257];
    NQ_CHAR **pRealms = NULL;
    NQ_WCHAR userNameW[CM_USERNAMELENGTH];
    NQ_CHAR userName[CM_USERNAMELENGTH];
    NQ_WCHAR domainNameW[UD_NQ_HOSTNAMESIZE];
    NQ_CHAR domainName[UD_NQ_HOSTNAMESIZE];
#ifdef UD_NQ_INCLUDETRACE
    const NQ_CHAR *errMessage = NULL;
#endif /* UD_NQ_INCLUDETRACE */
    NQ_STATUS errorCode = NQ_ERR_KERBEROSERROR;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "serverName:%s isSmb2:%s signingOn:%s", serverName ? serverName : "", isSmb2 ? "TRUE" : "FALSE", signingOn ? "TRUE" : "FALSE");

    syStrcpy(principal, serverName);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Principal: %s", principal);

    /*initialize_krb5_error_table();*/
                 
    /* Initialize contexts */
    ctx = (Context *)syMalloc(sizeof(*ctx));
    if (NULL == ctx)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate context");
        goto Exit;
    }

    ctx->krbCtx = NULL;
    ctx->authCtx = NULL;
    ctx->creds = NULL;
    syMemset(&ctx->in, 0, sizeof(ctx->in));
    syMemset(&ctx->out, 0, sizeof(ctx->out));
    syMemset(&creds, 0, sizeof(creds));

    ret = krb5_init_context(&ctx->krbCtx);
    if (0 != ret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to initialize Kerberos context: %d", ret);
        goto ExitCleanup;
    }
    ret = krb5_auth_con_init(ctx->krbCtx, &ctx->authCtx);
    if (0 != ret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to initialize Kerberos auth context: %d", ret);
        goto ExitCleanup;
    }

    ctx->in.data = NULL;
    ctx->out.data = NULL;

    ret = krb5_cc_resolve(ctx->krbCtx, krb5_cc_default_name(ctx->krbCtx), &cache);
    if (0 != ret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to resolve default cache: %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
        goto ExitCleanup;
    }

    if (!udGetCredentials(NULL, userNameW, NULL, domainNameW))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get user credentials");
        goto ExitCleanup;
    }

    /* verify realm doesn't differ from credentials domain name */
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Getting host realm for local host");

    ret = krb5_get_host_realm(ctx->krbCtx, NULL, &pRealms);
    if ((0 == ret) && (NULL != pRealms) && (NULL != pRealms[0]) && ('\0' != *pRealms[0]))
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Realm: %s", pRealms[0]);

        cmUnicodeToAnsi(domainName, domainNameW);
        if (cmAStricmp(pRealms[0], domainName) != 0)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Domain name supplied: %s differs from realm", domainName);
            /*errorCode = NQ_ERR_USERNOTFOUND;*/
            goto ExitCleanup;
        }
    }
    else
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get host realm: %s, make sure local hostname has FQDN format", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
        goto ExitCleanup;
    }

    /* restrict encryption types for SMB1 connection with signing */
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "smb%s, signing %s", isSmb2 ? "2" : "1", signingOn ? "on" : "off");
    if (/*!isSmb2 && */signingOn)
    {
        const krb5_enctype enctypes[] = {   
                                            ENCTYPE_AES256_CTS_HMAC_SHA1_96,
                                            ENCTYPE_ARCFOUR_HMAC,
                                            ENCTYPE_DES_CBC_MD5, 
                                            ENCTYPE_DES_CBC_CRC, 
                                            ENCTYPE_NULL };

        ret = krb5_set_default_tgs_ktypes(ctx->krbCtx, enctypes);
        if (0 != ret)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to restrict enctypes: %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
            goto ExitCleanup;
        }
    }
    /*
     * obtain ticket by credentials:
     *  - create authentication context
     *  - find principal in the cache
     *  - get credentials (ticket) from the cache/kdc (several attempts)
     */

    ret = krb5_parse_name(ctx->krbCtx, principal, &server);
    if (0 != ret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to parse server name: %s, make sure local hostname has FQDN format", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
        goto ExitCleanup;
    }

    ret = krb5_copy_principal(ctx->krbCtx, server, &creds.server);
    if (0 != ret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to copy server principal: %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
        goto ExitCleanup;
    }
    ret = krb5_cc_get_principal(ctx->krbCtx, cache, &creds.client);
    if (0 != ret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get client principal: %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
        goto ExitCleanup;
    }

    cmUnicodeToAnsi(userName, userNameW);
    success = FALSE;
    for (i = 0; i < SYSASL_GET_TICKET_NUM_OF_RETRIES; i++)
    {
        char *name = NULL;
        time_t now;

        /* get service ticket using creds as input, output in ctx->creds,
         * last (default) credentials ('Default principal' in klist) are always used for obtaining the ticket */
        ret = krb5_get_credentials(ctx->krbCtx, 0, cache, &creds, &ctx->creds);
        if (0 != ret)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get credentials (service ticket): %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
            /* retry as this operation usually involves talking to KDC */
            continue;
        }

        ret = krb5_unparse_name(ctx->krbCtx, ctx->creds->client, &name);
        if (0 != ret)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to unparse the Kerberos name: %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
            goto ExitCleanup;
        }

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "NQ user: %s", userName);
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Ticket user:%s", name);

        ret = cmAStrincmp(name, userName, (NQ_COUNT)syStrlen(userName));
        krb5_free_unparsed(ctx->krbCtx, name);
        if (0 != ret)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Ticket for wrong user");
            /*errorCode = NQ_ERR_USERNOTFOUND;*/
            goto ExitCleanup;
        }

        /* check ticket expiration */
        now = time(NULL);
        if (ctx->creds->times.starttime > now)
        {
            int offset = (int)((unsigned)ctx->creds->times.starttime - (unsigned)now);
            krb5_set_real_time(ctx->krbCtx, (krb5_timestamp)(now + offset + 1), 0);
        }

        if (ctx->creds->times.endtime <= (now + SYSASL_TICKET_SECONDS_LEFT)) /* expires soon */
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Ticket expired, system time: %ld, endtime: %d", now, ctx->creds->times.endtime);
        }
        else
        {
            success = TRUE;
        }

        break;
    } /* end of for (i = 0; i < SYSASL_GET_TICKET_NUM_OF_RETRIES; i++) */

    if (SYSASL_GET_TICKET_NUM_OF_RETRIES == i)
    {
        LOGERR(CM_TRC_LEVEL_LOW_ERROR, "Failed to get credentials (service ticket), %d retries", i);
    }
  
ExitCleanup:
    if (NULL != ctx->krbCtx)
    {
#ifdef UD_NQ_INCLUDETRACE
        if (NULL != errMessage)
        {
            krb5_free_error_message(ctx->krbCtx, errMessage);
        }
#endif /* UD_NQ_INCLUDETRACE */
        if (NULL != pRealms)
        {
            krb5_free_host_realm(ctx->krbCtx, pRealms);
        }
        if (NULL != cache)
        {
            krb5_cc_close(ctx->krbCtx, cache);
        }
        if (NULL != server)
        {
            krb5_free_principal(ctx->krbCtx, server);
        }

        krb5_free_cred_contents(ctx->krbCtx, &creds);
    }

    if (FALSE == success)
    {
        sySaslContextDispose((NQ_BYTE *)ctx);
        ctx = NULL;

        sySetLastError(errorCode);
    }
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%p", ctx);
    return (NQ_BYTE*)ctx;
}

/*
 *====================================================================
 * PURPOSE: dispose SASL context
 *--------------------------------------------------------------------
 * PARAMS:  IN SASL context
 *
 * RETURNS: TRUE on SUCCESS, FALSE on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
sySaslContextDispose(
    NQ_BYTE* context
    )
{   
    NQ_BOOL res = FALSE;
    Context* ctx = (Context*)context;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "context:%p", context);

    if (NULL == ctx)
    {
        goto Exit;
    }
    if (NULL == ctx->krbCtx)
    {
        goto Exit;
    }
    if (NULL != ctx->authCtx)
    {
        krb5_auth_con_free(ctx->krbCtx, ctx->authCtx);
    }
    if (NULL != ctx->creds)
    {
        krb5_free_creds(ctx->krbCtx, ctx->creds);
    }
    if (NULL != ctx->out.data)
    {
        krb5_free_data_contents(ctx->krbCtx, &ctx->out);
    }
    if (NULL != ctx->in.data)
    {
        krb5_free_data_contents(ctx->krbCtx, &ctx->in);
    }

    krb5_free_context(ctx->krbCtx);
    syFree(ctx);

    res = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "res:%s", res ? "true" : "false");
    return res;
}

/*
 *====================================================================
 * PURPOSE: set security mechanism for this context
 *--------------------------------------------------------------------
 * PARAMS:  IN SASL context
 *          IN mechanism name
 *
 * RETURNS: TRUE on SUCCESS, FALSE on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
sySaslClientSetMechanism(
    NQ_BYTE* context,
    const NQ_CHAR* name
    )
{
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: generate first client-side blob
 *--------------------------------------------------------------------
 * PARAMS:  IN SASL context
 *          IN list for security mechanisms
 *          OUT buffer for blob pointer
 *          OUT buffer for blob length
 *
 * RETURNS: TRUE on SUCCESS, FALSE on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
sySaslClientGenerateFirstRequest(
    NQ_BYTE* context,
    const NQ_CHAR* mechList,
    NQ_BYTE** blob,
    NQ_COUNT* blobLen
    )
{
    Context* ctx = (Context*)context;
    int ret;
    krb5_flags options = AP_OPTS_USE_SUBKEY
#ifdef SYSASL_REQUIRE_MUTUAL_AUTH
            | AP_OPTS_MUTUAL_REQUIRED
#endif /* SYSASL_REQUIRE_MUTUAL_AUTH */
            ;
    NQ_BOOL result = FALSE;
#ifdef UD_NQ_INCLUDETRACE
    const NQ_CHAR *errMessage = NULL;
#endif /* UD_NQ_INCLUDETRACE */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "context:%p list:%p blob:%p len:%p", context, mechList, blob, blobLen);
#ifdef SYSASL_REQUIRE_MUTUAL_AUTH
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Mutual authentication required");
#endif /* SYSASL_REQUIRE_MUTUAL_AUTH */

    if (NULL != ctx->out.data)
    {
        krb5_free_data_contents(ctx->krbCtx, &ctx->out);
    }

    ret = krb5_mk_req_extended(ctx->krbCtx, &ctx->authCtx, options, &ctx->in, ctx->creds, &ctx->out);
    if (0 != ret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to generate blob: %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
        goto Exit;
    }

    *blob = (NQ_BYTE *)ctx->out.data;
    *blobLen = (NQ_COUNT)ctx->out.length;
    result = TRUE;

Exit:
#ifdef UD_NQ_INCLUDETRACE
    if (NULL != errMessage)
    {
        krb5_free_error_message(ctx->krbCtx, errMessage);
    }
#endif /* UD_NQ_INCLUDETRACE */

    if (NULL != ctx->in.data)
    {
        krb5_free_data_contents(ctx->krbCtx, &ctx->in);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: %d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: generate next client-side blob
 *--------------------------------------------------------------------
 * PARAMS:  IN SASL context
 *          IN server response blob
 *          IN server response length
 *          OUT buffer for request blob pointer
 *          OUT buffer for request blob length
 *          IN connection pointer (no use)
 * RETURNS: TRUE on SUCCESS, FALSE on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
sySaslClientGenerateNextRequest(
    NQ_BYTE* context,
    const NQ_BYTE* inBlob,
    NQ_COUNT inBlobLen,
    NQ_BYTE** outBlob,
    NQ_COUNT* outBlobLen,
    NQ_BYTE* con
    )
{
    Context* ctx = (Context*)context;
    int ret;
    krb5_flags options = AP_OPTS_USE_SUBKEY
#ifdef SYSASL_REQUIRE_MUTUAL_AUTH
            | AP_OPTS_MUTUAL_REQUIRED
#endif /* SYSASL_REQUIRE_MUTUAL_AUTH */
            ;
    NQ_BOOL result = FALSE;
#ifdef UD_NQ_INCLUDETRACE
    const NQ_CHAR *errMessage = NULL;
#endif /* UD_NQ_INCLUDETRACE */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "context:%p inBlob:%p inLen:%d outBlob:%p outLen:%p con:%p", context, inBlob, inBlobLen, outBlob, outBlobLen, con);
#ifdef SYSASL_REQUIRE_MUTUAL_AUTH
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Mutual authentication required");
#endif /* SYSASL_REQUIRE_MUTUAL_AUTH */

    if (NULL != ctx->in.data)
    {
        krb5_free_data_contents(ctx->krbCtx, &ctx->in);
    }
    ctx->in.data = (char*)inBlob;
    ctx->in.length = inBlobLen;

    if (NULL != ctx->out.data)
    {
        krb5_free_data_contents(ctx->krbCtx, &ctx->out);
    }

    krb5_auth_con_free(ctx->krbCtx, ctx->authCtx);
    ret = krb5_auth_con_init(ctx->krbCtx, &ctx->authCtx);
    if (0 != ret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to initialize Kerberos auth context: %d", ret);
        goto Exit;
    }

    ret = krb5_mk_req_extended(ctx->krbCtx, &ctx->authCtx, options, &ctx->in, ctx->creds, &ctx->out);
    if (0 != ret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to generate blob: %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
        goto Exit;
    }

    *outBlob = (NQ_BYTE *)ctx->out.data;
    *outBlobLen = (NQ_COUNT)ctx->out.length;
    result = TRUE;

Exit:
#ifdef UD_NQ_INCLUDETRACE
    if (NULL != errMessage)
    {
        krb5_free_error_message(ctx->krbCtx, errMessage);
    }
#endif /* UD_NQ_INCLUDETRACE */

    ctx->in.data = NULL; /* allocated and must be freed by caller */
    ctx->in.length = 0;

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: %d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: encrypt packet according to negotiated security rules
 *--------------------------------------------------------------------
 * PARAMS:  IN SASL context
 *          IN/OUT packet to encrypt
 *          IN/OUT packet length
 *
 * RETURNS: TRUE on SUCCESS, FALSE on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
sySaslEncode(
    NQ_BYTE* context,
    NQ_BYTE* packet,
    NQ_COUNT* len
    )
{
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: get session key
 *--------------------------------------------------------------------
 * PARAMS:  IN SASL context
 *          IN buffer for session key
 *          IN buffer length, OUT key length
 *          IN response buffer (required for mutual authentication)
 *          IN response buffer length
 *
 * RETURNS: TRUE on SUCCESS, FALSE on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
sySaslGetSessionKey(
    NQ_BYTE* context,
    NQ_BYTE* buffer,
    NQ_COUNT* len,
    NQ_BYTE* resp,
    NQ_COUNT lenResp
    )
{
    NQ_BOOL result = FALSE;
    Context* ctx = (Context*)context;
    krb5_keyblock *krbKey = NULL;
    int ret;
#ifdef UD_NQ_INCLUDETRACE
    const NQ_CHAR *errMessage = NULL;
#endif /* UD_NQ_INCLUDETRACE */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "context:%p buff:%p len:%d resp:%p lenResp:%u", context, buffer, *len, resp, lenResp);

#ifdef SYSASL_REQUIRE_MUTUAL_AUTH
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Mutual authentication required");
    if (NULL == resp)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Mutual blob is missing in Server's response");
        /* do not fail currently */
    }
#endif /* SYSASL_REQUIRE_MUTUAL_AUTH */

    /* mutual authentication blob received */
    if (NULL != resp)
    {
        krb5_ap_rep_enc_part *repl;

        ctx->in.data = (char*)resp;
        ctx->in.length = lenResp;

        ret = krb5_rd_rep(ctx->krbCtx, ctx->authCtx, &ctx->in, &repl);
        ctx->in.data = NULL;
        if (0 != ret)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to decrypt AP-REP (mutual auth): %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
            goto Exit;
        }
        krb5_free_ap_rep_enc_part(ctx->krbCtx, repl);

#ifdef HEIMDAL
       /* Heimdal's krb5_rd_rep() in the contrary to MIT's doesn't create new local session key which results in wrong SMB signing,
        * we replace the local session key with the remote
        */
        {
            krb5_keyblock *keyRemote = NULL;

            ret = krb5_auth_con_getremotesubkey(ctx->krbCtx, ctx->authCtx, &keyRemote);
            if (0 != ret)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get remote session key (mutual auth): %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
                goto Exit;
            }
            if ((NULL != keyRemote) && (NULL != KEYDATA(keyRemote)))
            {
                ret = krb5_auth_con_setlocalsubkey(ctx->krbCtx, ctx->authCtx, keyRemote);
                krb5_free_keyblock(ctx->krbCtx, keyRemote);
                if (0 != ret)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Failed to replace local session key with remote session key (mutual auth): %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
                    goto Exit;
                }
            }
            else
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Null remote session key (mutual auth): %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
                goto Exit;
            }
        }
#endif /* HEIMDAL */
    }

    ret = krb5_auth_con_getlocalsubkey(ctx->krbCtx, ctx->authCtx, &krbKey);
    if (0 != ret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get session key: %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
        goto Exit;
    }
    if ((NULL != krbKey) && (NULL != KEYDATA(krbKey)))
    {
        if (KEYLEN(krbKey) > *len)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Session key is too long, %lu > %u", (NQ_ULONG)KEYLEN(krbKey), *len);
            /* copy session key up to *len size */
            syMemcpy(buffer, KEYDATA(krbKey), *len);
        }
        else
        {
            syMemcpy(buffer, KEYDATA(krbKey), KEYLEN(krbKey));
            *len = (NQ_COUNT)KEYLEN(krbKey);
        }
        LOGDUMP(CM_TRC_LEVEL_MESS_ALWAYS, "Session key (KRB5)", buffer, (NQ_UINT)KEYLEN(krbKey));

        krb5_free_keyblock(ctx->krbCtx, krbKey);

        result = TRUE;
    }
    else
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Null session key: %s", errMessage = krb5_get_err_text(ctx->krbCtx, ret));
        goto Exit;
    }
Exit:
#ifdef UD_NQ_INCLUDETRACE
    if (NULL != errMessage)
    {
        krb5_free_error_message(ctx->krbCtx, errMessage);
    }
#endif /* UD_NQ_INCLUDETRACE */

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: check SASL context 
 *--------------------------------------------------------------------
 * PARAMS:  IN SASL context
 *
 * RETURNS: TRUE on success, FALSE on error
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
sySaslContextIsValid(
    const NQ_BYTE* c
    )
{
    return c != NULL;
}    


#endif /* UD_CC_INCLUDEEXTENDEDSECURITY && UD_CC_INCLUDEEXTENDEDSECURITY_KERBEROS */
