/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Light-weight DNS client
 *--------------------------------------------------------------------
 * MODULE        : Network 
 * DEPENDENCIES  :
 ********************************************************************/

#include "udparams.h"
#include "nsapi.h"
#include "cmresolver.h"
#include "cmbuf.h"
#include "amapiin.h"
#include "nsdns.h"

#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)

#define NSDNS_DATAGRAM_BUFSIZE (CM_NB_DATAGRAMBUFFERSIZE - 40)
typedef struct
{
    NQ_CHAR dnsDomain[CM_NQ_HOSTNAMESIZE + 1];
    NQ_IPADDRESS dnsServers[UD_NQ_MAXDNSSERVERS * UD_NS_MAXADAPTERS];
    NQ_COUNT numServers;
    NQ_UINT16 id;
    NQ_IPADDRESS llmnrIp4;
    NQ_IPADDRESS llmnrIp6;
    SYMutex dataGuard;
    SYMutex idGuard;
    NQ_BOOL isRegistered;
    NQ_BOOL isNewServerSet;   /* boolean to check is cmDnsSetServers has been called*/
}
StaticData;

static NQ_BOOL isModuleInitialized = FALSE;
static NQ_COUNT moduleInitCount = 0;
#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

typedef struct  /* secure exchange data */
{
    const NQ_IPADDRESS * ip;            /* server IP */
    SYSocketHandle socket;              /* TCP socket */
    CMBlob tkey;                        /* TKEY */
    NQ_INT originalIdOffset;            /* offset from in tkey to original ID field */
    NQ_CHAR name[100];                  /* TKEY name */
    CMBlob sessionKey;                  /* not used - for compatibility with GSSAPI */
    CMBlob macKey;                      /* not used - for compatibility with GSSAPI */
} 
TkeyContext;

/* -- Static functions -- */

#ifndef UD_CM_DONOTREGISTERHOSTNAMEDNS
/* Place new record ID in the "Original ID' field of TKEY */
static void setTkeyOriginalId(const TkeyContext * pTkey, NQ_UINT16 id)
{
    CMRpcPacketDescriptor writer;       /* to pack data */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)

    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE *)(pTkey->tkey.data + pTkey->originalIdOffset), 2)

    cmRpcSetDescriptor(&writer, bufPos, TRUE);
    cmRpcPackUint16(&writer, id);
}
#endif /* UD_CM_DONOTREGISTERHOSTNAMEDNS */

/* pack DNS header */
void nsWriteDnsHeader(CMRpcPacketDescriptor * writer, const DnsHeader * pHeader)
{
    cmRpcPackUint16(writer, pHeader->id);
    cmRpcPackByte(writer, pHeader->flags1);
    cmRpcPackByte(writer, pHeader->flags2);
    cmRpcPackUint16(writer, pHeader->questions);
    cmRpcPackUint16(writer, pHeader->answers);
    cmRpcPackUint16(writer, pHeader->authority);
    cmRpcPackUint16(writer, pHeader->additional);
}

/* parse DNS header */
void nsReadDnsHeader(CMRpcPacketDescriptor * reader, DnsHeader * pHeader)
{
    cmRpcParseUint16(reader, &pHeader->id);
    cmRpcParseByte(reader, &pHeader->flags1);
    cmRpcParseByte(reader, &pHeader->flags2);
    cmRpcParseUint16(reader, &pHeader->questions);
    cmRpcParseUint16(reader, &pHeader->answers);
    cmRpcParseUint16(reader, &pHeader->authority);
    cmRpcParseUint16(reader, &pHeader->additional);
}

/* encode fully qualified name as DNS string, return total bytes written */
NQ_COUNT nsDnsEncodeName(CMRpcPacketDescriptor * writer, const NQ_CHAR *name)
{
    NQ_CHAR *s;     /* pointer in string */
    NQ_UINT length; /* next segment length */
    NQ_COUNT lengthWritten = cmRpcGetDataCount(writer); /* length written till now */

    do
    {
        s = (NQ_CHAR *)syStrchr(name, '.');
        length = (NQ_UINT)(s ? (NQ_UINT)(s - name) : syStrlen(name));
        cmRpcPackByte(writer, (NQ_BYTE)length);
        cmRpcPackBytes(writer, (NQ_BYTE *)name, length);
        if (s)
        {
            name = s + 1;
        }
    }
    while (s);

    cmRpcPackByte(writer, 0);

    return (NQ_COUNT)(cmRpcGetDataCount(writer) - lengthWritten);
}

/* convert DNS string into fully qualified name */
/* Assumption - size of name is CM_NQ_HOSTNAMESIZE + 1 */
NQ_STATUS nsDnsDecodeName(CMRpcPacketDescriptor * reader, NQ_CHAR * name)
{
    NQ_BOOL jump;       /* jump indicator */
    NQ_UINT length;     /* segment length */
    NQ_CHAR * t = NULL; /* pointer in name string */
    NQ_BYTE temp;
    CMRpcPacketDescriptor readerClone;
    NQ_STATUS result = NQ_SUCCESS;
    NQ_UINT cumulativeNameLength = 0;

    cmRpcCloneDescriptor(reader, &readerClone);

    if (name)
    {
        *name = '\0';
        t = name;
    }

    jump = FALSE;
    cmRpcParseByte(&readerClone, &temp);
    for (; temp != '\0'; )
    {
        if (0xc0 == temp && !jump)
        {
            cmRpcParseByte(&readerClone, &temp);

            cmRpcResetDescriptor(&readerClone);
            cmRpcParseSkip(&readerClone, temp);
            cmRpcParseByte(&readerClone, &temp);
            jump = TRUE;
            cmRpcParseSkip(reader, 2);
            continue;
        }

        length = temp;

        if (name)
        {
            NQ_CHAR *p = (NQ_CHAR*)cmRpcGetPositionBytePtr(&readerClone, 0, length);

            if (NULL == p)
            {
                result = NQ_FAIL;
                goto Exit;
            }

            cumulativeNameLength += length;
            if (cumulativeNameLength > CM_NQ_HOSTNAMESIZE) /* Assume that the size of name is CM_NQ_HOSTNAMESIZE */
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "The supplied DNS string too long");
                result = NQ_FAIL;
                goto Exit;
            }

            syStrncat((NQ_CHAR*)name, p, length);
            t += length;
        }
        cmRpcParseSkip(&readerClone, (NQ_INT)length);

        cmRpcParseByte(&readerClone, &temp);
        if (name && temp)
        {
            cumulativeNameLength ++;
            if (cumulativeNameLength > CM_NQ_HOSTNAMESIZE) /* Assume that the size of name is CM_NQ_HOSTNAMESIZE */
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "The supplied DNS string too long");
                result = NQ_FAIL;
                goto Exit;
            }

            syStrcat((NQ_CHAR*)name, ".");
            t++;
        }
        if (!jump)
        {
            cmRpcParseSkip(reader, (NQ_INT)(length + 1));
        }
    }

    if (name)
    {
        *t = '\0';
    }

    if (!jump)
    {
        cmRpcParseSkip(reader, 1);
    }
Exit:
    return result;
}

/* add domain name to a name if it is not FQ */
static void dnsNormalizeName(const NQ_CHAR * host, NQ_CHAR *tmp)
{
    NQ_IPADDRESS ip;    /* dummy */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "host:%p tmp:%p", host, tmp);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (NQ_SUCCESS == cmAsciiToIp((NQ_CHAR *)host + 2, &ip))
    {
        goto Exit;
    }

    syStrncpy(tmp, host, CM_NQ_HOSTNAMESIZE);
    tmp[CM_NQ_HOSTNAMESIZE] = '\0';
    if (!syStrchr(tmp, '.'))
    {
        if (CM_NQ_HOSTNAMESIZE >= syStrlen(tmp) + syStrlen(staticData->dnsDomain) + 1)
        {
            syStrcat(tmp, ".");
            syStrcat(tmp, staticData->dnsDomain);
        }
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/* create a reverse-order ASCII representation of IP address
   a new string is allocated and the caller should free it
*/
NQ_CHAR * nsDnsCreateReversedName(const NQ_IPADDRESS * ip)
{
    NQ_CHAR * buffer;           /* buffer to compose name */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "ip:%p", ip);
    buffer = (NQ_CHAR *)cmMemoryAllocate(CM_DNS_NAMELEN + 1);
    if (NULL == buffer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    buffer[0] = '\0';

    switch (CM_IPADDR_VERSION(*ip))
    {
#ifdef UD_NQ_USETRANSPORTIPV4
    case CM_IPADDR_IPV4:
        {
            NQ_CHAR ipBuffer[CM_IPADDR_MAXLEN]; /* for converting IP into ascii */
            NQ_CHAR * p;                        /* pointer inside */

            cmIpToAscii(ipBuffer, ip);
            for (p = syStrrchr(ipBuffer, '.'); p != NULL ; p = syStrrchr(ipBuffer, '.'))
            {
                syStrcat(buffer, p + 1);
                syStrcat(buffer, ".");
                *p = '\0';
            }
            syStrcat(buffer, ipBuffer);
            syStrcat(buffer, ".in-addr.arpa");
        }
        break;
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
    case CM_IPADDR_IPV6:
        {
            NQ_INT i;               /* octet index in IPv6 */
            NQ_CHAR * p = buffer;   /* pointer inside the buffer */
            const NQ_CHAR hex[] = {'0','1','2','3','4','5','6','7','8','9','a','b','c','d','e','f'};

            for (i = 7; i >= 0; i--)
            {
                *p++ = hex[ip->addr.v6[i] >> 8 & 0xf];
                *p++ = '.';
                *p++ = hex[ip->addr.v6[i] >> 12 & 0xf];
                *p++ = '.';
                *p++ = hex[ip->addr.v6[i] & 0xf];
                *p++ = '.';
                *p++ = hex[ip->addr.v6[i] >> 4 & 0xf];
                *p++ = '.';
            }
            *p = '\0';
            syStrcat(buffer, "ip6.arpa");
        }
        break;
#endif /* UD_NQ_USETRANSPORTIPV6 */
#ifdef UD_NQ_USETRANSPORTIPVR
        case CM_IPADDR_IPVR:
        {
            NQ_IPADDRESS ipvr;

            if (CM_IPADDR_IPV6 == ip->addr.vr.version)
                CM_IPADDR_ASSIGN6(ipvr, ip->addr.vr.addr.v6)
            else
                CM_IPADDR_ASSIGN4(ipvr, ip->addr.vr.addr.v4)

            buffer =  nsDnsCreateReversedName((const NQ_IPADDRESS *)&ipvr);
            goto Exit;
        }
#endif /* UD_NQ_USETRANSPORTIPVR */
    default:
        cmMemoryFree(buffer);
        buffer = NULL;
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", buffer);
    return buffer;
}

NQ_IPADDRESS *  nsDnsParseReversedName(NQ_CHAR * name , NQ_UINT16 type)
{
    NQ_IPADDRESS * resIp = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "name:%p type:%u", name, type);

    resIp = (NQ_IPADDRESS *)cmMemoryAllocate(sizeof(NQ_IPADDRESS));
    if (NULL == resIp)
    {
        goto Exit;
    }

    switch (type)
    {
#ifdef UD_NQ_USETRANSPORTIPV4
    case CM_IPADDR_IPV4:
        {
            NQ_CHAR * p;                        /* pointer inside */
#define IPV4_REVERSED_NAME ".in-addr.arpa"

            p = name + syStrlen(name) - syStrlen(IPV4_REVERSED_NAME);
            if (syStrcmp(p, IPV4_REVERSED_NAME) != 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "IP version 4 reversed name doesn't match");
                cmMemoryFree(resIp);
                resIp = NULL;
                goto Exit;
            }

            syMemset(p , 0 , syStrlen(IPV4_REVERSED_NAME));
            if (cmAsciiToIp(name , resIp) != NQ_SUCCESS)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Wrong IP string");
                cmMemoryFree(resIp);
                resIp = NULL;
                goto Exit;
            }

            CM_IPADDR_ASSIGN4(*resIp, cmChangeByteOrder32(CM_IPADDR_GET4(*resIp)));
        }
        break;
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
    case CM_IPADDR_IPV6:
        {

            NQ_INT i;               /* octet index in IPv6 */
            NQ_CHAR   ipBuffer[CM_DNS_NAMELEN + 1];
            NQ_CHAR * p , * pt;   /* pointer inside the buffer */

            p = syStrchr(name , 'i');
            p--;
            p--;
            pt = ipBuffer;
            syMemset(p , 0 , syStrlen("ip6.arpa"));
            for (i = 7; i >= 0; i--)
            {
                *pt++ = *p;
                p--;
                *pt++ = *p;
                p--;
                *pt++ = *p;
                p--;
                *pt++ = *p;
                p--;
                *pt++ = ':';
            }
            cmAsciiToIp(ipBuffer , resIp);
        }
        break;
#endif /* UD_NQ_USETRANSPORTIPV6 */
    default:
        cmMemoryFree(resIp);
        resIp = NULL;
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", resIp);
    return resIp;
}

static void writeBlock(CMRpcPacketDescriptor * writer, const NQ_CHAR * string, const NQ_BYTE * pad, NQ_COUNT sizeOfPad)
{
    nsDnsEncodeName(writer, string);
    cmRpcPackBytes(writer, pad, sizeOfPad);
}

void nsDnsWriteAnswer(CMRpcPacketDescriptor * writer, const NQ_CHAR * string, NQ_UINT16 type , void * answer , NQ_UINT16 answerLen)
{
    nsDnsEncodeName(writer, string);
    cmRpcPackUint16(writer, type);
    cmRpcPackUint16(writer, 0x01);
    cmRpcPackUint32(writer, 0); /* Time To Live set to 0*/

    switch (type)
    {
        case NS_DNS_A:
        {
            NQ_UINT32 * ipAns;

            cmRpcPackUint16(writer, answerLen);
            ipAns = (NQ_UINT32 *)answer;
            cmBufferWriteUint32(writer, syHton32(*ipAns));
            break;
        }
        case NS_DNS_PTR:
        {
            NQ_COUNT lengthWritten;
            NQ_IOBufPos p1;
            NQ_IOBufPos p2;

            p1 = cmRpcGetPosition(writer, 0, 0);
            cmRpcPackSkip(writer, sizeof(answerLen));
            lengthWritten = nsDnsEncodeName(writer, (const NQ_CHAR *)answer);
            p2 = cmRpcGetPosition(writer, 0, 0);
            cmRpcSetPositionBytePtr(writer, IOBUF_GETBYTEPTR(p1));
            cmRpcPackUint16(writer, (NQ_UINT16)lengthWritten);
            cmRpcSetPositionBytePtr(writer, IOBUF_GETBYTEPTR(p2));

            break;
        }
    }
}

static NQ_COUNT dnsCreateQueryRequest(
    NQ_IOBufPos buffer,
    NQ_BYTE type,
    const NQ_CHAR * name
    )
{
    /* predefined DNS blocks */
    NQ_BYTE query[] = {0, 0, 0, 0x01};
    DnsHeader header;                       /* header */
    CMRpcPacketDescriptor writer;           /* packet writer */
    NQ_COUNT result = 0;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buffer:%p type:0x%x name:%s", buffer, type, name);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    cmRpcSetDescriptor(&writer, buffer, TRUE);
    syMutexTake(&staticData->idGuard);
    header.id = ++staticData->id;
    syMutexGive(&staticData->idGuard);
    header.flags1 = DNS_QUERY;
    header.flags2 = 0;
    header.questions = 1;
    header.answers = 0;
    header.authority = 0;
    header.additional = 0;

    query[1] = type;
    nsWriteDnsHeader(&writer, &header);
    writeBlock(&writer, name, query, sizeof(query));

    result = (NQ_COUNT)(cmRpcGetDataCount(&writer));

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "transaction id: 0x%x", header.id);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS dnsParseQueryResponse(
    const NQ_BYTE * buffer,
    const NQ_BYTE type,
    const NQ_CHAR * host,
    NQ_IPADDRESS * ip,
    NQ_INT * pNumIps,
    NQ_CHAR * name
    )
{
    DnsHeader header;               /* header */
    CMRpcPacketDescriptor reader;   /* for parsing */
    NQ_INT i;                       /* just a counter */
    NQ_INT answer = 0;              /* current answer */
    NQ_COUNT length = 0;            /* string length */
    NQ_COUNT maxIps = 0;            /* maximum room in IPs */
    NQ_STATUS res = NQ_FAIL;        /* operation result */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buffer:%p type:0x%x host:%p ip:%p pNumIps:%p name:%p", buffer, type, host, ip, pNumIps, name);

    if (NULL != pNumIps)
    {
        maxIps = (NQ_COUNT)*pNumIps;
        *pNumIps = 0;
    }
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)buffer, NSDNS_DATAGRAM_BUFSIZE)
    cmRpcSetDescriptor(&reader, bufPos, TRUE);
    nsReadDnsHeader(&reader, &header);

    if (DNS_REPLY_CODE_REFUSED == (header.flags2 & DNS_REPLY_CODE))
    {
        /* force secure exchange */
        LOGERR(CM_TRC_LEVEL_ERROR, "DNS_REPLY_CODE_REFUSED");
        res = NQ_ERR_ACCESS;
        goto Exit;
    }
    if (header.answers < 1 || DNS_REPLY_CODE_NO_SUCH_NAME == (header.flags2 & DNS_REPLY_CODE))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "answers:%u, flags2:%d", header.answers, header.flags2);
        goto Exit;
    }
    for (i = header.questions; --i >= 0;)
    {
        if (nsDnsDecodeName(&reader, NULL) == NQ_FAIL)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "nsDnsDecodeName() failed");
            goto Exit;
        }
        cmRpcParseSkip(&reader, 4);
    }
    
    for (answer = 0; answer < header.answers; answer++)
    {  
        NQ_UINT16 t;                        /* next answer type */
        NQ_UINT16 dataLen;                  /* variable data length in answer */

        if (nsDnsDecodeName(&reader, NULL) == NQ_FAIL)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "nsDnsDecodeName() failed");
            goto Exit;
        }
        cmRpcParseUint16(&reader, &t);
        cmRpcParseSkip(&reader, 6);
        cmRpcParseUint16(&reader, &dataLen);

        if (t == type)
        {
            switch (t)
            {
                case NS_DNS_A:
                    {
                        NQ_IPADDRESS4 p4;    /* temporary pointer */

                        if (!ip)
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "NULL ip");
                            goto Exit;
                        }

                        if (maxIps > 0)
                        {
                            CMRpcPacketDescriptor readerTmp;

                            cmBufferReaderInit(&readerTmp, cmBufferReaderGetPositionBytePtr(&reader, (NQ_COUNT)sizeof(NQ_IPADDRESS4)), (NQ_COUNT)sizeof(NQ_IPADDRESS4));
                            cmBufferReadBytes(&readerTmp, (NQ_BYTE *)&p4, (NQ_COUNT)sizeof(NQ_IPADDRESS4));
                            CM_IPADDR_ASSIGN4(*ip, p4);
                            *pNumIps += 1;
                            maxIps--;
                            ip++;
                            res = NQ_SUCCESS;
                        }

                        cmRpcParseSkip(&reader, dataLen);
                        break;
                    }    
#ifdef UD_NQ_USETRANSPORTIPV6
                case NS_DNS_AAAA:
                    {
                        NQ_IPADDRESS6 * p6;    /* temporary pointer */

                        if (!ip)
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "NULL ip");
                            goto Exit;
                        }
                        if (maxIps > 0)
                        {
                            p6 = (NQ_IPADDRESS6 *)cmBufferReaderGetPositionBytePtr(&reader, sizeof(NQ_IPADDRESS6));
                            CM_IPADDR_ASSIGN6(*ip, *p6);
                            *pNumIps += 1;
                            maxIps--;
                            ip++;
                            res = NQ_SUCCESS;
                        }
                        cmRpcParseSkip(&reader, dataLen);
                        break;
                    }
#endif /* UD_NQ_USETRANSPORTIPV6 */
    
                case NS_DNS_CNAME:
                    {
                        NQ_CHAR str[CM_NQ_HOSTNAMESIZE + 1];    /* host size */

                        if (nsDnsDecodeName(&reader, str) == NQ_FAIL)
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "nsDnsDecodeName() failed");
                            goto Exit;
                        }
                        if (name)
                        {
                            syStrncpy(name, str, CM_NQ_HOSTNAMESIZE);
                            name[CM_NQ_HOSTNAMESIZE] = '\0';
                            res = NQ_SUCCESS;
                            goto Exit;
                        }
                        LOGERR(CM_TRC_LEVEL_ERROR, "NULL name");
                        goto Exit;
                    }
                case NS_DNS_SRV:
                    {
                        NQ_CHAR str[CM_NQ_HOSTNAMESIZE + 1];    /* host size */

                        cmRpcParseSkip(&reader, 6);
                        if (nsDnsDecodeName(&reader, str) == NQ_FAIL)
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "nsDnsDecodeName() failed");
                            goto Exit;
                        }
                        if (name && pNumIps)
                        {   
                            if (*pNumIps == 0)
                            {
                                length = (NQ_COUNT)syStrlen(str);
                                syStrcpy(name, str);
                            }
                            else
                            {                               
                                syStrcpy(&name[length + 1], str);
                                length += (NQ_COUNT)(syStrlen(str) + 1);
                            }
                            *pNumIps += 1;                      
                            res = NQ_SUCCESS;
                        }
                        break;
                    }
                case NS_DNS_PTR:
                    {
                        NQ_CHAR str[CM_NQ_HOSTNAMESIZE + 1];    /* host size */

                        if (nsDnsDecodeName(&reader, str) == NQ_FAIL)
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "nsDnsDecodeName() failed");
                            goto Exit;
                        }
                        if (!name)
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "NULL name");
                            goto Exit;
                        }

                        syStrncpy(name, str, CM_NQ_HOSTNAMESIZE);
                        name[CM_NQ_HOSTNAMESIZE] = '\0';
                        res = NQ_SUCCESS;
                        goto Exit;
                    }
                default:
                    {
                        goto Exit;
                    }
            }
        }
        else
        {
            cmRpcParseSkip(&reader, dataLen);
        }
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d, numIps:%d", res, pNumIps != NULL ? *pNumIps : -1);
    return res;
}

static NQ_STATUS requestByNameAndType(SYSocketHandle socket, const NQ_CHAR * name, NQ_BYTE type, void * context, const NQ_IPADDRESS * serverIp)
{
    NQ_BYTE buffer[NSDNS_DATAGRAM_BUFSIZE];     /* output buffer */
    NQ_UINT length;                             /* outgoing packet length */
    NQ_STATUS result = NQ_SUCCESS;              /* operation result */
    NQ_BOOL isFQDN = FALSE;                     /* whether name is fully qualified */
    NQ_BOOL wasSendAttempt = FALSE;             /* whether send was attempted */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)
    
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket:%d, name:%s, type:0x%x, context:%p, serverIp:%s", socket, name, type, context, cmIPDump(serverIp));

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        result = NQ_ERR_NOTREADY;
        goto Exit;
    }

    if (NULL == name)
    {
        result = NQ_ERR_BADPARAM;
        goto Exit;
    }

    isFQDN = (NULL != syStrchr(name, '.'));

    IOBUF_POSCONSTRUCTOR(bufPos, buffer, sizeof(buffer))
    length = dnsCreateQueryRequest(bufPos, type, name);
#ifdef UD_NQ_USETRANSPORTIPV4
    if (CM_IPADDR_EQUAL(staticData->llmnrIp4, *serverIp))
    {
        if (FALSE == isFQDN)
        {
            wasSendAttempt = TRUE;
            result = sySendMulticast(socket, buffer, length, serverIp, syHton16(LLMNR_PORT));
        }
    }
    else
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
    if (CM_IPADDR_EQUAL(staticData->llmnrIp6, *serverIp))
    {
        if (FALSE == isFQDN)
        {
            wasSendAttempt = TRUE;
            result = sySendToSocket(socket, buffer, length, serverIp, syHton16(LLMNR_PORT));
        }
    }
    else
#endif /* UD_NQ_USETRANSPORTIPV6 */
    {
        wasSendAttempt = TRUE;
        result = sySendToSocket(socket, buffer, length, serverIp, syHton16(DNS_PORT));
    }

    result = (result > 0) ? NQ_SUCCESS : (wasSendAttempt ? NQ_ERR_SOCKETSEND : NQ_ERR_GENERAL);
Exit:
    sySetLastError(result);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS dnsRequestByName(SYSocketHandle socket, const NQ_WCHAR * name, void * context, const NQ_IPADDRESS * serverIp, NQ_COUNT * numOfSentRequests)
{
    NQ_STATUS result = NQ_SUCCESS;      /* operation result */
    NQ_CHAR * nameA = NULL;             /* name as ACSII */
    
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket:%d, name:%s, context:%p, serverIp:%s", socket, cmWDump(name), context, cmIPDump(serverIp));

    *numOfSentRequests = 0;

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        result = NQ_ERR_NOTREADY;
        goto Exit;
    }

    nameA = cmMemoryCloneWStringAsAscii(name);
    if (NULL == nameA)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        result = NQ_ERR_NOMEM;
        goto Exit;
    }
    if (NULL == syStrchr(nameA, '.'))
    {
        NQ_CHAR * qualifiedName;

        /* try to resolve unqualified name as is */
#ifdef UD_NQ_USETRANSPORTIPV4
        result = requestByNameAndType(socket, nameA, NS_DNS_A, context, serverIp);
        if (NQ_SUCCESS != result)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "requestByNameAndType() failed for: %s NS_DNS_A", nameA);
        }
        else
        {
            (*numOfSentRequests)++;
        }
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
        result = requestByNameAndType(socket, nameA, NS_DNS_AAAA, context, serverIp);
        if (NQ_SUCCESS != result)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "requestByNameAndType() failed for: %s NS_DNS_AAAA", nameA);
        }
        else
        {
            (*numOfSentRequests)++;
        }
#endif /* UD_NQ_USETRANSPORTIPV6 */

        /* construct qualified name using default domain */
        qualifiedName = (NQ_CHAR *)cmMemoryAllocate((NQ_UINT)(syStrlen(nameA) + syStrlen(staticData->dnsDomain) + 2));
        if (NULL == qualifiedName)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            result = NQ_ERR_NOMEM;
            goto Exit;
        }
        syStrcpy(qualifiedName, nameA);
        cmMemoryFree(nameA);
        if (syStrlen(staticData->dnsDomain) != 0)
        {
            syStrcat(qualifiedName, ".");
            syStrcat(qualifiedName, staticData->dnsDomain);
        }
        nameA = qualifiedName;
    }

#ifdef UD_NQ_USETRANSPORTIPV4
    result = requestByNameAndType(socket, nameA, NS_DNS_A, context, serverIp);
    if (NQ_SUCCESS != result)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "requestByNameAndType() failed for: %s NS_DNS_A", nameA);
    }
    else
    {
        (*numOfSentRequests)++;
    }
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
    result = requestByNameAndType(socket, nameA, NS_DNS_AAAA, context, serverIp);
    if (NQ_SUCCESS != result)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "requestByNameAndType() failed for: %s NS_DNS_AAAA", nameA);
    }
    else
    {
        (*numOfSentRequests)++;
    }
#endif /* UD_NQ_USETRANSPORTIPV6 */

Exit:
    if (*numOfSentRequests > 0)
    {
        result = NQ_SUCCESS;
    }

    if (NQ_SUCCESS != result)
    {
        sySetLastError(result);
    }

    cmMemoryFree(nameA);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS dnsRequestByNameForDC(SYSocketHandle socket, const NQ_WCHAR * name, void * context, const NQ_IPADDRESS * serverIp, NQ_COUNT * numOfSentRequests)
{
    NQ_CHAR * serviceName = (NQ_CHAR *)"_ldap._tcp.dc._msdcs.";
    NQ_STATUS result = NQ_SUCCESS;          /* operation result */
    NQ_CHAR * serviceA = NULL; /* service */
    NQ_CHAR * nameA = NULL;    /* name as ACSII */
    NQ_CHAR * qualifiedName = NULL;
    
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket:%d, name:%s, context:%p, serverIp:%s", socket, cmWDump(name), context, cmIPDump(serverIp));

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        result = NQ_ERR_NOTREADY;
        goto Exit;
    }

    *numOfSentRequests = 0;
    if (NULL != name)
    {
        nameA = cmMemoryCloneWStringAsAscii(name);
        if (NULL == nameA)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            result = NQ_ERR_NOMEM;
            goto Exit;
        }
    }

    qualifiedName = (NQ_CHAR *)cmMemoryAllocate((NQ_UINT)((nameA ? syStrlen(nameA) : 0) + syStrlen(staticData->dnsDomain) + 2));
    if (NULL == qualifiedName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        result = NQ_ERR_NOMEM;
        goto Exit;
    }

    if (NULL == nameA)
    {
        if (0 != syStrlen(staticData->dnsDomain))
        {
            syStrcat(qualifiedName, staticData->dnsDomain);
        }
        else
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Empty name");
            result = NQ_ERR_BADPATH;
            goto Exit;
        }
    }
    else
    {
        syStrcpy(qualifiedName, nameA);
        /*
        if (NULL == syStrchr(nameA, '.') && (0 != syStrlen(staticData->dnsDomain)) && (NULL != syStrchr(staticData->dnsDomain, '.')))
        {
            syStrcat(qualifiedName, staticData->dnsDomain);
        }
        */
        cmMemoryFree(nameA);
    }

    nameA = qualifiedName;
    qualifiedName = NULL;
    serviceA = (NQ_CHAR *)cmMemoryAllocate((NQ_UINT)(syStrlen(nameA) + syStrlen(serviceName) + 2));
    if (NULL == serviceA)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        result = NQ_ERR_NOMEM;
        goto Exit;
    }
    syStrcpy(serviceA, serviceName);
    syStrcat(serviceA, nameA);

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "service: %s", serviceA);
    result = requestByNameAndType(socket, serviceA, NS_DNS_SRV, context, serverIp);
    if (NQ_SUCCESS == result)
    {
        (*numOfSentRequests)++;
    }

Exit:
    cmMemoryFree(nameA);
    cmMemoryFree(serviceA);
    cmMemoryFree(qualifiedName);
    sySetLastError(result);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS dnsResponseByName(SYSocketHandle socket, NQ_IPADDRESS ** pAddressArray, NQ_INT * numIps, void ** pContext)
{
#define MAX_RESPONSE_IPS 10
    NQ_BYTE buffer[NSDNS_DATAGRAM_BUFSIZE];       /* input buffer */
    NQ_INT count;               /* number of bytes in the incoming datagram */
    NQ_IPADDRESS srcIp;         /* source IP */
    NQ_PORT srcPort;            /* source port */
    NQ_STATUS res;              /* operation result */
    NQ_IPADDRESS result[MAX_RESPONSE_IPS];    /* resulted IPs */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket:%d pAddressArray:%p numIps:%p pContext:%p", socket, pAddressArray, numIps, pContext);

    *numIps = MAX_RESPONSE_IPS;       /* max number of IPs */
    count = syRecvFromSocket(socket, buffer, sizeof(buffer), &srcIp, &srcPort);
    if (count <= 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error receiving DNS response");
        res = (NQ_STATUS)syGetLastError();
        goto Exit;
    }

    res = dnsParseQueryResponse(buffer, NS_DNS_A, NULL, result, numIps, NULL); 
#ifdef UD_NQ_USETRANSPORTIPV6
    if (NQ_SUCCESS != res)
    {
        *numIps = MAX_RESPONSE_IPS;       /* max number of IPs */
        res = dnsParseQueryResponse(buffer, NS_DNS_AAAA, NULL, result, numIps, NULL); 
    }
#endif /* UD_NQ_USETRANSPORTIPV6 */
    if (NQ_SUCCESS != res || *numIps == 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error parsing DNS response");
        goto Exit;
    }
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "resolved %d ips, 1st address: %s", *numIps, cmIPDump(&result[0]));
    
    *pAddressArray = (NQ_IPADDRESS *)cmMemoryAllocate((NQ_UINT)(sizeof(NQ_IPADDRESS) * (NQ_UINT)(*numIps)));
    if (NULL == *pAddressArray)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_NOMEM;
        goto Exit;
    }
    syMemcpy(*pAddressArray, result, sizeof(NQ_IPADDRESS) * (NQ_UINT)(*numIps));
    res = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", res);
    return res;
}

static NQ_STATUS dnsRequestByIp(SYSocketHandle socket, const NQ_IPADDRESS * ip, void * context, const NQ_IPADDRESS * serverIp, NQ_COUNT * numOfSentRequests)
{
    NQ_BYTE buffer[NSDNS_DATAGRAM_BUFSIZE];     /* output buffer */
    NQ_UINT length;                             /* outgoing packet length */
    NQ_STATUS result = NQ_SUCCESS;              /* operation result */
    NQ_CHAR * ipA;                              /* IP as ACSII */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)
    
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket:%d ip:%p context:%p serverIp:%p", socket, ip, context, serverIp);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        result = NQ_ERR_NOTREADY;
        goto Exit;
    }

    *numOfSentRequests = 0;
    ipA = nsDnsCreateReversedName(ip);
    if (NULL == ipA)
    {
        result = NQ_ERR_NOMEM;
        goto Exit;
    }

    IOBUF_POSCONSTRUCTOR(bufPos, buffer, sizeof(buffer))
    length = dnsCreateQueryRequest(bufPos, NS_DNS_PTR, ipA);
    cmMemoryFree(ipA);
#ifdef UD_NQ_USETRANSPORTIPV4
    if (CM_IPADDR_EQUAL(staticData->llmnrIp4, *serverIp))
    {
        result = sySendMulticast(socket, buffer, length, serverIp, syHton16(LLMNR_PORT));
    }
    else 
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
    if (CM_IPADDR_EQUAL(staticData->llmnrIp6, *serverIp))
    {   
        result = sySendToSocket(socket, buffer, length, serverIp, syHton16(LLMNR_PORT));
    }
    else
#endif /* UD_NQ_USETRANSPORTIPV6 */
    {
        result = sySendToSocket(socket, buffer, length, serverIp, syHton16(DNS_PORT));
    }

    if (result > 0)
    {
        (*numOfSentRequests)++;
        result = NQ_SUCCESS;
    }
    else
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Send failed");
        result = NQ_ERR_SOCKETSEND;
    }

Exit:
    sySetLastError(result);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", result);
    return result;
}

static NQ_STATUS dnsResponseByNameForDC(SYSocketHandle socket, const NQ_WCHAR ** pName, void ** pContex)
{
    NQ_BYTE buffer[1024];                  /* input buffer */
    NQ_INT count;                          /* number of bytes in the incoming datagram */
    NQ_IPADDRESS srcIp;                    /* source IP */
    NQ_PORT srcPort;                       /* source port */
    NQ_CHAR *pDcList = NULL, *pa = NULL;   /* result in ASCII */
    NQ_WCHAR *pDcListW = NULL, *pw = NULL; /* result in WCHAR */
    NQ_INT numDcs = 0;                     /* result number of parsed DC servers */
    NQ_STATUS res = NQ_FAIL;               /* operation result */
    NQ_INT *num;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket:%d pName:%p pContex:%p", socket, pName, pContex);

    count = syRecvFromSocket(socket, buffer, sizeof(buffer), &srcIp, &srcPort);
    if (count <= 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error receiving DNS response");
        res = (NQ_STATUS)syGetLastError();
        goto Exit;
    }

    /* allocate max space for DNS names as max UDP DNS packet length */
    pDcList = (NQ_CHAR *)cmMemoryAllocate(1024);
    if (NULL == pDcList)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_NOMEM;
        goto Exit;
    }
    pDcList[0] = '\0';
    res = dnsParseQueryResponse(buffer, NS_DNS_SRV, NULL, 0, &numDcs, pDcList);  
    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error parsing DNS response");
        goto Exit;
    }

    pDcListW = (NQ_WCHAR *)cmMemoryAllocate((NQ_UINT)numDcs * (NQ_UINT)sizeof(NQ_WCHAR) * (CM_DNS_NAMELEN + 1));
    if (NULL == pDcListW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_NOMEM;
        goto Exit;
    }

    pDcListW[0] = cmWChar('\0');
    for (pw = pDcListW, pa = pDcList, count = 0; count < numDcs; count++)
    {
        NQ_UINT lenW = 0;
        NQ_WCHAR *p = cmMemoryCloneAString(pa);
        
        if (NULL != p)
        {
            lenW = cmWStrlen(p);
            cmWStrcpy(pw, p);
            pw[lenW] = cmWChar('\0');
            cmMemoryFree(p);
        }
        pw += lenW + 1;
        pa += syStrlen(pa) + 1;        
    }
    *pName = pDcListW;
    *pContex = num = (NQ_INT *)cmMemoryAllocate(sizeof(numDcs));
    if (NULL == num)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_NOMEM;
        goto Exit;
    }
    *num = numDcs;
    res = NQ_SUCCESS;

Exit:
    cmMemoryFree(pDcList);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", res);
    return res;
}

static NQ_STATUS dnsResponseByIp(SYSocketHandle socket, const NQ_WCHAR ** pName, void ** pContex)
{
    NQ_BYTE buffer[1024];    /* input buffer */
    NQ_INT count;            /* number of bytes in the incoming datagram */
    NQ_IPADDRESS srcIp;      /* source IP */
    NQ_PORT srcPort;         /* source port */
    NQ_CHAR * pNameA = NULL; /* result in ASCII */
    NQ_STATUS res;           /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket:%d pName:%p pContex:%p", socket, pName, pContex);

    count = syRecvFromSocket(socket, buffer, sizeof(buffer), &srcIp, &srcPort);
    if (count <= 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error receiving DNS response");
        res = (NQ_STATUS)syGetLastError();
        goto Exit;
    }

    pNameA = (NQ_CHAR *)cmMemoryAllocate(CM_DNS_NAMELEN + 1);
    if (NULL == pNameA)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_NOMEM;
        goto Exit;
    }
    res = dnsParseQueryResponse(buffer, NS_DNS_PTR, NULL, 0, NULL, pNameA); 
    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error parsing DNS response");
        goto Exit;
    }
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "resolved name: %s", pNameA);

    *pName = cmMemoryCloneAString(pNameA);
    res = ( NULL == *pName? NQ_ERR_NOMEM : NQ_SUCCESS );

Exit:
    cmMemoryFree(pNameA);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", res);
    return res;
}

/*
 *====================================================================
 * PURPOSE: Register a list of servers as a new resolver methods
 *--------------------------------------------------------------------
 * PARAMS:  list - semicolon delimited list of DNS servers
 *
 * RETURNS: NONE
 *====================================================================
 */
static void registerServerList(NQ_WCHAR * list)
{
    CMResolverMethodDescriptor  method;                                 /* next method descriptor */
    CMResolverMethodDescriptor  methodDC;                               /* next method descriptor */
    NQ_WCHAR                    *curServer = NULL;                      /* pointer to the current server IP */
    NQ_CHAR                     aServer[CM_IPADDR_MAXLEN];              /* the same in ASCII */
    NQ_HANDLE                   parseHandle;                            /* handle to parse servers list */
    NQ_WCHAR                    delimiter = cmWChar(CM_NQ_DELIMITER);   /* delimiter */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "list:%s", list ? cmWDump(list) : "");

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit1;
    }

    method.type = NQ_RESOLVER_DNS;
    method.isMulticast = FALSE;     /* unicast */
    method.activationPriority = 2;
    method.timeout.low = 1000;          /* milliseconds */
    method.timeout.high = 0;            /* milliseconds */
    method.waitAnyway = TRUE;
    method.requestByName = dnsRequestByName;
    method.responseByName = dnsResponseByName;
    method.requestByIp = dnsRequestByIp;
    method.responseByIp = dnsResponseByIp;

    methodDC.type = NQ_RESOLVER_DNS_DC;
    methodDC.isMulticast = FALSE;   /* unicast */
    methodDC.activationPriority = 2;
    methodDC.timeout.low = 1000;        /* milliseconds */
    methodDC.timeout.high = 0;          /* milliseconds */
    methodDC.waitAnyway = FALSE;
    methodDC.requestByName = dnsRequestByNameForDC;
    methodDC.responseByName = NULL;
    methodDC.requestByIp = NULL;
    methodDC.responseByIp = dnsResponseByNameForDC;

    syMutexTake(&staticData->dataGuard);

    /* parse servers string */
    staticData->numServers = 0;
    parseHandle = cmSpStartParsing((void *)list, (void *)&delimiter, TRUE, (void *)&curServer, TRUE);
    if (NULL == parseHandle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to start parsing");
        goto Exit2;
    }

    while((staticData->numServers < sizeof(staticData->dnsServers) / sizeof(staticData->dnsServers[0])) && (NULL != curServer))
    {
        NQ_STATUS res;                      /* operation status */

        if (cmWStrlen(curServer) < CM_IPADDR_MAXLEN)
        {
            cmUnicodeToAnsiN(aServer, sizeof(aServer), curServer, CM_IPADDR_MAXLEN * sizeof(NQ_WCHAR));
            res = cmAsciiToIp(aServer, &staticData->dnsServers[staticData->numServers]);
            if (NQ_SUCCESS == res)
            {
                /* register DNS with Resolver */
                cmResolverRemoveMethod(&method, &staticData->dnsServers[staticData->numServers]);
                cmResolverRegisterMethod(&method, &staticData->dnsServers[staticData->numServers]);
                /* register DNS DC with Resolver */
                cmResolverRemoveMethod(&methodDC, &staticData->dnsServers[staticData->numServers]);
                cmResolverRegisterMethod(&methodDC, &staticData->dnsServers[staticData->numServers]);

                staticData->numServers++;
            }
        }

        curServer = cmSpGetNextString(parseHandle);
    }

    cmSpTerminateParsing(parseHandle);

Exit2:
    syMutexGive(&staticData->dataGuard);
Exit1:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return;
}

static void dnsGetServers(void)
{
    NQ_WCHAR * serverList = NULL;                          /* buffer for DNS servers string */
    NQ_WCHAR * serverListW = NULL;                          /* buffer for DNS servers string in Unicode */
    NQ_WCHAR * domain = NULL;                              /* buffer for server name */
    CMResolverMethodDescriptor method;                      /* next method descriptor */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (staticData->isNewServerSet)
    {
        goto Exit;
    }

    serverList = (NQ_WCHAR *)cmMemoryAllocate(UD_DNS_SERVERSTRINGSIZE * sizeof(NQ_WCHAR));
    domain = (NQ_WCHAR *)cmMemoryAllocate((CM_NQ_HOSTNAMESIZE + 1) * sizeof(NQ_WCHAR));

    if (NULL == serverList || NULL == domain)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }

    udGetDnsParams(domain, serverList);
    cmUnicodeToAnsi(staticData->dnsDomain, domain);

    serverListW = cmMemoryCloneWString(serverList);
    if (NULL == serverListW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    registerServerList(serverListW);

    method.type = NQ_RESOLVER_DNS;
    method.isMulticast = TRUE;
    method.activationPriority = 4;
    method.timeout.low = 1000; /* milliseconds */
    method.timeout.high = 0;   /* milliseconds */
    method.waitAnyway = TRUE;
    method.requestByName = dnsRequestByName;
    method.responseByName = dnsResponseByName;
    method.requestByIp = dnsRequestByIp;
    method.responseByIp = dnsResponseByIp;
#ifdef UD_NQ_USETRANSPORTIPV4
    cmAsciiToIp((NQ_CHAR *)"224.0.0.252", &staticData->llmnrIp4);
    cmResolverRemoveMethod(&method, &staticData->llmnrIp4);
    cmResolverRegisterMethod(&method, &staticData->llmnrIp4);
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
    cmAsciiToIp("FF02:0:0:0:0:0:1:3", &staticData->llmnrIp6);
    cmResolverRemoveMethod(&method, &staticData->llmnrIp6);
    cmResolverRegisterMethod(&method, &staticData->llmnrIp6);
#endif /* UD_NQ_USETRANSPORTIPV6 */

Exit:
    cmMemoryFree(serverListW);  /* takes care of NULL */
    cmMemoryFree(domain);       /* take care of NULL */
    cmMemoryFree(serverList);   /* takes care of NULL */
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/* send and receive DNS datagram */
static NQ_INT dnsDatagramExchange(
    NQ_IPADDRESS * serverIp,
    NQ_BYTE * buffer,
    NQ_COUNT dataLength,
    NQ_COUNT bufferSize
    )
{
    SYSocketHandle socket;      /* socket */
    SYSocketSet set;            /* socket set */
    NQ_IPADDRESS tip;           /* dummy */
    NQ_PORT tport;              /* dummy */
    NQ_INT res;                 /* function result */
    NQ_INT result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "serverIp:%p buffer:%p dataLength:%d bufferSize:%d", serverIp, buffer, dataLength, bufferSize);

    socket = syCreateSocket(FALSE, CM_IPADDR_VERSION(*serverIp));
    if (!syIsValidSocket(socket))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error creating DNS socket");
        goto Exit;
    }

    if (sySendToSocket(socket, buffer, dataLength, serverIp, syHton16(DNS_PORT)) == NQ_FAIL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error sending DNS request");
        goto Error;
    }

    syClearSocketSet(&set);
    syAddSocketToSet(socket, &set);

    switch (sySelectSocket(&set, DNS_TIMEOUT))
    {
    case 0:  /* timeout */
        LOGERR(CM_TRC_LEVEL_ERROR, "DNS Timeout occured");
        goto Error;
    case -1: /* error or exit */
        LOGERR(CM_TRC_LEVEL_ERROR, "DNS Error occured");
        goto Error;
    };

    res = syRecvFromSocket(socket, buffer, bufferSize, &tip, &tport);
    if (res <= 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "DNS Error occured");
        goto Error;
    };
    result = res;

Error:
    syCloseSocket(socket);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

#if defined(UD_NQ_INCLUDECIFSCLIENT) && defined(UD_CC_INCLUDEEXTENDEDSECURITY)

/* create a unique TKEY name */
#ifndef UD_CM_DONOTREGISTERHOSTNAMEDNS

static void createTkeyName(NQ_CHAR * name)
{
    NQ_IPADDRESS ip;        /* show id as IP */
    NQ_IPADDRESS4 ip4;      /* show id as IP */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "name:%s", name);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    /*  Tkey name has the format:
            vs-nq-<n>-<time>
        <n> is ID number modulo 10.
        <time> is current Unix time in secs, formatted as IP address.
        Why IP address? just for fun and since
        we have an appopriate function in hand */
    syMutexTake(&staticData->dataGuard);
    syStrcpy (name, "vs-nq-x-");
    name[6] = (NQ_CHAR)('0' + (staticData->id%10));
    ip4 = (NQ_IPADDRESS4)syGetTimeInSec();
    CM_IPADDR_ASSIGN4(ip, ip4);
    cmIpToAscii(name + 8, &ip);
    syMutexGive(&staticData->dataGuard);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static NQ_STATUS dnsQueryTkey(void * context, const CMBlob * in, CMBlob * out)
{
    NQ_BYTE buffer[NSDNS_DATAGRAM_BUFSIZE];               /* for header + queries */
    CMRpcPacketDescriptor descr;        /* packer and parser */
    DnsHeader header;                   /* header */
    NQ_STATUS status;                   /* operation result */
    SYSocketSet set;                    /* socket set */
    NQ_INT count;                       /* receive count */
    NQ_BYTE tempByte;                   /* for parsing byte values */
    NQ_UINT16 temp16;                   /* for parsing two-byte values */
    TkeyContext * pTkey;                /* casted pointer */
    NQ_STATUS result = NQ_FAIL;         /* return value */
    const NQ_BYTE queryData[] = {0, 0xf9, 0, 0x01};
    const NQ_BYTE additionalData[] = {0, 0xf9, 0, 0xff, 0, 0, 0, 0};
    static NQ_BYTE other[] = {0, 0};
    NQ_UINT16 packetSize;
    NQ_IOBufPos tmpBuf;
#ifdef UD_NQ_USEIOVECS
    NQ_IOBufState bufState;
    NQ_IOBufOrigin bufOrig;
#endif
    NQ_IOBuf sendBuf;
    IOBUF_POSCONSTRUCTORINIT(tmpBuf)
#define ALGORITHM_NAME "gss-tsig"
#define TKEY_EXTRALEN 26
#define SECS_IN_DAY (60 * 60 * 24)
#define MODE_GSSAPI 3

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "context:%p in:%p out:%p", context, in, out);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    pTkey = (TkeyContext *)context;

    /* compose and transmit DNS header + queries */
    IOBUF_POSCONSTRUCTOR(tmpBuf, buffer, sizeof(buffer))
    cmRpcSetDescriptor(&descr, tmpBuf, TRUE);  /* nbo */
    cmRpcPackSkip(&descr, sizeof(NQ_UINT16));  /* skip packet length */
    syMutexTake(&staticData->idGuard);
    header.id = ++staticData->id;
    syMutexGive(&staticData->idGuard);
    header.flags1 = 0;
    header.flags2 = 0;
    header.questions = 1;
    header.answers = 0;
    header.authority = 0;
    header.additional = 1;
    nsWriteDnsHeader(&descr, &header);
    writeBlock(&descr, pTkey->name, queryData, sizeof(queryData));
    writeBlock(&descr, pTkey->name, additionalData, sizeof(additionalData));
    cmRpcPackUint16(&descr, (NQ_UINT16)(in->len + TKEY_EXTRALEN));           /* data length */
    cmRpcPackByte(&descr, sizeof(ALGORITHM_NAME) - 1);          /* algorithm as size-prefixed null-terminated */
    cmRpcPackAscii(&descr, ALGORITHM_NAME, CM_RP_NULLTERM);     /* algorithm */
    cmRpcPackUint32(&descr, (NQ_UINT32)syGetTimeInSec());                       /* signature creation */
    cmRpcPackUint32(&descr, (NQ_UINT32)(syGetTimeInSec() + SECS_IN_DAY));         /* signature expiration */
    cmRpcPackUint16(&descr, MODE_GSSAPI);                       /* mode */
    cmRpcPackUint16(&descr, 0);                                 /* no error */
    cmRpcPackUint16(&descr, (NQ_UINT16)in->len);                /* key size */
    /* get total packet size */
    packetSize = (NQ_UINT16)cmRpcGetDataCount(&descr);
    /* move back and write packet size */
    cmRpcResetDescriptor(&descr);
    cmRpcPackUint16(&descr, (NQ_UINT16)((packetSize - 2) + (NQ_UINT16)in->len + (NQ_UINT16)sizeof(other)));   /* packet length */
    sendBuf = IOBUF_GETBUFPTR(tmpBuf);
    IOBUF_PREPARELENGTH(sendBuf, bufOrig, bufState, packetSize)
    status = sySendSocket(pTkey->socket, sendBuf, (NQ_UINT)(packetSize));
    IOBUF_IOVECBUF_RESTORE(sendBuf, bufState, bufOrig)
    if (NQ_FAIL == status)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to transmit TKEY query");
        goto Exit;
    }

    /* send TKEY payload */
    IOBUF_POSCONSTRUCTOR(tmpBuf, in->data, in->len)
    sendBuf = IOBUF_GETBUFPTR(tmpBuf);
    IOBUF_PREPARELENGTH(sendBuf, bufOrig, bufState, in->len)
    status = sySendSocket(pTkey->socket, sendBuf, (NQ_UINT)in->len);
    IOBUF_IOVECBUF_RESTORE(sendBuf, bufState, bufOrig)
    if (NQ_FAIL == status)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to transmit TKEY query");
        goto Exit;
    }

    /* send "other" */
    IOBUF_POSCONSTRUCTOR(tmpBuf, other, sizeof(other))
    sendBuf = IOBUF_GETBUFPTR(tmpBuf);
    IOBUF_PREPARELENGTH(sendBuf, bufOrig, bufState, sizeof(other))
    status = sySendSocket(pTkey->socket, sendBuf, sizeof(other));
    IOBUF_IOVECBUF_RESTORE(sendBuf, bufState, bufOrig)
    if (NQ_FAIL == status)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to transmit TKEY query");
        goto Exit;
    }

    /* receive response */
    syClearSocketSet(&set);
    syAddSocketToSet(pTkey->socket, &set);
    switch (sySelectSocket(&set, DNS_TIMEOUT))
    {
        case 0:  /* timeout */
            LOGMSG(CM_TRC_LEVEL_ERROR, "DNS Timeout occurred for TKey request");
            goto Exit;

        case -1: /* error or exit */
            LOGERR(CM_TRC_LEVEL_ERROR, "DNS Error occurred for TKey request");
            goto Exit;
    };

    IOBUF_POSCONSTRUCTOR(tmpBuf, buffer, sizeof(buffer))
    sendBuf = IOBUF_GETBUFPTR(tmpBuf);
    IOBUF_PREPARELENGTH(sendBuf, bufOrig, bufState, sizeof(buffer))
    count = syRecvSocket(pTkey->socket, sendBuf, sizeof(buffer));
    IOBUF_IOVECBUF_RESTORE(sendBuf, bufState, bufOrig)
    if (count <= 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error receiving DNS response");
        goto Exit;
    }

    /* skip header and queries */
    IOBUF_POSCONSTRUCTOR(tmpBuf, buffer, sizeof(buffer))
    cmRpcSetDescriptor(&descr, tmpBuf, TRUE);       /* nbo */
    cmRpcParseSkip(&descr, sizeof(NQ_UINT16));      /* packet length */
    nsReadDnsHeader(&descr, &header);
    if ((header.flags2 & 0xF) != 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error DNS response 0x%x", header.flags2);
        goto Exit;
    }
    
    if (header.answers == 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "No answers in response");
        goto Exit;
    }

    if (nsDnsDecodeName(&descr, NULL) == NQ_FAIL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsDnsDecodeName() failed");
        goto Exit;
    }
    cmRpcParseSkip(&descr, 4);

    /* parse answer */
    if (nsDnsDecodeName(&descr, NULL) == NQ_FAIL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsDnsDecodeName() failed");
        goto Exit;
    }
    cmRpcParseSkip(&descr, 10);            /* type, class, TTL, data length */
    cmRpcParseByte(&descr, &tempByte);     /* algorithm name length */
    cmRpcParseSkip(&descr, (NQ_INT32)(tempByte + 1));  /* algorithm name */
    cmRpcParseSkip(&descr, 10);            /* times, mode */
    cmRpcParseUint16(&descr, &temp16);     /* error */
    if (0 != temp16)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Server failed the request");
        goto Exit;
    }
    cmRpcParseUint16(&descr, &temp16);     /* key size */
    out->data = (NQ_BYTE *)cmMemoryAllocate(temp16);
    if (NULL == out->data)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    out->len = temp16;
    cmRpcParseBytes(&descr, out->data, out->len);     /* key */
    cmRpcParseSkip(&descr, 2);                        /* other size */

    /* parse additional record */
    count = count - (NQ_INT)cmRpcGetDataCount(&descr);   /* tkey length */
    pTkey->tkey.data = (NQ_BYTE *)cmMemoryAllocate((NQ_COUNT)count);
    if (NULL == pTkey->tkey.data)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    pTkey->tkey.len = (NQ_COUNT)count;
    cmRpcParseBytes(&descr, pTkey->tkey.data, pTkey->tkey.len);     /* tkey */
    pTkey->originalIdOffset = (NQ_INT)(pTkey->tkey.len - 6);
    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS dnsStartTkey(TkeyContext * pTkey, const NQ_IPADDRESS * serverIp)
{
    NQ_INT ipVersion;               /* v4 or v6 */
    NQ_STATUS status = NQ_FAIL;     /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pTkey:%p serverIp:%p", pTkey, serverIp);

    pTkey->sessionKey.data = NULL;
    pTkey->macKey.data = NULL;
    pTkey->ip = serverIp;
    pTkey->tkey.data = NULL;

    /* create and connect socket */
    ipVersion = CM_IPADDR_VERSION(*serverIp);
    pTkey->socket = syCreateSocket(TRUE, (NQ_UINT)ipVersion);
    if (!syIsValidSocket(pTkey->socket))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to create TCP socket");
        goto Exit;
    }
    status = syConnectSocket(pTkey->socket, serverIp, syHton16(DNS_PORT));
    if (NQ_SUCCESS != status)
    {
        syCloseSocket(pTkey->socket);
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to connect to DNS server over TCP");
        goto Exit;
    }
    createTkeyName(pTkey->name);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", status);
    return status;
}

static void dnsFreeTkeyContext(TkeyContext * pTkey)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pTkey:%p", pTkey);

    cmMemoryFreeBlob(&pTkey->tkey);
    cmMemoryFreeBlob(&pTkey->sessionKey);
    cmMemoryFreeBlob(&pTkey->macKey);

    if (syIsSocketAlive(pTkey->socket))
    {
        syCloseSocket(pTkey->socket);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return;
}

#endif /* UD_CM_DONOTREGISTERHOSTNAMEDNS */
#endif /* defined(UD_NQ_INCLUDECIFSCLIENT) && defined(UD_CC_INCLUDEEXTENDEDSECURITY) */

static NQ_STATUS dnsQueryService(
    const NQ_CHAR * service,
    NQ_CHAR * host
    )
{
    NQ_COUNT i;                     /* index in DNS servers */
    NQ_STATUS result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "service:%p host:%p", service, host);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    /* try request with each of the DNS servers until one of them succeed */
    dnsGetServers();

    for (i = 0; i < staticData->numServers; i++)
    {
        NQ_BYTE buffer[NSDNS_DATAGRAM_BUFSIZE];             /* send/receive buffer */
        DnsHeader header;                                   /* header */
        CMRpcPacketDescriptor descr;                        /* packet writer */
        NQ_INT res;                                         /* operation result */
        const NQ_BYTE query[] = {0, NS_DNS_SRV, 0, 0x01};
        NQ_INT answer;                                      /* current answer */
        NQ_IOBufPos bufPos;
        IOBUF_POSCONSTRUCTORINIT(bufPos)
        IOBUF_POSCONSTRUCTOR(bufPos, buffer, sizeof(buffer))

        cmRpcSetDescriptor(&descr, bufPos, TRUE);
        syMutexTake(&staticData->idGuard);
        header.id = ++staticData->id;
        syMutexGive(&staticData->idGuard);
        header.flags1 = DNS_QUERY;
        header.flags2 = 0;
        header.questions = 1;
        header.answers = 0;
        header.authority = 0;
        header.additional = 0;
        nsWriteDnsHeader(&descr, &header);
        writeBlock(&descr, service, query, sizeof(query));
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "transaction id: 0x%x", header.id);
        res = dnsDatagramExchange(&staticData->dnsServers[i], buffer, cmRpcGetDataCount(&descr), sizeof(buffer));
        if (res <= 0)
        {
            continue;
        }

        IOBUF_POSCONSTRUCTOR(bufPos, buffer, sizeof(buffer))
        cmRpcSetDescriptor(&descr, bufPos, TRUE);
        nsReadDnsHeader(&descr, &header);
        if (header.answers < 1 || DNS_REPLY_CODE_NO_SUCH_NAME == (header.flags2 & DNS_REPLY_CODE))
        {
            continue;
        }
        if (nsDnsDecodeName(&descr, NULL) == NQ_FAIL)
        {
            continue;
        }
        cmRpcParseSkip(&descr, 4);

        for (answer = 0; answer < header.answers; answer++)
        {
            NQ_UINT16 t;                          /* next answer type */
            NQ_UINT16 dataLen;                  /* variable data length in answer */

            if (nsDnsDecodeName(&descr, NULL) == NQ_FAIL)
            {
                break;
            }
            cmRpcParseUint16(&descr, &t);
            cmRpcParseSkip(&descr, 6);
            cmRpcParseUint16(&descr, &dataLen);
            switch (t)
            {
                case NS_DNS_CNAME:
                case NS_DNS_SRV:
                case NS_DNS_PTR:
                    {
                        NQ_CHAR str[CM_NQ_HOSTNAMESIZE + 1];    /* redirection target */

                        cmRpcParseSkip(&descr, 6);
                        if (nsDnsDecodeName(&descr, str) == NQ_FAIL)
                        {
                            goto Exit;
                        }
                        syStrncpy(host, str, CM_NQ_HOSTNAMESIZE);
                        host[CM_NQ_HOSTNAMESIZE] = '\0';
                        result = NQ_SUCCESS;
                        break;
                    }
                default:
                    {
                        goto Exit;
                    }
            }
        }
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

#ifndef UD_CM_DONOTREGISTERHOSTNAMEDNS
/* update on one server with optional TKEY */
static NQ_STATUS dnsUpdateOneServer(
    NQ_UINT16 type,
    NQ_IPADDRESS * serverIp,
    NQ_CHAR * host,
    NQ_IPADDRESS *hostIPs,
    NQ_UINT16 numHostIPs,
    const TkeyContext * pTkey
    )
{
    NQ_BYTE buffer[NSDNS_DATAGRAM_BUFSIZE];   /* send and receive buffer */
    NQ_INT length;         /* data length */
    const NQ_BYTE zone[]    = {0, NS_DNS_SOA,   0, 0x01};
    const NQ_BYTE prereq[]  = {0, NS_DNS_CNAME, 0, 0xfe, 0, 0, 0, 0, 0, 0};
    NQ_BYTE update1[] = {0, 0, 0, 0xff, 0, 0, 0, 0, 0, 0};
    NQ_BYTE update2[] = {0, 0, 0, 0x01, 0, 0, 0x03, 0x84};
    DnsHeader header;                       /* header */
    NQ_CHAR * domain;                       /* pointer to domain inside FQN */
    NQ_COUNT i;                             /* just a counter */
    CMRpcPacketDescriptor descr;           /* packet reader/writer */
    NQ_STATUS result = NQ_FAIL;
    NQ_IOBufPos tmpBuf;
    IOBUF_POSCONSTRUCTORINIT(tmpBuf)

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "type:%u serverIp:%p host:%p hostIPs:%p numHostIPs:%u pTkey:%p", type, serverIp, host, hostIPs, numHostIPs, pTkey);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    /* compose request */
    IOBUF_POSCONSTRUCTOR(tmpBuf, buffer, sizeof(buffer))
    cmRpcSetDescriptor(&descr, tmpBuf, TRUE);
    descr.length = sizeof(buffer);
    syMutexTake(&staticData->idGuard);
    header.id = ++staticData->id;
    syMutexGive(&staticData->idGuard);
    header.flags1 = DNS_UPDATE;
    header.flags2 = 0;
    header.questions = 1;  
    header.answers   = 1;          /* zones and prerequisites */
    header.authority = (NQ_UINT16)(numHostIPs + 1);
    header.additional = NULL == pTkey? 0 : 1;
    nsWriteDnsHeader(&descr, &header);
    domain = syStrchr(host, '.');
    if (NULL == domain)
    {
        goto Exit;
    }
    domain++;
    writeBlock(&descr, domain, zone, sizeof(zone));
    writeBlock(&descr, host, prereq, sizeof(prereq));
    update1[1] = (NQ_BYTE)type;
    writeBlock(&descr, host, update1, sizeof(update1)); /* class ANY */
    if (NULL != hostIPs)
    {
        for (i = 0; i < numHostIPs; i++)
        {
            NQ_UINT byteToWriteForNextIP;

            update2[1] = (NQ_BYTE)type;
            byteToWriteForNextIP = (NQ_UINT)(syStrlen(host) + sizeof(update2) + CM_IPADDR_SIZE(hostIPs[i]) + 4); /* additional 4 bytes - 2 bytes of padding zeros and another 2 for length */
            if (NULL != pTkey)
            {
                byteToWriteForNextIP += pTkey->tkey.len;
            }

            if (byteToWriteForNextIP > cmRpcGetRemaining(&descr))
            {
                NQ_IOBufPos savedPosition;
                NQ_IOBufPos authorityPosition;

                if (0 == i)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Insufficient buffer space, required: %u, left: %u", byteToWriteForNextIP, cmRpcGetRemaining(&descr));
                    goto Exit;
                }

                cmRpcGetPosition(&descr, 0, &savedPosition);
                cmRpcGetStartPosition(&descr, 8, &authorityPosition); /* authority (part of DNS header) located 8 bytes from buffer beginning */
                cmRpcSetPosition(&descr, authorityPosition);
                cmRpcPackUint16(&descr, (NQ_UINT16)(i));
                cmRpcSetPosition(&descr, savedPosition);
                LOGERR(CM_TRC_LEVEL_ERROR, "Insufficient buffer space, required: %u, left: %u", byteToWriteForNextIP, cmRpcGetRemaining(&descr));
                break;
            }

            writeBlock(&descr, host, update2, sizeof(update2)); /* class IN */
            cmRpcPackByte(&descr, 0);
            cmRpcPackByte(&descr, CM_IPADDR_SIZE(hostIPs[i]));    /* data length */            
            switch (type)
            {
#ifdef UD_NQ_USETRANSPORTIPV4
                case NS_DNS_A:
                {
                    NQ_IPADDRESS4 ip4 = CM_IPADDR_GET4(hostIPs[i]);
                    cmRpcPackUint32(&descr, syNtoh32(ip4));
                    break;
                }
#endif /* UD_NQ_USETRANSPORTIPV4 */

#ifdef UD_NQ_USETRANSPORTIPV6
                case NS_DNS_AAAA:
                {
                    cmRpcPackBytes(&descr, (NQ_BYTE *)CM_IPADDR_GET6(hostIPs[i]), CM_IPADDR_SIZE(hostIPs[i]));
                    break;
                }
#endif /* UD_NQ_USETRANSPORTIPV6 */

                default:
                {
                    goto Exit;
                }
            }
        }
    }

    if (NULL != pTkey)
    {
        setTkeyOriginalId(pTkey, header.id); 
        cmRpcPackBytes(&descr, pTkey->tkey.data, pTkey->tkey.len);       /* TKEY record */
    }

    length = (NQ_INT)cmRpcGetDataCount(&descr);
    /* send request and receive response */
    length = dnsDatagramExchange(serverIp, buffer, (NQ_COUNT)length, sizeof(buffer));
    if (length <= 0)
    {
        goto Exit;
    }

    IOBUF_POSCONSTRUCTOR(tmpBuf, buffer, sizeof(buffer))
    cmRpcSetDescriptor(&descr, tmpBuf, TRUE);
    nsReadDnsHeader(&descr, &header);
    if (DNS_REPLY_CODE_REFUSED == (header.flags2 & DNS_REPLY_CODE))
    {
        /* force secure exchange */ 
        result = DNS_REPLY_CODE_REFUSED;
        goto Exit;
    }

    if (0 != (header.flags2 & DNS_REPLY_CODE))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "flags2:%d", header.flags2);
        goto Exit;
    }

    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}
#endif /* UD_CM_DONOTREGISTERHOSTNAMEDNS */

/* perform on all servers, forcing TKey query if needed */ 
static NQ_STATUS dnsUpdate(
    NQ_UINT16 type,
    NQ_IPADDRESS *hostIPs,
    NQ_UINT16 numHostIPs
    )
{
    NQ_STATUS result = NQ_FAIL;             /* return value */
#ifndef UD_CM_DONOTREGISTERHOSTNAMEDNS
    NQ_COUNT i;                             /* index in DNS servers */
    NQ_CHAR host[CM_NQ_HOSTNAMESIZE + 1];   /* self host name */
    NQ_STATUS res = NQ_FAIL;                /* update status */
    NQ_STATUS status = NQ_FAIL;             /* operation result */
#endif /* UD_CM_DONOTREGISTERHOSTNAMEDNS */
#if defined(UD_NQ_INCLUDECIFSCLIENT) && defined(UD_CC_INCLUDEEXTENDEDSECURITY)
    AMCredentials * pCredentials = NULL;  /* user credentials */
    const NQ_WCHAR * hostW = NULL;         /* host name in Unicode */
#endif /* defined(UD_NQ_INCLUDECIFSCLIENT) && defined(UD_CC_INCLUDEEXTENDEDSECURITY) */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "type:%d hostIPs:%p numHostIPs:%u", type, hostIPs, numHostIPs);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit2;
    }
#ifdef UD_CM_DONOTREGISTERHOSTNAMEDNS
    result = NQ_SUCCESS;
    goto Exit1;
#else /* UD_CM_DONOTREGISTERHOSTNAMEDNS */

    dnsNormalizeName(cmGetFullHostName(), host);

    /* try request with each of the DNS servers until one of them succeeds */
    dnsGetServers();

    for (i = 0; i < staticData->numServers; i++)
    {
        status = dnsUpdateOneServer(type, &staticData->dnsServers[i], host, hostIPs, numHostIPs, NULL);
#if defined(UD_NQ_INCLUDECIFSCLIENT) && defined(UD_CC_INCLUDEEXTENDEDSECURITY)
        if (DNS_REPLY_CODE_REFUSED == status)    /* secure exchange required */
        {
            TkeyContext context;            /* secure authentication exchange */
            pCredentials = (AMCredentials *)cmMemoryAllocate(sizeof(AMCredentials));
            hostW = cmMemoryCloneAString(host);
            if (NULL == pCredentials || NULL == hostW)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                sySetLastError(NQ_ERR_NOMEM);
                goto Exit;
            }
            if (!udGetCredentials(hostW, pCredentials->user, pCredentials->password, pCredentials->domain.name))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "udGetCredentials break by user");
                sySetLastError(NQ_ERR_BADPARAM);
                goto Exit;
            }

            cmWStrupr(pCredentials->domain.name);

            status = dnsStartTkey(&context, &staticData->dnsServers[i]);
            if (NQ_SUCCESS != status)
            {
                goto Exit;
            }

            status = amSpnegoClientLogon(&context, NULL, pCredentials, FALSE, NULL, &context.sessionKey, &context.macKey, dnsQueryTkey, NULL);
            if (AM_SPNEGO_SUCCESS == status)
            {
                status = dnsUpdateOneServer(type, &staticData->dnsServers[i], host, hostIPs, numHostIPs, &context);
            }
            dnsFreeTkeyContext(&context);

            cmMemoryFree(hostW);
            hostW = NULL;

            cmMemoryFree(pCredentials);
            pCredentials = NULL;
        }
#endif /* defined(UD_NQ_INCLUDECIFSCLIENT) && defined(UD_CC_INCLUDEEXTENDEDSECURITY) */
        if (NQ_SUCCESS == status)
        {
            res = NQ_SUCCESS;
        }
    }
    staticData->isRegistered = TRUE;
    result = res;
#endif /* UD_CM_DONOTREGISTERHOSTNAMEDNS */

#if defined(UD_NQ_INCLUDECIFSCLIENT) && defined(UD_CC_INCLUDEEXTENDEDSECURITY)
#ifndef UD_CM_DONOTREGISTERHOSTNAMEDNS
Exit:
#endif /* UD_CM_DONOTREGISTERHOSTNAMEDNS */
    cmMemoryFree(hostW);
    cmMemoryFree(pCredentials);
#endif /* defined(UD_NQ_INCLUDECIFSCLIENT) && defined(UD_CC_INCLUDEEXTENDEDSECURITY) */
#ifdef UD_CM_DONOTREGISTERHOSTNAMEDNS
Exit1:
#endif /* UD_CM_DONOTREGISTERHOSTNAMEDNS */
Exit2:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

/* -- API functions -- */

NQ_STATUS nsDnsInit(void)
{
    NQ_STATUS result = NQ_SUCCESS;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (TRUE == isModuleInitialized)
    {
        ++moduleInitCount;
        goto Exit;
    }

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocateStartup(sizeof(*staticData));
    if (NULL == staticData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate DNS data");
        result = NQ_FAIL;
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    syMutexCreate(&staticData->dataGuard);
    syMutexCreate(&staticData->idGuard);
    staticData->id = 1;
    staticData->isRegistered = FALSE;
    staticData->isNewServerSet = FALSE;
    isModuleInitialized = TRUE;
    ++moduleInitCount;
    dnsGetServers();    /* to have domain name */

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

void nsDnsExit(void)
{
    if ((TRUE == isModuleInitialized) && (0 == --moduleInitCount))
    {
#ifdef UD_NQ_USETRANSPORTIPV4
        nsDnsClearTargetAddress(NS_DNS_A);
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
        nsDnsClearTargetAddress(NS_DNS_AAAA);
#endif /* UD_NQ_USETRANSPORTIPV6 */
        syMutexDelete(&staticData->dataGuard);
        syMutexDelete(&staticData->idGuard);
        /* release memory */
#ifdef SY_FORCEALLOCATION
        if (NULL != staticData)
        {
            cmMemoryFreeShutdown(staticData);
            staticData = NULL;
        }
#endif /* SY_FORCEALLOCATION */
        isModuleInitialized = FALSE;
    }
}

NQ_STATUS nsDnsGetHostNameByService(const NQ_CHAR * service, NQ_CHAR * name)
{
    NQ_STATUS result = NQ_FAIL;
    NQ_CHAR   tmp[CM_NQ_HOSTNAMESIZE + 1];

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "service:%p name:%p", service, name);

    if (NULL == name || NULL == service || 0 == syStrlen(service))
    {
        goto Exit;
    }

    dnsNormalizeName(service, tmp);
    result = dnsQueryService(tmp, name);


Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: Updates the DNS with all addresses of the target
 *--------------------------------------------------------------------
 * PARAMS:  type - the address type: NS_DNS_A or NS_DNS_AAAA
 *
 * RETURNS: NQ_SUCCESS on success
 *
 * NOTE:
 *====================================================================
 */
NQ_STATUS nsDnsSetTargetAddresses(void)
{
    NQ_IPADDRESS hostIps[UD_NS_MAXADAPTERS];    /* array of host IPs of just one type */
    NQ_COUNT numHostIps = 0;                    /* number of host IPs of just one type */
    NQ_STATUS status = NQ_FAIL;                 /* function result */
    const CMSelfIp * nextIp = NULL;             /* next self IP address */ 

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (*(cmNetBiosGetHostNameInfo()->name) == '\0')  
    {
        status = NQ_SUCCESS;
        goto Exit;
    }

#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTNETBIOS)
    cmSelfipIterate();
    for (numHostIps = 0; ;)
    {
        nextIp = cmSelfipNext();
        if (NULL == nextIp)
            break;
        if (CM_IPADDR_IPV4 == CM_IPADDR_VERSION(nextIp->ip))
            hostIps[numHostIps++] = nextIp->ip;
    }
    cmSelfipTerminate();
    if (numHostIps > 0)
        status = dnsUpdate(NS_DNS_A, hostIps, (NQ_UINT16)numHostIps);

#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTNETBIOS) */

#if defined(UD_NQ_USETRANSPORTIPV6)
    cmSelfipIterate();
    for (numHostIps = 0; ;)
    {
        nextIp = cmSelfipNext();
        if (NULL == nextIp)
            break;
        if (CM_IPADDR_IPV6 == CM_IPADDR_VERSION(nextIp->ip))
            hostIps[numHostIps++] = nextIp->ip;
    }
    cmSelfipTerminate();
    if (numHostIps > 0)
        status = dnsUpdate(NS_DNS_AAAA, hostIps, (NQ_UINT16)numHostIps);
#endif /* defined(UD_NQ_USETRANSPORTIPV6) */

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", status);
    return status;
}

void cmDnsSetServersA(const NQ_CHAR * servers)
{
    const NQ_WCHAR * serversW;      /* unicode copy */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "servers:%p", servers);

    if (NULL == servers)
    {
        cmDnsSetServers(NULL);
        goto Exit;
    }

    serversW = cmMemoryCloneAString(servers);
    if (NULL != serversW)
    {
        cmDnsSetServers(serversW);
    }
    cmMemoryFree(serversW);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 *====================================================================
 * PURPOSE: Sets a new list of DNS servers
 *--------------------------------------------------------------------
 * PARAMS:  servers - semicolon delimited list of DNS servers
 *
 * RETURNS: NONE
 *
 * NOTES: This method will remove all the current registered methods
 *          before registering the new servers list
 *====================================================================
 */
void cmDnsSetServers(const NQ_WCHAR * servers)
{
    NQ_INT idx;                             /* index in servers */
    CMResolverMethodDescriptor descriptor;  /* method descriptor */
    NQ_WCHAR * aCopy;                       /* server list copy */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "servers:%s", servers ? cmWDump(servers) : "");

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    syMutexTake(&staticData->dataGuard);
    /* remove per DNS server methods */
    descriptor.type = NQ_RESOLVER_DNS;  /* only type and multicast flag are required */
    descriptor.isMulticast = FALSE;
    staticData->isNewServerSet = TRUE;
    for (idx = 0; idx < (NQ_INT)staticData->numServers; idx++)
    {
        cmResolverRemoveMethod(&descriptor, &staticData->dnsServers[idx]);
    }

    descriptor.type = NQ_RESOLVER_DNS_DC;  /* only type and multicast flag are required */
    descriptor.isMulticast = FALSE;
    for (idx = 0; idx < (NQ_INT)staticData->numServers; idx++)
    {
        cmResolverRemoveMethod(&descriptor, &staticData->dnsServers[idx]);
    }

    staticData->numServers = 0;
    if (NULL != servers && 0 != syWStrlen(servers))
    {
        aCopy = cmMemoryCloneWString(servers);
        if (NULL != aCopy)
        {
            registerServerList(aCopy);
            cmMemoryFree(aCopy);
        }
    }

    syMutexGive(&staticData->dataGuard);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 *====================================================================
 * PURPOSE: Clears from the DNS address of the target
 *--------------------------------------------------------------------
 * PARAMS:  type - the address type: NS_DNS_A or NS_DNS_AAAA
 *
 * RETURNS: NQ_SUCCESS on success
 *
 * NOTE:
 *====================================================================
 */
NQ_STATUS nsDnsClearTargetAddress(NQ_BYTE type)
{
    NQ_STATUS result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "type:%u", type);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (staticData->isRegistered)
    {
        dnsUpdate(NS_DNS_A, NULL, 0);
#ifdef UD_NQ_USETRANSPORTIPV6
        dnsUpdate(NS_DNS_AAAA, NULL, 0);
#endif /* UD_NQ_USETRANSPORTIPV6 */
    }

    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

NQ_STATUS cmDnsSetDomainA(
        const NQ_CHAR * domainName
        )
{
    NQ_STATUS result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "domainName:%s", domainName ? domainName : "");

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (NULL == domainName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "NULL domainName");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if ((CM_NQ_HOSTNAMESIZE < syStrlen(domainName)) || (1 >= syStrlen(domainName)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid domainName size");
        sySetLastError(NQ_ERR_INVALIDNAMELEN);
        goto Exit;
    }

    syMutexTake(&staticData->dataGuard);

    syStrncpy(staticData->dnsDomain , domainName, CM_NQ_HOSTNAMESIZE);
    staticData->dnsDomain[CM_NQ_HOSTNAMESIZE] = '\0';
    staticData->isNewServerSet = TRUE;

    syMutexGive(&staticData->dataGuard);

    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

NQ_STATUS cmDnsSetDomain(
        const NQ_WCHAR * domainName
        )
{
    NQ_STATUS result = NQ_FAIL;
    NQ_CHAR   domainNameA[CM_NQ_HOSTNAMESIZE + 1];

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "domainName:%s", cmWDump(domainName));

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (NULL == domainName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "NULL domainName");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if ((CM_NQ_HOSTNAMESIZE < syWStrlen(domainName)) || (1 >= syWStrlen(domainName)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid domainName size");
        sySetLastError(NQ_ERR_INVALIDNAMELEN);
        goto Exit;
    }

    cmUnicodeToAnsi((NQ_CHAR *)&domainNameA , domainName);

    result = cmDnsSetDomainA((const NQ_CHAR *)&domainNameA);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

NQ_IPADDRESS * cmDnsGetServer(NQ_COUNT dnsID)
{
    NQ_IPADDRESS * ret = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "dnsID:%u", dnsID);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (dnsID <= staticData->numServers)
    {
        ret = &staticData->dnsServers[dnsID];
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", ret);
    return ret;
}

NQ_COUNT cmDnsGetNumDnsServers(void)
{
    NQ_COUNT numServers = 0;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (FALSE == isModuleInitialized)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsdns is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    numServers = staticData->numServers;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "numServers:%u", numServers);
    return numServers;
}

NQ_COUNT nsDnsRegisterMethods(const NQ_WCHAR * list)
{
    CMResolverMethodDescriptor method;                                          /* next method descriptor */
    CMResolverMethodDescriptor methodDC;                                        /* next method descriptor */
    NQ_WCHAR * curServer = NULL;                                                /* pointer to the current server IP */
    NQ_CHAR aServer[CM_IPADDR_MAXLEN];                                          /* the same in ASCII */
    NQ_COUNT numRegistered = 0;                                                 /* number of methods that were registered */
    NQ_HANDLE parseHandle;                                                      /* handle */
    NQ_WCHAR delimiter = cmWChar(CM_NQ_DELIMITER);                        /* delimiter */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "list:%p", list);

    method.type = NQ_RESOLVER_DNS;
    method.isMulticast = FALSE;         /* unicast */
    method.activationPriority = 2;
    method.timeout.low = 1000;          /* milliseconds */
    method.timeout.high = 0;            /* milliseconds */
    method.waitAnyway = TRUE;
    method.requestByName = dnsRequestByName;
    method.responseByName = dnsResponseByName;
    method.requestByIp = dnsRequestByIp;
    method.responseByIp = dnsResponseByIp;

    methodDC.type = NQ_RESOLVER_DNS_DC;
    methodDC.isMulticast = FALSE;       /* unicast */
    methodDC.activationPriority = 2;
    methodDC.timeout.low = 1000;        /* milliseconds */
    methodDC.timeout.high = 0;          /* milliseconds */
    methodDC.waitAnyway = FALSE;
    methodDC.requestByName = dnsRequestByNameForDC;
    methodDC.responseByName = NULL;
    methodDC.requestByIp = NULL;
    methodDC.responseByIp = dnsResponseByNameForDC;

    /* parse servers string */
    parseHandle = cmSpStartParsing((void *)list, (void *)&delimiter, TRUE, (void *)&curServer, TRUE);
    if (NULL == parseHandle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to start parsing");
        goto Exit;
    }

    while(NULL != curServer)
    {
        if (cmWStrlen(curServer) < CM_IPADDR_MAXLEN)
        {
            NQ_IPADDRESS ip;

            cmUnicodeToAnsiN(aServer, sizeof(aServer), curServer, CM_IPADDR_MAXLEN * sizeof(NQ_WCHAR));
            if (NQ_SUCCESS == cmAsciiToIp(aServer, &ip))
            {
                /* register DNS with Resolver */
                cmResolverRemoveMethod(&method, &ip);
                cmResolverRegisterMethod(&method, &ip);
                /* register DNS DC with Resolver */
                cmResolverRemoveMethod(&methodDC, &ip);
                cmResolverRegisterMethod(&methodDC, &ip);
                numRegistered++;
            }
        }

        curServer = cmSpGetNextString(parseHandle);
    }

    cmSpTerminateParsing(parseHandle);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "numRegistered:%d", numRegistered);
    return numRegistered;
}

#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */
