/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Unicode string manipulation
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 ********************************************************************/

#include "cmunicod.h"

#define CM_WDUMP_NUMOFCHARS 2049 /* max file name in Linux is 4096 for unicode. This is 4096 / 2 + 1 */

/* This file implements UNICODE equivalents to the ansiString.h functions
   We assume that strings is not necessarily word aligned. */

typedef struct{
    NQ_WCHAR start;
    NQ_WCHAR end;
    NQ_WCHAR diff;
} CMRange;

static const CMRange plainRangeTable[] = {
    {0x0061, 0x007a, 0x0020},
    {0x00e0, 0x00f6, 0x0020},
    {0x00f8, 0x00fe, 0x0020},
    {0x00ff, 0x00ff, 0xff87},
    {0x0188, 0x0188, 0x0001},
    {0x018C, 0x018C, 0x0001},
    {0x0192, 0x0192, 0x0001},
    {0x0199, 0x0199, 0x0001},
    {0x01a8, 0x01a8, 0x0001},
    {0x01ad, 0x01ad, 0x0001},
    {0x01b0, 0x01b0, 0x0001},
    {0x01b9, 0x01b9, 0x0001},
    {0x01bd, 0x01bd, 0x0001},
    {0x01c6, 0x01c6, 0x0002},
    {0x01c9, 0x01c9, 0x0002},
    {0x01cc, 0x01cc, 0x0002},
    {0x01DD, 0x01dd, 0x004f},
    {0x01f3, 0x01f3, 0x0002},
    {0x01F5, 0x01f5, 0x0001},
    {0x0253, 0x0253, 0x00d2},
    {0x0254, 0x0254, 0x00ce},
    {0x0256, 0x0257, 0x00cd},
    {0x0259, 0x0259, 0x00ca},
    {0x025b, 0x025b, 0x00cb},
    {0x0260, 0x0260, 0x00cd},
    {0x0263, 0x0263, 0x00cf},
    {0x0268, 0x0268, 0x00d1},
    {0x0269, 0x0269, 0x00d3},
    {0x026F, 0x026F, 0x00d3},
    {0x0272, 0x0272, 0x00d5},
    {0x0275, 0x0275, 0x00d6},
    {0x0283, 0x0283, 0x00da},
    {0x0288, 0x0288, 0x00da},
    {0x028A, 0x028b, 0x00d9},
    {0x0292, 0x0292, 0x00db},
    {0x03ac, 0x03ac, 0x0026},
    {0x03ad, 0x03af, 0x0025},
    {0x03b1, 0x03c1, 0x0020},
    {0x03c2, 0x03c2, 0x001f},
    {0x03c3, 0x03cb, 0x0020},
    {0x03cc, 0x03cc, 0x0040},
    {0x03cd, 0x03ce, 0x003f},
    {0x0430, 0x044f, 0x0020},
    {0x0451, 0x045c, 0x0050},
    {0x045e, 0x045f, 0x0050},
    {0x04c8, 0x04c8, 0x0001},
    {0x04cc, 0x04cc, 0x0001},
    {0x04f9, 0x04f9, 0x0001},
    {0x0561, 0x0586, 0x0030},
    {0x1f00, 0x1f07, 0xfff8},
    {0x1f10, 0x1f15, 0xfff8},
    {0x1f20, 0x1f27, 0xfff8},
    {0x1f30, 0x1f37, 0xfff8},
    {0x1f40, 0x1f45, 0xfff8},
    {0x1f60, 0x1f67, 0xfff8},
    {0x1f70, 0x1f71, 0xffb6},
    {0x1f72, 0x1f75, 0xffaa},
    {0x1f76, 0x1f77, 0xff9c},
    {0x1f78, 0x1f79, 0xff80},
    {0x1f7a, 0x1f7b, 0xff90},
    {0x1f7c, 0x1f7d, 0xff82},
    {0x1fb0, 0x1fb1, 0xfff8},
    {0x1fd0, 0x1fd1, 0xfff8},
    {0x1fe0, 0x1fe1, 0xfff8},
    {0x1fe5, 0x1fe5, 0xfff9},
    {0x2170, 0x217f, 0x0010},
    {0x24d0, 0x24e9, 0x001a},
    {0xff41, 0xff5a, 0x0020}
};

#define PLAIN_TABLE_SIZE (sizeof(plainRangeTable) / sizeof(CMRange))

static const CMRange oddRangeTable[] = {
    {0x0101, 0x012f, 0x0001},
    {0x0133, 0x0137, 0x0001},
    {0x013a, 0x0148, 0x0001},
    {0x014b, 0x0177, 0x0001},
    {0x017a, 0x017e, 0x0001},
    {0x0183, 0x0185, 0x0001},
    {0x01a1, 0x01a5, 0x0001},
    {0x01b4, 0x01b6, 0x0001},
    {0x01ce, 0x01dc, 0x0001},
    {0x01df, 0x01ef, 0x0001},
    {0x01fb, 0x0217, 0x0001},
    {0x03e3, 0x03ef, 0x0001},
    {0x0461, 0x0481, 0x0001},
    {0x0491, 0x04bf, 0x0001},
    {0x04c2, 0x04c4, 0x0001},
    {0x04d1, 0x04eb, 0x0001},
    {0x04ef, 0x04f5, 0x0001},
    {0x1e01, 0x1ef9, 0x0001},
    {0x1f51, 0x1f57, 0xfff8}
};

#define ODD_TABLE_SIZE (sizeof(oddRangeTable) / sizeof(CMRange))

/* convert one character to upper case */
static NQ_WCHAR     /* converted character */
unicodeToupper(
    NQ_WCHAR w      /* source character */
    );

/*
 *====================================================================
 * PURPOSE: Calculate string length
 *--------------------------------------------------------------------
 * PARAMS:  IN source string
 *
 * RETURNS: number of characters in the string not including the
 *          terminator
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT
cmWStrlen(
    const NQ_WCHAR* s
    )
{
    const NQ_WCHAR* p;

    p = (NQ_WCHAR*)cmAllignTwo(s);
    while (*p++)
    {};
    return (NQ_UINT)((NQ_INT)((NQ_BYTE*)(--p) - (NQ_BYTE*)s)/(NQ_INT)sizeof(NQ_WCHAR));
}

#ifdef UD_NQ_USEIOVECS
NQ_INT
cmIOWStrcpyV2F(
    NQ_WCHAR* to,
    NQ_IOVecPosition iobuf /* from current IOBUF position */
    )
{
    NQ_WCHAR *from = (NQ_WCHAR*)IOBUF_GETBYTEPTR(iobuf);
    to = (NQ_WCHAR*)cmAllignTwo(to);
    NQ_COUNT segment = iobuf.segment;
    NQ_COUNT byteOffset = iobuf.byteOffset;
    NQ_COUNT numCopied = 0;
    NQ_INT result = NQ_FAIL;

    if ((NQ_BYTE *)from > IOBUF_GETBYTEPTR(iobuf) && byteOffset + 1 >= iobuf.ioBuf->iovec[segment].iov_len)
    {
        /* if align moved p forward and we had buffer overflow */
        if (segment >= iobuf.ioBuf->iovSegmentCnt)
        {
            /* end of buffer. bad call. */
            goto Exit;
        }
        byteOffset = 0;
        from = (NQ_WCHAR*)iobuf.ioBuf->iovec[++segment].iov_base;
    }

    for(;*from;)
    {
        if (byteOffset + sizeof(NQ_WCHAR) > iobuf.ioBuf->iovec[segment].iov_len)
        {
            if (segment >= iobuf.ioBuf->iovSegmentCnt)
            {
                /* end of buffer. bad call. */
                goto Exit;
            }
            if (byteOffset + 1 >= iobuf.ioBuf->iovec[segment].iov_len)
            {
                /* wchar cut to half */
                if (0 == *(iobuf.ioBuf->iovec[segment].iov_base + byteOffset) &&
                    0 == *(iobuf.ioBuf->iovec[segment + 1].iov_base))
                {
                    /* reached '/0' */
                    *to = cmWChar('\0'); /* move forward since we move backward on exit */
                    result = NQ_SUCCESS;
                    ++numCopied;
                    goto Exit;
                }
                else
                {
                    *((NQ_BYTE *)to) = *(iobuf.ioBuf->iovec[segment].iov_base + byteOffset) ;
                    *(((NQ_BYTE *)to) + 1) = *(iobuf.ioBuf->iovec[++segment].iov_base) ;
                    ++to;
                    from = (NQ_WCHAR *)(iobuf.ioBuf->iovec[segment].iov_base + 1);
                    byteOffset = 1;
                    ++numCopied;
                }
            }
        }
        else
        {
            *to++ = *from++;
            byteOffset += (NQ_COUNT)sizeof(NQ_WCHAR);
            ++numCopied;
            if (byteOffset == iobuf.ioBuf->iovec[segment].iov_len)
            {
                byteOffset = 0;
                from = (NQ_WCHAR*)iobuf.ioBuf->iovec[++segment].iov_base;
            }
        }
    }

    result = (NQ_INT)numCopied;

Exit:

    return result;
}

NQ_UINT
cmIOWStrlen(
    NQ_IOVecPosition iobuf
    )
{
    const NQ_WCHAR* p;
    NQ_COUNT segment = iobuf.segment;
    NQ_COUNT byteOffset = iobuf.byteOffset;
    NQ_COUNT length = 0;

    p = (NQ_WCHAR*)cmAllignTwo(IOBUF_GETBYTEPTR(iobuf));
    if ((NQ_BYTE*)p > IOBUF_GETBYTEPTR(iobuf) && byteOffset + 1 >= iobuf.ioBuf->iovec[segment].iov_len)
    {
        /* if align moved p forward and we had buffer overflow */
        if (segment >= iobuf.ioBuf->iovSegmentCnt)
        {
            /* end of buffer. bad call. */
            return 0;
        }
        byteOffset = 0;
        p = (NQ_WCHAR*)iobuf.ioBuf->iovec[++segment].iov_base;
    }

    for(;*p;)
    {
        if (byteOffset + sizeof(NQ_WCHAR) > iobuf.ioBuf->iovec[segment].iov_len)
        {
            /* if end of segment */
            if (segment >= iobuf.ioBuf->iovSegmentCnt)
            {
                /* end of buffer. bad call. */
                return 0;
            }
            if (byteOffset + 1 == iobuf.ioBuf->iovec[segment].iov_len)
            {
                /* if WChar cut to half */
                if (0 == *(iobuf.ioBuf->iovec[segment].iov_base + byteOffset) &&
                    0 == *(iobuf.ioBuf->iovec[segment + 1].iov_base))
                {
                    /* reached '/0' */
                    ++p; /* move forward since we move backward on exit */
                    goto Exit;
                }
                else
                {
                    p = (NQ_WCHAR *) iobuf.ioBuf->iovec[segment + 1].iov_base + 1;
                    byteOffset = 1;
                }
            }
            else
            {
                /* wchar not cut to half */
                p = (NQ_WCHAR *)iobuf.ioBuf->iovec[segment + 1].iov_base;
                byteOffset = 0;
            }
            ++segment;
        }
        else
        {
            /* no skip */
            ++p;
            byteOffset += (NQ_COUNT)sizeof(NQ_WCHAR);
        }
        ++length;
    }

Exit:
    return length;
}
#endif /* UD_NQ_USEIOVECS */

/*
 *====================================================================
 * PURPOSE: Copy a string
 *--------------------------------------------------------------------
 * PARAMS:  OUT destination string
 *          IN source string
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

void
cmWStrcpy(
    NQ_WCHAR* to,
    const NQ_WCHAR* from
    )
{
    to = (NQ_WCHAR*)cmAllignTwo(to);
    from = (NQ_WCHAR*)cmAllignTwo(from);
    do
    {
        *to++ = *from++;
    }
    while (*(from-1));
}

/*
 *====================================================================
 * PURPOSE: Concatenate two strings
 *--------------------------------------------------------------------
 * PARAMS:  OUT destination string
 *          IN source string
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

void
cmWStrcat(
    NQ_WCHAR* to,
    const NQ_WCHAR* from
    )
{
    to += cmWStrlen(to);
    from = (NQ_WCHAR*)cmAllignTwo(from);
    do
    {
        *to++ = *from++;
    }
    while (*(from-1));
}

/*
 *====================================================================
 * PURPOSE: Copies maxNumCharsToCopy wide characters from source to
 *          destination.
 *--------------------------------------------------------------------
 * PARAMS:  OUT destination string
 *          IN  source string
 *          IN  maximum number of wide characters to be copied from source
 *
 * RETURNS: Number of copied wide characters
 *
 * NOTES: The function stop copying if source string reach to NULL
 *        character (copying also the NULL if maxNumCharsToCopy didn't exceed)
 *        or after maxNumCharsToCopy wide characters have been copied.
 *
 *====================================================================
 */

NQ_UINT
cmWStrncpy(
    NQ_WCHAR* outWStr,
    const NQ_WCHAR* inWStr,
    NQ_UINT maxNumCharsToCopy
    )
{
    NQ_UINT numCharsToCopy;
    NQ_UINT result = 0;

    if (0 == maxNumCharsToCopy)
    {
        goto Exit;
    }

    outWStr = (NQ_WCHAR*)cmAllignTwo(outWStr);
    inWStr = (const NQ_WCHAR*)cmAllignTwo(inWStr);

    numCharsToCopy = maxNumCharsToCopy;
    do
    {
        *outWStr++ = *inWStr++;
    }
    while (--numCharsToCopy && *(inWStr -1));


    result = maxNumCharsToCopy - numCharsToCopy;
Exit:
    return result;
}

/*
 *====================================================================
 * PURPOSE: Compare two strings
 *--------------------------------------------------------------------
 * PARAMS:  IN first string
 *          IN second string
 *
 * RETURNS: 1, 0, -1
 *
 * NOTES:
 *====================================================================
 */

NQ_INT
cmWStrcmp(
    const NQ_WCHAR* s1,
    const NQ_WCHAR* s2
    )
{
    NQ_INT result = 0;

    s1 = (const NQ_WCHAR*)cmAllignTwo(s1);
    s2 = (const NQ_WCHAR*)cmAllignTwo(s2);

    while (*s1 == *s2)
    {
        if (*s1 == 0)
        {
            goto Exit;
        }

        s1++;
        s2++;
    }

    result = *s1 - *s2;

Exit:
    return result;
}

/*
 *====================================================================
 * PURPOSE: Compare the rightmost fragments of two strings
 *--------------------------------------------------------------------
 * PARAMS:  IN first string
 *          IN second string
 *          IN number of characters to compare
 *
 * RETURNS: 1, 0, -1
 *
 * NOTES:
 *====================================================================
 */

NQ_INT
cmWStrncmp(
    const NQ_WCHAR* s1,
    const NQ_WCHAR* s2,
    NQ_UINT n
    )
{
    NQ_INT result = 0;

    if (0 == n)
        goto Exit;

    s1 = (const NQ_WCHAR*)cmAllignTwo(s1);
    s2 = (const NQ_WCHAR*)cmAllignTwo(s2);
    while (*s1 == *s2 && n-- > 0)
    {
        if (*s1 == 0)
            goto Exit;
        s1++;
        s2++;
    }
    result = ( n==0 )? 0 : *s1 - *s2;

Exit:
    return result;
}

/*
 *====================================================================
 * PURPOSE: Compare the two strings ignoring case
 *--------------------------------------------------------------------
 * PARAMS:  IN first string
 *          IN second string
 *
 * RETURNS: 1, 0, -1
 *
 * NOTES:
 *====================================================================
 */

NQ_INT
cmWStricmp(
    const NQ_WCHAR* s1,
    const NQ_WCHAR* s2
    )
{
    NQ_WCHAR c1, c2;

    s1 = (const NQ_WCHAR*)cmAllignTwo(s1);
    s2 = (const NQ_WCHAR*)cmAllignTwo(s2);
    for (;;)
    {
        c1 = *s1++;
        c2 = *s2++;
        c1 = unicodeToupper(c1);
        c2 = unicodeToupper(c2);
        if (c1 != c2)
            break;
        if (c1 == 0)
            break;
    }

    return (c1 - c2);
}

/*
 *====================================================================
 * PURPOSE: Compare the beginning of two strings ignoring case
 *--------------------------------------------------------------------
 * PARAMS:  IN first string
 *          IN second string
 *          IN number of chars to compare
 *
 * RETURNS: 1, 0, -1
 *
 * NOTES:
 *====================================================================
 */

NQ_INT
cmWStrincmp(
    const NQ_WCHAR* s1,
    const NQ_WCHAR* s2,
    NQ_UINT n
    )
{
    NQ_WCHAR c1, c2;
    NQ_INT result = 0;

    s1 = (const NQ_WCHAR*)cmAllignTwo(s1);
    s2 = (const NQ_WCHAR*)cmAllignTwo(s2);

    if (0 == n)
        goto Exit;

    for (;;)
    {
        c1 = *s1++;
        c2 = *s2++;
        c1 = unicodeToupper(c1);
        c2 = unicodeToupper(c2);
        if (c1 != c2 || n-- <= 0)
            break;
        if (c1 == 0)
            goto Exit;
    }
    if (n == 0)
        goto Exit;

    result = c1 - c2;

Exit:
    return result;
}

/*
 *====================================================================
 * PURPOSE: Find character in a string
 *--------------------------------------------------------------------
 * PARAMS:  IN string
 *          IN character
 *
 * RETURNS: Pointer to the 1st occurence of this characater in the string
 *          or NULL
 *
 * NOTES:
 *====================================================================
 */

NQ_WCHAR*
cmWStrchr(
    const NQ_WCHAR* s,
    NQ_WCHAR c
    )
{
    NQ_WCHAR* pResult = NULL;

    s = (const NQ_WCHAR*)cmAllignTwo(s);
    while (*s != c)
    {
        if (!*s++)
            goto Exit;
    }
    pResult = (NQ_WCHAR*)s;

Exit:
    return pResult;
}

/*
 *====================================================================
 * PURPOSE: Split Unicode string into tokens
 *--------------------------------------------------------------------
 * DESCRIPTION :A sequence of calls to this function split str into tokens,
 * which are sequences of contiguous characters separated by any of the
 * characters that are part of delimiters. On a first call, the function
 * expects a Unicode string as argument for str, whose first character
 * is used as the starting location to scan for tokens. In subsequent calls,
 * the function expects a null pointer and uses the position right after
 * the end of the last token as the new starting location for scanning.
 *
 * To determine the beginning and the end of a token, the function first
 * scans from the starting location for the first character not contained
 * in delimiters (which becomes the beginning of the token),
 * And then scans starting from this beginning of the token
 * for the first character contained in delimiters, which becomes the end
 * of the token.The scan also stops if the terminating null character is found.
 *
 * This end of the token is automatically replaced by a null-character,
 * and the beginning of the token is returned by the function.
 *
 * Once the terminating null character of str is found in a call to strtok,
 * all subsequent calls to this function (with a null pointer as the
 * first argument) return a null pointer.
 *
 * The point where the next token will be found is kept internally
 * by the function to be used on the next call.
 *
 * PARAMS:  IN string
 *          IN delimiters
 *
 * RETURNS: If a token is found, a pointer to the beginning of the token.
 * Otherwise, a null pointer.
 * A null pointer is always returned when the end of the string (i.e., a null
 * character) is reached in the string being scanned.
 *
 * NOTES:
 *====================================================================
 */

NQ_WCHAR*
cmWStrtok(
    NQ_WCHAR* str,
    const NQ_WCHAR *delimiters
    )
{
    static NQ_WCHAR *  next_point = NULL;
    NQ_WCHAR endWStr = cmWChar('\0');
    NQ_UINT i;

    /* delimiters cannot be NULL, else return NULL as well*/
    if (delimiters == NULL || (str == NULL && next_point == NULL))
        return NULL;

    delimiters = (const NQ_WCHAR*)cmAllignTwo(delimiters);

    if(NULL == str)
    {
        /* not first call of the function */
        str = next_point;
    }
    else
    {
        str = (NQ_WCHAR*)cmAllignTwo(str);
        /*first call of the function - bypass the delimeters character in the begining of the string no need to back empty token*/
        while( (*(str) != endWStr) &&  cmWStrchr(delimiters,*(str)) )
        {
            str++;
        }
    }

    for( i=0 ; *(str+i) != endWStr ; i++ )
    {
        if( cmWStrchr(delimiters,*(str+i)) )
        {
            break;
        }

    }

    /* any delimiters not found */
    if(*(str+i) == endWStr)
    {
        next_point = NULL;
        return str;
    }

    /* any delimiters found in str+i */
    *(str+i) = endWStr;

    /* bypass all the delimiters which follow the first delimiter was found */
    i++;
    while( (*(str+i) != endWStr) &&  cmWStrchr(delimiters,*(str+i)) )
    {
        i++;
    }

    if( *(str+i) != endWStr )
    {
        next_point = str+i;
    }
    else
    {
        next_point = NULL;
    }

    return str;

}

/*
 *====================================================================
 * PURPOSE: Find the rightmost occurence of a character in a string
 *--------------------------------------------------------------------
 * PARAMS:  IN string
 *          IN character
 *
 * RETURNS: Pointer to the rightmost occurence of this characater in the string
 *          or NULL
 *
 * NOTES:
 *====================================================================
 */

NQ_WCHAR*
cmWStrrchr(
    const NQ_WCHAR* s,
    NQ_WCHAR c
    )
{
    const NQ_WCHAR* s1;
    NQ_UINT i;
    NQ_WCHAR* pResult;

    s = (const NQ_WCHAR*)cmAllignTwo(s);
    i = cmWStrlen(s);
    s1 = (s + i);

    for (; i > 0; i--)
    {
        if (*--s1 == c)
        {
            pResult = (NQ_WCHAR*)s1;
            goto Exit;
        }
    }
    pResult = NULL;

Exit:
    return pResult;
}

/*
 *====================================================================
 * PURPOSE: Convert UNICODE string to ANSI string
 *--------------------------------------------------------------------
 * PARAMS:  OUT destination string
 *          IN source string
 *
 * RETURNS: None
 *
 * NOTES:   Converts only ASCII characters
 *====================================================================
 */

void
cmUnicodeToAnsi(
    NQ_CHAR *outStr,
    const NQ_WCHAR* inWStr
    )
{
    if ( (NULL == outStr) || (NULL == inWStr) )
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    cmUnicodeToAnsiN(outStr, CM_IGNORE_LENGTH, inWStr, (NQ_UINT)sizeof(NQ_WCHAR) * ((NQ_UINT)(syWStrlen(inWStr) + CM_TRAILING_NULL)));
Exit:
    return;
}

/*
 *====================================================================
 * PURPOSE: Convert ANSII string to UNICODE string
 *--------------------------------------------------------------------
 * PARAMS:  OUT destination string
 *          IN source string
 *
 * RETURNS: None
 *
 * NOTES:   Converts only ASCII characters
 *====================================================================
 */

void
cmAnsiToUnicode(
    NQ_WCHAR *outWStr,
    const NQ_CHAR *inStr
    )
{
    if ( (NULL == inStr) || (NULL == outWStr) )
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    outWStr = (NQ_WCHAR*)cmAllignTwo(outWStr);
    cmAnsiToUnicodeN(outWStr, CM_IGNORE_LENGTH, inStr, (NQ_UINT)(syStrlen(inStr) + CM_TRAILING_NULL));
Exit:
    return;
}

/*
 *====================================================================
 * PURPOSE: Convert UNICODE string to ANSI string
 *--------------------------------------------------------------------
 * PARAMS:  OUT destination string
 *          IN destination buffer size (bytes)
 *          IN source string
 *          IN source buffer size (bytes)
 *
 * RETURNS: None
 *
 * NOTES:   Converts only ASCII characters. The destination buffer should
 *          be filled with no more than outLength bytes (include NULL terminator
 *          which added at the end of the string by the function). The
 *          string truncate if the destination buffer too small.
 *          The function must not read more than inLength bytes from
 *          the source buffer even if the last character is not NULL.
 *====================================================================
 */

void
cmUnicodeToAnsiN(
    NQ_CHAR *outStr,
    NQ_UINT outLength,
    const NQ_WCHAR* inWStr,
    NQ_UINT inLength
    )
{
#ifndef UD_NQ_INCLUDECODEPAGE
    NQ_UINT inNumOfCharacters;
#endif /* UD_NQ_INCLUDECODEPAGE */
    NQ_BOOL ignoreInLength = (inLength == CM_IGNORE_LENGTH);
    NQ_BOOL ignoreOutLength = (outLength == CM_IGNORE_LENGTH);

    if ( (NULL == outStr) || (NULL == inWStr) )
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if ( ((!ignoreInLength) && (inLength < sizeof(NQ_WCHAR))) || ((!ignoreOutLength) && (outLength < 1)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal buffer length");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

#ifdef UD_NQ_INCLUDECODEPAGE
    cmCpUnicodeToAnsi(outStr, outLength, inWStr, inLength);
#else
    inNumOfCharacters = inLength / ((NQ_UINT)sizeof(NQ_WCHAR));  /* calculate number of characters for source buffer */
    outLength--;                                      /* reserve place for trailing zero */

    for (; (ignoreOutLength || (outLength > 0)) && (ignoreInLength || (inNumOfCharacters > 0)) && (*inWStr != 0); inNumOfCharacters--, outLength--, inWStr++, outStr++)
    {
        *outStr = (NQ_CHAR)cmLtoh16(*inWStr);
    }

    /* Place trailing zero */
    *outStr = 0;

#endif
Exit:
    return;
}

/*
 *====================================================================
 * PURPOSE: Convert ANSII string to UNICODE string
 *--------------------------------------------------------------------
 * PARAMS:  OUT destination string
 *          IN destination buffer size (bytes)
 *          IN source string
 *          IN source buffer size (bytes)
 *
 * RETURNS: None
 *
 * NOTES:   Converts only ASCII characters. The destination buffer should
 *          be filled with no more than outLength bytes (include NULL terminator
 *          which added at the end of the string by the function). The
 *          string truncate if the destination buffer too small.
 *          The function must not read more than inLength bytes from
 *          the source buffer even if the last character is not NULL.
 *====================================================================
 */

void
cmAnsiToUnicodeN(
    NQ_WCHAR* outWStr,
    NQ_UINT outLength,
    const NQ_CHAR *inStr,
    NQ_UINT inLength
    )
{
#ifndef UD_NQ_INCLUDECODEPAGE
    NQ_UINT outNumOfCharacters;
#endif /* UD_NQ_INCLUDECODEPAGE */
    NQ_BOOL ignoreInLength = (inLength == CM_IGNORE_LENGTH);
    NQ_BOOL ignoreOutLength = (outLength == CM_IGNORE_LENGTH);

    if ( (NULL == inStr) || (NULL == outWStr) )
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if ( ((!ignoreInLength) && (inLength < 1)) || ((!ignoreOutLength) && (outLength < sizeof(NQ_WCHAR))))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal buffer length");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    outWStr = (NQ_WCHAR*)cmAllignTwo(outWStr);
#ifdef UD_NQ_INCLUDECODEPAGE
    cmCpAnsiToUnicode(outWStr, outLength, inStr, inLength);
#else
    outNumOfCharacters = outLength / ((NQ_UINT)sizeof(NQ_WCHAR));  /* calculate number of characters for destination buffer */
    outNumOfCharacters--;                               /* reserve place for trailing zero */

    for (; (ignoreOutLength || (outNumOfCharacters > 0)) && (ignoreInLength || (inLength > 0)) && (*inStr != 0); outNumOfCharacters--, inLength--, inStr++)
    {
        NQ_WCHAR c = (NQ_BYTE)*inStr;
        *outWStr++ = (NQ_WCHAR)cmHtol16(c);
    }

    /* Place trailing zero */
    *outWStr = 0;

#endif
    Exit:
        return;
}

/*
 *====================================================================
 * PURPOSE: Convert Unicode string into an ANSI string for immediate printout
 *--------------------------------------------------------------------
 * PARAMS:  IN Unicode string
 *
 * RETURNS: output string converted to ANSI
 *
 * NOTES: When calling a function it must not be used on more than one
 *        argument because the output string is declared as static.
 *====================================================================
 */

NQ_CHAR*
cmWDump(
    const NQ_WCHAR* inWStr
    )
{
    static NQ_CHAR outStr[CM_WDUMP_NUMOFCHARS * sizeof(NQ_WCHAR) + CM_TRAILING_NULL];

    if (NULL != inWStr)
    {
        cmUnicodeToAnsiN(outStr, sizeof(outStr), inWStr, sizeof(outStr) * sizeof(NQ_WCHAR));
    }
    else
    {
        outStr[0] = '\0';
    }

    return outStr;
}

/*
 *====================================================================
 * PURPOSE: Convert WCHAR string to uppercase
 *--------------------------------------------------------------------
 * PARAMS:  IN string
 *
 * RETURNS: none
 *
 * NOTES:

 *====================================================================
 */

void
cmWStrupr(
    NQ_WCHAR* s
    )
{
    while (*s != 0)
    {
        *s = unicodeToupper(*s);
        s++;
    }
}

/*
 *====================================================================
 * PURPOSE: Convert a single WCHAR to uppercase
 *--------------------------------------------------------------------
 * PARAMS:  OUT destination buffer
 *          IN pointer to sourc character
 *
 * RETURNS: number of bytes converted
 *
 * NOTES:

 *====================================================================
 */

NQ_INT
cmWToupper(
    NQ_WCHAR* to,
    const NQ_WCHAR* from
    )
{
    *to = unicodeToupper(*from);
    return 1;
}

/*
 *====================================================================
 * PURPOSE: Convert a single WCHAR to uppercase
 *--------------------------------------------------------------------
 * PARAMS:  IN WCHAR
 *
 * RETURNS: converted WCHAR
 *
 * NOTES:

 *====================================================================
 */

static NQ_WCHAR
unicodeToupper(
    NQ_WCHAR w
    )
{
    NQ_INDEX i;
    NQ_WCHAR c = (NQ_WCHAR)cmLtoh16(w);
    NQ_WCHAR result;

    for (i = 0; i < PLAIN_TABLE_SIZE && plainRangeTable[i].start <= c ; i++)
    {
        if (c <= plainRangeTable[i].end)
        {
            result = (NQ_WCHAR)cmHtol16((NQ_WCHAR)(c - plainRangeTable[i].diff));
            goto Exit;
        }
    }

    for (i = 0; i < ODD_TABLE_SIZE && oddRangeTable[i].start <= c; i++)
    {
        if (c <= oddRangeTable[i].end)
        {
            if ((c - oddRangeTable[i].start) % 2 == 0)
            {
                result = (NQ_WCHAR)cmHtol16((NQ_WCHAR)(c - oddRangeTable[i].diff));
                goto Exit;
            }
        }
    }

    result = (NQ_WCHAR)cmHtol16(c);

Exit:
    return result;
}
