/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Trace common definitions
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 ********************************************************************/

#include "cmtrace.h"

#ifdef UD_NQ_INCLUDETRACE

#include "udconfig.h"

#define CM_DEBUG_DUMP_BLOCK_SIZE                16
#define CM_TRC_LENGTHOFLOGFILENAMEEXTENSIONS    50
#define CM_TRC_RECORD_INFO_SIZE                 256
#define CM_LOG_CREATE_RETRYCOUNT                3   /* Used for retry count for each new log file creation.
                                                       At least 1 attempt to create a log file will be performed regardless the value of this parameter */

/*  Format definition of an internal log message. */
typedef enum{

    TYPE_VERSION    = 'V',   /* Version record. This record is placed once at the very beginning of the log. */
    TYPE_START      = 'S',   /* Thread start. This record appears once on a thread start. It marks a new thread and contains its name. */
    TYPE_STOP       = 'T',   /* Thread stop. This record appears once on a thread end. */
    TYPE_ENTER      = 'F',   /* Function entrance. This record marks an entrance into a function execution. */
    TYPE_LEAVE      = 'L',   /* Function exit. This record marks an exit from a function execution. */
    TYPE_ERROR      = 'E',   /* Error message. This record contains error text and system error code. */
    TYPE_INFO       = 'I'    /* Information message. This record contains general purpose text. */
}
CMRecordTypes;

typedef struct{
    NQ_UINT32  id;                                  /* Record ID. Used for sequencing. */
    NQ_CHAR    type;                                /* Record type. See CMRecordTypes above. */
    NQ_ULONG   thread;                              /* Thread ID. This number is provided by OS. */
    NQ_UINT32  timestamp;                           /* Record time. The system time when this record was created. */
    NQ_UINT    level;                               /* Severity. Abstract record priority, used by viewer. */
    NQ_CHAR    file[64];                            /* Source file. This should be supported by the OS. */
    NQ_CHAR    function[64];                        /* Called function. This should be supported by the OS. */
    NQ_UINT    line;                                /* Source line. This should be supported by the OS. */
    NQ_UINT32  error;                               /* Error code. This should be supported by the OS. */
    NQ_CHAR    data[CM_TRC_RECORD_INFO_SIZE];       /* Record information. Pointer to the variable argument list starting with format string and followed by format arguments. */
}
CMLogRecord;

typedef struct{
    CMLogRecord record;
    NQ_CHAR buffer[1460];
}
CMTrace;

static NQ_CHAR productInfoString[CM_TRC_RECORD_INFO_SIZE];
static NQ_BYTE initialized = FALSE;
static NQ_BYTE shuttingDown = FALSE;
static NQ_UINT traceLevelThreshold = CM_TRC_DEBUG_LEVEL;
static NQ_COUNT numberOfLines = 0;
static SYThread thread;
static NQ_BOOL  canWrite = FALSE;
static SYSocketHandle sendSocket;
static NQ_PORT sendPort;
static NQ_PORT dynPort = 0;
#ifdef UD_CM_LOG_TOREMOTE
static NQ_IPADDRESS addr;
static SYSocketHandle remoteLogSocket;
#endif /* UD_CM_LOG_TOREMOTE */
SYSocketHandle threadSocket;  /* receiving socket */
static SYMutex  endMutex;
static SYMutex  cycleMutex;
#ifdef UD_CM_LOG_TOFILE
static NQ_COUNT fileIndex = 0;
static SYFile file = syInvalidFile();
static NQ_WCHAR filename[CM_BUFFERLENGTH(NQ_WCHAR, UD_FS_MAXPATHLEN)];

/* This function will try to create new log file, if from any reason the create operation failed, will retry CM_LOG_CREATE_RETRYCOUNT times
 * Note: At least 1 attempts to create a new log file will be performed */
static NQ_BOOL createLogFile(NQ_WCHAR *fileName)
{
    NQ_INT  counter = CM_LOG_CREATE_RETRYCOUNT;    /* operation attempts counter */
    NQ_BOOL res = FALSE;

    /* Do at least 1 attempt to create a log file */
    do
    {
        file = syTraceCreateFile(fileName);
        if (TRUE == syIsValidFile(file))
        {
            res = TRUE;
            break;
        }
    }
    while (0 < --counter);

    return res;
}

#endif /* UD_CM_LOG_TOFILE */

static const NQ_CHAR *truncateFileName(const NQ_CHAR *path)
{
    const NQ_CHAR *p = syStrrchr(path, SY_PATHSEPARATOR);
    return (p == NULL ? path : p + 1);
}

static void traceHeader(CMLogRecord *record, NQ_CHAR mt, NQ_UINT level, const NQ_CHAR *file, const NQ_CHAR *function, NQ_UINT line)
{
    static NQ_UINT32 id = 1;
    NQ_CHAR *parenthesis = "()";

    record->id = ++id;
    record->thread = (NQ_ULONG)syThreadGetCurrent();
    record->timestamp = (NQ_UINT32)syGetTimeInSec();
    record->level = level;
    syStrcpy(record->file, truncateFileName(file));
    syStrcpy(record->function, function);
    syStrncat(record->function, parenthesis, syStrlen(parenthesis));
    record->line = line;
    record->type = mt;
    record->data[0] = '\0';
}

#ifdef UD_CM_LOG_TOREMOTE
static void writeToRemoteLog(const NQ_CHAR *buffer, NQ_UINT32 size)
{
    NQ_INT res;

    if (initialized && syIsValidSocket(remoteLogSocket))
    {
        res = sySendToSocket(remoteLogSocket, (const NQ_BYTE *)buffer, (NQ_COUNT)size, &addr, syHton16(UD_CM_LOG_REMOTESRVPORT));
        if (res != size)
        {
            syPrintf("Traces sendto returned %d, error: %d\n", res, syGetLastError());
        }
    }
}
#endif /* UD_CM_LOG_TOREMOTE */

#ifdef UD_CM_LOG_TOFILE
static void writeToFileLog(const NQ_CHAR *buffer, NQ_UINT32 size)
{
    NQ_IOBuf tmpBuf;
    IOBUF_CONSTRUCTORINIT(tmpBuf)

    IOBUF_CONSTRUCTOR(tmpBuf, (NQ_BYTE *)buffer, (NQ_COUNT)size);
    syWriteFile(file, tmpBuf, (NQ_COUNT)size);
}
#endif /* UD_CM_LOG_TOFILE */

static void writeTrace(CMTrace *trc)
{
    NQ_UINT32 size;
    NQ_INT res;
    CMLogRecord record = trc->record;

    if (canWrite)
    {
        size = (NQ_UINT32)sySnprintf(trc->buffer, sizeof(trc->buffer), "%c;%lu;%lu;%lu;%d;%s;%s;%d", record.type, (NQ_ULONG)record.id, (NQ_ULONG)record.thread, (NQ_ULONG)record.timestamp, (NQ_UINT)record.level, record.file, record.function, record.line);
        switch (record.type)
        {
            case TYPE_ERROR:
                size += (NQ_UINT32)sySnprintf(trc->buffer + size, (NQ_UINT)(sizeof(trc->buffer) - size), ";%lu;%s\n", (NQ_ULONG)record.error, record.data);
                break;
            case TYPE_VERSION:
            case TYPE_START:
            case TYPE_STOP:
            case TYPE_INFO:
            case TYPE_ENTER:
            case TYPE_LEAVE:
                size += (NQ_UINT32)sySnprintf(trc->buffer + size, sizeof(trc->buffer) - size, ";%s\n", record.data);
                break;
            default:
                syPrintf("unknown record->type = %c\n", record.type);
                break;
        }

        /* send datagram */
        if (initialized && !shuttingDown && 0 != dynPort)
        {
            syMutexTake(&cycleMutex);
            res = sySendToSocket(sendSocket, (NQ_BYTE *)trc->buffer, (NQ_COUNT)size, cmSelfipGetLocalHostIp(), syHton16(dynPort));
            if (res != size)
            {
                syPrintf("Traces sendto returned %d, error: %d\n", res, syGetLastError());
            }
            syMutexGive(&cycleMutex);
        }
    }
}

static void writeProductInfoToLog(NQ_CHAR *productInfoStr, NQ_UINT32 size)
{
#ifdef UD_CM_LOG_TOFILE
    writeToFileLog(productInfoStr, size);
#endif /* UD_CM_LOG_TOFILE */
#ifdef UD_CM_LOG_TOCONSOLE
    syPrintf("%s", productInfoStr);
    syFflush(stdout);
#endif /* UD_CM_LOG_TOCONSOLE */
#ifdef UD_CM_LOG_TOREMOTE
    writeToRemoteLog(productInfoStr, size);
#endif /* UD_CM_LOG_TOREMOTE */
}

static void threadBodyTrace(void)
{
    NQ_BOOL active = TRUE;  /* flag to execute the body */
    NQ_CHAR buffer[1461];   /* full MTU + 1 null termination */
#ifdef UD_CM_LOG_TOFILE
    NQ_CHAR copyName[256];
    NQ_UINT logFolderLen;
    NQ_CHAR *name = copyName;
    NQ_CHAR tmpName[] = UD_CM_LOG_FILENAME;
    NQ_CHAR* ptrExtension = cmAStrrchr(tmpName, '.');
    NQ_UINT  nameSize= 0;
    NQ_STATIC NQ_UINT currentFileSizeBytes = 0;
    NQ_BOOL res = FALSE;    /* Flag to determine if createLogFile succeeded or not */
#endif /* UD_CM_LOG_TOFILE*/

    threadSocket = syCreateSocket(FALSE, CM_IPADDR_IPV4);
    if (!syIsValidSocket(threadSocket))
    {
        syPrintf("Unable to create trace socket - %d\n", syGetLastError());
        goto Exit;
    }

    dynPort = cmDynamicPortBindSocket(threadSocket, FALSE); /* not all OS support reuse address */
    if (0 == dynPort)
    {
        syPrintf("Unable to bind trace socket - %d\n", syGetLastError());
        syCloseSocket(threadSocket);
        goto Exit;
    }

#ifdef UD_CM_LOG_TOFILE
    if (NULL == ptrExtension)
    {
        syPrintf("Could not create log file. Specify the extension.\n");
        goto Exit;
    }

    *ptrExtension = '\0';

    udGetLogFileBaseFolder(copyName, sizeof(copyName));
    logFolderLen = (NQ_UINT)syStrlen(copyName);
    if (logFolderLen > 0)
    {
        if (logFolderLen + syStrlen(UD_CM_LOG_FILENAME) + CM_TRC_LENGTHOFLOGFILENAMEEXTENSIONS >= sizeof(copyName))
        {
            syPrintf("Could not create log file. The path of log file base folder is too long.\n");
            goto Exit;
        }

        if (copyName[logFolderLen - 1] != SY_PATHSEPARATOR)
        {
            copyName[logFolderLen] = SY_PATHSEPARATOR;
            copyName[logFolderLen + 1] = 0;
        }

        nameSize= (NQ_UINT)syStrlen(copyName);
        name = &copyName[nameSize];
    }

    sySnprintf(name, sizeof(copyName) - nameSize, "%s_%d_%d.%s", tmpName, fileIndex++, syGetPid(), ptrExtension + 1);
    cmAnsiToUnicode(filename, copyName);
    res = createLogFile(filename);
    if (FALSE == res)
    {
        syPrintf("Could not create log file:'%s', error: %d\n", copyName, syGetLastError());
        goto Exit;
    }
    else
    {
        syPrintf("Log file: %s\n", copyName);
    }

    /* update first log file size and line number due to product info */
    numberOfLines = 1;
    currentFileSizeBytes = (NQ_UINT)syStrlen(productInfoString);
#endif /* UD_CM_LOG_TOFILE */

    /* write product info to first line of log */
    writeProductInfoToLog(productInfoString, (NQ_UINT32)syStrlen(productInfoString));

    syMutexTake(&endMutex);
    while (active)
    {
        SYSocketSet set;          /* select argument */

        syClearSocketSet(&set);
        syAddSocketToSet(threadSocket, &set);

        if (shuttingDown)
        {
            syCloseSocket(threadSocket);
            cmDynamicPortRelease(dynPort);
#ifdef UD_CM_LOG_TOFILE
            syCloseFile(file);
            file = syInvalidFile();
#endif /* UD_CM_LOG_TOFILE */
            syMutexGive(&endMutex);
            active = FALSE;
            break;
        }

        switch (sySelectSocket(&set, 1))
        {
            case NQ_FAIL:
            {
                break;
            }
            case 0:
            {
                continue;   /* timeout */
            }
            default:
            {
                NQ_INT size;              /* message size */
                NQ_IPADDRESS ip;
                NQ_PORT port;

                size = syRecvFromSocket(threadSocket, (NQ_BYTE *)buffer, sizeof(buffer), &ip, &port);
                if (size <= 0)
                {
                    continue;
                }
#ifdef UD_CM_LOG_TOCONSOLE
                buffer[size] = '\0';
                syPrintf("%s", buffer);
                syFflush(stdout);
#endif /* UD_CM_LOG_TOCONSOLE */
#ifdef UD_CM_LOG_TOREMOTE
                writeToRemoteLog(buffer, (NQ_UINT32)size);
#endif /* UD_CM_LOG_TOREMOTE */
#ifdef UD_CM_LOG_TOFILE
                currentFileSizeBytes += (NQ_UINT)size;

                if (UD_CM_LOG_FILESIZE_BYTES < currentFileSizeBytes)
                {
                    /* file full. switch to next file */
                    name = &copyName[nameSize];
                    sySnprintf(name, sizeof(copyName) - nameSize, "%s_%d_%d.%s", tmpName, fileIndex, syGetPid(), ptrExtension + 1);
                    cmAnsiToUnicode(filename, copyName);
                    syCloseFile(file);
                    res = createLogFile(filename);
                    if (FALSE == res)
                    {
                        syPrintf("Could not create log file:'%s', error: %d\n", copyName, syGetLastError());
                        goto Exit;
                    }

                    /* write product info to first line of log file */
                    writeToFileLog(productInfoString, (NQ_UINT32)syStrlen(productInfoString));
                    numberOfLines = 1;
                    currentFileSizeBytes = (NQ_UINT)syStrlen(productInfoString);
#ifdef UD_CM_LOG_NUMBEROFFILES
                    if (UD_CM_LOG_NUMBEROFFILES - 1 < fileIndex)
                    {
                        /* delete oldest file. */
                        name = &copyName[nameSize];
                        sySnprintf(name, sizeof(copyName) - nameSize, "%s_%d_%d.%s", tmpName, (fileIndex - UD_CM_LOG_NUMBEROFFILES), syGetPid(), ptrExtension + 1);
                        cmAnsiToUnicode(filename, copyName);
                        syTraceDeleteFile(filename);
                    }
#endif /* UD_CM_LOG_NUMBEROFFILES */
                    fileIndex++;
                }

                writeToFileLog(buffer, (NQ_UINT32)size);
#endif /* UD_CM_LOG_TOFILE */
                break;
            }
        }
    }

Exit:
    return;
}

void cmTraceInit(void)
{
    if (!udGetInternalTrace())
    {
#ifdef UD_PRINT_STARTUP_INFO
        syPrintf("UD has cancelled internal trace\n");
#endif /* UD_PRINT_STARTUP_INFO */
        goto Exit;
    }

    if (!initialized)
    {
        CMProductInfo   *productInfo;

        /* create product information string */
        productInfo = cmGetCurrentProductInformation();
        sySprintf(productInfoString, "%c;product name: %s, product type: %s, product version: %s\n", TYPE_VERSION, productInfo->productName, productInfo->productType, productInfo->version);

        sendSocket = syCreateSocket(FALSE, CM_IPADDR_IPV4);
        if (syIsValidSocket(sendSocket))
        {
            sendPort = cmDynamicPortBindSocket(sendSocket, FALSE);
            if (0 != sendPort)
            {
                syMutexCreate(&endMutex);
                syMutexCreate(&cycleMutex);
                shuttingDown = FALSE;
                syThreadStart(FALSE, NQ_THREAD_PRIORITY_NONE, &thread, threadBodyTrace, TRUE);
                initialized = TRUE;
            }
            else
            {
                syCloseSocket(sendSocket);
                syPrintf("Unable to start logs module - %d\n", syGetLastError());
            }

        }
        else
        {
            syPrintf("Unable to start logs module - %d\n", syGetLastError());
        }

#ifdef UD_CM_LOG_TOREMOTE
        cmAsciiToIp(UD_CM_LOG_REMOTESRVIP, &addr);
        remoteLogSocket = syCreateSocket(FALSE, CM_IPADDR_IPV4);
        if (!syIsValidSocket(remoteLogSocket))
        {
            syPrintf("Could not create socket for remote log, error: %d\n", syGetLastError());
        }
#ifdef UD_CM_LOG_REMOTEBROADCAST
        if (syAllowBroadcastsSocket(remoteLogSocket) == NQ_FAIL)
        {
            syCloseSocket(remoteLogSocket);
            syPrintf("Could not set socket option for remote log, error: %d\n", syGetLastError());
        }
#endif /* UD_CM_LOG_REMOTEBROADCAST */
#endif /* UD_CM_LOG_TOREMOTE */
        canWrite = TRUE;
    }
Exit:
    return;
}

void cmTraceFinish(void)
{
    syMutexTake(&cycleMutex);
    if (initialized)
    {
        shuttingDown = TRUE;
        syMutexGive(&cycleMutex);
        /* this sleep is relevant only if NQ server uses multiple workers */
        /* sySleep(1); */
        syMutexTake(&cycleMutex);
        numberOfLines = 0;
#ifdef UD_CM_LOG_TOFILE
        syCloseFile(file);
        file = syInvalidFile();
#endif /* UD_CM_LOG_TOFILE */
#ifdef UD_CM_LOG_TOREMOTE
        syCloseSocket(remoteLogSocket);
#endif /* UD_CM_LOG_TOREMOTE */
        if (syIsValidSocket(sendSocket))
        {
            syCloseSocket(sendSocket);
            cmDynamicPortRelease(sendPort);
            sendSocket = syInvalidSocket();
            syMutexGive(&cycleMutex);
            syMutexDelete(&cycleMutex);
            syMutexDelete(&endMutex);
            initialized = FALSE;
            canWrite = FALSE;
            syThreadDestroy(thread);
            goto Exit;
        }
        canWrite = FALSE;
        syThreadDestroy(thread);
    }
    syMutexGive(&cycleMutex);
Exit:
    return;
}

void cmTraceMessage(const NQ_CHAR *file, const NQ_CHAR *function, NQ_UINT line, NQ_UINT level, const NQ_CHAR *format, ...)
{
    if (level <= traceLevelThreshold && !shuttingDown && initialized)
    {
        static CMTrace trace;
        va_list args;
        CMLogRecord *record = &trace.record;
        NQ_UINT32 lastError;

        lastError = (NQ_UINT32)syGetLastError();
        traceHeader(record, TYPE_INFO, level, file, function, line);

        va_start(args, format);
        syVsnprintf(record->data, sizeof(record->data), format, args);
        va_end(args);

        writeTrace(&trace);
        sySetLastError(lastError);
    }
}

void cmTraceError(const NQ_CHAR *file, const NQ_CHAR *function, NQ_UINT line, NQ_UINT level, const NQ_CHAR *format, ...)
{
    if (level <= traceLevelThreshold && !shuttingDown && initialized)
    {
        static CMTrace trace;
        va_list args;
        CMLogRecord *record = &trace.record;
        NQ_UINT32 lastError;

        lastError = (NQ_UINT32)syGetLastError();
        traceHeader(record, TYPE_ERROR, level, file, function, line);
        record->error = ((NQ_UINT32)syGetLastError());

        va_start(args, format);
        syVsnprintf(record->data, sizeof(record->data), format, args);
        va_end(args);

        writeTrace(&trace);
        sySetLastError(lastError);
    }
}

void cmTraceFuncEnter(const NQ_CHAR *file, const NQ_CHAR *function, NQ_UINT line, NQ_UINT level, const NQ_CHAR *format, ...)
{
    if (level <= traceLevelThreshold && !shuttingDown && initialized)
    {
        static CMTrace trace;
        va_list args;
        CMLogRecord *record = &trace.record;
        NQ_UINT32 lastError;

        lastError = (NQ_UINT32)syGetLastError();
        traceHeader(record, TYPE_ENTER, level, file, function, line);

        va_start(args, format);
        syVsnprintf(record->data, sizeof(record->data), format, args);
        va_end(args);

        writeTrace(&trace);
        sySetLastError(lastError);
    }
}

void cmTraceFuncLeave(const NQ_CHAR *file, const NQ_CHAR *function, NQ_UINT line, NQ_UINT level, const NQ_CHAR *format, ...)
{
    if (level <= traceLevelThreshold && !shuttingDown && initialized)
    {
        static CMTrace trace;
        va_list args;
        CMLogRecord *record = &trace.record;
        NQ_UINT32 lastError;

        lastError = (NQ_UINT32)syGetLastError();
        traceHeader(record, TYPE_LEAVE, level, file, function, line);
        record->error = lastError;

        va_start(args, format);
        syVsnprintf(record->data, sizeof(record->data), format, args);
        va_end(args);

        writeTrace(&trace);
        sySetLastError(lastError);
    }
}

void cmTraceStart(const NQ_CHAR *file, const NQ_CHAR *function, NQ_UINT line, NQ_UINT level, const NQ_CHAR *name)
{
    if (level <= traceLevelThreshold && !shuttingDown && initialized)
    {
        static CMTrace trace;
        CMLogRecord *record = &trace.record;

        traceHeader(record, TYPE_START, level, file, function, line);
        syStrcpy(record->data, name);
        writeTrace(&trace);
    }
}

void cmTraceStop(const NQ_CHAR *file, const NQ_CHAR *function, NQ_UINT line, NQ_UINT level, const NQ_CHAR *name)
{
    if (level <= traceLevelThreshold && !shuttingDown && initialized)
    {
        static CMTrace trace;
        CMLogRecord *record = &trace.record;

        traceHeader(record, TYPE_STOP, level, file, function, line);
        syStrcpy(record->data, name);
        writeTrace(&trace);
    }
}

static NQ_CHAR c(NQ_BYTE p)
{
    return (NQ_CHAR)((p < 32 || p >= 127) ? '.' : p);
}

void cmTraceDump(const NQ_CHAR *file, const NQ_CHAR *function, NQ_UINT line, NQ_UINT level, const NQ_CHAR *str, const void *addr, NQ_UINT nBytes)
{
    if (level <= traceLevelThreshold && !shuttingDown && initialized)
    {
        NQ_INDEX i = 0;
        NQ_BYTE p[CM_DEBUG_DUMP_BLOCK_SIZE];
        NQ_BYTE * pAddr = (NQ_BYTE *)addr;

        cmTraceMessage(file, function, line, level, "%s (%d bytes) (address: %p):", str, nBytes, pAddr);
        if (nBytes < 1)
        {
            goto Exit;
        }
        if (pAddr != NULL)
        {
            for (; nBytes > 0; i+= CM_DEBUG_DUMP_BLOCK_SIZE, pAddr += CM_DEBUG_DUMP_BLOCK_SIZE, nBytes = nBytes > CM_DEBUG_DUMP_BLOCK_SIZE? nBytes - CM_DEBUG_DUMP_BLOCK_SIZE : 0)
            {
                syMemset(p, 0, sizeof(p));
                syMemcpy(p, pAddr, nBytes > CM_DEBUG_DUMP_BLOCK_SIZE? CM_DEBUG_DUMP_BLOCK_SIZE : nBytes);
                cmTraceMessage(file, function, line, level, "%03x: 0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x,  0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x, 0x%02x "
                        "| %c%c%c%c%c%c%c%c  %c%c%c%c%c%c%c%c |",
                        i,
                        p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9], p[10], p[11], p[12], p[13], p[14], p[15],
                        c(p[0]), c(p[1]), c(p[2]), c(p[3]), c(p[4]), c(p[5]), c(p[6]), c(p[7]),
                        c(p[8]), c(p[9]), c(p[10]), c(p[11]), c(p[12]), c(p[13]), c(p[14]), c(p[15]));
            }
        }
        else
        {
            cmTraceMessage(file, function, line, level, "Dump address is NULL");
        }
    }
Exit:
    return;    
}

#ifdef UD_NQ_USEIOVECS
void cmTraceIODump(const NQ_CHAR *file, const NQ_CHAR *function, NQ_UINT line, NQ_UINT level, const NQ_CHAR *str, const NQ_IOBufPos ioPos, NQ_UINT nBytes)
{
    if (level <= traceLevelThreshold && !shutDwn && initialized)
    {
        NQ_INDEX i = 0;
        NQ_BYTE p[CM_DEBUG_DUMP_BLOCK_SIZE];
        NQ_BYTE * pAddr = NULL;

        cmTraceMessage(file, function, line, level, "%s (%d bytes) (address: 0x%x):", str, nBytes, ioPos);
        if (nBytes < 1)
        {
            return;
        }
        if (IOBUF_ISNULL(ioPos))
        {
            NQ_INDEX j = 0;
            NQ_UINT32 segBytes;

            for (j = 0; j < ioPos.ioBuf->iovSegmentCnt; j++)
            {
                pAddr = ioPos.ioBuf->iovec[j].iov_base;
                segBytes = ioPos.ioBuf->iovec[j].iov_len;
                for (; segBytes > 0; i+= CM_DEBUG_DUMP_BLOCK_SIZE, pAddr += CM_DEBUG_DUMP_BLOCK_SIZE, segBytes = segBytes > CM_DEBUG_DUMP_BLOCK_SIZE? segBytes - CM_DEBUG_DUMP_BLOCK_SIZE : 0)
                {
                    syMemset(p, 0, sizeof(p));
                    syMemcpy(p, pAddr, segBytes > CM_DEBUG_DUMP_BLOCK_SIZE? CM_DEBUG_DUMP_BLOCK_SIZE : segBytes);
                    cmTraceMessage(file, function, line, level, "%03x: %02x %02x %02x %02x %02x %02x %02x %02x  %02x %02x %02x %02x %02x %02x %02x %02x | %c%c%c%c%c%c%c%c  %c%c%c%c%c%c%c%c |",
                            i,
                            p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9], p[10], p[11], p[12], p[13], p[14], p[15],
                            c(p[0]), c(p[1]), c(p[2]), c(p[3]), c(p[4]), c(p[5]), c(p[6]), c(p[7]),
                            c(p[8]), c(p[9]), c(p[10]), c(p[11]), c(p[12]), c(p[13]), c(p[14]), c(p[15]));
                }
            }
        }
        else
        {
            cmTraceMessage(file, function, line, level, "Dump address is NULL");
        }
    }
}
#endif /* UD_NQ_USEIOVECS */

void cmTraceThresholdSet(NQ_UINT newValue)
{
    traceLevelThreshold = newValue;
}

NQ_UINT cmTraceThresholdGet(void)
{
    return traceLevelThreshold;
}

void nqEnableTraceLog(NQ_BOOL on)
{
    canWrite = on;
}

#endif /* UD_NQ_INCLUDETRACE */


