/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Common self IP operations
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 *************************************************************************/

#include "cmselfip.h"

/* Definitions: */
typedef struct 
{
    CMItem item;        /* inheritance */
    CMSelfIp selfIp;    /* IP address and broadcast address */
}
SelfIp;

typedef struct
{
    CMList ips;                     /* list of self IPs */
    CMIterator iterator;            /* IP iterator */
}
StaticData;

static NQ_IPADDRESS zeroIp = CM_IPADDR_ZERO;
static NQ_IPADDRESS anyIPVer4 = CM_IPADDR_ANY4;
static NQ_IPADDRESS localHostIP = CM_IPADDR_LOCAL;
#ifdef UD_NQ_USETRANSPORTIPV6
static NQ_IPADDRESS anyIPVer6 = CM_IPADDR_ANY6;
static NQ_IPADDRESS localHostIPVer6 = CM_IPADDR_LOCAL6;
#endif /* UD_NQ_USETRANSPORTIPV6 */

static NQ_BOOL isModuleInitialized = FALSE;
#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

/* API functions */

NQ_BOOL cmSelfipStart(void)
{
    NQ_BOOL result = TRUE;

    if (isModuleInitialized)
    {
        goto Exit;
    }

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocateStartup(sizeof(*staticData));
    if (NULL == staticData)
    {
        result = FALSE;
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate SelfIp data");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    cmListStart(&staticData->ips);
    isModuleInitialized = TRUE;

Exit:
    return result;
}

void cmSelfipShutdown(void)
{
    if (isModuleInitialized)
    {
        cmListShutdown(&staticData->ips);
#ifdef SY_FORCEALLOCATION
        if (NULL != staticData)
        {
            cmMemoryFreeShutdown(staticData);
        }

        staticData = NULL;
#endif /* SY_FORCEALLOCATION */
        isModuleInitialized = FALSE;
    }
}

void cmSelfipIterate(void)
{
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    syMutexTake(&staticData->ips.guard);

    /* load IPs from system */
    {
        NQ_IPADDRESS4 ip;                   /* next IP address */
        NQ_IPADDRESS6 ip6;                  /* next IPv6 address */
#ifdef UD_NQ_USETRANSPORTIPV6
        NQ_IPADDRESS6 ip6zero;              /* zero IPv6 address */
#endif /* UD_NQ_USETRANSPORTIPV6 */
        NQ_IPADDRESS4 subnet;               /* next subnet mask */
        NQ_IPADDRESS4 bcast;                /* next bcast address */
        NQ_WCHAR wins[UD_NQ_MAXWINSSERVERS * UD_NS_MAXADAPTERS * CM_IP4ADDR_MAXLEN];    /* next WINS IP addresses */
        NQ_WCHAR dns[UD_NQ_MAXDNSSERVERS * UD_NS_MAXADAPTERS * CM_IPADDR_MAXLEN];       /* next DNS IP addresses */
        NQ_INDEX idx;                       /* index in the list of adapters */
        NQ_INDEX osIdx = 0;                 /* index in the OS */
#ifdef CM_NQ_STORAGE
        NQ_BOOL isRdma = FALSE;             /* TRUE for an RDMA-capable adapter */
#endif /* CM_NQ_STORAGE */
        NQ_COUNT numOfWinsServers = cmNetBiosGetNumWinsServers();   /* number of WINS servers that if registered in the system */
#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)
        NQ_COUNT numOfDnsServers = cmDnsGetNumDnsServers();         /* number of DNS servers that if registered in the system */
        NQ_IPADDRESS *toDns;                /* DNS servers temporary IP */
#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */
        NQ_COUNT serversCounter;
        NQ_IPADDRESS4 toWins;               /* WINS servers temporary IP */

        /* empty the list */
        cmListRemoveAndDisposeAll(&staticData->ips);

#ifdef UD_NQ_USETRANSPORTIPV6
        syMemset(ip6, 0, sizeof(ip6));
        syMemset(ip6zero, 0, sizeof(ip6zero));
#endif /* UD_NQ_USETRANSPORTIPV6 */
#ifdef CM_NQ_STORAGE
        for (idx = 0; ((NQ_SUCCESS == syGetAdapter(idx, &osIdx, &isRdma, &ip, &ip6, &subnet, &bcast, &wins)) && (idx < UD_NS_MAXADAPTERS)); idx++)
#else
        for (idx = 0; ((NQ_SUCCESS == syGetAdapter(idx, &osIdx, &ip, &ip6, &subnet, &bcast, wins, dns)) && (idx < UD_NS_MAXADAPTERS)); idx++)
#endif /* CM_NQ_STORAGE */
        {
            SelfIp *pSelf = NULL; /* pointer to next IP entry */

            LOGMSG(CM_TRC_LEVEL_FUNC_COMMON, "Adapter found: ip: %s", cmIPDumpV4(ip));
            LOGMSG(CM_TRC_LEVEL_FUNC_COMMON, "               subnet: %s", cmIPDumpV4(subnet));
            LOGMSG(CM_TRC_LEVEL_FUNC_COMMON, "               bcast: %s", cmIPDumpV4(bcast));

            if (CM_IPADDR_ZERO4 != ip)
            {
                pSelf = (SelfIp *)cmListItemCreateAndAdd(&staticData->ips, sizeof(SelfIp), NULL, NULL, CM_LISTITEM_NOLOCK , FALSE);
                if (NULL == pSelf)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    goto Exit;
                }

                CM_IPADDR_ASSIGN4(pSelf->selfIp.ip, ip);
                pSelf->selfIp.bcast = bcast;
                pSelf->selfIp.subnet = subnet;
                pSelf->selfIp.numWinsServers = 0;
                pSelf->selfIp.numDnsServers = 0;

                /* Add each WINS server to the winsServers list based on the ip subnet */
                for (serversCounter = 0; serversCounter < numOfWinsServers; serversCounter++)
                {
                    toWins = cmNetBiosGetWins(serversCounter);

                    if ((pSelf->selfIp.subnet & toWins) != (pSelf->selfIp.subnet & CM_IPADDR_GET4(pSelf->selfIp.ip)))
                    {
                        continue;
                    }
                    else
                    {
                        /* In case the adapter and the WINS server are on the same subnet add the server to the winsServers list */

                        CM_IPADDR_ASSIGN4(pSelf->selfIp.winsServers[pSelf->selfIp.numWinsServers] , toWins);
                        pSelf->selfIp.numWinsServers++;
                    }
                }

#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)
                /* Add each DNS server to the dnsServers list based on the ip subnet */
                for (serversCounter = 0; serversCounter < numOfDnsServers; serversCounter++)
                {
                    toDns = cmDnsGetServer(serversCounter);

                    if ((pSelf->selfIp.subnet & CM_IPADDR_GET4(*toDns)) != (pSelf->selfIp.subnet & CM_IPADDR_GET4(pSelf->selfIp.ip)))
                    {
                        continue;
                    }
                    else
                    {
                        /* In case the adapter and the DNS server are on the same subnet add the server to the dnsServers list */

                        CM_IPADDR_ASSIGN4(pSelf->selfIp.dnsServers[pSelf->selfIp.numDnsServers] , CM_IPADDR_GET4(*toDns));
                        pSelf->selfIp.numDnsServers++;
                    }
                }
#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */
#ifdef CM_NQ_STORAGE
                pSelf->selfIp.osIndex = osIdx;
                pSelf->selfIp.rdmaCapable = isRdma;
#endif /* CM_NQ_STORAGE */
            }
#ifdef UD_NQ_USETRANSPORTIPV6
            if (0 != syMemcmp(ip6, ip6zero, sizeof(ip6zero)))
            {
                pSelf = (SelfIp *)cmListItemCreateAndAdd(&staticData->ips, sizeof(SelfIp), NULL, NULL, CM_LISTITEM_NOLOCK, FALSE);
                if (NULL == pSelf)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    goto Exit;
                }

                CM_IPADDR_ASSIGN6(pSelf->selfIp.ip, ip6);
                LOGMSG(CM_TRC_LEVEL_FUNC_COMMON, "               ip6: %s", cmIPDump((const NQ_IPADDRESS *)&pSelf->selfIp.ip));
                pSelf->selfIp.bcast = 0L;
                pSelf->selfIp.subnet = subnet;
                pSelf->selfIp.numWinsServers = 0;
                pSelf->selfIp.numDnsServers = 0;
#ifdef CM_NQ_STORAGE
                pSelf->selfIp.osIndex = osIdx;
                pSelf->selfIp.rdmaCapable = FALSE;
#endif /* CM_NQ_STORAGE */
#ifdef UD_NQ_USETRANSPORTIPV4
                /* Add each DNS server to the dnsServers list */
                for (serversCounter = 0; serversCounter < numOfDnsServers; serversCounter++)
                {
                    toDns = cmDnsGetServer(serversCounter);

                    CM_IPADDR_ASSIGN6(pSelf->selfIp.dnsServers[pSelf->selfIp.numDnsServers], CM_IPADDR_GET6(*toDns));
                    pSelf->selfIp.numDnsServers++;
                }
#endif /* UD_NQ_USETRANSPORTIPV4 */
            }

            syMemset(ip6, 0, sizeof(ip6));
#endif /* UD_NQ_USETRANSPORTIPV6 */
        }
    }

    result = TRUE;

Exit:
    syMutexGive(&staticData->ips.guard);
    if (TRUE == result)
    {
        cmListIteratorStart(&staticData->ips, &staticData->iterator);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

const CMSelfIp * cmSelfipNext(void)
{
    const CMSelfIp * pResult = NULL;

    if (cmListIteratorHasNext(&staticData->iterator))
    {
        const SelfIp * selfIp = (const SelfIp *)cmListIteratorNext(&staticData->iterator);
        pResult = &selfIp->selfIp;
    }
    return pResult;
}

void cmSelfipTerminate(void)
{
    cmListIteratorTerminate(&staticData->iterator);  
}

NQ_IPADDRESS * cmSelfipGetZeroIp(void)
{
    return (&zeroIp);
}

NQ_IPADDRESS * cmSelfipGetAnyIpVersion4(void)
{
    return (&anyIPVer4);
}

NQ_IPADDRESS * cmSelfipGetLocalHostIp(void)
{
    return (&localHostIP);
}

#ifdef UD_NQ_USETRANSPORTIPV6
NQ_IPADDRESS * cmSelfipGetAnyIpVersion6(void)
{
    return (&anyIPVer6);
}

NQ_IPADDRESS * cmSelfipGetLocalHostIpVersion6(void)
{
    return (&localHostIPVer6);
}
#endif /* UD_NQ_USETRANSPORTIPV6 */

