/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Common memory operations
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 *************************************************************************/

#ifndef _CMMEMORY_H_ 
#define _CMMEMORY_H_

#include "cmapi.h"

/* Alignment size
 * we will align each memory buffer allocated by SYMalloc to this size.
 * 2 should be the minimum alignment size, 64 might work well for performance since it is memory caching size in many systems */

#define MEM_ALLIGNMENT_BYTES 2

/* -- Functions -- */

/* Description
   This function starts memory management module.
   Parameters
   maxBytes :  Maximum number of bytes that NQ is allowed to
               allocate in total. A zero value means unlimited
               memory usage.
   Returns
   TRUE on success, FALSE on failure.                          */
NQ_BOOL cmMemoryStart(NQ_UINT64 maxBytes);

/* Description
   This function terminates memory management module.
   Returns
   None */
void cmMemoryShutdown(void);

void * cmMemoryAllocNonDebug(NQ_UINT size);

#ifdef UD_NQ_INCLUDETRACE
void * cmMemoryAllocDebug(NQ_UINT size, const NQ_CHAR *text, const NQ_UINT line);
#endif /* UD_NQ_INCLUDETRACE */

void * cmMemoryAllocStartupNonDebug(NQ_UINT size);

#ifdef UD_NQ_INCLUDETRACE
void * cmMemoryAllocStartupDebug(NQ_UINT size, const NQ_CHAR *text, const NQ_UINT line);
#endif /* UD_NQ_INCLUDETRACE */

#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryAllocate(_size) cmMemoryAllocDebug(_size, SY_LOG_FUNCTION, SY_LOG_LINE) 
#else
#define cmMemoryAllocate(_size) cmMemoryAllocNonDebug(_size)
#endif /* UD_NQ_INCLUDETRACE */

#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryAllocateStartup(_size) cmMemoryAllocStartupDebug(_size, SY_LOG_FUNCTION, SY_LOG_LINE)
#else
#define cmMemoryAllocateStartup(_size) cmMemoryAllocStartupNonDebug(_size)
#endif /* UD_NQ_INCLUDETRACE */


/* Description
   This function allocates a block of memory.
   Parameters
   size :  Number of bytes to allocate.
   Returns
   Pointer to the block or NULL on error.   */
/*void * cmMemoryAllocate(NQ_UINT size);*/


#ifdef UD_NQ_INCLUDETRACE
NQ_WCHAR * cmMemoryCloneWStringDebug(const NQ_WCHAR * str, const NQ_CHAR *function, const NQ_UINT line);
#endif /* UD_NQ_INCLUDETRACE */
NQ_WCHAR * cmMemoryCloneWStringNonDebug(const NQ_WCHAR * str);

#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryCloneWString(_str) cmMemoryCloneWStringDebug(_str, SY_LOG_FUNCTION, SY_LOG_LINE) 
#else
#define cmMemoryCloneWString(_str) cmMemoryCloneWStringNonDebug(_str)
#endif /* UD_NQ_INCLUDETRACE */

/* Description
   This function creates a string in allocated memory with the
   same contents as the original string.
   
   It is caller's responsibility to release this string.
   Parameters
   str :  The original string.
   Returns
   Pointer to new string or NULL on error.                     */
/*NQ_WCHAR * cmMemoryCloneWString(const NQ_WCHAR * str);*/

#ifdef UD_NQ_INCLUDETRACE
NQ_WCHAR * cmMemoryCloneAStringDebug(const NQ_CHAR * str, const NQ_CHAR *function, const NQ_UINT line);
#endif /* UD_NQ_INCLUDETRACE */
NQ_WCHAR * cmMemoryCloneAStringNonDebug(const NQ_CHAR * str);

#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryCloneAString(_str) cmMemoryCloneAStringDebug(_str, SY_LOG_FUNCTION, SY_LOG_LINE) 
#else
#define cmMemoryCloneAString(_str) cmMemoryCloneAStringNonDebug(_str)
#endif /* UD_NQ_INCLUDETRACE */

/* Description
   This function creates a string in allocated memory with the
   same contents as the original string.

   It is caller's responsibility to release this string.
   Parameters
   str :  The original string.
   Returns
   Pointer to new string or NULL on error.                     */
/*NQ_CHAR * cmMemoryCloneAStringAsAscii(const NQ_CHAR * str);*/

#ifdef UD_NQ_INCLUDETRACE
NQ_CHAR * cmMemoryCloneAStringAsAsciiDebug(const NQ_CHAR * str, const NQ_CHAR *function, const NQ_UINT line);
#endif /* UD_NQ_INCLUDETRACE */
NQ_CHAR * cmMemoryCloneAStringAsAsciiNonDebug(const NQ_CHAR * str);

#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryCloneAStringAsAscii(_str) cmMemoryCloneAStringAsAsciiDebug(_str, SY_LOG_FUNCTION, SY_LOG_LINE)
#else
#define cmMemoryCloneAStringAsAscii(_str) cmMemoryCloneAStringAsAsciiNonDebug(_str)
#endif /* UD_NQ_INCLUDETRACE */

/* Description
   This function creates a string in allocated memory with the
   same contents as the original ASCII string after conversion.
   
   It is caller's responsibility to release this string.
   Parameters
   str :  The original string.
   Returns
   Pointer to new string or NULL on error.                      */
/*NQ_WCHAR * cmMemoryCloneAString(const NQ_CHAR * str);*/

/* Description
   This function creates an ASCII string in allocated memory
   with the same contents as the original string.
   
   It is caller's responsibility to release this string.
   Parameters
   str :  The original string.
   Returns
   Pointer to new string or NULL on error.                   */
/*NQ_CHAR * cmMemoryCloneWStringAsAscii(const NQ_WCHAR * str);*/
#ifdef UD_NQ_INCLUDETRACE
NQ_CHAR * cmMemoryCloneWStringAsAsciiDebug(const NQ_WCHAR * str, const NQ_CHAR *function, const NQ_UINT line);
#endif /* UD_NQ_INCLUDETRACE */
NQ_CHAR * cmMemoryCloneWStringAsAsciiNonDebug(const NQ_WCHAR * str);

#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryCloneWStringAsAscii(_str) cmMemoryCloneWStringAsAsciiDebug(_str, SY_LOG_FUNCTION, SY_LOG_LINE) 
#else
#define cmMemoryCloneWStringAsAscii(_str) cmMemoryCloneWStringAsAsciiNonDebug(_str)
#endif /* UD_NQ_INCLUDETRACE */

CMIOBlob memoryCloneIOBlob(const CMIOBlob * origin, const NQ_CHAR *function, const NQ_UINT line);
CMBlob memoryCloneBlob(const CMBlob * origin, const NQ_CHAR *function, const NQ_UINT line);
CMIOBlob memoryCloneBlobToIOBlob(const CMBlob * origin, const NQ_CHAR *function, const NQ_UINT line);
CMBlob memoryCloneIOBlobToBlob(const CMIOBlob * origin, const NQ_CHAR *function, const NQ_UINT line);

/* Description
   This function creates a new blob and copies into it the contents of the original blob.
   
   It is caller's responsibility to release new blob data.
   Parameters
   origin :  The original blob.
   Returns
   New blob with allocated data or NULL data on error.                   */
/*CMIOBlob cmMemoryCloneBlob(const CMIOBlob * origin);*/
#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryCloneIOBlob(_origin) memoryCloneIOBlob(_origin, SY_LOG_FUNCTION, SY_LOG_LINE);
#define cmMemoryCloneBlob(_origin)   memoryCloneBlob(_origin, SY_LOG_FUNCTION, SY_LOG_LINE);
#define cmMemoryCloneBlobtoIOblob(_origin)   memoryCloneBlobToIOBlob(_origin, SY_LOG_FUNCTION, SY_LOG_LINE);
#define cmMemoryCloneIOBlobToBlob(_origin)   memoryCloneIOBlobToBlob(_origin, SY_LOG_FUNCTION, SY_LOG_LINE);
#else
#define cmMemoryCloneIOBlob(_origin) memoryCloneIOBlob(_origin, NULL, 0);
#define cmMemoryCloneBlob(_origin)   memoryCloneBlob(_origin, NULL, 0);
#define cmMemoryCloneBlobtoIOblob(_origin) memoryCloneBlobToIOBlob(_origin, NULL, 0);
#define cmMemoryCloneIOBlobToBlob(_origin) memoryCloneIOBlobToBlob(_origin, NULL, 0);
#endif /* UD_NQ_INCLUDETRACE */

/* Description
   This function disposes blob data and sets its pointer to NULL.
   
   Parameters
   blob :  The original blob.
   Returns
   None.                   */

void memoryFreeBlob(CMBlob * blob, const NQ_CHAR *function, const NQ_UINT line);
void memoryFreeIOBlob(CMIOBlob * blob, const NQ_CHAR *function, const NQ_UINT line);

#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryFreeBlob(_blob) memoryFreeBlob(_blob, SY_LOG_FUNCTION, SY_LOG_LINE);
#define cmMemoryFreeIOBlob(_blob) memoryFreeIOBlob(_blob, SY_LOG_FUNCTION, SY_LOG_LINE);
#else
#define cmMemoryFreeBlob(_blob)   memoryFreeBlob(_blob, NULL, 0);
#define cmMemoryFreeIOBlob(_blob) memoryFreeIOBlob(_blob, NULL, 0);
#endif /* UD_NQ_INCLUDETRACE */

/* Description
   This function disposes a block of memory.

   Parameters
   block :  Pointer to a block.

   Returns
   None                                      */
/*void cmMemoryFree(const void * block);*/
#ifdef UD_NQ_INCLUDETRACE
void cmMemoryFreeDebug(const void * block, const NQ_CHAR *function, const NQ_UINT line);
#endif /* UD_NQ_INCLUDETRACE */
void cmMemoryFreeNonDebug(const void * block);

void cmMemoryFreeShutdownNonDebug(const void * block);
#ifdef UD_NQ_INCLUDETRACE
void cmMemoryFreeShutdownDebug(const void * block, const NQ_CHAR *function, const NQ_UINT line);
#endif /* UD_NQ_INCLUDETRACE */

#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryFree(_block) cmMemoryFreeDebug(_block, SY_LOG_FUNCTION, SY_LOG_LINE) 
#else
#define cmMemoryFree(_block) cmMemoryFreeNonDebug(_block)
#endif /* UD_NQ_INCLUDETRACE */

#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryFreeShutdown(_block) cmMemoryFreeShutdownDebug(_block, SY_LOG_FUNCTION, SY_LOG_LINE)
#else
#define cmMemoryFreeShutdown(_block) cmMemoryFreeShutdownNonDebug(_block)
#endif /* UD_NQ_INCLUDETRACE */

/* Description
   This function reports memory usage statistics.

   Parameters
   memMaxAllowed : Number of bytes allowed (0 returned for unlimited).
   memInUse :      Number of bytes allocated so far.
   Returns
   None.               */
void cmMemoryStatistics(NQ_UINT64 * memMaxAllowed, NQ_UINT64 * memInUse);

#ifdef UD_NQ_INCLUDETRACE

/* Description
   This function reports full memory usage statistics.
   
   It is available in debug mode only.
   Parameters
   memMaxAllowed : Number of bytes allowed (0 returned for unlimited).
   memAlloc :      Number of bytes allocated so far.
   memDedealloc :  Number of bytes deallocated so far.
   blockAlloc :    Number of blocks allocated so far.
   blockDealloc :  Number of blocks deallocated so far.
   Returns
   None.               */
void cmMemoryStatisticsFull(NQ_UINT64 * memMaxAllowed, NQ_UINT64 * memAlloc, NQ_UINT32 * memDealloc, NQ_UINT32 * blockAlloc, NQ_UINT32 * blockDealloc);

/* Description
   This function prints memory usage statistics.
   
   It is available in debug mode only.
   Returns
   None.               */
void cmMemoryDump(void);

#endif /* UD_NQ_INCLUDETRACE */

#endif /* _CMMEMORY_H_ */
