/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Cryptographic library and password handling
 *                 contains DES, MD4 and MD5 algorithms implementation
 *                 and higher level LM, NTLM, LMv2 and NTLMv2 security
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  : None
 ********************************************************************/

#include "cmfscifs.h"
#include "cmcrypt.h"
#include "cmbufman.h"
#include "cmsmb2.h"
#include "aminaesgcm.h"

#define NTLM_CLIENT_SIGN_CONST "session key to client-to-server signing key magic constant"
#define NTLM_CLIENT_SEAL_CONST "session key to client-to-server sealing key magic constant"
#define NTLM_SERVER_SIGN_CONST "session key to server-to-client signing key magic constant"
#define NTLM_SERVER_SEAL_CONST "session key to server-to-client sealing key magic constant"

static const CMCrypterList internalCrypters = 
{
    md4Internal,
    md5Internal,
    hmacmd5Internal,
#ifdef UD_NQ_INCLUDESMB2
    sha256Internal,
#ifdef UD_NQ_INCLUDESMB3
    aes128cmacInternal,
    sha512Internal,
    aes128ccmEncryptionInternal,
    aes128ccmDecryptionInternal,
    aes128GcmEncryptInternal,
    aes128GcmDecryptInternal
#else /* UD_NQ_INCLUDESMB3 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
#endif /* UD_NQ_INCLUDESMB3 */
#else /* UD_NQ_INCLUDESMB2 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
#endif /* UD_NQ_INCLUDESMB2 */
};

static CMCrypterList currentCrypters = 
{
    md4Internal,
    md5Internal,
    hmacmd5Internal,
#ifdef UD_NQ_INCLUDESMB2
    sha256Internal,
#ifdef UD_NQ_INCLUDESMB3
    aes128cmacInternal,
    sha512Internal,
    aes128ccmEncryptionInternal,
    aes128ccmDecryptionInternal,
    aes128GcmEncryptInternal,
    aes128GcmDecryptInternal
#else /* UD_NQ_INCLUDESMB3 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
#endif /* UD_NQ_INCLUDESMB3 */
#else /* UD_NQ_INCLUDESMB2 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
#endif /* UD_NQ_INCLUDESMB2 */
};

void cmSetExternalCrypters(const CMCrypterList * newCrypters)
{
    if (NULL == newCrypters)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        return;
    }

    if (newCrypters->md4 != NULL)
    {
        currentCrypters.md4 = newCrypters->md4;
    }

    if (newCrypters->md5 != NULL)
    {
        currentCrypters.md5 = newCrypters->md5;
    }

    if (newCrypters->hmacmd5 != NULL)
    {
        currentCrypters.hmacmd5 = newCrypters->hmacmd5;
    }

    if (newCrypters->sha256 != NULL)
    {
        currentCrypters.sha256 = newCrypters->sha256;
    }

    if (newCrypters->aes128cmac != NULL)
    {
        currentCrypters.aes128cmac = newCrypters->aes128cmac;
    }

    if (newCrypters->sha512 != NULL)
    {
        currentCrypters.sha512 = newCrypters->sha512;
    }

    if (newCrypters->aes128ccmEncryption != NULL)
    {
        currentCrypters.aes128ccmEncryption = newCrypters->aes128ccmEncryption;
    }

    if (newCrypters->aes128ccmDecryption != NULL)
    {
        currentCrypters.aes128ccmDecryption = newCrypters->aes128ccmDecryption;
    }

    if (newCrypters->aes128gcmEncryption != NULL)
    {
        currentCrypters.aes128gcmEncryption = newCrypters->aes128gcmEncryption;
    }

    if (newCrypters->aes128gcmDecryption != NULL)
    {
        currentCrypters.aes128gcmDecryption = newCrypters->aes128gcmDecryption;
    }
}

void cmResetExternalCrypters(void)
{
    currentCrypters = internalCrypters;
}

/*
 *====================================================================
 * PURPOSE: Calculate MD4 wrapper
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT signature buffer
 *          IN     data to sign
 *          IN     data length
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmMD4(
        NQ_BYTE *out,
        NQ_BYTE *in,
        NQ_UINT   n
      )
{
    (*currentCrypters.md4)(in, out, n);
}

/*
 *====================================================================
 * PURPOSE: Calculate MD5 wrapper
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT signature buffer
 *          IN     data to sign
 *          IN     data length
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmMD5(
        NQ_BYTE *out,
        NQ_BYTE *in,
        NQ_UINT n
      )
{
    CMIOBlob fragment;
    NQ_IOBufPos bufInPos;
    IOBUF_POSCONSTRUCTORINIT(bufInPos)

    IOBUF_POSCONSTRUCTOR(bufInPos, in, n)
    fragment.data = bufInPos;
    fragment.len = n;

    (*currentCrypters.md5)(NULL, NULL, &fragment, 1,out, 16);
}

/*
 *====================================================================
 * PURPOSE: Calculate MD5 wrapper using number of data fragments
 *--------------------------------------------------------------------
 * PARAMS:  IN     key
 *          IN     key1
 *          IN     array of data fragments
 *          IN     number of data fragments
 *          IN/OUT buffer for result
 *          IN     size of buffer
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmMD5Fragments(const CMBlob * key, const CMBlob * key1, const CMIOBlob dataFragments[], NQ_COUNT numFragments, NQ_BYTE * buffer, NQ_COUNT bufferSize)
{
    (*currentCrypters.md5)(key, key1, dataFragments, numFragments, buffer, bufferSize);
}

/*
 *====================================================================
 * PURPOSE: Calculate HMACMD5 wrapper using number of data fragments
 *--------------------------------------------------------------------
 * PARAMS:  IN     key
 *          IN     key1
 *          IN     array of data fragments
 *          IN     number of data fragments
 *          IN/OUT buffer for result
 *          IN     size of buffer
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmHMACMD5Fragments(const CMBlob * key, const CMBlob * key1, const CMIOBlob dataFragments[], NQ_COUNT numFragments, NQ_BYTE * buffer, NQ_COUNT bufferSize)
{
    (*currentCrypters.hmacmd5)(key, key1, dataFragments, numFragments, buffer, bufferSize);
}

/*
 *====================================================================
 * PURPOSE: MD5 hash function based message authentication code impl.
 *--------------------------------------------------------------------
 * PARAMS:  IN     key
 *          IN     key length
 *          IN     data to sign
 *          IN     data length
 *          IN/OUT signature buffer
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmHMACMD5(
    const NQ_BYTE *key,
    NQ_UINT keyLen,
    const NQ_BYTE *data,
    NQ_UINT data_len,
    NQ_BYTE *md
    )
{
    CMIOBlob fragments[1];
    CMBlob keyBlob;
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)

    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE *)data, data_len)
    fragments[0].data = bufPos;
    fragments[0].len = data_len;
    keyBlob.data = (NQ_BYTE *)key;
    keyBlob.len = keyLen;

    (*currentCrypters.hmacmd5)(&keyBlob, NULL, fragments, 1, md, 16);
}

/*
 *====================================================================
 * PURPOSE: create v2 hash
 *--------------------------------------------------------------------
 * PARAMS:  IN  domain name
 *          IN  user name
 *          IN  *hashed* password
 *          OUT v2 hash
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void cmCreateV2Hash(
    const NQ_WCHAR *domain,
    NQ_BOOL caseSensitiveDomain,
    const NQ_WCHAR *user,
    const NQ_BYTE  *password,
    NQ_UINT pwdlen,
    NQ_BYTE *hash
   )
{
    NQ_WCHAR data[CM_BUFFERLENGTH(NQ_WCHAR, CM_USERNAMELENGTH + CM_NQ_HOSTNAMESIZE)];

    cmWStrcpy(data, user);

    if (caseSensitiveDomain)
        cmWStrupr(data);

    cmWStrcpy(data + cmWStrlen(data), domain);

    if (!caseSensitiveDomain)
        cmWStrupr(data);

    cmHMACMD5(password, pwdlen, (NQ_BYTE*)data, (NQ_UINT)(cmWStrlen(data) * sizeof(NQ_WCHAR)), hash);
}


#define SHA256_DIGEST_SIZE 32
#define SHA256_BLOCK_SIZE  64

/*
 *====================================================================
 * PURPOSE: Calculate hash with SHA256
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT 
 *          IN     
 *          IN     
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmSmb2CalculateMessageSignature(
        const NQ_BYTE *key,
        NQ_UINT keyLen,
        NQ_IOBufPos buffer1,
        NQ_UINT size1,
        NQ_IOBufPos buffer2,
        NQ_UINT size2,
        NQ_BYTE *signature
        )
{
    /* signature buffer must be 16 bytes long */
    NQ_BYTE ipad[SHA256_BLOCK_SIZE];
    NQ_BYTE opad[SHA256_BLOCK_SIZE];
    NQ_BYTE hash[SHA256_DIGEST_SIZE];
    NQ_UINT i;
    CMIOBlob    frag1[3];
    CMIOBlob    frag2[2];
    NQ_IOBufPos ipadPos;
    NQ_IOBufPos opadPos;
    NQ_IOBufPos hashPos;
    IOBUF_POSCONSTRUCTORINIT(ipadPos)
    IOBUF_POSCONSTRUCTORINIT(opadPos)
    IOBUF_POSCONSTRUCTORINIT(hashPos)

    IOBUF_POSCONSTRUCTOR(ipadPos, ipad, SHA256_BLOCK_SIZE)
    IOBUF_POSCONSTRUCTOR(opadPos, opad, SHA256_BLOCK_SIZE)
    IOBUF_POSCONSTRUCTOR(hashPos, hash, SHA256_DIGEST_SIZE)

    frag1[0].data = ipadPos;
    frag1[0].len = sizeof(ipad);
    frag1[1].data = buffer1;
    frag1[1].len = size1;
    frag1[2].data = buffer2;
    frag1[2].len = size2;
    frag2[0].data = opadPos;
    frag2[0].len = sizeof(opad);
    frag2[1].data = hashPos;
    frag2[1].len = sizeof(hash);

    LOGFB(CM_TRC_LEVEL_CRYPT, "key:%p keyLen:%u buffer1:%p size1:%u buffer2:%p size2:%u signature:%p", key, keyLen, buffer1, size1, buffer2, size2, signature);

    LOGDUMP(CM_TRC_LEVEL_CRYPT, "key", key, (NQ_UINT)keyLen);

    syMemset(ipad, 0x36, sizeof(ipad));
    syMemset(opad, 0x5C, sizeof(opad));

    for (i = 0; i < keyLen; ++i)
    {
        ipad[i] ^= key[i];
        opad[i] ^= key[i];
    }

    syMemset(signature, 0, SMB2_SECURITY_SIGNATURE_SIZE);

    (*currentCrypters.sha256)(NULL, NULL, frag1, 3, hash, sizeof(hash));
    (*currentCrypters.sha256)(NULL, NULL, frag2, 2, hash, sizeof(hash));

    syMemcpy(signature, hash, SMB2_SECURITY_SIGNATURE_SIZE);
    LOGDUMP(CM_TRC_LEVEL_CRYPT, "signature", signature, SMB2_SECURITY_SIGNATURE_SIZE);

    LOGFE(CM_TRC_LEVEL_CRYPT);
}

/*
 *====================================================================
 * PURPOSE: Encrypt hashed NTLM password v2
 *--------------------------------------------------------------------
 * PARAMS:  IN  working connection
 *          IN  hashed NTLM password
 *          OUT encrypted password length
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void cmEncryptNTLMv2Password(
        const NQ_BYTE *key,
        const NQ_BYTE *v2hash,
        const NQ_BYTE *blob,
        NQ_UINT16 bloblen,
        NQ_BYTE   *encrypted,
        NQ_UINT16 *enclen
        )
{
    CMBlob  v2Hkey;
    CMIOBlob fragments[2];
    NQ_IOBufPos bufKeyPos;
    NQ_IOBufPos bufBlobPos;
    IOBUF_POSCONSTRUCTORINIT(bufKeyPos)
    IOBUF_POSCONSTRUCTORINIT(bufBlobPos)

    IOBUF_POSCONSTRUCTOR(bufKeyPos, (NQ_BYTE *)key, 8)
    IOBUF_POSCONSTRUCTOR(bufBlobPos, (NQ_BYTE *)blob, bloblen)
    v2Hkey.data = (NQ_BYTE *)v2hash;
    v2Hkey.len = 16;
    fragments[0].data = bufKeyPos;
    fragments[0].len = 8;
    fragments[1].data = bufBlobPos;
    fragments[1].len = bloblen;

    (*currentCrypters.hmacmd5)(&v2Hkey, NULL, fragments, 2, encrypted, CM_CRYPT_ENCLMv2HMACSIZE);
    *enclen = CM_CRYPT_ENCLMv2HMACSIZE;
}

/*
 *====================================================================
 * PURPOSE: Generate extended security session key
 *--------------------------------------------------------------------
 * PARAMS:  IN start of the client response
 *          IN pointer to NTLM hashed password
 *          IN/OUT encrypted password
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmGenerateExtSecuritySessionKey(
        const NQ_BYTE* v2hash,
        const NQ_BYTE* encrypted,
        NQ_BYTE* out
        )
{
    CMBlob  key;
    CMIOBlob fragment;
    NQ_IOBufPos bufEncryptedPos;
    IOBUF_POSCONSTRUCTORINIT(bufEncryptedPos)

    LOGFB(CM_TRC_LEVEL_CRYPT, "v2hash:%p encrypted:%p out:%p", v2hash, encrypted, out);

    key.data = (NQ_BYTE *)v2hash;
    key.len = 16;
    IOBUF_POSCONSTRUCTOR(bufEncryptedPos, (NQ_BYTE *)encrypted, 16)
    fragment.data = bufEncryptedPos;
    fragment.len = 16;

    (*currentCrypters.hmacmd5)(&key, NULL, &fragment, 1, out, 16);

    LOGFE(CM_TRC_LEVEL_CRYPT);
}

/*
 *====================================================================
 * PURPOSE: Calculate session key to be used in secure channel (strong key, md5)
 *--------------------------------------------------------------------
 * PARAMS:
 *          IN     in client challenge (8 bytes)
 *          IN     in server challenge (8 bytes)
 *          IN     key (16 bytes)
 *          OUT    session key (16 bytes)
 *
 * RETURNS: none
 *====================================================================
 */
void cmNetlogonCredentialsGenerateSessKey(
         const NQ_BYTE *clientChallenge,
         const NQ_BYTE *serverChallenge,
         const NQ_BYTE *key,  /* secret */
         NQ_BYTE *sessKey
         )
{
    NQ_BYTE zero[4], temp[16];
    CMIOBlob frag1[3];
    CMIOBlob frag2;
    CMBlob keyBlob;
    NQ_IOBufPos bufZeroPos;
    NQ_IOBufPos bufClientChPos;
    NQ_IOBufPos bufSrvChPos;
    NQ_IOBufPos bufTmpPos;
    IOBUF_POSCONSTRUCTORINIT(bufZeroPos)
    IOBUF_POSCONSTRUCTORINIT(bufClientChPos)
    IOBUF_POSCONSTRUCTORINIT(bufSrvChPos)
    IOBUF_POSCONSTRUCTORINIT(bufTmpPos)

    IOBUF_POSCONSTRUCTOR(bufZeroPos, zero, sizeof(zero))
    IOBUF_POSCONSTRUCTOR(bufClientChPos, (NQ_BYTE *)clientChallenge, 8)
    IOBUF_POSCONSTRUCTOR(bufSrvChPos, (NQ_BYTE *)serverChallenge, 8)
    IOBUF_POSCONSTRUCTOR(bufTmpPos, temp, 16)

    frag1[0].data = bufZeroPos;
    frag1[0].len = 4;
    frag1[1].data = bufClientChPos;
    frag1[1].len = 8;
    frag1[2].data = bufSrvChPos;
    frag1[2].len = 8;
    frag2.data = bufTmpPos;
    frag2.len = 16; 
    keyBlob.data = (NQ_BYTE *)key;
    keyBlob.len = 16;

    /* Generate the session key */
    syMemset(sessKey, 0, 16);
    syMemset(zero, 0, sizeof(zero));

    (*currentCrypters.md5)(NULL, NULL, frag1, 3, temp, 16);
    (*currentCrypters.hmacmd5)(&keyBlob, NULL, &frag2, 1, sessKey, 16);
}

/*
 *====================================================================
 * PURPOSE: Create next Netlogon credentials in chain
 *--------------------------------------------------------------------
 * PARAMS:
 *          IN     whether to use AES algorithm
 *          IN     session key (16 bytes)
 *          IN     in challenge (8 bytes)
 *          OUT    out challenge (8 bytes)
 *
 * RETURNS: none
 *====================================================================
 */
void cmNetlogonCredentialsCryptNext(NQ_BOOL doAES, const NQ_BYTE *sessKey, const NQ_BYTE *inChallenge, NQ_BYTE *outChallenge)
{
    if (TRUE == doAES)
    {
        NQ_BYTE initVector[16] = {0};

        AES_128_CFB8_Encrypt(sessKey, initVector, inChallenge, 8, outChallenge);
    }
    else
    {
        cmDES112(outChallenge, inChallenge, sessKey);
    }
}

/*
 *====================================================================
 * PURPOSE: Calculate session key to be used in secure channel (AES)
 *--------------------------------------------------------------------
 * PARAMS:
 *          IN     in client challenge (8 bytes)
 *          IN     in server challenge (8 bytes)
 *          IN     key (16 bytes)
 *          OUT    session key (16 bytes)
 *
 * RETURNS: none
 *====================================================================
 */
void cmNetlogonCredentialsGenerateSessKeyEx(const NQ_BYTE *clientChallenge, const NQ_BYTE *serverChallenge, const NQ_BYTE *key, NQ_BYTE *sessKey)
{
    NQ_BYTE hash[SHA256_DIGEST_SIZE];
    NQ_BYTE ipad[SHA256_BLOCK_SIZE];
    NQ_BYTE opad[SHA256_BLOCK_SIZE];
    NQ_UINT i;
    CMIOBlob frag1[3];
    CMIOBlob frag2[2];
    NQ_IOBufPos cliChlPos;
    NQ_IOBufPos srvChlPos;
    NQ_IOBufPos ipadPos;
    NQ_IOBufPos opadPos;
    NQ_IOBufPos hashPos;
    IOBUF_POSCONSTRUCTORINIT(cliChlPos)
    IOBUF_POSCONSTRUCTORINIT(srvChlPos)
    IOBUF_POSCONSTRUCTORINIT(ipadPos)
    IOBUF_POSCONSTRUCTORINIT(opadPos)
    IOBUF_POSCONSTRUCTORINIT(hashPos)

    IOBUF_POSCONSTRUCTOR(cliChlPos, (NQ_IOBufPos)clientChallenge, 8)
    IOBUF_POSCONSTRUCTOR(srvChlPos, (NQ_IOBufPos)serverChallenge, 8)
    IOBUF_POSCONSTRUCTOR(ipadPos, ipad, SHA256_BLOCK_SIZE)
    IOBUF_POSCONSTRUCTOR(opadPos, opad, SHA256_BLOCK_SIZE)
    IOBUF_POSCONSTRUCTOR(hashPos, hash, SHA256_DIGEST_SIZE)

    syMemset(ipad, 0x36, sizeof(ipad));
    syMemset(opad, 0x5C, sizeof(opad));

    for (i = 0; i < 16; ++i)
    {
        ipad[i] ^= key[i];
        opad[i] ^= key[i];
    }

    frag1[0].data = ipadPos;
    frag1[0].len = sizeof(ipad);
    frag1[1].data = cliChlPos;
    frag1[1].len = 8;
    frag1[2].data = srvChlPos;
    frag1[2].len = 8;

    frag2[0].data = opadPos;
    frag2[0].len = sizeof(opad);
    frag2[1].data = hashPos;
    frag2[1].len = sizeof(hash);

    (*currentCrypters.sha256)(NULL, NULL, frag1, 3, hash, sizeof(hash));
    (*currentCrypters.sha256)(NULL, NULL, frag2, 2, hash, sizeof(hash));

    /* Use 16 hash bytes as session key */
    syMemcpy(sessKey, hash, 16);
}

/*
 *====================================================================
 * PURPOSE: Netlogon schannel - sign the data (compute the checksum) (AES)
 *--------------------------------------------------------------------
 * PARAMS:
 *          IN     in session key (16 bytes)
 *          IN     in header (8 bytes)
 *          IN     in confounder (8 bytes)
 *          IN     in data to sign
 *          IN     in length of data to sign
 *          OUT    checksum (digest) (8 bytes)
 *
 * RETURNS: none
 *====================================================================
 */
void cmNetlogonSchannelSign(const NQ_BYTE *sessKey, const NQ_BYTE *header,
                            const NQ_BYTE *confounder, const NQ_BYTE *data, NQ_UINT length, NQ_BYTE *checksum)
{
    NQ_BYTE hash[SHA256_DIGEST_SIZE];
    NQ_BYTE ipad[SHA256_BLOCK_SIZE];
    NQ_BYTE opad[SHA256_BLOCK_SIZE];
    NQ_UINT i;
    CMIOBlob frag1[4];  /* 3 or 4 when confounder */
    CMIOBlob frag2[2];
    NQ_IOBufPos headerPos;
    NQ_IOBufPos confounderPos;
    NQ_IOBufPos dataPos;
    NQ_IOBufPos ipadPos;
    NQ_IOBufPos opadPos;
    NQ_IOBufPos hashPos;
    IOBUF_POSCONSTRUCTORINIT(headerPos)
    IOBUF_POSCONSTRUCTORINIT(confounderPos)
    IOBUF_POSCONSTRUCTORINIT(dataPos)
    IOBUF_POSCONSTRUCTORINIT(ipadPos)
    IOBUF_POSCONSTRUCTORINIT(opadPos)
    IOBUF_POSCONSTRUCTORINIT(hashPos)

    IOBUF_POSCONSTRUCTOR(headerPos, (NQ_IOBufPos)header, 8)
    IOBUF_POSCONSTRUCTOR(confounderPos, (NQ_IOBufPos)confounder, 8)
    IOBUF_POSCONSTRUCTOR(dataPos, (NQ_IOBufPos)data, length)
    IOBUF_POSCONSTRUCTOR(ipadPos, ipad, SHA256_BLOCK_SIZE)
    IOBUF_POSCONSTRUCTOR(opadPos, opad, SHA256_BLOCK_SIZE)
    IOBUF_POSCONSTRUCTOR(hashPos, hash, SHA256_DIGEST_SIZE)

    syMemset(ipad, 0x36, sizeof(ipad));
    syMemset(opad, 0x5C, sizeof(opad));

    for (i = 0; i < 16; ++i)
    {
        ipad[i] ^= sessKey[i];
        opad[i] ^= sessKey[i];
    }

    i = 0;
    frag1[i].data = ipadPos;
    frag1[i++].len = sizeof(ipad);
    frag1[i].data = headerPos;
    frag1[i++].len = 8;
    if (NULL != confounder)
    {
        frag1[i].data = confounderPos;
        frag1[i++].len = 8;
    }
    frag1[i].data = dataPos;
    frag1[i++].len = length;

    frag2[0].data = opadPos;
    frag2[0].len = sizeof(opad);
    frag2[1].data = hashPos;
    frag2[1].len = sizeof(hash);

    (*currentCrypters.sha256)(NULL, NULL, frag1, i, hash, sizeof(hash));
    (*currentCrypters.sha256)(NULL, NULL, frag2, 2, hash, sizeof(hash));

    /* Use first 8 bytes out of hash */
    syMemcpy(checksum, hash, 8);
}

/*
 *====================================================================
 * PURPOSE: 
 *--------------------------------------------------------------------
 * PARAMS:  IN
 *          IN
 *          IN/OUT
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmSmbCalculateMessageSignature(
    const NQ_BYTE *key,
    NQ_UINT keyLen,
    NQ_UINT32      sequence,
    const NQ_IOBufPos buffer1,
    NQ_UINT size1,
    const NQ_IOBufPos buffer2,
    NQ_UINT size2,
    const NQ_BYTE *password,
    NQ_COUNT       passwordLen,
    NQ_IOBufPos signature
    )
{
    NQ_BYTE hash[16];
    NQ_UINT32 sn = cmHtol32(sequence);
    CMIOBlob    fragments[4];
    NQ_IOBufPos bufKeyPos;
    NQ_IOBufPos bufPassPos;
    IOBUF_POSCONSTRUCTORINIT(bufKeyPos)
    IOBUF_POSCONSTRUCTORINIT(bufPassPos)

    IOBUF_POSCONSTRUCTOR(bufKeyPos, (NQ_BYTE *)key, keyLen)
    IOBUF_POSCONSTRUCTOR(bufPassPos, (NQ_BYTE *)password, passwordLen)
    fragments[0].data = bufKeyPos;
    fragments[0].len = keyLen;
    fragments[1].data = bufPassPos;
    fragments[1].len = passwordLen;
    fragments[2].data = (NQ_IOBufPos)buffer1;
    fragments[2].len = size1;
    fragments[3].data = (NQ_IOBufPos)buffer2;
    fragments[3].len = size2;
    
    LOGFB(CM_TRC_LEVEL_CRYPT, "key:%p keyLen:%u sequence:%u buffer1:%p size1:%u buffer2:%p size2:%u password:%p passwordLen:%d signature:%p", key, keyLen, sequence, buffer1, size1, buffer2, size2, password, passwordLen, signature);
    
    LOGMSG(CM_TRC_LEVEL_CRYPT, "seq: %lu", (NQ_ULONG)sequence);
    LOGDUMP(CM_TRC_LEVEL_CRYPT, "key", key, (NQ_UINT)keyLen);

    IOBUF_MEMSET(signature, 0, SMB_SECURITY_SIGNATURE_LENGTH);
    cmPutUint32(IOBUF_GETBYTEPTR(signature), sn);

    (*currentCrypters.md5)(NULL , NULL, fragments , 4 , hash , SMB_SECURITY_SIGNATURE_LENGTH);

    IOBUF_MEMCPY_F2V(signature, hash, SMB_SECURITY_SIGNATURE_LENGTH);

    LOGDUMP(CM_TRC_LEVEL_CRYPT, "signature", hash, SMB_SECURITY_SIGNATURE_LENGTH);
    LOGFE(CM_TRC_LEVEL_CRYPT);
}

/*
 *====================================================================
 * PURPOSE: 
 *--------------------------------------------------------------------
 * PARAMS:  IN
 *          IN
 *          IN/OUT
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmSmb311CalcMessagesHash(    
        NQ_IOBufPos buffer,
        NQ_UINT size,   
        NQ_BYTE *digest,
        NQ_BYTE *ctxBuff
        )
{
    CMIOBlob fragments[2];
    NQ_IOBufPos ioBufPosTmp;
    IOBUF_POSCONSTRUCTORINIT(ioBufPosTmp)

    LOGFB(CM_TRC_LEVEL_CRYPT, "buffer:%p size:%u digest:%p ctxBuff:%p", buffer, size, digest, ctxBuff);

    IOBUF_POSCONSTRUCTOR(ioBufPosTmp, digest, SMB3_PREAUTH_INTEG_HASH_LENGTH)

    /* start with incoming digest result - 1st round its all zero */
    fragments[0].data = ioBufPosTmp;
    fragments[0].len = SMB3_PREAUTH_INTEG_HASH_LENGTH;

    /* buffer */ 
    fragments[1].data = buffer;
    fragments[1].len = size;

    (*currentCrypters.sha512)(NULL, NULL, fragments, 2, ioBufPosTmp, SMB3_PREAUTH_INTEG_HASH_LENGTH, ctxBuff);
    LOGFE(CM_TRC_LEVEL_CRYPT);
}

/*
 *====================================================================
 * PURPOSE: 
 *--------------------------------------------------------------------
 * PARAMS:  IN
 *          IN
 *          IN/OUT
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmSmb3CalculateMessageSignature(
        const NQ_BYTE *key,
        NQ_UINT keyLen,
        NQ_IOBufPos buffer1,
        NQ_UINT size1,
        NQ_IOBufPos buffer2,
        NQ_UINT size2,
        NQ_BYTE *signature
        )
{
    CMIOBlob fragments[2];
    CMBlob keyBlob;

    LOGFB(CM_TRC_LEVEL_CRYPT, "key:%p keyLen:%u buffer1:%p size1:%u buffer2:%p size2:%u signature:%p", key, keyLen, buffer1, size1, buffer2, size2, signature);

    fragments[0].data = buffer1;
    fragments[0].len = size1;
    fragments[1].data = buffer2;
    fragments[1].len = size2;
    keyBlob.data = (NQ_BYTE *)key;
    keyBlob.len = keyLen;

    (*currentCrypters.aes128cmac)(&keyBlob, NULL, fragments, 2, signature, SMB2_SECURITY_SIGNATURE_SIZE);

    LOGDUMP(CM_TRC_LEVEL_CRYPT, "signature", signature, SMB2_SECURITY_SIGNATURE_SIZE);
    LOGFE(CM_TRC_LEVEL_CRYPT);
}

/*
 *====================================================================
 * PURPOSE: Ker Derivation for SMB3 keys
 *--------------------------------------------------------------------
 * PARAMS:  IN     Session Key
 *          IN     Key length
 *          IN     Label
 *          IN     Label length
 *          IN     Context
 *          IN     Context length
 *          OUT    Derived key
 *
 * RETURNS: none
 *====================================================================
 */
void cmKeyDerivation(const NQ_BYTE * key, NQ_UINT keyLen, NQ_BYTE * label, NQ_UINT labelLen, NQ_BYTE * context, NQ_UINT contextLen, NQ_BYTE * derivedKey)
{
    NQ_BYTE temp1[4] = { 0x00, 0x00, 0x00, 0x01};
    NQ_BYTE temp2[4] = { 0x00, 0x00, 0x00, 0x80};
    NQ_BYTE zero = 0x00;
    NQ_BYTE ipad[SHA256_BLOCK_SIZE];
    NQ_BYTE opad[SHA256_BLOCK_SIZE];
    NQ_BYTE digest[SHA256_DIGEST_SIZE];
    NQ_IOBufPos temp1Pos;
    NQ_IOBufPos temp2Pos;
    NQ_IOBufPos zeroPos;
    NQ_IOBufPos ipadPos;
    NQ_IOBufPos opadPos;
    NQ_IOBufPos labelPos;
    NQ_IOBufPos contextPos;
    NQ_IOBufPos digestPos;
    IOBUF_POSCONSTRUCTORINIT(temp1Pos)
    IOBUF_POSCONSTRUCTORINIT(temp2Pos)
    IOBUF_POSCONSTRUCTORINIT(zeroPos)
    IOBUF_POSCONSTRUCTORINIT(ipadPos)
    IOBUF_POSCONSTRUCTORINIT(opadPos)
    IOBUF_POSCONSTRUCTORINIT(labelPos)
    IOBUF_POSCONSTRUCTORINIT(contextPos)
    IOBUF_POSCONSTRUCTORINIT(digestPos)

    NQ_UINT i;
    CMIOBlob fragments1[6];
    CMIOBlob fragments2[2];

    LOGFB(CM_TRC_LEVEL_CRYPT, "key:%p keyLen:%d label:%s labelLen:%d context:%p contextLen:%d derivedKey:%p", key, keyLen, (NQ_CHAR *)label, labelLen, context, contextLen, derivedKey);

    LOGDUMP(CM_TRC_LEVEL_CRYPT, "key", key, keyLen);

    IOBUF_POSCONSTRUCTOR(temp1Pos, temp1, sizeof(temp1))
    IOBUF_POSCONSTRUCTOR(temp2Pos, temp2, sizeof(temp2))
    IOBUF_POSCONSTRUCTOR(zeroPos, &zero, sizeof(zero))
    IOBUF_POSCONSTRUCTOR(ipadPos, ipad, sizeof(ipad))
    IOBUF_POSCONSTRUCTOR(opadPos, opad, sizeof(opad))
    IOBUF_POSCONSTRUCTOR(labelPos, label, labelLen)
    IOBUF_POSCONSTRUCTOR(contextPos, context, contextLen)
    IOBUF_POSCONSTRUCTOR(digestPos, digest, sizeof(digest))

    fragments1[0].data = ipadPos;
    fragments1[0].len = sizeof(ipad);
    fragments1[1].data = temp1Pos;
    fragments1[1].len = sizeof(temp1);
    fragments1[2].data = labelPos;
    fragments1[2].len = labelLen;
    fragments1[3].data = zeroPos;
    fragments1[3].len = 1;
    fragments1[4].data = contextPos;
    fragments1[4].len = contextLen;
    fragments1[5].data = temp2Pos;
    fragments1[5].len = sizeof(temp2);

    fragments2[0].data = opadPos;
    fragments2[0].len = sizeof(opad);
    fragments2[1].data = digestPos;
    fragments2[1].len = sizeof(digest);

    syMemset(ipad, 0x36, sizeof(ipad));
    syMemset(opad, 0x5C, sizeof(opad));

    for (i = 0; i < keyLen; ++i)
    {
        ipad[i] ^= key[i];
        opad[i] ^= key[i];
    }

    (*currentCrypters.sha256)(NULL, NULL, fragments1, 6, digest, sizeof(digest));
    (*currentCrypters.sha256)(NULL, NULL, fragments2, 2, digest, sizeof(digest));

    syMemcpy(derivedKey, digest, 16);

    LOGDUMP(CM_TRC_LEVEL_CRYPT, "derivedKey", derivedKey, 16);
    LOGFE(CM_TRC_LEVEL_CRYPT);
}

/*
 *====================================================================
 * PURPOSE: Encrypt SMB3 Message with AES_CCM
 *--------------------------------------------------------------------
 * PARAMS:  IN     Encrypting Key
 *          IN     Nonce
 *          IN     SMB3 Message
 *          IN     SMB3 Message length
 *          IN     Additional Message (SMB2 TRANSFORM HEADER excluding protocolID, Signature, Nonce)
 *          IN     Additional Message length
 *          OUT    Encrypted authentication value (signature for TF-Header)
 *
 * RETURNS: none
 *====================================================================
 */
void AES_128_CCM_Encrypt(NQ_BYTE * key, NQ_BYTE * nonce, NQ_IOBufPos msgBuf, NQ_UINT msgLen, NQ_IOBufPos addBuf, NQ_UINT addLen, NQ_BYTE * outMac)
{
    CMBlob keyBlob, key1Blob;
    CMIOBlob prefixBlob, msgBlob;

    keyBlob.data = (NQ_BYTE *) key;
    keyBlob.len = 16;
    key1Blob.data = (NQ_BYTE *) nonce;
    key1Blob.len = SMB2_AES128_CCM_NONCE_SIZE;

    prefixBlob.data = addBuf;
    prefixBlob.len = addLen;
    msgBlob.data = msgBuf;
    msgBlob.len = msgLen;

    (*currentCrypters.aes128ccmEncryption)(&keyBlob, &key1Blob, &prefixBlob, &msgBlob, outMac);
}

/*
 *====================================================================
 * PURPOSE: Decryption SMB3 Message with AES_CCM
 *--------------------------------------------------------------------
 * PARAMS:  IN     Encrypting Key
 *          IN     Nonce
 *          IN     SMB3 Message
 *          IN     SMB3 Message length
 *          IN     Additional Message (SMB2 TRANSFORM HEADER excluding protocolID, Signature, Nonce)
 *          IN     Additional Message length
 *          OUT    Encrypted authentication value (signature for TF-Header)
 *
 * RETURNS: TRUE  -> if calculated authentication value equals to received value.
 *          FALSE -> if calculated value differs from received. The Decrypted messages should be IGNORED in this case
 *====================================================================
 */
NQ_BOOL AES_128_CCM_Decrypt(NQ_BYTE * key, NQ_BYTE * nonce, NQ_IOBufPos msgBuf, NQ_UINT msgLen, NQ_IOBufPos addBuf, NQ_UINT addLen, NQ_BYTE * authValue)
{
    CMBlob keyBlob, key1Blob;
    CMIOBlob prefixBlob, msgBlob;

    keyBlob.data = (NQ_BYTE *) key;
    keyBlob.len = 16;
    key1Blob.data = (NQ_BYTE *) nonce;
    key1Blob.len = SMB2_AES128_CCM_NONCE_SIZE;

    prefixBlob.data = addBuf;
    prefixBlob.len = addLen;
    msgBlob.data = msgBuf;
    msgBlob.len = msgLen;

    return (*currentCrypters.aes128ccmDecryption)(&keyBlob, &key1Blob, &prefixBlob, &msgBlob, authValue);
}

/*
 *====================================================================
 * PURPOSE: 
 *--------------------------------------------------------------------
 * PARAMS:  IN
 *          IN
 *          IN/OUT
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
NQ_BOOL cmSmb3DecryptMessage(
        NQ_BYTE *key,
        NQ_BYTE *nonce,
        NQ_IOBufPos crptMsg,
        NQ_UINT msgLen,
        NQ_IOBufPos authMsg,
        NQ_UINT authLen,
        NQ_BYTE *signature,
        NQ_BOOL IsAesGCM
        )
{
    if (IsAesGCM)
        return aes128GcmDecrypt(key, nonce, crptMsg, msgLen, authMsg, authLen,signature, NULL, NULL);

    return AES_128_CCM_Decrypt(key, nonce, crptMsg, msgLen, authMsg, authLen,signature);
}

/*
 *====================================================================
 * PURPOSE: 
 *--------------------------------------------------------------------
 * PARAMS:  IN
 *          IN
 *          IN/OUT
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmSmb3EncryptMessage(
        NQ_BYTE * key,
        NQ_BYTE * nonce,
        NQ_IOBufPos msg,
        NQ_UINT msgLen,
        NQ_IOBufPos authMsg,
        NQ_UINT authLen,
        NQ_BYTE * signature,
        NQ_BOOL isAesGCM
        )
{
    LOGFB(CM_TRC_LEVEL_CRYPT, "key:%p nonce:%p msg:%p msgLen:%u authMsg:%p authLen:%u signature:%p isAesGCM:%d", key, nonce, msg, msgLen, authMsg, authLen, signature, isAesGCM);

    if (isAesGCM)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Cipher in use: AES-128-GCM");
        aes128GcmEncrypt(key, nonce, msg, msgLen, authMsg, authLen, signature, NULL, NULL);
    }
    else
    {
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Cipher in use: AES-128-CCM");
        AES_128_CCM_Encrypt(key, nonce, msg, msgLen, authMsg, authLen, signature);
    }

    LOGFE(CM_TRC_LEVEL_CRYPT);
}

/*
 *====================================================================
 * PURPOSE: 
 *--------------------------------------------------------------------
 * PARAMS:  IN
 *          IN
 *          IN/OUT
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void aes128GcmEncrypt(NQ_BYTE *key, NQ_BYTE *nonce, NQ_IOBufPos msgBuf, NQ_UINT msgLen, NQ_IOBufPos addBuf, NQ_UINT addLen, NQ_BYTE * outMac, NQ_BYTE *keyBuffer, NQ_BYTE *msgBuffer)
{
    CMBlob keyBlob, IVBlob;
    CMIOBlob AADBlob, msgBlob;

    keyBlob.data = (NQ_BYTE *) key;
    keyBlob.len = SMB_SESSIONKEY_LENGTH;
    IVBlob.data = (NQ_BYTE *) nonce;
    IVBlob.len = SMB2_AES128_GCM_NONCE_SIZE;

    AADBlob.data = addBuf;
    AADBlob.len = addLen;
    msgBlob.data = msgBuf;
    msgBlob.len = msgLen;

    (*currentCrypters.aes128gcmEncryption)(&keyBlob, &IVBlob, &AADBlob, &msgBlob, outMac, keyBuffer, msgBuffer);
}

/*
 *====================================================================
 * PURPOSE: 
 *--------------------------------------------------------------------
 * PARAMS:  IN
 *          IN
 *          IN/OUT
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
NQ_BOOL aes128GcmDecrypt(NQ_BYTE * key, NQ_BYTE * nonce, NQ_IOBufPos msgBuf, NQ_UINT msgLen, NQ_IOBufPos addBuf, NQ_UINT addLen, NQ_BYTE * outMac, NQ_BYTE *keyBuffer, NQ_BYTE *msgBuffer)
{
    CMBlob keyBlob, IVBlob;
    CMIOBlob AADBlob, msgBlob;

    keyBlob.data = (NQ_BYTE *) key;
    keyBlob.len = SMB_SESSIONKEY_LENGTH;
    IVBlob.data = (NQ_BYTE *) nonce;
    IVBlob.len = SMB2_AES128_GCM_NONCE_SIZE;

    AADBlob.data = addBuf;
    AADBlob.len = addLen;
    msgBlob.data = msgBuf;
    msgBlob.len = msgLen;

    return (*currentCrypters.aes128gcmDecryption)(&keyBlob, &IVBlob, &AADBlob, &msgBlob, outMac, keyBuffer, msgBuffer);
}

CMCrypterList *cmGetCurrentCrypters()
{
    return &currentCrypters;
}

void cmCalculateNtlmSigningKey(const NQ_BYTE *sessionKey, NQ_BYTE *outKey, NQ_UINT16 flag)
{
    CMBlob fragments[2];

    LOGFB(CM_TRC_LEVEL_CRYPT, "sessionKey:%p outKey:%p flag:%u", sessionKey, outKey, flag);

    switch (flag)
    {
        case CM_CRYPT_NTLM_TO_SERVER_SIGNING:
        {
            fragments[0].data = (NQ_BYTE *)sessionKey;
            fragments[0].len = SMB_SESSIONKEY_LENGTH;
            fragments[1].data = (NQ_BYTE *)NTLM_CLIENT_SIGN_CONST;
            fragments[1].len = (NQ_COUNT)syStrlen((NQ_CHAR *)NTLM_CLIENT_SIGN_CONST) + 1;
            (*currentCrypters.md5)(NULL , NULL , fragments , 2 ,outKey , SMB_SESSIONKEY_LENGTH);
        }
        break;
        case CM_CRYPT_NTLM_FROM_SERVER_SIGNING:
        {
            fragments[0].data = (NQ_BYTE *)sessionKey;
            fragments[0].len = SMB_SESSIONKEY_LENGTH;
            fragments[1].data = (NQ_BYTE *)NTLM_SERVER_SIGN_CONST;
            fragments[1].len = (NQ_COUNT)syStrlen((NQ_CHAR *)NTLM_SERVER_SIGN_CONST) + 1;
            (*currentCrypters.md5)(NULL , NULL , fragments , 2 ,outKey , SMB_SESSIONKEY_LENGTH);
        }
        break;
        case CM_CRYPT_NTLM_TO_SERVER_SEALING:
        {
            NQ_BYTE digest[16];

            fragments[0].data = (NQ_BYTE *)sessionKey;
            fragments[0].len = SMB_SESSIONKEY_LENGTH;
            fragments[1].data = (NQ_BYTE *)NTLM_CLIENT_SEAL_CONST;
            fragments[1].len = (NQ_COUNT)syStrlen((NQ_CHAR *)NTLM_CLIENT_SEAL_CONST) + 1;
            (*currentCrypters.md5)(NULL , NULL , fragments , 2 ,digest , SMB_SESSIONKEY_LENGTH);

            cmArcfourPrepareState(digest , 16 , outKey);
        }
        break;
        case CM_CRYPT_NTLM_FROM_SERVER_SEALING:
        {
            NQ_BYTE digest[16];

            fragments[0].data = (NQ_BYTE *)sessionKey;
            fragments[0].len = SMB_SESSIONKEY_LENGTH;
            fragments[1].data = (NQ_BYTE *)NTLM_SERVER_SEAL_CONST;
            fragments[1].len = (NQ_COUNT)syStrlen((NQ_CHAR *)NTLM_SERVER_SEAL_CONST) + 1;
            (*currentCrypters.md5)(NULL , NULL , fragments , 2 ,digest , SMB_SESSIONKEY_LENGTH);

            cmArcfourPrepareState(digest , 16 , outKey);
        }
        break;
        default:
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "  illegal flag");
        }
    }
    LOGFE(CM_TRC_LEVEL_CRYPT);
}

void
cmCalculateDcerpcSignature(
    NQ_BYTE *data,
    NQ_UINT16 dataLen,
    NQ_BYTE *signingKey,
    NQ_BYTE *sealingKey,
    NQ_UINT32 sequence,
    NQ_BOOL doSeal,
    NQ_BYTE *signature
    )
{
    CMBlob  fragments[2];
    NQ_BYTE seqNum[4];
    NQ_UINT32 *pSeqBuf;
    CMBlob  keyBlob;

    LOGFB(CM_TRC_LEVEL_CRYPT, "data:%p dataLen:%d signingKey:%p sealingKey:%p sequence:%d doSeal:%d signature:%p", data, dataLen, signingKey, sealingKey, sequence, doSeal, signature);


    syMemset(seqNum, 0, sizeof(seqNum));
    pSeqBuf = (NQ_UINT32 *)&seqNum;
    cmPutUint32(pSeqBuf, sequence);

    fragments[0].data = seqNum;
    fragments[0].len = sizeof(seqNum);
    fragments[1].data = (NQ_BYTE *)data;
    fragments[1].len = dataLen;
    keyBlob.data = (NQ_BYTE *)signingKey;
    keyBlob.len = SMB_SESSIONKEY_LENGTH;

    (*currentCrypters.hmacmd5)(&keyBlob, NULL, fragments, 2, signature, SMB_SESSIONKEY_LENGTH);

    if (TRUE == doSeal)
    {
        cmArcfourCryptWithState(signature, 8, sealingKey);
    }

    LOGFE(CM_TRC_LEVEL_CRYPT);
}

NQ_BOOL
cmCheckDcerpcSignature(
    NQ_BYTE *data,
    NQ_UINT16 dataLen,
    NQ_BYTE *signingKey,
    NQ_BYTE *sealingKey,
    NQ_UINT32 sequence,
    NQ_BOOL doSeal,
    NQ_BYTE *signature
)
{
    NQ_BYTE calcSig[8];
    NQ_BOOL result;

    LOGFB(CM_TRC_LEVEL_CRYPT, "data:%p dataLen:%d signingKey:%p sealingKey:%p sequence:%d doSeal:%d signature:%p", data, dataLen, signingKey, sealingKey, sequence, doSeal, signature);

    cmCalculateDcerpcSignature(data, dataLen, signingKey, sealingKey, sequence, doSeal, calcSig);
    result = 0 == syMemcmp(calcSig, signature, sizeof(calcSig));

    LOGFE(CM_TRC_LEVEL_CRYPT, "seq:%d, result:%d: signatures %s match", sequence, result, result ? "" : "not");
    return result;
}
