/*********************************************************************
*
*           Copyright (c) 2021 by Visuality Systems, Ltd.
*
*********************************************************************
* FILE NAME     : $Workfile:$
* ID            : $Header:$
* REVISION      : $Revision:$
*--------------------------------------------------------------------
* DESCRIPTION   : Buffer manipulation
*--------------------------------------------------------------------
* MODULE        : Common
* DEPENDENCIES  :
********************************************************************/

#include "cmbuf.h"
#include "syapi.h"

/* Defines */

#define NQ_UINT32_ASCII_STRING_LEN  9        /* 2 * sizeof(NQ_UINT32) + 1 null termination */
#define NQ_UINT16_ASCII_STRING_LEN  5        /* 2 * sizeof(NQ_UINT16) + 1 null termination */
#define NQ_BYTE_ASCII_STRING_LEN    3        /* 2 * sizeof(NQ_BYTE) + 1 null termination */

/* Static functions */

static NQ_UINT32 align(NQ_UINT32 what, NQ_UINT32 alignment)
{
    --alignment;

    return (what + alignment) & ~alignment;
}

/* Buffer reader */

void cmBufferReaderInit(CMBufferReader *reader, NQ_IOBufPos buffer, NQ_COUNT size)
{
    cmRpcSetDescriptor(reader, buffer, FALSE);

    reader->length = size;
}

void cmBufferReaderReset(CMBufferReader *reader)
{
    reader->current = reader->origin;
}

void cmBufferReaderStart(CMBufferReader *reader)
{
    reader->length -= cmBufferReaderGetDataCount(reader);
    reader->origin = reader->current;
}

void cmBufferReaderClone(const CMBufferReader *from, CMBufferReader *to)
{
    to->origin = from->origin;
    to->current = from->current;
    to->length = from->length;
    to->nbo = from->nbo;
}

void cmBufferReaderSetByteOrder(CMBufferReader *reader, NQ_BOOL le)
{
    reader->nbo = !le;
}

NQ_IOBufPos cmBufferReaderGetStart(const CMBufferReader *reader)
{
    return reader->origin;
}

NQ_BYTE* cmBufferReaderGetPositionBytePtr(const CMBufferReader *reader, NQ_COUNT requiredLength)
{
    IOBUF_BUFFER_CANUSEFLAT_ASSERT(reader->current, requiredLength);
    return IOBUF_GETBYTEPTR(reader->current);
}

NQ_IOBufPos cmBufferReaderGetPosition(const CMBufferReader *reader)
{
    return reader->current;
}

void cmBufferReaderSetPosition(CMBufferReader *reader, NQ_IOBufPos position)
{
    ASSERT_IOBUFPOSPTR((&position));
    reader->current = position;
}

NQ_BOOL cmBufferReaderSetPositionIfValid(CMBufferReader *reader, NQ_IOBufPos position)
{
    NQ_BOOL result = TRUE;

    /* check that the new position is inside the buffer */
    if (position > (reader->origin + reader->length))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid position");
        result = FALSE;
        goto Exit;
    }

    ASSERT_IOBUFPOSPTR((&position));
    reader->current = position;

Exit:
    return result;
}

void cmBufferReaderSetPositionNULL(CMBufferReader *reader)
{
    IOBUF_POSINIT(reader->current);
}

void cmBufferReaderSetOffset(CMBufferReader *reader, NQ_UINT32 offset)
{
    reader->current = reader->origin;
    IOBUF_MOVEBYTES(reader->current, (NQ_INT)offset);
}

NQ_COUNT cmBufferReaderGetDataCount(const CMBufferReader *reader)
{
    return (NQ_COUNT)IOBUF_GETPOINTERSDIF(reader->current, reader->origin);
}

NQ_INT cmBufferReaderGetRemaining(const CMBufferReader *reader)
{
    return (NQ_INT)(reader->length - cmBufferReaderGetDataCount(reader));
}

NQ_COUNT cmBufferReaderGetTotalSize(const CMBufferReader *reader)
{
    return reader->length;
}

void cmBufferReaderAlignToOrigin(CMBufferReader *reader, NQ_UINT alignment)
{
    cmBufferReaderAlign(reader, reader->origin, alignment);
}

void cmBufferReaderAlign(CMBufferReader *reader, NQ_IOBufPos anchor, NQ_UINT alignment)
{
    /* assert: reader->current >= anchor */
    NQ_COUNT bytesDiff = (NQ_COUNT)IOBUF_GETPOINTERSDIF(reader->current, anchor);
    reader->current = anchor;
    IOBUF_MOVEBYTES(reader->current, (NQ_INT)align(bytesDiff, alignment));
}

void cmBufferReaderSkip(CMBufferReader *reader, NQ_UINT bytes)
{
    IOBUF_MOVEBYTES(reader->current, (NQ_INT)bytes);
}

void cmBufferReaderMoveBackward(CMBufferReader *reader, NQ_UINT bytes)
{
    IOBUF_MOVEBYTES(reader->current, (-1) * (NQ_INT)bytes);
}

void cmBufferReadByte(CMBufferReader *reader, NQ_BYTE *to)
{
    cmRpcParseByte(reader, to);
}

void cmBufferReadBytes(CMBufferReader *reader, NQ_BYTE *to, NQ_COUNT size)
{
    cmRpcParseBytes(reader, to, size);
}

void cmBufferReadUint16(CMBufferReader *reader, NQ_UINT16 *to)
{
    cmRpcParseUint16(reader, to);
}

void cmBufferReadUint32(CMBufferReader *reader, NQ_UINT32 *to)
{
    cmRpcParseUint32(reader, to);
}

void cmBufferReadUint64(CMBufferReader *reader, NQ_UINT64 *to)
{
    cmBufferReadUint32(reader, &to->low);
    cmBufferReadUint32(reader, &to->high);
}

void cmBufferReadUuid(CMBufferReader *reader, NQ_Uuid * pUuid)
{
    cmBufferReadUint32(reader, (NQ_UINT32 *)&pUuid->timeLow);
    cmBufferReadUint16(reader, (NQ_UINT16 *)&pUuid->timeMid);
    cmBufferReadUint16(reader, (NQ_UINT16 *)&pUuid->timeHiVersion);
    cmBufferReadBytes(reader, pUuid->clockSeq, sizeof(pUuid->clockSeq));
    cmBufferReadBytes(reader, pUuid->node, sizeof(pUuid->node));
}

NQ_INT cmBufferReaderMemCmp(const CMBufferReader *reader, NQ_BYTE *with, NQ_COUNT size)
{
    return (IOBUF_MEMCMP_V2F(reader->current, with, size));
}

NQ_BOOL cmBufferReaderIsCurrentPositionBigger(const CMBufferReader *reader, NQ_IOBufPos position)
{
    if (IOBUF_ISPTRBIGGER(reader->current, position))
    {
        return TRUE;
    }
    return FALSE;
}

NQ_BOOL cmBufferReaderIsCurrentPositionSmaller(const CMBufferReader *reader, NQ_IOBufPos position)
{
    if (IOBUF_ISPTRSMALLER(reader->current, position))
    {
        return TRUE;
    }
    return FALSE;
}

/* Buffer writer */

void cmBufferWriterInit(CMBufferWriter *writer, NQ_IOBufPos buffer, NQ_COUNT capacity)
{
    cmRpcSetDescriptor(writer, buffer, FALSE);

    writer->length = capacity;
}

void cmBufferWriterStart(CMBufferWriter *writer)
{
    writer->length -= cmBufferWriterGetDataCount(writer);
    writer->origin = writer->current;
}

void cmBufferWriterReset(CMBufferWriter *writer)
{
    writer->current = writer->origin;
}

void cmBufferWriterSetByteOrder(CMBufferWriter *writer, NQ_BOOL le)
{
    writer->nbo = !le;
}

void cmBufferWriterClone(const CMBufferWriter *what, CMBufferWriter *to, NQ_UINT offset)
{
    to->origin = what->origin;
    to->current = what->current;
    IOBUF_MOVEBYTES(to->current, (NQ_INT)offset);
    to->nbo = what->nbo;
    to->length = what->length;
}

void cmBufferWriterCloneAndSkip(CMBufferWriter *what, CMBufferWriter *to, NQ_UINT offset)
{
    cmBufferWriterClone(what, to, 0);
    cmBufferWriterSkip(what, offset);
}

void cmBufferWriterBranch(const CMBufferWriter *what, CMBufferWriter *to, NQ_UINT offset)
{
    NQ_IOBufPos bufPos = what->current;
    to->origin = to->current = IOBUF_MOVEBYTES(bufPos, (NQ_INT)offset);
    to->length = what->length - (cmBufferWriterGetDataCount(what) + offset);
    to->nbo = what->nbo;
}

void cmBufferWriterSync(CMBufferWriter *what, const CMBufferWriter *with)
{
    what->current = with->current;
}

NQ_IOBufPos cmBufferWriterGetStart(const CMBufferWriter *writer)
{
    return writer->origin;
}

NQ_IOBufPos cmBufferWriterGetPosition(const CMBufferWriter *writer)
{
    return writer->current;
}

NQ_BYTE* cmBufferWriterGetPositionBytePtr(const CMBufferWriter *writer, NQ_COUNT requiredLength)
{
    IOBUF_BUFFER_CANUSEFLAT_ASSERT(writer->current, requiredLength);
    return IOBUF_GETBYTEPTR(writer->current);
}

NQ_IOBufPos cmBufferWriterGetPositionWithOff(const CMBufferWriter *writer, NQ_COUNT offset)
{
    NQ_IOBufPos bufPos = writer->current;
    IOBUF_MOVEBYTES(bufPos, (NQ_INT)offset);
    return bufPos;
}

void cmBufferWriterSetPosition(CMBufferWriter *writer, NQ_IOBufPos position)
{
    writer->current = position;
}

/*
void cmBufferWriterSetOffset(CMBufferWriter *writer, NQ_UINT offset)
{
    writer->current = writer->origin + offset;
}
*/

NQ_COUNT cmBufferWriterGetDataCount(const CMBufferWriter *writer)
{
    return (NQ_COUNT)IOBUF_GETPOINTERSDIF(writer->current, writer->origin);
}

NQ_COUNT cmBufferWriterGetDataCountFromOffset(const CMBufferWriter *writer, NQ_IOBufPos offset)
{
    return (NQ_COUNT)IOBUF_GETPOINTERSDIF(writer->current, offset);
}

NQ_COUNT cmBufferWriterGetRemaining(const CMBufferWriter *writer)
{
    return writer->length - cmBufferWriterGetDataCount(writer);
}

NQ_UINT32 cmBufferWriterAlignToOrigin(CMBufferWriter *writer, NQ_UINT alignment)
{
    return cmBufferWriterAlign(writer, writer->origin, alignment);
}

NQ_UINT32 cmBufferWriterAlign(CMBufferWriter *writer, NQ_IOBufPos anchor, NQ_UINT alignment)
{
    /* assert: writer->current >= anchor */
    NQ_UINT32 diff = (NQ_UINT32)IOBUF_GETPOINTERSDIF(writer->current, anchor);
    NQ_UINT32 aligned = align(diff, alignment);

    IOBUF_MEMSET(writer->current, 0, (aligned - diff));
    writer->current = IOBUF_SKIPBYTE(anchor, (NQ_INT)aligned);

    return (aligned - diff);
}

void cmBufferWriterSkip(CMBufferWriter *writer, NQ_UINT bytes)
{
    IOBUF_MOVEBYTES(writer->current, (NQ_INT)bytes);
}

void cmBufferWriteByte(CMBufferWriter *writer, NQ_BYTE value)
{
    cmRpcPackByte(writer, value);
}

void cmBufferWriteBytes(CMBufferWriter *writer, const NQ_BYTE *bytes, NQ_COUNT size)
{
    cmRpcPackBytes(writer, bytes, size);
}

void cmBufferWriteIOBytes(CMBufferWriter *writer, NQ_IOBufPos bytes, NQ_COUNT size)
{
    cmRpcPackIOBytes(writer, bytes, size);
}

void cmBufferWriteZeroes(CMBufferWriter *writer, NQ_COUNT size)
{
    for (; size > 0; --size)
        cmBufferWriteByte(writer, 0);
}

void cmBufferWriteUint16(CMBufferWriter *writer, NQ_UINT16 value)
{
    cmRpcPackUint16(writer, value);
}

void cmBufferWriteUint32(CMBufferWriter *writer, NQ_UINT32 value)
{
    cmRpcPackUint32(writer, value);
}

void cmBufferWriteInt32(CMBufferWriter *writer, NQ_INT32 value)
{
    cmRpcPackInt32(writer, value);
}

void cmBufferWriteUint64(CMBufferWriter *writer, const NQ_UINT64 *value)
{
    if (writer->nbo)
    {
        cmRpcPackUint32(writer, value->high);
        cmRpcPackUint32(writer, value->low);
    }
    else
    {
        cmRpcPackUint32(writer, value->low);
        cmRpcPackUint32(writer, value->high);
    }
}

void cmBufferWriteUuid(CMBufferWriter * writer, const NQ_Uuid * pUuid)
{
    cmBufferWriteUint32(writer, (NQ_UINT32)cmGetSUint32(pUuid->timeLow));
    cmBufferWriteUint16(writer, (NQ_UINT16)cmGetSUint16(pUuid->timeMid));
    cmBufferWriteUint16(writer, (NQ_UINT16)cmGetSUint16(pUuid->timeHiVersion));
    cmBufferWriteBytes(writer, pUuid->clockSeq, sizeof(pUuid->clockSeq));
    cmBufferWriteBytes(writer, pUuid->node, sizeof(pUuid->node));
}

void cmBufferWriteUuidAsString(CMBufferWriter * writer, const NQ_Uuid * pUuid)
{
    NQ_CHAR strHexaUint32[NQ_UINT32_ASCII_STRING_LEN]; /* for hexa NQ_UINT32 ascii representation with null */
    NQ_CHAR strHexaUint16[NQ_UINT16_ASCII_STRING_LEN]; /* for hexa NQ_UINT16 ascii representation with null */
    NQ_CHAR strHexaByte[NQ_BYTE_ASCII_STRING_LEN];     /* for hexa NQ_BYTE ascii representation with null */
    NQ_COUNT i;

    sySprintf(strHexaUint32, "%08x", (NQ_UINT32)cmGetSUint32(pUuid->timeLow));
    cmBufferWriteString(writer, TRUE, (NQ_BYTE *)strHexaUint32, FALSE, CM_BSF_NOFLAGS);
    cmBufferWriteByte(writer, '-');

    sySprintf(strHexaUint16, "%04x", (NQ_UINT16)cmGetSUint16(pUuid->timeMid));
    cmBufferWriteString(writer, TRUE, (NQ_BYTE *)strHexaUint16, FALSE, CM_BSF_NOFLAGS);
    cmBufferWriteByte(writer, '-');

    sySprintf(strHexaUint16, "%04x", (NQ_UINT16)cmGetSUint16(pUuid->timeHiVersion));
    cmBufferWriteString(writer, TRUE, (NQ_BYTE *)strHexaUint16, FALSE, CM_BSF_NOFLAGS);
    cmBufferWriteByte(writer, '-');

    for (i = 0; i < sizeof(pUuid->clockSeq) / sizeof(pUuid->clockSeq[0]); i++)
    {
        sySprintf(strHexaByte, "%02x", pUuid->clockSeq[i]);
        cmBufferWriteString(writer, TRUE, (NQ_BYTE *)strHexaByte, FALSE, CM_BSF_NOFLAGS);
    }
    cmBufferWriteByte(writer, '-');

    for (i = 0; i < sizeof(pUuid->node) / sizeof(pUuid->node[0]); i++)
    {
        sySprintf(strHexaByte, "%02x", pUuid->node[i]);
        cmBufferWriteString(writer, TRUE, (NQ_BYTE *)strHexaByte, FALSE, CM_BSF_NOFLAGS);
    }
}

void cmBufferWriteAsciiAsUnicodeN(CMBufferWriter *writer, const NQ_CHAR *string, NQ_UINT length, CMBufferStringFlags flags)
{
    cmAnsiToUnicodeN((NQ_WCHAR *)cmBufferWriterGetPositionBytePtr(writer, (NQ_COUNT)((length + CM_TRAILING_NULL) * sizeof(NQ_WCHAR))), (length + CM_TRAILING_NULL) * (NQ_UINT)sizeof(NQ_WCHAR), string, length);
    cmBufferWriterSkip(writer, (NQ_UINT)(length * sizeof(NQ_WCHAR)));
    /* if target string should be null terminated write trailing 2 zero bytes */
    if (flags & CM_BSF_WRITENULLTERM)
    {
        cmBufferWriteUint16(writer, 0);
    }
}

void cmBufferWriteUnicode(CMBufferWriter *writer, const NQ_WCHAR *string)
{
    NQ_UINT length;

    length = (NQ_UINT)cmWStrlen(string);
    cmWStrncpy((NQ_WCHAR *)cmBufferWriterGetPositionBytePtr(writer, length), string, length);
    cmBufferWriterSkip(writer, (NQ_UINT)(length * sizeof(NQ_WCHAR)));
    cmBufferWriteUint16(writer, 0);
}

void cmBufferWriteUnicodeNoNull(CMBufferWriter *writer, const NQ_WCHAR *string)
{
    NQ_UINT length;

    length = (NQ_UINT)cmWStrlen(string);
    cmBufferWriteBytes(writer, (NQ_BYTE*)string, (NQ_COUNT)(length * sizeof(NQ_WCHAR)));
}

void cmBufferWriteRandomBytes(CMBufferWriter *writer, NQ_COUNT size)
{
    for (; size > 0; --size)
        cmBufferWriteByte(writer, (NQ_BYTE)syRand());
}

void cmBufferWriteString(CMBufferWriter *writer, NQ_BOOL outAscii, const NQ_BYTE *string, NQ_BOOL inUnicode, CMBufferStringFlags flags)
{
    if (inUnicode)
    {
        if (outAscii)
        {
            NQ_UINT length = cmWStrlen((NQ_WCHAR *)string);
            NQ_CHAR * strA = (NQ_CHAR *)cmBufferWriterGetPositionBytePtr(writer, length);

            cmUnicodeToAnsi(strA, (NQ_WCHAR *)string);
            cmBufferWriterSkip(writer, length);
            if (flags & CM_BSF_WRITENULLTERM)
            {
                cmBufferWriteByte(writer, 0);
            }
        }
        else
        {
            cmBufferWriteUnicodeNoNull(writer, (const NQ_WCHAR *)string);
            if (flags & CM_BSF_WRITENULLTERM)
            {
                cmBufferWriteUint16(writer, 0);
            }
        }
    }
    else
    {
        NQ_UINT length = (NQ_UINT)syStrlen((const NQ_CHAR*)string);

        if (outAscii)
        {
            cmBufferWriteBytes(writer, string, length);
            if (flags & CM_BSF_WRITENULLTERM)
            {
                cmBufferWriteByte(writer, 0);
            }
        }
        else
        {
            cmBufferWriteAsciiAsUnicodeN(writer, (const NQ_CHAR *)string, length, flags);
        }
    }
}
