/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Initialization
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 ********************************************************************/

#include "cmapi.h"
#include "cmtrace.h"
#include "cmfinddc.h"
#if (!defined(UD_NQ_DYNAMICALLOCATION) || defined(REPOSITORY_SIZE_TUNING))
#include "cmmemoryrepo.h"
#endif /* !defined(UD_NQ_DYNAMICALLOCATION) || defined(REPOSITORY_SIZE_TUNING) */

/*
  CM consists of two components: NetBIOS library and CIFS library. We initialize both
  of them.
 */

static NQ_UINT initCount = 0;

/*
 *====================================================================
 * PURPOSE: Initialize the library
 *--------------------------------------------------------------------
 * PARAMS:  component (client, server, nd daemon)
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *====================================================================
 */

NQ_STATUS
cmInit(
    NQ_UINT32 component
    )
{
    NQ_STATUS result = NQ_FAIL;
    NQ_UINT64 zero = {0,0};

    if (sizeof(NQ_UINT32) != 4)
    {
        syPrintf("Type 'long' is not 4 bytes on the target platform. Define SY_INT32 in SYCOMPIL.H\n");
        sySetLastError(NQ_ERR_INVALIDUINT32SIZE);
        goto Exit;
    }
    if (!cmMemoryStart(zero))
    {
        syPrintf("cmMemoryStart() failed\n");
        goto Exit;
    }

#if (!defined(UD_NQ_DYNAMICALLOCATION) || defined(REPOSITORY_SIZE_TUNING))
    if (!cmMemoryRepoStart(0))
    {
        syPrintf("cmMemoryRepoStart() failed\n");
        goto Error1;
    }
#endif /* !defined(UD_NQ_DYNAMICALLOCATION) || defined(REPOSITORY_SIZE_TUNING) */

    if (NQ_FAIL == cmNetBiosInit())
    {
        syPrintf("cmNetBiosInit() failed\n");
        goto Error2;
    }

    if (NQ_FAIL == cmCifsInit())
    {
        syPrintf("cmCifsInit() failed\n");
        goto Error3;
    }

    if (!cmSpStart())
    {
        syPrintf("cmSpStart() failed\n");
        goto Error4;
    }

    if (!cmSelfipStart())
    {
        syPrintf("cmSelfipStart() failed\n");
        goto Error5;
    }

    if (!cmResolverStart())
    {
        syPrintf("cmResolverStart() failed\n");
        goto Error6;
    }

    if (!cmThreadStart())
    {
        syPrintf("cmThreadStart() failed\n");
        goto Error7;
    }

    if (!cmPortManageStart())
    {
        syPrintf("cmPortManageStart() failed\n");
        goto Error8;
    }

    if ((NQ_NDDAEMON != component) && (NQ_FAIL == cmFindDCInit()))
    {
        syPrintf("cmFindDCInit() failed\n");
        goto Error9;
    }

#ifdef UD_NQ_INCLUDETRACE
    cmTraceInit();
#endif /* UD_NQ_INCLUDETRACE */
#ifdef UD_NQ_EXTERNALTRACE
    syTraceInit();
#endif /* UD_NQ_EXTERNALTRACE */

#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS)
    if ((NQ_NDDAEMON != component) && (NQ_FAIL == cmSdInit()))
    {
        syPrintf("cmSdInit() failed\n");
        goto Error10;
    }
#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) */

    if (NQ_NDDAEMON != component)
    {
        cmBufManStart();
    }

#ifdef UD_NQ_INCLUDESMBCAPTURE
    if (NQ_NDDAEMON != component)
    {
        cmCaptureStart();
    }
#endif  /*UD_NQ_INCLUDESMBCAPTURE*/

    initCount++;

    result = NQ_SUCCESS;
    goto Exit;

#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS)
Error10:
    cmFindDCExit();
#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) */
Error9:
   cmPortManageShutdown();
Error8:
   cmThreadShutdown();
Error7:
    cmResolverShutdown();
Error6:
    cmSelfipShutdown();
Error5:
    cmSpShutdown();
Error4:
    cmCifsExit();
Error3:
    cmNetBiosExit();
Error2:
#if (!defined(UD_NQ_DYNAMICALLOCATION) || defined(REPOSITORY_SIZE_TUNING))
    cmMemoryRepoShutdown();
Error1:
#endif /* #if (!defined(UD_NQ_DYNAMICALLOCATION) || defined(REPOSITORY_SIZE_TUNING)) */
    cmMemoryShutdown();

Exit:
    return result;
}

/*
 *====================================================================
 * PURPOSE: This function performs the shutdown of the Common Module
 *--------------------------------------------------------------------
 * PARAMS:  One of the following: NQ_CLIENT, NQ_CLIENT Macro,
 *                     NQ_NDDAEMON, NQ_NDDAEMON Macro, NQ_SERVER, NQ_SERVER Macro
 * RETURNS: NONE
 *====================================================================
 */

void
cmExit(
    NQ_UINT32 component
    )
{
    static NQ_BOOL isForceExit = FALSE; /* This parameter will be changed to TRUE if initCount > 0 and the component is not NQ_NDDAEMON */

    if (--initCount > 0)
    {
        if (NQ_NDDAEMON != component)
        {
            isForceExit = TRUE;
        }

        return;
    }

#ifdef UD_NQ_INCLUDESMBCAPTURE
    cmCaptureShutdown();
#endif  /*UD_NQ_INCLUDESMBCAPTURE*/
    if ((NQ_NDDAEMON != component) || (TRUE == isForceExit))
    {
        cmBufManShutdown();
    }
    cmNetBiosExit();
    cmCifsExit();
#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS)
    if ((NQ_NDDAEMON != component) || (TRUE == isForceExit))
    {
        cmSdExit();
    }
#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) */
    if ((NQ_NDDAEMON != component) || (TRUE == isForceExit))
    {
        cmFindDCExit();
    }
    /* new modules */
    cmResolverShutdown();
#ifdef UD_NQ_INCLUDETRACE
    cmTraceFinish();
#endif /* UD_NQ_INCLUDETRACE */
#ifdef UD_NQ_EXTERNALTRACE
    syTraceShutdown();
#endif /* UD_NQ_EXTERNALTRACE */
    cmThreadShutdown();
    cmSelfipShutdown();
    cmSpShutdown();
    cmPortManageShutdown();
#if (!defined(UD_NQ_DYNAMICALLOCATION) || defined(REPOSITORY_SIZE_TUNING))
    cmMemoryRepoShutdown();
#endif /* UD_NQ_DYNAMICALLOCATION */
    cmMemoryShutdown();
    isForceExit = FALSE;
}
