/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : SMB2 Write command handler
 *--------------------------------------------------------------------
 * MODULE        : Server
 * DEPENDENCIES  :
 ********************************************************************/

#include "csparams.h"
#include "csutils.h"
#include "csdcerpc.h"
#include "cs2disp.h"
#ifdef UD_CS_INCLUDERPC_SPOOLSS
#include "csspools.h"
#endif /* UD_CS_INCLUDERPC_SPOOLSS */

#if defined(UD_NQ_INCLUDECIFSSERVER) && defined(UD_NQ_INCLUDESMB2)

/* 
 * Local functions and data
 * ------------------------
 */ 

static NQ_UINT32 dataCount;     /* number of bytes in WRITE */
static CSFile* pFile;           /* pointer to file descriptor */
#ifdef UD_CS_INCLUDERPC

/* saving late response for Write */
static void
writeSmb2LateResponseSave(
    CSLateResponseContext* context
    );

/* preparing late response for Write */
static NQ_BOOL
writeSmb2LateResponsePrepare(
    CSLateResponseContext* context
    );

/* sending late response for Write */
static NQ_BOOL
writeSmb2LateResponseSend(
    CSLateResponseContext* context,
    NQ_UINT32 status,
    NQ_COUNT dataLength
    );

#endif /* UD_CS_INCLUDERPC */

#if defined(UD_CS_INCLUDERPC_SPOOLSS) && !defined(UD_CS_SPOOLSS_PRN_USESPOOLERFILE)
/* sending data to a printer */
static NQ_UINT32
prnWriteToPrinter(
    CSFile  *pFile,
    NQ_BYTE *pData              /* pointer to the buffer */
    );

static NQ_UINT32
prnPrintDisorderedPackets(
    CSFile  *pFile
    );

static NQ_UINT32
prnSaveDisorderedPackets(
    CSFile      *pFile,
    NQ_BYTE     *pData,     /* pointer to the buffer */
    NQ_UINT64   *pOffset    /* write offset */
    );
#endif /* defined(UD_CS_INCLUDERPC_SPOOLSS) && !defined(UD_CS_SPOOLSS_PRN_USESPOOLERFILE) */

#define RESPONSE_LENGTH 16  /* length of the write response not including data */
#define DATA_OFFSET 0x70    /* The offset of the data in a write request, for now this offset is constant */

/*====================================================================
 * PURPOSE: Perform Write processing
 *--------------------------------------------------------------------
 * PARAMS:  IN in - pointer to the parsed SMB2 header descriptor
 *          OUT out - pointer to the response header structure
 *          IN reader - request reader pointing to the second command field
 *          IN connection - pointer to the session structure
 *          IN user - pointer to the user structure
 *          IN tree - pointer to the tree structure
 *          OUT writer - pointer to the response writer
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   This function is called on SMB2 Create command.
 *====================================================================
 */

NQ_UINT32 csSmb2OnWrite(CMSmb2Header *in, CMSmb2Header *out, CMBufferReader *reader, CSSession *connection, CSUser *user, CSTree *tree, CMBufferWriter *writer)
{
    CSFid       fid;                /* fid of the file to close */
    NQ_UINT32   minCount;           /* buffer length */
    NQ_UINT16   dataOffset;         /* offset of the data portion from SMB2 start*/
    NQ_UINT64   offset;             /* write offset */
    NQ_UINT32   returnValue;        /* return code */
    NQ_BYTE *   pData;              /* pointer to the buffer */
#ifdef UD_CS_FORCEINTERIMRESPONSES
    NQ_UINT32 asyncId = 0;          /* generated Async ID */
#endif /* UD_CS_FORCEINTERIMRESPONSES */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    UDFileAccessEvent   eventInfo;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "in:%p out:%p reader:%p connection:%p user:%p tree:%p writer:%p", in, out, reader, connection, user, tree, writer);

    /* parse request */
    cmBufferReadUint16(reader, &dataOffset);    /* data offset */
    cmBufferReadUint32(reader, &dataCount);     /* length */
    cmBufferReadUint64(reader, &offset);        /* offset */
    cmBufferReadUint16(reader, &fid);           /* file ID */
    cs2ParseFid(&fid);
    cmBufferReaderSkip(reader, 14);             /* the rest of the file ID */
    cmBufferReadUint32(reader, &minCount);      /* channel */

    if (DATA_OFFSET != dataOffset)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid data offset value expected: 0x%x, actual: 0x%x", DATA_OFFSET, dataOffset);
        returnValue = SMB_STATUS_INVALID_PARAMETER;
        goto Exit;
    }

    pData = in->_start + dataOffset;
#ifndef UD_CS_INCLUDEDIRECTTRANSFER
    if (dataCount > (CIFS_MAX_DATA_SIZE - SMB2_HEADERSIZE - 16))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Buffer overflow: write length too big");
        returnValue = SMB_STATUS_INVALID_PARAMETER;
        goto Exit;
    }
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
    
    /* find file descriptor */
    pFile = csGetFileByFid(fid, tree->tid, user->uid);
    if (pFile == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unknown FID");
        returnValue =  SMB_STATUS_INVALID_HANDLE;
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "pFile:%p file:%d dataCount:%lu offset:%lu%lu", pFile, pFile->file, (NQ_ULONG)dataCount, (NQ_ULONG)(offset.high), (NQ_ULONG)(offset.low));

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.fileName = csGetFileName(pFile->fid);
    eventInfo.tid = tree->tid;
    eventInfo.rid = csGetUserRid(user);
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

#ifdef UD_CS_INCLUDERPC
    if (pFile->isPipe)
    {
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
        /* discard possible DirectTransfer and read the payload */
        csDispatchDtDiscard();
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

        csDcerpcSetLateResponseCallbacks(
                writeSmb2LateResponseSave,
                writeSmb2LateResponsePrepare,
                writeSmb2LateResponseSend
        );
        returnValue = csDcerpcWrite(
                pFile, 
                pData, 
                (NQ_UINT)dataCount, 
                FALSE 
                );
        if (returnValue != 0)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "error writing to pipe");
            goto Exit;
        }
    }
    else
#endif /* UD_CS_INCLUDERPC */
    {
#ifdef UD_CS_INCLUDERPC_SPOOLSS
        if (pFile->isPrint)
        {
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
            /* discard possible DirectTransfer and read the payload */
            csDispatchDtDiscard();
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
            /* write into spooler file */
            if ((NQ_UINT32)NQ_FAIL == sySeekFileStart(pFile->fileSpooler, offset.low, offset.high))
            {
                returnValue = csErrorGetLast();
                LOGERR(CM_TRC_LEVEL_ERROR, "LSEEK failed");
                goto Exit;
            }

            dataCount = (NQ_UINT32)syWriteFile(pFile->fileSpooler, pData, (NQ_COUNT)dataCount);
            if ((NQ_INT)dataCount < 0)
            {
                returnValue = csErrorGetLast();
                LOGERR(CM_TRC_LEVEL_ERROR, "WRITE failed");
                goto Exit;
            }

            /* update the count */
            cmU64AddU32(&pFile->fileSpoolerDataCount, dataCount);

            /* check whether it's the last portion and actually send to the printer */
            if (0 == cmU64Cmp(&pFile->fileSpoolerDataCount, &pFile->printSize))
            {
                returnValue = csSpoolssSendToPrinter(pFile, writeSmb2LateResponseSave, writeSmb2LateResponsePrepare, writeSmb2LateResponseSend);
                if (NQ_SUCCESS != returnValue)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "WRITE to printer failed");
                    goto Exit;
                }
            }
#else
            /* Set the divider only once - at the beginning */
            if (0 == pFile->memorySpooler.divider)
            {
                pFile->memorySpooler.divider = dataCount;
            }

            csDcerpcSetLateResponseCallbacks(
                    writeSmb2LateResponseSave,
                    writeSmb2LateResponsePrepare,
                    writeSmb2LateResponseSend
            );

            if (0 == cmU64Cmp(&offset, &((CSPrnDisorderedContext*)(pFile->memorySpooler.context))->currentOffset))
            {
                /* Write the current packet data */
                returnValue = prnWriteToPrinter(pFile, pData);
                if (NQ_SUCCESS != returnValue)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "failed to write prn file to printer");
                    goto Exit;
                }

                /* If there are available disordered packets, print them */
                if (0 != pFile->memorySpooler.numberOfItems)
                {
                    returnValue = prnPrintDisorderedPackets(pFile);
                    if (NQ_SUCCESS != returnValue)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "failed to write prn file to printer");
                        goto Exit;
                    }
                }
            }
            else /* This packet offset is not the current offset that needs to be written and hence needed to be saved */
            {
                returnValue = prnSaveDisorderedPackets(pFile, pData, &offset);
                if (NQ_SUCCESS != returnValue)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "failed to save prn data");
                    goto Exit;
                }
            }
#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
        } /* pFile->isPrint */
        else
#endif /* UD_CS_INCLUDERPC_SPOOLSS */       
        {
            /* send interim response */
#ifdef UD_CS_FORCEINTERIMRESPONSES
            asyncId = csSmb2SendInterimResponse(in);
            if (0 == asyncId)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "error sending interim write response");
                returnValue = SMB_STATUS_INVALID;
                goto Exit;
            }
            out->flags |= SMB2_FLAG_ASYNC_COMMAND;
            out->aid.low = asyncId;
            out->aid.high = 0;
            out->credits = 0;
#endif /* UD_CS_FORCEINTERIMRESPONSES */

            if (dataCount == 0)
            {
                /* truncate file */
                returnValue = csTruncateFile(pFile, NULL, offset.low, offset.high);
                if (returnValue != NQ_SUCCESS)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "truncate failed");
                    goto Exit;
                }
            }
            else
            {
                /* shift to the write position */
                if (pFile->offsetLow != offset.low || pFile->offsetHigh != offset.high)
                {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                    eventInfo.offsetLow = offset.low;
                    eventInfo.offsetHigh = offset.high;
                    eventInfo.before = TRUE;
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_SEEK,
                        user->name,
                        user->ip,
                        0,
                        (const NQ_BYTE*)&eventInfo
                    );
                    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
                    if (sySeekFileStart(pFile->file, offset.low, offset.high) == (NQ_UINT32)NQ_FAIL)
                    {
                        returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                        udEventLog(
                            UD_LOG_MODULE_CS,
                            UD_LOG_CLASS_FILE,
                            UD_LOG_FILE_SEEK,
                            user->name,
                            user->ip,
                            error,
                            (const NQ_BYTE*)&eventInfo
                        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
                        LOGERR(CM_TRC_LEVEL_ERROR, "LSEEK failed");
                        goto Exit;
                    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_SEEK,
                        user->name,
                        user->ip,
                        0,
                        (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
                    pFile->offsetLow = offset.low;
                    pFile->offsetHigh = offset.high;
                }
        
                /* write to file */
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
                if (csDispatchIsDtIn())
                {
                    csDispatchDtSet(pFile->file, dataCount);
                }
                else
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
                {
                    dataCount = (NQ_UINT32)syWriteFile(pFile->file, pData, (NQ_COUNT)dataCount);
                    if ((NQ_INT)dataCount < 0)
                    {
                        returnValue = csErrorGetLast();
                        LOGERR(CM_TRC_LEVEL_ERROR, "WRITE failed");
                        goto Exit;
                    }

#ifdef UD_FS_FLUSHIFMODIFIED
                    pFile->wasModified = TRUE;
#endif /* UD_FS_FLUSHIFMODIFIED */
                }
                csGetNameByNid(pFile->nid)->isDirty = TRUE;   
            }
    
            /* update file offsets */
            pFile->offsetHigh = offset.high;
            pFile->offsetLow = offset.low + dataCount;
            if (pFile->offsetLow < offset.low)
            {
                pFile->offsetHigh++;
            }
        }
    }

    /* compose the response */
    cmBufferWriteUint16(writer, 17);            /* structure length */
    cmBufferWriteUint16(writer, 0);             /* reserved */
    cmBufferWriteUint32(writer, dataCount);     /* written */
    cmBufferWriteUint32(writer, 0);             /* remaining */
    cmBufferWriteUint16(writer, 0);             /* WriteChannelInfoOffset - unused*/
    cmBufferWriteUint16(writer, 0);             /* WriteChannelInfoLength - unused */

    returnValue = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: 0x%x", returnValue);
    return returnValue;
}

#ifdef UD_CS_INCLUDERPC

/*====================================================================
 * PURPOSE: save IOCTL parameters in late response context
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *
 * RETURNS: NONE
 *
 * NOTES:   skips Transact header
 *====================================================================
 */

static void
writeSmb2LateResponseSave(
    CSLateResponseContext* context
    )
{
    CMSmb2Header * pHeader;
    pHeader = cs2DispatchGetCurrentHeader();
    pHeader->aid.low = csSmb2SendInterimResponse(pHeader);
    pHeader->aid.high = 0;
    context->prot.smb2.commandData.write.dataCount = dataCount;
    context->file = pFile;

    /* write request information into the file descriptor */
    csDispatchSaveResponseContext(context);

    return;
}

/*====================================================================
 * PURPOSE: calculate command data pointer and size
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *
 * RETURNS: NQ_SUCCESS or error code
 *
 * NOTES:   skips Transact header
 *====================================================================
 */

static NQ_STATUS
writeSmb2LateResponsePrepare(
    CSLateResponseContext* context
    )
{
    csDispatchPrepareLateResponse(context);
    context->commandData += RESPONSE_LENGTH;
    context->commandDataSize -= RESPONSE_LENGTH;

    return NQ_SUCCESS;
}

/*====================================================================
 * PURPOSE: send a response using saved context
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *          IN status to return
 *          IN number of bytes to return in the data section
 *
 * RETURNS: TRUE on success
 *
 * NOTES:   data is ignored
 *====================================================================
 */

static NQ_BOOL
writeSmb2LateResponseSend(
    CSLateResponseContext* context,
    NQ_UINT32 status,
    NQ_COUNT dataLength
    )
{
    CMBufferWriter writer;

    /* save response for subsequent READ */
    csDcerpcSaveCompleteResponse((CSFile*)context->file, context->commandData, dataLength);
    
    /* compose Write response */    
    context->commandData -= RESPONSE_LENGTH;
    cmBufferWriterInit(&writer, context->commandData, RESPONSE_LENGTH + 10);
    cmBufferWriteUint16(&writer, 17);            /* structure length */
    cmBufferWriteUint16(&writer, 0);             /* reserved */
    cmBufferWriteUint32(&writer, context->prot.smb2.commandData.write.dataCount);
    cmBufferWriteUint32(&writer, 0);             /* remaining */
    cmBufferWriteUint16(&writer, 0);             /* WriteChannelInfoOffset - unused*/
    cmBufferWriteUint16(&writer, 0);             /* WriteChannelInfoL:ength - unused */

    return csDispatchSendLateResponse(context, status, RESPONSE_LENGTH);
}

#endif /* UD_CS_INCLUDERPC */

#if defined(UD_CS_INCLUDERPC_SPOOLSS) && !defined(UD_CS_SPOOLSS_PRN_USESPOOLERFILE)
/*====================================================================
 * PURPOSE: send a data to a printer
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to an opened file handle
 *          IN pointer to a data buffer
 *
 * RETURNS: NQ_SUCCESS on success, error code otherwise
 *
 * NOTES:   data is ignored
 *====================================================================
 */

static NQ_UINT32
prnWriteToPrinter(
    CSFile  *pFile,
    NQ_BYTE *pData              /* pointer to the buffer */
    )
{
    void            *p = NULL;
    NQ_UINT32       writtenCount;
    SYPrinterJob    *pJob = NULL;
    NQ_UINT32       returnValue = NQ_SUCCESS;        /* return code */

    writtenCount = (NQ_UINT32)syWritePrintData(pFile->printerHandle, (NQ_UINT32)pFile->file, pData, dataCount, &p);
    pJob = (SYPrinterJob *)p;
    if (0 >= writtenCount)
    {
        if (0 == writtenCount)
        {
            /* 0 bytes written or response has to be delayed */
            if ((NULL != pJob) && (NULL != pJob->delayed.ctx))
            {
                /* response has to be delayed */
                csDcerpcSaveResponseContext(FALSE, NULL, (CSDcerpcResponseContext *)pJob->delayed.ctx);
                returnValue = SMB_STATUS_NORESPONSE;
                goto Exit;
            }
        }

        returnValue =  csErrorGetLast();
        goto Exit;
    }

    cmU64AddU32(&((CSPrnDisorderedContext*)(pFile->memorySpooler.context))->currentOffset, writtenCount);

Exit:
    return returnValue;
}

/*====================================================================
 * PURPOSE: print all available packets data
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to an opened file handle
 *
 * RETURNS: NQ_SUCCESS on success, error code otherwise
 *
 * NOTES:   data is ignored
 *====================================================================
 */

static NQ_UINT32
prnPrintDisorderedPackets(
    CSFile  *pFile
    )
{
    NQ_UINT32   returnValue = NQ_SUCCESS;                               /* return code */
    NQ_UINT     *currentPosition = &pFile->memorySpooler.currentPosition;   /* A pointer to the current position */

    /* Write the next packet data (if exists) */
    while (NULL != pFile->memorySpooler.items[*currentPosition])
    {
        returnValue = prnWriteToPrinter(pFile, ((CSPrnDisorderedData*)pFile->memorySpooler.items[*currentPosition])->buffer);
        if (NQ_SUCCESS != returnValue)
        {
            goto Exit;
        }

        /* Add the memory back to the pool for reuse */
        cmListItemAdd(&pFile->memorySpooler.memoryPool, (CMItem*)pFile->memorySpooler.items[*currentPosition], NULL);
        pFile->memorySpooler.items[*currentPosition] = NULL;
        *currentPosition = (((*currentPosition) + 1) % UD_CS_SPOOLSS_PRN_MAXDISORDEREDPACKETS);
        pFile->memorySpooler.numberOfItems--;
    }

    /* If there are no more items in the array initialize the currentPosition counter */
    if (0 == pFile->memorySpooler.numberOfItems)
    {
        *currentPosition = 0;
    }
    else
    {
        /* In case we reached NULL pointer we need to */
        *currentPosition = (((*currentPosition) + 1) % UD_CS_SPOOLSS_PRN_MAXDISORDEREDPACKETS);
    }

Exit:
    return returnValue;
}

/*====================================================================
 * PURPOSE: save a disordered packet data
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to an opened file handle
 *          IN pointer to a data buffer
 *          IN pointer to the offset of the new packet
 *
 * RETURNS: NQ_SUCCESS on success, error code otherwise
 *
 * NOTES:   data is ignored
 *====================================================================
 */

static NQ_UINT32
prnSaveDisorderedPackets(
    CSFile      *pFile,
    NQ_BYTE     *pData,     /* pointer to the buffer */
    NQ_UINT64   *pOffset    /* write offset */
    )
{
    NQ_UINT32       returnValue = NQ_SUCCESS;   /* return code */
    CSPrnDisorderedData  *nextWrite;                 /* a buffer to store the disordered packet data */
    NQ_UINT64       tempOffset;                 /* a temp buffer to calculate the offset between the new packet to the current offset */
    NQ_UINT         newPosition;                /* a temp buffer to calculate the position that the new disordered packet data will be stored
                                                   inside the array of pointers */

    if (UD_CS_SPOOLSS_PRN_MAXDISORDEREDPACKETS <= pFile->memorySpooler.numberOfItems)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Reached upper limit of allocation, enlarge UD_CS_SPOOLSS_PRN_MAXDISORDEREDPACKETS value");
        returnValue = SMB_STATUS_NO_MEMORY;
        goto Exit;
    }

    cmU64SubU64U64(&tempOffset, pOffset, &((CSPrnDisorderedContext*)(pFile->memorySpooler.context))->currentOffset);
    newPosition = (((tempOffset.low / pFile->memorySpooler.divider) + pFile->memorySpooler.currentPosition - 1) % UD_CS_SPOOLSS_PRN_MAXDISORDEREDPACKETS); /* -1 for array alignment (starts from 0) */

    /* If there is an available unused memory block in the memoryPool, use it */
    if (NULL != pFile->memorySpooler.memoryPool.first)
    {
        nextWrite = (CSPrnDisorderedData*)pFile->memorySpooler.memoryPool.first;
        cmListItemRemove(pFile->memorySpooler.memoryPool.first);
    }
    else /* Else allocate a new memory block */
    {
        nextWrite = (CSPrnDisorderedData*)cmListItemCreate(sizeof(CSPrnDisorderedData), NULL, CM_LISTITEM_NOLOCK, FALSE);
        if (NULL == nextWrite)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            returnValue = SMB_STATUS_NO_MEMORY;
            goto Exit;
        }
    }

    /* Point to the new data */
    pFile->memorySpooler.items[newPosition] = nextWrite;
    syMemcpy(((CSPrnDisorderedData*)pFile->memorySpooler.items[newPosition])->buffer, pData, dataCount);
    pFile->memorySpooler.numberOfItems++;

Exit:
    return returnValue;
}
#endif /* defined(UD_CS_INCLUDERPC_SPOOLSS) && !defined(UD_CS_SPOOLSS_PRN_USESPOOLERFILE) */

#endif /* defined(UD_NQ_INCLUDECIFSSERVER) && defined(UD_NQ_INCLUDESMB2) */

