/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : SMB2 Read command handler
 *--------------------------------------------------------------------
 * MODULE        : Server
 * DEPENDENCIES  :
 ********************************************************************/

#include "csparams.h"
#include "csutils.h"
#include "csdcerpc.h"
#include "cs2disp.h"

#if defined(UD_NQ_INCLUDECIFSSERVER) && defined(UD_NQ_INCLUDESMB2)

#define RESPONSE_LENGTH 16  /* length of the read response not including data */

/*====================================================================
 * PURPOSE: Perform Read processing
 *--------------------------------------------------------------------
 * PARAMS:  IN in - pointer to the parsed SMB2 header descriptor
 *          OUT out - pointer to the response header structure
 *          IN reader - request reader pointing to the second command field
 *          IN connection - pointer to the session structure
 *          IN user - pointer to the user structure
 *          IN tree - pointer to the tree structure
 *          OUT writer - pointer to the response writer
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   This function is called on SMB2 Create command.
 *====================================================================
 */

NQ_UINT32 csSmb2OnRead(CMSmb2Header *in, CMSmb2Header *out, CMBufferReader *reader, CSSession *connection, CSUser *user, CSTree *tree, CMBufferWriter *writer)
{
    CSFile* pFile;                          /* pointer to file descriptor */
    CSFid fid;                              /* fid of the file to close */
    NQ_UINT32 dataCount;                    /* requested data count */
    NQ_UINT32 maxCount;                     /* available data count */
    NQ_UINT32 minCount;                     /* buffer length */
    NQ_UINT64 offset;                       /* read offset */
    NQ_BYTE * pDataCount;                   /* saved pointer to data length in response */
    NQ_BYTE * pData;                        /* pointer to the buffer */
#ifdef UD_CS_FORCEINTERIMRESPONSES
    NQ_UINT32 asyncId = 0;                  /* generated Async ID */
#endif /* UD_CS_FORCEINTERIMRESPONSES */
    NQ_UINT32 readCount;                    /* read count */
    NQ_UINT32 immediateDataCount;           /* number of bytes in the packet (not including DT) */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    UDFileAccessEvent eventInfo;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    NQ_UINT32 returnValue;                  /* return code */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "in:%p out:%p reader:%p connection:%p user:%p tree:%p writer:%p", in, out, reader, connection, user, tree, writer);

    /* parse request */
    cmBufferReaderSkip(reader, 2);  /* padding + reserved */
    cmBufferReadUint32(reader, &dataCount);
    cmBufferReadUint64(reader, &offset);
    cmBufferReadUint16(reader, &fid);
    cs2ParseFid(&fid);
    cmBufferReaderSkip(reader, 14); /* the rest of the file ID */
    cmBufferReadUint32(reader, &minCount);
   
    if (dataCount > (CIFS_MAX_DATA_SIZE - SMB2_HEADERSIZE - 16))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Buffer overflow: read length too big");
        returnValue = SMB_STATUS_INVALID_PARAMETER;
        goto Exit;
    }
    
    /* find file descriptor */
    pFile = csGetFileByFid(fid, tree->tid, user->uid);
    if (pFile == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unknown FID");
        returnValue = SMB_STATUS_INVALID_HANDLE;
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "pFile:%p file:%d dataCount:%lu offset:%lu%lu", pFile, pFile->file, (NQ_ULONG)dataCount, (NQ_ULONG)(offset.high), (NQ_ULONG)(offset.low));

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.fileName = csGetFileName(pFile->fid);
    eventInfo.tid = tree->tid;
    eventInfo.rid = csGetUserRid(user);
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    /* check available room in the buffer */
    maxCount = CS_SMB2_MAX_READ_SIZE;
    if (dataCount > maxCount)
    {
        dataCount = maxCount;
    }

    /* start composing response */
    cmBufferWriteUint16(writer, 17);   /* structure length */
    cmBufferWriteByte(writer, SMB2_HEADERSIZE + RESPONSE_LENGTH);
    cmBufferWriteByte(writer, 0);   /* reserved */
    pDataCount = cmBufferWriterGetPosition(writer);
    cmBufferWriteUint32(writer, 0); /* data count - initially */
    cmBufferWriteUint32(writer, 0); /* remaining */
    cmBufferWriteZeroes(writer, 4); /* reserved */
    pData = cmBufferWriterGetPosition(writer);

#ifdef UD_CS_INCLUDERPC
    if (pFile->isPipe)
    {
        /* read from pipe */
        readCount = csDcerpcRead(pFile, pData, (NQ_UINT)dataCount, NULL);
        if (0 == readCount)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "error reading from pipe");
            returnValue = SMB_STATUS_INVALID_PIPE_STATE;
            goto Exit;
        }

        immediateDataCount = readCount;
    }
    else
#endif /* UD_CS_INCLUDERPC */
    {
        /* send interim response */
#ifdef UD_CS_FORCEINTERIMRESPONSES
        asyncId = csSmb2SendInterimResponse(in);
        if (0 == asyncId)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "error sending interim read response");
            returnValue = SMB_STATUS_INVALID;
            goto Exit;
        }
        out->flags |= SMB2_FLAG_ASYNC_COMMAND;
        out->aid.low = asyncId;
        out->aid.high = 0;
        out->credits = 0;
#endif /* UD_CS_FORCEINTERIMRESPONSES */

        /* position to the offset */
        if (pFile->offsetLow != offset.low || pFile->offsetHigh != offset.high)
        {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            eventInfo.offsetLow = offset.low;
            eventInfo.offsetHigh = offset.high;
            eventInfo.before = TRUE;
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_SEEK,
                user->name,
                user->ip,
                0,
                (const NQ_BYTE*)&eventInfo
            );
            eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
            if ((maxCount = sySeekFileStart(pFile->file, offset.low, offset.high)) != offset.low)
            {
                returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_SEEK,
                    user->name,
                    user->ip,
                    returnValue,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
                LOGERR(CM_TRC_LEVEL_ERROR, "Seek failed. Expected: %d, returned %d", offset.low, maxCount);
                goto Exit;
            }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_SEEK,
                user->name,
                user->ip,
                0,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        }

        /* read from file */
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
        /* for DT determine end of file before actual reading from file into socket,
         * and actual read count (can be less than requested due to end of file),
         * because SMB header with status and dataCount is returned first, then the payload of DT */
        if (csDispatchIsDtOut() && (dataCount > 0))
        {
            NQ_UINT64 remaining;        /* remaining file portion to read */
            NQ_UINT64 fileSize;         /* file size */
            NQ_UINT64 count;            /* dataCount in uint64 */

            /* get file size */
            if (NQ_FAIL == syGetFileSize(pFile->file, &fileSize))
            {
                returnValue = csErrorGetLast();
                LOGERR(CM_TRC_LEVEL_ERROR, "DT Read failed: file size");
                goto Exit;
            }

            /* offset >= fileSize */
            if (0 <= cmU64Cmp(&offset, &fileSize))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "DT Read failed: end of file");
                returnValue =  SMB_STATUS_END_OF_FILE;
                goto Exit;
            }

            /* remaining = fileSize - offset */
            cmU64SubU64U64(&remaining, &fileSize, &offset);

            /* remaining < count */
            count.high = 0;
            count.low = dataCount;
            if (-1 == cmU64Cmp(&remaining, &count))
            {
                dataCount = remaining.low;
            }

            csDispatchDtSet(pFile->file, dataCount);
            immediateDataCount = 0;
            readCount = dataCount;
        }
        else
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
        {
            NQ_INT readResult;

            readResult = syReadFile(pFile->file, pData, (NQ_COUNT)dataCount);
            if ((readResult <= 0) && (dataCount != 0))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Read failed: end of file");

                /* update file pointer */
                pFile->offsetHigh = pFile->offsetLow = 0;
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                eventInfo.offsetLow = pFile->offsetLow;
                eventInfo.offsetHigh = pFile->offsetHigh;
                eventInfo.before = TRUE;
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_SEEK,
                    user->name,
                    user->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                );
                eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
                sySeekFileStart(pFile->file, pFile->offsetLow, pFile->offsetHigh);
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_SEEK,
                    user->name,
                    user->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
                returnValue =  SMB_STATUS_END_OF_FILE;
                goto Exit;
            }

            if (readResult < 0)
            {
                returnValue = csErrorGetLast();
                LOGERR(CM_TRC_LEVEL_ERROR, "Read failed");
                goto Exit;
            }
            readCount = (NQ_UINT32)readResult;
            immediateDataCount = readCount;
        }
        
        /* update file offsets */
        pFile->offsetHigh = offset.high;
        pFile->offsetLow = offset.low + readCount;
        if (pFile->offsetLow < offset.low)
        {
            pFile->offsetHigh++;
        }
    } /* !pFile->isPipe */

    /* compose the response */
    pData += immediateDataCount;
    cmBufferWriterSetPosition(writer, pDataCount);
    cmBufferWriteUint32(writer, readCount);
    cmBufferWriterSetPosition(writer, pData);

    returnValue = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: 0x%x", returnValue);
    return returnValue;
}

#endif /* defined(UD_NQ_INCLUDECIFSSERVER) && defined(UD_NQ_INCLUDESMB2) */

