/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : SMB2 command dispatcher
 *--------------------------------------------------------------------
 * MODULE        : Server
 * DEPENDENCIES  :
 ********************************************************************/

#include "csnotify.h"
#include "cs2disp.h"
#include "csdispat.h"
#include "nssocket.h"
#ifdef UD_CS_MESSAGESIGNINGPOLICY
#include "cssignin.h"
#endif

#if defined(UD_NQ_INCLUDECIFSSERVER) && defined(UD_NQ_INCLUDESMB2)

typedef struct
{
    NQ_BYTE responseBuffer[CM_NB_DATAGRAMBUFFERSIZE];/* buffer for late response */
    NSSocketHandle savedSocket;         /* handle of the current socket */
    CSFid quickFid;                     /* saved fid for compounded requests */
    CMSmb2Header header;                /* buffer for a current header (do not use CMSmb2Header._start) */
    NQ_BOOL isEncrypedPacket;           /* whether packet is encrypted */
    NSRecvDescr *recvDescr;             /* saved descriptor */
    NQ_BYTE *response;                  /* saved response buffer */
    CMBufferWriter *primary;            /* saved primary response buffer writer */
#ifdef UD_CS_MESSAGESIGNINGPOLICY
    CMBufferWriter *packet;             /* saved signed response buffer writer */
#endif /* UD_CS_MESSAGESIGNINGPOLICY */
}
StaticData;

#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

/*
 * SMB2 command handler
 * In terms of SMB2:
 * CSSession -> connection
 * CSUser    -> session
 * CSTree    -> tree
 */
typedef NQ_UINT32 (*Smb2CommandHandler)(CMSmb2Header *in, CMSmb2Header *out, CMBufferReader *reader, CSSession *connection, CSUser *session, CSTree *tree, CMBufferWriter *writer);

/* SMB2 command table entry */
typedef struct
{
/*  NQ CHAR * name; */
    Smb2CommandHandler handler;
    NQ_UINT16 size;
    NQ_UINT16 flags;
}
Entry;

#define FLAG_NOCONNECTION   0x0001
#define FLAG_NOSESSION      0x0002
#define FLAG_NOTREE         0x0004
#define FLAG_DTIN           0x0008  /* incoming Direct Transfer */
#define FLAG_DTOUT          0x0010  /* outgoing Direct Transfer */
#define FLAG_ASYNC          0x0020  /* support asynchronous request */

/*
 * SMB2 command table
 */
static const Entry _entries[] = {
    /* SMB2_COM_NEGOTIATE       0x0000 */  {/*"NEGOTIATE",*/       csSmb2OnNegotiate,      36, FLAG_NOCONNECTION | FLAG_NOSESSION | FLAG_NOTREE},
    /* SMB2_COM_SESSION_SETUP   0x0001 */  {/*"SESSION_SETUP",*/   csSmb2OnSessionSetup,   25, FLAG_NOSESSION | FLAG_NOTREE},
    /* SMB2_COM_LOGOFF          0x0002 */  {/*"LOGOFF",*/          csSmb2OnLogoff,          4, FLAG_NOTREE},
    /* SMB2_COM_TREE_CONNECT    0x0003 */  {/*"TREE_CONNECT",*/    csSmb2OnTreeConnect,     9, FLAG_NOTREE},
    /* SMB2_COM_TREE_DISCONNECT 0x0004 */  {/*"TREE_DISCONNECT",*/ csSmb2OnTreeDisconnect,  4, 0},
    /* SMB2_COM_CREATE          0x0005 */  {/*"CREATE",*/          csSmb2OnCreate,         57, 0},
    /* SMB2_COM_CLOSE           0x0006 */  {/*"CLOSE",*/           csSmb2OnClose,          24, 0},
    /* SMB2_COM_FLUSH           0x0007 */  {/*"FLUSH",*/           csSmb2OnFlush,          24, 0},
    /* SMB2_COM_READ            0x0008 */  {/*"READ",*/            csSmb2OnRead,           49, FLAG_DTOUT},
    /* SMB2_COM_WRITE           0x0009 */  {/*"WRITE",*/           csSmb2OnWrite,          49, FLAG_DTIN},
    /* SMB2_COM_LOCK            0x000A */  {/*"LOCK",*/            csSmb2OnLock,           48, 0},
    /* SMB2_COM_IOCTL           0x000B */  {/*"IOCTL",*/           csSmb2OnIoctl,          57, 0},
    /* SMB2_COM_CANCEL          0x000C */  {/*"CANCEL",*/          csSmb2OnCancel,          4, FLAG_ASYNC}, /*[MS-SMB2] section 2.2.1*/
    /* SMB2_COM_ECHO            0x000D */  {/*"ECHO",*/            csSmb2OnEcho,            4, FLAG_NOSESSION | FLAG_NOTREE},
    /* SMB2_COM_QUERY_DIRECTORY 0x000E */  {/*"QUERY_DIRECTORY",*/ csSmb2OnQueryDirectory, 33, 0},
    /* SMB2_COM_CHANGE_NOTIFY   0x000F */  {/*"CHANGE_NOTIFY",*/   csSmb2OnChangeNotify,   32, 0},
    /* SMB2_COM_QUERY_INFO      0x0010 */  {/*"QUERY_INFO",*/      csSmb2OnQueryInfo,      41, 0},
    /* SMB2_COM_SET_INFO        0x0011 */  {/*"SET_INFO",*/        csSmb2OnSetInfo,        33, 0},
    /* SMB2_COM_OPLOCK_BREAK    0x0012 */  {/*"OPLOCK_BREAK",*/    csSmb2OnOplockBreak,    24, 0},
};

static void releaseCallback(const NQ_BYTE* buffer)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);
    nsPutBuffer((NQ_BYTE*)buffer);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static void writeErrorResponseData(CMBufferWriter *writer)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);
    cmBufferWriteUint16(writer, 9);  /* size */
    cmBufferWriteUint16(writer, 0);  /* reserved */
    cmBufferWriteUint32(writer, 0);  /* byte count */
    cmBufferWriteByte(writer, 0);    /* 1 byte of data (required if ByteCount == 0) */
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static void setQuickFid(CSFid* fid)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);
    staticData->quickFid = *fid;
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static void getQuickFid(CSFid* fid)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);
    *fid = staticData->quickFid;
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static void resetQuickFid()
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);
    staticData->quickFid = CS_ILLEGALID;
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*====================================================================
 * PURPOSE: Parse FID
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT pointer to fid
 *
 * RETURNS: none
 *
 * NOTES:   According to fid value save it (for usage of further requests
 *          in the same packet) or get previously saved one.
 *====================================================================
 */
void cs2ParseFid(CSFid* fid)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);
    (*fid == CS_ILLEGALID) ? getQuickFid(fid) : setQuickFid(fid);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}


/*====================================================================
 * PURPOSE: initialize resources
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *====================================================================
 */

NQ_STATUS
cs2DispatchInit(
    void
    )
{
    NQ_STATUS res = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocate(sizeof(*staticData));
    if (NULL == staticData)
    {
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */
    staticData->isEncrypedPacket = FALSE;
    staticData->response = NULL;
    staticData->recvDescr = NULL;
    staticData->primary = NULL;
    staticData->savedSocket = NULL;
    staticData->quickFid = CS_ILLEGALID;
#ifdef UD_CS_MESSAGESIGNINGPOLICY
    staticData->packet = NULL;
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

    res = NQ_SUCCESS;
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", res);
    return res;
}

/*====================================================================
 * PURPOSE: release resources
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
cs2DispatchExit(
    void
    )
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    /* release memory */
#ifdef SY_FORCEALLOCATION
    if (NULL != staticData)
    {
        cmMemoryFree(staticData);
    }
    staticData = NULL;
#endif /* SY_FORCEALLOCATION */

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*====================================================================
 * PURPOSE: SMB2 async ID generator
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for next async ID
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void cs2GenerateNextAsyncId(NQ_UINT64 * id)
{
    static NQ_UINT32 nextAsyncId = 1;

    nextAsyncId++;
    if (0 == nextAsyncId)
    {
        nextAsyncId = 1;
    }
    id->high = 0;
    id->low = nextAsyncId;
}

/*====================================================================
 * PURPOSE: Get current command header
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: Pointer to the header
 *
 * NOTES:   This function should be called only inside the csSmb2DispatchRequest()
 *          processing
 *====================================================================
 */

CMSmb2Header *
cs2DispatchGetCurrentHeader(
    void
    )
{
    return &staticData->header;
}

/* Write the response packet (header, signing) */
void cs2WriteResponse(
        NQ_UINT32 result,
        CMSmb2Header *in,
        CMSmb2Header *out,
        CMBufferWriter *data,
        CSSession *connection
)
{
    CMBufferWriter *primary = staticData->primary;
#ifdef UD_NQ_INCLUDESMB3
    NQ_BOOL isEncrypedPacket = staticData->isEncrypedPacket;
#endif /* UD_NQ_INCLUDESMB3 */
#ifdef UD_CS_MESSAGESIGNINGPOLICY
    CMBufferWriter *packet = staticData->packet;
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "result:0x%08x in:%p out:%p data:%p connection:%p", result, in, out, data, connection);

    /* on error write error response data part */
    if (result != SMB_STATUS_INTERNAL_RELEASE_REQUEST_RESPONSE)
    {
        out->status = result;
        if (out->status != 0 && out->status != SMB_STATUS_MORE_PROCESSING_REQUIRED)
        {
            cmBufferWriterReset(data);
            writeErrorResponseData(data);
        }
    }

    /* if compounded request align writer and advance reader properly */
    if (in->next > 0)
    {
        cmSmb2HeaderAlignWriter(out, data, 8);
        out->next = cmSmb2HeaderGetWriterOffset(out, data);
    }

    /* write response header (including processing status) */
    cmSmb2HeaderWrite(out, primary);

    /* prepare main writer for the next header */
    cmBufferWriterSync(primary, data);

#ifdef UD_CS_MESSAGESIGNINGPOLICY
    cmBufferWriterSync(packet, primary);
#ifdef UD_NQ_INCLUDESMB3
    if (connection->dialect >= CS_DIALECT_SMB30)
    {
        if (!isEncrypedPacket)
        {
            csCreateMessageSignatureSMB3(out->sid.low, cmBufferWriterGetStart(packet), cmBufferWriterGetDataCount(packet));
        }
    }
    else
#endif /* UD_NQ_INCLUDESMB3 */
    {
        csCreateMessageSignatureSMB2(out->sid.low, cmBufferWriterGetStart(packet), cmBufferWriterGetDataCount(packet));
    }
    cmBufferWriterBranch(primary, packet, 0);
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Response: command=%u, mid=%u/%u, sid.low=0x%x, signed:%d, async:%d, pid(async.high)=0x%x, tid(async.low)=0x%x, chained: %d, connection: %p",
            out->command, out->mid.high, out->mid.low, out->sid.low, (out->flags & SMB2_FLAG_SIGNED) > 0, (out->flags & SMB2_FLAG_ASYNC_COMMAND) > 0,
            out->flags & SMB2_FLAG_ASYNC_COMMAND ? out->aid.high : out->pid, out->flags & SMB2_FLAG_ASYNC_COMMAND ? out->aid.low : out->tid, out->next > 0, connection);

    LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
}

/* sends the packet, always releases response buffer */
NQ_BOOL cs2SendResponse(
        CMSmb2Header *out,
        CSSession *connection
        )
{
#if defined(UD_CS_INCLUDEDIRECTTRANSFER) || defined(UD_NQ_INCLUDESMB3)
    NSRecvDescr *recvDescr = staticData->recvDescr;
#endif /* defined(UD_CS_INCLUDEDIRECTTRANSFER) || defined(UD_NQ_INCLUDESMB3) */
    NQ_BYTE *response = staticData->response;
    CMBufferWriter *primary = staticData->primary;
    NQ_INT written, sent;
    NQ_COUNT packetLen;
#ifdef UD_NQ_INCLUDESMBCAPTURE
    NQ_UINT packetLenForIntCap;                     /* packet length that will be written to internal capture */
    NQ_UINT sizeToWriteToIntCap;                    /* size of the next buffer that will be written to internal capture */
    NQ_BYTE* tempBufferForIntCap = NULL;            /* temporary buffer to save the data that should be written later to internal capture */
    CSSocketDescriptor * sockDescr;
#endif /* UD_NQ_INCLUDESMBCAPTURE */
#ifdef UD_NQ_INCLUDESMB3
    static NQ_BYTE ctxBuff[SHA512_CTXSIZE];
    NQ_BOOL isEncrypedPacket = staticData->isEncrypedPacket;
#endif /* UD_NQ_INCLUDESMB3 */
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "out:%p connection:%p", out, connection);

    written = (NQ_INT)cmBufferWriterGetDataCount(primary);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "response packet size=%d", written);

#ifdef UD_CS_INCLUDEDIRECTTRANSFER
    if (csDispatchIsDtOut() && csDispatchDtAvailable())
    {
        packetLen = (NQ_COUNT)((NQ_COUNT)written + csDispatchDtGetCount());
    }
    else
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
    {
        packetLen = (NQ_COUNT)written;
    }

#ifdef UD_NQ_INCLUDESMBCAPTURE
    /* fill tempBufferForIntCap with data to write it into internal capture after the packet will be sent */
    packetLenForIntCap = (NQ_UINT)packetLen;
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
    if (csDispatchIsDtOut() && csDispatchDtAvailable())
    {
        sizeToWriteToIntCap = (NQ_UINT)written;
    }
    else
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
    {
        sizeToWriteToIntCap = (NQ_UINT)packetLen;
    }

    tempBufferForIntCap = cmMemoryAllocate(sizeToWriteToIntCap);
    if (NULL != tempBufferForIntCap)
    {
        syMemcpy(tempBufferForIntCap, response + 4, sizeToWriteToIntCap);
    }

#endif /* UD_NQ_INCLUDESMBCAPTURE */

#ifdef UD_NQ_INCLUDESMB3
    /* calculate hash on response packets. request calculated above */
    if (connection->dialect == CS_DIALECT_SMB311)
    {
        if (out->command == SMB2_CMD_NEGOTIATE && connection->preauthIntegOn)
        {
            cmSmb311CalcMessagesHash(primary->origin, (NQ_UINT)(written), connection->preauthIntegHashVal, ctxBuff);
            connection->preauthIntegOn = FALSE;
        }
        if (out->command == SMB2_CMD_SESSIONSETUP)
        {
            CSUser *pUser = csGetUserByUid((CSUid)sessionIdToUid(out->sid.low));

            if (pUser != NULL && pUser->preauthIntegOn)
            {
                cmSmb311CalcMessagesHash(primary->origin, (NQ_UINT)(written), pUser->preauthIntegHashVal, ctxBuff);
            }
        }
    }

    if (isEncrypedPacket)
    {
        NQ_INT hdrSize = SMB2_TRANSFORMHEADER_SIZE;

        hdrSize -= (nsSkipHeader(recvDescr->socket, response) == response) ? 0 : 4;
        cs2TransformHeaderEncrypt(csGetUserByUid((CSUid)sessionIdToUid(out->sid.low)), response - hdrSize, (NQ_COUNT)written);
        response -= SMB2_TRANSFORMHEADER_SIZE;
        written += SMB2_TRANSFORMHEADER_SIZE;
        packetLen = (NQ_COUNT)written;
    }
#endif /* UD_NQ_INCLUDESMB3 */

    written = (NQ_INT)nsPrepareNBBuffer(response, (NQ_UINT)packetLen, (NQ_UINT)written);
    if (0 == written)
    {
        /* error */
        LOGERR(CM_TRC_LEVEL_ERROR, "prepare buffer failed");
        releaseCallback((const NQ_BYTE *)response);
        goto Exit;
    }

    if ((sent = nsSendFromBuffer(
                    staticData->savedSocket,
                    response,
                    (NQ_UINT)packetLen,
                    (NQ_UINT)written,
                    &releaseCallback)
        ) != written)
    {
        /* error */
        LOGERR(CM_TRC_LEVEL_ERROR, "sending response packet failed: size=%d, sent=%d", written, sent);
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "sent response packet: size=%d", sent);

#ifdef UD_NQ_INCLUDESMBCAPTURE
    sockDescr = csGetClientSocketDescriptorBySocket(staticData->savedSocket);
    if (NULL != sockDescr)
    {
        sockDescr->captureHdr.receiving = FALSE;
        cmCapturePacketWriteStart(&sockDescr->captureHdr, packetLenForIntCap);
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
        if (csDispatchIsDtOut() && csDispatchDtAvailable())
        {
            NQ_BYTE *   tempBuf;
            NQ_COUNT    len;

            if (NULL != tempBufferForIntCap)
            {
                cmCapturePacketWritePacket(tempBufferForIntCap, sizeToWriteToIntCap);
            }

            len = csDispatchDtGetCount();
            tempBuf = (NQ_BYTE *)cmMemoryAllocate(len);
            if (NULL != tempBuf)
            {
                syMemset(tempBuf , 0 , len);
                cmCapturePacketWritePacket(tempBuf , len);
                cmMemoryFree(tempBuf);
            }
        }
        else
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
        {
            if (NULL != tempBufferForIntCap)
            {
                cmCapturePacketWritePacket(tempBufferForIntCap, sizeToWriteToIntCap);
            }
        }

        cmCapturePacketWriteEnd();
    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */

    result = TRUE;

#ifdef UD_CS_INCLUDEDIRECTTRANSFER
    if (csDispatchIsDtOut())
    {
        /* Transfer bytes from file to socket */
        if (!csDispatchDtToSocket(recvDescr))
        {
            result = TRUE;
            goto Exit;
        }
        syDtEndPacket(((SocketSlot*)recvDescr->socket)->socket);
    }
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

Exit:
#ifdef UD_NQ_INCLUDESMBCAPTURE
    if (NULL != tempBufferForIntCap)
    {
        cmMemoryFree(tempBufferForIntCap);
        tempBufferForIntCap = NULL;
    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */

    staticData->response = NULL; /* freed already */
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/*====================================================================
 * PURPOSE: SMB2 command dispatcher
 *--------------------------------------------------------------------
 * PARAMS:  IN receive descriptor
 *          IN request - pointer to the incoming request packet
 *          IN length -  incoming packet length without four bytes of signature
 *
 * RETURNS: TRUE on success or FALSE on any error
 *
 * NOTES:   The buffer passed from the outside will be released by caller!
 *          4 bytes of SMB2 signature are already read through the descriptor
 *====================================================================
 */

NQ_BOOL
csSmb2DispatchRequest(
    NSRecvDescr * recvDescr,
    NQ_BYTE * request,
    NQ_COUNT length
    )
{
    CMSmb2Header in, out;
    CMBufferReader reader;
    CMBufferWriter primary, data;
    NQ_BYTE   *allocatedResponse;
    NQ_UINT32 commandResult;
    NQ_BYTE *pBuf = request + sizeof(cmSmb2TrnsfrmHdrProtocolId);
    NQ_BOOL isFirstInChain = TRUE;
    CSUser *session = NULL;
    CSSession *connection = NULL;
#ifdef UD_CS_MESSAGESIGNINGPOLICY
    NQ_COUNT dataLength;
    CMBufferWriter packet;
#endif /* UD_CS_MESSAGESIGNINGPOLICY */
#ifdef UD_CS_SMB2_INCLUDEMULTICREDIT
    NQ_UINT16 creditCharge = 0;     /* number of charged credits */
    NQ_UINT creditsForResponse = 0;
#endif /* UD_CS_SMB2_INCLUDEMULTICREDIT */
    NQ_BOOL result = FALSE;
#ifdef UD_NQ_INCLUDESMBCAPTURE
    NQ_BOOL isPaddingNeededForCapture = FALSE; /* indicate if zeros padding is needed for internal capture */
    NQ_BYTE *captureBuf;                    /* buffer to write zeros into capture */
    NQ_UINT numOfZerosToFill = length;      /* number of zeros to fill for the last uncompleted packet */
#endif /* UD_NQ_INCLUDESMBCAPTURE */

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "recvDescr:%p request:%p length:%d", recvDescr, request, length);

#ifdef UD_NQ_INCLUDESMB3
    staticData->isEncrypedPacket = (0 == syMemcmp(request, cmSmb2TrnsfrmHdrProtocolId, sizeof(cmSmb2TrnsfrmHdrProtocolId)));
#ifdef UD_NQ_INCLUDESMBCAPTURE
    if (staticData->isEncrypedPacket)
    {
        numOfZerosToFill = (NQ_UINT)length + (NQ_UINT)sizeof(cmSmb2ProtocolId) - SMB2_TRANSFORMHEADER_SIZE;
    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
#else /* UD_NQ_INCLUDESMB3 */
    staticData->isEncrypedPacket = FALSE;
#endif /* UD_NQ_INCLUDESMB3 */

    if ((length + sizeof(cmSmb2ProtocolId)) < SMB2_HEADERSIZE || (length + sizeof(cmSmb2ProtocolId)) > UD_NS_BUFFERSIZE)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid packet length:%d", (NQ_INT)(length + sizeof(cmSmb2ProtocolId)));
#ifdef UD_NQ_INCLUDESMBCAPTURE
        isPaddingNeededForCapture = TRUE;
#endif /* UD_NQ_INCLUDESMBCAPTURE */
        goto Exit;
    }

    if (NULL == (connection = csGetSessionBySocket()))
    {
        if (NULL == (connection = csGetNewSession()))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "connection not found (reached max)");
#ifdef UD_NQ_INCLUDESMBCAPTURE
            isPaddingNeededForCapture = TRUE;
#endif /* UD_NQ_INCLUDESMBCAPTURE */
            goto Exit;
        }
    }

    staticData->savedSocket = recvDescr->socket;
    staticData->recvDescr = recvDescr;

    /* allocate response buffer, according to implementation can't be NULL */
    allocatedResponse = staticData->response = nsGetBuffer();

    /* always respond NT STATUS */
    csDispatchSetNtError(TRUE);
#ifdef UD_NQ_INCLUDESMB3
    if (staticData->isEncrypedPacket)
    {
        if (cs2TransformHeaderDecrypt(recvDescr, request, length))
        {
            pBuf = request + SMB2_TRANSFORMHEADER_SIZE + (NQ_COUNT)sizeof(cmSmb2ProtocolId);
            length = length - SMB2_TRANSFORMHEADER_SIZE + (NQ_COUNT)sizeof(cmSmb2ProtocolId);
        }
        else
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Encrypted Packet Signature doesn't match");
#ifdef UD_NQ_INCLUDESMBCAPTURE
            isPaddingNeededForCapture = TRUE;
#endif /* UD_NQ_INCLUDESMBCAPTURE */
            goto ExitEndRecv;
        }
    }
#endif /* UD_NQ_INCLUDESMB3 */

    if (!staticData->isEncrypedPacket)
    {
        if (NQ_FAIL == nsRecvIntoBuffer(recvDescr, pBuf, SMB2_HEADERANDSTRUCTSIZE - sizeof(cmSmb2ProtocolId))) /* read the rest of the header + StructureSize */
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Error reading from socket");
#ifdef UD_NQ_INCLUDESMBCAPTURE
            isPaddingNeededForCapture = TRUE;
#endif /* UD_NQ_INCLUDESMBCAPTURE */
            goto ExitEndRecv;
        }
    }
    else
    {
        syMemset(staticData->response, 0, UD_NS_BUFFERSIZE);
        staticData->response += SMB2_TRANSFORMHEADER_SIZE;
    }

#ifdef UD_NQ_INCLUDESMBCAPTURE
    if (!staticData->isEncrypedPacket)
    {
        cmCapturePacketWritePacket(request + sizeof(cmSmb2ProtocolId), SMB2_HEADERANDSTRUCTSIZE - sizeof(cmSmb2ProtocolId));
    }
    else
    {
        cmCapturePacketWritePacket(pBuf - sizeof(cmSmb2ProtocolId) , length);
        cmCapturePacketWriteEnd();
    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    cmBufferReaderInit(&reader, !staticData->isEncrypedPacket ? request : (pBuf - sizeof(cmSmb2ProtocolId)), !staticData->isEncrypedPacket ? (length + (NQ_COUNT)sizeof(cmSmb2ProtocolId)) : length);
    cmBufferWriterInit(&primary, nsSkipHeader(recvDescr->socket, staticData->response), UD_NS_BUFFERSIZE);

    staticData->primary = &primary;

#ifdef UD_CS_MESSAGESIGNINGPOLICY
    cmBufferWriterBranch(&primary, &packet, 0);
    staticData->packet = &packet;
    dataLength = length + (NQ_COUNT)sizeof(cmSmb2ProtocolId);
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

    pBuf += SMB2_HEADERANDSTRUCTSIZE - sizeof(cmSmb2ProtocolId);
    length -= (NQ_COUNT)(SMB2_HEADERANDSTRUCTSIZE - sizeof(cmSmb2ProtocolId));

    resetQuickFid();    /* reset quick fid for next packet */

    do
    {
        NQ_UINT16 size;
        NQ_INT msgLen;
        NQ_UINT creditsGranted;
        const Entry *e;
        NQ_BOOL nosess = TRUE;

        /* buffer overrun check: header + 2 bytes for payload size */
        if (cmBufferReaderGetRemaining(&reader) < SMB2_HEADERANDSTRUCTSIZE - sizeof(cmSmb2ProtocolId)) /* minus 4 bytes already read before */
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "insufficient request length %d", cmBufferReaderGetRemaining(&reader));
#ifdef UD_NQ_INCLUDESMBCAPTURE
            if ((!staticData->isEncrypedPacket) && (isFirstInChain))
            {
                numOfZerosToFill = length;
                isPaddingNeededForCapture = TRUE;
            }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
            goto ExitEndRecv;
        }

        syMemset(&in, 0, sizeof(CMSmb2Header));

        /* read request packet header */
        cmSmb2HeaderRead(&in, &reader);
        staticData->header = in;
        e = &_entries[in.command];

        /* read payload size */
        cmBufferReadUint16(&reader, &size);

        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Request: command=%u, mid=%u/%u, sid.low=0x%x, signed:%d, async:%d, pid(async.high)=0x%x, tid(async.low)=0x%x, chained: %d, connection: %p",
                in.command, in.mid.high, in.mid.low, in.sid.low, (in.flags & SMB2_FLAG_SIGNED) > 0, (in.flags & SMB2_FLAG_ASYNC_COMMAND) > 0,
                in.flags & SMB2_FLAG_ASYNC_COMMAND ? in.aid.high : in.pid, in.flags & SMB2_FLAG_ASYNC_COMMAND ? in.aid.low : in.tid, in.next > 0, connection);

        /* command range check */
        if (in.command >= CM_ARRAY_SIZE(_entries))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "invalid command code %u", in.command);
#ifdef UD_NQ_INCLUDESMBCAPTURE
            if ((!staticData->isEncrypedPacket) && (isFirstInChain))
            {
                numOfZerosToFill = length;
                isPaddingNeededForCapture = TRUE;
            }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
            goto ExitEndRecv;
        }

#ifdef UD_CS_INCLUDEDIRECTTRANSFER
        /* try DT IN */
        csDispatchSetDtOut(FALSE);
        csDispatchSetDtIn(FALSE);
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
        if (isFirstInChain)
        {
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
            if ((_entries[in.command].flags & FLAG_DTIN)
                    && in.next == 0
#ifdef UD_CS_MESSAGESIGNINGPOLICY
                    && ((in.flags & SMB2_FLAG_SIGNED) == 0)
#endif /* UD_CS_MESSAGESIGNINGPOLICY */
                    && !staticData->isEncrypedPacket
            )
            {
                /* use DirectTransfer - read according to word count */
                csDispatchSetDtIn(TRUE);
                msgLen = (NQ_INT)nsRecvIntoBuffer(recvDescr, pBuf, (NQ_COUNT)(size - 3)); /* read command structure */
                csDispatchDtSaveParameters(pBuf + size - 3, recvDescr);
            }
            else
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
            {
                msgLen = (!staticData->isEncrypedPacket) ? nsRecvIntoBuffer(recvDescr, pBuf, length) : (NQ_INT)length; /* read the rest of the packet */
            }
            if (msgLen == NQ_FAIL)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "error receiving command");
#ifdef UD_NQ_INCLUDESMBCAPTURE
                if (!staticData->isEncrypedPacket)
                {
                    numOfZerosToFill = length;
                    isPaddingNeededForCapture = TRUE;
                }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
                goto ExitEndRecv;
            }

#ifdef UD_NQ_INCLUDESMBCAPTURE
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
            if (e->flags & FLAG_DTIN && !staticData->isEncrypedPacket)
            {
                NQ_BYTE *   tempBuf;
                NQ_UINT32   dataLen;
                NQ_UINT16   offset;
                CMBufferReader  readerDT;

                cmBufferReaderInit(&readerDT, pBuf, (NQ_COUNT)(size - 3));
                cmBufferReadUint16(&readerDT, &offset);
                cmBufferReadUint32(&readerDT, &dataLen);

                if (offset > (size - 1 + SMB2_HEADERSIZE))
                {
                    dataLen += (NQ_UINT32)(offset - (size - 1 + SMB2_HEADERSIZE));
                }

                tempBuf = (NQ_BYTE *)cmMemoryAllocate(dataLen);
                syMemset(tempBuf, 0, dataLen);

                cmCapturePacketWritePacket(pBuf, (NQ_UINT)size - 3);
                cmCapturePacketWritePacket(tempBuf, (NQ_UINT)dataLen);
                cmCapturePacketWriteEnd();

                cmMemoryFree(tempBuf);
            }
            else
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
            {
                if (!staticData->isEncrypedPacket)
                {
                    cmCapturePacketWritePacket(pBuf, (NQ_UINT)msgLen);
                    cmCapturePacketWriteEnd();
                }
            }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
        }

#ifdef UD_CS_INCLUDEDIRECTTRANSFER
        /* try DT OUT */
        if (isFirstInChain &&
                (_entries[in.command].flags & FLAG_DTOUT) &&
                in.next == 0
#ifdef UD_CS_MESSAGESIGNINGPOLICY
                && ((in.flags & SMB2_FLAG_SIGNED) == 0)
#endif /* UD_CS_MESSAGESIGNINGPOLICY */
                && !staticData->isEncrypedPacket
        )
        {
            /* use DirectTransfer - prepare socket */
            syDtStartPacket(((SocketSlot*)recvDescr->socket)->socket);
            csDispatchSetDtOut(TRUE);
        }
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

        /* setup data writer */
        cmBufferWriterBranch(&primary, &data, SMB2_HEADERSIZE);

        /* setup response header */
        out = in;

        /* calculate number of credits */
#ifdef UD_CS_SMB2_INCLUDEMULTICREDIT
        if (in.command != SMB2_CMD_NEGOTIATE)
        {
            creditCharge = (NQ_UINT16)(((in.creditCharge > 0) || (in.command == SMB2_CMD_CANCEL)) ? in.creditCharge : 1);
        }
        connection->credits += (NQ_UINT)creditCharge;

        if ((UD_CS_SMB2_NUMCREDITS < (connection->credits - 1)) && (in.command != SMB2_CMD_CANCEL) && (in.command != SMB2_CMD_NEGOTIATE))
        {
            if (!(connection->capabilities & SMB2_CAPABILITY_LARGE_MTU))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "client asks to charge more than 1 credit, but SMB2_CAPABILITY_LARGE_MTU not supported");
                cmSmb2HeaderSetForResponse(&out, &primary, 1);
                out.status = commandResult = SMB_STATUS_INVALID_PARAMETER;
                cmBufferWriterReset(&data);
                writeErrorResponseData(&data);
                cmSmb2HeaderWrite(&out, &primary);
                cmBufferWriterSync(&primary, &data);
                dataLength -= (NQ_COUNT)in.next;
                break;
            }
            else
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "client asks to charge more credit than available");
                result = FALSE;
                goto ExitEndRecv;
            }
        }

        if ((in.command == SMB2_CMD_NEGOTIATE) || ((UD_CS_SMB2_NUMCREDITS == (connection->credits - 1)) && (0 == in.credits)))
        {
            creditsGranted = 1; /* give one credit even if not asked */
        }
        else
        {
            connection->creditsMaxRequested = connection->credits < in.credits ? in.credits : connection->credits;
            creditsGranted = connection->credits > in.credits ? in.credits : connection->credits;
            if ((0 == creditsGranted) && (in.command != SMB2_CMD_CANCEL) && (connection->creditsMaxRequested > 0))
            {
                creditsGranted = 1;
                connection->creditsMaxRequested--;
            }
            connection->creditsToGrant = creditsGranted;
        }
        connection->credits -= creditsGranted;
        if (in.creditCharge != 0)
        {
            connection->credits -= (NQ_UINT)(creditCharge - in.creditCharge);
        }
        creditsForResponse += creditsGranted;
         cmSmb2HeaderSetForResponse(&out, &primary, (NQ_UINT16)(0 != in.next ? 0 : creditsForResponse));
#else
        if (in.command == SMB2_CMD_NEGOTIATE)
        {
            creditsGranted = 1;
        }
        else
        {
            creditsGranted = connection->credits > in.credits? in.credits : connection->credits;
            connection->creditsToGrant = creditsGranted;
        }

        cmSmb2HeaderSetForResponse(&out, &primary, (NQ_UINT16)creditsGranted);

        connection->credits -=  creditsGranted - 1;

#endif /* UD_CS_SMB2_INCLUDEMULTICREDIT */

        if (size != e->size)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "command structure size is %u, expected %u", size, e->size);
            goto ExitPutBuffer;
        }

        nosess = (e->flags & FLAG_NOSESSION) != 0;
        session = nosess ? NULL : csGetUserByUid((CSUid)sessionIdToUid(in.sid.low));

        if ((in.flags & SMB2_FLAG_SIGNED) && (SMB2_CMD_NEGOTIATE == in.command))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "unexpected signed negotiate request");
            out.status = commandResult = SMB_STATUS_INVALID_PARAMETER;
            cmSmb2HeaderSetForResponse(&out, &primary, 1);
            cmBufferWriterReset(&data);
            writeErrorResponseData(&data);
            cmSmb2HeaderWrite(&out, &primary);
            cmBufferWriterSync(&primary, &data);
            break;
        }

        if ((in.flags & SMB2_FLAG_SIGNED) && (NULL == session))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "signed request for not existing user session");
            out.status = commandResult = SMB_STATUS_USER_SESSION_DELETED;
            cmSmb2HeaderSetForResponse(&out, &primary, 1);
            cmBufferWriterReset(&data);
            writeErrorResponseData(&data);
            cmSmb2HeaderWrite(&out, &primary);
            cmBufferWriterSync(&primary, &data);
            break;
         }

        if (NULL != session)
        {
            /* renew session time stamp */
            csRenewUserTimeStamp(session);

            /* unencrypted request is not allowed for 3.xx dialects if session required encryption */
#ifdef UD_NQ_INCLUDESMB3
#if defined(UD_NQ_INCLUDESMB311) || !defined(UD_CS_ALLOW_NONENCRYPTED_ACCESS_TO_ENCRYPTED_SHARE)
            if ((connection->dialect >= CS_DIALECT_SMB30) && (session->isEncrypted) && (!staticData->isEncrypedPacket))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "This session requires encrypted access and received packet is not encrypted");
                out.status = commandResult = SMB_STATUS_ACCESS_DENIED;
                cmBufferWriterReset(&data);
                writeErrorResponseData(&data);
                cmSmb2HeaderWrite(&out, &primary);
                cmBufferWriterSync(&primary, &data);
                dataLength -= (NQ_COUNT)in.next;
                break;
            }
#endif /*  defined(UD_NQ_INCLUDESMB311) || defined(UD_CS_ALLOW_NONENCRYPTED_ACCESS_TO_ENCRYPTED_SHARE) */
#endif /* UD_NQ_INCLUDESMB3 */
        }

        if (nosess || session != NULL)
        {
            NQ_BOOL notree = (e->flags & FLAG_NOTREE) != 0;
            /*in the spec , just cancel request must be in async format , others may use async header but YNQ does not support it.*/
            NQ_BOOL async = ( (in.flags & SMB2_FLAG_ASYNC_COMMAND) != 0 ) & ( (e->flags & FLAG_ASYNC) != 0 );
            CSTree *tree = (notree || async) ? NULL : csGetTreeByTid((CSTid)in.tid);

            if (notree || async || tree != NULL)
            {
#ifdef UD_CS_MESSAGESIGNINGPOLICY
#ifdef UD_NQ_INCLUDESMB311
                if (!staticData->isEncrypedPacket && (CS_DIALECT_SMB311 == connection->dialect) && (SMB2_CMD_TREECONNECT == in.command)
                        && (NULL != session) && (!session->isAnonymous) && (!session->isGuest) && (0 == (in.flags & SMB2_FLAG_SIGNED)))
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Tree connect request over dialect 3.11 has to be signed");
                    result = FALSE;
                    goto ExitEndRecv;
                }
#endif /* UD_NQ_INCLUDESMB311 */
#ifdef UD_NQ_INCLUDESMB3
                if (connection->dialect >= CS_DIALECT_SMB30)
                {
                    if (!staticData->isEncrypedPacket &&
                        !csCheckMessageSignatureSMB3(session, (NQ_BYTE*)in._start , (NQ_COUNT)(in.next == 0 ? dataLength : in.next) , in.flags))
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "incoming signature doesn't match");
                        out.status = commandResult = SMB_STATUS_ACCESS_DENIED;
                        cmBufferWriterReset(&data);
                        writeErrorResponseData(&data);
                        cmSmb2HeaderWrite(&out, &primary);
                        cmBufferWriterSync(&primary, &data);
                        dataLength -= (NQ_COUNT)in.next;
                        break;
                    }
                }
                /* check incoming message signature */
                else
#endif /* UD_NQ_INCLUDESMB3 */
                {
                    if (!csCheckMessageSignatureSMB2(session, (NQ_BYTE*)in._start, (NQ_COUNT)(in.next == 0 ? dataLength : in.next), in.flags))
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "incoming signature doesn't match");
                        out.status = commandResult = SMB_STATUS_ACCESS_DENIED;
                        cmBufferWriterReset(&data);
                        writeErrorResponseData(&data);
                        cmSmb2HeaderWrite(&out, &primary);
                        cmBufferWriterSync(&primary, &data);
                        dataLength -= (NQ_COUNT)in.next;
                        break;
                    }
                }

                dataLength -= (NQ_COUNT)in.next;
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

#ifdef UD_NQ_INCLUDESMB3
                if (NULL != tree)
                {
#ifdef UD_NQ_INCLUDESMB311
                    if((tree->share->isEncrypted) && (connection && (0 != (connection->capabilities & SMB2_CAPABILITY_ENCRYPTION))) && (!staticData->isEncrypedPacket))
                    {
                         LOGERR(CM_TRC_LEVEL_ERROR, "This share requires encrypted access");
                         out.status = commandResult = SMB_STATUS_ACCESS_DENIED;
                         cmBufferWriterReset(&data);
                         writeErrorResponseData(&data);
                         cmSmb2HeaderWrite(&out, &primary);
                         cmBufferWriterSync(&primary, &data);
                         break;
                    }
#endif /* UD_NQ_INCLUDESMB311 */
#ifndef UD_CS_ALLOW_NONENCRYPTED_ACCESS_TO_ENCRYPTED_SHARE
                   if(((csIsServerEncrypted() || (tree->share->isEncrypted)) && (connection && (0 != (connection->capabilities & SMB2_CAPABILITY_ENCRYPTION))) && (!staticData->isEncrypedPacket))
                      || ((csIsServerEncrypted() || (tree->share->isEncrypted) || (staticData->isEncrypedPacket)) && (connection && (0 == (connection->capabilities & SMB2_CAPABILITY_ENCRYPTION)))))
                {
                        LOGERR(CM_TRC_LEVEL_ERROR, "This share requires encrypted access");
                    out.status = commandResult = SMB_STATUS_ACCESS_DENIED;
                    cmBufferWriterReset(&data);
                    writeErrorResponseData(&data);
                    cmSmb2HeaderWrite(&out, &primary);
                    cmBufferWriterSync(&primary, &data);
                    break;
                }
#endif /* UD_CS_ALLOW_NONENCRYPTED_ACCESS_TO_ENCRYPTED_SHARE */
                }
#endif /* UD_NQ_INCLUDESMB3 */

                /* process request and set the status */
                commandResult = e->handler ? e->handler(&in, &out, &reader, connection, session, tree, &data) : SMB_STATUS_NOT_IMPLEMENTED;
                switch (commandResult)
                {
                    case SMB_STATUS_DISCONNECT:
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "command handler returned status SMB_STATUS_DISCONNECT");
                        goto ExitPutBuffer;
                    }
                    case SMB_STATUS_NORESPONSE:
                    {
                        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "command handler returned status SMB_STATUS_NORESPONSE");
                        result = TRUE;
                        goto ExitEndRecv;
                    }
                }
            }
            else
            {
                commandResult = SMB_STATUS_NETWORK_NAME_DELETED;
                LOGERR(CM_TRC_LEVEL_ERROR, "tree not found");
            }
        }
        else
        {
            commandResult = SMB_STATUS_USER_SESSION_DELETED;
            LOGERR(CM_TRC_LEVEL_ERROR, "session not found");
        }

        /* write the response packet (header, signing) */
        cs2WriteResponse(commandResult, &in, &out, &data, connection);

        isFirstInChain = FALSE;
    }
    while (cmSmb2HeaderShiftNext(&in, &reader));

#ifdef UD_CS_INCLUDEDIRECTTRANSFER
    if (csDispatchIsDtIn())
    {
        if (!csDispatchDtFromSocket(recvDescr, csDispatchDtGetCount()))
        {
            result = TRUE;
            goto ExitEndRecv;
        }
    }
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

    nsEndRecvIntoBuffer(recvDescr);

    /* send the response */
    result = cs2SendResponse(&out, connection);
    goto Exit;

ExitEndRecv:
    nsEndRecvIntoBuffer(recvDescr);
ExitPutBuffer:
    if (NULL != staticData->response)
    {
        nsPutBuffer(allocatedResponse);
        staticData->response = NULL;
    }

Exit:
#ifdef UD_NQ_INCLUDESMBCAPTURE
    if (TRUE == isPaddingNeededForCapture)
    {
        captureBuf = (NQ_BYTE *)cmMemoryAllocate(numOfZerosToFill);
        if (NULL != captureBuf)
        {
            syMemset(captureBuf, 0, numOfZerosToFill);
            cmCapturePacketWritePacket(captureBuf, numOfZerosToFill);
            cmMemoryFree(captureBuf);
        }

        cmCapturePacketWriteEnd();
    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/*
 *====================================================================
 * PURPOSE: save information for a delayed response
 *--------------------------------------------------------------------
 * PARAMS:  OUT pointer to the buffer for response context
 *          IN pointer to the interim response header
 *          IN expected length of the response command
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
cs2DispatchSaveResponseContext(
    CSLateResponseContext * contextBuffer,
    const CMSmb2Header * header
    )
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);
    contextBuffer->prot.smb2.flags = header->flags;
    contextBuffer->prot.smb2.tid = header->tid;
    contextBuffer->prot.smb2.sid = header->sid;
    contextBuffer->prot.smb2.mid = header->mid;
    contextBuffer->prot.smb2.pid = header->pid;
    contextBuffer->prot.smb2.command = (NQ_BYTE)header->command;
    contextBuffer->prot.smb2.aid = header->aid;
    contextBuffer->socket = staticData->savedSocket;
#ifdef UD_NQ_INCLUDESMB3
    contextBuffer->doEncrypt = staticData->isEncrypedPacket;
#endif
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 *====================================================================
 * PURPOSE: get new context for this file - for late responses on file requests.
 *--------------------------------------------------------------------
 * PARAMS:  pFile - pointer to a file descriptor
 *
 * RETURNS: pointer to context
 *
 * NOTES:
 *====================================================================
 */
/*
CSLateResponseContext
cs2DispatchGetFreeResponseContext(
    CSFile * pFile)
{
    NQ_COUNT i;
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    for (i = 0, i < CM_RPC_MAXNUMOF_PENDINGNOTIFYCTXS; ++i)
    {
        if (pFile->notifyContext[i].status == 0)
            return &(pFile->notifyContext[i]);
    }
    return NULL;

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}
*/
/*
 *====================================================================
 * PURPOSE: compose header and calculate command data pointer and size
 *--------------------------------------------------------------------
 * PARAMS:  IN saved context
 *          IN status to return
 *
 * RETURNS: NQ_SUCCESS or error code
 *
 * NOTES:   prepares CIFS header
 *====================================================================
 */

NQ_STATUS
cs2DispatchPrepareLateResponse(
    CSLateResponseContext* context,
    NQ_UINT32 status
    )
{
    CMBufferWriter writer;  /* header writer */
    CMSmb2Header out;       /* header data */
#ifdef UD_CS_MESSAGESIGNINGPOLICY
    CSUser *pUser;
    CSSession *pSession;
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    cmBufferWriterInit(&writer, nsSkipHeader(context->socket, staticData->responseBuffer), UD_NS_BUFFERSIZE);
    cmSmb2HeaderInitForResponse(&out, &writer, 1);
    out.command = context->prot.smb2.command;
    out.mid = context->prot.smb2.mid;
    out.pid = context->prot.smb2.pid;
    out.sid = context->prot.smb2.sid;
    out.tid = context->prot.smb2.tid;
    out.aid = context->prot.smb2.aid;
    out.flags = SMB2_FLAG_SERVER_TO_REDIR | SMB2_FLAG_ASYNC_COMMAND;
#ifdef UD_CS_MESSAGESIGNINGPOLICY
    if (((pUser = csGetUserByUid((CSUid)sessionIdToUid(out.sid.low))) && (pSession = csGetSessionById(pUser->session)) && pSession->signingOn) ||
       (context->prot.smb2.flags & SMB2_FLAG_SIGNED))
        out.flags |= SMB2_FLAG_SIGNED;
#endif /* UD_CS_MESSAGESIGNINGPOLICY */
    out.status = status;
    cmSmb2HeaderWrite(&out, &writer);
    context->commandData = writer.current;
    context->commandDataSize = writer.length - (NQ_COUNT)(writer.current - writer.origin);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return NQ_SUCCESS;
}

/*
 *====================================================================
 * PURPOSE: send a response using saved context
 *--------------------------------------------------------------------
 * PARAMS:  IN saved context
 *          IN command data length
 *
 * RETURNS: TRUE for success
 *====================================================================
 */

NQ_BOOL
cs2DispatchSendLateResponse(
    CSLateResponseContext* context,
    NQ_COUNT dataLength
    )
{
    CSSession * pSession;
    NQ_COUNT packetLen;  /* packet length, no NB header */
#ifdef UD_NQ_INCLUDESMBCAPTURE
    CSSocketDescriptor *    sockDescr;
#endif /* UD_NQ_INCLUDESMBCAPTURE */
#ifdef UD_NQ_INCLUDESMB3
    NQ_BYTE encryptBuf[CM_NB_DATAGRAMBUFFERSIZE + SMB2_TRANSFORMHEADER_SIZE];

    NQ_BOOL isEncrypted = context->doEncrypt;
#endif

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "context:%p dataLength:%d", context, dataLength);

    pSession = csGetSessionBySpecificSocket(context->socket);
    packetLen = (NQ_COUNT)(context->commandData + dataLength - nsSkipHeader(context->socket, staticData->responseBuffer));
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "packetLen:%d", packetLen);

#ifdef UD_CS_MESSAGESIGNINGPOLICY
    {
        CMBufferReader  reader;
        CMBufferWriter  writer;
        CMSmb2Header hdr;

        cmBufferReaderInit(&reader , nsSkipHeader(context->socket, staticData->responseBuffer) , 64);
        cmSmb2HeaderRead(&hdr, &reader);
        cmBufferWriterInit(&writer , nsSkipHeader(context->socket, staticData->responseBuffer) , 64);
        cmSmb2HeaderWrite(&hdr , &writer);

        if (pSession->dialect < CS_DIALECT_SMB30)
        {
            csCreateMessageSignatureSMB2(context->prot.smb2.sid.low, nsSkipHeader(context->socket, staticData->responseBuffer), packetLen);
        }
#ifdef UD_NQ_INCLUDESMB3
        else if (pSession->dialect >= CS_DIALECT_SMB30)
        {
            csCreateMessageSignatureSMB3(context->prot.smb2.sid.low, nsSkipHeader(context->socket, staticData->responseBuffer), packetLen);
        }
#endif /* UD_NQ_INCLUDESMB3 */
    }
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

#ifdef UD_NQ_INCLUDESMBCAPTURE
    sockDescr = csGetClientSocketDescriptorBySocket(context->socket);
    if (sockDescr != NULL)
    {
        sockDescr->captureHdr.receiving = FALSE;
        cmCapturePacketWriteStart(&sockDescr->captureHdr , packetLen);
        cmCapturePacketWritePacket(staticData->responseBuffer + 4, packetLen);
        cmCapturePacketWriteEnd();
    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
#ifdef UD_NQ_INCLUDESMB3
    if (isEncrypted)
    {
        NQ_INT  nbHeader;

        nbHeader = nsSkipHeader(staticData->savedSocket, staticData->responseBuffer) == (NQ_BYTE *)staticData->responseBuffer ? 0 : 4;
        syMemcpy(&encryptBuf[SMB2_TRANSFORMHEADER_SIZE + nbHeader], &staticData->responseBuffer[nbHeader] , packetLen);
        cs2TransformHeaderEncrypt( NULL , &encryptBuf[nbHeader] , packetLen);
        packetLen = packetLen + SMB2_TRANSFORMHEADER_SIZE;
    }
#endif /* UD_NQ_INCLUDESMB3 */
    packetLen = nsPrepareNBBuffer(
#ifdef UD_NQ_INCLUDESMB3
            isEncrypted ? encryptBuf :
#endif
                    staticData->responseBuffer, packetLen, packetLen);
    if (0 == packetLen)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error prepare buffer for late response");
        LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
        return FALSE;
    }

    if (packetLen != (NQ_COUNT)nsSendFromBuffer(
                context->socket,
#ifdef UD_NQ_INCLUDESMB3
                isEncrypted ? encryptBuf :
#endif
                        staticData->responseBuffer,
                packetLen,
                packetLen,
                NULL
                )
          )
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error sending late response");
        LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
        return FALSE;
    }

    LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
    return TRUE;
}

/*====================================================================
 * PURPOSE: Send an interim response
 *--------------------------------------------------------------------
 * PARAMS:  IN in - pointer to the incoming request header
 *
 * RETURNS: Generated Async ID
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT32 csSmb2SendInterimResponse(
    CMSmb2Header * in
    )
{
    CMSmb2Header out;                               /* outgoing header */
    CMBufferWriter writer;                          /* response writer */
    NQ_BYTE outBuffer[SMB2_HEADERSIZE + 10 + 8];    /* response buffer */
    NQ_COUNT expected;                              /* expected packet length to send */
    NQ_INT sent = 0;                                /* actually sent bytes */
    CSSession *pSession = NULL;
    CSUser *pUser = NULL;
#ifdef UD_NQ_INCLUDESMBCAPTURE
    CSSocketDescriptor *    sockDescr;
#endif /* UD_NQ_INCLUDESMBCAPTURE */
#ifdef UD_NQ_INCLUDESMB3
    NQ_BYTE encryptedBuffer[SMB2_TRANSFORMHEADER_SIZE + SMB2_HEADERSIZE + 10 + 8];    /* encrypted response buffer */
    NQ_BOOL isEncrypted = staticData->isEncrypedPacket;
#endif /* UD_NQ_INCLUDESMB3 */

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL);

    pUser = csGetUserByUid((CSUid)sessionIdToUid(in->sid.low));
    if (NULL == pUser)
    {
         LOGERR(CM_TRC_LEVEL_ERROR, "Sending interim response failed, invalid session");
         LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
         return 0;
    }
    pSession = csGetSessionById(pUser->session);
    if (NULL == pSession)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Sending interim response failed, invalid session");
        LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
        return 0;
    }

    out = *in;      /* copy fields */
    cmBufferWriterInit(&writer, nsSkipHeader(staticData->savedSocket, outBuffer), sizeof(outBuffer));
    cmSmb2HeaderSetForResponse(&out, &writer, (NQ_UINT16)pSession->creditsToGrant);
    out.flags |= SMB2_FLAG_ASYNC_COMMAND;
#ifdef UD_CS_MESSAGESIGNINGPOLICY
    if (pSession->signingOn)
        out.flags |= SMB2_FLAG_SIGNED;
#endif /* UD_CS_MESSAGESIGNINGPOLICY */
    out.status = SMB_STATUS_PENDING;
    cs2GenerateNextAsyncId(&out.aid);
    cmSmb2HeaderWrite(&out, &writer);
    writeErrorResponseData(&writer);
    expected = cmBufferWriterGetDataCount(&writer);
#ifdef UD_CS_MESSAGESIGNINGPOLICY
    if (pSession->dialect < CS_DIALECT_SMB30)
    {
        csCreateMessageSignatureSMB2(out.sid.low, nsSkipHeader(staticData->savedSocket, outBuffer), expected);
    }
#ifdef UD_NQ_INCLUDESMB3
    else if (pSession->dialect >= CS_DIALECT_SMB30 && !isEncrypted)
    {
        csCreateMessageSignatureSMB3(out.sid.low, nsSkipHeader(staticData->savedSocket, outBuffer), expected);
    }
#endif /* UD_NQ_INCLUDESMB3 */
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

#ifdef UD_NQ_INCLUDESMBCAPTURE
    sockDescr = csGetClientSocketDescriptorBySocket(staticData->savedSocket);
    if (sockDescr != NULL)
    {
        sockDescr->captureHdr.receiving = FALSE;
        cmCapturePacketWriteStart(&sockDescr->captureHdr, expected);
        cmCapturePacketWritePacket(outBuffer + 4, expected);
        cmCapturePacketWriteEnd();
    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
#ifdef UD_NQ_INCLUDESMB3
    if (isEncrypted)
    {
        NQ_INT  nbHeader;
        NQ_INT  skip = SMB2_TRANSFORMHEADER_SIZE;

        syMemset(&encryptedBuffer , 0 , sizeof(encryptedBuffer));
        nbHeader = (nsSkipHeader(staticData->savedSocket, outBuffer) == (NQ_BYTE *)outBuffer) ? 0 : 4;
        skip += nbHeader;

        syMemcpy((NQ_BYTE *)&encryptedBuffer[skip], (NQ_BYTE *)&outBuffer[nbHeader] , expected);
        cs2TransformHeaderEncrypt(NULL, (NQ_BYTE *)&encryptedBuffer[nbHeader], expected);
        expected = expected + SMB2_TRANSFORMHEADER_SIZE;
    }
#endif /* UD_NQ_INCLUDESMB3 */
    expected = nsPrepareNBBuffer(
#ifdef UD_NQ_INCLUDESMB3
            isEncrypted ? encryptedBuffer :
#endif /* UD_NQ_INCLUDESMB3 */
            outBuffer, expected, expected);
    if (0 == expected)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "prepare buffer for interim response failed");
        LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
        return 0;
    }

    sent = nsSendFromBuffer(
                            staticData->savedSocket,
#ifdef UD_NQ_INCLUDESMB3
                            isEncrypted ? encryptedBuffer :
#endif /* UD_NQ_INCLUDESMB3 */
                            outBuffer,
                            expected,
                            expected,
                            NULL
                            );

    if (sent != expected)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Sending interim response failed: size=%d, sent=%d", expected, sent);
        LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
        return 0;
    }

    LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
    return out.aid.low;
}

/*
 *====================================================================
 * PURPOSE: Responding with error
 *--------------------------------------------------------------------
 * PARAMS:  IN socket that has an incoming packet
 *          IN SMB error to send
 *          IN buffer filled with SMB packet from the socket
 *          IN expected number of bytes in the packet
 *
 * RETURNS: NQ_FAIL or NQ_SUCCESS
 *
 * NOTES:   The buffer passed from the outside will be released by caller
 *====================================================================
 */

NQ_STATUS
cs2DispSendError(
    NSSocketHandle socket,
    NQ_UINT32 smbError,
    NQ_BYTE* receiveBuf,
    NQ_INT expected
    )
{
    NQ_INT msgLen;                          /* message length */
    NQ_INT sndLen;                          /* number of bytes that were sent */
    NQ_BYTE* sendBuf;                       /* pointer to the send buffer */
    CMBufferReader  reader;
    CMBufferWriter  writer, data;
    CMSmb2Header    smb2Header;
    NQ_STATUS result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "socket:%p SMB error:0x%x DOS received buffer:%p expected bytes:%d", socket, smbError, receiveBuf, expected);

    if (expected < SMB2_HEADERSIZE)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid packet length:%d", expected);
        goto Exit;
    }

    sendBuf = nsGetBuffer();

    cmBufferReaderInit(&reader, receiveBuf, (NQ_COUNT)expected);
    cmBufferWriterInit(&writer, nsSkipHeader(socket, sendBuf), UD_NS_BUFFERSIZE);
    /* read request packet header */
    cmSmb2HeaderRead(&smb2Header, &reader);
    cmSmb2HeaderSetForResponse(&smb2Header, &writer, 1);
    smb2Header.status = smbError;
    /* setup data writer */
    cmBufferWriterBranch(&writer, &data, SMB2_HEADERSIZE);
    cmBufferWriterReset(&data);
    writeErrorResponseData(&data);
    cmSmb2HeaderWrite(&smb2Header, &writer);
    cmBufferWriterSync(&writer, &data);

    msgLen = (NQ_INT)cmBufferWriterGetDataCount(&writer);

    msgLen = (NQ_INT)nsPrepareNBBuffer(sendBuf, (NQ_UINT)msgLen, (NQ_UINT)msgLen);
    if (0 == msgLen)
    {
        nsPutBuffer(sendBuf);
        LOGERR(CM_TRC_LEVEL_ERROR, "Error prepare buffer for response");
        goto Exit;
    }

    /* send the response - this will also release the buffer */
    sndLen = nsSendFromBuffer(socket, sendBuf, (NQ_UINT)msgLen, (NQ_UINT)msgLen, &releaseCallback);
    if (sndLen != msgLen)
    {
        LOGERR(CM_TRC_LEVEL_ERROR,"Error sending response, required: %d, sent: %d", msgLen, sndLen);
        goto Exit;
    }

    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result:%d", result);
    return result;
}

#ifdef UD_NQ_INCLUDESMB3
NQ_BOOL isCurrentPacketEncrypted(void)
{
    return staticData->isEncrypedPacket;
}
#endif /* UD_NQ_INCLUDESMB3 */

#endif /* defined(UD_NQ_INCLUDECIFSSERVER) && defined(UD_NQ_INCLUDESMB2) */
