/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : NETLOGON RPC client
 *--------------------------------------------------------------------
 * MODULE        : rpc - rpccore
 * DEPENDENCIES  : None
 ********************************************************************/

#include "ccnetlgn.h"
#include "ccparams.h"
#include "cmbuf.h"
#include "ccerrors.h"
#include "cmcrypt.h"
#include "ccrpc.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT


#define NETRSERVERREQUESTCHALLENGE_NETLOGON_OPNUM    4
#define NETRSERVERAUTHENTICATE2_NETLOGON_OPNUM      15
#define NETRSERVERAUTHENTICATE3_NETLOGON_OPNUM      26
#define DSRENUMERATEDOMAINTRUSTS_NETLOGON_OPNUM     40

#define NETLOGON_TRUST_FLAGS                0x0000003f

typedef struct
{
    const NQ_WCHAR *server;
    const NQ_WCHAR *computer;
    CCNetlogonCredential *credential;
    NQ_UINT32 status;
}
ParamsNetrServerReqChallenge;

typedef struct
{
    const NQ_WCHAR *server;
    const NQ_WCHAR *computer;
    CCNetlogonCredential *credential;
    NQ_UINT32 flags;
    NQ_UINT32 status;
}
ParamsNetrServerAuthenticate2and3;

typedef struct
{
    const NQ_WCHAR *server;                 /* server name */
    CCNetrEnumerateNamesCallback callback;  /* add name callback */
    void *list;                             /* list to add name to */
    NQ_UINT32 status;                       /* operation status */
}
ParamsDsrEnumerateDomainTrusts;

/* NETLOGON pipe descriptor */
static const NQ_WCHAR pipeName[] = { cmWChar('n'), cmWChar('e'), cmWChar('t'), cmWChar('l'), cmWChar('o'), cmWChar('g'), cmWChar('o'), cmWChar('n'), cmWChar('\0') };
static const CCDcerpcPipeDescriptor _nlpd = {
    pipeName,
    {cmPack32(0x12345678),cmPack16(0x1234),cmPack16(0xabcd),{0xef,0x00},{0x01,0x23,0x45,0x67,0xcf,0xfb}},
    cmRpcVersion(1, 0),
    0
};

const CCDcerpcPipeDescriptor *ccNetlogonGetPipe(void)
{
    return &_nlpd;
}

static NQ_COUNT composeNetrServerReqChallenge(NQ_IOBufPos buffer, NQ_COUNT size, void *params, NQ_BOOL *moreData)
{
    CMBufferWriter w;               /* for composing request */
    ParamsNetrServerReqChallenge *p = (ParamsNetrServerReqChallenge *)params;
    NQ_UINT32 ref = 0;              /* ref id */
    NQ_UINT32 sz;                   /* string size */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buff:%p size:%d params:%p more:%p", IOBUF_GETBUFPTR(buffer), size, params, moreData);

    cmBufferWriterInit(&w, buffer, size);

    cmBufferWriteUint16(&w, NETRSERVERREQUESTCHALLENGE_NETLOGON_OPNUM);

    /* logon server name prefixed by double back slash */
    sz = 3 + (NQ_UINT32)cmWStrlen(p->server);

    cmBufferWriteUint32(&w, ++ref);                /* ref ID */
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteAsciiAsUnicodeN(&w, "\\\\", 2, CM_BSF_NOFLAGS);
    cmBufferWriteUnicode(&w, p->server);
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTE(buffer, 2), 4);        /* 4 byte alignment */
    IOBUF_SKIPBACKBYTE(buffer, 2);

    /* own computer name (no trailing '$') */
    sz = 1 + (NQ_UINT32)cmWStrlen(p->computer);

    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteUnicode(&w, p->computer);

    /* client credential */
    cmBufferWriteBytes(&w, p->credential->client, sizeof(p->credential->client));

    *moreData = FALSE;

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return cmBufferWriterGetDataCount(&w);
}

static NQ_STATUS processNetrServerReqChallenge(NQ_IOBufPos data, NQ_COUNT size, void *params, NQ_BOOL moreData)
{
    CMBufferReader r;
    ParamsNetrServerReqChallenge *p = (ParamsNetrServerReqChallenge *)params;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "data:%p size:%d params:%p more:%s", IOBUF_GETBUFPTR(data), size, params, moreData ? "TRUE" : "FALSE");

    cmBufferReaderInit(&r, data, size);
    cmBufferReadBytes(&r, p->credential->server, sizeof(p->credential->server));
    cmBufferReadUint32(&r, &p->status);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", p->status);
    return (NQ_STATUS)p->status;
}

NQ_UINT32 ccNetrServerReqChallenge(NQ_HANDLE netlogon, const NQ_WCHAR *server, const NQ_WCHAR *computer, CCNetlogonCredential *credential)
{
    ParamsNetrServerReqChallenge p;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "logon:%p server:%p computer:%p credential:%p", netlogon, server, computer, credential);
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "server:%s", cmWDump(server));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "computer:%s", cmWDump(computer));

    p.server = server;
    p.computer = computer;
    p.credential = credential;
    p.status = 0;

    /* call NETLOGON::NetrServerReqChallenge */
    if (FALSE == ccDcerpcCall(netlogon, composeNetrServerReqChallenge, processNetrServerReqChallenge, &p))
    {
        p.status = (p.status == 0) ? (NQ_UINT32)syGetLastError() : (NQ_UINT32)ccErrorsStatusToNq(p.status, TRUE);
        LOGERR(CM_TRC_LEVEL_ERROR, "NETLOGON::NetrServerReqChallenge");
    }

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%u", p.status);
    return p.status;
}

NQ_UINT32 ccNetrServerReqChallengeEx(NQ_HANDLE netlogon, const NQ_WCHAR *server, const NQ_WCHAR *computer, CCNetlogonCredential *credential)
{
    ParamsNetrServerReqChallenge p;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "logon:%p server:%p computer:%p credential:%p", netlogon, server, computer, credential);
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "server:%s", cmWDump(server));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "computer:%s", cmWDump(computer));

    p.server = server;
    p.computer = computer;
    p.credential = credential;
    p.status = 0;

    /* call NETLOGON::NetrServerReqChallenge */
    if (FALSE == ccDcerpcOverTcpCall(netlogon, composeNetrServerReqChallenge, processNetrServerReqChallenge, &p))
    {
        p.status = (p.status == 0) ? (NQ_UINT32)syGetLastError() : (NQ_UINT32)ccErrorsStatusToNq(p.status, TRUE);
        LOGERR(CM_TRC_LEVEL_ERROR, "NETLOGON::NetrServerReqChallenge");
    }

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%u", p.status);
    return p.status;
}

static NQ_COUNT composeNetrServerAuthenticate2(NQ_IOBufPos buffer, NQ_COUNT size, void *params, NQ_BOOL *moreData)
{
    CMBufferWriter w;       /* for composing request */
    ParamsNetrServerAuthenticate2and3 *p = (ParamsNetrServerAuthenticate2and3 *)params;
    NQ_UINT32 ref = 0;      /* ref if */
    NQ_UINT32 sz;           /* string size */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buff:%p size:%d params:%p more:%p", IOBUF_GETBUFPTR(buffer), size, params, moreData);

    cmBufferWriterInit(&w, buffer, size);

    cmBufferWriteUint16(&w, NETRSERVERAUTHENTICATE2_NETLOGON_OPNUM);

    /* logon server name prefixed by double back slash */
    sz = 3 + (NQ_UINT32)cmWStrlen(p->server);

    cmBufferWriteUint32(&w, ++ref);                /* ref ID */
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteAsciiAsUnicodeN(&w, "\\\\", 2, CM_BSF_NOFLAGS);
    cmBufferWriteUnicode(&w, p->server);
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTE(buffer, 2), 4);        /* 4 byte alignment */
    IOBUF_SKIPBACKBYTE(buffer, 2);

    /* user name = own computer name (with trailing '$')*/
    sz = 2 + (NQ_UINT32)cmWStrlen(p->computer);

    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteUnicodeNoNull(&w, p->computer);
    cmBufferWriteAsciiAsUnicodeN(&w, "$", 1, CM_BSF_WRITENULLTERM);
    cmBufferWriteUint16(&w, 2);                    /* type: workstation */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTE(buffer, 2), 4);        /* 4 byte alignment */
    IOBUF_SKIPBACKBYTE(buffer, 2);

    /* own computer name (no trailing '$') */
    sz = 1 + (NQ_UINT32)cmWStrlen(p->computer);

    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteUnicode(&w, p->computer);

    /* client credential */
    cmBufferWriteBytes(&w, p->credential->client, sizeof(p->credential->client));
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTE(buffer, 2), 4);        /* 4 byte alignment */
    IOBUF_SKIPBACKBYTE(buffer, 2);

    /* negotiation flags */
    cmBufferWriteUint32(&w, p->flags);

    *moreData = FALSE;

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return cmBufferWriterGetDataCount(&w);
}

static NQ_STATUS processNetrServerAuthenticate2(NQ_IOBufPos data, NQ_COUNT size, void *params, NQ_BOOL moreData)
{
    CMBufferReader r;
    ParamsNetrServerAuthenticate2and3 *p = (ParamsNetrServerAuthenticate2and3 *)params;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "data:%p size:%d params:%p more:%s", IOBUF_GETBUFPTR(data), size, params, moreData ? "TRUE" : "FALSE");

    cmBufferReaderInit(&r, data, size);
    cmBufferReadBytes(&r, p->credential->server, sizeof(p->credential->server));
    cmBufferReadUint32(&r, &p->flags);
    cmBufferReadUint32(&r, &p->status);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", p->status);
    return (NQ_STATUS)p->status;
}

NQ_UINT32 ccNetrServerAuthenticate2(NQ_HANDLE netlogon, const NQ_WCHAR *server, const NQ_WCHAR *computer, CCNetlogonCredential *credential, NQ_UINT32 *flags)
{
    ParamsNetrServerAuthenticate2and3 p;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "handle:%p server:%s computer:%s credential:%p flags:%p", netlogon, cmWDump(server), cmWDump(computer), credential, flags);

    p.server = server;
    p.computer = computer;
    p.credential = credential;
    p.flags = (NULL != flags) ? *flags : 0;
    p.status = 0;

    /* call NETLOGON::NetrServerAuthenticate2 */
    if (TRUE == ccDcerpcCall(netlogon, composeNetrServerAuthenticate2, processNetrServerAuthenticate2, &p))
    {
        if (NULL != flags)
        {
            *flags = p.flags;
        }
    }
    else
    {
        p.status = (p.status == 0) ? (NQ_UINT32)syGetLastError() : (NQ_UINT32)ccErrorsStatusToNq(p.status, TRUE);
        LOGERR(CM_TRC_LEVEL_ERROR, "NETLOGON::NetrServerAuthenticate2");
    }

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%u", p.status);
    return p.status;
}

NQ_UINT32 ccNetrServerAuthenticate2Ex(NQ_HANDLE netlogon, const NQ_WCHAR *server, const NQ_WCHAR *computer, CCNetlogonCredential *credential, NQ_UINT32 *flags)
{
    ParamsNetrServerAuthenticate2and3 p;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "handle:%p server:%s computer:%s credential:%p flags:%p", netlogon, cmWDump(server), cmWDump(computer), credential, flags);

    p.server = server;
    p.computer = computer;
    p.credential = credential;
    p.flags = (NULL != flags) ? *flags : 0;
    p.status = 0;

    /* call NETLOGON::NetrServerAuthenticate2 */
    if (TRUE == ccDcerpcOverTcpCall(netlogon, composeNetrServerAuthenticate2, processNetrServerAuthenticate2, &p))
    {
        if (NULL != flags)
        {
            *flags = p.flags;
        }
    }
    else
    {
        p.status = (p.status == 0) ? (NQ_UINT32)syGetLastError() : (NQ_UINT32)ccErrorsStatusToNq(p.status, TRUE);
        LOGERR(CM_TRC_LEVEL_ERROR, "NETLOGON::NetrServerAuthenticate2");
    }

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%u", p.status);
    return p.status;
}

static NQ_COUNT composeNetrServerAuthenticate3(NQ_IOBufPos buffer, NQ_COUNT size, void *params, NQ_BOOL *moreData)
{
    CMBufferWriter w;       /* for composing request */
    ParamsNetrServerAuthenticate2and3 *p = (ParamsNetrServerAuthenticate2and3 *)params;
    NQ_UINT32 ref = 0;      /* ref if */
    NQ_UINT32 sz;           /* string size */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buff:%p size:%d params:%p more:%p", IOBUF_GETBUFPTR(buffer), size, params, moreData);

    cmBufferWriterInit(&w, buffer, size);

    cmBufferWriteUint16(&w, NETRSERVERAUTHENTICATE3_NETLOGON_OPNUM);

    /* logon server name prefixed by double back slash */
    sz = 3 + (NQ_UINT32)cmWStrlen(p->server);

    cmBufferWriteUint32(&w, ++ref);                /* ref ID */
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteAsciiAsUnicodeN(&w, "\\\\", 2, CM_BSF_NOFLAGS);
    cmBufferWriteUnicode(&w, p->server);
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTE(buffer, 2), 4);        /* 4 byte alignment */
    IOBUF_SKIPBACKBYTE(buffer, 2);

    /* user name = own computer name (with trailing '$')*/
    sz = 2 + (NQ_UINT32)cmWStrlen(p->computer);

    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteUnicodeNoNull(&w, p->computer);
    cmBufferWriteAsciiAsUnicodeN(&w, "$", 1, CM_BSF_WRITENULLTERM);
    cmBufferWriteUint16(&w, 2);                    /* type: workstation */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTE(buffer, 2), 4);        /* 4 byte alignment */
    IOBUF_SKIPBACKBYTE(buffer, 2);

    /* own computer name (no trailing '$') */
    sz = 1 + (NQ_UINT32)cmWStrlen(p->computer);

    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteUnicode(&w, p->computer);

    /* client credential */
    cmBufferWriteBytes(&w, p->credential->client, sizeof(p->credential->client));
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTE(buffer, 2), 4);        /* 4 byte alignment */
    IOBUF_SKIPBACKBYTE(buffer, 2);

    /* negotiation flags */
    cmBufferWriteUint32(&w, p->flags);

    *moreData = FALSE;

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return cmBufferWriterGetDataCount(&w);
}

static NQ_STATUS processNetrServerAuthenticate3(NQ_IOBufPos data, NQ_COUNT size, void *params, NQ_BOOL moreData)
{
    CMBufferReader r;
    ParamsNetrServerAuthenticate2and3 *p = (ParamsNetrServerAuthenticate2and3 *)params;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "data:%p size:%d params:%p more:%s", IOBUF_GETBUFPTR(data), size, params, moreData ? "TRUE" : "FALSE");

    cmBufferReaderInit(&r, data, size);
    cmBufferReadBytes(&r, p->credential->server, sizeof(p->credential->server));
    cmBufferReadUint32(&r, &p->flags);          /* negotiation flags */
    cmBufferReaderSkip(&r, sizeof(NQ_UINT32));  /* account rid */
    cmBufferReadUint32(&r, &p->status);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", p->status);
    return (NQ_STATUS)p->status;
}

NQ_UINT32 ccNetrServerAuthenticate3Ex(NQ_HANDLE netlogon, const NQ_WCHAR *server, const NQ_WCHAR *computer, CCNetlogonCredential *credential, NQ_UINT32 *flags)
{
    ParamsNetrServerAuthenticate2and3 p;  /* parameters struct shared between 2 methods */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "handle:%p server:%s computer:%s credential:%p flags:%p", netlogon, cmWDump(server), cmWDump(computer), credential, flags);

    p.server = server;
    p.computer = computer;
    p.credential = credential;
    p.flags = *flags;
    p.status = 0;

    /* call NETLOGON::NetrServerAuthenticate3 */
    if (TRUE == ccDcerpcOverTcpCall(netlogon, composeNetrServerAuthenticate3, processNetrServerAuthenticate3, &p))
    {
        *flags = p.flags;
    }
    else
    {
        p.status = (p.status == 0) ? (NQ_UINT32)syGetLastError() : (NQ_UINT32)ccErrorsStatusToNq(p.status, TRUE);
        LOGERR(CM_TRC_LEVEL_ERROR, "NETLOGON::NetrServerAuthenticate3");
    }

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%u", p.status);
    return p.status;
}

static NQ_COUNT composeDsrEnumerateDomainTrusts(NQ_IOBufPos buffer, NQ_COUNT size, void *params, NQ_BOOL *moreData)
{
    CMBufferWriter w;   /* for composing request */
    ParamsDsrEnumerateDomainTrusts  *p = (ParamsDsrEnumerateDomainTrusts  *)params;
    NQ_UINT32 ref = 0;  /* ref id */
    NQ_UINT32 sz;       /* string size */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buff:%p size:%d params:%p more:%p", IOBUF_GETBUFPTR(buffer), size, params, moreData);

    cmBufferWriterInit(&w, buffer, size);

    cmBufferWriteUint16(&w, DSRENUMERATEDOMAINTRUSTS_NETLOGON_OPNUM);

    /* server name */
    sz = 1 + (NQ_UINT32)cmWStrlen(p->server);
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteUnicode(&w, p->server);
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTE(buffer, 2), 4);        /* 4 byte alignment */
    IOBUF_SKIPBACKBYTE(buffer, 2);

    /* trust flags */
    cmBufferWriteUint32(&w, NETLOGON_TRUST_FLAGS);

    *moreData = FALSE;

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return cmBufferWriterGetDataCount(&w);
}

static NQ_STATUS processDsrEnumerateDomainTrusts(NQ_IOBufPos data, NQ_COUNT size, void *params, NQ_BOOL moreData)
{
    CMBufferReader structs;             /* parser for structures */
    CMBufferReader strings;             /* parser for strings */
    ParamsDsrEnumerateDomainTrusts * p = (ParamsDsrEnumerateDomainTrusts *)params;
    NQ_UINT32 count = 0;                /* number of answers */
    NQ_INT i;                           /* just a counter */
    NQ_IOBufPos bufPos;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "data:%p size:%d params:%p more:%s", IOBUF_GETBUFPTR(data), size, params, moreData ? "TRUE" : "FALSE");

    cmBufferReaderInit(&structs, data, size);
    cmBufferReadUint32(&structs, &count);
    cmBufferReaderSkip(&structs, 2 * 4);

    bufPos = cmBufferReaderGetPosition(&structs);
    cmBufferReaderInit(&strings, IOBUF_SKIPBYTE(bufPos, (NQ_INT)(44 * count)), size);

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Count: %d", count);
    if (count == 0)
    {
        cmBufferReadUint32(&structs, &p->status);
        goto Exit;
    }

    for (i = 0; i < (NQ_INT)count; i++)
    {
        NQ_UINT32 maxCount;                 /* name length */
        NQ_UINT32 refIdNetBIOS;             /* refId */
        NQ_UINT32 refIdDNS;                 /* refId */
        NQ_UINT32 refIdSID;                 /* refId */

        cmBufferReadUint32(&structs, &refIdNetBIOS);
        cmBufferReadUint32(&structs, &refIdDNS);
        cmBufferReaderSkip(&structs,(NQ_UINT)(4 * 4));     /* 4 fields in structure */
        cmBufferReadUint32(&structs, &refIdSID);
        cmBufferReaderSkip(&structs, 16);                 /* GUID */

        /* write NetBIOS domain names to output buffer */
        if (0 != refIdNetBIOS)
        {
            NQ_IOBufPos bufPos;

            cmBufferReadUint32(&strings, &maxCount);                                  /* read NetBIOS domain name length */
            cmBufferReaderSkip(&strings, 4 * 2);                                      /* skip offset and actual count */
            bufPos = cmBufferReaderGetPosition(&strings);
            IOBUF_BUFFER_CANUSEFLAT_ASSERT(bufPos, (NQ_UINT)(maxCount * sizeof(NQ_WCHAR)));
            (*p->callback)((NQ_WCHAR *)IOBUF_GETBYTEPTR(bufPos), NULL, p->list);
            cmBufferReaderSkip(&strings,(NQ_UINT)(maxCount * sizeof(NQ_WCHAR)));     /* skip name */
            cmBufferReaderAlign(&strings, data , 4);                      /* 4 byte alignment */
        }
        if (0 != refIdDNS)
        {
            cmBufferReadUint32(&strings, &maxCount);                                  /* read DNS domain name length */
            cmBufferReaderSkip(&strings,(NQ_UINT)( 4 * 2 + maxCount * sizeof(NQ_WCHAR)));        /* skip DNS domain name */
            cmBufferReaderAlign(&strings, data , 4);                       /* 4 byte alignment */
        }
        if (0 != refIdSID)
        {
            cmBufferReaderSkip(&strings, 4 + 24);                                     /* skip SID */
        }
    }

    cmBufferReadUint32(&strings, &p->status);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", p->status);
    return (NQ_STATUS)p->status;
}


/* returns list of domain names, up to buffer space supplied */
NQ_UINT32 ccDsrEnumerateDomainTrusts(NQ_HANDLE netlogon, const NQ_WCHAR *server, CCNetrEnumerateNamesCallback callback, CMList *list)
{
    ParamsDsrEnumerateDomainTrusts p;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "logon:%p server:%s callback:%p list:%p", netlogon, cmWDump(server), callback, list);

    p.server = server;
    p.callback = callback;
    p.list = list;
    p.status = 0;

    /* call NETLOGON::DsrEnumerateDomainTrusts */
    if (TRUE == ccDcerpcCall(netlogon, composeDsrEnumerateDomainTrusts, processDsrEnumerateDomainTrusts, &p))
    {

    }
    else
    {
        p.status = (p.status == 0) ? (NQ_UINT32)syGetLastError() : (NQ_UINT32)ccErrorsStatusToNq(p.status, TRUE);
        LOGERR(CM_TRC_LEVEL_ERROR, "NETLOGON::DsrEnumerateDomainTrusts");
    }

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%u", p.status);
    return p.status;
}


void ccNetlogonCredentialsRandom(NQ_BYTE *buffer, NQ_UINT32 size)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buffer:%p size:%u", buffer, size);

    cmCreateRandomByteSequence(buffer, size);

    /* first 5 bytes must be unique */
    while ((buffer[0] == buffer[1]) && (buffer[0] == buffer[2]) && (buffer[0] == buffer[3]) && (buffer[0] == buffer[4]))
    {
        cmCreateRandomByteSequence(buffer, size);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/* init and generate next??? netlogon credentials, called after NetrServerReqChallenge */
void ccNetlogonCredentialsInit(CCNetlogonCredential *creds, const NQ_BYTE secret[16], NQ_UINT32 flags)
{
    NQ_BOOL doAES = (flags == NETLOGON_NEG_FLAGS_AUTH3);

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "creds:%p, secret:%p, flags:0x%x", creds, &secret, flags);

    /* creds->client and creds->server are already populated by NetrServerReqChallenge call */
    creds->sequence = (NQ_UINT32)syGetTimeInSec();
    creds->negotFlags = flags;
    syMemcpy(creds->secret, secret, sizeof(*secret));

    /* calculate session key and client credentials for next step */
    if (TRUE == doAES)
    {
        cmNetlogonCredentialsGenerateSessKeyEx((const NQ_BYTE*)creds->client, (const NQ_BYTE*)creds->server, secret, creds->sessionKey);
    }
    else
    {
        /* not supported yet, less secure anyway */
        /* if (creds->negotFlags & NETLOGON_NEG_FLAGS_STRONG_KEYS) */
        cmNetlogonCredentialsGenerateSessKey((const NQ_BYTE*)creds->client, (const NQ_BYTE*)creds->server, secret, creds->sessionKey);
    }

    /* generate the next client and next server creds using old creds and newly created session key */
    cmNetlogonCredentialsCryptNext(doAES, (const NQ_BYTE*)creds->sessionKey, (const NQ_BYTE*)creds->client, creds->client);
    cmNetlogonCredentialsCryptNext(doAES, (const NQ_BYTE*)creds->sessionKey, (const NQ_BYTE*)creds->server, creds->server);

    /* save newly created client challenge */
    syMemcpy(creds->seed, creds->client, sizeof(creds->seed));

    dumpNetlogonCredentials("Init done", creds);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/* once secure channel is established, call it before each rpc call */
void ccNetlogonCredentialsNext(CCNetlogonCredential *creds)
{
    NQ_BOOL doAES = (creds->negotFlags & NETLOGON_NEG_FLAGS_SUPPORTS_AES);
    NQ_BYTE data[8];
    NQ_UINT32 *p;
    NQ_UINT32 temp;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "creds:%p", creds);

    syMemcpy(data, creds->seed, sizeof(data));

    p = (NQ_UINT32*)data;
    temp = cmLtoh32(*p);
    *p = cmHtol32(temp + creds->sequence);

    cmNetlogonCredentialsCryptNext(doAES, (const NQ_BYTE *)creds->sessionKey, (const NQ_BYTE *)data, creds->client);

    syMemcpy(data, creds->seed, sizeof(data));

    p = (NQ_UINT32*)data;
    temp = cmLtoh32(*p);
    *p = cmHtol32(temp + creds->sequence + 1);

    cmNetlogonCredentialsCryptNext(doAES, (const NQ_BYTE *)creds->sessionKey, (const NQ_BYTE *)data, creds->server);

    /*creds->sequence += 2; */

    dumpNetlogonCredentials("after netlogonCredentialsNext", creds);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/* used by ccDomainLogon() only */
void ccNetlogonCredentialsGenerate(CCNetlogonCredential *credential, const NQ_BYTE secret[16])
{
    cmNetlogonCredentialsGenerateSessKey((const NQ_BYTE*)credential->client, (const NQ_BYTE*)credential->server, secret, credential->sessionKey);
    cmNetlogonCredentialsCryptNext(FALSE, (const NQ_BYTE *)credential->sessionKey, (const NQ_BYTE *)credential->client, credential->client);
    cmNetlogonCredentialsCryptNext(FALSE, (const NQ_BYTE *)credential->sessionKey, (const NQ_BYTE *)credential->server, credential->server);
}

void dumpNetlogonCredentials(const NQ_CHAR *text, CCNetlogonCredential *cred)
{
#ifdef UD_NQ_INCLUDETRACE
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "%s cred:%p", text, cred);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "flags:0x%x negotFlags:0x%x time:%u seq:%u", cred->flags, cred->negotFlags, cred->time, cred->sequence);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "sesskey", cred->sessionKey, 8);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "client ", cred->client, 8);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "server ", cred->server, 8);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "seed ", cred->seed, 8);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
#endif /* UD_NQ_INCLUDETRACE */
}

static void dumpSchannelState(CCSchannel *pSchannel)
{
#ifdef UD_NQ_INCLUDETRACE
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pSchannel:%p", pSchannel);
    dumpNetlogonCredentials(SY_LOG_FUNCTION, &pSchannel->creds);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "schannel sequence:0x%x 0x%x\n", pSchannel->sequence.high, pSchannel->sequence.low);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
#endif /* UD_NQ_INCLUDETRACE */
}

static void dumpSchannelAuthSignature(NQ_BYTE *pSig, NQ_UINT16 sigLen)
{
#ifdef UD_NQ_INCLUDETRACE
    CMBufferReader reader;
    NQ_UINT16 uint16;
    NQ_BYTE bytes[24];

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pSig:%p sigLen:%u", pSig, sigLen);
    cmBufferReaderInit(&reader, pSig, sigLen);
    cmBufferReadUint16(&reader, &uint16);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Sign algo:0x%x", uint16);
    cmBufferReadUint16(&reader, &uint16);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Seal algo:0x%x", uint16);
    cmBufferReadUint16(&reader, &uint16);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Pad:0x%x", uint16);
    cmBufferReadUint16(&reader, &uint16);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Flags:0x%x", uint16);
    cmBufferReadBytes(&reader, bytes, 8);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "Sequence:", bytes, 8);
    cmBufferReadBytes(&reader, bytes, 8);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "Checksum (digest):", bytes, 8);
    cmBufferReadBytes(&reader, bytes, 8);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "Confounder:", bytes, 8);
    cmBufferReadBytes(&reader, bytes, sizeof(bytes));
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL,"Reserved 24:", bytes, sizeof(bytes));
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
#endif /* UD_NQ_INCLUDETRACE */
}

/* generate next schannel sequence number based on current */
void ccSchannelGenerateSequenceNumber(NQ_BYTE result[8], NQ_UINT64 currSeqNum, NQ_BOOL isOutgoing)
{
    NQ_UINT32 *pResult = (NQ_UINT32 *)result;
    NQ_UINT32 partLow = currSeqNum.low;
    NQ_UINT32 partHigh = currSeqNum.high | ((TRUE == isOutgoing) ? 0x80000000 : 0);

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "result:%p currSeqNum.high:%u currSeqNum.low:%u isOutgoing:%d ", result, currSeqNum.high, currSeqNum.low, isOutgoing);

    *pResult = syHton32(partLow);
    ++pResult;
    *pResult = syHton32(partHigh);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/* channel sequence number - encrypt always the sequence number with session key and calculated signature, write into packet */
void ccSchannelSequence(CCSchannel *pSchannel, const NQ_BYTE sequence[8], NQ_BYTE *pSig)
{
    NQ_BOOL doAes = (0 != (pSchannel->creds.negotFlags & NETLOGON_NEG_FLAGS_SUPPORTS_AES));
    NQ_BYTE *pChecksum = pSig + SCHANNEL_CHECKSUM_OFFSET;
    NQ_BYTE *pSequence = pSig + SCHANNEL_SEQUENCE_OFFSET;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pSchannel:%p sequence:%p pSig:%p", pSchannel, sequence, pSig);

    if (TRUE == doAes)
    {
        NQ_BYTE initVector[16] = {0};

        /* init vector with twice checksum */
        syMemcpy(initVector, pChecksum, 8);
        syMemcpy(initVector + 8, pChecksum, 8);

        AES_128_CFB8_Encrypt(pSchannel->creds.sessionKey, initVector, sequence, 8, pSequence);
    }
    else
    {
        /* not supported yet, less secure anyway (RC4 key) */
    }

    cmU64Inc(&pSchannel->sequence);

    dumpSchannelAuthSignature(pSig, 56); /* max len */

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/* sign the data and fill the signature trailer fields */
void ccSchannelSign(CCSchannel *pSchannel, const NQ_BYTE *pData, NQ_UINT dataLen, NQ_BOOL doSeal, NQ_BYTE *pSig, NQ_UINT16 sigLen, NQ_BOOL isOutgoing)
{
    NQ_BOOL doAes = (0 != (pSchannel->creds.negotFlags & NETLOGON_NEG_FLAGS_SUPPORTS_AES));
    CMBufferWriter writer;  /* write the auth trailer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pSchannel:%p pData:%p dataLen:%u doSeal:%d pSig:%p sigLen:%u ", pSchannel, pData, dataLen, doSeal, pSig, sigLen);

    dumpSchannelState(pSchannel);

    cmBufferWriterInit(&writer, pSig, sigLen);

    if (TRUE == doAes)
    {
        /* NL_AUTH_SHA2_SIGNATURE */

        /* write the 'header' of token, except the checksum */
        cmBufferWriteUint16(&writer, RPC_SIGN_ALGO_HMAC_SHA256);    /* sign algorithm */
        if (TRUE == doSeal)
        {
            cmBufferWriteUint16(&writer, RPC_SEAL_ALGO_AES_128);    /* seal algorithm */
        }
        else
        {
            cmBufferWriteUint16(&writer, RPC_SEAL_ALGO_NONE);       /* seal algorithm  - not encrypted */
        }
        cmBufferWriteUint16(&writer, 0xFFFF);                       /* padding must be 0xFFFF */
        cmBufferWriteUint16(&writer, 0);                            /* flags must be 0 */
        cmBufferWriterSkip(&writer, 8);                             /* skip sequence, calculated last */

        if (TRUE == doSeal)
        {
            NQ_BYTE confounder[8];
            NQ_BYTE *pConfounder = NULL;

            if (TRUE == isOutgoing)
            {
                ccNetlogonCredentialsRandom(confounder, sizeof(confounder));
                pConfounder = confounder;
            }
            else
            {
                pConfounder = pSig + SCHANNEL_CONFOUNDER_OFFSET;
            }

            /* compute the checksum and write into the packet */
            cmNetlogonSchannelSign(pSchannel->creds.sessionKey, pSig, pConfounder,
                                   pData, dataLen, cmBufferWriterGetPosition(&writer));
            cmBufferWriterSkip(&writer, 8);                                 /* skip checksum, already in packet */
            cmBufferWriteBytes(&writer, pConfounder, 8);                    /* confounder */
        }
        else
        {
            /* compute the checksum and write into the packet */
            cmNetlogonSchannelSign(pSchannel->creds.sessionKey, pSig, NULL,
                                   pData, dataLen, cmBufferWriterGetPosition(&writer));
            cmBufferWriterSkip(&writer, 8);                                 /* skip checksum, already in packet */
            cmBufferWriteZeroes(&writer, 8);                                /* confounder */
        }
        cmBufferWriteZeroes(&writer, 24);                                   /* reserved 24 bytes zeroed */

        dumpSchannelAuthSignature(pSig, sigLen);
    }
    else
    {
        /* not supported yet, less secure anyway  */
        /* NL_AUTH_SIGNATURE */

        NQ_BYTE zero[4] = {0};
        NQ_BYTE digest[16];
        NQ_UINT fragmentsNumber = 0;
        CMBlob fragments[4];
        CMBlob sessKey;

        /* write the 'header' of token */
        cmBufferWriteUint16(&writer, RPC_SIGN_ALGO_HMAC_MDA5);  /* sign algorithm */
        if (TRUE == doSeal)
        {
            cmBufferWriteUint16(&writer, RPC_SEAL_ALGO_RC4);    /* seal algorithm */
        }
        else
        {
            cmBufferWriteUint16(&writer, RPC_SEAL_ALGO_NONE);   /* seal algorithm  - not encrypted */
        }
        cmBufferWriteUint16(&writer, 0xFFFF);                   /* padding must be 0xFFFF */
        cmBufferWriteUint16(&writer, 0);                        /* flags must be 0 */

        fragments[fragmentsNumber].data = zero;
        fragments[fragmentsNumber].len = sizeof(zero);
        fragments[++fragmentsNumber].data = pSig;               /* header of token */
        fragments[fragmentsNumber].len = 8;
        if (TRUE == doSeal)
        {
            cmCreateRandomByteSequence(pSig + SCHANNEL_CONFOUNDER_OFFSET, 8);
            fragments[++fragmentsNumber].data = pSig + SCHANNEL_CONFOUNDER_OFFSET;   /* confounder */
            fragments[fragmentsNumber].len = 8;
        }

        fragments[++fragmentsNumber].data = (NQ_BYTE *)pData;   /* data */
        fragments[fragmentsNumber++].len = dataLen;

        /* first digest all over */
        cmMD5Fragments(NULL, NULL, fragments, fragmentsNumber, digest, sizeof(digest));

        /* then md5 with session key write into checksum */
        sessKey.data = pSchannel->creds.sessionKey;
        sessKey.len = sizeof(pSchannel->creds.sessionKey);
        fragments[0].data = digest;
        fragments[0].len = sizeof(digest);
        /* write the checksum (digest) field */
        cmHMACMD5Fragments(&sessKey, NULL, fragments, 1, pSig + SCHANNEL_CHECKSUM_OFFSET, 8);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/* seal (encrypt) the data, write the modified confounder into packet */
void ccSchannelSeal(CCSchannel *pSchannel, NQ_BYTE *pData, NQ_UINT dataLen, const NQ_BYTE sequence[8], NQ_BYTE *pSig, NQ_UINT16 sigLen, NQ_BOOL isOutgoing)
{
    NQ_BOOL doAes = (0 != (pSchannel->creds.negotFlags & NETLOGON_NEG_FLAGS_SUPPORTS_AES));
    CMBufferWriter writer;  /* write the token */

    cmBufferWriterInit(&writer, pSig, sigLen);

    if (TRUE == doAes)
    {
        NQ_COUNT i;
        NQ_BYTE key[16];
        NQ_BYTE initVector[16] = {0};
        NQ_BYTE *pConfounder = pSig + SCHANNEL_CONFOUNDER_OFFSET;

        for (i = 0; i < sizeof(key); i++)
        {
            key[i] = pSchannel->creds.sessionKey[i] ^ 0xf0;
        }

        /* init vector with twice sequence */
        syMemcpy(initVector, sequence, 8);
        syMemcpy(initVector + 8, sequence, 8);

        if (TRUE == isOutgoing)
        {
            AES_128_CFB8_Encrypt(key, initVector, pConfounder, 8, pConfounder);
            AES_128_CFB8_Encrypt(key, initVector, pData, dataLen, pData);
        }
        else
        {
            AES_128_CFB8_Decrypt(key, initVector, pConfounder, 8, pConfounder);
            AES_128_CFB8_Decrypt(key, initVector, pData, dataLen, pData);
        }
    }
    else
    {
        /* not supported yet, less secure anyway */
    }
}


#endif /* UD_NQ_INCLUDECIFSCLIENT */

