/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client file operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#ifndef _CCFILE_H_
#define _CCFILE_H_

#include "ccshare.h"
#include "ccmount.h"

/* -- Constants -- */

/* Description
 * What is the durable file state.
 */
#define DURABLE_REQUIRED        1
#define DURABLE_NOTREQUIRED     2
#define DURABLE_GRANTED         3 /* granted durable handle by server */
#define DURABLE_CANCELED        4 /* durable handle canceled by server via break */

#define FILE_OPLOCK_LEVEL_NONE  1
#define FILE_OPLOCK_LEVEL_II    2
#define FILE_OPLOCK_LEVEL_BATCH 3

#define RESUME_KEY_SIZE         24 /* define the resume key array size as defined in the MS-SMB2 spec */

/* -- Structures -- */

/* Description
   This structure describes open file.
   
   Since this structure inherits from <link CMItem> the file name
   is designated as item name. 
   
   It is using unlock callback. It references the respective server. */
typedef struct _ccfile
{
    CMItem item;                /* List item. */
    CCShare * share;            /* Pointer to the remote share descriptor. */
    CCMountIdentifier  mountPointID; /* Mount point identifier. */
    NQ_BYTE fid[16];            /* File ID. The usage of this field and its structure
                                   depends on SMB dialect. */
    NQ_UINT32 accessMask;       /* NT Format access rights. */
    NQ_UINT32 sharedAccess;     /* Share access. */
    NQ_UINT32 disposition;      /* How to open file. */
    NQ_UINT32 options;          /* Create/open options. */
    NQ_UINT32 attributes;       /* File attributes. */
    NQ_BOOL open;               /* TRUE when file has been open. */
    NQ_UINT64 offset;           /* Current offset in the file */
    NQ_UINT16 maxRpcXmit;       /* Maximum length of RPC transmit fragment */
    NQ_UINT16 maxRpcRecv;       /* Maximum length of RPC receive fragment */
    NQ_BOOL isPipe;             /* TRUE when this is a pipe */
    NQ_BYTE grantedOplock;      /* Level of the oplock that has been granted*/
    NQ_BOOL disconnected;       /* TRUE when connection was disconnected */
#ifdef UD_NQ_INCLUDESMB2
    NQ_UINT durableState;       /* States: durable required / durable not required / durable granted. see above */
    NQ_Uuid durableHandle;      /* Durable handle. */
    NQ_UINT32 durableTimeout;   /* Timeout for durable or persistent handle granted by server. */
    NQ_UINT32 durableFlags;     /* Durable or persistent flag granted by server. */
#endif /* UD_NQ_INCLUDESMB2 */
    NQ_UINT64 volumeId;         /* Volume Id within which the file is opened.*/
} 
CCFile; /* Open file descriptor. */

/* Description
   This structure describes A 24-byte resume key.
   Generated by the server that can be subsequently used by the client to uniquely identify the source file
   in an FSCTL_SRV_COPYCHUNK or FSCTL_SRV_COPYCHUNK_WRITE request.

   The array size defined in the MS-SMB2 spec. */
typedef struct _ccresumekey
{
    NQ_BYTE key[RESUME_KEY_SIZE];
} CCResumeKey;

/* Description
   This structure describes an individual data range to copy in a single FSCTL_SRV_COPYCHUNK or FSCTL_SRV_COPYCHUNK_WRITE request.   */
typedef struct _ccchunk
{
    NQ_UINT64 sourceOffset; /* The offset, in bytes, from the beginning of the source file to the location from which the data will be copied */
    NQ_UINT64 targetOffset; /* The offset, in bytes, from the beginning of the destination file to where the data will be copied */
    NQ_UINT32 length;       /* The number of bytes of data to copy */
} CCChunk;

/* Description
   This structure holds a CCChunk array pointer and the size of this array.   */
typedef struct _ccchunks
{
    CCChunk *   chunks;             /* A pointer to the chunks array */
    NQ_UINT     numberOfChunks;     /* Number of allocated chunks */
} CCChunks;

/* Description
   This structure describes a response from a single FSCTL_SRV_COPYCHUNK or FSCTL_SRV_COPYCHUNK_WRITE request.   */
typedef struct _ccchunkstatus
{
    NQ_STATUS status;              /* The Status field in the SMB2 header of the server-side copy operation response, it can be STATUS_SUCCESS or STATUS_INVALID_PARAMETER */
    NQ_UINT32 chunksWritten;       /* This value indicates the number of chunks that were successfully written.
                                       In case of an error, this value indicates the maximum number of chunks that the server will accept in a single request.
                                       This would allow the client to correctly reissue the request */
    NQ_UINT32 chunkBytesWritten;   /* This value indicates the number of bytes written in the last chunk that did not successfully process (if a partial write occurred).
                                       In case of an error, this value indicates the maximum number of bytes the server will allow to be written in a single chunk. */
    NQ_UINT32 totalBytesWritten;    /* This value indicates the total number of bytes written in the server-side copy operation.
                                       In case of an error, this value indicates the maximum number of bytes the server will accept to copy in a single request */
} CCChunksStatus;

/* -- API Functions */
NQ_HANDLE ccCreateFileNoBatch(
    const NQ_WCHAR * fileName,
    NQ_INT access,
    NQ_INT shareMode,
    NQ_INT locality,
    NQ_BOOL writeThrough,
    NQ_UINT16 attributes,
    NQ_INT createAction,
    NQ_INT openAction
    );

/* Description
   Initialize this module.
   Returns 
   None
 */
NQ_BOOL ccFileStart(void);

/* Description
   Release resources used by this module.
   Returns 
   None
 */
void ccFileShutdown(void);

/* Description
   Find open file by name.
   Parameters
   pShare :  Pointer to share to look files on.
   path : Local path to the file, starting from mount point.
   Returns
   Pointer to file descriptor or NULL if it was not found. */
CCFile * ccFileFind(CCShare * pShare, const NQ_WCHAR * path);

/* Description
   Find open file by file id.

   This function traverses the tree of users and shares on the given server.
   Parameters
   pServer :  Pointer to the server object.
   id : File ID to look for.
   Returns
   Pointer to file descriptor or NULL if it was not found. */
CCFile * ccFileFindById(CCServer * pServer, const NQ_BYTE * id);

/* Description
   Add new file descriptor to the list of open files for the given share.
   
   This call:
     * Locates share;
     * Creates and links a file object;
   Parameters
   pShare : Pointer to share to create file on.
   pMount : Pointer to mount point.
   path : Local path to the file, starting from mount point.
   Returns
   Pointer to file descriptor or NULL on failure. Creating a
   file may fail for the following reasons:
     * out of memory.*/
CCFile * ccFileCreate(CCShare * pShare, CCMount * pMount, const NQ_WCHAR * path);

/* Description
   Create/Open file on server.
   
   This call creates and links a file object on a known share.
   Parameters
   pShare :        Pointer to share to create share on.
   path :          Local path to the file, starting from mount
                   point.
   pathIsLocal:   TRUE if path is local (e.g mounted share)
                      FALSE if path is remote (e.g srvsvc)
   access :        Access mode (see <i>ccCreateFileA/W</i>).
   shareMode :     Share mode (see <i>ccCreateFileA/W</i>).
   locality :      Locality (see <i>ccCreateFileA/W</i>)
   writeThrough :  Write through indicator (see <i>ccCreateFileA/W</i>).
   attributes :    File attributes (see <i>ccCreateFileA/W</i>).
   createAction :  Create action (see<i> ccCreateFileA/W</i>).
   openAction :    Open action (see <i>ccCreateFileA/W</i>).
   isPipe :        <i>TRUE</i> if the file is pipe, <i>FALSE</i>
                   for a regular file.
   Returns
   Pointer to an allocated file descriptor or NULL on failure.
   Creating a file may fail for the following reasons:
     * out of memory.                                                    */
CCFile * ccFileCreateOnServer(
        CCShare * pShare, 
        const NQ_WCHAR * path,
        NQ_BOOL pathIsLocal,
        NQ_INT access, 
        NQ_INT shareMode,
        NQ_INT locality, 
        NQ_BOOL writeThrough, 
        NQ_UINT16 attributes, 
        NQ_INT createAction, 
        NQ_INT openAction,
        NQ_BOOL isPipe,
        NQ_INT oplockLevel
        );

/* Description
   Starts operations caused by (temporary) disconnect.
   
   NQ call this function when it detects a condition that may
   indicate loss of connection to a the server. This call:
     * Locates server;
     * Calls server's reconnect method (this will cause
       reconnecting of all its underneath objects);
     * Reports reconnect results (either reconnected of failed);
   Parameters
   pFile : Pointer to the file object.
   Returns
   This call returns TRUE when the server was successfully
   reconnected. This call returns FALSE in one of the following
   cases:
     * Server connection still exists &#45; this was a false
       alarm;
     * NQ cannot reconnect to the server;
     * NQ cannot reconnect to the share;
     * NQ cannot reopen this file;                               */
NQ_BOOL ccFileReportDisconnect(CCFile * pFile);

/* Description
   This call re-opens previously open file by restoring its handle.
   
   This happens when NQ detects a connection loss (see <link ccFileReportDisconnect@CCFile *, ccFileReportDisconnect()>.
   When NQ calls this function it has also restored the enclosing context:
     * Connected to the server;
     * Conveyed negotiations;
     * Logged on;
     * Connected to file's share;
   This function attempts to restore file handle by using
   underlying protocol method. If the protocol does not support
   durable file handle, the protocol layer will fail the
   operation.
   Parameters
   pFile :  Pointer to the file object.
   Returns
   This call returns TRUE if file handle was restored
   successfully .It returns FALSE in one of the following cases:
     * The SMB dialect agreed upon during negotiations does not
       support durable handles;
     * Server does not support durable handles;
     * Server did not grant a durable handle before connection
       loss;
     * Re&#45;opening failed;                                                                                            */ 
NQ_BOOL ccFileRestore(CCFile * pFile);

NQ_BOOL ccValidateFileHandle(NQ_HANDLE handle);

void ccFileTake(NQ_HANDLE handle);

void ccFileGive(NQ_HANDLE handle);

NQ_HANDLE ccGetFileHandleByName(const NQ_CHAR *path, NQ_INT desiredAccess, NQ_UINT32 desiredSharedAccess, NQ_BYTE desiredOpLock, NQ_UINT16 attributes);

#endif /* _CCFILE_H_ */
