/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client dummy smb operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "ccapi.h"
#include "ccsmb20.h"
#include "cmsmb2.h"
#include "ccserver.h"
#include "ccshare.h"
#include "ccuser.h"
#include "ccfile.h"
#include "ccsearch.h"
#include "ccinfo.h"
#include "ccsmb2common.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT


/* CCCifsSmb methods */
static void * allocateContext(CCServer * server);
static void freeContext(void * context, void * server);
static void setSolo(NQ_BOOL set){};
static NQ_STATUS doNegotiate(CCServer * pServer, CMBlob * blob);
static NQ_STATUS doSessionSetup(CCUser * pUser, const CMBlob * pass1, const CMBlob * pass2);
static NQ_STATUS doSessionSetupExtended(CCUser * pUser, const CMBlob * outBlob, CMBlob * inBlob);
static NQ_STATUS doLogOff(CCUser * pUser);
static NQ_STATUS doTreeConnect(CCShare * pShare);
static NQ_STATUS doTreeDisconnect(CCShare * pShare);
static NQ_STATUS doCreate(CCFile * pFile);
static NQ_STATUS doRestoreHandle(CCFile * pFile);
static NQ_STATUS doClose(CCFile * pFile);
static NQ_STATUS doQueryDfsReferrals(CCShare * share, const NQ_WCHAR * path, CCCifsParseReferral parser, CMList * list);
static NQ_STATUS doFindOpen(CCSearch * pSearch);
static NQ_STATUS doFindMore(CCSearch * pSearch);
static NQ_STATUS doFindClose(CCSearch * pSearch);
static NQ_STATUS doWrite(CCFile * pFile, const NQ_IOBufPos data, NQ_UINT bytesToWrite, CCCifsWriteCallback callback, void * context, void *hook);
static NQ_STATUS doRead(CCFile * pFile, const NQ_IOBufPos data, NQ_UINT bytesToRead, CCCifsReadCallback callback, void * context, void *hook);
static NQ_STATUS doQueryResumeFileKey(CCFile * pSrcFile, CCResumeKey * pKey);
static NQ_STATUS doServerSideDataCopy(CCFile * pDstFile, NQ_BOOL isReadAccess, CCResumeKey * pSrcFileKey, CCChunks * pChunks, CCChunksStatus * pChunkStatus);
#ifdef UD_CC_INCLUDESECURITYDESCRIPTORS
static NQ_STATUS doQuerySecurityDescriptor(CCFile * pFile, CMSdSecurityDescriptor * sd);
static NQ_STATUS doSetSecurityDescriptor(CCFile * pFile, const CMSdSecurityDescriptor * sd);
#endif /* UD_CC_INCLUDESECURITYDESCRIPTORS */
static NQ_STATUS doQueryFsInfo(CCShare * pShare, const NQ_WCHAR * pathName, CCVolumeInfo * pInfo);
static NQ_STATUS doQueryFileInfoByName(CCShare * pShare, const NQ_WCHAR * fileName, CCFileInfo * pInfo);
static NQ_STATUS doQueryFileInfoByHandle(CCFile * pFile, CCFileInfo * pInfo);
static NQ_STATUS doSetFileAttributes(CCFile * pFile, NQ_UINT32 attributes);
static NQ_STATUS doSetFileSize(CCFile * pFile, NQ_UINT64 size);
static NQ_STATUS doSetFileTime(CCFile * pFile, NQ_UINT64 creationTime, NQ_UINT64 lastAccessTime, NQ_UINT64 lastWriteTime);
static NQ_STATUS doSetFileDeleteOnClose(CCFile * pFile);
static NQ_STATUS doRename(CCFile * pFile, const NQ_WCHAR * newName);
static NQ_STATUS doFlush(CCFile * pFile);
static NQ_STATUS doRapTransaction(void * pShare, const CMBlob * inData, CMBlob * outParams, CMBlob * outData);
static NQ_STATUS doEcho(CCShare * pShare);
static NQ_BOOL   validateNegotiate(void *pServ, void *pUser, void *pShare);

static void handleWaitingNotifyResponse(void *pServer, void *pFile);
static NQ_STATUS sendRequest(CCServer * pServer, CCUser * pUser, Request * pRequest, Match * pMatch, NQ_BOOL (*callback)(CMItem * pItem));
static NQ_STATUS sendReceive(CCServer * pServer, CCUser * pUser, Request * pRequest, Response * pResponse);
static void anyResponseCallback(void * transport);

static NQ_STATUS keyDerivation(void * user);
static void signalAllMatches(void * pTransport);
/* -- Static data */

static const NQ_WCHAR rpcPrefix[] = { 0 };  /* value to prefix RPC pipe names */

static const CCCifsSmb dialect =
{
        SMB2_DIALECTSTRING,
        SMB2_DIALECTREVISION,
        16,
        TRUE,
        rpcPrefix,
        (void * (*)(void *))allocateContext,
        freeContext,
        setSolo,
        (NQ_STATUS (*)(void *, CMBlob *))doNegotiate,
        (NQ_STATUS (*)(void *, const CMBlob *, const CMBlob *))doSessionSetup,
        (NQ_STATUS (*)(void *, const CMBlob *, CMBlob *))doSessionSetupExtended,
        (NQ_STATUS (*)(void *))doLogOff,
        (NQ_STATUS (*)(void *))doTreeConnect,
        (NQ_STATUS (*)(void *))doTreeDisconnect,
        (NQ_STATUS (*)(void *))doCreate,
        (NQ_STATUS (*)(void *))doRestoreHandle,
        (NQ_STATUS (*)(void *))doClose,
        (NQ_STATUS (*)(void *, const NQ_WCHAR *, CCCifsParseReferral, CMList *))doQueryDfsReferrals,
        (NQ_STATUS (*)(void *))doFindOpen,
        (NQ_STATUS (*)(void *))doFindMore,
        (NQ_STATUS (*)(void *))doFindClose,
        (NQ_STATUS (*)(void *, const NQ_IOBufPos, NQ_UINT, CCCifsWriteCallback, void *, void *))doWrite,
        (NQ_STATUS (*)(void *, const NQ_IOBufPos, NQ_UINT, CCCifsReadCallback, void *, void *))doRead,
        (NQ_STATUS (*)(void *, void *))doQueryResumeFileKey,
        (NQ_STATUS (*)(void *, NQ_BOOL, void *, void *, void *))doServerSideDataCopy,
#ifdef UD_CC_INCLUDESECURITYDESCRIPTORS
        (NQ_STATUS (*)(void *, CMSdSecurityDescriptor *))doQuerySecurityDescriptor,
        (NQ_STATUS (*)(void *, const CMSdSecurityDescriptor *))doSetSecurityDescriptor,
#endif /* UD_CC_INCLUDESECURITYDESCRIPTORS */
        (NQ_STATUS (*)(void *, const NQ_WCHAR *, void *))doQueryFsInfo,
        (NQ_STATUS (*)(void *, const NQ_WCHAR *, void *))doQueryFileInfoByName,
        (NQ_STATUS (*)(void *, void *))doQueryFileInfoByHandle,
        (NQ_STATUS (*)(void *, NQ_UINT32))doSetFileAttributes,
        (NQ_STATUS (*)(void *, NQ_UINT64))doSetFileSize,
        (NQ_STATUS (*)(void *, NQ_UINT64, NQ_UINT64, NQ_UINT64))doSetFileTime,

        (NQ_STATUS (*)(void *))doSetFileDeleteOnClose,
        (NQ_STATUS (*)(void *, const NQ_WCHAR *))doRename,
        (NQ_STATUS (*)(void * pFile))doFlush,
        (NQ_STATUS(*)(void *, const CMBlob *, CMBlob *, CMBlob *))doRapTransaction,
        (NQ_STATUS (*)(void *))doEcho,
        (NQ_STATUS (*)(void * pServer, void * pUser, void * pRequest, void * pMatch, NQ_BOOL (*callback)(CMItem * pItem)))sendRequest,
        (NQ_STATUS (*)(void * pServer, void * pUser, void * pRequest, void * pResponse))sendReceive,
        anyResponseCallback,
        keyDerivation,
        signalAllMatches,
        handleWaitingNotifyResponse,
        validateNegotiate,
        FALSE,
        TRUE
};

/* -- API Functions */

NQ_BOOL ccSmbStart()
{
    return TRUE;
}

NQ_BOOL ccSmbShutdown()
{
    return TRUE;
}

const CCCifsSmb *ccSmbDummyGetCifs(void)
{
    return &dialect;
}

/* -- Static functions -- */

static void * allocateContext(CCServer * pServer)
{
    void *result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pServer:%p",pServer);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NULL;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->allocateContext(pServer);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return result;
}

static void freeContext(void * context, void * server)
{
    CCServer * pServer = (CCServer *)server;    /* casted pointer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "context:%p server:%p", context, server);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    pServer->smb->freeContext(context, server);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return;
}


static NQ_STATUS sendRequest(CCServer * pServer, CCUser * pUser, Request * pRequest, Match * pMatch, NQ_BOOL (*callback)(CMItem * pItem))
{
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p user:%p request:%p match:%p", pServer, pUser, pRequest, pMatch);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->sendRequest(pServer, pUser, pRequest, pMatch, callback);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", result);
    return result;
}

static NQ_STATUS sendReceive(CCServer * pServer, CCUser * pUser, Request * pRequest, Response * pResponse)
{
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p user:%p request:%p response:%p", pServer, pUser, pRequest, pResponse);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->sendReceive(pServer, pUser, pRequest, pResponse);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", result);
    return result;
}

/*
     * Server sends notify responses == response without request. Ex: break notification
     * If file ID for sent response isn't found. we save the response and try again on newly created files.
     * To avoid missing a break notification that is handled while file creation on our side still in process.
     */
static void handleWaitingNotifyResponse(void *pserver, void *pfile)
{
    CCServer * pServer = (CCServer *)pserver;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pServer:%p pfile:%p", pserver, pfile);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    pServer->smb->handleWaitingNotifyResponses(pserver, pfile);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return;
}

static void anyResponseCallback(void * transport)
{
    CCTransport * pTransport = (CCTransport *)transport;    /* casted to transport entry */
    CCServer * pServer = (CCServer *)pTransport->context;                           /* casted pointer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "transport:%p", transport);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    pServer->smb->anyResponseCallback(transport);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return;
}

static NQ_STATUS doNegotiate(CCServer * pServer, CMBlob * inBlob)
{
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p inBlob:%p", pServer, inBlob);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doNegotiate(pServer, inBlob);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}


static NQ_STATUS doSessionSetup(CCUser * pUser, const CMBlob * pass1, const CMBlob * pass2)
{
    CCServer *pServer = pUser->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user:%p pass1:xxx pass2:xxx", pUser);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doSessionSetup(pUser, pass1, pass2);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doSessionSetupExtended(CCUser * pUser, const CMBlob * outBlob, CMBlob * inBlob)
{
    CCServer * pServer;     /* server object pointer */
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user:%p outBlob:%p inBlob:%p", pUser, outBlob, inBlob);

    pServer = pUser->server;
    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doSessionSetup(pUser, outBlob, inBlob);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", result);
    return result;
}

static NQ_STATUS doLogOff(CCUser * pUser)
{
    CCServer * pServer = pUser->server;     /* server object pointer */
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user:%p", pUser);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doLogOff(pUser);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doTreeConnect(CCShare * pShare)
{
    CCServer * pServer = pShare->user->server;      /* server object pointer */
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "share:%p", pShare);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doTreeConnect(pShare);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doTreeDisconnect(CCShare * pShare)
{
    CCServer * pServer = pShare->user->server;      /* server object pointer */
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "share:%p", pShare);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doTreeDisconnect(pShare);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doCreate(CCFile * pFile)
{
    CCServer * pServer = pFile->share->user->server;        /* server object pointer */
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p", pFile);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doCreate(pFile);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doRestoreHandle(CCFile * pFile)
{
    CCServer * pServer = pFile->share->user->server;        /* server object pointer */
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p", pFile);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pFile->share->user->server->smb->doRestoreHandle(pFile);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doClose(CCFile * pFile)
{
    CCServer * pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p", pFile);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pFile->share->user->server->smb->doClose(pFile);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doQueryDfsReferrals(CCShare * share, const NQ_WCHAR * path, CCCifsParseReferral parser, CMList * list)
{
    CCServer * pServer = share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "share:%p path:%s parser:%p list:%p", share, cmWDump(path), parser, list);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doQueryDfsReferrals(share, path, parser, list);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doFindOpen(CCSearch * pSearch)
{
    CCServer * pServer = pSearch->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "search:%p", pSearch);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doFindOpen(pSearch);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doFindMore(CCSearch * pSearch)
{
    CCServer * pServer = pSearch->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "search:%p", pSearch);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doFindOpen(pSearch);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doFindClose(CCSearch * pSearch)
{
    CCServer * pServer = pSearch->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "search:%p", pSearch);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doFindClose(pSearch);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}


static NQ_STATUS doWrite(CCFile * pFile, const NQ_IOBufPos data, NQ_UINT bytesToWrite, CCCifsWriteCallback callback, void * context, void *hook)
{
    CCServer  *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p dtat:%p bytes:%u callback:%p context:%p", pFile, data, bytesToWrite, callback, context);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doWrite(pFile, data, bytesToWrite, callback, context, hook);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}


static NQ_STATUS doRead(CCFile * pFile, const NQ_IOBufPos buffer, NQ_UINT bytesToRead, CCCifsReadCallback callback, void * context, void *hook)
{
    CCServer  *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p buff:%p bytes:%u callback:%p context:%p", pFile, buffer, bytesToRead, callback, context);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doRead(pFile, buffer, bytesToRead, callback, context, hook);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doQueryResumeFileKey(CCFile * pSrcFile, CCResumeKey * pKey)
{
    CCServer    *pServer = pSrcFile->share->user->server;
    NQ_STATUS   result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pSrcFile:%p pKey:%p", pSrcFile, pKey);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doQueryResumeFileKey(pSrcFile, pKey);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doServerSideDataCopy(CCFile * pDstFile, NQ_BOOL isReadAccess, CCResumeKey * pSrcFileKey, CCChunks * pChunks, CCChunksStatus * pChunkStatus)
{
    CCServer    *pServer = pDstFile->share->user->server;
    NQ_STATUS   result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "destFile:%p readAccess:%s pSrcFileKey:%p pChunks:%p pChunkStatus:%p",
                                        pDstFile, isReadAccess ? "TRUE" : "FALSE", pSrcFileKey, pChunks, pChunkStatus);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doServerSideDataCopy(pDstFile, isReadAccess, pSrcFileKey, pChunks, pChunkStatus);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

#ifdef UD_CC_INCLUDESECURITYDESCRIPTORS

static NQ_STATUS doQuerySecurityDescriptor(CCFile * pFile, CMSdSecurityDescriptor * sd)
{
    CCServer  *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p sd:%p", pFile, sd);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doQuerySecurityDescriptor(pFile, sd);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doSetSecurityDescriptor(CCFile * pFile, const CMSdSecurityDescriptor * sd)
{
    CCServer *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p sd:%p", pFile, sd);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doSetSecurityDescriptor(pFile, sd);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

#endif /* UD_CC_INCLUDESECURITYDESCRIPTORS */


static NQ_STATUS doQueryFileInfoByHandle(CCFile * pFile, CCFileInfo * pInfo)
{
    CCServer *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p info:%p", pFile, pInfo);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doQueryFileInfoByHandle(pFile, pInfo);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doQueryFileInfoByName(CCShare * pShare, const NQ_WCHAR * fileName, CCFileInfo * pInfo)
{
    CCServer *pServer = pShare->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "share:%p file:%s info:%p", pShare, cmWDump(fileName), pInfo);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doQueryFileInfoByName(pShare, fileName, pInfo);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doQueryFsInfo(CCShare * pShare, const NQ_WCHAR * pathName, CCVolumeInfo * pInfo)
{
    CCServer *pServer = pShare->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "share:%p pathName:%s info:%p", pShare, cmWDump(pathName), pInfo);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doQueryFsInfo(pShare, pathName, pInfo);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doSetFileAttributes(CCFile * pFile, NQ_UINT32 attributes)
{
    CCServer *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p attr:0x%x", pFile, attributes);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doSetFileAttributes(pFile, attributes);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doSetFileSize(CCFile * pFile, NQ_UINT64 size)
{
    CCServer  *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p size(low,high):%u,%u", pFile, size.low, size.high);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doSetFileSize(pFile, size);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doSetFileTime(CCFile * pFile, NQ_UINT64 creationTime, NQ_UINT64 lastAccessTime, NQ_UINT64 lastWriteTime)
{
    CCServer *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p creationTime(low,high):%u,%u lastAccessTime(low,high):%u,%u lastWriteTime(low,high):%u,%u",
                pFile, creationTime.low, creationTime.high, lastAccessTime.low, lastAccessTime.high, lastWriteTime.low, lastWriteTime.high);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doSetFileTime(pFile, creationTime, lastAccessTime, lastWriteTime);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doSetFileDeleteOnClose(CCFile * pFile)
{
    CCServer *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p", pFile);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doSetFileDeleteOnClose(pFile);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doRename(CCFile * pFile, const NQ_WCHAR * newName)
{
    CCServer *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p name:%s", pFile, cmWDump(newName));

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doRename(pFile, newName);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS doFlush(CCFile * pFile)
{
    CCServer *pServer = pFile->share->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p", pFile);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doFlush(pFile);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static  NQ_STATUS doRapTransaction(void * pshare, const CMBlob * inData, CMBlob * outParams, CMBlob * outData)
{
    CCShare *pShare = (CCShare *) pshare;
    CCServer *pServer = pShare->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "share:%p inData:%p params:%p outData:%p", pshare, inData, outParams, outData);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doRapTransaction(pshare, inData, outParams, outData);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}


static NQ_STATUS doEcho(CCShare * pShare)
{
    CCServer *pServer = pShare->user->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "share:%p", pShare);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->doEcho(pShare);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_STATUS keyDerivation(void * user)
{
    CCUser *pUser = (CCUser *)user;
    CCServer *pServer = pUser->server;
    NQ_STATUS result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user:%p", user);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->keyDerivation(user);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", result);
    return result;
}

static void signalAllMatches(void * trans)
{
    CCTransport *   pTransport = (CCTransport *)trans;
    CCServer *      pServer = (CCServer *)pTransport->server;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "transport:%p", trans);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    pServer->smb->signalAllMatch(trans);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return;
}

static NQ_BOOL validateNegotiate(void *pServ, void *_pUser, void *pShare)
{
    CCServer *pServer = (CCServer *)pServ;
    NQ_BOOL  result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pServ: %p pUser: %p pShare:%p", pServ, _pUser, pShare);

    cmListItemTake(&pServer->item);
    if (FALSE == ccTransportIsConnected(&pServer->transport))
    {
        cmListItemGive(&pServer->item);
        LOGERR(CM_TRC_LEVEL_ERROR, "The server is not connected");
        result = FALSE;
        goto Exit;
    }

    cmListItemGive(&pServer->item);
    result = pServer->smb->validateNegotiate(pServer, _pUser, pShare);

Exit:
    return result;
}


#endif /* UD_NQ_INCLUDECIFSCLIENT */

