/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : System epoll operations
 *--------------------------------------------------------------------
 * MODULE        : Linux - SY
 * DEPENDENCIES  :
 *************************************************************************/

#ifndef _SYEPOLL_H_
#define _SYEPOLL_H_

#include "cmapi.h"
#include "syinclud.h"
#include "syopsyst.h"

/*@@SY_EPOLL_AVAILABLE
   Description
   NQ may use either an epoll-like mechanism or a select-like
   mechanism for polling events on sockets.
   
   Define this parameter to use <i>epoll</i> or alike. Disable
   this parameter to use <i>select</i> or alike.               */
#define SY_EPOLL_AVAILABLE
/*#undef SY_EPOLL_AVAILABLE*/

/*@@SYEpollHandle
   Description
   An abstract handle to the event polling mechanism.  */
#define SYEpollHandle int

/*@@syEpollIsValid
   \ \ 
   Description
   This call checks that a given handle is a valid handle to an
   event polling mechanism.                                     */
#define syEpollIsValid(_handle_) (_handle_ != -1)

/*@@syEpollInvalidHandle
   \ \ 
   Description
   This call invalidates a handle.  */
#define syEpollInvalidHandle() (-1)

/*@@syEpollCreate@NQ_COUNT
   Description
   NQ calls this function on startup to create an event polling
   mechanism.
   Parameters
   num :  This parameter must be ignored. 
   Returns
   A handle of the polling mechanism.                           */
SYEpollHandle syEpollCreate(NQ_COUNT num);

/*@@
   Description
   NQ calls this function on shutdown to dispose all resorces associated with
   the event polling mechanism.
   Parameters
   handle :   The handle obtained through a call to the <link syEpollCreate@NQ_COUNT, syEpollCreate Function>.
   Returns
   None                 */
void syEpollDelete(SYEpollHandle handle);

/*@@syEpollAddSocket@SYEpollHandle@SYSocketHandle@void *
   Description
   NQ calls this function to add a socket to the event polling
   mechanism.
   Parameters
   handle :   The handle obtained through a call to the <link syEpollCreate@NQ_COUNT, syEpollCreate Function>.
   sock :     Handle of the socket to add.
   context :  An abstract pointer to be returned on the
              respective event. See <link syEpollWait@SYEpollHandle@void **@NQ_UINT32, syEpollWait Function>
              for details.
   Returns
   None                                                                                                        */
void syEpollAddSocket(SYEpollHandle handle, SYSocketHandle sock, void * context);

/*@@syEpollRearmSocket@SYEpollHandle@SYSocketHandle@void *
   Description
   NQ calls this function after there was on event on the given
   socket. After this call the socket will be polled again.
   Parameters
   handle :   The handle obtained through a call to the <link syEpollCreate@NQ_COUNT, syEpollCreate Function>.
   sock :     Handle of the socket to re\-arm.
   context :  An abstract pointer to be returned on the
              respective event. See <link syEpollWait@SYEpollHandle@void **@NQ_UINT32, syEpollWait Function>
              for details.
   Returns
   None                                                                                                        */
void syEpollRearmSocket(SYEpollHandle handle, SYSocketHandle sock, void * context);

/*@@syEpollRemoveSocket@SYEpollHandle@SYSocketHandle
   Description
   NQ calls this function to stop polling events on a given
   socket.
   Parameters
   handle :  The handle obtained through a call to the <link syEpollCreate@NQ_COUNT, syEpollCreate Function>.
   sock :    Handle of the socket to remove.
   Returns
   None                                                                                                       */
void syEpollRemoveSocket(SYEpollHandle handle, SYSocketHandle sock);

/*@@syEpollWait@SYEpollHandle@void **@NQ_UINT32
   Description
   NQ calls this function to wait for events.
   
   The implementation must guarantee that events will be
   proceeded one by one. This means that when two events happen,
   this function implementation must return for each one of them
   separately.
   Parameters
   handle :   The handle obtained through a call to the <link syEpollCreate@NQ_COUNT, syEpollCreate Function>.
   context :  A buffer for the event context. On a successful
              socket event this buffer must be filled with the
              context of the respective socket as was specified
              in either in a <link syEpollAddSocket@SYEpollHandle@SYSocketHandle@void *, syEpollAddSocket Function>
              call or in a <link syEpollRearmSocket@SYEpollHandle@SYSocketHandle@void *, syEpollRearmSocket Function>
              call. 
   timeout :  Number of milliseconds to wait for events.
   Returns
   This function returns one of the following:
     * Number of events happened. This value must be always one.
     * Zero on timeout.
     * NQ_FAIL on error.                                                                                              */
NQ_INT syEpollWait(SYEpollHandle handle, void ** context, NQ_UINT32 timeout);

#endif /* _SYEPOLL_H_ */
