/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Common memory operations
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 *************************************************************************/

#include "cmmemory.h"
#include "cmapi.h"
#if (!defined(UD_NQ_DYNAMICALLOCATION) || defined(REPOSITORY_SIZE_TUNING))
#include "cmmemoryrepo.h"
#endif /* !defined(UD_NQ_DYNAMICALLOCATION) || defined(REPOSITORY_SIZE_TUNING) */

/* -- Constants and typedefs -- */

/*#define PRINTOUT*/                    /* define this to have debug printouts on the screen */
/*#define MEMORY_SHUTDOWN_PRINTOUT*/    /* define this to get a report if there is any unreleased memory */

#if defined(MEMORY_SHUTDOWN_PRINTOUT) && !defined(UD_NQ_INCLUDETRACE)
#error "MEMORY_SHUTDOWN_PRINTOUT requires UD_NQ_INCLUDETRACE"
#endif /* defined(MEMORY_SHUTDOWN_PRINTOUT) && !defined(UD_NQ_INCLUDETRACE) */

/* each allocated memory block is prefixed by a 32-bit signature value and a 32-bit size */
#define SIGNATURE ((NQ_UINT32)0xFED12345)    

typedef struct
{
#ifdef UD_NQ_INCLUDETRACE
    CMItem item;                /* element in linked list */
    NQ_UINT line;               /* line number */
#endif /* UD_NQ_INCLUDETRACE */
    NQ_UINT32 size;             /* block size */
    NQ_UINT32 signature;        /* NQ signature */
    NQ_UINT seq;                /* sequence number */
    NQ_BYTE* originalBufPtr;    /* since we align - need to keep original pointer */
} Extra;

typedef struct 
{
    NQ_UINT64 maxAllowedBytes;
    NQ_UINT64 bytesInUse;
#ifdef UD_NQ_INCLUDETRACE
    NQ_UINT32 bytesAllocated;
    NQ_UINT32 bytesDeallocated;
    NQ_UINT32 blocksAllocated;
    NQ_UINT32 blocksDeallocated;
    NQ_UINT32 blocksInUse;
#endif /* UD_NQ_INCLUDETRACE */
} Statistics;


/* -- Static data -- */

static NQ_BOOL isModuleInitialized = FALSE;
static Statistics statistics;   /* memory usage statistics */
static NQ_INT nextSeq;          /* next sequence number */
#ifdef UD_NQ_INCLUDETRACE 
static CMList blocks;           /* list of allocated memory blocks */
#endif /* UD_NQ_INCLUDETRACE */


/* -- Static functions --*/

static void * memoryAllocate(NQ_UINT size, const NQ_CHAR *text, const NQ_UINT line);
static void memoryFree(const void * block, const NQ_CHAR *function, const NQ_UINT line);
static NQ_WCHAR * memoryCloneWString(const NQ_WCHAR * str, const NQ_CHAR *function, const NQ_UINT line);
static NQ_WCHAR * memoryCloneAString(const NQ_CHAR * str, const NQ_CHAR *function, const NQ_UINT line);
static NQ_CHAR * memoryCloneWStringAsAscii(const NQ_WCHAR * str, const NQ_CHAR *function, const NQ_UINT line);

#ifdef UD_NQ_INCLUDETRACE
static void dumpBlock(CMItem *pItem)
{
    Extra *extra = (Extra*)((NQ_BYTE *)pItem - sizeof(NQ_UINT32) * 2 - sizeof(NQ_UINT));       
    LOGMSG(CM_TRC_LEVEL_MEMORY, "       Memory block # %d %p of %d bytes not released yet", extra->seq, extra + 1 ,extra->size);
}
#endif /* UD_NQ_INCLUDETRACE */

/* -- API functions -- */

NQ_BOOL cmMemoryStart(NQ_UINT64 maxBytes)
{
    if (isModuleInitialized)
        return TRUE;

    syMemset(&statistics, 0, sizeof(statistics));
    cmU64AssignU64(&statistics.maxAllowedBytes, &maxBytes);
    cmU64Zero(&statistics.bytesInUse);
    nextSeq = 0;
#ifdef UD_NQ_INCLUDETRACE
    cmListStart(&blocks);
    blocks.name = "memory";
#endif /* UD_NQ_INCLUDETRACE */

    /* set mem alignment MACRO */
#if (2 == MEM_ALLIGNMENT_BYTES)
#define memAllignMacro(ptr) cmAllignTwo(ptr)
#elif (4 == MEM_ALLIGNMENT_BYTES)
#define memAllignMacro(ptr) cmAllignFour(ptr)
#elif (8 == MEM_ALLIGNMENT_BYTES)
#define memAllignMacro(ptr) cmAllign8(ptr)
#elif (16 == MEM_ALLIGNMENT_BYTES)
#define memAllignMacro(ptr) cmAllign16(ptr)
#elif (32 == MEM_ALLIGNMENT_BYTES)
#define memAllignMacro(ptr) cmAllign32(ptr)
#elif (64 == MEM_ALLIGNMENT_BYTES)
#define memAllignMacro(ptr) cmAllign64(ptr)
#else
#error "Bad value for MEM_ALLIGNMENT_BYTES, please define one of: 2, 4, 8, 16, 32, 64."
#endif

    isModuleInitialized = TRUE;
    return TRUE;
}

void cmMemoryShutdown(void)
{
    if (isModuleInitialized)
    {
#ifdef UD_NQ_INCLUDETRACE
        CMIterator  memItr;

        cmListIteratorStart(&blocks,&memItr);
#ifdef MEMORY_SHUTDOWN_PRINTOUT
        syPrintf("\n =========Unreleased Memory List:============= \n");
#endif /* MEMORY_SHUTDOWN_PRINTOUT */
        if (cmListIteratorHasNext(&memItr))
        {
#ifdef MEMORY_SHUTDOWN_PRINTOUT
            Extra *extra;
            syPrintf("||Memory usage:\n");
            syPrintf("||  bytes in use %u %u\n", statistics.bytesInUse.high, statistics.bytesInUse.low);
            syPrintf("||  blocks in use %d\n", statistics.blocksInUse);
#endif /* MEMORY_SHUTDOWN_PRINTOUT */
            while (cmListIteratorHasNext(&memItr))
            {
                CMItem  *pItem;

                pItem = cmListIteratorNext(&memItr);

                cmListItemRemove(pItem);

#ifdef MEMORY_SHUTDOWN_PRINTOUT
                extra = (Extra*)pItem;
                syPrintf(" Memory Item Address: %p Name: %s line: %u\n" , pItem , cmWDump(pItem->name), extra->line);
                syPrintf("  ->Memory block #%d %p of %d bytes not released yet \n", extra->seq, extra + 1, extra->size);
#endif /* MEMORY_SHUTDOWN_PRINTOUT */
                if (NULL != pItem->name)
                {
                    syFree(pItem->name);
                }
                if (NULL != pItem->guard)
                {
                    syMutexDelete(pItem->guard);
                    syFree(pItem->guard);
                }

                syFree(pItem);
                pItem = NULL;
            }
        }
#ifdef MEMORY_SHUTDOWN_PRINTOUT
        else
        {
            syPrintf("\tNo unreleased blocks were found \n");
        }
        syPrintf(" ============================================= \n");
#endif /* MEMORY_SHUTDOWN_PRINTOUT */
        cmListIteratorTerminate(&memItr);
#endif /* UD_NQ_INCLUDETRACE */
        isModuleInitialized = FALSE;
    }
}


void * cmMemoryAllocNonDebug(NQ_UINT size)
{
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    return memoryAllocate(size, NULL, 0);
#else
    return cmMemoryRepoAllocate(size , NULL , 0);
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
}


#ifdef UD_NQ_INCLUDETRACE
void * cmMemoryAllocDebug(NQ_UINT size, const NQ_CHAR *function, const NQ_UINT line)
{
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    return memoryAllocate(size, function, line);
#else
    return cmMemoryRepoAllocate(size , function , line);
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
}
#endif /* UD_NQ_INCLUDETRACE */


void * cmMemoryAllocStartupNonDebug(NQ_UINT size)
{
    return memoryAllocate(size, NULL, 0);
}


#ifdef UD_NQ_INCLUDETRACE
void * cmMemoryAllocStartupDebug(NQ_UINT size, const NQ_CHAR *function, const NQ_UINT line)
{
    return memoryAllocate(size, function, line);
}
#endif /* UD_NQ_INCLUDETRACE */


static void * memoryAllocate(NQ_UINT size, const NQ_CHAR *function, const NQ_UINT line)
{
    Extra * extra = NULL;
    void * pResult = NULL;
    NQ_UINT64 zero = {0,0};
    NQ_BYTE *allocBuf = NULL;

    if (size == 0)
    {
#ifdef PRINTOUT
        syPrintf("ALLOC: Trying to allocate 0 bytes");
#endif /* PRINTOUT */
        LOGERR(CM_TRC_LEVEL_MEMORY, "Trying to allocate 0 bytes");
        goto Exit;
    }

    /* if memory usage is limited */
    if (cmU64Cmp(&statistics.maxAllowedBytes, &zero) == 1) /* maxAllowedBytes > 0 */
    {
        NQ_UINT64 newAlloced;

        cmU64AssignU64(&newAlloced, &statistics.bytesInUse);
        cmU64AddU32(&newAlloced, size);
        if (cmU64Cmp(&newAlloced, &statistics.maxAllowedBytes) == 1) /* newAlloced > statistics.maxAllowedBytes */
        {
#ifdef PRINTOUT
            syPrintf("ALLOC: NQ run out of allowed memory; statistics.maxBytes:%u %u, statistics.bytesInUse:%u %u\n", statistics.maxAllowedBytes.high, statistics.maxAllowedBytes.low , statistics.bytesInUse.high, statistics.bytesInUse.low);
#endif /* PRINTOUT */
            LOGERR(CM_TRC_LEVEL_ERROR, "NQ run out of allowed memory; statistics.maxBytes:%u %u, statistics.bytesInUse:%u %u", statistics.maxAllowedBytes.high, statistics.maxAllowedBytes.low , statistics.bytesInUse.high, statistics.bytesInUse.low);
            goto Exit;
        }
    }

    size += (NQ_UINT)sizeof(Extra) + MEM_ALLIGNMENT_BYTES - 1;  /* signature + size */
    /* Valgrind find out that allocBuf is problematic:
     * Uninitialised value was created by a heap allocation.
     * In order to solve it a memset is needed for allocBuf, however, this memory block it too big in order to be set each time to zeros. */
    allocBuf = (NQ_BYTE *)syMalloc((NQ_UINT)size);
    if (NULL != allocBuf)
    {
#ifdef UD_NQ_INCLUDETRACE
        NQ_BOOL isAdded;

        statistics.bytesAllocated += (NQ_COUNT)(size - sizeof(Extra));
        statistics.blocksAllocated += 1;
        statistics.blocksInUse += 1;
#endif /* UD_NQ_INCLUDETRACE */
        extra = (Extra *)(memAllignMacro(allocBuf));
        extra->originalBufPtr = allocBuf;
        cmU64AddU32(&statistics.bytesInUse, (NQ_UINT32)(size - sizeof(Extra)));

        extra->signature = SIGNATURE;
        extra->size = (NQ_UINT32)(size - sizeof(Extra));
        extra->seq = (NQ_UINT)++nextSeq;
#ifdef UD_NQ_INCLUDETRACE
        cmListItemInit(&extra->item);
        extra->item.guard = (SYMutex *)syMalloc(sizeof(*extra->item.guard));
        if(NULL == extra->item.guard)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "syMalloc: Out of memory");
            goto Exit;
        }
        syMutexCreate(extra->item.guard);
        cmListStart(&extra->item.references);
        if (function)
        {
            extra->item.name = (NQ_WCHAR *)syMalloc((syStrlen(function) + 1)* sizeof(NQ_WCHAR));
            if (extra->item.name)
            {
                cmAnsiToUnicode(extra->item.name, function);
            }
        }
        isAdded = cmListItemAdd(&blocks, &extra->item, NULL);
        extra->line = line;
        extra->item.dump = dumpBlock;
        LOGMSG(CM_TRC_LEVEL_MEMORY, "ALLOC: bloc #%d, %d bytes, %p (called by %s() line %d) %s", extra->seq, extra->size, extra + 1, function ? function : "[]", line,  
            isAdded ? "" : "(not added to the list!)");
#endif /* UD_NQ_INCLUDETRACE */
#ifdef PRINTOUT
        syPrintf("ALLOC: bloc #%4d, %5d bytes by %s (line %d), %p\n", extra->seq, extra->size, function ? function : "[]", line, extra + 1);
#endif /* PRINTOUT */

        pResult = extra + 1;
        goto Exit;
    }
#ifdef UD_NQ_INCLUDETRACE   
    LOGERR(CM_TRC_LEVEL_ERROR, "ALLOC: failed %s (line %d), %d bytes", function, line, size);
    cmMemoryDump();
#endif /* UD_NQ_INCLUDETRACE */
#ifdef PRINTOUT
    syPrintf("ALLOC: failed %d bytes by %s (line %d)\n", size, function ? function : "[]", line);
#endif /* PRINTOUT */

Exit:
    return pResult;
}

void cmMemoryFreeNonDebug(const void * block)
{
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    memoryFree(block, NULL, 0);
#else
    cmMemoryRepoFree(block , NULL , 0);
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
}

#ifdef UD_NQ_INCLUDETRACE
void cmMemoryFreeDebug(const void * block, const NQ_CHAR *function, const NQ_UINT line)
{
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    memoryFree(block, function, line);
#else
    cmMemoryRepoFree(block , function , line);
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
}
#endif /* UD_NQ_INCLUDETRACE */

void cmMemoryFreeShutdownNonDebug(const void * block)
{
    memoryFree(block, NULL, 0);
}

#ifdef UD_NQ_INCLUDETRACE
void cmMemoryFreeShutdownDebug(const void * block, const NQ_CHAR *function, const NQ_UINT line)
{
    memoryFree(block, function, line);
}
#endif /* UD_NQ_INCLUDETRACE */

static void memoryFree(const void * block, const NQ_CHAR *function, const NQ_UINT line)
{
    Extra * extra;

    if (NULL == block)
    {   
#ifdef UD_NQ_INCLUDETRACE
        LOGERR(CM_TRC_LEVEL_MEMORY, "An attempt to free null block (called by %s() line %d)", function ? function : "[]", line);
#endif /* UD_NQ_INCLUDETRACE */
#ifdef PRINTOUT
        syPrintf("FREE: An attempt to free null block\n");
#endif /* PRINTOUT */
        return;
    }

    extra = (Extra *)block - 1;
    if (extra->signature == SIGNATURE)
    {
        NQ_UINT32 size = extra->size;
#ifdef UD_NQ_INCLUDETRACE       
        NQ_BOOL isRemoved;
#endif /* UD_NQ_INCLUDETRACE */

#ifdef PRINTOUT
       syPrintf("FREE:  bloc #%4d, %5d bytes by %s (line %d), %p\n", extra->seq, extra->size, function ? function : "[]", line, block);
#endif /* PRINTOUT */
        
#ifdef UD_NQ_INCLUDETRACE
        /* item name is not unique anyway 
        {
            CMItem *pItem;

            pItem = cmListItemFind(&blocks, extra->item.name, TRUE , FALSE);
            if (!pItem)
            {
                LOGERR(CM_TRC_LEVEL_MEMORY, "FREE: item %p %s not found in list!", block, extra->item.name ? cmWDump(extra->item.name) : "<none>");
            }
        }*/
        isRemoved = cmListItemRemove(&extra->item);
        LOGMSG(CM_TRC_LEVEL_MEMORY, "FREE: bloc #%d, %d bytes, %p (called by %s() line %d, allocated by %s() line %d) %s", extra->seq, extra->size, block, function ? function : "[]", line, 
            extra->item.name ? cmWDump(extra->item.name) : "<none>", extra->line, isRemoved ? "" : "Failed to remove block from the list");
        
        if (extra->item.name)
        {
            syFree(extra->item.name);
            extra->item.name = NULL;
        }
        if (extra->item.guard)
        {
            syMutexDelete(extra->item.guard);
            syFree(extra->item.guard);
            extra->item.guard = NULL;
        }        
        cmListShutdown(&extra->item.references);
#endif /* UD_NQ_INCLUDETRACE */
        syFree((void *)extra->originalBufPtr);
        cmU64SubU64U32(&statistics.bytesInUse, size);
#ifdef UD_NQ_INCLUDETRACE
        statistics.bytesDeallocated += size;
        statistics.blocksDeallocated += 1;
        statistics.blocksInUse -= 1;
#endif /* UD_NQ_INCLUDETRACE */
    }
    else
    {       
#ifdef UD_NQ_INCLUDETRACE
        LOGERR(CM_TRC_LEVEL_MEMORY, "An attempt to release bad memory block %p by %s (line %d)", block, function, line);
#endif /* UD_NQ_INCLUDETRACE */
#ifdef PRINTOUT
       syPrintf("FREE: An attempt to release bad memory block %p by %s (line %d)\n", block, function ? function : "[]", line);
#endif /* PRINTOUT */
       syFree((void *)block);
    }
}

#ifdef UD_NQ_INCLUDETRACE
NQ_WCHAR * cmMemoryCloneWStringDebug(const NQ_WCHAR * str, const NQ_CHAR *function, const NQ_UINT line)
{
    return memoryCloneWString(str, function, line);
}
#endif /* UD_NQ_INCLUDETRACE */

NQ_WCHAR * cmMemoryCloneWStringNonDebug(const NQ_WCHAR * str)
{
    return memoryCloneWString(str, NULL, 0);
}

static NQ_WCHAR * memoryCloneWString(const NQ_WCHAR * str, const NQ_CHAR *function, const NQ_UINT line)
{
    NQ_WCHAR * res = NULL;

    if (NULL == str)
    {
        goto Exit;
    }

#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    res = (NQ_WCHAR *)memoryAllocate((NQ_UINT)(sizeof(NQ_WCHAR) * (1 + cmWStrlen(str))), function, line);
#else
    res = (NQ_WCHAR *)cmMemoryRepoAllocate((NQ_UINT)(sizeof(NQ_WCHAR) * (1 + cmWStrlen(str))), function, line);
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
    if (NULL != res)
    {
        cmWStrcpy(res, str);
    }

Exit:
    return res;
}

#ifdef UD_NQ_INCLUDETRACE
NQ_WCHAR * cmMemoryCloneAStringDebug(const NQ_CHAR * str, const NQ_CHAR *function, const NQ_UINT line)
{
    return memoryCloneAString(str, function, line);
}
#endif /* UD_NQ_INCLUDETRACE */
NQ_WCHAR * cmMemoryCloneAStringNonDebug(const NQ_CHAR * str)
{
    return memoryCloneAString(str, NULL, 0);
}

static NQ_WCHAR * memoryCloneAString(const NQ_CHAR * str, const NQ_CHAR *function, const NQ_UINT line)
{
    NQ_WCHAR * res = NULL;

    if (NULL == str)
    {
        goto Exit;
    }

#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    res = (NQ_WCHAR *)memoryAllocate((NQ_UINT)(sizeof(NQ_WCHAR) * (1 + syStrlen(str))), function, line);
#else
    res = (NQ_WCHAR *)cmMemoryRepoAllocate((NQ_UINT)(sizeof(NQ_WCHAR) * (1 + syStrlen(str))), function, line);
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
    if (NULL != res)
    {
        cmAnsiToUnicode(res, str);
    }

Exit:
    return res;
}

#ifdef UD_NQ_INCLUDETRACE
NQ_CHAR * cmMemoryCloneWStringAsAsciiDebug(const NQ_WCHAR * str, const NQ_CHAR *function, const NQ_UINT line)
{
    return memoryCloneWStringAsAscii(str, function, line);
}
#endif /* UD_NQ_INCLUDETRACE */
NQ_CHAR * cmMemoryCloneWStringAsAsciiNonDebug(const NQ_WCHAR * str)
{
    return memoryCloneWStringAsAscii(str, NULL, 0);
}

static NQ_CHAR * memoryCloneAStringAsAscii(const NQ_CHAR * str, const NQ_CHAR *function, const NQ_UINT line)
{
    NQ_CHAR * res = NULL;

    if (NULL == str)
    {
        goto Exit;
    }

#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    res = (NQ_CHAR *)memoryAllocate((NQ_UINT)(sizeof(NQ_CHAR) * (1 + syStrlen(str))), function, line);
#else
    res = (NQ_CHAR *)cmMemoryRepoAllocate((NQ_UINT)(sizeof(NQ_CHAR) * (1 + syStrlen(str))), function, line);
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
    if (NULL != res)
    {
        syStrcpy(res, str);
    }

Exit:
    return res;
}

#ifdef UD_NQ_INCLUDETRACE
NQ_CHAR * cmMemoryCloneAStringAsAsciiDebug(const NQ_CHAR * str, const NQ_CHAR *function, const NQ_UINT line)
{
    return memoryCloneAStringAsAscii(str, function, line);
}
#endif /* UD_NQ_INCLUDETRACE */

NQ_CHAR * cmMemoryCloneAStringAsAsciiNonDebug(const NQ_CHAR * str)
{
    return memoryCloneAStringAsAscii(str, NULL, 0);
}

#ifdef UD_NQ_INCLUDETRACE
#define cmMemoryCloneWStringAsAscii(_str) cmMemoryCloneWStringAsAsciiDebug(_str, SY_LOG_FUNCTION, SY_LOG_LINE) 
#else
#endif /* UD_NQ_INCLUDETRACE */


static NQ_CHAR * memoryCloneWStringAsAscii(const NQ_WCHAR * str, const NQ_CHAR *function, const NQ_UINT line)
{
    NQ_CHAR * res = NULL;

    if (NULL == str)
    {
        goto Exit;
    }

    /* we allocate 4 x string length because of multibyte character strings (Chinese) */
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    res = (NQ_CHAR *)memoryAllocate((NQ_UINT)(4 * sizeof(NQ_CHAR) * (1 + syWStrlen(str))), function, line);
#else
    res = (NQ_CHAR *)cmMemoryRepoAllocate((NQ_UINT)(4 * sizeof(NQ_CHAR) * (1 + syWStrlen(str))), function, line);
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
    if (NULL != res)
    {
        cmUnicodeToAnsi(res, str);
    }

Exit:
    return res;
}

CMBlob memoryCloneBlob(const CMBlob * origin, const NQ_CHAR *function, const NQ_UINT line)
{
    CMBlob newBlob;
    
    newBlob.data = NULL;
    newBlob.len = origin->len;
    if (NULL != origin->data)
    {
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    {
        newBlob.data = (NQ_BYTE *)memoryAllocate(origin->len, function, line);
    }
#else
    {
        newBlob.data = (NQ_BYTE *)cmMemoryRepoAllocate(origin->len, function, line);
    }
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
        if (NULL != newBlob.data)
        {
            syMemcpy(newBlob.data, origin->data, newBlob.len);
        }
    }
    return newBlob;
}

CMIOBlob memoryCloneBlobToIOBlob(const CMBlob * origin, const NQ_CHAR *function, const NQ_UINT line)
{
    CMIOBlob newBlob;
    NQ_BYTE* bufPtr;

    IOBUF_POSINIT(newBlob.data);
    newBlob.len = origin->len;
    if (NULL != origin->data)
    {
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
        {
            bufPtr = (NQ_BYTE *)memoryAllocate(origin->len + (NQ_COUNT)IOBUF_POSCONSTRUCT_EXTRASIZE, function, line);
        }
#else
        {
            bufPtr = (NQ_BYTE *)cmMemoryRepoAllocate((origin->len + (NQ_COUNT)IOBUF_POSCONSTRUCT_EXTRASIZE), function, line);
        }
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
        if (NULL != bufPtr)
        {
            IOBUF_POSCONSTRUCTOR_INPLACE(newBlob.data, bufPtr, newBlob.len);
            IOBUF_MEMCPY_F2V(newBlob.data, origin->data, newBlob.len);
        }
    }
    return newBlob;
}

CMIOBlob memoryCloneIOBlob(const CMIOBlob * origin, const NQ_CHAR *function, const NQ_UINT line)
{
    CMIOBlob newBlob;
    NQ_BYTE* bufPtr;

    IOBUF_POSINIT(newBlob.data);
    newBlob.len = origin->len;
    if (!IOBUF_ISNULL(origin->data))
    {
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
        {
            bufPtr = (NQ_BYTE *)memoryAllocate(origin->len + (NQ_COUNT)IOBUF_POSCONSTRUCT_EXTRASIZE, function, line);
        }
#else
        {
            bufPtr = (NQ_BYTE *)cmMemoryRepoAllocate((origin->len + (NQ_COUNT)IOBUF_POSCONSTRUCT_EXTRASIZE), function, line);
        }
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
        if (NULL != bufPtr)
        {
            IOBUF_POSCONSTRUCTOR_INPLACE(newBlob.data, bufPtr, newBlob.len);
            IOBUF_MEMCPY_V2V(newBlob.data, origin->data, newBlob.len);
        }
    }
    return newBlob;
}

CMBlob memoryCloneIOBlobToBlob(const CMIOBlob * origin, const NQ_CHAR *function, const NQ_UINT line)
{
    CMBlob newBlob;
    NQ_BYTE* bufPtr;

    newBlob.len = 0;
    newBlob.data = NULL;
    if (!IOBUF_ISNULL(origin->data))
    {
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
        {
            bufPtr = (NQ_BYTE *)memoryAllocate(origin->len, function, line);
        }
#else
        {
            bufPtr = (NQ_BYTE *)cmMemoryRepoAllocate((origin->len), function, line);
        }
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
        if (NULL != bufPtr)
        {
            newBlob.len = origin->len;
            newBlob.data = bufPtr;
            IOBUF_MEMCPY_V2F(newBlob.data, origin->data, newBlob.len);
        }
    }
    return newBlob;
}


void memoryFreeBlob(CMBlob * blob, const NQ_CHAR *function, const NQ_UINT line)
{
    if (NULL != blob->data)
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    {
        memoryFree(blob->data, function, line);
    }
#else
    {
        cmMemoryRepoFree(blob->data, function, line);
    }
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
    blob->data = NULL;
    blob->len = 0;
}


void memoryFreeIOBlob(CMIOBlob * blob, const NQ_CHAR *function, const NQ_UINT line)
{
    if (!IOBUF_ISNULL(blob->data))
#if defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING)
    {
        memoryFree((NQ_BYTE*)IOBUF_GETBUFPTR(blob->data), function, line);
    }
#else
    {
        cmMemoryRepoFree((NQ_BYTE*)IOBUF_GETBUFPTR(blob->data), function, line);
    }
#endif /* defined(UD_NQ_DYNAMICALLOCATION) && !defined(REPOSITORY_SIZE_TUNING) */
    IOBUF_POSINIT(blob->data);
    blob->len = 0;
}


#ifdef UD_NQ_INCLUDETRACE

void cmMemoryStatisticsFull(NQ_UINT64 * memMaxAllowed, NQ_UINT64 * memAlloc, NQ_UINT32 * memDealloc, NQ_UINT32 * blockAlloc, NQ_UINT32 * blockDealloc)
{
    if (memMaxAllowed)  cmU64AssignU64(memMaxAllowed, &statistics.maxAllowedBytes);
    if (memAlloc)       cmU64AssignU64(memAlloc, &statistics.bytesInUse);
    if (memDealloc)     *memDealloc = statistics.bytesDeallocated;
    if (blockAlloc)     *blockAlloc = statistics.blocksAllocated;
    if (blockDealloc)   *blockDealloc = statistics.blocksDeallocated;
}

void cmMemoryDump(void)
{
    LOGFB(CM_TRC_LEVEL_MEMORY);
    LOGMSG(CM_TRC_LEVEL_MEMORY, "Memory usage:");
    LOGMSG(CM_TRC_LEVEL_MEMORY, "  bytes allocated %u", statistics.bytesAllocated);
    LOGMSG(CM_TRC_LEVEL_MEMORY, "  bytes deallocated %u", statistics.bytesDeallocated);
    LOGMSG(CM_TRC_LEVEL_MEMORY, "  blocks allocated %u", statistics.blocksAllocated);
    LOGMSG(CM_TRC_LEVEL_MEMORY, "  blocks deallocated %u", statistics.blocksDeallocated);
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "  bytes in use %u %u", statistics.bytesInUse.high, statistics.bytesInUse.low);
    LOGMSG(CM_TRC_LEVEL_MEMORY, "  blocks in use %u", statistics.blocksInUse);
    LOGMSG(CM_TRC_LEVEL_MEMORY, "  block dump:");
    cmListDump(&blocks);
    LOGFE(CM_TRC_LEVEL_MEMORY);
}

#endif /* UD_NQ_INCLUDETRACE */

void cmMemoryStatistics(NQ_UINT64 * memMaxAllowed, NQ_UINT64 * memInUse)
{
    if (memMaxAllowed)  cmU64AssignU64(memMaxAllowed, &statistics.maxAllowedBytes);
    if (memInUse)       cmU64AssignU64(memInUse, &statistics.bytesInUse);
}
