/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : The most basic definitions, common for all NQ modules
 *                 (this source assumes that system-dependent definitions
 *                 were not defined yet)
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 ********************************************************************/

#ifndef _CMCOMMON_H_
#define _CMCOMMON_H_

#ifndef NQ_SUCCESS
#define NQ_SUCCESS  (0)
#endif

#ifndef NQ_FAIL
#define NQ_FAIL     (-1)
#endif

#ifdef UD_NQ_USETRANSPORTIPVR
#define CM_IPADDR_IPVR 1
#endif /* UD_NQ_USETRANSPORTIPVR */
#define CM_IPADDR_IPV4 4
#ifdef UD_NQ_USETRANSPORTIPV6
#define CM_IPADDR_IPV6 6

#define CM_IPADDR_MAXLEN 46     /* max length of textual representation + trailing null */
#define CM_IP4ADDR_MAXLEN 16    /* max length of textual representation + trailing null */
#define CM_IPADDR_ZERO   {0, {SY_ZEROIP}}
#define CM_IPADDR_ANY4   {CM_IPADDR_IPV4, {SY_ANYIP4}}
#define CM_IPADDR_ANY6   {CM_IPADDR_IPV6, {SY_ANYIP6}}
#define CM_IPADDR_LOCAL  {CM_IPADDR_IPV4, {SY_LOCALHOSTIP4}}
#define CM_IPADDR_LOCAL6 {CM_IPADDR_IPV6, {SY_LOCALHOSTIP6}}

/*@@CM_IPADDR_VERSION
   Description
   Get IP address type.

   This call allows determining the concrete type of an abstract
   IP address.
   Parameters
   _addr :  IP address
   Returns
   CM_IPADDR_IPV4 or CM_IPADDR_IPV6 (see <link Other Constants>). */
#define CM_IPADDR_VERSION(_addr)       (_addr).version

/*@@CM_IPADDR_SIZE
   Description
   Get the actual size of an IP address.
   
   This call returns the number of bytes needed to hold the
   actual address.
   Parameters
   _addr :  IP address. 
   Returns
   Actual size of the address.                              */
#ifndef UD_NQ_USETRANSPORTIPVR
#define CM_IPADDR_SIZE(_addr) \
    (((_addr).version == CM_IPADDR_IPV6) ? sizeof(NQ_IPADDRESS6) \
    : (((_addr).version == CM_IPADDR_IPV4) ? sizeof(NQ_IPADDRESS4) :0 ))

/*@@CM_IPADDR_EQUAL
   Description
   Compare two abstract IP addresses.

   Both abstract addresses are equal when they are of the same
   type and their address values are equal.
   Parameters
   _addr :   First IP address.
   _other :  Second IP address.
   Returns
   <i>TRUE</i> when both addresses are equal, <i>FALSE</i>
   otherwise.                                                  */
#define CM_IPADDR_EQUAL(_addr, _other) \
    ((_addr).version == (_other).version && ((_addr).version == CM_IPADDR_IPV4 \
        ? ((_addr).addr.v4 == (_other).addr.v4) \
        : (syMemcmp((_addr).addr.v6, (_other).addr.v6, sizeof(NQ_IPADDRESS6)) == 0)))
#endif /* UD_NQ_USETRANSPORTIPVR */

/*@@CM_IPADDR_GET4
   Description
   Withdraw an IPv4 address from an abstract IP address.
   
   An IPv4 address as a loadable value.
   Parameters
   _addr :  IP address. 
   Returns
   IPv4 address.                                         */
#define CM_IPADDR_GET4(_addr) (_addr).addr.v4

/*@@CM_IPADDR_GET6
   Description
   Withdraw an IPv6 address from an abstract IP address.
   
   An IPv6 address as a pointer.
   Parameters
   _addr :  IP address. 
   Returns
   IPv6 address as a pointer.                            */
#define CM_IPADDR_GET6(_addr) (_addr).addr.v6

/*@@CM_IPADDR_ASSIGN4
   Description
   Set an abstract IP address as IPv4 address.
   Parameters
   _addr :  Pointer to an abstract IP address.
   _ip4 :   IPv4 address to assign to the abstract IP address.
   Returns
   None                                                        */
#define CM_IPADDR_ASSIGN4(_addr, _ip4) \
    {(_addr).version = CM_IPADDR_IPV4; (_addr).addr.v4 = _ip4;}

/*@@CM_IPADDR_ASSIGN6
   Description
   Set an abstract IP address as IPv6 address.
   Parameters
   _addr :  Pointer to an abstract IP address.
   _ip4 :   Pointer to IPv6 address to assign to the abstract IP
            address.
   Returns
   None                                                          */
#define CM_IPADDR_ASSIGN6(_addr, _ip6) \
    {(_addr).version = CM_IPADDR_IPV6; syMemcpy((_addr).addr.v6, _ip6, sizeof(NQ_IPADDRESS6));}

/*@@CM_IPADDR_EQUAL4
   Description
   Compare an abstract IP address with an IPv4 address.
   
   To match, he abstract address should be of the IPv4 type and
   its address value should match the IPv4 address.
   Parameters
   _addr :  Pointer to the abstract IP address.
   _ip4 :   IPv4 address to compare with.
   Returns
   <i>TRUE</i> when abstract IP address is the same as the IPv4
   address, <i>FALSE</i> otherwise.                             */
#define CM_IPADDR_EQUAL4(_addr, _other) \
    ((_addr).version == CM_IPADDR_IPV4 && (_addr).addr.v4 == _other)

/*@@CM_IPADDR_EQUAL6
   Description
   Compare an abstract IP address with an IPv6 address.
   
   To match, the abstract address should be of the IPv6 type and
   its address value should match the IPv6 address.
   Parameters
   _addr :  Pointer to the abstract IP address.
   _ip6 :   Pointer to the IPv6 address to compare with.
   Returns
   <i>TRUE</i> when abstract IP address is the same as the IPv6
   address, <i>FALSE</i> otherwise.                              */
#define CM_IPADDR_EQUAL6(_addr, _other) \
    ((_addr).version == CM_IPADDR_IPV6 && \
    (syMemcmp((_addr).addr.v6, _other, sizeof(NQ_IPADDRESS6)) == 0))

/*@@INT_TO_ADDR4
   Description
   Generate INT to 4 INTs = IP address.

   Parameters
   _addr :   First IP address v4.
   Returns
   None                                                  */
#define INT_TO_ADDR4(_addr) \
(_addr & 0xFF), \
(_addr >> 8 & 0xFF), \
(_addr >> 16 & 0xFF), \
(_addr >> 24 & 0xFF)

#else /* UD_NQ_USETRANSPORTIPV6 */

#define CM_IPADDR_MAXLEN 16                 /* max length of textual representation + trailing null */
#define CM_IP4ADDR_MAXLEN CM_IPADDR_MAXLEN  /* alignment to UD_NQ_USETRANSPORTIPV6 section */
#define CM_IPADDR_ZERO {CM_IPADDR_IPV4, {SY_ZEROIP}}
#define CM_IPADDR_ANY4 {CM_IPADDR_IPV4, {SY_ANYIP}}
#define CM_IPADDR_LOCAL {CM_IPADDR_IPV4, {SY_LOCALHOSTIP}}

/*@@CM_IPADDR_VERSION
   Description
   Get IP address type.

   This call allows determining the concrete type of an abstract
   IP address.
   Parameters
   _addr :  IP address
   Returns
   CM_IPADDR_IPV4 or CM_IPADDR_IPV6 (see <link Other Constants>). */
#define CM_IPADDR_VERSION(_addr)       (_addr).version
#define CM_IPADDR_SIZE(_addr)          sizeof(NQ_IPADDRESS4)
#define CM_IPADDR_GET4(_addr)          (_addr).addr.v4
#define CM_IPADDR_ASSIGN4(_addr, _ip4)  {(_addr).version = CM_IPADDR_IPV4; (_addr).addr.v4 = _ip4;}
#define CM_IPADDR_EQUAL4(_addr, _other) ((_addr).version == CM_IPADDR_IPV4 && (_addr).addr.v4 == _other)
#define CM_IPADDR_EQUAL(_addr, _other)  ((_addr).version == (_other).version && ((_addr).addr.v4 == (_other).addr.v4))

#endif /* UD_NQ_USETRANSPORTIPV6 */

#ifdef CM_NQ_STORAGE
#ifdef UD_NQ_USETRANSPORTIPVR

/*@@SY_ANYIPVR
   Description
   Definition of "loopback address" to use for socket binding.
   This may be different for different environments. The
   standard value is all zeroes.                               */
#define SY_ANYIPVR       {0x0000, 0x00, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000}

/*@@
 Description
 Definition of local host IP. */
#define SY_LOCALHOSTIPVR {0x0000, 0x00, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000}

#define CM_IPADDR_ANYVR   {CM_IPADDR_IPVR, {{0}}}
#define CM_IPADDR_LOCALVR {CM_IPADDR_IPVR, {{0}}}

/*@@CM_IPADDR_GETVR
   Description
   Withdraw an IPv6 address from an abstract IP address.

   An IPv6 address as a pointer.
   Parameters
   _addr :  IP address.
   Returns
   IPv6 address as a pointer.                            */
#define CM_IPADDR_GETVR(_addr) (_addr).addr.vr

/*@@CM_IPADDR_ASSIGNVR
   Description
   Set an abstract IP address as IPv6 address.
   Parameters
   _addr :  Pointer to an abstract IP address.
   _ipvr :  Pointer to IPvr address to assign to the abstract IP
            address.
   Returns
   None                                                          */
#define CM_IPADDR_ASSIGNVR(_addr, _ipvr) \
    {(_addr).version = CM_IPADDR_IPVR; (_addr).addr.vr.id = (_ipvr).id; \
    (_ipvr).addr.vr.version == CM_IPADDR_IPV6 ? CM_IPADDR_ASSIGN6((_addr).addr.vr, (_ipvr).addr.v6)  \
    : CM_IPADDR_ASSIGN4((_addr).addr.vr, (_ipvr).addr.v4); }

/*@@CM_IPADDR_EQUALVR
   Description
   Compare an abstract IP address with an IPv6 address.

   To match, the abstract address should be of the IPv6 type and
   its address value should match the IPv6 address.
   Parameters
   _addr :  Pointer to the abstract IP address.
   _ip6 :   Pointer to the IPv6 address to compare with.
   Returns
   <i>TRUE</i> when abstract IP address is the same as the IPv6
   address, <i>FALSE</i> otherwise.                              */
#define CM_IPADDR_EQUALVR(_addr, _other) \
    ((_addr).version == (_other).version && (_addr).version == CM_IPADDR_IPVR && \
    (_addr).addr.vr.id == (_other).addr.vr.id && \
    ((_addr).addr.vr.version == (_other).addr.vr.version && \
    (_addr).addr.vr.version == CM_IPADDR_IPV6 ? CM_IPADDR_EQUAL6((_addr).addr.vr, (_other).addr.vr.addr.v6) : \
    CM_IPADDR_EQUAL4((_addr).addr.vr, (_other).addr.vr.addr.v4)))

#define CM_IPADDR_SIZE(_addr) \
    (((_addr).version == CM_IPADDR_IPV6) ? sizeof(NQ_IPADDRESS6) \
    : (((_addr).version == CM_IPADDR_IPV4) ? sizeof(NQ_IPADDRESS4) \
    : (((_addr).version == CM_IPADDR_IPVR) ? sizeof(NQ_IPADDRESSVR) : 0 )))

/*@@CM_IPADDR_EQUAL
   Description
   Compare two abstract IP addresses.

   Both abstract addresses are equal when they are of the same
   type and their address values are equal.
   Parameters
   _addr :   First IP address.
   _other :  Second IP address.
   Returns
   <i>TRUE</i> when both addresses are equal, <i>FALSE</i>
   otherwise.                                                  */
#define CM_IPADDR_EQUAL(_addr, _other) \
    ((_addr).version == (_other).version && ((_addr).version == CM_IPADDR_IPV4 \
        ? ((_addr).addr.v4 == (_other).addr.v4) \
        : ((_addr).version == CM_IPADDR_IPV6 ? (syMemcmp((_addr).addr.v6, (_other).addr.v6, sizeof(NQ_IPADDRESS6)) == 0) \
        : CM_IPADDR_EQUALVR(_addr, _other))))

#endif /* UD_NQ_USETRANSPORTIPVR */
#endif /* CM_NQ_STORAGE */

#define CM_IPADDR_ZERO4 SY_ZEROIP4

#define CM_MAX_UINT16                       (0xFFFF)
#define CM_MAX_UINT32                       (0xFFFFFFFF)
#define CM_NUMBER_OF_BYTES_IN_MEGABYTES     (1024 * 1024)


#ifndef SY_PACK_ATTR
#define SY_PACK_ATTR
#endif

#ifndef SY_PACK_PREFIX
#define SY_PACK_PREFIX
#endif

/*@@cmAsciiToIp@NQ_CHAR *@NQ_IPADDRESS *
   Description
   Convert ASCII string representation of IP address into an abstract
   IP structure.
   
   This function converts IP address according to its type.
   Parameters
   str :   String containing text representation of IP address.
   addr :  IP address/
   Returns
   <i>NQ_SUCCESS</i> when conversion was done, <i>NQ_FAIL</i>
   otherwise.                                                   */
NQ_STATUS cmAsciiToIp(NQ_CHAR *ascii, NQ_IPADDRESS *ip);

/*@@cmWcharToIp@NQ_WCHAR *@NQ_IPADDRESS *
    Description
    Convert WCHAR string representation of IP address into an abstract
    IP structure.

    This function converts IP address according to its type.
    Parameters
    str :   String containing text representation of IP address.
    addr :  IP address/
    Returns
    <i>NQ_SUCCESS</i> when conversion was done, <i>NQ_FAIL</i>
    otherwise.                                                   */
NQ_STATUS cmWcharToIp(NQ_WCHAR *wchar, NQ_IPADDRESS *ip);

 /*@@cmIpToAscii@NQ_CHAR *@NQ_IPADDRESS *
    Description
    Convert an abstract IP address into printable text
    representation.
    
    This function converts IP address according to its type.
    Parameters
    buffer :  Text buffer of enough size.
    addr :    the IP address to convert.
    Returns
    <i>NQ_SUCCESS</i> when conversion was done, <i>NQ_FAIL</i>
    otherwise                                                  */
NQ_STATUS cmIpToAscii(NQ_CHAR *ascii, const NQ_IPADDRESS *ip);

/*@@cmIsIPv6Literal@NQ_WCHAR *
    Description
    checks if a received string is an ipv6 literal/

    this function checks a string name suffix
    if equal to "ipv6.literal".

    Returns
    <i>TRUE</i> when name is ipv6 literal, <i>FALSE</i>
    otherwise                                                  */
NQ_BOOL cmIsIPv6Literal(NQ_WCHAR *name);

/*@@cmIPDump@NQ_IPADDRESS *
   Description
   Converts IP address to a text representation into an embedded
   buffer. This function converts IP address according to its type.
   
   Since the result resides in a static buffer this function
   should not be used twice as a parameter in a printf() call.
   This function is not thread-safe.
   Parameters
   addr :    the IP address to convert.
   Returns
   Pointer to the string representation of the provided IP.      */
NQ_CHAR * cmIPDump(const NQ_IPADDRESS *ip);

/*@@cmIPDumpV4@NQ_IPADDRESS4
   Description
   Converts IPv4 address to a text representation into an embedded
   buffer.

   Since the result resides in a static buffer this function
   should not be used twice as a parameter in a printf() call.
   This function is not thread-safe.
   Parameters
   addr :    the IPv4 address to convert.
   Returns
   Pointer to the string representation of the provided IP.      */
NQ_CHAR * cmIPDumpV4(NQ_IPADDRESS4 ip);

/*@@cmIPCShareName@void
   Description
   Returns the pointer to 'IPC$' share name, null terminated.

   Parameters
   none
   Returns
   Pointer to the string, null terminated.      */
const NQ_WCHAR * cmIPCShareName(void);

#ifdef UD_NQ_USEIOVECS
typedef struct {
    NQ_IOVECBUF *ioBuf; /* The iovec buffer. */
    NQ_COUNT segment;    /* the data segment where offset resides */
    NQ_COUNT byteOffset;/* byte offset inside relevant segment */
}NQ_IOVecPosition;

typedef struct{
    NQ_IOVEC *origIovec;
    NQ_COUNT origSegmentCount;
    NQ_COUNT reducedSegment; /* segment bytes reduced from */
    NQ_COUNT reducedSegmentNumBytes;
}NQ_IOVecOrigin;

/* Abstract data buffer
   Description
   According to compile-time parameters, this type may evaluate
   to one of the following:
     * Plain buffer. Data is placed as a contiguous array of
       bytes.
     * IOVEC scatter&#45;gather structure.
   NQ manipulates this type the same manner regardless of its
   implementation.                                              */

#define NQ_IOBufPos NQ_IOVecPosition
#define NQ_IOBufState NQ_IOVEC
#define NQ_IOBufOrigin NQ_IOVecOrigin
#else
#define NQ_IOBufPos NQ_BYTE*
#define NQ_IOBufState
#define NQ_IOBufOrigin
#endif

#define NQContext NQ_IOBufPos

/* abstract blob */
typedef struct
{
    NQ_BYTE* data;  /* data pointer */
    NQ_COUNT len;   /* data length */
} CMBlob;

#ifdef UD_NQ_USEIOVECS
/* Description
   A wrapper around IOVEC to represent a message.                 */
typedef struct {
    NQ_IOBufPos data;    /* IOVec buffer or byte *     */
    NQ_COUNT len;       /* total packet length         */
}CMIOVecPosBlob;

#define CMIOBlob CMIOVecPosBlob
#else
#define CMIOBlob CMBlob
#endif

#ifdef UD_NQ_USEIOVECS
NQ_IOVecPosition cmIovecSkipBytes(NQ_IOVecPosition *buffer, NQ_INT skipVal);
NQ_COUNT cmIovecNumByteDiff(const NQ_IOVecPosition *base, const NQ_IOVecPosition *end);
void *cmIovecMemcpyVec2Flat(NQ_BYTE* to, NQ_IOVecPosition from, NQ_COUNT size);
NQ_IOVecPosition cmIovecMemcpyFlat2Vec(NQ_IOVecPosition *toIOBuf, const NQ_BYTE* from, NQ_COUNT numBytes);
NQ_IOVecPosition cmIovecMemcpyVec2Vec(NQ_IOVecPosition *toIOBuf, const NQ_IOVecPosition* fromIOBuf, NQ_COUNT numBytes);
NQ_IOVecPosition cmIovecMemcpyVec2VecBackwards(NQ_IOVecPosition *toIOBuf, const NQ_IOVecPosition* fromIOBuf, NQ_COUNT numBytes);
NQ_INT cmIovecMemcmpVec2Vec(const NQ_IOVecPosition *toIOBuf, const NQ_IOVecPosition* fromIOBuf, NQ_COUNT numBytes);
NQ_INT cmIovecMemcmpVec2Flat(const NQ_IOVecPosition *IOBuf, const NQ_BYTE* buff, NQ_COUNT numBytes);
NQ_BOOL cmIovecMemset(NQ_IOVecPosition *IOBuf, NQ_INT val, NQ_COUNT numBytes);
NQ_COUNT cmIOStrlen(NQ_IOBufPos data);
#define IOBUF_ISNULL(__IOBUFPos) (__IOBUFPos.ioBuf == NULL)
#define IOBUF_POSINIT(__IOBUFPos) __IOBUFPos.ioBuf = NULL; \
                               __IOBUFPos.byteOffset = 0; \
                               __IOBUFPos.segment = 0;
#define IOBUF_BUFINIT(__IOBUF) __IOBUF->iovec = NULL; \
                               __IOBUF->iovSegmentCnt = 0;
#define IOBUF_BUFPTRINIT(__IOBUF) __IOBUF = NULL;
/* create iovec with one segment  */
#define IOBUF_POSCONSTRUCTORINIT(__IOBUFPos)    \
                           NQ_IOVEC ioBufTmp##__IOBUFPos; \
                           NQ_IOVECBUF ioVecBufTmp##__IOBUFPos;
#define IOBUF_POSCONSTRUCTOR(__IOBUFPos, __base, __len) \
                           ioBufTmp##__IOBUFPos.iov_base = __base; \
                           ioBufTmp##__IOBUFPos.iov_len = __len; \
                           ioVecBufTmp##__IOBUFPos.iovec = &ioBufTmp##__IOBUFPos; \
                           ioVecBufTmp##__IOBUFPos.iovSegmentCnt = 1; \
                           __IOBUFPos.byteOffset = 0; \
                           __IOBUFPos.segment = 0; \
                           __IOBUFPos.ioBuf = &ioVecBufTmp##__IOBUFPos;
#define IOBUF_POSSET(__IOBUFPos, __IOvec) \
                           __IOBUFPos.segment = 0; \
                           __IOBUFPos.byteOffset = 0; \
                           __IOBUFPos.ioBuf = __IOvec;
#define IOBUF_POSRESET(__IOBUFPos) __IOBUFPos.segment = 0; \
                                  __IOBUFPos.byteOffset = 0;
#define IOBUF_POSCONSTRUCTOR_INPLACE(__IOBUFPos, __buf, __size) \
                                    { \
                                         NQ_BYTE* __ptr = __buf; /* can't move __buf pointer which might be later used again. */\
                                         __IOBUFPos.segment = 0; \
                                         __IOBUFPos.byteOffset = 0; \
                                         __IOBUFPos.ioBuf = (NQ_IOVECBUF *)__ptr; \
                                         __ptr += sizeof(NQ_IOVECBUF); \
                                         *((NQ_UINT *)__ptr) = (NQ_UINT)0xAAAAAAAA; \
                                         __ptr += sizeof (NQ_INT); \
                                         __IOBUFPos.ioBuf->iovec = (NQ_IOVEC *) __ptr; \
                                         __IOBUFPos.ioBuf->iovSegmentCnt = 1; \
                                         __ptr += sizeof(NQ_IOVEC); \
                                         *((NQ_UINT *)__ptr) = (NQ_UINT)0xAAAAAAAA; \
                                         __ptr += sizeof (NQ_INT); \
                                         __IOBUFPos.ioBuf->iovec[0].iov_base = __ptr; \
                                         __IOBUFPos.ioBuf->iovec[0].iov_len = __size; \
                                    }
#define IOBUF_POSCONSTRUCT_EXTRASIZE (sizeof(NQ_IOVECBUF) + sizeof (NQ_INT) + sizeof(NQ_IOVEC) + sizeof (NQ_INT))
#define IOBUF_CONSTRUCTORINIT(__IOBUF)    \
                            NQ_IOVEC iovec##__IOBUF; \
                            NQ_IOVECBUF iovecBuf##__IOBUF;
#define IOBUF_CONSTRUCTOR(__IOBUF, __base, __len) \
                           __IOBUF = &iovecBuf##__IOBUF; \
                           __IOBUF->iovec = &iovec##__IOBUF; \
                           __IOBUF->iovSegmentCnt = 1; \
                           __IOBUF->iovec->iov_base = __base; \
                           __IOBUF->iovec->iov_len = __len;
/* move forward or backward */
#define IOBUF_SKIPBYTESEP(__IOBufPos, __ByteCount, __IOVByteCount)    cmIovecSkipBytes(&__IOBufPos, __IOVByteCount) /* seperate values for iovec and non iovec*/
#define IOBUF_SKIPBYTE(__IOBufPos, __ByteCount)    cmIovecSkipBytes(&__IOBufPos, __ByteCount)
#define IOBUF_SKIPBACKBYTE(__IOBufPos, __ByteCount)    cmIovecSkipBytes(&__IOBufPos, -(__ByteCount))
#define IOBUF_MOVEBYTES(__IOBufPos, __ByteCount) cmIovecSkipBytes(&__IOBufPos, __ByteCount);
#define IOBUF_MOVEBASE_BACKWARD(__IOBufPos, __ByteCount) __IOBufPos.ioBuf->iovec[0].iov_base -= __ByteCount; \
                                                         __IOBufPos.ioBuf->iovec[0].iov_len += (NQ_COUNT)__ByteCount;
#define IOBUF_MOVEBASE_FORWARD(__IOBufPos, __ByteCount) __IOBufPos.ioBuf->iovec[0].iov_base += __ByteCount; \
                                                        __IOBufPos.ioBuf->iovec[0].iov_len -= (NQ_COUNT)__ByteCount;

#define IOBUF_GETPOINTERSDIF(__end, __start) cmIovecNumByteDiff(&__start, &__end)
#define IOBUF_ISPTRBIGGER(__ptr,__ref) ((__ptr.segment > __ref.segment) || \
                                        (__ptr.segment == __ref.segment && __ptr.byteOffset > __ref.byteOffset))
#define IOBUF_ISPTRSMALLER(__ptr,__ref) ((__ptr.segment < __ref.segment) || \
                                         (__ptr.segment == __ref.segment && __ptr.byteOffset < __ref.byteOffset))
/* prepare iovec for send. This has to be last operation before sending. later then restore it. */
#define IOBUF_IOVECSET_START(__IOBufPos, __IOBufPosSave) if (NULL != __IOBufPosSave) *(NQ_IOVecPosition *)__IOBufPosSave = __IOBufPos; \
                                                         __IOBufPos.ioBuf->iovec += __IOBufPos.segment; \
                                                             __IOBufPos.ioBuf->iovSegmentCnt -= __IOBufPos.segment; \
                                                             __IOBufPos.ioBuf->iovec[0].iov_base += __IOBufPos.byteOffset; \
                                                             __IOBufPos.ioBuf->iovec[0].iov_len -= __IOBufPos.byteOffset;
#define IOBUF_IOVEC_RESTORE(__IOBufPos, __IOBufRestore) if (NULL != __IOBufRestore) __IOBufPos = *(NQ_IOVecPosition *) __IOBufRestore; \
                                                                __IOBufPos.ioBuf->iovec[0].iov_base -= __IOBufPos.byteOffset; \
                                                                __IOBufPos.ioBuf->iovec[0].iov_len += __IOBufPos.byteOffset; \
                                                                __IOBufPos.ioBuf->iovec -= __IOBufPos.segment; \
                                                                __IOBufPos.ioBuf->iovSegmentCnt += __IOBufPos.segment;
#define IOBUF_SETORIGIN(__IOBuf, __iovecOrigin, __bufState) \
                                                __bufState = __IOBuf->iovec[0]; \
                                                __iovecOrigin.origIovec = __IOBuf->iovec; \
                                                __iovecOrigin.origSegmentCount = __IOBuf->iovSegmentCnt; \
                                                __iovecOrigin.reducedSegment = 0;  \
                                                __iovecOrigin.reducedSegmentNumBytes = __IOBuf->iovec[0].iov_len; /* actually not reduced */
#define IOBUF_PREPARELENGTH(__IOBufToSend, __iovecOrigin, __bufState, __packetLen) \
                                                __iovecOrigin.origIovec = __IOBufToSend->iovec; \
                                                __iovecOrigin.origSegmentCount = __IOBufToSend->iovSegmentCnt; \
                                                __bufState = __IOBufToSend->iovec[0]; \
                                                { /* calculate where actual length should cut segments */ \
                                                    NQ_COUNT lenToSend = __packetLen; \
                                                    __IOBufToSend->iovSegmentCnt = 1; \
                                                    while (lenToSend > __IOBufToSend->iovec[__IOBufToSend->iovSegmentCnt - 1].iov_len) \
                                                    { \
                                                        lenToSend -= __IOBufToSend->iovec[__IOBufToSend->iovSegmentCnt - 1].iov_len; \
                                                        if (++__IOBufToSend->iovSegmentCnt == __iovecOrigin.origSegmentCount) break;  /* shouldn't happen */ \
                                                    } \
                                                    __iovecOrigin.reducedSegment = __IOBufToSend->iovSegmentCnt - 1; /* this segment is cut */ \
                                                    if (lenToSend > __IOBufToSend->iovec[__IOBufToSend->iovSegmentCnt - 1].iov_len) syAssert(FALSE); \
                                                    __iovecOrigin.reducedSegmentNumBytes = __IOBufToSend->iovec[__IOBufToSend->iovSegmentCnt - 1].iov_len; \
                                                    __IOBufToSend->iovec[__IOBufToSend->iovSegmentCnt - 1].iov_len = lenToSend; \
                                                }
#define IOBUF_IOVECBUF_ADVANCE_SAVE(__ioBuf, __numSentBytes, __iovecState) \
                                                     if (__numSentBytes > __ioBuf->iovec[0].iov_len) /* skip segment required */\
                                                     { \
                                                         if (__ioBuf->iovSegmentCnt <= 0) syAssert (FALSE); \
                                                         __numSentBytes -= __ioBuf->iovec[0].iov_len; /* these bytes were sent */\
                                                         __ioBuf->iovec[0].iov_len  = __iovecState.iov_len; \
                                                         __ioBuf->iovec[0].iov_base = __iovecState.iov_base; \
                                                         ++__ioBuf->iovec; /* advance iovec */ \
                                                         --__ioBuf->iovSegmentCnt; \
                                                         while(__numSentBytes > __ioBuf->iovec[0].iov_len) \
                                                         { \
                                                             if (__ioBuf->iovSegmentCnt <= 0) syAssert (FALSE); \
                                                             __numSentBytes -= __ioBuf->iovec[0].iov_len; \
                                                             ++__ioBuf->iovec; /* advance iovec */ \
                                                             --__ioBuf->iovSegmentCnt; \
                                                         } \
                                                         __iovecState.iov_len  = __ioBuf->iovec[0].iov_len; \
                                                         __iovecState.iov_base = __ioBuf->iovec[0].iov_base; \
                                                     } \
                                                     __ioBuf->iovec[0].iov_len -= __numSentBytes; \
                                                     __ioBuf->iovec[0].iov_base += __numSentBytes;
#define IOBUF_IOVECBUF_RESTORE(__ioBuf, __iovecState, __iovecOrigin) \
                                                    __ioBuf->iovec[0].iov_len  = __iovecState.iov_len; \
                                                    __ioBuf->iovec[0].iov_base = __iovecState.iov_base; \
                                                    __ioBuf->iovSegmentCnt = __iovecOrigin.origSegmentCount; \
                                                    __ioBuf->iovec = __iovecOrigin.origIovec; \
                                                    __ioBuf->iovec[__iovecOrigin.reducedSegment].iov_len = __iovecOrigin.reducedSegmentNumBytes;
#define IOBUF_GETBUFPTR(_IOBufPos) _IOBufPos.ioBuf
#define IOBUF_GETBUF(_IOBufPos) *_IOBufPos.ioBuf
#define IOBUF_GETBYTEPTR(_IOBufPos) (_IOBufPos.ioBuf->iovec[_IOBufPos.segment].iov_base + _IOBufPos.byteOffset)
#define IOBUF_GETSTARTPTR(_IOBufPos) (_IOBufPos.ioBuf->iovec[_IOBufPos.segment].iov_base)
#define IOBUF_IOVEC_GETBYTEPTR(_IOBuf) (_IOBuf->iovec[0].iov_base)
/* make sure legal iovec */
#define ASSERT_IOBUF(__IOBufPos) \
                    if (NULL == __IOBufPos) {\
                        LOGERR(CM_TRC_LEVEL_ERROR, "IOVec buffer overflow."); \
                        exit(1); \
                    }
#define ASSERT_IOBUFPOSPTR(__IOBufPos) syAssert(__IOBufPos->segment <= __IOBufPos->ioBuf->iovSegmentCnt && __IOBufPos->byteOffset <= __IOBufPos->ioBuf->iovec[__IOBufPos->segment].iov_len)
/* can use as flat buffer on current pinter and size  */
#define IOBUF_BUFFER_CANUSEFLAT_ASSERT(_IOBufPos, __size) if ((_IOBufPos.ioBuf->iovec[_IOBufPos.segment].iov_len - _IOBufPos.byteOffset) < __size) \
                                                          { \
                                                            syPrintf("len: %d, offset: %d, size: %d\n", _IOBufPos.ioBuf->iovec[_IOBufPos.segment].iov_len,_IOBufPos.byteOffset,__size); \
                                                            syAssert(FALSE); \
                                                          }
#define IOBUF_BUF_CANUSEFLAT_ASSERT(_IOBuf, __size) if (_IOBuf->iovec[0].iov_len < __size) \
                                                          { \
                                                            syPrintf("len: %d, size: %d\n", _IOBuf->iovec[0].iov_len,__size); \
                                                            syAssert(FALSE); \
                                                          }
#define IOBUF_BUFFER_CANUSEFLAT(_IOBufPos, __size) (((NQ_INT)_IOBufPos.ioBuf->iovec[_IOBufPos.segment].iov_len - (NQ_INT)_IOBufPos.byteOffset) >= __size)
#define IOBUF_BUFFER_ISINSEGMENT(_IOBufPos, __ptr) ((__ptr >= _IOBufPos.ioBuf->iovec[_IOBufPos.segment].iov_base) && \
                                                    (__ptr < (_IOBufPos.ioBuf->iovec[_IOBufPos.segment].iov_base +  \
                                                              _IOBufPos.ioBuf->iovec[_IOBufPos.segment].iov_len)))
/* memory macros. V=vec F=flat*/
#define IOBUF_MEMCPY_V2F(__to, __from, __size ) cmIovecMemcpyVec2Flat(__to, __from, __size)
#define IOBUF_MEMCPY_F2V(__to, __from, __size ) cmIovecMemcpyFlat2Vec(&__to, __from, __size)
#define IOBUF_MEMCPY_V2V(__to, __from, __size ) cmIovecMemcpyVec2Vec(&__to, &__from, __size)
#define IOBUF_MEMCPYBACK_V2V(__to, __from, __size ) cmIovecMemcpyVec2VecBackwards(&__to, &__from, __size)
#define IOBUF_MEMCMP_V2V(__IOBUFPos1, __IOBUFPos2, __size ) cmIovecMemcmpVec2Vec(__IOBUFPos1, __IOBUFPos2, __size)
#define IOBUF_MEMCMP_V2F(__IOBUFPos, __FlatBuf, __size ) cmIovecMemcmpVec2Flat(&__IOBUFPos, __FlatBuf, __size)
#define IOBUF_MEMSET(__IOBufPos, __val, __numBytes) cmIovecMemset(&__IOBufPos, __val, __numBytes)
#define IOBUF_WSTRLEN(__IOBufPos) cmIOWStrlen(__IOBufPos)
#define IOBUF_STRLEN(__IOBufPos) cmIOStrlen(__IOBufPos)
#define IOBUF_WSTRCPY_V2F(__IOBufPosFrom, __WCharTo) cmIOWStrcpyV2F(__WCharTo, __IOBufPosFrom)
#else
#define IOBUF_ISNULL(__IOBUFPos) (__IOBUFPos == NULL)
#define IOBUF_POSINIT(__IOBUFPos) (__IOBUFPos = NULL);
#define IOBUF_BUFINIT(__IOBUF) __IOBUF = NULL;
#define IOBUF_BUFPTRINIT(__IOBUF) __IOBUF = NULL;
#define ASSERT_IOBUF(__IOBufPos)
#define ASSERT_IOBUFPOSPTR(__IOBufPos)
#define IOBUF_POSCONSTRUCTORINIT(__IOBUFPos)
#define IOBUF_POSCONSTRUCTOR(__IOBUFPos, __base, __len)    __IOBUFPos = __base;
#define IOBUF_CONSTRUCTORINIT(__IOBUF)
#define IOBUF_CONSTRUCTOR(__IOBUF, __base, __len)     __IOBUF = __base;
#define IOBUF_POSSET(__IOBUFPos, __IOvec) __IOBUFPos = __IOvec;
#define IOBUF_POSRESET(__IOBUFPos)
#define IOBUF_POSCONSTRUCTOR_INPLACE(__IOBUFPos, __buf, __size) __IOBUFPos = __buf;
#define IOBUF_POSCONSTRUCT_EXTRASIZE 0
/* move forward or backward */
#define IOBUF_SKIPBYTESEP(__IOBufPos, __ByteCount, __IOVByteCount)    (__IOBufPos + __ByteCount)
#define IOBUF_SKIPBYTE(__IOBufPos, __ByteCount)    (__IOBufPos + __ByteCount)
#define IOBUF_SKIPBACKBYTE(__IOBufPos, __ByteCount)
#define IOBUF_MOVEBYTES(__IOBufPos, __ByteCount) (__IOBufPos += __ByteCount);
#define IOBUF_MOVEBASE_BACKWARD(__IOBufPos, __ByteCount) (__IOBufPos -= __ByteCount)
#define IOBUF_MOVEBASE_FORWARD(__IOBufPos, __ByteCount) (__IOBufPos += __ByteCount)
#define IOBUF_GETPOINTERSDIF(__end, __start)    (__end - __start)
#define IOBUF_ISPTRBIGGER(__ptr,__ref) (__ptr > __ref)
#define IOBUF_ISPTRSMALLER(__ptr,__ref) (__ptr < __ref)
#define IOBUF_IOVECSET_START(__IOBufPos, __IOBufPosSave)
#define IOBUF_IOVEC_RESTORE(__IOBufPos, __IOBufRestore)
#define IOBUF_SETORIGIN(__ioBuf, __iovecOrigin, __bufState)
#define IOBUF_IOVECBUF_ADVANCE_SAVE(__ioBuf, __numBytes, __iovecState) __ioBuf += __numBytes;
#define IOBUF_IOVECBUF_RESTORE(__ioBuf, __iovecState, __iovecOrigin)
#define IOBUF_GETBUFPTR(__IOBufPos)   __IOBufPos
#define IOBUF_PREPARELENGTH(__IOBufToSend, __iovecOrigin, __bufState, __packetLen)
#define IOBUF_GETBUF(__IOBufPos)     __IOBufPos
#define IOBUF_GETBYTEPTR(__IOBufPos) __IOBufPos
#define IOBUF_GETSTARTPTR(__IOBufPos) __IOBufPos
#define IOBUF_IOVEC_GETBYTEPTR(_IOBuf) _IOBuf
/* can use as flat buffer on current segment */
#define IOBUF_BUFFER_CANUSEFLAT_ASSERT(_IOBufPos, __size)
#define IOBUF_BUF_CANUSEFLAT_ASSERT(_IOBuf, __size)
#define IOBUF_BUFFER_CANUSEFLAT(_IOBufPos, __size) TRUE
#define IOBUF_BUFFER_ISINSEGMENT(_IOBufPos, __ptr) TRUE
/* memcpy macros. V=vec F=flat*/
#define IOBUF_MEMCPY_V2F(__to, __from, __size ) syMemcpy(__to, __from, __size)
#define IOBUF_MEMCPY_F2V(__to, __from, __size ) syMemcpy(__to, __from, __size)
#define IOBUF_MEMCPY_V2V(__to, __from, __size ) syMemcpy(__to, __from, __size)
#define IOBUF_MEMCMP_V2V(__to, __from, __size ) syMemcmp(__to, __from, __size)
#define IOBUF_MEMCMP_V2F(__to, __from, __size ) syMemcmp(__to, __from, __size)
#define IOBUF_MEMSET(__IOBufPos, __val, __numBytes) syMemset(__IOBufPos, __val, __numBytes)
#define IOBUF_WSTRLEN(__IOBufPos) cmWStrlen((NQ_WCHAR*)__IOBufPos)
#define IOBUF_STRLEN(__IOBufPos) syStrlen((NQ_CHAR*)__IOBufPos)
#define IOBUF_WSTRCPY_V2F(__IOBufPosFrom, __WCharTo) cmWStrcpy(__WCharTo, __IOBufPosFrom)
#endif

/**** Virtual "static" definition.
 * Virtual static always designates a function-scope
 * variable. Depending on project decision it may be
 * compiled as either automatic or static.
 * Static variables save stack room but require more
 * RAM.
 */

#ifdef SY_BIGSTACK
#define NQ_STATIC                 /* defined as automatic */
#else
#define NQ_STATIC    static       /* defined as static */
#endif


#define CM_ARRAY_SIZE(a) (sizeof(a)/sizeof(a[0]))

NQ_CHAR * cmFileIdDump(const NQ_BYTE id[16]);

/* create random byte sequence */
void
cmCreateRandomByteSequence(
    NQ_BYTE *buffer,
    NQ_UINT32 size
    );

/* bind socket to first available dynamic port local host */
NQ_PORT cmDynamicPortBindSocket(SYSocketHandle sock, NQ_BOOL doReuseAddress);

/* release dynamic port (to be called when socket is closed) */
void cmDynamicPortRelease(NQ_PORT dynPort);

#endif  /* _CMCOMMON_H_ */

