/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Common Buffer operations
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 *************************************************************************/

#include "cmbufman.h"
#include "cmrepository.h"

#define BUFREPOSITORY_SIZE_MEDIUM       (UD_NS_BUFFERSIZE + 52 + 100)       /* 64K + transform header + space for command structure */
#define BUFREPOSITORY_SIZE_LARGE        (1048576 + 52 + 100)                /* 1MB + transform header + space for command structure */

/* Do not allocate large buffer in case it is smaller than medium */
#if (BUFREPOSITORY_SIZE_LARGE > BUFREPOSITORY_SIZE_MEDIUM)
#define BUFREPOSITORY_USE_LARGE_BUFFERS /* use buffers of size BUFREPOSITORY_SIZE_LARGE */
#endif /* (BUFREPOSITORY_SIZE_LARGE > BUFREPOSITORY_SIZE_MEDIUM) */

#define BUFREPOSITORY_NUM_OF_BUFFERS    10


static NQ_BOOL isModuleInitialized = FALSE;
static CMRepository BufRepoMedium;
#if (defined(CM_NQ_STORAGE) && defined(UD_NQ_INCLUDESMB3)) || (!defined(CM_NQ_STORAGE) && defined(BUFREPOSITORY_USE_LARGE_BUFFERS))
static CMRepository BufRepoLarge;
#endif /* (defined(CM_NQ_STORAGE) && defined(UD_NQ_INCLUDESMB3)) || (!defined(CM_NQ_STORAGE) && defined(BUFREPOSITORY_USE_LARGE_BUFFERS)) */


typedef struct
{
    CMItem * item;
    CMRepository * rep;
}
Buffer;


/* -- API Functions */

NQ_BOOL cmBufManStart(void)
{
    NQ_COUNT numBuffers = BUFREPOSITORY_NUM_OF_BUFFERS;

    if (isModuleInitialized)
        return TRUE;

#ifdef CM_NQ_STORAGE
    numBuffers = UD_CS_NUMBER_CONNECTIONS / 2 > BUFREPOSITORY_NUM_OF_BUFFERS ? UD_CS_NUMBER_CONNECTIONS / 2 : BUFREPOSITORY_NUM_OF_BUFFERS;
#endif /* CM_NQ_STORAGE */
    cmRepositoryInit(&BufRepoMedium, 0, NULL, NULL, "bufMan_medium");
    cmRepositoryItemPoolAlloc(&BufRepoMedium, numBuffers, sizeof(Buffer) + BUFREPOSITORY_SIZE_MEDIUM);
#if (defined(CM_NQ_STORAGE) && defined(UD_NQ_INCLUDESMB3)) || (!defined(CM_NQ_STORAGE) && defined(BUFREPOSITORY_USE_LARGE_BUFFERS))
    cmRepositoryInit(&BufRepoLarge, 0, NULL, NULL, "bufMan_large");
    cmRepositoryItemPoolAlloc(&BufRepoLarge, numBuffers, sizeof(Buffer) + BUFREPOSITORY_SIZE_LARGE);
#endif /* (defined(CM_NQ_STORAGE) && defined(UD_NQ_INCLUDESMB3)) || (!defined(CM_NQ_STORAGE) && defined(BUFREPOSITORY_USE_LARGE_BUFFERS)) */
    isModuleInitialized = TRUE;
    return TRUE;
}

void cmBufManShutdown(void)
{
    if (isModuleInitialized)
    {
        cmRepositoryShutdown(&BufRepoMedium);

#if (defined(CM_NQ_STORAGE) && defined(UD_NQ_INCLUDESMB3)) || (!defined(CM_NQ_STORAGE) && defined(BUFREPOSITORY_USE_LARGE_BUFFERS))
        cmRepositoryShutdown(&BufRepoLarge);
#endif /* (defined(CM_NQ_STORAGE) && defined(UD_NQ_INCLUDESMB3)) || (!defined(CM_NQ_STORAGE) && defined(BUFREPOSITORY_USE_LARGE_BUFFERS)) */
        isModuleInitialized = FALSE;
    }
}

NQ_BYTE * cmBufManTake(NQ_COUNT size)
{
    Buffer *pRepBuffer;
    NQ_BYTE *buf = NULL;
    
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "size:%d", size);

    if (size <= BUFREPOSITORY_SIZE_MEDIUM)
    {
        pRepBuffer = (Buffer *)cmRepositoryGetNewItem(&BufRepoMedium);
        if (NULL == pRepBuffer)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Memory allocation failed BufRepoMedium repository.");
            goto Exit;
        }
        pRepBuffer->rep = &BufRepoMedium;
        buf = (NQ_BYTE *)(++pRepBuffer);
    }
#if (defined(CM_NQ_STORAGE) && defined(UD_NQ_INCLUDESMB3)) || (!defined(CM_NQ_STORAGE) && defined(BUFREPOSITORY_USE_LARGE_BUFFERS))
    else if (size <= BUFREPOSITORY_SIZE_LARGE)
    {
        pRepBuffer = (Buffer *)cmRepositoryGetNewItem(&BufRepoLarge);
        if (NULL == pRepBuffer)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Memory allocation failed BufRepoLarge repository.");
            goto Exit;
        }
        pRepBuffer->rep = &BufRepoLarge;
        buf = (NQ_BYTE *)(++pRepBuffer);
    }
#endif /* (defined(CM_NQ_STORAGE) && defined(UD_NQ_INCLUDESMB3)) || (!defined(CM_NQ_STORAGE) && defined(BUFREPOSITORY_USE_LARGE_BUFFERS)) */
    else
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Size not allowed:%d", size);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "buf:%p", buf);

Exit:
    return buf;
}

NQ_IOBufPos cmIOBufManTake(NQ_COUNT size)
{
    NQ_IOBufPos ioBufPos;
    NQ_BYTE * pBuf = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "size:%d", size);

    pBuf = cmBufManTake(size + (NQ_COUNT)IOBUF_POSCONSTRUCT_EXTRASIZE);

    IOBUF_POSCONSTRUCTOR_INPLACE(ioBufPos, pBuf, size)

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "buf:%p", pBuf);

    return ioBufPos;
}

void cmBufManGive(NQ_BYTE * buffer)
{
    Buffer *pRepBuffer;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buf:%p", buffer);

    if (NULL != buffer)
    {
        pRepBuffer = (Buffer *)buffer - 1;
        cmRepositoryReleaseItem(pRepBuffer->rep, (CMItem *)pRepBuffer);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

void cmIOBufManGive(NQ_IOBufPos ioBufPos)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buf:%p", IOBUF_GETBUFPTR(ioBufPos));

    if (!IOBUF_ISNULL(ioBufPos))
    {
        cmBufManGive((NQ_BYTE*)IOBUF_GETBUFPTR(ioBufPos));
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

#ifdef NQDEBUG
/* Description
   This function prints internal information about buffer management.
   Returns
   None                                                                       */
void cmBufManDump(void)
{
    
}
#endif /* NQDEBUG */

