/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : SHA256 encryption implementation
 *--------------------------------------------------------------------
 * MODULE        : Auth - AM
 * DEPENDENCIES  : None
 ********************************************************************/
#include "cmapi.h"

#ifdef UD_NQ_INCLUDESMB2
#include "cmsmb2.h"

#define SHA256_DIGEST_SIZE 32
#define SHA256_BLOCK_SIZE  64

#define SHFR(x, n)    (x >> n)
#define ROTR(x, n)   ((x >> n) | (x << ((sizeof(x) << 3) - n)))
#define ROTL(x, n)   ((x << n) | (x >> ((sizeof(x) << 3) - n)))
#define CH(x, y, z)  ((x & y) ^ (~x & z))
#define MAJ(x, y, z) ((x & y) ^ (x & z) ^ (y & z))

#define SHA256_F1(x) (ROTR(x,  2) ^ ROTR(x, 13) ^ ROTR(x, 22))
#define SHA256_F2(x) (ROTR(x,  6) ^ ROTR(x, 11) ^ ROTR(x, 25))
#define SHA256_F3(x) (ROTR(x,  7) ^ ROTR(x, 18) ^ SHFR(x,  3))
#define SHA256_F4(x) (ROTR(x, 17) ^ ROTR(x, 19) ^ SHFR(x, 10))

#define UNPACK32(x, str)                        \
{                                               \
    *((str) + 3) = (NQ_BYTE) ((x)      );   \
    *((str) + 2) = (NQ_BYTE) ((x) >>  8);   \
    *((str) + 1) = (NQ_BYTE) ((x) >> 16);   \
    *((str) + 0) = (NQ_BYTE) ((x) >> 24);   \
}

#define PACK32(str, x)                        \
{                                             \
    *(x) =   ((NQ_UINT32) *((str) + 3))   \
    | ((NQ_UINT32) *((str) + 2) <<  8)    \
    | ((NQ_UINT32) *((str) + 1) << 16)    \
    | ((NQ_UINT32) *((str) + 0) << 24);   \
}

NQ_UINT32 sha256_h0[8] = {
    0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a,
    0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19
    };

NQ_UINT32 sha256_k[64] = {
    0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5,
    0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
    0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3,
    0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
    0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc,
    0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
    0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7,
    0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
    0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13,
    0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
    0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3,
    0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
    0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5,
    0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
    0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208,
    0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2
    };

typedef struct {
    NQ_UINT tot_len;
    NQ_UINT len;
    NQ_BYTE block[2 * SHA256_BLOCK_SIZE];
    NQ_UINT32 h[8];
} sha256_ctx;

static void sha256_init(sha256_ctx *ctx)
{
    NQ_UINT i;

    for (i = 0; i < 8; i++)
    {
        ctx->h[i] = sha256_h0[i];
    }

    ctx->len = 0;
    ctx->tot_len = 0;
}

static void sha256_transf(sha256_ctx *ctx, const NQ_IOBufPos message, NQ_UINT block_nb)
{
    NQ_UINT32 w[64];
    NQ_UINT32 wv[8];
    NQ_UINT32 t1, t2;
    NQ_IOBufPos sub_block;
    NQ_UINT i, j;

    for (i = 0; i < block_nb; ++i)
    {
        sub_block = (NQ_IOBufPos)message;
        IOBUF_MOVEBYTES(sub_block, (NQ_INT)(i << 6));

        for (j = 0; j < 16; j++) {
            IOBUF_BUFFER_CANUSEFLAT_ASSERT(sub_block, sizeof(NQ_INT32));
            PACK32(IOBUF_GETBYTEPTR(sub_block), &w[j]);
            IOBUF_MOVEBYTES(sub_block, (NQ_INT)(1 << 2));
        }
        IOBUF_MOVEBYTES(sub_block, (NQ_INT)(-(1 << 2)));

        for (j = 16; j < 64; j++) {
            w[j] =  SHA256_F4(w[j - 2]) + w[j - 7] + SHA256_F3(w[j - 15]) + w[j - 16];
        }

        for (j = 0; j < 8; j++) {
            wv[j] = ctx->h[j];
        }

        for (j = 0; j < 64; j++) {
            t1 = wv[7] + SHA256_F2(wv[4]) + CH(wv[4], wv[5], wv[6])
                + sha256_k[j] + w[j];
            t2 = SHA256_F1(wv[0]) + MAJ(wv[0], wv[1], wv[2]);
            wv[7] = wv[6];
            wv[6] = wv[5];
            wv[5] = wv[4];
            wv[4] = wv[3] + t1;
            wv[3] = wv[2];
            wv[2] = wv[1];
            wv[1] = wv[0];
            wv[0] = t1 + t2;
        }

        for (j = 0; j < 8; j++) {
            ctx->h[j] += wv[j];
        }
    }
}

static void sha256_update(sha256_ctx *ctx, const NQ_IOBufPos message, NQ_UINT len)
{
    NQ_UINT block_nb, new_len, copy_len, tmp_len;
    NQ_IOBufPos shifted_message;
    NQ_IOBufPos ctxBlock;
    IOBUF_POSCONSTRUCTORINIT(ctxBlock);

    tmp_len = SHA256_BLOCK_SIZE - ctx->len;
    copy_len = len < tmp_len ? len : tmp_len;

    /* copy start of message to reamining length of previous message (saved in ctx->block) */
    IOBUF_MEMCPY_V2F(&ctx->block[ctx->len], message, copy_len);
    ctx->len += len;

    if (ctx->len < SHA256_BLOCK_SIZE) {
        /* if all this bytes don't add up to a full block. return */
        return;
    }

    new_len = len - copy_len;
    /* calcualte how many blocks we have in the remaining bytes of message that arrived */
    block_nb = new_len / SHA256_BLOCK_SIZE;

    /* shift message according to number of bytes we copied to previous block */
    shifted_message = (NQ_IOBufPos)message;
    IOBUF_MOVEBYTES(shifted_message, (NQ_INT)copy_len);

    /* hash thet first block */
    IOBUF_POSCONSTRUCTOR(ctxBlock, ctx->block, SHA256_BLOCK_SIZE);
    sha256_transf(ctx, ctxBlock, 1);

    /* hash rest of new message */
    sha256_transf(ctx, shifted_message, block_nb);

    copy_len = new_len % SHA256_BLOCK_SIZE;

    /* copy remaining bytes of new message to ctx->block */
    IOBUF_MEMCPY_V2F(ctx->block, IOBUF_SKIPBYTE(shifted_message , (NQ_INT)(block_nb << 6)), copy_len);

    ctx->len = copy_len;
    ctx->tot_len += (block_nb + 1) << 6;
}

static void sha256_final(sha256_ctx *ctx, NQ_BYTE *digest)
{
    NQ_UINT block_nb, pm_len, len_b, i;
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);

    block_nb = (1 + ((SHA256_BLOCK_SIZE - 9)
                < (ctx->len % SHA256_BLOCK_SIZE)));

    len_b = (ctx->tot_len + ctx->len) << 3;
    pm_len = block_nb << 6;

    syMemset(ctx->block + ctx->len, 0, pm_len - ctx->len);
    ctx->block[ctx->len] = 0x80;
    UNPACK32(len_b, ctx->block + pm_len - 4);

    IOBUF_POSCONSTRUCTOR(bufPos, ctx->block, sizeof(ctx->block))
        sha256_transf(ctx, bufPos, block_nb);

    for (i = 0 ; i < 8; i++) {
        UNPACK32(ctx->h[i], &digest[i << 2]);
    }
}

void sha256Internal(const CMBlob * key, const CMBlob * key1, const CMIOBlob dataFragments[], NQ_COUNT numFragments, NQ_BYTE * buffer, NQ_COUNT bufferSize)
{
    sha256_ctx ctx;
    NQ_COUNT i;

    sha256_init(&ctx);
    for (i = 0; i < numFragments; i++)
    {
        if (!IOBUF_ISNULL(dataFragments[i].data) && (dataFragments[i].len > 0))
        {
            sha256_update(&ctx, dataFragments[i].data, dataFragments[i].len);
        }
    }
    sha256_final(&ctx, buffer);
}


#endif /* UD_NQ_INCLUDESMB2 */
