/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : LM, NTLM hash implementation
 *--------------------------------------------------------------------
 * MODULE        : Auth - AM
 * DEPENDENCIES  : None
 ********************************************************************/

#include "amapiin.h"
#include "cmcrypt.h"

#define CM_CRYPT_ENCLMPWDSIZE    24
#define CM_CRYPT_ENCNTLMPWDSIZE  24
#define CM_CRYPT_ENCLMv2BLIPSIZE 8
#define CM_CRYPT_ENCLMv2HMACSIZE 16
#define CM_CRYPT_ENCLMv2PWDSIZE  24


static void NQ_E_P16(
        NQ_BYTE *p14,
        NQ_BYTE *p16
    )
{
    NQ_BYTE sp8[8] = {0x4b, 0x47, 0x53, 0x21, 0x40, 0x23, 0x24, 0x25};
    amSmbHash(p16, sp8, p14, 1);
    amSmbHash(p16 + 8, sp8, p14 + 7, 1);
}

static void NQ_E_P24(
        NQ_BYTE *p21,
        const NQ_BYTE *c8,
        NQ_BYTE *p24
    )
{
    amSmbHash(p24, c8, p21, 1);
    amSmbHash(p24 + 8, c8, p21 + 7, 1);
    amSmbHash(p24 + 16, c8, p21 + 14, 1);
}

/*
 *====================================================================
 * PURPOSE: Encrypt hashed password
 *--------------------------------------------------------------------
 * PARAMS:  IN     hashed password
 *          IN     key
 *          IN/OUT encrypted password
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmEncryptHashedPassword(
        const NQ_BYTE hashed[16],
        const NQ_BYTE key[8],
        NQ_BYTE encrypted[24]
        )
{
    NQ_BYTE hshPasswdBuf[21];

    syMemset(hshPasswdBuf,cmWChar('\0'),sizeof(hshPasswdBuf));
    syMemcpy(hshPasswdBuf, hashed, 16);
    NQ_E_P24(hshPasswdBuf, key, encrypted);
}

/*
 *====================================================================
 * PURPOSE: Hash plain text password
 *--------------------------------------------------------------------
 * PARAMS:  IN     plain text password
 *          IN/OUT hashed password
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmHashPassword(
        const NQ_BYTE *password,
        NQ_BYTE hashed[16]
          )
{
    NQ_BYTE ptPasswdBuf[15];
    NQ_BYTE hshPasswdBuf[21];
    NQ_INT i;

    syMemset(ptPasswdBuf,cmWChar('\0'), sizeof(ptPasswdBuf));
    syMemset(hshPasswdBuf,cmWChar('\0'), sizeof(hshPasswdBuf));
    syStrncpy((NQ_CHAR *) ptPasswdBuf, (NQ_CHAR *)password, sizeof(ptPasswdBuf) - 1);
    for (i = 0; ptPasswdBuf[i] != 0; i++)
    {
        ptPasswdBuf[i] = (NQ_BYTE)syToupper(ptPasswdBuf[i]);
    }
    NQ_E_P16(ptPasswdBuf, hshPasswdBuf);
    syMemcpy(hashed, hshPasswdBuf, 16);
}

/*
 *====================================================================
 * PURPOSE: Encrypt plain text password
 *--------------------------------------------------------------------
 * PARAMS:  IN     plain text password
 *          IN     key
 *          IN/OUT encrypted password
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */
void cmEncryptPlainTextPassword(
        const NQ_BYTE *password,
        const NQ_BYTE *key,
        NQ_BYTE *encrypted
        )
{
    NQ_BYTE hshPasswdBuf[16];

    cmHashPassword(password, hshPasswdBuf);
    cmEncryptHashedPassword(hshPasswdBuf, key, encrypted);
}

/*
 *====================================================================
 * PURPOSE: encrypt hashed LM password
 *--------------------------------------------------------------------
 * PARAMS:  IN  *hashed* LM password to encrypt
 *          IN  encryption key
 *          IN  buffer where encrypted password will be stored (at least 24 bytes)
 *          OUT encrypted password length
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void cmEncryptLMPassword(
        const NQ_BYTE *key,
        const NQ_BYTE *password,
        NQ_BYTE *encrypted,
        NQ_UINT16 *len
        )
{
    cmEncryptHashedPassword(password, key, encrypted);
    *len = CM_CRYPT_ENCLMPWDSIZE;
}

/*
 *====================================================================
 * PURPOSE: Encrypt hashed NTLM password
 *--------------------------------------------------------------------
 * PARAMS:  IN  encryption key
 *          IN  *hashed* NTLM password to encrypt
 *          OUT buffer where encrypted password will be stored (at least 24 bytes)
 *          OUT encrypted password length
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void cmEncryptNTLMPassword(
        const NQ_BYTE *key,
        const NQ_BYTE *password,
        NQ_BYTE *encrypted,
        NQ_UINT16 *enclen
        )
{
    cmEncryptHashedPassword(password, key, encrypted);
    *enclen = CM_CRYPT_ENCNTLMPWDSIZE;
}

/*
 *====================================================================
 * PURPOSE: Encrypt hashed NTLM password
 *--------------------------------------------------------------------
 * PARAMS:  IN  encryption key
 *          IN/OUT  IN - encrypted password OUT - decrypted password
 *          IN whether to perform MD5 before
 *
 * RETURNS: None
 *
 * NOTES:   password is 516 bytes long. Next 16 bytes are for crypted
 *          digest. After decryption the last four bytes of the password
 *          (from offset 512) contain password length. Decrypted password
 *          starts from byte (512 - length).
 *====================================================================
 */

void
cmDecryptPassword(
    const NQ_BYTE *key,
    NQ_BYTE *password,
    NQ_BOOL doMd5
    )
{
    NQ_BYTE coKey[16];  /* new key */

    if (doMd5)
    {
        CMBlob fragments[2];

        fragments[0].data = password+516;
        fragments[0].len = 16;
        fragments[1].data = (NQ_BYTE *)key;
        fragments[1].len = SMB_SESSIONKEY_LENGTH;
        (cmGetCurrentCrypters()->md5)(NULL, NULL, fragments, 2, coKey, 16);

    }
    else
    {
        syMemset(coKey, 0, sizeof(coKey));
        syMemcpy(coKey, key, SMB_SESSIONKEY_LENGTH);
    }
    cmArcfourCrypt(password, 516, coKey, sizeof(coKey));
}
