/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Authentication internal API
 *--------------------------------------------------------------------
 * MODULE        : Auth - AM
 * DEPENDENCIES  :
 *************************************************************************/

#ifndef _AMAPIIN_H_
#define _AMAPIIN_H_

#include "amapi.h"

/** -- Constants -- */

/* SPNEGO negotiation statuses: */
#define AM_SPNEGO_SUCCESS   0   /* SPNEGO status: security exchange succeeded. */
#define AM_SPNEGO_NONE      1   /* SPNEGO status: security did not start yet. */
#define AM_SPNEGO_FAILED    2   /* SPNEGO status: security exchange failed. For instance - bad format. */
#define AM_SPNEGO_CONTINUE  3   /* SPNEGO status: security exchange should be continued. */
#define AM_SPNEGO_DENIED    4   /* SPNEGO status: security exchange failed because server denied logon. */

/* server parser statuses */
#define AM_STATUS_AUTHENTICATED             1   /* user was authenticated */
#define AM_STATUS_NOT_AUTHENTICATED         2   /* message parsed but user was not authenticated yet */
#define AM_STATUS_MORE_PROCESSING_REQUIRED  3   /* insufficient information, challenge created */
#define AM_STATUS_BAD_FORMAT                10  /* parse error */
#define AM_STATUS_GENERIC                   11  /* generic error */
#define AM_STATUS_INSUFFICIENT_RESOURCES    12  /* out of memory */

/** -- Structures --*/

typedef struct
{
    const NQ_BYTE* pLm;         /* LM blob pointer */
    const NQ_BYTE* pNtlm;       /* NTLM blob pointer */
    NQ_UINT16 lmLen;            /* LM blob length */
    NQ_UINT16 ntlmLen;          /* NTLM blob length */
    NQ_BOOL isLmAuthenticated;  /* whether authentication performed using LM client response */
    NQ_BOOL isNtlmAuthenticated;/* whether authentication performed using NTLM client response */
    NQ_BOOL isKrbAuthenticated; /* whether authentication performed using Kerberos */
    NQ_UINT32 flags;            /* ntlm flags */
}
AMNtlmDescriptor;

/* -- Functions -- */

#ifdef UD_NQ_INCLUDECIFSCLIENT


/* Description
   This function frees memory of a previously allocated key.
   Parameters
   key :  Pointer to a blob with a previously allocated session
          key. This value can be NULL.
   Returns
   None.
   See Also
   <link amSpnegoClientLogon@void *@NQ_WCHAR *@AMCredentials *@NQ_BOOL@CMBlob *@CMBlob *@CMBlob *@AMSpnegoClientExchange, amSpnegoClientLogon()>

   <link amSpnegoGeneratePasswordBlobs@AMCredentials *@NQ_INT@CMBlob *@CMBlob *@CMBlob *@CMBlob *, amSpnegoGeneratePasswordBlobs()>              */
void amSpnegoFreeKey(CMBlob * key);
void amSpnegoFreeIOKey(CMIOBlob * key);

/* Description
   This function generates encrypted passwords for low-security
   logon, when extended security was not negotiated.

   The two blobs will be generated by the crypters by the
   designated <i>level</i>. The list of crypters per level can
   be modified by calling <link amClientDefineLevel@NQ_UINT@NQ_UINT@NQ_UINT@NQ_UINT32, amClientDefineLevel()>.
   Parameters
   credentials :    Pointer to user logon credentials.
   level :          Security level. This value should be greater
                    or equal to zero and it should not exceed the
                    maximum security level as defined in <link AM_MAXSECURITYLEVEL>.
                    An illegal value is replaced with <link AM_MAXSECURITYLEVEL>.
   pass1 :          Pointer to the first blob. Upon successful
                    return, NQ places the first blob into this
                    structure. It is caller's responsibility to
                    free this blob data.
   pass2 :          Pointer to the second blob. Upon successful
                    return, NQ places the first blob into this
                    structure. It is caller's responsibility to
                    free this blob data.
   sessionKey :     On entry, this blob should contain session
                    key provided by server. Upon successful
                    return, NQ places message signing challenge
                    (aka \- response) key into this structure. It
                    is caller's responsibility to free this blob
                    data.
   macSessionKey :  Pointer to the signature key to be created.
                    Upon successful return, NQ places mac session
                    key (aka message signing key) into this
                    structure. It is caller's responsibility to
                    free this blob data.
   Returns
   One of the constants defined in this module.                                                                */
NQ_STATUS amSpnegoGeneratePasswordBlobs(const AMCredentials * credentials, NQ_INT level, CMBlob *pass1, CMBlob * pass2, CMBlob * sessionKey, CMBlob * macSessionKey);

/* Description
   AM module uses external function for SPNEGO exchange. This
   prototype designates a function which performs one step of
   SPNEGO exchange.

   It should transmit a blob to the authenticating server and
   receive a response blob from it.
   Parameters
   context :        Pointer to SPNEGO context.
   pass1 :          Pointer to the first blob. Upon successful
                    return, NQ places the first blob into this
                    structure. It is caller's responsibility to
                    free this blob data.
   pass2 :          Pointer to the second blob. Upon successful
                    return, NQ places the first blob into this
                    structure. It is caller's responsibility to
                    free this blob data.
   sessionKey :     Pointer to the session key. On entry, this
                    blob should contain session key provided by
                    server. Upon successful return, NQ places
                    session key into this structure. It is
                    caller's responsibility to free this blob
                    data.
   macSessionKey :  Pointer to the signature key to be created.
                    Upon successful return, NQ places session key
                    into this structure. It is caller's
                    responsibility to free this blob data.
   Returns
   One of the constants defined in this module.                   */
typedef NQ_STATUS (* AMSpnegoClientExchange)(void * context, const CMBlob * send, CMBlob * receive);

/* Description
   This function performs SPNEGO negotiation process.

   It repeatedly executes the exchange callback as designated by
   the 'exchange' parameter until the process either succeeds or
   fails.

   The two blobs will be generated by the crypters. NQ is
   starting from the highest level and descends to the least
   level until authentication succeeds. The list of crypters per
   level can be modified by calling <link amClientDefineLevel@NQ_UINT@NQ_UINT@NQ_UINT@NQ_UINT32, amClientDefineLevel()>.
   Parameters
   callingContext :     Pointer to an abstract context. When
                        exchange is needed, this function calls
                        the the <link AMSpnegoClientExchange>
                        callback and passes this pointer to it.
   serverName :         Name of the authenticating server.
   credentials :        User credentials to use.
   restrictCrypters :   TRUE to restrict crypters for Kerberos
                        exchange only, FALSE for all crypters.
                        The TRUE value is used with message
                        signing only since some Kerberos
                        implementations do not provide message
                        signing keys for some crypters.
   firstSecurityBlob :  The blob obtained during negotiation, for
                        instance \- during SMB Negotiate. This
                        blob is expected to contain the list of
                        security mechanisms. It can be NULL in
                        which case NTLMSSP will be used by
                        default.
   sessionKey :         On entry, this blob should contain
                        session key provided by server. Upon
                        successful return, NQ places message
                        signing challenge (sometimes referenced
                        as response) key into this structure. It
                        is caller's responsibility to free this
                        blob data.
   macKey :             Pointer to the signature key to be
                        created. Upon successful return, NQ
                        places mac session key (sometimes
                        referenced as message signing key) into
                        this structure. It is caller's
                        responsibility to free this blob data.
   exchange :           The callback to be used for security
                        exchange. See <link AMSpnegoClientExchange>.
   cleanCallback:       Callback to be used when switching between
                        security levels.
   Returns
   One of the constants defined in this module.*/
NQ_STATUS amSpnegoClientLogon(
    void * callingContext,
    const NQ_WCHAR * serverName,
    const AMCredentials * credentials,
    NQ_BOOL restrictCrypters,
    const CMBlob * firstSecurityBlob,
    CMBlob * sessionKey,
    CMBlob * macKey,
    AMSpnegoClientExchange exchange,
    void (*cleanCallback)(void * context)
    );
#endif /* UD_NQ_INCLUDECIFSCLIENT */

#ifdef UD_NQ_INCLUDECIFSSERVER

/* Description
   This function generates the list of supported security mechanisms (e..g, to use in SMB Negotiate response).

   Returns
   A blob with the list of security mechanisms. It is callers responsibility to release this blob.                                                                                   */
CMIOBlob amSpnegoServerGenerateMechList(void);

/* Description
   This server function parses incoming blob and according to
   its context performs one of the following operations:
     * If the incoming blob carries an SPNEGO Negotiate, this
       function generates SPNEGO Challenge;
     * If the incoming blob carries SPNEGO Response, this
       function composes tries to authenticate user.
   Parameters
   mechs :     A blob with the list of mechanisms as available
               after negotiation (e.g., \- SMB Negotiate).
   pOwnDomain : Poniter to own domain name.
   pOwnHostname : Pointer to own hostname.
   inBlob :    Incoming blob. It is expected to contains either
               SPNEGO Negotiate or SPNEGO Response.
   outBlob :   Pointer to blob to be filled with outgoing
               payload. it is caller's responsibility to release
               that blob.
   userName :  A buffer for user name. Should be of 256
               characters at least.
   pClientHostName : Pointer to client's host name.
   pDomain :   A pointer to domain name. On SPNEGO Response it will point to the domain name in
               the incoming blob.
   pSessionKey : A double pointer to the session key (sometimes referenced as message signing key).  On SPNEGO Response this pointer will refer to the key.
   ntlmDescr : Pointer to a descriptor to be filled with hashed passwords.
   Returns
   One of the constants defined in this module.                   */
NQ_UINT32 amSpnegoServerAcceptBlob(
    const void ** pMechBuf,
    const NQ_WCHAR *pOwnDomain,
    const NQ_WCHAR *pOwnHostname,
    CMIOBlob * inBlob,
    CMIOBlob * outBlob,
    NQ_WCHAR * userName,
    NQ_WCHAR * pClientHostName,
    NQ_WCHAR * pDomain,
    const NQ_BYTE ** pSessionKey,
    AMNtlmDescriptor * ntlmDescr
    );

/* ASCII version */
NQ_UINT32 amSpnegoServerAcceptBlobA(
    const void ** pMechBuf,
    const NQ_CHAR *pOwnDomain,
    const NQ_CHAR *pOwnHostname,
    CMIOBlob * inBlob,
    CMIOBlob * outBlob,
    NQ_CHAR * userName,
    NQ_CHAR * pClientHostName,
    NQ_WCHAR * pDomain,
    const NQ_BYTE ** pSessionKey,
    AMNtlmDescriptor * ntlmDescr
    );


/* Description
   The prototype for a server function to retrieve session key for the current connection.

   This function should be implemented in respect to the very project and may be modified if needed.

   Parameters
   key :    A double pointer to the session key.
   nonce :  A double pointer to nonce.
   Returns
   Session key length.*/
NQ_UINT amSpnegoServerGetCurrentSessionKey(NQ_BYTE ** key, NQ_BYTE **nonce);

/* Description
   The prototype for a server function to retrieve session key for the current connection.

   This function should be implemented in respect to the very project and may be modified if needed.

   Parameters
   key :    A double pointer to the session key.
   nonce :  A double pointer to nonce.
   Returns
   Session key length.*/
typedef NQ_UINT (* AMSpnegoServerGetCurrentSessionKey)(NQ_BYTE ** key, NQ_BYTE **nonce);

/* Description
   This function installs callback for retrieving session key for the current connection.

   This function should be implemented in respect to the very project and may be modified if needed.

   Parameters
   func :    The callback function.
   Returns
   None.*/
void amSpnegoServerSetSessionKeyCallback(AMSpnegoServerGetCurrentSessionKey func);

#endif /* UD_NQ_INCLUDECIFSSERVER */

void amSmbHash(NQ_BYTE *out, const NQ_BYTE *in, const NQ_BYTE *key, NQ_INT forw);

#endif /* _AMAPIIN_H_ */
