/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : The main loop of the CIFS server
 *--------------------------------------------------------------------
 * MODULE        : Server
 * DEPENDENCIES  :
 ********************************************************************/

#include "csapi.h"
#include "nsapi.h"
#include "amapi.h"

#include "csbrowse.h"
#include "csnotify.h"
#include "csdispat.h"
#ifdef UD_NQ_INCLUDESMB2
#include "cs2disp.h"
#endif /* UD_NQ_INCLUDESMB2 */
#include "csdataba.h"
#include "csutils.h"
#include "csauth.h"
#include "csdcerpc.h"
#include "cmsdescr.h"
#ifdef UD_CS_INCLUDECONTROL
#include "cscontrl.h"
#endif /* UD_CS_INCLUDECONTROL */
#include "cmbuf.h"
#ifdef UD_CS_INCLUDEDOMAINMEMBERSHIP
#include "ccapi.h"
#endif /* UD_CS_INCLUDEDOMAINMEMBERSHIP */
#ifdef UD_CS_INCLUDEPASSTHROUGH
#include "cmfinddc.h"
#include "ccdcerpc.h"
#include "ccsamrpc.h"
#include "cclsarpc.h"
#endif /*UD_CS_INCLUDEPASSTHROUGH*/
#ifdef UD_NQ_INCLUDESMBCAPTURE
#include "nssocket.h"
#endif /* UD_NQ_INCLUDESMBCAPTURE */

#ifdef UD_NQ_INCLUDECIFSSERVER

/* This code implements the main loop of the server
 */

/*
    Static data
    -----------
 */

#ifdef UD_CS_INCLUDECONTROL
/* abstract response parser - returns a pointer to the 
 * static parameter structure for this command */
typedef NQ_BOOL (*CommandProcessor)(
    CMBufferReader * reader, 
    CMBufferWriter * writer
    ); 

typedef struct          /* descriptor for a control command */
{
    NQ_UINT32 code;             /* command code */
    CommandProcessor processor; /* command processor */
}
ControlCommand;
#endif /* UD_CS_INCLUDECONTROL */

/* command processors */
static NQ_BOOL stopServer(CMBufferReader * reader, CMBufferWriter * writer);
#ifdef UD_CS_INCLUDECONTROL
static NQ_BOOL restartServer(CMBufferReader * reader, CMBufferWriter * writer);
#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION
static NQ_BOOL addShare(CMBufferReader * reader, CMBufferWriter * writer);
static NQ_BOOL removeShare(CMBufferReader * reader, CMBufferWriter * writer);
#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */
static NQ_BOOL enumShares(CMBufferReader * reader, CMBufferWriter * writer);
#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT
static NQ_BOOL addUser(CMBufferReader * reader, CMBufferWriter * writer);
static NQ_BOOL removeUser(CMBufferReader * reader, CMBufferWriter * writer);
static NQ_BOOL cleanUserCons(CMBufferReader * reader, CMBufferWriter * writer);
static NQ_BOOL enumUsers(CMBufferReader * reader, CMBufferWriter * writer);
#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */
static NQ_BOOL enumClients(CMBufferReader * reader, CMBufferWriter * writer);
static NQ_BOOL changeAuthenticationEncryptionLevel(CMBufferReader * reader, CMBufferWriter * writer);
#ifdef UD_CS_MESSAGESIGNINGPOLICY
static NQ_BOOL changeMsgSign(CMBufferReader * reader, CMBufferWriter * writer);
#endif /*UD_CS_MESSAGESIGNINGPOLICY*/
static NQ_BOOL enumFiles(CMBufferReader * reader, CMBufferWriter * writer);
#ifdef UD_NQ_INCLUDESMB1
static NQ_BOOL setSMB1Support(CMBufferReader * reader, CMBufferWriter * writer);
#endif /* UD_NQ_INCLUDESMB1 */

static const ControlCommand controlCommands[] = 
{
    { CS_CONTROL_STOP, stopServer },
    { CS_CONTROL_RESTART, restartServer },
#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION
    { CS_CONTROL_ADDSHARE, addShare },
    { CS_CONTROL_REMOVESHARE, removeShare },
#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */
    { CS_CONTROL_ENUMSHARES, enumShares },
#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT
    { CS_CONTROL_ADDUSER, addUser },
    { CS_CONTROL_REMOVEUSER, removeUser },
    { CS_CONTROL_CLEANUSERCONS, cleanUserCons },
    { CS_CONTROL_ENUMUSERS, enumUsers },
#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */
    { CS_CONTROL_ENUMCLIENTS, enumClients},
    { CS_CONTROL_CHANGEENCRYPTION, changeAuthenticationEncryptionLevel},
#ifdef UD_CS_MESSAGESIGNINGPOLICY
    { CS_CONTROL_CHANGEMSGSIGN, changeMsgSign},
#endif /*UD_CS_MESSAGESIGNINGPOLICY*/
#ifdef UD_NQ_INCLUDESMB1
    { CS_CONTROL_SETSMB1SUPPORT, setSMB1Support},
#endif /* UD_NQ_INCLUDESMB1 */
    { CS_CONTROL_ENUMFILES, enumFiles}
};
#endif /* UD_CS_INCLUDECONTROL */

typedef struct
{
#ifdef UD_CS_INCLUDECONTROL
    NSSocketHandle serverSocketUDP;                                     /* server internal UDP socket used to signal server to stop its execution */
#endif /* UD_CS_INCLUDECONTROL */
    NQ_BOOL restart;                                                    /* when TRUE - restart the server cycle */
    NQ_BOOL exitNow;                                                    /* TRUE signals to the server to stop execution */
#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) && defined(UD_CS_INCLUDEPASSTHROUGH)
    CMSdDomainSid domainSid;                                            /* SID of the configured domain */
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS && UD_CS_INCLUDEPASSTHROUGH */
#ifdef UD_NQ_USETRANSPORTNETBIOS
    NQ_UINT32 nextAnnouncementInterval;                                 /* next interval between announcements up to CM_FS_MINHOSTANNOUNCEMENTINTERVAL */
    NQ_UINT32 lastTimeout;                                              /* time of the last timeout */
    NSSocketHandle serverSocketNB;                                      /* server NetBIOS TCP socket */
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_NQ_USETRANSPORTIPV4
    NSSocketHandle serverSocketV4;                                      /* server TCPv4 socket */
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
    NSSocketHandle serverSocketV6;      /* server TCPv6 socket */
#endif /* UD_NQ_USETRANSPORTIPV6 */
    NSSocketSet socketSet;                                              /* for nsSelect() */
    CSSocketDescriptor clientSockets[UD_FS_NUMSERVERSESSIONS];          /* list of client sockets */
    SYMutex dbGuard;                                                    /* mutex for access to the database */
#ifdef UD_CS_INCLUDEPASSTHROUGH
    NQ_WCHAR domain[CM_BUFFERLENGTH(NQ_WCHAR, CM_NQ_HOSTNAMESIZE)];     /* buffer for client domain */
    NQ_WCHAR pdcName[CM_BUFFERLENGTH(NQ_WCHAR, CM_NQ_HOSTNAMESIZE)];    /* buffer for PDC name */
    NQ_BOOL pdcNameFlag;                                                /* flag for this name */
#endif /* UD_CS_INCLUDEPASSTHROUGH */
    NQ_WCHAR name[CM_USERNAMELENGTH];                                   /* buffer for user name */
    NQ_WCHAR fullName[CM_USERNAMELENGTH];                               /* buffer for full name */
    NQ_WCHAR description[UD_FS_MAXDESCRIPTIONLEN];                      /* buffer for description */
}
StaticData;

#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

/*
    Forward definitions
 */
static NQ_STATUS
serverCycle(
    void
    );

static void
closeServerSockets(
    void
    );

/* accept incoming connection and processes it */
static NQ_BOOL
acceptSocket(
    NSSocketHandle serverSocket,
    NQ_BOOL isNetBios,
    NQ_UINT32 time
    );

/* pause server for performing changes in the database */
static void
pauseServer(
    void
    );

/* resume server after database changes are over */
static void
resumeServer(
    void
    );

/* release allocated memory */
static void
releaseResources(
    void
    );

#ifdef UD_CS_INCLUDECONTROL
/* prepare internal UDP server socket */    
static NSSocketHandle
prepareUdpServerSocket(
    void
    );

/* perform a control command */
NQ_BOOL     /* TRUE - continue, FALSE - exit server */
doControl(
    SYSocketHandle sock     /* socket with pending command UDP */ 
    );
#endif /* UD_CS_INCLUDECONTROL */

/*
 *====================================================================
 * PURPOSE: main server loop
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:   As any other server, this one implements a listening loop
 *          we listen to one "server" socket (TCP) and several "client" sockets.
 *          The "server" socket accepts new connections, while "client" sockets
 *          represent those connections.
 *====================================================================
 */

NQ_STATUS
csStart(
    void
    )
{
    NQ_STATUS res = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL);

#ifndef UD_NQ_INCLUDESMB2
    if (!udDefGetServerSMB1Support())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "All dialects are disabled");
        goto Exit;
    }
#endif

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocate(sizeof(*staticData));
    if (NULL == staticData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate server data");
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    staticData->restart = FALSE;
    staticData->exitNow = FALSE;
    do
    {
        res = serverCycle();
        if (NQ_FAIL == res)
        {
            break;
        }
    }
    while (TRUE == staticData->restart);

#ifdef SY_FORCEALLOCATION
    if (NULL != staticData)
    {
        cmMemoryFree(staticData);
    }

    staticData = NULL;
#endif /* SY_FORCEALLOCATION */

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "res:%d", res);
    return res;
}


static NQ_INT prepareServerSockets()
{
    NQ_STATUS res = NQ_FAIL;       /* value returned from various calls */

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL);

#ifdef UD_CS_INCLUDECONTROL
    if ((staticData->serverSocketUDP = prepareUdpServerSocket()) == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Server internal UDP socket initialization failed");
        syPrintf("Server internal UDP socket initialization failed, It will not be possible to use server control tool\n");
    }
#endif /* UD_CS_INCLUDECONTROL */

#ifdef UD_NQ_USETRANSPORTNETBIOS
    if (udGetTransportPriority(NS_TRANSPORT_NETBIOS) &&
        (staticData->serverSocketNB = csPrepareSocket(NS_SOCKET_STREAM, NS_TRANSPORT_NETBIOS)) == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "NetBIOS socket initialization failed");
        goto Error1;
    }
#endif /* UD_NQ_USETRANSPORTNETBIOS */

#ifdef UD_NQ_USETRANSPORTIPV4
    if (udGetTransportPriority(NS_TRANSPORT_IPV4) &&
        (staticData->serverSocketV4 = csPrepareSocket(NS_SOCKET_STREAM, NS_TRANSPORT_IPV4)) == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "IPv4 socket initialization failed");
        goto Error2;
    }
#endif /* UD_NQ_USETRANSPORTIPV4 */

#ifdef UD_NQ_USETRANSPORTIPV6
    if (udGetTransportPriority(NS_TRANSPORT_IPV6) &&
        (staticData->serverSocketV6 = csPrepareSocket(NS_SOCKET_STREAM, NS_TRANSPORT_IPV6)) == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "IPv6 socket initialization failed");
        goto Error3;
    }
#endif /* UD_NQ_USETRANSPORTIPV6 */

    res = NQ_SUCCESS;
    goto Exit;

#ifdef UD_NQ_USETRANSPORTIPV6
Error3:
#ifdef UD_NQ_USETRANSPORTIPV4
    nsClose(staticData->serverSocketV4);
    staticData->serverSocketV4 = NULL;
#endif /* UD_NQ_USETRANSPORTIPV4 */
#endif /* UD_NQ_USETRANSPORTIPV6 */
#ifdef UD_NQ_USETRANSPORTIPV4
Error2:
#endif /* UD_NQ_USETRANSPORTIPV4 */
#if (defined(UD_NQ_USETRANSPORTNETBIOS) && (defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)))
    nsClose(staticData->serverSocketNB);
    staticData->serverSocketNB = NULL;
#endif /* (defined(UD_NQ_USETRANSPORTNETBIOS) && (defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6))) */
#ifdef UD_NQ_USETRANSPORTNETBIOS
Error1:
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_CS_INCLUDECONTROL
    nsClose(staticData->serverSocketUDP);
    staticData->serverSocketUDP = NULL;
#endif /* UD_CS_INCLUDECONTROL */
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "res:%d", res);
    return res;
}

static NQ_INT initServerCycle()
{
    NQ_STATUS res = NQ_FAIL;       /* value returned from various calls */
    NQ_UINT idx;                /* index in the table of client sockets */

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL);

    if (FALSE == staticData->restart)
    {
        syMutexCreate(&staticData->dbGuard);
#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) && defined(UD_CS_INCLUDEPASSTHROUGH)
        staticData->pdcNameFlag = FALSE;
#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) && defined(UD_CS_INCLUDEPASSTHROUGH) */
#ifdef UD_NQ_USETRANSPORTIPV4
        staticData->serverSocketV4 = NULL;
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
        staticData->serverSocketV6 = NULL;
#endif /* UD_NQ_USETRANSPORTIPV6 */
#ifdef UD_NQ_USETRANSPORTNETBIOS
        staticData->serverSocketNB = NULL;
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_CS_INCLUDECONTROL
        staticData->serverSocketUDP = NULL;
#endif /* UD_CS_INCLUDECONTROL */
    }

    /* clean up section is essential when this task is re-entered after csStop():
        - zero sockets
        - close server socket if still open */
    if (FALSE == staticData->restart)
    {
        closeServerSockets();
    }

    for (idx = 0; idx <UD_FS_NUMSERVERSESSIONS; idx++)
    {
        staticData->clientSockets[idx].socket = NULL;
    }
    
    /* Initialization:
        - Database
        - NetBIOS
        - socket set
        - stream server socket bound to the name of the machine */
    if (FALSE == staticData->restart)
    {
        if (NQ_FAIL == nsInit(TRUE, NQ_SERVER)) /* we are initializing a task - not a driver */
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "NS initialization failed");
            goto Exit;
        }
    }

    if (NQ_FAIL == csInitDatabase(&pauseServer, &resumeServer))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Database initialization failed");
        goto Error1;
    }

#ifdef UD_CS_INCLUDERPC
    if (NQ_FAIL == csDcerpcInit())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Dcerpc initialization failed");
        goto Error2;
    }
#endif /* UD_CS_INCLUDERPC */

#ifdef UD_NQ_USETRANSPORTNETBIOS
    if (NQ_FAIL == csInitBrowse())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Browser initialization failed");
        goto Error3;
    }
#endif /* UD_NQ_USETRANSPORTNETBIOS */
    if (NQ_FAIL == csFnamesInit())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Fnames initialization failed");
        goto Error4;
    }  

    if (!csAuthInit())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Authentication initialization failed");
        goto Error5;
    }

    if (NQ_FAIL == csDispatchInit())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Dispatcher initialization failed");
        goto Error6;
    }

#ifdef UD_NQ_INCLUDESMB2
    if (NQ_FAIL == cs2DispatchInit())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "SMB2 Dispatcher initialization failed");
        goto Error7;
    }
#endif /* UD_NQ_INCLUDESMB2 */

    if (NQ_FAIL == csNotifyInit())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Notify initialization failed");
        goto Error8;
    }

    if (!amStart())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "AM Spnego initialization failed");
        goto Error9;
    }

    if (FALSE == staticData->restart)
    {
        if (NQ_FAIL == prepareServerSockets())
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Prepare server sockets failed");
            goto Error10;
        }

#ifdef UD_NQ_USETRANSPORTNETBIOS
        /* announce our host to the domain for the first time and start the announcement timeout */
        if (NQ_FAIL == (NQ_INT)(staticData->nextAnnouncementInterval = csAnnounceServer()))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Host announcement failed");
            goto Error11;
        }
#endif /* UD_NQ_USETRANSPORTNETBIOS */
    }

    if (TRUE == staticData->restart)
    {
        staticData->restart = FALSE;
    }

    res = NQ_SUCCESS;
    goto Exit;

#ifdef UD_NQ_USETRANSPORTNETBIOS
Error11:
    closeServerSockets();
#endif /* UD_NQ_USETRANSPORTNETBIOS */
Error10:
    amShutdown();
Error9:
    csNotifyExit();
Error8:
#ifdef UD_NQ_INCLUDESMB2
    cs2DispatchExit();
Error7:
#endif /* UD_NQ_INCLUDESMB2 */
    csDispatchExit();
Error6:
    csAuthShutdown();
Error5:
    csFnamesExit();
Error4:
#ifdef UD_NQ_USETRANSPORTNETBIOS
    csStopBrowse();
Error3:
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_CS_INCLUDERPC
    csDcerpcStop();
Error2:
#endif /* UD_CS_INCLUDERPC */
    csCloseDatabase();
Error1:
    if (FALSE == staticData->restart)
    {
        nsExit(TRUE, NQ_SERVER);
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "res:%d", res);
    return res;
}

#ifdef UD_CS_INCLUDEDOMAINMEMBERSHIP

static void joinDomain()
{
    const NQ_WCHAR *domainW = NULL;
    AMCredentials adminW;

    if (NULL == (domainW = cmGetFullDomainNameW()))
    {
        if (cmNetBiosGetDomain()->isGroup)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "workgroup mode");
#ifdef UD_PRINT_STARTUP_INFO
            syPrintf("NQCS: workgroup mode\n");
#endif /* UD_PRINT_STARTUP_INFO */
            return;
        }

        domainW = cmNetBiosGetDomainAuthW();
    }
#ifdef UD_PRINT_STARTUP_INFO
    syPrintf("NQCS: default domain: %s, %s\n", cmGetFullDomainName(), cmNetBiosGetDomainAuth()->name);
#endif /* UD_PRINT_STARTUP_INFO */
    if (!udGetComputerSecret(NULL))
    {
        NQ_BYTE secret[16];

        /* get the credentials of user with administrative rights (join domain right) */
        if (!udGetCredentials(NULL, adminW.user, adminW.password, adminW.domain.name))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "admin credentials were not provided");
            return;
        }

        syAnsiToUnicode(adminW.domain.name, cmNetBiosGetDomain()->name);
        cmWStrcpy(adminW.domain.realm, adminW.domain.name);
        adminW.type = AM_CREDENTIALS_PASSWORD_PLAINTEXT;

        /* join domain (create computer account in AD) and store a secret,
           if computer account already exists, just update the secret */
        if (ccDomainUpdateSecret(domainW, cmNetBiosGetHostNameZeroedW(), &adminW, secret))
        {
            udSetComputerSecretByDomain(secret, domainW, cmNetBiosGetDomainAuthW());
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "joined default domain: %s, %s", cmWDump(domainW), cmNetBiosGetDomainAuth()->name);
#ifdef UD_PRINT_STARTUP_INFO
            syPrintf("NQCS: joined default domain: %s, %s\n", cmGetFullDomainName(), cmNetBiosGetDomainAuth()->name);
#endif /* UD_PRINT_STARTUP_INFO */
        }
        else
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "failed to join default domain: %s, error:0x%X", cmWDump(domainW), syGetLastError());
#ifdef UD_PRINT_STARTUP_INFO
            syPrintf("NQCS: failed to join default domain: %s, error:0x%X\n", cmWDump(domainW), syGetLastError());
#endif /* UD_PRINT_STARTUP_INFO */
        }
    }
}
#endif /* UD_CS_INCLUDEDOMAINMEMBERSHIP */

/*
 *====================================================================
 * PURPOSE: main server loop
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:   As any other server, this one implements a listening loop
 *          we listen to one "server" socket (TCP) and several "client" sockets.
 *          The "server" socket accepts new connections, while "client" sockets
 *          represent those connections.
 *====================================================================
 */

static NQ_STATUS
serverCycle(
    void
    )
{
    NQ_STATUS res;              /* value returned from various calls */
    NQ_UINT idx;                /* index in the table of client sockets */
    NQ_UINT32 curTime;          /* current system time */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL);

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "YNQ Server %s", staticData->restart ? "restarting" : "starting");

    if (NQ_FAIL == initServerCycle())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to init server");
        res = NQ_FAIL;
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "NetBios host name registered: %s", cmNetBiosGetHostNameZeroed());

#ifdef UD_CS_INCLUDEPASSTHROUGH
    /* resolve this domain SID on DC */
    if (!cmNetBiosGetDomainAuth()->isGroup && !staticData->pdcNameFlag)
    {
        const NQ_WCHAR * pdcName = csAuthGetPDCName();

        if (NULL != pdcName)
        {
            NQ_BOOL isDomainAuthSet = FALSE;
            CCLsaPolicyInfoDomain domainInfo;
            AMCredentials compCreds; /* computer account credentials */
            AMCredentials *pCreds = NULL;
            NQ_BYTE secret[16];
            NQ_BYTE *pSecret = secret;

            staticData->pdcNameFlag = TRUE;
            cmWStrcpy(staticData->pdcName, pdcName);
            cmAnsiToUnicode(staticData->domain, cmGetFullDomainName());

            /* get machine account credentials */
            if (udGetComputerSecret(&pSecret))
            {
                NQ_WCHAR dollar[] = { cmWChar('$'), cmWChar(0)};

                amCredentialsInit(&compCreds, staticData->domain, cmNetBiosGetHostNameZeroedW(), (const NQ_WCHAR *)pSecret, AM_CREDENTIALS_PASSWORD_MD4);
                cmWStrcat(compCreds.user, dollar);
                cmWStrupr(compCreds.user);

                pCreds = &compCreds;
            }

            /* get NetBIOS domain name and domain SID */
            if (NQ_SUCCESS == ccLsaPolicyQueryInfoDomain(pCreds, staticData->pdcName, &domainInfo))
            {
                cmNetBiosSetDomainAuth(domainInfo.name, FALSE);
                cmDCAddToCache(domainInfo.name, pdcName);
                isDomainAuthSet = TRUE;
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
                if (!cmSdIsDomainSidSet())
                {
                    cmSdSetDomainSid(&domainInfo.sid);
                }
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
            }
            /* get domain GUID (gets NetBIOS domain name as well) */
            if (NQ_SUCCESS == ccLsaDsRoleGetPrimaryDomainInformation(staticData->pdcName, pCreds, &domainInfo))
            {
                cmSetDomainGuid(&domainInfo.guid);
                if (!isDomainAuthSet)
                {
                    cmNetBiosSetDomainAuth(domainInfo.name, FALSE);
                    cmDCAddToCache(domainInfo.name, pdcName);
                }
            }

            cmAnsiToUnicode(staticData->domain, cmNetBiosGetDomainAuth()->name);

#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
            /* get domain SID (another attempt) */
            if (!cmSdIsDomainSidSet() && (NQ_SUCCESS == ccGetDomainSid(pCreds, staticData->domain, staticData->pdcName, &domainInfo.sid)))
            {
                cmSdSetDomainSid(&domainInfo.sid);
            }
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
        }
        else /* end of (NULL != pdcName) */
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Pass through authentication is not initialized yet");
        }
    }
#endif /* UD_CS_INCLUDEPASSTHROUGH */

    /* setup random number generator for creating encryption keys */
    sySetRand();

#ifdef UD_NQ_USETRANSPORTNETBIOS
    staticData->lastTimeout = (NQ_UINT32)syGetTimeInSec();
#endif

    /* join default domain */
#ifdef UD_CS_INCLUDEDOMAINMEMBERSHIP
    joinDomain();
#endif /* UD_CS_INCLUDEDOMAINMEMBERSHIP */

#ifdef UD_NQ_INCLUDEEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_GEN,
        UD_LOG_GEN_START,
        NULL,
        NULL,
        0,
        NULL
    );
#endif /* UD_NQ_INCLUDEEVENTLOG */

    /* from here we accept incoming client connections */
    udCifsServerStarted();

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Entering the main loop");
    while (TRUE)
    {
        if ((TRUE == staticData->restart) || (TRUE == staticData->exitNow))
        {
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Exit from the server cycle");
            break;
        }

        /* compose the set of sockets for select:
           1) the server listening socket
           2) client session sockets */

        nsClearSocketSet(&staticData->socketSet);
#ifdef UD_CS_INCLUDECONTROL
        if (NULL != staticData->serverSocketUDP)
        {
            nsAddSocketToSet(&staticData->socketSet, staticData->serverSocketUDP);        /* add internal UDP server socket */
        }
#endif /* UD_CS_INCLUDECONTROL */
#ifdef UD_NQ_USETRANSPORTNETBIOS
        if (udGetTransportPriority(NS_TRANSPORT_NETBIOS))
        {
            nsAddSocketToSet(&staticData->socketSet, staticData->serverSocketNB);     /* add NetBIOS server socket */
        }
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_NQ_USETRANSPORTIPV4
        if (udGetTransportPriority(NS_TRANSPORT_IPV4))
        {
            nsAddSocketToSet(&staticData->socketSet, staticData->serverSocketV4);     /* add v4 server socket */
        }
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
        if (udGetTransportPriority(NS_TRANSPORT_IPV6))
        {
            nsAddSocketToSet(&staticData->socketSet, staticData->serverSocketV6);     /* add v6 server socket */
        }
#endif /* UD_NQ_USETRANSPORTIPV6 */

        /* add client sockets */
        for (idx = 0; idx < UD_FS_NUMSERVERSESSIONS; idx++)
        {
            if (staticData->clientSockets[idx].socket != NULL)
            {
                /* if the socket actually was closed inside the server clear it here as well */
                if (!nsAddSocketToSet(&staticData->socketSet, staticData->clientSockets[idx].socket))
                {
                    staticData->clientSockets[idx].socket = NULL;
                }
            }
        }

#ifdef UD_NQ_USETRANSPORTNETBIOS
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "SERVER --->> Select, next timeout = %lu sec", (NQ_ULONG)(staticData->nextAnnouncementInterval));
        res = nsSelect(&staticData->socketSet, staticData->nextAnnouncementInterval);
#else /* UD_NQ_USETRANSPORTNETBIOS */
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "SERVER --->> Select, next timeout = %ld sec", SMB_MAX_SERVER_ANNOUNCEMENT_INTERVAL);
        res = nsSelect(&staticData->socketSet, SMB_MAX_SERVER_ANNOUNCEMENT_INTERVAL);
#endif /* UD_NQ_USETRANSPORTNETBIOS */

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "SERVER --->> Select returned: %d", res);

        /* call user defined processing */
        udServerDataIn();

        curTime = (NQ_UINT32)syGetTimeInSec();
#ifdef UD_NQ_USETRANSPORTNETBIOS
        /* Check for timeout and calculate the time to announce the server*/
        if (res == 0 || curTime >= (staticData->lastTimeout + staticData->nextAnnouncementInterval))       /* timeout */
        {
            staticData->lastTimeout = curTime;
            if ((NQ_INT)(staticData->nextAnnouncementInterval = csAnnounceServer()) == NQ_FAIL)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Host announcement failed, exit from the server cycle");
                break; /* exit server */
            }
        }
#endif /* UD_NQ_USETRANSPORTNETBIOS */

        /* on timeout do not continue processing */
        if (res == 0)
        {
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Select timeout");
            continue;
        }

        /* if select failed - one of sockets has disconnected:
           clean up disconnected client sockets */
        if (res == NQ_FAIL)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Select failed");

            if (FALSE
#ifdef UD_NQ_USETRANSPORTNETBIOS
                  || (!nsIsSocketAlive(staticData->serverSocketNB))
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_NQ_USETRANSPORTIPV4
                  || (!nsIsSocketAlive(staticData->serverSocketV4))
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
                  || (!nsIsSocketAlive(staticData->serverSocketV6))
#endif /* UD_NQ_USETRANSPORTIPV6 */
            )
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Select failed: one of server socket failed, exit from the server cycle");
                break;          /* exit server */
            }
            for (idx = 0; idx < UD_FS_NUMSERVERSESSIONS; idx++)
            {
                if (staticData->clientSockets[idx].socket != NULL)
                {
                    if (!nsIsSocketAlive(staticData->clientSockets[idx].socket))
                    {
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Select failed: a dead client socket found, cleaning up");

                        csReleaseSessions(staticData->clientSockets[idx].socket, FALSE);
                        nsClose(staticData->clientSockets[idx].socket);
                        staticData->clientSockets[idx].socket = NULL;
                    }
                }
            }
            
            continue;
        } /* end of if (res == NQ_FAIL) */
        
#ifdef UD_CS_INCLUDECONTROL
        /* if data arrived on internal UDP server socket it means a command was sent */
        if (NULL != staticData->serverSocketUDP)
        {
            if (nsSocketInSet(&staticData->socketSet, staticData->serverSocketUDP))
            {
                if (FALSE == doControl(nsGetSySocket(staticData->serverSocketUDP)))
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Control command execution failed");
                }
            }
        }
#endif /* UD_CS_INCLUDECONTROL */

        /* if new client connecting: accept new socket */
#ifdef UD_NQ_USETRANSPORTNETBIOS
        if (nsSocketInSet(&staticData->socketSet, staticData->serverSocketNB))
        {
            acceptSocket(staticData->serverSocketNB, TRUE, curTime);
        }
#endif /* UD_NQ_USETRANSPORTNETBIOS */

#ifdef UD_NQ_USETRANSPORTIPV4
        if (nsSocketInSet(&staticData->socketSet, staticData->serverSocketV4))
        {
            acceptSocket(staticData->serverSocketV4, FALSE, 0);
        }
#endif /* UD_NQ_USETRANSPORTIPV4 */

#ifdef UD_NQ_USETRANSPORTIPV6
        if (nsSocketInSet(&staticData->socketSet, staticData->serverSocketV6))
        {
            acceptSocket(staticData->serverSocketV6, FALSE, 0);
        }
#endif /* UD_NQ_USETRANSPORTIPV6 */

        /* if a session packet has arrived at an already accepted socket, this means SMB message */
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "checking accepted sockets");

        for (idx = 0; idx < UD_FS_NUMSERVERSESSIONS; idx++)
        {
            if (staticData->clientSockets[idx].socket != NULL)
            {
                if (!nsIsSocketAlive(staticData->clientSockets[idx].socket))
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "a dead session socket found, cleaning up");

                    csReleaseSessions(staticData->clientSockets[idx].socket , TRUE);
                    nsClose(staticData->clientSockets[idx].socket);
                    staticData->clientSockets[idx].socket = NULL;
                }
                else if (nsSocketInSet(&staticData->socketSet, staticData->clientSockets[idx].socket))
                {
#ifdef UD_NQ_USETRANSPORTNETBIOS
                    /* process NBT Session Request */
                    if (staticData->clientSockets[idx].requestExpected)
                    {
                        staticData->clientSockets[idx].requestExpected = (NQ_SUCCESS != nsPostAccept(&staticData->clientSockets[idx].socket));
                        continue;
                    }
#endif /* UD_NQ_USETRANSPORTNETBIOS */

                    syMutexTake(&staticData->dbGuard);
                    res = csDispatchRequest(&staticData->clientSockets[idx]);    /* process SMB message */
                    syMutexGive(&staticData->dbGuard);
                    
                    staticData->clientSockets[idx].lastActivityTime = curTime;

                    if (res == NQ_FAIL)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Error in performing SMB command");

                        csReleaseSessions(staticData->clientSockets[idx].socket , FALSE);
                        nsClose(staticData->clientSockets[idx].socket);
                        staticData->clientSockets[idx].socket = NULL;
                    }
                }
#ifdef UD_NQ_USETRANSPORTNETBIOS
                else
                {
                    /* clean up sockets with NBT Session Request timed out */
                    if (staticData->clientSockets[idx].requestExpected &&
                      ((curTime - staticData->clientSockets[idx].requestTimeout) > CM_NB_UNICASTREQRETRYTIMEOUT)
                       )
                    {
                        nsClose(staticData->clientSockets[idx].socket);
                        staticData->clientSockets[idx].socket = NULL;
                    }
                }
#endif /* UD_NQ_USETRANSPORTNETBIOS */
            }
        }
    }/* end of main loop */

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Exiting the YNQ Server %s", staticData->restart ? "(for restart)" : "");
    
    /* close server sockets when shutting down */
    if (FALSE == staticData->restart)
    {
        closeServerSockets();
    }

    /* close clients sockets always */
    for (idx = 0; idx < UD_FS_NUMSERVERSESSIONS; idx++)
    {
        if (staticData->clientSockets[idx].socket != NULL)
        {
            csReleaseSessions(staticData->clientSockets[idx].socket , TRUE);
            if (nsIsSocketAlive(staticData->clientSockets[idx].socket))
            {
                nsClose(staticData->clientSockets[idx].socket);
                staticData->clientSockets[idx].socket = NULL;
            }
        }
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_GEN,
        UD_LOG_GEN_STOP,
        NULL,
        NULL,
        0,
        NULL
    );
#endif /* UD_NQ_INCLUDEEVENTLOG */

    releaseResources();
    res = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "res:%d", res);
    return res;

}  

#ifdef UD_CS_INCLUDECONTROL
/*
 *====================================================================
 * PURPOSE: create a UDP socket and bind it to any port
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: socket handle or NULL on error
 *
 * NOTES:   This internal UDP server socket is used to signal server to
 *          exit the main execution loop.
 *====================================================================
 */
static NSSocketHandle
prepareUdpServerSocket(
    void
    )
{
    NSSocketHandle socket = NULL;
    NQ_COUNT i;
    NQ_UINT transArr[] = {
#ifdef UD_NQ_USETRANSPORTIPV6
       NS_TRANSPORT_IPV6,
#endif
#ifdef UD_NQ_USETRANSPORTIPV4
       NS_TRANSPORT_IPV4,
#endif
#ifdef UD_NQ_USETRANSPORTNETBIOS
       NS_TRANSPORT_NETBIOS
#endif
                }; /* array to check transports*/
    NQ_IPADDRESS * pLocalHost = cmSelfipGetLocalHostIp();

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL);

    for (i = 0; i < sizeof(transArr)/sizeof(transArr[0]) ; i++)
    {
        if (!udGetTransportPriority(transArr[i]))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Transport %d is not used" , transArr[i]);
            continue;
        }

        /* create a UDP socket */
        if (NULL == (socket = nsSocket(NS_SOCKET_DATAGRAM, transArr[i])))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unable to create internal UDP server socket with transport %d " , transArr[i]);
            continue;
        }
#ifdef UD_NQ_USETRANSPORTIPV6
        if (transArr[i] == NS_TRANSPORT_IPV6)
        {
            pLocalHost = cmSelfipGetLocalHostIpVersion6();
        }
#endif /*UD_NQ_USETRANSPORTIPV6*/

        /* bind to CS_CONTROL_PORT */
        if (nsBindInet(socket, pLocalHost, syHton16(CS_CONTROL_PORT)) == NQ_FAIL)
        {
            nsClose(socket);
            LOGERR(CM_TRC_LEVEL_ERROR, "Unable to bind internal UDP server socket to CS_CONTROL_PORT with transport %d", transArr[i]);
            continue;
        }

        goto Exit;
    }

    LOGERR(CM_TRC_LEVEL_ERROR, "Unable to create or bind internal UDP server socket");
    socket = NULL;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "socket:%p", socket);
    return socket;
}
#endif /* UD_CS_INCLUDECONTROL */

/*
 *====================================================================
 * PURPOSE: close server sockets
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES: releases registered NetBIOS names
 *====================================================================
 */

static void
closeServerSockets(
    void
    )
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (staticData != NULL)
    {
#ifdef UD_NQ_USETRANSPORTNETBIOS
        if (staticData->serverSocketNB != NULL)
        {
            nsClose(staticData->serverSocketNB);
            staticData->serverSocketNB = NULL;
        }
#endif /* UD_NQ_USETRANSPORTNETBIOS */

#ifdef UD_NQ_USETRANSPORTIPV4
        if (staticData->serverSocketV4 != NULL)
        {
            nsClose(staticData->serverSocketV4);
            staticData->serverSocketV4 = NULL;
        }
#endif /* UD_NQ_USETRANSPORTIPV4 */

#ifdef UD_NQ_USETRANSPORTIPV6
        if (staticData->serverSocketV6 != NULL)
        {
            nsClose(staticData->serverSocketV6);
            staticData->serverSocketV6 = NULL;
        }
#endif /* UD_NQ_USETRANSPORTIPV6 */
        
#ifdef UD_CS_INCLUDECONTROL
        if (staticData->serverSocketUDP != NULL)
        {
            nsClose(staticData->serverSocketUDP);  
            staticData->serverSocketUDP = NULL;
        }
#endif /* UD_CS_INCLUDECONTROL */
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 *====================================================================
 * PURPOSE: debug only
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

#ifdef NQDEBUG

void
csDumpSockets(
    )
{
    NQ_UINT i;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    syPrintf("\n================ Sockets ==============\n");

    for (i = 0; i < UD_FS_NUMSERVERSESSIONS; i++)
    {
        syPrintf(
            "socket: %d, mapped on: %p, with peer IP: %s\n",
            i,
            staticData->clientSockets[i].socket,
            cmIPDump(&staticData->clientSockets[i].ip));
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL,
                "socket: %d, mapped on: %p, with peer IP: %s\n",
                i,
                staticData->clientSockets[i].socket,
                cmIPDump(&staticData->clientSockets[i].ip));
    }

    syPrintf("================         ==============\n\n");
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

#endif /* NQDEBUG */

/*
 *====================================================================
 * PURPOSE: pause server for performing changes in the database
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

static void
pauseServer(
    void
    )
{
    syMutexTake(&staticData->dbGuard);
}

/*
 *====================================================================
 * PURPOSE: resume server after changes in the database
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

static void
resumeServer(
    void
    )
{
    syMutexGive(&staticData->dbGuard);
}

/*
 *====================================================================
 * PURPOSE: accept new client socket
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE when done and FALSE on error
 *
 * NOTES:
 *====================================================================
 */

static NQ_BOOL
acceptSocket(
    NSSocketHandle serverSocket,
    NQ_BOOL isNetBios,
    NQ_UINT32 time
    )
{
    NSSocketHandle newSocket;           /* an accepted socket */
    NQ_UINT idx;                        /* index in the table of client sockets */
    NQ_IPADDRESS ip;                    /* IP on the next side of the socket */

    nsResetBufferPool();
    newSocket = nsAccept(serverSocket, &ip);
    if (newSocket == NULL)
    {
    #ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(UD_LOG_MODULE_CS,
            UD_LOG_CLASS_CONNECTION,
            UD_LOG_CONNECTION_CONNECT,
            NULL,
            &ip,
            (NQ_UINT32)syGetLastError(),
            NULL);
    #endif
        TRCERR("nsAccept failed");
        return FALSE;
    }

    /* save this socket in an empty record in the client socket table */
    for (idx = 0; idx < UD_FS_NUMSERVERSESSIONS; idx++)
    {
        if (staticData->clientSockets[idx].socket == NULL)
        {
            break;
        }
    }

    if (idx == UD_FS_NUMSERVERSESSIONS)
    {
#ifdef UD_CS_REFUSEONSESSIONTABLEOVERFLOW
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(UD_LOG_MODULE_CS,
            UD_LOG_CLASS_CONNECTION,
            UD_LOG_CONNECTION_CONNECT,
            NULL,
            &ip,
            (NQ_UINT32)SMB_STATUS_INSUFFICIENT_RESOURCES,
            NULL);
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR(" Server Session Table Overflow - Refusing Connection");
        csDispatchSendError(newSocket, SMB_STATUS_INSUFFICIENT_RESOURCES, SRV_ERRnoresource);
        nsClose(newSocket);
        return FALSE;
#else
        NQ_UINT stepIdx = 0;        /* Index of the oldest inactive session so far */ 
        NQ_UINT32 stepTime = (NQ_UINT32)-1;  /* Last activity time of the */

        /* no more connections may be accepted - 
         * close the connection with the latest activity 
         */
        for (idx = 0; idx < UD_FS_NUMSERVERSESSIONS; idx++)
        {
            if (stepTime == (NQ_UINT32)-1 || stepTime > staticData->clientSockets[idx].lastActivityTime)
            {
                stepTime = staticData->clientSockets[idx].lastActivityTime;
                stepIdx = idx;
            }
        }

        csReleaseSessions(staticData->clientSockets[stepIdx].socket , FALSE);
        nsClose(staticData->clientSockets[stepIdx].socket);
        staticData->clientSockets[stepIdx].socket = NULL;
        idx = stepIdx;
#endif
        
    }
#ifdef UD_NQ_INCLUDEEVENTLOG
    udEventLog(UD_LOG_MODULE_CS,
        UD_LOG_CLASS_CONNECTION,
        UD_LOG_CONNECTION_CONNECT,
        NULL,
        &ip,
        NQ_SUCCESS,
        NULL);
#endif
    /* save the connection socket in an empty slot */
    staticData->clientSockets[idx].socket = newSocket;
    staticData->clientSockets[idx].ip = ip;
    staticData->clientSockets[idx].lastActivityTime = time;
#ifdef UD_NQ_USETRANSPORTNETBIOS
    staticData->clientSockets[idx].requestTimeout = time;
    staticData->clientSockets[idx].requestExpected = isNetBios;
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_NQ_INCLUDESMBCAPTURE
    {
        NQ_IPADDRESS    serverIp;
        NQ_PORT         serverPort;
        SocketSlot *    serverSock = (SocketSlot *)serverSocket;
        SocketSlot * clientSock = (SocketSlot *)newSocket;

        syGetSocketPortAndIP(serverSock->socket , &serverIp , &serverPort);
        staticData->clientSockets[idx].captureHdr.dstIP = clientSock->remoteIP;
        staticData->clientSockets[idx].captureHdr.dstPort = clientSock->port;
        staticData->clientSockets[idx].captureHdr.srcIP = serverIp;
        staticData->clientSockets[idx].captureHdr.srcPort = serverPort; 
        staticData->clientSockets[idx].captureHdr.receiving = TRUE;

    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: release allocated memory
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

static void
releaseResources(
    void
    )
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    amShutdown();
    csFnamesExit();
    csAuthShutdown();   
    csNotifyExit();
    csDispatchExit();
#ifdef UD_NQ_INCLUDESMB2
    cs2DispatchExit();
#endif /* UD_NQ_INCLUDESMB2 */
#ifdef UD_NQ_USETRANSPORTNETBIOS
    csStopBrowse();
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_CS_INCLUDERPC
    csDcerpcStop();
#endif /* UD_CS_INCLUDERPC */
    csCloseDatabase();
    syMutexDelete(&staticData->dbGuard);
    if (!staticData->restart)
    {
        nsExit(TRUE, NQ_SERVER);
    }

    udCifsServerClosed();

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

#ifdef UD_CS_INCLUDECONTROL
/*
 *====================================================================
 * PURPOSE: perform a control command
 *--------------------------------------------------------------------
 * PARAMS:  IN socket with pending UDP 
 *
 * RETURNS: TRUE - continue, FALSE - exit server
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL     
doControl(
    SYSocketHandle sock      
    )
{
    NQ_IPADDRESS ip;            /* IP address */
    NQ_PORT port;               /* port number */
    NQ_BYTE buf[CS_CONTROL_MAXMSG]; /* command buffer */
    NQ_UINT32 code;             /* command code */
    CMBufferReader reader;      /* command parser */
    CMBufferReader writer;      /* command packer */
    NQ_COUNT i;                 /* just a counter */
    NQ_INT comLen;              /* command length */
    NQ_BOOL res = FALSE;        /* command result */
    
    LOGFB(CM_TRC_LEVEL_FUNC_TOOL);

    /* read command */
    comLen = syRecvFromSocket(sock, buf, sizeof(buf), &ip, &port);
    if (comLen <= 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error receiving control command");
        goto Exit;
    }

    /* parse response code */
    cmBufferReaderInit(&reader, buf, (NQ_COUNT)comLen);
    cmBufferReadUint32(&reader, &code);
    
    /* find command */
    for (i = 0; i < sizeof(controlCommands)/sizeof(controlCommands[0]); i++)
    {
        if (controlCommands[i].code == code)
        {
            break;
        }
    }
    
    /* prepare packer */
    cmBufferWriterInit(&writer, buf, sizeof(buf));
    if (i < sizeof(controlCommands)/sizeof(controlCommands[0]))
    {
        res = (*controlCommands[i].processor)(&reader, &writer);
    }
    else
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Command not supported");
        cmBufferWriteInt32(&writer, NQ_ERR_NOSUPPORT);
    }
    
    /* send response */
    comLen = sySendToSocket(sock, buf, (NQ_COUNT)(writer.current - buf), &ip, port);
    if (comLen <= 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error sending control response");
        res = FALSE;
        goto Exit;
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result: %s", res ? "TRUE" : "FALSE");
    return res;
}
#endif /* UD_CS_INCLUDECONTROL */

/*
 *====================================================================
 * PURPOSE: stop the server
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

void
csStop(
    void
    )
{
    stopServer(NULL, NULL);
}

/*
 *====================================================================
 * PURPOSE: get data for all open files and directories.
 *--------------------------------------------------------------------
 * PARAMS:  in - file index
 *          out - file data
 *
 * RETURNS: TRUE - this file index exists.
 *          FALSE - file index doesn't exist *
 *
 * NOTES:   call this function in a loop which continues as long as TRUE value is returned.
 *          Be sure to advance index by one per call.
 *          If a file is opened or closed during the loop, 
 *          the results might not reflect the exact open files situation.
 *====================================================================
 */

NQ_BOOL csEnumerateOpenFiles(NQ_UINT index, CSFileData *fileData)
{
    const CSFile    *pFile;       /* pointer to share slot */
    CSUser          *pUser;
    CSSession       *pSession;
    const NQ_WCHAR  *fileName;
    NQ_CHAR IP[CM_IPADDR_MAXLEN];
    NQ_BOOL result = TRUE;

    TRCB();

    /* perform */
    pFile = csGetFileByIndex(index);
    if (pFile != NULL)
    {
        pSession = csGetSessionById(pFile->session);
        pUser = csGetUserBySession(pSession);
        fileName = csGetFileName(pFile->fid);

        cmIpToAscii(IP , &pSession->ip);
        cmAnsiToUnicode(fileData->IP, IP);

        syWStrcpy(fileData->fileNamePath, fileName);
        syWStrcpy(fileData->userName, pUser->name);
        fileData->isDirectory = pFile->isDirectory;

        goto Exit;
    }

    result = FALSE;

Exit:
    TRCE();
    return result;
}

NQ_BOOL csEnumerateOpenFilesA(NQ_UINT index, FileDataA_t *fileData)
{
    CSFileData fileDataW;

    TRCB();

    if (csEnumerateOpenFiles(index, &fileDataW))
    {
        cmUnicodeToAnsi(fileData->IP,fileDataW.IP);
        cmUnicodeToAnsi(fileData->userName,fileDataW.userName);
        cmUnicodeToAnsi(fileData->fileNamePath,fileDataW.fileNamePath);
        fileData->isDirectory = fileDataW.isDirectory;

        TRCE();
        return TRUE;
    }

    TRCE();
    return FALSE;
}

#ifdef UD_NQ_INCLUDESMBCAPTURE
CSSocketDescriptor * csGetClientSocketDescriptorBySocket(NSSocketHandle socket)
{
    NQ_INT  i;

    if (socket == NULL)
    {
        return NULL;
    }

    for (i = 0; i < UD_FS_NUMSERVERSESSIONS ; i++)
    {
        if (staticData->clientSockets[i].socket == socket)
        {
            return (CSSocketDescriptor *)&staticData->clientSockets[i];
        }
    }

    return NULL;
}
#endif /* UD_NQ_INCLUDESMBCAPTURE */

/* 
 * command processors 
 */

static NQ_BOOL stopServer(CMBufferReader * reader, CMBufferWriter * writer)
{
    if (NULL != writer)
    {
        cmBufferWriteUint32(writer, NQ_SUCCESS);
    }

    if (NULL != staticData)
    {
        staticData->restart = FALSE;
        staticData->exitNow = TRUE;
    }

    return FALSE;
}

#ifdef UD_CS_INCLUDECONTROL
static NQ_BOOL restartServer(CMBufferReader * reader, CMBufferWriter * writer)
{
    cmBufferWriteUint32(writer, NQ_SUCCESS);

    /* reset server parameters, they will be reloaded later from cs_cfg file when needed */
    udResetServerParams();
    if (NULL != staticData)
    {
        staticData->restart = TRUE;
    }
    return FALSE;
}

#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION
static NQ_BOOL addShare(CMBufferReader * reader, CMBufferWriter * writer)
{
    const NQ_WCHAR *name;      /* share name */
    const NQ_WCHAR *path;      /* share mapping */
    const NQ_WCHAR *comment;   /* share comment */
    NQ_BOOL isPrinter;         /* TRUE for printer */
    NQ_UINT16 len;             /* string length */
    NQ_STATUS res;             /* result status */
    CSShare *pShare;           /* share descriptor */
    
    /* parse the command */
    cmBufferReadUint16(reader, &len);
    name = (const NQ_WCHAR*)cmBufferReaderGetPosition(reader);
    cmBufferReaderSkip(reader, (NQ_UINT)((NQ_UINT)(len + 1) * sizeof(NQ_WCHAR)));
    cmBufferReadUint16(reader, &len);
    path = (const NQ_WCHAR*)cmBufferReaderGetPosition(reader);
    cmBufferReaderSkip(reader, (NQ_UINT)((NQ_UINT)(len + 1) * sizeof(NQ_WCHAR)));
    cmBufferReadUint16(reader, &len);
    isPrinter = len == 1? TRUE : FALSE;
    cmBufferReadUint16(reader, &len);
    comment = (const NQ_WCHAR*)cmBufferReaderGetPosition(reader);
    cmBufferReaderSkip(reader, (NQ_UINT)((NQ_UINT)(len + 1) * sizeof(NQ_WCHAR)));
    
    /* perform */
    res = csAddShare(name, path, isPrinter, comment, NULL);
    if (NQ_SUCCESS != res)
    {
        goto Exit1;
    }

    pShare = csGetShareByName(name);
    if (NULL == pShare)
    {
        res = NQ_FAIL;
        goto Exit2;
    }

    /* check the existence of the underlying path */
    if (FALSE == csCheckShareMapping(pShare))
    {
        res = NQ_FAIL;
        goto Exit2;
    }

    if (FALSE == isPrinter)
    {
        /* add share persistently, it is achieved by saving the share information in server configuration file */
        if (TRUE != udSaveShareInformation(NULL, name, path, comment))
        {
            res = NQ_FAIL;
            goto Exit2;
        }
    }

    goto Exit1;

Exit2:
    csRemoveShare(name);
Exit1:
    /* pack response */
    cmBufferWriteInt32(writer, res == NQ_SUCCESS? NQ_SUCCESS : NQ_ERR_ERROR);
    return TRUE;
}

static NQ_BOOL removeShare(CMBufferReader * reader, CMBufferWriter * writer)
{
    const NQ_WCHAR * name;      /* share name */
    NQ_UINT16 len;              /* string length */
    NQ_STATUS res;              /* result status */
    
    /* parse the command */
    cmBufferReadUint16(reader, &len);
    name = (const NQ_WCHAR*)cmBufferReaderGetPosition(reader);
    cmBufferReaderSkip(reader, (NQ_UINT)((NQ_UINT)(len + 1) * sizeof(NQ_WCHAR)));
    
    /* perform */
    res = csRemoveShare(name);

    if (NQ_SUCCESS == res)
    {
        /* remove share persistently, it is achieved by removing the share from server configuration file */
        if (TRUE != udRemoveShare(name))
        {
            res = NQ_FAIL;
        }
    }

    /* pack response */
    cmBufferWriteInt32(writer, res == NQ_SUCCESS? NQ_SUCCESS : NQ_ERR_ERROR);
    
    return TRUE;
}
#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */

static NQ_BOOL enumShares(CMBufferReader * reader, CMBufferWriter * writer)
{
    NQ_UINT32 res;              /* result status */
    NQ_UINT16 index;            /* index */
    const CSShare *share;       /* pointer to share slot */

    TRCB();
    
    /* parse the command */
    cmBufferReadUint16(reader, &index);
        
    /* perform */
    res = (share = csGetShareByIndex(index)) != NULL ? NQ_SUCCESS : (NQ_UINT32)NQ_ERR_ERROR;

    /* pack response */
    cmBufferWriteUint32(writer, res);
    if (res == NQ_SUCCESS)
    {
        cmBufferWriteUint16(writer, (NQ_UINT16)syWStrlen(share->name));
        cmBufferWriteUnicode(writer, share->name);
        cmBufferWriteUint16(writer, (NQ_UINT16)syWStrlen(share->map));
        cmBufferWriteUnicode(writer, share->map);
        cmBufferWriteUint16(writer, share->isPrintQueue ? 1 : 0);
        cmBufferWriteUint16(writer, (NQ_UINT16)syWStrlen(share->description));
        cmBufferWriteUnicode(writer, share->description);
    }

    TRCE();
    return TRUE;
}

#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT

static NQ_BOOL addUser(CMBufferReader * reader, CMBufferWriter * writer)
{
    const NQ_WCHAR* name;       /* logon name */
    const NQ_WCHAR* fullName;   /* full name */
    const NQ_WCHAR* description;/* user descripton */
    const NQ_WCHAR* password;   /* password */
    NQ_BOOL isAdmin;            /* TRUE for Admistrator rights */
    NQ_UINT16 len;              /* string length */
    NQ_STATUS res;              /* result status */
    NQ_UINT32 rid;              /* user RID */
    NQ_WCHAR passwordW[CM_BUFFERLENGTH(NQ_WCHAR, 256)];

    
    /* parse the command */
    cmBufferReadUint16(reader, &len);
    name = (const NQ_WCHAR*)cmBufferReaderGetPosition(reader);
    cmBufferReaderSkip(reader, (NQ_UINT)((NQ_UINT)(len + 1) * sizeof(NQ_WCHAR)));
    cmBufferReadUint16(reader, &len);
    fullName = (const NQ_WCHAR*)cmBufferReaderGetPosition(reader);
    cmBufferReaderSkip(reader, (NQ_UINT)((NQ_UINT)(len + 1) * sizeof(NQ_WCHAR)));
    cmBufferReadUint16(reader, &len);
    description = (const NQ_WCHAR*)cmBufferReaderGetPosition(reader);
    cmBufferReaderSkip(reader, (NQ_UINT)((NQ_UINT)(len + 1) * sizeof(NQ_WCHAR)));
    cmBufferReadUint16(reader, &len);
    password = (const NQ_WCHAR*)cmBufferReaderGetPosition(reader);
    cmBufferReaderSkip(reader, (NQ_UINT)((NQ_UINT)(len + 1) * sizeof(NQ_WCHAR)));
    cmBufferReadUint16(reader, &len);
    isAdmin = len == 1? TRUE : FALSE;
    
    /* perform */
    res = udCreateUser(name, fullName, description);
    if (res)
    {
        res = udGetUserRidByName(name, &rid);
    }

    if (res)
    {
        syWStrcpy(passwordW, password);
        res = udSetUserInfo(rid, name, fullName, description, passwordW);
    }

    if (res)
    {
        res = udSetUserAsAdministrator(rid, isAdmin);
    }

    /* pack response */
    cmBufferWriteInt32(writer, res? NQ_SUCCESS : NQ_ERR_ERROR);
    
    return TRUE;
}

static NQ_BOOL removeUser(CMBufferReader * reader, CMBufferWriter * writer)
{
    const NQ_WCHAR * name;      /* user name */
    NQ_UINT16 len;              /* string length */
    NQ_STATUS res;              /* result status */
    NQ_UINT32 rid;              /* user RID */
    NQ_UINT i;                  /* user counter */
    
    /* parse the command */
    cmBufferReadUint16(reader, &len);
    name = (const NQ_WCHAR*)cmBufferReaderGetPosition(reader);
    cmBufferReaderSkip(reader, (NQ_UINT)((NQ_UINT)(len + 1) * sizeof(NQ_WCHAR)));
    
    /* perform */
    res = udGetUserRidByName(name, &rid);
    if (res)
    {
        res = udDeleteUserByRid(rid);
    }
    
    /* release user from the database */
    for (i = 0; ; i++)
    {
        CSUser* pUser = csGetUserByIndex(i); /* pointer to user descriptor */
        if (NULL == pUser)
        {
            break;
        }

        if (pUser->token.rids[0] == rid)
        {
            csReleaseUser(pUser->uid, TRUE);
            break;
        }
    }
    
    /* pack response */
    cmBufferWriteInt32(writer, res? NQ_SUCCESS : NQ_ERR_ERROR);
    
    return TRUE;
}

static NQ_BOOL cleanUserCons(CMBufferReader * reader, CMBufferWriter * writer)
{
    const NQ_WCHAR * name;      /* user name */
    NQ_UINT16 len;              /* string length */
    NQ_STATUS res;              /* result status */
    NQ_UINT16 isDomainUser;     /* user type */

    /* parse the command */
    cmBufferReadUint16(reader, &isDomainUser);
    cmBufferReadUint16(reader, &len);
    name = (const NQ_WCHAR*)cmBufferReaderGetPosition(reader);
    cmBufferReaderSkip(reader, (NQ_UINT)((NQ_UINT)(len + 1) * sizeof(NQ_WCHAR)));

    /* perform */
    res = csCleanUserServerConnections(name, isDomainUser);
    
    /* pack response */
    cmBufferWriteInt32(writer, res == NQ_SUCCESS ? NQ_SUCCESS : NQ_ERR_ERROR);
    return TRUE;
}

static NQ_BOOL enumUsers(CMBufferReader * reader, CMBufferWriter * writer)
{
    NQ_UINT16 index;            /* index */
    NQ_UINT32 rid;              /* user RID */
    
    /* parse the command */
    cmBufferReadUint16(reader, &index);
        
    /* perform */
    if (udGetUserInfo(index, &rid, staticData->name, staticData->fullName, staticData->description))
    {
        /* pack response */
        cmBufferWriteUint32(writer, NQ_SUCCESS);
        cmBufferWriteUint16(writer, (NQ_UINT16)syWStrlen(staticData->name));
        cmBufferWriteUnicode(writer, staticData->name);
        cmBufferWriteUint16(writer, (NQ_UINT16)syWStrlen(staticData->fullName));
        cmBufferWriteUnicode(writer, staticData->fullName);
        cmBufferWriteUint16(writer, (NQ_UINT16)syWStrlen(staticData->description));
        cmBufferWriteUnicode(writer, staticData->description);
        cmBufferWriteUint16(writer, (NQ_UINT16)(((NQ_INT)rid) < 0 ? 1 : 0));
    }
    else
    {
        /* pack response */
        cmBufferWriteInt32(writer, NQ_ERR_ERROR);
    }

    return TRUE;
}

#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */

static NQ_BOOL enumClients(CMBufferReader * reader, CMBufferWriter * writer)
{
    NQ_UINT16   index;      /* index*/
    NQ_UINT16   count;
    NQ_INT      i;

    cmBufferReadUint16(reader , &index);
    count = index;

    for (i = 0; i < UD_FS_NUMSERVERSESSIONS ; i++)
    {
        CSSession * pSession;

        pSession = csGetSessionById((CSSessionKey)i);
        if (pSession != NULL)
        {
            if (count == 0 && pSession->key != CS_ILLEGALID)
            {
                NQ_CHAR * ip;
                NQ_WCHAR * ipW;
                ip = (NQ_CHAR *)cmMemoryAllocate(CM_IPADDR_MAXLEN* sizeof(NQ_CHAR));
                ipW = (NQ_WCHAR *)cmMemoryAllocate(CM_IPADDR_MAXLEN* sizeof(NQ_WCHAR));
                if (ip == NULL || ipW == NULL)
                {
                    cmMemoryFree(ip); /* can handle NULL */
                    cmMemoryFree(ipW); /* can handle NULL */
                    break;
                }

                cmIpToAscii(ip , &pSession->ip);
                cmAnsiToUnicode(ipW, ip);
                cmBufferWriteUint32(writer, NQ_SUCCESS);
                cmBufferWriteUint16(writer , (NQ_UINT16)syWStrlen(ipW));
                cmBufferWriteUnicode(writer , ipW);
                cmBufferWriteUint16(writer , (NQ_UINT16)pSession->dialect);
                cmMemoryFree(ip);
                cmMemoryFree(ipW);
                return TRUE;
            }

            if (pSession->key != CS_ILLEGALID)
            {
                count == 0 ? count = 0 : count--;
            }
        }

    }

    cmBufferWriteInt32(writer, NQ_ERR_ERROR);

    return TRUE;
}

static NQ_BOOL enumFiles(CMBufferReader * reader, CMBufferWriter * writer)
{
    NQ_UINT16       index;            /* index */
    const CSFile *  pFile;       /* pointer to share slot */
    CSUser  *       pUser;
    CSSession   *   pSession;
    const NQ_WCHAR  *   fileName;
    NQ_CHAR * ip;

    TRCB();

    /* parse the command */
    cmBufferReadUint16(reader, &index);

    /* perform */
    pFile = csGetFileByIndex(index);
    if (pFile != NULL)
    {
        pSession = csGetSessionById(pFile->session);
        pUser = csGetUserBySession(pSession);
        fileName = csGetFileName(pFile->fid);
        /* pack response */
        cmBufferWriteUint32(writer, NQ_SUCCESS);

        ip = (NQ_CHAR *)cmMemoryAllocate(CM_IPADDR_MAXLEN* sizeof(NQ_CHAR));
        if (ip == NULL)
        {
            return FALSE;
        }

        cmIpToAscii(ip , &pSession->ip);
        cmBufferWriteUint16(writer , (NQ_UINT16)syStrlen(ip));
        cmBufferWriteBytes(writer, (NQ_BYTE *)ip , (NQ_COUNT)syStrlen(ip) + 1);
        cmMemoryFree(ip);
        cmBufferWriteUint16(writer, (NQ_UINT16)syWStrlen(fileName));
        cmBufferWriteUnicode(writer, fileName);
        cmBufferWriteUint16(writer, (NQ_UINT16)syWStrlen(pUser->name));
        cmBufferWriteUnicode(writer, pUser->name);
        cmBufferWriteUint16(writer, (NQ_UINT16)pFile->isDirectory);

        TRCE();
        return TRUE;
    }

    cmBufferWriteInt32(writer, NQ_ERR_ERROR);
    TRCE();
    return TRUE;
}

#ifdef UD_NQ_INCLUDESMB1
static NQ_BOOL setSMB1Support(CMBufferReader * reader, CMBufferWriter * writer)
{
    NQ_UINT16   support;
    NQ_BOOL     isSupport;
    NQ_STATUS   status;

    cmBufferReadUint16(reader , &support);
    if (0 == support)
    {
        isSupport = FALSE;
    }
    else
    {
        isSupport = TRUE;
    }

    status = udDefSetServerSMB1Support(isSupport);
    cmBufferWriteInt32(writer, status);

    return TRUE;
}
#endif /* UD_NQ_INCLUDESMB1 */

static NQ_BOOL changeAuthenticationEncryptionLevel(CMBufferReader * reader, CMBufferWriter * writer)
{
    NQ_UINT16 maxLevel;
    NQ_UINT16 newLevel;
    NQ_BOOL result;

    cmBufferReadUint16(reader , &maxLevel);
    switch (maxLevel)
    {
        case CS_CONTROL_MAX_LEVEL_AUTH_ENCRYPTION_LM:
        {
            newLevel = CS_AUTH_ENCRYPTION_LM;
            break;
        }

        case CS_CONTROL_MAX_LEVEL_AUTH_ENCRYPTION_NTLM:
        {
            newLevel = CS_AUTH_ENCRYPTION_LM | CS_AUTH_ENCRYPTION_NTLM;
            break;
        }
        case CS_CONTROL_MAX_LEVEL_AUTH_ENCRYPTION_LMV2:
        {
            newLevel = CS_AUTH_ENCRYPTION_LM | CS_AUTH_ENCRYPTION_NTLM | CS_AUTH_ENCRYPTION_LMV2;
            break;
        }

        case CS_CONTROL_MAX_LEVEL_AUTH_ENCRYPTION_NTLMV2:
        {
            newLevel = CS_AUTH_ENCRYPTION_ALL;
            break;
        }

        default:
        {
            newLevel = CS_AUTH_ENCRYPTION_ALL;
            break;
        }
    }

    result = csChangeAuthenticationEncryptionLevel(newLevel);
    cmBufferWriteInt32(writer, result ? NQ_SUCCESS : NQ_ERR_ERROR);
    return TRUE;
}

#ifdef UD_CS_MESSAGESIGNINGPOLICY
static NQ_BOOL changeMsgSign(CMBufferReader * reader, CMBufferWriter * writer)
{
    NQ_UINT16 newPolicy;

    cmBufferReadUint16(reader , &newPolicy);
    csSetMessageSigningPolicy((NQ_INT)newPolicy);

    cmBufferWriteUint32(writer, NQ_SUCCESS);

    return TRUE;
}
#endif /*UD_CS_MESSAGESIGNINGPOLICY*/

#endif /* UD_CS_INCLUDECONTROL */

#endif /* UD_NQ_INCLUDECIFSSERVER */
