/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Implementation of Open and Create commands
 *--------------------------------------------------------------------
 * MODULE        : Server
 * DEPENDENCIES  :
 ********************************************************************/

#include "csdataba.h"
#include "csparams.h"
#include "csdispat.h"
#include "csutils.h"
#include "cstrans2.h"
#include "csnttran.h"
#include "csnotify.h"
#include "csutils.h"
#include "csdcerpc.h"
#include "cscreate.h"
#include "csbreak.h"
#include "csinform.h"

#ifdef UD_NQ_INCLUDECIFSSERVER

/* This code implements file open and create commands over different subprotocols.
   These commands share the same local functions */

/*
    A couple of the following tables define what happens when there are multiple
    openings of the same file. Each attempt has an open mode and a deny mode. So,
    the result is determined by four values: old (previous) open mode, old deny mode,
    new open mode and new deny mode.

    The main code table (grantedAccessTable) defines access, granted for each
    combination of the above values. This tables is packed, meaning that its
    dimensions are all used. For this reason we encode open mode and deny mode
    into packed values (values, starting from 0 and using all numbers).

    Each element in the table defines four values: granted access for a
    combination of EXE/non EXE files and same client/different clients.
 */

/* packed values fro deny mode */
#define DENY_COM   0  /* compatibility mode index */
#define DENY_ALL   1  /* index for deny read/write/execute (exclusive) */
#define DENY_WRITE 2  /* index for deny write */
#define DENY_READ  3  /* index for deny read/execute */
#define DENY_NONE  4  /* index for deny none */
#define DENY_FCB   5  /* index for deny FCB */
#define DENY_SIZE  6  /* size of the deny dimension */

/* packed values for open mode */
#define ACCESS_RDWR    0 /* index for read/write access */
#define ACCESS_RDONLY  1 /* index for read access */
#define ACCESS_WRONLY  2 /* index for write access */
#define ACCESS_EXECUTE 3 /* index for execute access */
#define ACCESS_SIZE    4 /* size of the access dimension */

/* granted access values */
#define GRANT_NONE 0xF  /* no access can be granted, this is internally used*/
#define GRANT_READ SMB_ACCESS_A_READ
#define GRANT_WRIT SMB_ACCESS_A_WRITE
#define GRANT_RDWR SMB_ACCESS_A_READWRITE

/* table for encoding SMB deny modes into our "packed" deny modes. A value in the table is
   a "packed" mode. An index in the table is an SMB mode. */
static const NQ_UINT16 denyCodeTable[] = {
    DENY_COM,       /* 0x0 */
    DENY_ALL,       /* 0x1 */
    DENY_WRITE,     /* 0x2 */
    DENY_READ,      /* 0x3 */
    DENY_NONE,      /* 0x4 */
    (NQ_UINT16)-1,  /* 0x5 */
    (NQ_UINT16)-1,  /* 0x6 */
    (NQ_UINT16)-1,  /* 0x7 */
    DENY_COM,       /* 0x8 */
    DENY_ALL,       /* 0x9 */
    DENY_WRITE,     /* 0xA */
    DENY_READ,      /* 0xB */
    DENY_NONE,      /* 0xC */
    (NQ_UINT16)-1,  /* 0xD */
    (NQ_UINT16)-1,  /* 0xE */
    DENY_FCB        /* 0xF */
};

/* table for encoding SMB open modes into our "packed" open modes. A value in the table is
   a "packed" mode. An index in the table is an SMB mode. */
static const NQ_UINT16 accessCodeTable[] = {
    ACCESS_RDONLY,  /* 0x0 */
    ACCESS_WRONLY,  /* 0x1 */
    ACCESS_RDWR,    /* 0x2 */
    ACCESS_EXECUTE, /* 0x3 */
    ACCESS_RDONLY,  /* 0x4 */
    ACCESS_RDONLY,  /* 0x5 */
    ACCESS_RDONLY,  /* 0x6 */
    ACCESS_RDONLY,  /* 0x7 */
    ACCESS_RDONLY,  /* 0x8 */
    ACCESS_WRONLY,  /* 0x9 */
    ACCESS_RDWR,    /* 0xA */
    ACCESS_EXECUTE, /* 0xB */
    ACCESS_RDONLY,  /* 0xC */
    ACCESS_RDONLY,  /* 0xD */
    ACCESS_RDONLY,  /* 0xE */
    ACCESS_RDWR     /* 0xF */
};

/* An element of the granted access table */
static struct
{
    NQ_BYTE nonSame;    /* granted access: non-EXE, same client */
    NQ_BYTE exeSame;    /* granted access: EXE, same client */
    NQ_BYTE nonDiff;    /* granted access: non-EXE, different clients */
    NQ_BYTE exeDiff;    /* granted access: EXE, different clients */
} const grantedAccessTable[] = {

{ /*  ACCESS_RDWR,   DENY_COM,   ACCESS_RDWR,   DENY_COM, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_WRONLY,   DENY_COM, */ GRANT_WRIT, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*  ACCESS_RDWR,   DENY_COM,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_COM,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_COM,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_COM,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*  ACCESS_RDWR,   DENY_COM,   ACCESS_RDWR,   DENY_FCB, */ GRANT_RDWR, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_RDONLY,   DENY_FCB, */ GRANT_RDWR, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_WRONLY,   DENY_FCB, */ GRANT_RDWR, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_COM, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_RDWR, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_COM,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_RDONLY,   DENY_COM,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_COM,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_RDONLY, DENY_WRITE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_RDONLY,   DENY_COM,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_COM,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_RDONLY,  DENY_NONE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_EXECUTE, DENY_NONE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_RDONLY,   DENY_COM,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_COM, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_COM,   ACCESS_RDWR,   DENY_COM, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_WRONLY,   DENY_COM, */ GRANT_WRIT, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_WRONLY,   DENY_COM,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_COM,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_COM,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_WRONLY,   DENY_COM,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },

{ /*ACCESS_WRONLY,   DENY_COM,   ACCESS_RDWR,   DENY_FCB, */ GRANT_RDWR, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_RDONLY,   DENY_FCB, */ GRANT_RDWR, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_WRONLY,   DENY_FCB, */ GRANT_RDWR, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_COM, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_RDWR, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_COM,   ACCESS_RDWR,   DENY_COM, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_WRONLY,   DENY_COM, */ GRANT_WRIT, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_EXECUTE,  DENY_COM,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_COM,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_COM,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_READ },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_READ },

{ /*ACCESS_EXECUTE,  DENY_COM,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_EXECUTE,  DENY_COM,   ACCESS_RDWR,   DENY_FCB, */ GRANT_RDWR, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_RDONLY,   DENY_FCB, */ GRANT_READ, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_WRONLY,   DENY_FCB, */ GRANT_WRIT, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_COM, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_RDWR, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_ALL,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_ALL,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_ALL,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_ALL,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_ALL,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_ALL,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_ALL, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_ALL,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_ALL,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_ALL,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_ALL,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_ALL,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_ALL,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_ALL, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_ALL,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_ALL,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_ALL,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_ALL,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_ALL,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_ALL,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_ALL, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_ALL,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_ALL,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_ALL,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_ALL,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_ALL,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_ALL,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_ALL, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR, DENY_WRITE,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*  ACCESS_RDWR, DENY_WRITE,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR, DENY_WRITE,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR, DENY_WRITE,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR, DENY_WRITE,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_RDONLY,  DENY_NONE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_EXECUTE, DENY_NONE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*  ACCESS_RDWR, DENY_WRITE,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR, DENY_WRITE, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY, DENY_WRITE,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_RDONLY, DENY_WRITE,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY, DENY_WRITE,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_RDONLY, DENY_WRITE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_RDONLY, DENY_WRITE,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY, DENY_WRITE,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_RDONLY,  DENY_NONE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_EXECUTE, DENY_NONE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_RDONLY, DENY_WRITE,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY, DENY_WRITE, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY, DENY_WRITE,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_WRONLY, DENY_WRITE,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY, DENY_WRITE,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY, DENY_WRITE,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_RDONLY,  DENY_READ, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_EXECUTE, DENY_READ, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_WRONLY, DENY_WRITE,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_RDONLY,  DENY_NONE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_EXECUTE, DENY_NONE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_WRONLY, DENY_WRITE,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY, DENY_WRITE, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,DENY_WRITE,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_RDONLY,   DENY_COM, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_EXECUTE,DENY_WRITE,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,DENY_WRITE,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_RDONLY, DENY_WRITE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_EXECUTE,DENY_WRITE,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,DENY_WRITE,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_RDONLY,  DENY_NONE, */ GRANT_READ, GRANT_READ, GRANT_NONE , GRANT_READ },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_EXECUTE, DENY_NONE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_EXECUTE,DENY_WRITE,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,DENY_WRITE, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,  DENY_READ,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,  DENY_READ,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,  DENY_READ,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,  DENY_READ,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,  DENY_READ,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_WRONLY,  DENY_NONE, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,  DENY_READ,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_READ, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,  DENY_READ,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,  DENY_READ,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,  DENY_READ,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_WRONLY, DENY_WRITE, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,  DENY_READ,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,  DENY_READ,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_WRONLY,  DENY_NONE, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,  DENY_READ,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_READ, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,  DENY_READ,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,  DENY_READ,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,  DENY_READ,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,  DENY_READ,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_WRONLY,  DENY_READ, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,  DENY_READ,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_WRONLY,  DENY_NONE, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,  DENY_READ,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_READ, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE, DENY_READ,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE, DENY_READ,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE, DENY_READ,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_WRONLY, DENY_WRITE, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE, DENY_READ,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE, DENY_READ,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_WRONLY,  DENY_NONE, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE, DENY_READ,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_READ, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,  DENY_NONE,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*  ACCESS_RDWR,  DENY_NONE,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,  DENY_NONE,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,  DENY_NONE,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,  DENY_NONE,   ACCESS_RDWR,  DENY_NONE, */ GRANT_RDWR, GRANT_RDWR, GRANT_RDWR, GRANT_RDWR },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_RDONLY,  DENY_NONE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_WRONLY,  DENY_NONE, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_EXECUTE, DENY_NONE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*  ACCESS_RDWR,  DENY_NONE,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,  DENY_NONE, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,  DENY_NONE,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_RDONLY,  DENY_NONE,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,  DENY_NONE,   ACCESS_RDWR, DENY_WRITE, */ GRANT_RDWR, GRANT_RDWR, GRANT_RDWR, GRANT_RDWR },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_RDONLY, DENY_WRITE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_WRONLY, DENY_WRITE, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_RDONLY,  DENY_NONE,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,  DENY_NONE,   ACCESS_RDWR,  DENY_NONE, */ GRANT_RDWR, GRANT_RDWR, GRANT_RDWR, GRANT_RDWR },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_RDONLY,  DENY_NONE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_WRONLY,  DENY_NONE, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_EXECUTE, DENY_NONE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_RDONLY,  DENY_NONE,   ACCESS_RDWR,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,  DENY_NONE, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,  DENY_NONE,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_RDONLY,   DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_WRONLY,  DENY_NONE,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,  DENY_NONE,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,  DENY_NONE,   ACCESS_RDWR,  DENY_READ, */ GRANT_RDWR, GRANT_RDWR, GRANT_RDWR, GRANT_RDWR },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_RDONLY,  DENY_READ, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_WRONLY,  DENY_READ, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_EXECUTE, DENY_READ, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_WRONLY,  DENY_NONE,   ACCESS_RDWR,  DENY_NONE, */ GRANT_RDWR, GRANT_RDWR, GRANT_RDWR, GRANT_RDWR },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_RDONLY,  DENY_NONE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_WRONLY,  DENY_NONE, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_EXECUTE, DENY_NONE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_WRONLY,  DENY_NONE,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,  DENY_NONE, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE, DENY_NONE,   ACCESS_RDWR,   DENY_COM, */ GRANT_NONE, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_WRONLY,   DENY_COM, */ GRANT_NONE, GRANT_WRIT, GRANT_NONE, GRANT_WRIT },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_EXECUTE,  DENY_COM, */ GRANT_NONE, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_EXECUTE, DENY_NONE,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE, DENY_NONE,   ACCESS_RDWR, DENY_WRITE, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_RDWR },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_RDONLY, DENY_WRITE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_WRONLY, DENY_WRITE, */ GRANT_WRIT, GRANT_WRIT, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_READ },

{ /*ACCESS_EXECUTE, DENY_NONE,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE, DENY_NONE,   ACCESS_RDWR,  DENY_NONE, */ GRANT_RDWR, GRANT_RDWR, GRANT_RDWR, GRANT_RDWR },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_RDONLY,  DENY_NONE, */ GRANT_READ, GRANT_READ, GRANT_READ, GRANT_READ },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_WRONLY,  DENY_NONE, */ GRANT_WRIT, GRANT_WRIT, GRANT_WRIT, GRANT_WRIT },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_EXECUTE, DENY_NONE, */ GRANT_RDWR, GRANT_READ, GRANT_RDWR, GRANT_READ },

{ /*ACCESS_EXECUTE, DENY_NONE,   ACCESS_RDWR,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_RDONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_WRONLY,   DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE, DENY_NONE, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_FCB,   ACCESS_RDWR,   DENY_COM, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_WRONLY,   DENY_COM, */ GRANT_WRIT, GRANT_WRIT, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_EXECUTE,  DENY_COM, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_FCB,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_FCB,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_FCB,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_FCB,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*  ACCESS_RDWR,   DENY_FCB,   ACCESS_RDWR,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_RDONLY,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_WRONLY,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*  ACCESS_RDWR,   DENY_FCB, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_FCB,   ACCESS_RDWR,   DENY_COM, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_WRONLY,   DENY_COM, */ GRANT_WRIT, GRANT_WRIT, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_EXECUTE,  DENY_COM, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_FCB,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_FCB,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_FCB,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_FCB,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_RDONLY,   DENY_FCB,   ACCESS_RDWR,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_RDONLY,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_WRONLY,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_RDONLY,   DENY_FCB, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_FCB,   ACCESS_RDWR,   DENY_COM, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_WRONLY,   DENY_COM, */ GRANT_WRIT, GRANT_WRIT, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_EXECUTE,  DENY_COM, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_FCB,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_FCB,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_FCB,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_FCB,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_WRONLY,   DENY_FCB,   ACCESS_RDWR,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_RDONLY,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_WRONLY,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_WRONLY,   DENY_FCB, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_FCB,   ACCESS_RDWR,   DENY_COM, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_RDONLY,   DENY_COM, */ GRANT_READ, GRANT_READ, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_WRONLY,   DENY_COM, */ GRANT_WRIT, GRANT_WRIT, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_EXECUTE,  DENY_COM, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_FCB,   ACCESS_RDWR,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_RDONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_WRONLY,   DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_EXECUTE,  DENY_ALL, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_FCB,   ACCESS_RDWR, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_RDONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_WRONLY, DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_EXECUTE,DENY_WRITE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_FCB,   ACCESS_RDWR,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_RDONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_WRONLY,  DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_EXECUTE, DENY_READ, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_FCB,   ACCESS_RDWR,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_RDONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_WRONLY,  DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_EXECUTE, DENY_NONE, */ GRANT_NONE, GRANT_NONE, GRANT_NONE, GRANT_NONE },

{ /*ACCESS_EXECUTE,  DENY_FCB,   ACCESS_RDWR,   DENY_FCB, */ GRANT_RDWR, GRANT_RDWR, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_RDONLY,   DENY_FCB, */ GRANT_READ, GRANT_READ, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_WRONLY,   DENY_FCB, */ GRANT_WRIT, GRANT_WRIT, GRANT_NONE, GRANT_NONE },
{ /*ACCESS_EXECUTE,  DENY_FCB, ACCESS_EXECUTE,  DENY_FCB, */ GRANT_RDWR, GRANT_READ, GRANT_NONE, GRANT_NONE },


};

static NQ_WCHAR fileNameBuff[CM_MAXFILENAMELEN];

#define MAX_NUM_OPLOCK_OPEN_FILES           ((UD_FS_NUMSERVERFILEOPEN * 3) / 4)
#define MAX_NUM_OPLOCK_OPEN_UNIQUE_FILES    ((UD_FS_NUMSERVERFILENAMES * 3) / 4)
/*
    Static functions
    ----------------
 */

/* open file and allocate FID */

static NQ_UINT32                          /* error code or 0 */
openFile(
    const NQ_WCHAR* pFileName,  /* file name */
    CSTid tid,                  /* master tree ID */
    NQ_UINT16 desiredAccess,    /* desired access as in the request */
    NQ_UINT16* grantedAccess,   /* place for granted access */
    CSFile** pFile              /* OUT: file descriptor */
    );

/* create file or directory and allocate FID */

static NQ_UINT32                   /* error code or 0 */
createFile(
    NQ_BOOL directoryRequired,  /* TRUE for directory */
    const NQ_WCHAR* pFileName,   /* file name */
    CSTid tid,                  /* master tree ID */
    NQ_UINT16 desiredAccess,    /* file access bits */
    CSFile** pFile              /* OUT: file descriptor */
    );

/* compose an OPEN-style desired access from NT-style parameters */

static NQ_UINT16                           /* desired access */
convertNtAccessToDesiredAccess(
    NQ_UINT32 ntAccess,             /* desired access in an NT-style */
    NQ_UINT32 ntShare               /* deny flags */
    );

/* check if another open may happen */

static NQ_UINT32                    /* granted access */
isAccessAllowed(
    CSName* pName,                  /* name descriptor pointer */
    NQ_UINT16* newAccess,           /* desired access, may be modified */
    CSUid newUid,                   /* new client UID */
    NQ_BOOL readOnly                /* whether file is read-only */
    );

#ifdef UD_CS_INCLUDERPC_SPOOLSS
/* create a file for printing */

static NQ_UINT32                    /* error code or 0 */
createPrintFile(
    const NQ_WCHAR* pFileName,      /* file name */
    const CSShare* pShare,          /* pointer to the share */
    CSTid tid,                      /* master tree ID */
    NQ_UINT16 desiredAccess,        /* file access bits */
    CSFile** pFile,                 /* OUT: file descriptor */
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
    NQ_BOOL createSpoolerFile,      /* whether to create spooler file (SMB1 not relevant) */
#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
    CSUser * pUser                  /* user context */
    );
#endif

/* desired access bits for exclusive access */

#define EXCLUSIVE_ACCESS (SMB_ACCESS_A_READWRITE | SMB_ACCESS_S_READWRITEEXECUTE)

/*====================================================================
 * PURPOSE: Perform OPEN command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComOpen(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsOpenFileRequest* openRequest = NULL;     /* casted request */
    CMCifsOpenFileResponse* openResponse = NULL;   /* casted response */
    NQ_UINT32 returnValue;                  /* error code in NT format or 0 for no error */
    NQ_BOOL unicodeRequired;                /* whether client requires UNICODE */
    CMCifsStatus error;                     /* for composing DOS-style error */
    CSFile* pFile;                          /* pointer to file descriptor */
    const CSShare* pShare;                  /* pointer to the share */
    NQ_WCHAR* pFileName = NULL;                    /* filename to open */
    CSUid uid;                              /* required UID */
    CSTid tid;                              /* required TID */
    CSPid pid;                              /* required PID */
    SYFileInformation fileInfo;             /* for querying file information */
    NQ_UINT16 desiredAccess;                /* as in the request */
    NQ_UINT16 grantedAccess;                /* as in the response */
    CSUser* pUser;                          /* pointer to the user descriptor */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    UDFileAccessEvent   eventInfo;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    TRCB();

    /* check space in output buffer */

    if ((returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, sizeof(*openResponse))
        ) != 0
       )
    {
        TRCE();
        return returnValue;
    }

    /* read unicode flag */

    unicodeRequired = cmLtoh16(cmGetSUint16(pHeaderOut->flags2)) & SMB_FLAGS2_UNICODE;

    /* withdraw UID, TID and PID */

    uid = (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid));
    tid = (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    pid = csGetPidFromHeader(pHeaderOut);

    pUser = csGetUserByUid(uid);
    if (pUser == NULL)
    {
        TRCERR("Illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvuid);
    }

    pShare = csGetShareByUidTid(uid, tid);
    if (pShare == NULL)
    {
        TRCERR("Illegal TID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvtid);
    }
    /* we do not open pseudo-files */

#ifndef UD_CS_INCLUDERPC
    if (pShare->ipcFlag)
    {
        TRCERR("Pseudo-files (pipes) not supported");
        TRCE();
        return csErrorReturn(SMB_STATUS_NOT_SUPPORTED, (NQ_UINT32)SRV_ERRnosupport);
    }
#endif

    /* cast pointers */

    openRequest = (CMCifsOpenFileRequest*) pRequest;
    openResponse = (CMCifsOpenFileResponse*) *pResponse;

    /* check counts */

    if (   openRequest->wordCount != SMB_OPEN_REQUEST_WORDCOUNT
        || cmLtoh16(cmGetSUint16(openRequest->byteCount)) < SMB_OPEN_REQUEST_MINBYTES)
    {
        TRCERR("Illegal WordCount or ByteCount ");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* convert filename to host filename */

    if ((pFileName = cmCifsNtohFilename(
                        fileNameBuff,
                        pShare->map,
                        (NQ_WCHAR*)(openRequest + 1),
                        unicodeRequired,
                        TRUE
                        )
        ) == NULL
       )
    {
        TRCERR("Illegal filename");
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.tid = tid;
    eventInfo.rid = csGetUserRid(pUser);
    eventInfo.fileName = pFileName;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        pUser->preservesCase ? UD_LOG_FILE_ATTRIBGET : (pShare != NULL && syWStrcmp(pShare->map , pFileName) == 0) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    
    /* look for the file */

    if (!csCheckPathAndFile(pShare, pFileName, (NQ_UINT)syWStrlen(pShare->map), pUser->preservesCase))
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            pUser->preservesCase ? UD_LOG_FILE_ATTRIBGET : (pShare != NULL && syWStrcmp(pShare->map , pFileName) == 0) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile),
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        TRCERR("file does not exist");
        TRC1P(" file: %s", cmWDump(pFileName));
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        pUser->preservesCase ? UD_LOG_FILE_ATTRIBGET : (pShare != NULL && syWStrcmp(pShare->map , pFileName) == 0) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

     /* do not allow open for directory */
        
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_ATTRIBGET,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (syGetFileInformationByName(pFileName, &fileInfo) != NQ_SUCCESS)
    {
        error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_ATTRIBGET,
            pUser->name,
            pUser->ip,
            error,
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        TRCERR("Unable to read file information");
        TRCE();
        return error;
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_ATTRIBGET,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (fileInfo.attributes & SMB_ATTR_DIRECTORY)  
    {
        TRCERR("Not a file");
        TRCE();
        return csErrorReturn(SMB_STATUS_FILE_IS_A_DIRECTORY, DOS_ERRnoaccess);
    }

    /* open file */

    desiredAccess = (NQ_UINT16)cmLtoh16(cmGetSUint16(openRequest->desiredAccess));

    returnValue = openFile(pFileName, tid, desiredAccess, &grantedAccess, &pFile);
    if (returnValue != 0)
    {
        TRCE();
        return returnValue;
    }

    /* store file information */

    pFile->tid = tid;
    pFile->uid = uid;
    pFile->pid = pid;

    /* get the file attributes */

    if (csGetFileInformation(pFile, pFileName, &fileInfo) != NQ_SUCCESS)
    {
        error = csErrorGetLast();
        csReleaseFile(pFile->fid);      /* also closes the file */
        TRCERR("Unable to read file information");

        return error;
    }

    /* compose the response */

    openResponse->wordCount = SMB_OPEN_RESPONSE_WORDCOUNT;
    cmPutSUint16(openResponse->byteCount, 0);

    cmPutSUint16(openResponse->fid, cmHtol16(pFile->fid));
    cmPutSUint16(openResponse->fileAttributes, cmHtol16((NQ_UINT16)fileInfo.attributes));

    {
        NQ_UINT16 smbTime;     /* temporary time in SMB_TIME format */
        NQ_UINT16 smbDate;     /* temporary date in SMB_DATE format */

        cmCifsTimeToSmbTime(fileInfo.lastWriteTime, &smbTime, &smbDate);
        cmPutSUint16(openResponse->lastWriteTime, cmHtol16(smbTime));
        cmPutSUint16(openResponse->lastWriteDate, cmHtol16(smbDate));
    }

    cmPutSUint32(openResponse->dataSize, cmHtol32(fileInfo.sizeLow));
    cmPutSUint16(openResponse->grantedAccess, cmHtol16(grantedAccess));

    /* advance the outgoing response pointer */

    *pResponse += sizeof(*openResponse);

    TRCE();
    return 0;
}

/*====================================================================
 * PURPOSE: Perform OPEN_ANDX command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComOpenAndX(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsOpenAndXRequest* openRequest;     /* casted request */
    CMCifsOpenAndXResponse* openResponse;   /* casted response */
    NQ_UINT32 returnValue;                  /* error code in NT format or 0 for no error */
    NQ_BOOL unicodeRequired;                /* whether client requires UNICODE */
    CMCifsStatus error;                     /* for composing DOS-style error */
    CSFile* pFile;                          /* pointer to file descriptor */
    const CSShare* pShare;                  /* pointer to the share */
    NQ_WCHAR* pFileName;                    /* filename to open */
    CSUid uid;                              /* required UID */
    CSTid tid;                              /* required TID */
    CSPid pid;                              /* required PID */
    SYFileInformation fileInfo;             /* for querying file information */
    NQ_UINT16 desiredAction;                /* action from  request */
    NQ_UINT16 takenAction;                  /* action in the response */
    NQ_UINT16 desiredAccess;                /* as in the request */
    NQ_UINT16 grantedAccess;                /* for the response */
    NQ_BOOL fileExists;                     /* whether the file exists */
    CSUser* pUser;                          /* pointer to the user descriptor */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent eventInfo;    /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */


    TRCB();

    /* check space in output buffer and set unicode support */

    if ((returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, sizeof(*openResponse))
        ) != 0
       )
    {
        TRCE();
        return returnValue;
    }

    /* read unicode flag */

    unicodeRequired = cmLtoh16(cmGetSUint16(pHeaderOut->flags2)) & SMB_FLAGS2_UNICODE;

    /* cast pointers */

    openRequest = (CMCifsOpenAndXRequest*) pRequest;
    openResponse = (CMCifsOpenAndXResponse*) *pResponse;

    /* check counts */

    if (   openRequest->wordCount != SMB_OPENANDX_REQUEST_WORDCOUNT
        || cmLtoh16(cmGetSUint16(openRequest->byteCount)) < SMB_OPENANDX_REQUEST_MINBYTES)
    {
        TRCERR("Illegal WordCount or ByteCount ");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* check the next X command */

    switch (openRequest->andXCommand)
    {
    case SMB_COM_READ:
    case SMB_COM_READ_ANDX:
    case SMB_COM_IOCTL:
    case 0xFF:
        break;
    default:
        TRCERR("invalid next command");
        TRC1P("  value: %d", openRequest->andXCommand);
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* withdraw UID, TID and PID */

    uid = (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid));
    tid = (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    pid = csGetPidFromHeader(pHeaderOut);

    pUser = csGetUserByUid(uid);
    if (pUser == NULL)
    {
        TRCERR("Illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvuid);
    }

    pShare = csGetShareByUidTid(uid, tid);
    if (pShare == NULL)
    {
        TRCERR("Illegal TID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvtid);
    }


    /* we do not open pseudo-files */

#ifndef UD_CS_INCLUDERPC
    if (pShare->ipcFlag)
    {
        TRCERR("Pseudo-files (pipes) not supported");
        TRCE();
        return csErrorReturn(SMB_STATUS_NOT_SUPPORTED, (NQ_UINT32)SRV_ERRnosupport);
    }
#endif

    /* convert filename to host filename */

    if ((pFileName = cmCifsNtohFilename(
                        fileNameBuff,
                        pShare->map,
                        (NQ_WCHAR*)(openRequest + 1),
                        unicodeRequired,
                        TRUE
                        )
        ) == NULL
       )
    {
        TRCERR("Illegal filename");
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname);
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.tid = tid;
    eventInfo.rid = csGetUserRid(pUser);
    eventInfo.fileName = pFileName;
    eventInfo.access = cmLtoh16(cmGetSUint16(openRequest->desiredAccess));
#endif /* UD_NQ_INCLUDEEVENTLOG */
    /* look for the path */

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (!csCheckPath(pShare, pFileName, (NQ_UINT)syWStrlen(pShare->map), pUser->preservesCase))
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        eventInfo.before = TRUE;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
            (const NQ_BYTE*)&eventInfo
            );
        eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        TRCERR("path does not exist");
        TRC1P(" path: %s", cmWDump(pFileName));
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_CS_INCLUDERPC_SPOOLSS      
    if (pShare->isPrintQueue)
    {
        if (createPrintFile(pFileName, pShare, tid, cmLtoh16(cmGetSUint16(openRequest->desiredAccess)), &pFile,
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
                FALSE,
#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
                pUser) != NQ_SUCCESS)
        {
            TRCERR("Failed to create a printer file");
            TRCE();
            return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        }
        
        grantedAccess = SMB_ACCESS_A_READ;
        takenAction = SMB_OPENANDX_WASCREATED;
        syMemset(&fileInfo, 0, sizeof(fileInfo));          
    }
    else
#endif           
    {      
        /* open file */
        desiredAccess = (NQ_UINT16)cmLtoh16(cmGetSUint16(openRequest->desiredAccess));
        desiredAction = (NQ_UINT16)cmLtoh16(cmGetSUint16(openRequest->openFunction));

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        eventInfo.before = TRUE;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
            );
        eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        fileExists = csCheckFile(pShare, pFileName, pUser->preservesCase);
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        /* if file does not exists take actions as required in the
           request */

        if (!fileExists)
        {
            /* file does not exist */
            if ((desiredAction & SMB_OPENANDX_CREATEACTION) != SMB_OPENANDX_DOCREATE)
            {
                TRCERR("file not open and open function does not allow creation");
                TRCE();
                return csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
            }

            /* create file */

            TRC1P("Creating file: %s", (NQ_CHAR*)pFileName);                                       

            if ((returnValue = createFile(FALSE, pFileName, tid, desiredAccess, &pFile)) != 0)   
            {
                TRCE();
                return returnValue;
            }

            /* set file attributes */
            {
                SYFileInformation fileInfo;         /* buffer for file information */
                SYFile nullHandle;  /* invalid handle to the file */

                syInvalidateFile(&nullHandle);
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                eventInfo.before = TRUE;
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_ATTRIBGET,
                    pUser->name,
                    pUser->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                    );
                eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
                if (syGetFileInformationByName(pFileName, &fileInfo) != NQ_SUCCESS)
                {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_ATTRIBGET,
                        pUser->name,
                        pUser->ip,
                        csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile),
                        (const NQ_BYTE*)&eventInfo
                        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    TRCERR("cannot read file information");
                    TRCE();
                    return csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
                }
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_ATTRIBGET,
                    pUser->name,
                    pUser->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                fileInfo.attributes = (NQ_UINT32)cmLtoh16(cmGetSUint16(openRequest->fileAttributes));
                fileInfo.attributes |= SMB_ATTR_ARCHIVE;
#ifdef UD_NQ_INCLUDEEVENTLOG
                eventInfo.before = TRUE;
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_ATTRIBSET,
                    pUser->name,
                    pUser->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                    );
                eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
                if (NQ_FAIL == sySetFileInformation(pFileName, nullHandle,  &fileInfo))
                {
#ifdef UD_NQ_INCLUDEEVENTLOG
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_ATTRIBSET,
                        pUser->name,
                        pUser->ip,
                        0,
                        (const NQ_BYTE*)&eventInfo
                        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    TRCERR("cannot modify file information");
                    TRCE();
                    return csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
                }
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_ATTRIBSET,
                    pUser->name,
                    pUser->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            }
            grantedAccess = SMB_ACCESS_A_READWRITE;
            takenAction = SMB_OPENANDX_WASCREATED;
        }
        else
        {
            /* file exists: proceed according to the required "Open" action */

            if ((desiredAction & SMB_OPENANDX_OPENACTION) == SMB_OPENANDX_DOFAIL)
            {
                TRCERR("file exists while the request does not allow open");
                TRCE();
                return csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists);
            }

            /* do not allow open for directory */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            eventInfo.before = TRUE;
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_ATTRIBGET,
                pUser->name,
                pUser->ip,
                0,
                (const NQ_BYTE*)&eventInfo
                );
            eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
            if (syGetFileInformationByName(pFileName, &fileInfo) != NQ_SUCCESS)              
            {
                error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_ATTRIBGET,
                    pUser->name,
                    pUser->ip,
                    error,
                    (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("Unable to read file information");
                TRCE();
                return error;
            }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_ATTRIBGET,
                pUser->name,
                pUser->ip,
                0,
                (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            if (fileInfo.attributes & SMB_ATTR_DIRECTORY)  
            {
                TRCERR("Not a file");
                TRCE();
                return csErrorReturn(SMB_STATUS_FILE_IS_A_DIRECTORY, DOS_ERRnoaccess);
            }

            /* do not allow overwriting read only file */
            
            if ((desiredAction & SMB_OPENANDX_DOTRUNCATE) && (fileInfo.attributes & SMB_ATTR_READONLY))
            {
               TRCERR("Attempt to overwrite read only file");
               TRCE();
               return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
            }
            
            if ((returnValue = openFile(pFileName, tid, desiredAccess, &grantedAccess, &pFile)) != 0)   
            {
                TRCE();
                return returnValue;
            }

            takenAction = SMB_OPENANDX_WASOPENED;

            if (desiredAction & SMB_OPENANDX_DOTRUNCATE)
            {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                eventInfo.offsetLow = 0;
                eventInfo.offsetHigh = 0;
                eventInfo.before = TRUE;
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_SIZESET,
                    pUser->name,
                    pUser->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                    );
                eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
                if (syTruncateFile(pFile->file, 0, 0) == NQ_FAIL)
                {
                    error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_SIZESET,
                        pUser->name,
                        pUser->ip,
                        error,
                        (const NQ_BYTE*)&eventInfo
                        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    csReleaseFile(pFile->fid);      /* also closes the file */
                    TRCERR("cannot truncate the file");
                    TRCE();
                    return error;
                }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_SIZESET,
                    pUser->name,
                    pUser->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                takenAction = SMB_OPENANDX_WASTRUNCATED;
            }
        }

        /* signal to client the read-ahead possibility */

#ifdef UD_FS_READAHEAD
        if (pUser->supportsReadAhead)
        {
            takenAction  |= SMB_OPENANDX_WASLOCKED;
        }
#endif
        
        /* get the file attributes */

        if (NQ_SUCCESS != csGetFileInformationByName(pShare, pFileName, &fileInfo
#ifdef UD_NQ_INCLUDEEVENTLOG
                ,pUser
#endif /* UD_NQ_INCLUDEEVENTLOG */
                ))
        {
            error =  csErrorGetLast();
            csReleaseFile(pFile->fid);      /* also closes the file */
            TRCERR("Unable to read file information");
            TRCE();
            return error;
        }

        /* compare desired attributes with file attributes */

        if (!csMatchFileAttributes(
                (NQ_UINT16)cmLtoh16(cmGetSUint16(openRequest->searchAttributes)),
                (NQ_UINT16)fileInfo.attributes
                )
           )
        {
            csReleaseFile(pFile->fid);

            TRCERR("File attributes mismatch");
            TRC2P("  required: %02x, read: %02x", cmLtoh16(cmGetSUint16(openRequest->searchAttributes)), (NQ_INT)fileInfo.attributes);
            TRCE();
            return csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
        }

        if ((takenAction & SMB_OPENANDX_OPENACTION)== SMB_OPENANDX_WASCREATED)
        {
            NQ_UINT32 creationTime = cmLtoh32(cmGetSUint32(openRequest->creationTime));

            fileInfo.creationTime = cmTimeConvertSecToMSec(creationTime);
            fileInfo.lastAccessTime = cmTimeConvertSecToMSec(creationTime);
            fileInfo.lastChangeTime = cmTimeConvertSecToMSec(creationTime);
            fileInfo.lastWriteTime.high = 0;
            fileInfo.lastWriteTime.low = 0;
            if (csSetFileInformation(pFile, pFileName, &fileInfo))
            {
                csReleaseFile(pFile->fid);      /* also closes the file */

                TRCERR("unable to change file attributes/time");
                TRCE();
                return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
            }
            if (NQ_SUCCESS != csGetFileInformationByName(pShare, pFileName, &fileInfo
#ifdef UD_NQ_INCLUDEEVENTLOG
                    ,pUser
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    ))
            {
                error = csErrorGetLast();
                csReleaseFile(pFile->fid);      /* also closes the file */
                TRCERR("Unable to read file information");
                TRCE();
                return error;
            }
        }
    }

    /* store file information */
    pFile->tid = tid;
    pFile->uid = uid;
    pFile->pid = pid;

    /* advance the outgoing response pointer */

    *pResponse += sizeof(*openResponse);

    /* compose the response */

    openResponse->wordCount = SMB_OPENANDX_RESPONSE_WORDCOUNT;
    openResponse->andXCommand = openRequest->andXCommand;
    openResponse->andXReserved = 0;

    if (openResponse->andXCommand == 0xFF)
    {
        cmPutSUint16(openResponse->andXOffset, 0);
    }
    else
    {
        NQ_UINT16 offset;   /* for calculating offsets */

        offset = (NQ_UINT16)(*pResponse - (NQ_BYTE*)pHeaderOut);
        cmPutSUint16(openResponse->andXOffset, cmHtol16(offset));
    }

    cmPutSUint16(openResponse->fid, cmHtol16(pFile->fid));
    cmPutSUint16(openResponse->grantedAccess, cmHtol16(grantedAccess));
    cmPutSUint16(openResponse->action, cmHtol16(takenAction));
    cmPutSUint32(openResponse->dataSize, cmHtol32(fileInfo.sizeLow));

    /* if additional file information was requested */

    if (cmLtoh16(cmGetSUint16(openRequest->flags)) & SMB_OPENANDX_ADDITIONALINFORMATION)
    {
        NQ_UINT16   temp = fileInfo.attributes & 0x3F;
        cmPutSUint16(openResponse->fileAttributes, cmHtol16(temp));
        cmPutSUint32(openResponse->lastWriteTime, cmHtol32(cmTimeConvertMSecToSec(&fileInfo.lastWriteTime)));
    }
    else
    {
        cmPutSUint16(openResponse->fileAttributes, 0);
        cmPutSUint32(openResponse->lastWriteTime, 0L);
    }

    cmPutSUint16(openResponse->fileType, 0);
    cmPutSUint16(openResponse->deviceState, 0);
    cmPutSUint32(openResponse->serverFid, 0);
    cmPutSUint16(openResponse->reserved, 0);
    cmPutSUint16(openResponse->byteCount, 0);

    TRCE();
    return 0;
}

/*====================================================================
 * PURPOSE: Perform CREATE_NEW command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComCreateNew(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    return csComCreate(pRequest, pHeaderOut, pResponse);
}

/*====================================================================
 * PURPOSE: Perform CREATE command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *
 *          This function works for both CREATE and CREATE_NEW commands
 *====================================================================
 */

NQ_UINT32
csComCreate(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsCreateFileRequest* createRequest;     /* casted request */
    CMCifsCreateFileResponse* createResponse;   /* casted response */
    NQ_UINT32 returnValue;                      /* error code in NT format or 0 for no error */
    NQ_BOOL unicodeRequired;                    /* whether client requires UNICODE */
    CMCifsStatus error;                         /* for composing DOS-style error */
    CSFile* pFile;                              /* pointer to file descriptor */
    const CSShare* pShare;                      /* pointer to the share */
    NQ_WCHAR* pFileName;                        /* filename to open */
    CSUid uid;                                  /* required UID */
    CSTid tid;                                  /* required TID */
    CSPid pid;                                  /* required PID */
    SYFileInformation fileInfo;                 /* for querying file information */
    NQ_BOOL fileExists;                         /* whether the required file exists */
    CSUser* pUser;                              /* pointer to the user descriptor */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent eventInfo;                /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */

    TRCB();

    /* check space in output buffer */

    if ((returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, sizeof(*createResponse))
        ) != 0
       )
    {
        TRCE();
        return returnValue;
    }

    /* read unicode flag */

    unicodeRequired = cmLtoh16(cmGetSUint16(pHeaderOut->flags2)) & SMB_FLAGS2_UNICODE;

    /* cast pointers */

    createRequest = (CMCifsCreateFileRequest*) pRequest;
    createResponse = (CMCifsCreateFileResponse*) *pResponse;

    /* check counts */

    if (   createRequest->wordCount != SMB_CREATE_REQUEST_WORDCOUNT
        || cmLtoh16(cmGetSUint16(createRequest->byteCount)) < SMB_CREATE_REQUEST_MINBYTES)
    {
        TRCERR("Illegal WordCount or ByteCount ");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* withdraw UID, TID and PID */

    uid = (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid));
    tid = (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    pid = csGetPidFromHeader(pHeaderOut);

    pUser = csGetUserByUid(uid);
    if (pUser == NULL)
    {
        TRCERR("Illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvuid);
    }

    pShare = csGetShareByUidTid(uid, tid);
    if (pShare == NULL)
    {
        TRCERR("Illegal TID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvtid);
    }

    /* we do not open pseudo-files */

#ifndef UD_CS_INCLUDERPC
    if (pShare->ipcFlag)
    {
        TRCERR("Pseudo-files (pipes) not supported");
        TRCE();
        return csErrorReturn(SMB_STATUS_NOT_SUPPORTED, (NQ_UINT32)SRV_ERRnosupport);
    }
#endif

    /* convert filename to host filename */

    if ((pFileName = cmCifsNtohFilename(
                        fileNameBuff,
                        pShare->map,
                        (NQ_WCHAR*)(createRequest + 1),
                        unicodeRequired,
                        TRUE
                        )
        ) == NULL
       )
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.tid = tid;
        eventInfo.rid = csGetUserRid(pUser);
        eventInfo.before = TRUE;
        eventInfo.fileName = NULL;
        eventInfo.access = EXCLUSIVE_ACCESS;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Illegal filename");
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname);
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.tid = tid;
    eventInfo.rid = csGetUserRid(pUser);
    eventInfo.fileName = pFileName;
    eventInfo.access = EXCLUSIVE_ACCESS;
#endif /* UD_NQ_INCLUDEEVENTLOG */

    /* look for the path */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (!csCheckPath(pShare, pFileName, (NQ_UINT)syWStrlen(pShare->map), pUser->preservesCase))
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("path does not exist");
        TRC1P(" path: %s", cmWDump(pFileName));
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
    );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    fileExists = csCheckFile(pShare, pFileName, pUser->preservesCase);

    /* for the Create New command check that the file does not exist */

    if (pHeaderOut->command == SMB_COM_CREATE_NEW && fileExists)
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("File already exists");
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists);
    }

    /* create file */

#ifdef UD_CS_INCLUDERPC_SPOOLSS      
    if (pShare->isPrintQueue)
    {
        if (createPrintFile(pFileName, pShare, tid, EXCLUSIVE_ACCESS, &pFile,
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
                FALSE,
#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
                pUser) != NQ_SUCCESS)
        {
            TRCERR("Failed to create a printer file");
            TRCE();
            return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        }
        
        syMemset(&fileInfo, 0, sizeof(fileInfo));          
    }
    else
#endif           
    {
        returnValue = createFile(FALSE, pFileName, tid, EXCLUSIVE_ACCESS, &pFile);                  
        if (returnValue != 0)   
        {
            TRCE();
            return returnValue;
        }

        /* get the file attributes */

        if (csGetFileInformation(pFile, pFileName, &fileInfo) != NQ_SUCCESS)
        {
            error =  csErrorGetLast();
            csReleaseFile(pFile->fid);      /* also closes the file */
            TRCERR("Unable to read file information");
            TRCE();
            return error;
        }

        fileInfo.lastWriteTime.high = 0;
        fileInfo.lastWriteTime.low = 0;
        fileInfo.attributes = (NQ_UINT32)cmLtoh16(cmGetSUint16(createRequest->fileAttributes)) | SMB_ATTR_ARCHIVE;

        if (csSetFileInformation(pFile, pFileName, &fileInfo))
        {
            csReleaseFile(pFile->fid);
            TRCERR("unable to change file attributes/time");
            TRCE();
            return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
        }
    }
    
    /* store file information */

    pFile->tid = tid;
    pFile->uid = uid;
    pFile->pid = pid;

    /* advance the outgoing response pointer */

    *pResponse += sizeof(*createResponse);

    /* compose the response */

    createResponse->wordCount = SMB_CREATE_RESPONSE_WORDCOUNT;
    cmPutSUint16(createResponse->fid, cmHtol16(pFile->fid));
    cmPutSUint16(createResponse->byteCount, 0);

    TRCE();
    return 0;
}

/*====================================================================
 * PURPOSE: Perform CREATE_DIRECTORY command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *
 *          This function works for both CREATE and CREATE_NEW commands
 *====================================================================
 */

NQ_UINT32
csComCreateDirectory(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsCreateDirectoryRequest* createRequest;   /* casted request */
    CMCifsCreateDirectoryResponse* createResponse; /* casted response */
    NQ_UINT32 returnValue;                         /* error code in NT format or 0 for no error */
    NQ_BOOL unicodeRequired;                       /* whether client requires UNICODE */
    CMCifsStatus error;                         /* for composing DOS-style error */
    const CSShare* pShare;                      /* pointer to the share */
    NQ_WCHAR* pFileName;                        /* filename to open */
    CSTid tid;                                  /* tree ID for access check */
    CSUid uid;                                  /* user ID for access check */
    CSUser* pUser;                              /* pointer to the user descriptor */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent eventInfo;                /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */

    TRCB();

    /* check space in output buffer */

    if ((returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, sizeof(*createResponse))
        ) != 0
       )
    {
        TRCE();
        return returnValue;
    }

    /* read unicode flag */

    unicodeRequired = cmLtoh16(cmGetSUint16(pHeaderOut->flags2)) & SMB_FLAGS2_UNICODE;

    /* cast pointers */

    createRequest = (CMCifsCreateDirectoryRequest*) pRequest;
    createResponse = (CMCifsCreateDirectoryResponse*) *pResponse;

    /* check counts */

    if (   createRequest->wordCount != 0
        || cmLtoh16(cmGetSUint16(createRequest->byteCount)) < SMB_CREATEDIRECTORY_REQUEST_MINBYTES)
    {
        TRCERR("Illegal WordCount or ByteCount ");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* withdraw TID, UID */

    tid = (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    uid = (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid));

    pUser = csGetUserByUid(uid);
    if (pUser == NULL)
    {
        TRCERR("Illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvuid);
    }

    pShare = csGetShareByUidTid(uid, tid);
    if (pShare == NULL)
    {
        TRCERR("Illegal TID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvtid);
    }

    /* convert filename to host filename */

    if ((pFileName = cmCifsNtohFilename(
                        fileNameBuff,
                        pShare->map,
                        (NQ_WCHAR*)(createRequest + 1),
                        unicodeRequired,
                        TRUE
                        )
        ) == NULL
       )
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.before = TRUE;
        eventInfo.fileName = NULL;
        eventInfo.access = 0;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Illegal filename");
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname);
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.before = TRUE;
    eventInfo.tid = tid;
    eventInfo.rid = csGetUserRid(pUser);
    eventInfo.fileName = pFileName;
    eventInfo.access = 0;
#endif /* UD_NQ_INCLUDEEVENTLOG */

    /* look for the path */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        eventInfo.before = TRUE;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
            );
        eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (!csCheckPath(pShare, pFileName, (NQ_UINT)syWStrlen(pShare->map), pUser->preservesCase))
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("path does not exist");
        TRC1P(" path: %s", cmWDump(pFileName));
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    /* create directory */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (csCheckFile(pShare, pFileName, pUser->preservesCase))
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRalreadyexists),
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRalreadyexists),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Directory already exists");
        TRC1P(" name: %s", cmWDump(pFileName));
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRalreadyexists);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_CREATE,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
    );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
    if (syCreateDirectory(pFileName) == NQ_FAIL)
    {
        error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            error,
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Unable to create directory file");
        TRC1P("  required directory: %s", (NQ_CHAR*)pFileName);
        return error;
    }
#ifdef UD_NQ_INCLUDEEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_CREATE,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
    );
#endif /* UD_NQ_INCLUDEEVENTLOG */

    csNotifyImmediatelly(pFileName, SMB_NOTIFYCHANGE_ADDED, SMB_NOTIFYCHANGE_DIRNAME);

    /* advance the outgoing response pointer */

    *pResponse += sizeof(*createResponse);

    /* compose the response */

    createResponse->wordCount = 0;
    cmPutSUint16(createResponse->byteCount, 0);

    TRCE();
    return 0;
}

/*====================================================================
 * PURPOSE: Perform NT_CREATE_ANDX command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *
 *          This function works for both CREATE and CREATE_NEW commands
 *====================================================================
 */

NQ_UINT32
csComNtCreateAndX(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsNtCreateAndXRequest* createRequest;   /* casted request */
    CMCifsNtCreateAndXExtendedResponse* createExtResponse; /* casted extended response */
    NQ_UINT32 returnValue;          /* error code in NT format or 0 for no error */
    NQ_BOOL unicodeRequired;        /* whether client requires UNICODE */
    const CSShare* pShare;          /* pointer to the share */
    NQ_WCHAR* pFileName;            /* filename to open */
    CSUid uid;                      /* required UID */
    CSTid tid;                      /* required TID */
    CSPid pid;                      /* required PID */
    CSFid rootFid;                  /* root directory FID */
    CSUser* pUser;                  /* pointer to the user descriptor */
    NQ_STATIC NQ_WCHAR fileName[UD_FS_FILENAMELEN]; /* buffer for composing the filename */
    CSCreateParams params;          /* parameters for common post processing */
    NQ_UINT32 oplockLevel;          /* oplock level */
    CSName *pName;                  /* pointer to name */
    NQ_BOOL isExtendedResponse;     /* whether client requests extended response */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent eventInfo;    /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */

    TRCB();

    /* cast request pointer */
    createRequest = (CMCifsNtCreateAndXRequest*) pRequest;

    /* is extended response required */
    isExtendedResponse = (0 != ((cmLtoh32(cmGetSUint32(createRequest->flags))) & SMB_NTCREATEANDX_REQUESTEXTENDEDRESPONSE));

    /* check space in output buffer */

    if ((returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, isExtendedResponse? sizeof(CMCifsNtCreateAndXExtendedResponse) : sizeof(CMCifsNtCreateAndXResponse))
        ) != 0
       )
    {
        TRCE();
        return returnValue;
    }

    /* read unicode flag */
    unicodeRequired = cmLtoh16(cmGetSUint16(pHeaderOut->flags2)) & SMB_FLAGS2_UNICODE;

    /* cast response pointer - will use extended pointer for both extended and regular, since the fist part is same for both */
    createExtResponse = (CMCifsNtCreateAndXExtendedResponse*) *pResponse;

    syMemset(createExtResponse, 0, isExtendedResponse? sizeof (CMCifsNtCreateAndXExtendedResponse) : sizeof(CMCifsNtCreateAndXResponse));

    /* check counts */

    if (   createRequest->wordCount != SMB_NTCREATEANDX_REQUEST_WORDCOUNT
        || cmLtoh16(cmGetSUint16(createRequest->byteCount)) < SMB_NTCREATEANDX_REQUEST_MINBYTES)
    {
        TRCERR("Illegal WordCount or ByteCount ");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* check the next X command */

    switch (createRequest->andXCommand)
    {
    case SMB_COM_READ:
    case SMB_COM_READ_ANDX:
    case SMB_COM_IOCTL:
    case 0xFF:
        break;
    default:
        TRCERR("invalid next command");
        TRC1P("  value: %d", createRequest->andXCommand);
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* withdraw UID, TID and PID */

    uid = (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid));
    tid = (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    pid = csGetPidFromHeader(pHeaderOut);

    pShare = csGetShareByUidTid(uid, tid);
    if (pShare == NULL)
    {
         TRCERR("Illegal TID");
         TRCE();
         return csErrorReturn(0, SRV_ERRinvtid);
    }

    pUser = csGetUserByUid(uid);
    if (pUser == NULL)
    {
        TRCERR("Illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvuid);
    }

    /* we do not open pseudo-files */

#ifndef UD_CS_INCLUDERPC
    TRC2P("Share name: %s, ipcFlag: %d", pShare->name, pShare->ipcFlag);
    if (pShare->ipcFlag)
    {
        TRCERR("Pseudo-files (pipes) not supported");
        TRC1P("name: %s", pShare->name);
        TRCE();
        return csErrorReturn(SMB_STATUS_NOT_SUPPORTED, (NQ_UINT32)SRV_ERRnosupport);
    }
#endif

    /* add zeroes at the and of name (just in case for NT4) */

    {
        NQ_UINT16 byteCount;    /* number of data bytes */
        NQ_BYTE* pData;         /* data pointer */

        byteCount = (NQ_UINT16)cmLtoh16(cmGetSUint16(createRequest->byteCount));
        pData = (NQ_BYTE*)(createRequest + 1) + byteCount;
        *pData++ = 0;
        *pData = 0;
    }

    /* convert filename to host filename */
    if ((pFileName = cmCifsNtohFilename(
                        fileNameBuff,
                        pShare->map,
                        (NQ_WCHAR*)(createRequest + 1),
                        unicodeRequired,
                        TRUE
                        )
        ) == NULL
       )
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.before = FALSE;
        eventInfo.fileName = NULL;
        eventInfo.access = 0;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Illegal filename");
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname);
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.tid = tid;
    eventInfo.rid = csGetUserRid(pUser);
    eventInfo.fileName = pFileName;
    eventInfo.access = 0;
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */

    /* resolve a relative root when required */

    if ((rootFid = (NQ_UINT16)cmLtoh32(cmGetSUint32(createRequest->rootDirectoryFid))) != 0)
    {
        CSFile* pRootDirectory;                     /* file descriptor for root directory */
        CSName* pRootName;                          /* filename descriptor for the same */
        NQ_UINT length;                             /* name length */

        if ((pRootDirectory = csGetFileByFid(rootFid, tid, uid)) == NULL)
        {
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid),
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            TRCERR("Illegal root directory FID");
            TRC1P("  value %d", rootFid);
            TRCE();
            return csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
        }

        if ((pRootName = csGetNameByNid(pRootDirectory->nid)) == NULL)
        {
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid),
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            TRCERR("Illegal root directory NID");
            TRC1P("  value %d", pRootDirectory->nid);
            TRCE();
            return csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
        }

        syWStrcpy(fileName, pRootName->name);      /* root name */
        length = (NQ_UINT)syWStrlen(fileName);
        fileName[length] = cmWChar(SY_PATHSEPARATOR);
        fileName[length + 1] = cmWChar(0);
       
        length = (NQ_UINT)syWStrlen(csGetTreeByTid(pRootDirectory->tid)->share->map);
        if (cmWChar(SY_PATHSEPARATOR) == *(pFileName + length))
            length++;
        syWStrcat(fileName, pFileName + length);            /* add filename */
        pFileName = fileName;
    }

    /* prepare for common post processing */
    params.disposition = cmLtoh32(cmGetSUint32(createRequest->createDisposition)); 
    params.desiredAccess = cmLtoh32(cmGetSUint32(createRequest->desiredAccess));
    params.sharedAccess = cmLtoh32(cmGetSUint32(createRequest->shareAccess));
    params.createOptions = cmLtoh32(cmGetSUint32(createRequest->createOptions));
    params.fileName = pFileName;
    params.user = pUser;
    params.share = pShare;
    params.tid = tid;
    params.pid = pid;
    params.uid = uid;
    params.context.flags = 0;
    params.unicodeRequired = unicodeRequired;
    params.fileAttributes = cmLtoh32(cmGetSUint32(createRequest->fileAttributes));
#if defined(UD_CS_INCLUDERPC_SPOOLSS) && defined(UD_CS_SPOOLSS_PRN_USESPOOLERFILE)
    params.createSpoolerFile = FALSE;
#endif /* defined(UD_CS_INCLUDERPC_SPOOLSS) && defined(UD_CS_SPOOLSS_PRN_USESPOOLERFILE) */

    /* call common processing */
    returnValue = csCreateCommonProcessing(&params);
    if (0 != returnValue && SMB_STATUS_SHARING_VIOLATION != returnValue)
    {
        TRCE();
        return returnValue;
    }
    

    /* check for oplock */

    if ( SMB_STATUS_SHARING_VIOLATION != returnValue)
    {
        if ((pName = csGetNameByNid(params.file->nid)) == NULL)
        {
            TRC("Failed to get name slot pointer");   
            TRCE();
            return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        }

        oplockLevel = params.file->isDirectory || pName->wasOplockBroken ? 
                        SMB_NTCREATEANDX_REQUESTOPLOCKNONE : 
                        cmLtoh32(cmGetSUint32(createRequest->flags)) & (SMB_NTCREATEANDX_REQUESTOPLOCKEXCLUS | SMB_NTCREATEANDX_REQUESTOPLOCKBATCH);
#ifdef UD_CS_DISABLEOPLOCK
        oplockLevel = SMB_NTCREATEANDX_REQUESTOPLOCKNONE;
#endif /* UD_CS_DISABLEOPLOCK */
    }
    else
    {
        /* create fake file for oplock check */
        pName = csGetNameByName(params.fileName);
        if (NULL == pName)
        {
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror),
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get name slot pointer");   
            LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL);
            return SMB_STATUS_UNSUCCESSFUL;
        }
        params.file = csGetNewFile(csGetTreeByTid(tid), pName, 0);
        if (NULL == params.file)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "No more file slots");
            LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL);
            return SMB_STATUS_NO_MORE_ENTRIES;
        }
        params.file->isPipe = FALSE;
#ifdef UD_CS_INCLUDERPC_SPOOLSS
        params.file->isPrint = FALSE;
#endif
        params.file->nid = pName->nid;
        oplockLevel = SMB_NTCREATEANDX_REQUESTOPLOCKNONE;
    }

    TRC("oplockLevel requested: %d", oplockLevel);
    if ((oplockLevel != SMB_NTCREATEANDX_REQUESTOPLOCKNONE) && csBreakCheck(&params) == TRUE)
    {
        TRC("Breaking oplock was sent");
        TRCE();
        params.file->breakContext.status = (NQ_STATUS)returnValue;
        return SMB_STATUS_NORESPONSE;
    }
    else
    {
        if (SMB_STATUS_SHARING_VIOLATION == returnValue)
            csReleaseFile(params.file->fid);
    }
    if (SMB_STATUS_SHARING_VIOLATION == returnValue)
    {
        TRCE();
        return returnValue;
    }

    /* advance the outgoing response pointer */
    *pResponse += isExtendedResponse? sizeof(CMCifsNtCreateAndXExtendedResponse) : sizeof(CMCifsNtCreateAndXResponse);

    /* compose the response */
    if (isExtendedResponse)
        createExtResponse->wordCount = SMB_NTCREATEANDX_EXTENDED_RESPONSE_WORDCOUNT;
    else
        createExtResponse->wordCount = SMB_NTCREATEANDX_RESPONSE_WORDCOUNT;

    createExtResponse->andXCommand = createRequest->andXCommand;
    /*createExtResponse->andXReserved = 0; already zeroed */
   
    /* grant oplock if requested */

#ifdef UD_CS_INCLUDERPC
    if (params.file->isPipe)
    {
        createExtResponse->oplockLevel = SMB_NTCREATEANDX_RESPONSENOOPLOCK;
    }
    else
#endif /* UD_CS_INCLUDERPC */
    {    
        TRC("csGetFilesCount() %d, max : %d", csGetFilesCount(), MAX_NUM_OPLOCK_OPEN_FILES);
        TRC("csGetUniqueFilesCount() %d, max : %d", csGetUniqueFilesCount(), MAX_NUM_OPLOCK_OPEN_UNIQUE_FILES);

        if (csGetFilesCount() > MAX_NUM_OPLOCK_OPEN_FILES ||
            csGetUniqueFilesCount() > MAX_NUM_OPLOCK_OPEN_UNIQUE_FILES)
            createExtResponse->oplockLevel = SMB_NTCREATEANDX_RESPONSENOOPLOCK;
        else
            createExtResponse->oplockLevel = (oplockLevel == SMB_NTCREATEANDX_REQUESTOPLOCKNONE) ?
                                          SMB_NTCREATEANDX_RESPONSENOOPLOCK : SMB_NTCREATEANDX_RESPONSEOPLOCKBATCH; 
        TRC("oplockLevel granted: %d", createExtResponse->oplockLevel);

        params.file->oplockGranted = createExtResponse->oplockLevel != SMB_NTCREATEANDX_RESPONSENOOPLOCK;
        params.file->breakContext.socket = csDispatchGetSocket();
        params.file->breakContext.isSmb2 = FALSE;
    }

    if (createExtResponse->andXCommand == 0xFF)
    {
        /*cmPutSUint16(createExtResponse->andXOffset, 0); already zeroed */
    }
    else
    {
        NQ_UINT16 offset;   /* for calculating offset */

        offset = (NQ_UINT16)(*pResponse - (NQ_BYTE*)pHeaderOut);
        cmPutSUint16(createExtResponse->andXOffset, cmHtol16(offset));
    }

    cmPutSUint16(createExtResponse->fid, cmHtol16(params.file->fid));
    cmPutSUint32(createExtResponse->createAction, cmHtol32(params.takenAction));

    if (pShare->ipcFlag)
    {
        /*
         zeroed at beginning
        cmPutSUint32(createExtResponse->creationTime.high, 0);
        cmPutSUint32(createExtResponse->lastAccessTime.low, 0);
        cmPutSUint32(createExtResponse->lastAccessTime.high, 0);
        cmPutSUint32(createExtResponse->lastWriteTime.low, 0);
        cmPutSUint32(createExtResponse->lastWriteTime.high, 0);
        cmPutSUint32(createExtResponse->lastChangeTime.low, 0);
        cmPutSUint32(createExtResponse->lastChangeTime.high, 0);
        */
        cmPutSUint32(createExtResponse->fileAttributes, SMB_ATTR_NORMAL);
        cmPutSUint32(createExtResponse->allocationSize.low, cmHtol32((NQ_UINT32)(UD_NS_BUFFERSIZE - 0x44)));
        /*
        cmPutSUint32(createExtResponse->allocationSize.high, 0);
        cmPutSUint32(createExtResponse->endOfFile.low, 0);
        cmPutSUint32(createExtResponse->endOfFile.high, 0);
        zeroed at beginning
        */
        cmPutSUint16(createExtResponse->fileType, cmHtol16(SMB_NTCREATEANDX_MESSAGEPIPE));
        cmPutSUint16(createExtResponse->deviceState, cmHtol16((SMB_NTCREATEANDX_ICOUNT | SMB_NTCREATEANDX_READ | SMB_NTCREATEANDX_MESSAGETYPE)));
    }
    else
    {
        csWriteFileTimes(&params.fileInfo, pName, (NQ_BYTE *)&createExtResponse->creationTime.low);
        cmPutSUint32(createExtResponse->fileAttributes, cmHtol32(params.fileInfo.attributes));
        cmPutSUint32(createExtResponse->allocationSize.low, cmHtol32(params.fileInfo.allocSizeLow));
        cmPutSUint32(createExtResponse->allocationSize.high, cmHtol32(params.fileInfo.allocSizeHigh));
        cmPutSUint32(createExtResponse->endOfFile.low, cmHtol32(params.fileInfo.sizeLow));
        cmPutSUint32(createExtResponse->endOfFile.high, cmHtol32(params.fileInfo.sizeHigh));
        cmPutSUint16(createExtResponse->fileType, 0);
        cmPutSUint16(createExtResponse->deviceState, cmHtol16(7));  /* undocumented */
    }
    createExtResponse->directory = params.file->isDirectory ? 1 : 0;

    if (isExtendedResponse)
    {
        /*cmPutSUint32(createExtResponse->fid_full.low, (cmHtol32((NQ_UINT32)params.file->fid)));*/ /* short fid, rest is zeroes */
        cmPutSUint32(createExtResponse->maximalAccessRights, cmHtol32(csGetTreeByTid(tid)->maxAccessRights));
        /*cmPutSUint32(createExtResponse->maximalAccessRights, cmHtol32(convertNqAccessToNtAccess(params.file->access)));*/
        cmPutSUint32(createExtResponse->guestMaximalAccessRights, cmHtol32(csGetTreeByTid(tid)->maxAccessRights));
        /*cmPutSUint32(createExtResponse->guestMaximalAccessRights, 0);*/
        /*
         zeroed at beginning
        
        cmPutSUint16(createExtResponse->byteCount, 0);
        */
    }
    else
    {
        cmPutSUint16(((CMCifsNtCreateAndXResponse *)createExtResponse)->byteCount, 0);
    }

    TRCE();
    return 0;
}

/*====================================================================
 * PURPOSE: Perform common Create processing
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT pointer to the parameter structure
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   This function is called from both NT_CREATE_ANDX command handler and 
 *               from SMB2 Create command handler. 
 *               It performs processing command for these two commands.
 *====================================================================
 */

NQ_UINT32
csCreateCommonProcessing(
    CSCreateParams * params
    )
{
    NQ_UINT32 returnValue;          /* error code in NT format or 0 for no error */
    CMCifsStatus error;             /* for composing DOS-style error */
    NQ_BOOL fileExists;             /* flag for considering actions */
    NQ_BOOL directoryRequired;      /* whether operation was requested on a directory */
    NQ_BOOL fileRequired;           /* whether file cannot be a directory */
    NQ_BOOL directoryFound;         /* whether found file is a directory */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent eventInfo;    /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */

    TRCB();

    TRC("fileName: %s", cmWDump(params->fileName));

    /* look for the path */

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.fileName = params->fileName;
    eventInfo.access = params->desiredAccess;
    eventInfo.tid = params->tid;
    eventInfo.rid = csGetUserRid(params->user);
#endif /* UD_NQ_INCLUDEEVENTLOG */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        params->user->name,
        params->user->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (!csCheckPath(
            params->share, 
            params->fileName, 
            (NQ_UINT)syWStrlen(params->share->map), 
            params->user->preservesCase
            )
       )
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_QUERYDIRECTORY,
            params->user->name,
            params->user->ip,
            0,
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            params->user->name,
            params->user->ip,
            csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("path does not exist");
        TRC1P(" path: %s", cmWDump(params->fileName));
        TRCE();
        return (syGetLastSmbError() == DOS_ERRnoaccess) ? csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess) :
                                                          csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        params->user->name,
        params->user->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    /* further action depends on a combination of:
       1) whether file exists or not
       2) which action was required in this case
       3) if a directory required */
    params->desiredAccess = convertNtAccessToDesiredAccess(
                        params->desiredAccess,
                        params->sharedAccess
                        );
#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.access = params->desiredAccess;
#endif /* UD_NQ_INCLUDEEVENTLOG */
    
    fileRequired = params->createOptions & SMB_NTCREATEANDX_NONDIRECTORY;
    directoryRequired = params->createOptions & SMB_NTCREATEANDX_DIRECTORY;
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    {
        eventInfo.before = TRUE;
        udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    (params->user->preservesCase || (params->share != NULL && syWStrcmp( params->share->map , params->fileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
                    params->user->name,
                    params->user->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                );
        eventInfo.before = FALSE;
    }
#endif
    fileExists = csCheckFile(params->share, params->fileName, params->user->preservesCase);
    if (!fileExists && (syGetLastSmbError() == DOS_ERRnoaccess))
    {
        TRCERR("Access denied");
        TRCE();
        return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    {
        udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    (params->user->preservesCase || (params->share != NULL && syWStrcmp( params->share->map , params->fileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
                    params->user->name,
                    params->user->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                );
    }
#endif
#ifdef UD_CS_INCLUDERPC_SPOOLSS      
    if (params->share->isPrintQueue)
    {
        NQ_TIME currentTime = syGetTimeInMsec();

        if (createPrintFile(params->fileName, params->share, params->tid, (NQ_UINT16)params->desiredAccess, &params->file,
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
                params->createSpoolerFile,
#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
                params->user) != NQ_SUCCESS)
        {
            TRCERR("Failed to create a printer file");
            TRCE();
            return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        }
        
        params->takenAction = SMB_OPENANDX_WASCREATED;
        syMemset(&params->fileInfo, 0, sizeof(params->fileInfo));
        params->fileInfo.creationTime.high = currentTime.high;
        params->fileInfo.creationTime.low = currentTime.low;
        params->fileInfo.lastAccessTime.high = currentTime.high;
        params->fileInfo.lastAccessTime.low = currentTime.low;
        params->fileInfo.lastChangeTime.high = currentTime.high;
        params->fileInfo.lastChangeTime.low = currentTime.low;
        params->fileInfo.lastWriteTime.high = currentTime.high;
        params->fileInfo.lastWriteTime.low = currentTime.low;
        params->fileInfo.attributes = SMB_ATTR_NORMAL;
    }
    else
#endif    
    {
        if (fileExists)
        {
            /* file exists
               if the action is to create - fail */
            if (params->disposition == SMB_NTCREATEANDX_FILECREATE)
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_CREATE,
                    params->user->name,
                    params->user->ip,
                    csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists),
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("File exists while the command requires its creation");
                TRCE();
                return csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists);
            }
    
            /* define if the required file is a directory */
    
            if ( NQ_SUCCESS != csGetFileInformationByName(params->share, params->fileName, &params->fileInfo
#ifdef UD_NQ_INCLUDEEVENTLOG
                    ,params->user
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    ))
            {
                error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    params->user->name,
                    params->user->ip,
                    error,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("Unable to read file information");
                TRCE();
                return error;
            }
    
            directoryFound = params->fileInfo.attributes & SMB_ATTR_DIRECTORY;
            
            if (directoryFound && fileRequired)
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    params->user->name,
                    params->user->ip,
                    SMB_STATUS_FILE_IS_A_DIRECTORY,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("File required, directory found");
                TRCE();
                return SMB_STATUS_FILE_IS_A_DIRECTORY;
            }

            if (directoryRequired && !directoryFound)
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    params->user->name,
                    params->user->ip,
                    SMB_STATUS_FILE_IS_A_DIRECTORY,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("Directory required, file found");
                TRCE();
                return SMB_STATUS_NOT_A_DIRECTORY;

            }

            /* an existing directory should not be opened for overwrite access */
    
            if (directoryRequired && params->disposition == SMB_NTCREATEANDX_FILEOVERWRITE)
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    params->user->name,
                    params->user->ip,
                    csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRbaddirectory),
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("Directory write operations are not supported");
                TRCE();
                return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRbaddirectory);
            }

            /* do not allow opening for delete access of root folder */

             if ((0 != (params->desiredAccess & SMB_ACCESS_A_DELETE)) && directoryFound && (0 == cmWStricmp(params->share->map, params->fileName)))
             {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    params->user->name,
                    params->user->ip,
                    csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRbaddirectory),
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("Directory delete operations on root are not allowed");
                TRCE();
                return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRbaddirectory);
             }

            /* do not allow mark for deletion of read only file*/

            if ((params->createOptions & SMB_NTCREATEANDX_DELETEONCLOSE) && (params->fileInfo.attributes & SMB_ATTR_READONLY))
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    params->user->name,
                    params->user->ip,
                    csErrorReturn(SMB_STATUS_CANNOT_DELETE, DOS_ERRnoaccess),
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("Unable to mark file for deletion read only file");
                TRCE();
                return csErrorReturn(SMB_STATUS_CANNOT_DELETE, DOS_ERRnoaccess);
            }
    
            /* do not allow overwriting read only file */
            
            if ((params->disposition == SMB_NTCREATEANDX_FILEOVERWRITEIF) && (params->fileInfo.attributes & SMB_ATTR_READONLY))
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    params->user->name,
                    params->user->ip,
                    csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess),
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("Attempt to overwrite read only file");
                TRCE();
                return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);          
            }
    
            if ((returnValue = openFile(
                    params->fileName, 
                    params->tid, 
                    (NQ_UINT16)params->desiredAccess, 
                    NULL, 
                    &params->file
                    )
                 ) != 0)
            {
                TRCE();
                return returnValue;
            }

            if (   params->disposition == SMB_NTCREATEANDX_FILEOVERWRITE
                || params->disposition == SMB_NTCREATEANDX_FILEOVERWRITEIF
               )
            {

                if ((params->desiredAccess & SMB_ACCESS_A) == SMB_ACCESS_A_READ ||
                    (params->desiredAccess & SMB_ACCESS_A) == SMB_ACCESS_A_EXECUTE)
                {
                    /*  closing file and openings again for write */
                    LOGMSG(CM_TRC_LEVEL_FUNC_COMMON , " Create has overwrite and only read access [Closing and opening the file again for truncate].");
                    error = csCanWriteShare(params->tid);
                    if (error != NQ_SUCCESS)
                    {
                        TRCERR(" User doesn't have Write access");
                        TRCE();
                        return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
                    }
                    csReleaseFile(params->file->fid);
#ifdef UD_NQ_INCLUDEEVENTLOG
                    eventInfo.before = TRUE;
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_OPEN,
                        params->user->name,
                        params->user->ip,
                        0,
                        (const NQ_BYTE*)&eventInfo
                    );
                    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    params->file->file = syOpenFileForWrite(params->fileName , FALSE ,FALSE ,FALSE);
                    if (!syIsValidFile(params->file->file))
                    {
                        TRCERR("Failed to open file for write");
#ifdef UD_NQ_INCLUDEEVENTLOG
                        udEventLog(
                            UD_LOG_MODULE_CS,
                            UD_LOG_CLASS_FILE,
                            UD_LOG_FILE_OPEN,
                            params->user->name,
                            params->user->ip,
                            (NQ_UINT32)syGetLastError(),
                            (const NQ_BYTE*)&eventInfo
                        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                        TRCE();
                        return (NQ_UINT32)syGetLastError();
                    }
#ifdef UD_NQ_INCLUDEEVENTLOG
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_OPEN,
                        params->user->name,
                        params->user->ip,
                        0,
                        (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEVENTLOG */

                }
                /* truncate file */
#ifdef UD_NQ_INCLUDEEVENTLOG
                eventInfo.sizeLow = 0;
                eventInfo.sizeHigh = 0;
                eventInfo.before = TRUE;
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_SIZESET,
                    params->user->name,
                    params->user->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                );
                eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
                if (syTruncateFile(params->file->file, 0, 0) != NQ_SUCCESS)
                {
                    error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_SIZESET,
                        params->user->name,
                        params->user->ip,
                        error,
                        (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_OPEN,
                        params->user->name,
                        params->user->ip,
                        error,
                        (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    csReleaseFile(params->file->fid);      /* also closes the file */
                    TRCERR("Unable to truncate file");
                    TRCE();
                    return error;
                }
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_SIZESET,
                    params->user->name,
                    params->user->ip,
                    0,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                {
                    /*  closing file and opening again  */
                    if ((params->desiredAccess & SMB_ACCESS_A) == SMB_ACCESS_A_READ ||
                    (params->desiredAccess & SMB_ACCESS_A) == SMB_ACCESS_A_EXECUTE)
                    {
#ifdef UD_NQ_INCLUDEEVENTLOG
                        udEventLog(
                            UD_LOG_MODULE_CS,
                            UD_LOG_CLASS_FILE,
                            UD_LOG_FILE_CLOSE,
                            params->user->name,
                            params->user->ip,
                            0,
                            (const NQ_BYTE*)&eventInfo
                        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                        syCloseFile(params->file->file);
#ifdef UD_NQ_INCLUDEEVENTLOG
                        udEventLog(
                            UD_LOG_MODULE_CS,
                            UD_LOG_CLASS_FILE,
                            UD_LOG_FILE_CLOSE,
                            params->user->name,
                            params->user->ip,
                            (NQ_UINT32)syGetLastError(),
                            (const NQ_BYTE*)&eventInfo
                        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    }
                    else
                    {
                        csReleaseFile(params->file->fid);
                    }
                    if ((returnValue = openFile(
                                        params->fileName,
                                        params->tid,
                                        (NQ_UINT16)params->desiredAccess,
                                        NULL,
                                        &params->file
                                        )
                                     ) != 0)
                    {
                        TRCE();
                        return returnValue;
                    }
                }
    
                params->takenAction = SMB_OPENANDX_WASTRUNCATED;
            }
            else
            {
                params->takenAction = SMB_OPENANDX_WASOPENED;
            }

            params->file->isDirectory = directoryFound;
        }
        else /* if (fileExists) */
        {  
            /* file was not found
               if the action to take assumes that the file exists - fail */
    
            if (params->disposition == SMB_NTCREATEANDX_FILEOPEN)
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_CREATE,
                    params->user->name,
                    params->user->ip,
                    csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile),
                    (const NQ_BYTE*)&eventInfo
                );
 #endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("File does not exist while the command expects it to exist");
                TRCE();
                return csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
            }
    
            if (params->disposition == SMB_NTCREATEANDX_FILEOVERWRITE)
            {
                error = (directoryRequired)?
                        csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbaddirectory) :
                        csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_CREATE,
                    params->user->name,
                    params->user->ip,
                    error,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("File does not exist while the command expects it to exist");
                TRCE();
                return error;
            }
    
            /* create file/directory */
    
            returnValue = createFile(
                directoryRequired, 
                params->fileName, 
                params->tid, 
                (NQ_UINT16)params->desiredAccess, 
                &params->file
                );
    
            if (returnValue != 0)
            {
                TRCE();
                return returnValue;
            }
    
            params->takenAction = SMB_OPENANDX_WASCREATED;
            params->file->isDirectory = directoryRequired;
        }
            
        /* store file information */
#ifdef UD_NQ_INCLUDEEVENTLOG
        params->file->user = params->user;
#endif /* UD_NQ_INCLUDEEVENTLOG */
        params->file->tid = params->tid;
        params->file->uid = params->uid;
        params->file->pid = params->pid;
        params->file->options = params->createOptions;
    
        /* update file attributes */
    
        if (csGetFileInformationByName(
            params->share, 
            params->fileName, 
            &params->fileInfo
#ifdef UD_NQ_INCLUDEEVENTLOG
            ,params->user
#endif /* UD_NQ_INCLUDEEVENTLOG */
             )
             != NQ_SUCCESS)
        {
            error = csErrorGetLast();
            csReleaseFile(params->file->fid);      /* also closes the file */
            TRCERR("Unable to read file information");
            TRCE();
            return error;
        }
    
        if (params->takenAction == SMB_OPENANDX_WASCREATED || params->takenAction == SMB_OPENANDX_WASTRUNCATED)
        {
            params->fileInfo.attributes = csChangeFileAttributes(
                                    params->fileInfo.attributes,
                                      (NQ_UINT16)((params->fileAttributes & 0xFFFF)
                                    | SMB_ATTR_ARCHIVE)
                                    );
    
            params->fileInfo.lastWriteTime.high = 0;
            params->fileInfo.lastWriteTime.low = 0;
    
            if (!directoryRequired && csSetFileInformation(
                                        params->file, 
                                        params->fileName, 
                                        &params->fileInfo)
               )
            {
                csReleaseFile(params->file->fid);      /* also closes the file */
    
                TRCERR("unable to change file attributes/time");
                TRCE();
                return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
            }
            
            if (NQ_SUCCESS != csGetFileInformationByName(params->share, params->fileName, &params->fileInfo
#ifdef UD_NQ_INCLUDEEVENTLOG
                    ,params->user
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    ))
            {
                error = csErrorGetLast();
                csReleaseFile(params->file->fid);      /* also closes the file */
                TRCERR("Unable to read file information");
                TRCE();
                return error;
            }
        }
    }

    /* fix file size for directories */

    if (0 != (params->fileInfo.attributes & SMB_ATTR_DIRECTORY))
    {
        params->file->isDirectory = TRUE;
        params->fileInfo.sizeHigh = 0;
        params->fileInfo.sizeLow = 0;
    }
   
    TRCE();
    return 0;
}

/*====================================================================
 * PURPOSE: Perform NT_TRANSACT_CREATE subcommand of NT_TRANSACTION protocol
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT structure with subcommand parameters:
 *              IN pointer to the CIFS header
 *              IN pointer to the parameter area
 *              OUT pointer to the data area
 *              OUT length of the parameter area
 *              OUT length of the data area
 *
 * RETURNS: SMB error or 0 on success
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT32
csNtTransactionCreate(
    CSNtTransactionDescriptor* descriptor
    )
{
    CMCifsNtTransactionCreateRequest* createRequest;   /* casted request */
    CMCifsNtTransactionCreateResponse* createResponse; /* casted response */
    NQ_UINT32 returnValue;             /* error code in NT format or 0 for no error */
    NQ_BOOL unicodeRequired;           /* whether client requires UNICODE */
    CMCifsStatus error;             /* for composing DOS-style error */
    CSFile* pFile;                  /* pointer to file descriptor */
    const CSShare* pShare;          /* pointer to the share */
    NQ_WCHAR* pFileName;             /* filename to open */
    CSUid uid;                      /* required UID */
    CSTid tid;                      /* required TID */
    CSPid pid;                      /* required PID */
    SYFileInformation fileInfo;     /* for querying file information */
    CSFid rootFid;                  /* root directory FID */
    NQ_BOOL fileExists;             /* flag for considering actions */
    NQ_BOOL directoryRequired;      /* whether operation was requested on a directory */
    NQ_UINT32 createDisposition;    /* action to take */
    NQ_UINT32 takenAction;          /* action in response */
    NQ_UINT16 desiredAccess;        /* OPEN style access */
    CSUser* pUser;                          /* pointer to the user descriptor */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent eventInfo;            /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */

    TRCB();
    
    /* read unicode flag */

    unicodeRequired = cmLtoh16(cmGetSUint16(descriptor->pHeaderOut->flags2)) & SMB_FLAGS2_UNICODE;

    /* cast pointers */

    createRequest = (CMCifsNtTransactionCreateRequest*) (
                        (NQ_BYTE*)descriptor->requestData
                        + cmLtoh32(cmGetSUint32(descriptor->requestData->parameterOffset))
                        - sizeof(CMCifsHeader)
                        );

    createResponse = (CMCifsNtTransactionCreateResponse*) descriptor->pParams;
    
    /* withdraw UID, TID and PID */

    uid = (CSUid)cmLtoh16(cmGetSUint16(descriptor->pHeaderOut->uid));
    tid = (CSTid)cmLtoh16(cmGetSUint16(descriptor->pHeaderOut->tid));
    pid = csGetPidFromHeader(descriptor->pHeaderOut);

    pUser = csGetUserByUid(uid);
    if (pUser == NULL)
    {
        TRCERR("Illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvuid);
    }

    pShare = csGetShareByUidTid(uid, tid);
    if (pShare == NULL)
    {
        TRCERR("Illegal TID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvtid);
    }

    /* we do not open pseudo-files */

#ifndef UD_CS_INCLUDERPC
    if (pShare->ipcFlag)
    {
        TRCERR("Pseudo-files (pipes) not supported");
        TRCE();
        return csErrorReturn(SMB_STATUS_NOT_SUPPORTED, (NQ_UINT32)SRV_ERRnosupport);
    }
#endif
    
    /* convert filename to host filename */

    if ((pFileName = cmCifsNtohFilename(
                        fileNameBuff,
                        pShare->map,
                        (NQ_WCHAR*)(createRequest + 1),
                        unicodeRequired,
                        TRUE
                        )
        ) == NULL
       )
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.fileName = NULL;
        eventInfo.access = 0;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Illegal filename");
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname);
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.tid = tid;
    eventInfo.rid = csGetUserRid(pUser);
    eventInfo.fileName = pFileName;
    eventInfo.access = 0;
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */

    /* resolve a relative root when required */

    if ((rootFid = (NQ_UINT16)cmLtoh32(cmGetSUint32(createRequest->rootDirectoryFid))) != 0)
    {
        CSFile* pRootDirectory;                     /* file descriptor for root directory */
        CSName* pRootName;                          /* file descriptor for root directory */
        NQ_STATIC NQ_WCHAR fileName[UD_FS_FILENAMELEN]; /* buffer for composing the filename */
        NQ_UINT length;                                /* name length */

        if ((pRootDirectory = csGetFileByFid(rootFid, tid, uid)) == NULL)
        {
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid),
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            TRCERR("Illegal root directory FID");
            TRC1P("  value %d", rootFid);
            TRCE();
            return csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
        }

        if ((pRootName = csGetNameByNid(pRootDirectory->nid)) == NULL)
        {
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid),
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            TRCERR("Illegal root directory NID");
            TRC1P("  value %d", pRootDirectory->nid);
            TRCE();
            return csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
        }

        syWStrcpy(fileName, pRootName->name);      /* root name */
        length = (NQ_UINT)syWStrlen(fileName);
        if (cmWChar(SY_PATHSEPARATOR) != fileName[length - 1])
        {
            fileName[length] = cmWChar(SY_PATHSEPARATOR);
            fileName[length + 1] = cmWChar(0);
        }
        syWStrcat(fileName, pFileName);                /* add filename */
        pFileName = fileName;
    }

    desiredAccess = convertNtAccessToDesiredAccess(
                        cmLtoh32(cmGetSUint32(createRequest->desiredAccess)),
                        cmLtoh32(cmGetSUint32(createRequest->shareAccess))
                        );
#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.access = desiredAccess;
#endif /* UD_NQ_INCLUDEEVENTLOG */

    /* look for the path */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (!csCheckPath(pShare, pFileName, (NQ_UINT)syWStrlen(pShare->map), pUser->preservesCase))
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("path does not exist");
        TRC1P(" path: %s", cmWDump(pFileName));
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    /* futher action depends on a combination of:
       1) whether file exists or not
       2) which action was required in this case
       3) if a directory required */

    createDisposition = cmLtoh32(cmGetSUint32(createRequest->createDisposition));
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    fileExists = csCheckFile(pShare, pFileName, pUser->preservesCase);

#ifdef UD_CS_INCLUDERPC_SPOOLSS      
    if (pShare->isPrintQueue)
    {
        if (createPrintFile(pFileName, pShare, tid, desiredAccess, &pFile,
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
                FALSE,
#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
                pUser) != NQ_SUCCESS)
        {
            TRCERR("Failed to create a printer file");
            TRCE();
            return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        }

        takenAction = SMB_OPENANDX_WASCREATED;
        syMemset(&fileInfo, 0, sizeof(fileInfo));          
    }
    else
#endif 
    {          
        if (fileExists)
        {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
                pUser->name,
                pUser->ip,
                0,
                (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
            /* file exists
               if the action is to create - fail */
            if (createDisposition == SMB_NTCREATEANDX_FILECREATE)
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_CREATE,
                    pUser->name,
                    pUser->ip,
                    csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists),
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("File exists while the command requires its creation");
                TRCE();
                return csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists);
            }

            /* define if the required file is a directory */

            if (NQ_SUCCESS != csGetFileInformationByName(pShare, pFileName, &fileInfo
#ifdef UD_NQ_INCLUDEEVENTLOG
            ,pUser
#endif /* UD_NQ_INCLUDEEVENTLOG */
            ))
            {
                error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    pUser->name,
                    pUser->ip,
                    error,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("Unable to read file information");
                TRCE();
                return error;
            }

            directoryRequired = fileInfo.attributes & SMB_ATTR_DIRECTORY;

            if (!csMatchFileAttributes(
                    (NQ_UINT16)cmLtoh32(cmGetSUint32(createRequest->extFileAttributes)),
                    (NQ_UINT16)fileInfo.attributes
                    )
               )
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    pUser->name,
                    pUser->ip,
                    csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile),
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("File attributes mismatch");
                TRC2P("  required: %d, read: %d", (NQ_INT)cmLtoh32(cmGetSUint32(createRequest->extFileAttributes)), (NQ_INT)fileInfo.attributes);
                TRCE();
                return csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
            }

                returnValue =  openFile(pFileName, tid, desiredAccess, NULL, &pFile);   
                if (returnValue != 0)   
            {
                TRCE();
                return returnValue;
            }
            takenAction = SMB_OPENANDX_WASOPENED;
        }
        else
        {
            /* file was not found
               if the action to take assumes that the file exists - fail */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
                pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile),
                (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
            if (   createDisposition == SMB_NTCREATEANDX_FILEOPEN
                || createDisposition == SMB_NTCREATEANDX_FILEOVERWRITE)
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    pUser->name,
                    pUser->ip,
                    csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile),
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("File does not exist while the command expects it to exist");
                TRCE();
                return csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
            }

            /* create file */

            directoryRequired = cmLtoh32(cmGetSUint32(createRequest->createOptions)) & SMB_NTTRANSACTCREATE_DIRECTORY;

            TRC1P("creating file %s", (NQ_CHAR*)pFileName);

            returnValue = createFile(directoryRequired, pFileName, tid, desiredAccess, &pFile);
            if (returnValue != 0)
            {
                TRCE();
                return returnValue;
            }

            takenAction = SMB_OPENANDX_WASCREATED;
        }

        /* update file attributes */

        if (NQ_SUCCESS != csGetFileInformationByName(pShare, pFileName, &fileInfo
#ifdef UD_NQ_INCLUDEEVENTLOG
                ,pUser
#endif /* UD_NQ_INCLUDEEVENTLOG */
                ))
        {
            error = csErrorGetLast();
            if (pFile != NULL)
                csReleaseFile(pFile->fid);      /* also closes the file */
            TRCERR("Unable to read file information");
            TRCE();
            return error;
        }

        if (takenAction & SMB_OPENANDX_WASCREATED)
        {
            fileInfo.attributes = csChangeFileAttributes(
                            fileInfo.attributes,
                            (NQ_UINT16)cmLtoh32(cmGetSUint32(createRequest->extFileAttributes))
                            );
            fileInfo.lastWriteTime.high = 0;
            fileInfo.lastWriteTime.low = 0;

            if (csSetFileInformation(pFile, pFileName, &fileInfo))
            {
                csReleaseFile(pFile->fid);      /* also closes the file */

                TRCERR("unable to change file attributes/time");
                TRCE();
                return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
            }
            if (NQ_SUCCESS != csGetFileInformationByName(pShare, pFileName, &fileInfo
#ifdef UD_NQ_INCLUDEEVENTLOG
                    ,pUser
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    ))
            {
                error = csErrorGetLast();
                csReleaseFile(pFile->fid);      /* also closes the file */
                TRCERR("Unable to read file information");
                TRCE();
                return error;
            }
        }
    }

    /* store file information */
#ifdef UD_NQ_INCLUDEEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_CREATE,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */

    if (pFile != NULL)
    {
        pFile->tid = tid;
        pFile->uid = uid;
        pFile->pid = pid;
    }

    /* compose the response */

    cmPutSUint16(createResponse->fid, cmHtol16(pFile->fid));
    cmPutSUint32(createResponse->createAction, cmHtol32(takenAction));
    cmPutSUint32(createResponse->extFileAttributes, cmHtol32(fileInfo.attributes));

    {
        NQ_UINT32 timeLow;     /* low part of UTC time */
        NQ_UINT32 timeHigh;    /* high part of UTC time */

        cmCifsTimeToUTC(
            fileInfo.creationTime,
            &timeLow,
            &timeHigh
            );
        cmPutSUint32(createResponse->creationTime.low, cmHtol32(timeLow));
        cmPutSUint32(createResponse->creationTime.high, cmHtol32(timeHigh));
        cmCifsTimeToUTC(
            fileInfo.lastChangeTime,
            &timeLow,
            &timeHigh
            );
        cmPutSUint32(createResponse->lastChangeTime.low, cmHtol32(timeLow));
        cmPutSUint32(createResponse->lastChangeTime.high, cmHtol32(timeHigh));
        cmCifsTimeToUTC(
            fileInfo.lastAccessTime,
            &timeLow,
            &timeHigh
            );
        cmPutSUint32(createResponse->lastAccessTime.low, cmHtol32(timeLow));
        cmPutSUint32(createResponse->lastAccessTime.high, cmHtol32(timeHigh));
        cmCifsTimeToUTC(
            fileInfo.lastWriteTime,
            &timeLow,
            &timeHigh
            );
        cmPutSUint32(createResponse->lastWriteTime.low, cmHtol32(timeLow));
        cmPutSUint32(createResponse->lastWriteTime.high, cmHtol32(timeHigh));
    }

    
    /* fix file size for directories */
    if (0 != (fileInfo.attributes & SMB_ATTR_DIRECTORY))
    {
        fileInfo.sizeHigh = 0;
        fileInfo.sizeLow = 0;
    }

    cmPutSUint32(createResponse->allocationSize.low, cmHtol32(fileInfo.sizeLow));
    cmPutSUint32(createResponse->allocationSize.high, cmHtol32(fileInfo.sizeHigh));
    cmPutSUint32(createResponse->endOfFile.low, cmHtol32(fileInfo.sizeLow));
    cmPutSUint32(createResponse->endOfFile.high, cmHtol32(fileInfo.sizeHigh));
    cmPutSUint16(createResponse->fileType, 0);
    cmPutSUint16(createResponse->deviceState, cmHtol16(7));  /* undocumented */
    createResponse->directory = (fileInfo.attributes & SMB_ATTR_DIRECTORY)? 1:0;

    descriptor->parameterCount = sizeof(*createResponse);
    descriptor->dataCount = 0;
    descriptor->pData = (NQ_BYTE *)createResponse + sizeof(CMCifsNtTransactionCreateResponse);

    TRCE();
    return 0;
}

/*
 *====================================================================
 * PURPOSE: Perform Open2 subcommand of Transaction2 protocol
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT structure with subcommand parameters:
 *              IN pointer to the CIFS header
 *              IN pointer to the parameter area
 *              OUT pointer to the data area
 *              OUT length of the parameter area
 *              OUT length of the data area
 *
 * RETURNS: SMB error or 0 on success
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT32
csTransaction2Open(
    CSTransaction2Descriptor* descriptor
    )
{
    CMCifsOpen2Request* openRequest;    /* casted request */
    CMCifsOpen2Response* openResponse;  /* casted response */
    NQ_UINT32 returnValue = 0;          /* error code in NT format or 0 for no error */
    NQ_BOOL unicodeRequired;            /* whether client requires UNICODE */
    CMCifsStatus error;                 /* for composing DOS-style error */
    CSFile* pFile;                      /* pointer to file descriptor */
    const CSShare* pShare;              /* pointer to the share */
    NQ_WCHAR* pFileName;                 /* filename to open */
    CSUid uid;                          /* required UID */
    CSTid tid;                          /* required TID */
    CSPid pid;                          /* required PID */
    SYFileInformation fileInfo;         /* for querying file information */
    NQ_UINT16 desiredAction;            /* action from  request */
    NQ_UINT16 takenAction;              /* action in the response */
    NQ_UINT16 desiredAccess;            /* as in the request */
    NQ_UINT16 grantedAccess;            /* for the response */
    NQ_BOOL fileExists;                 /* whether the required fiel exists */
    CSUser* pUser;                          /* pointer to the user descriptor */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent eventInfo;            /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */

    TRCB();

    /* check unicode flag */

    unicodeRequired = cmLtoh16(cmGetSUint16(descriptor->pHeaderOut->flags2)) & SMB_FLAGS2_UNICODE;

    /* check space in output buffer */

    if (((NQ_UINT)(descriptor->pParams - (NQ_BYTE*)descriptor->pHeaderOut) + sizeof(*openResponse))
        > CS_MAXBUFFERSIZE
       )
    {
        TRCERR("Buffer overflow");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* cast pointers */

    openRequest = (CMCifsOpen2Request*)(
                          (NQ_BYTE*)descriptor->requestData
                        + cmLtoh16(cmGetSUint16(descriptor->requestData->transHeader.parameterOffset))
                        - sizeof(CMCifsHeader)
                        );

    openResponse = (CMCifsOpen2Response*) descriptor->pParams;


    /* withdraw UID, TID and PID */

    uid = (CSUid)cmLtoh16(cmGetSUint16(descriptor->pHeaderOut->uid));
    tid = (CSTid)cmLtoh16(cmGetSUint16(descriptor->pHeaderOut->tid));
    pid = csGetPidFromHeader(descriptor->pHeaderOut);

    pUser = csGetUserByUid(uid);
    if (pUser == NULL)
    {
        TRCERR("Illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvuid);
    }

    pShare = csGetShareByUidTid(uid, tid);
    if (pShare == NULL)
    {
        TRCERR("Illegal TID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvtid);
    }

    /* we do not open pseudo-files */

#ifndef UD_CS_INCLUDERPC
    if (pShare->ipcFlag)
    {
        TRCERR("Pseudo-files (pipes) not supported");
        TRCE();
        return csErrorReturn(SMB_STATUS_NOT_SUPPORTED, (NQ_UINT32)SRV_ERRnosupport);
    }
#endif

    /* convert filename to host filename */

    if ((pFileName = cmCifsNtohFilename(
                        fileNameBuff,
                        pShare->map,
                        (NQ_WCHAR*)(openRequest + 1),
                        unicodeRequired,
                        TRUE
                        )
        ) == NULL
       )
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.fileName = NULL;
        eventInfo.access = 0;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_OPEN,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Illegal filename");
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname);
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.before = TRUE;
    eventInfo.tid = tid;
    eventInfo.rid = csGetUserRid(pUser);
    eventInfo.fileName = pFileName;
    eventInfo.access = 0;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_OPEN,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */

    /* look for the path */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (!csCheckPath(pShare, pFileName, (NQ_UINT)syWStrlen(pShare->map), pUser->preservesCase))
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_OPEN,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("path does not exist");
        TRC1P(" path: %s", cmWDump(pFileName));
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_PATH_NOT_FOUND, DOS_ERRbadpath);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    /* open file */

    desiredAccess = (NQ_UINT16)cmLtoh16(cmGetSUint16(openRequest->desiredAccess));
#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.access = desiredAccess;
#endif /* UD_NQ_INCLUDEEVENTLOG */
    desiredAction = (NQ_UINT16)cmLtoh16(cmGetSUint16(openRequest->openFunction));
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    fileExists = csCheckFile(pShare, pFileName, pUser->preservesCase);

    /* if file was not opened (e.g., - not exists) take actions as required in the
       request */

    if (!fileExists)
    {
        /* file does not exist */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile),
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        if ((desiredAction & SMB_OPEN2_CREATEACTION) != SMB_OPEN2_DOCREATE)
        {
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_OPEN,
                pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile),
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            TRCERR("file not open and open function does not allow creation");
            TRCE();
            return csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
        }

        /* create file */

        TRC1P("Creating file: %s", (NQ_CHAR*)pFileName);

        if ((returnValue = createFile(FALSE, pFileName, tid, desiredAccess, &pFile)) != 0)
        {
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_OPEN,
                pUser->name,
                pUser->ip,
                returnValue,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            TRCE();
            return returnValue;
        }

        grantedAccess = SMB_ACCESS_A_READWRITE;
        takenAction = SMB_OPEN2_WASCREATED;
    }
    else
    {
        /* file exists: proceed according to the required "Open" action */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        if ((desiredAction & SMB_OPEN2_OPENACTION) == SMB_OPEN2_DOFAIL)
        {
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_OPEN,
                pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists),
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            TRCERR("file exists while the request does not allow open");
            TRCE();
            return csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists);
        }

        grantedAccess = desiredAccess;
        takenAction = SMB_OPEN2_WASOPENED;

        if ((returnValue = openFile(pFileName, tid, desiredAccess, &grantedAccess, &pFile)) != 0)
        {
            TRCE();
            return returnValue;
        }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        eventInfo.before = TRUE;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
            );
        eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        if (desiredAction & SMB_OPEN2_DOTRUNCATE)
        {
#ifdef UD_NQ_INCLUDEEVENTLOG
            eventInfo.before = TRUE;
            eventInfo.sizeHigh = 0;
            eventInfo.sizeLow = 0;
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_SIZESET,
                pUser->name,
                pUser->ip,
                0,
                (const NQ_BYTE*)&eventInfo
            );
            eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
            if (syTruncateFile(pFile->file, 0, 0) == NQ_FAIL)
            {
                error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_SIZESET,
                    pUser->name,
                    pUser->ip,
                    error,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    pUser->name,
                    pUser->ip,
                    error,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                csReleaseFile(pFile->fid);      /* also closes the file */
                TRCERR("cannot truncate the file");
                TRCE();
                return error;
            }

            takenAction = SMB_OPEN2_WASTRUNCATED;
        }
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_CREATE,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
    /* store file information */

    pFile->tid = tid;
    pFile->uid = uid;
    pFile->pid = pid;

    /* get the file attributes */

    if (NQ_SUCCESS != csGetFileInformationByName(pShare, pFileName, &fileInfo
#ifdef UD_NQ_INCLUDEEVENTLOG
            ,pUser
#endif /* UD_NQ_INCLUDEEVENTLOG */
            ))
    {
        error = csErrorGetLast();
        csReleaseFile(pFile->fid);      /* also closes the file */
        TRCERR("Unable to read file information");
        TRCE();
        return error;
    }

    fileInfo.lastWriteTime.high = 0;
    fileInfo.lastWriteTime.low = 0;
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_ATTRIBSET,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
    );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (csSetFileInformation(pFile, pFileName, &fileInfo))
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_ATTRIBSET,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_ACCESS_DENIED , DOS_ERRnoaccess),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        csReleaseFile(pFile->fid);      /* also closes the file */

        TRCERR("unable to change file attributes/time");
        TRCE();
        return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_ATTRIBSET,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
    );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    /* mark areas */

    descriptor->parameterCount = sizeof(*openResponse);
    descriptor->pData = descriptor->pParams + descriptor->parameterCount;
    descriptor->pData = cmAllignTwo(descriptor->pData);
    descriptor->dataCount = 0;

    /* compose the response */

    cmPutSUint16(openResponse->fid, cmHtol16(pFile->fid));
    cmPutSUint16(openResponse->grantedAccess, cmHtol16(grantedAccess));
    cmPutSUint16(openResponse->action, cmHtol16(takenAction));

    /* if additional file information was requested */

    if (cmLtoh16(cmGetSUint16(openRequest->flags)) & SMB_OPEN2_ADDITIONALINFORMATION)
    {
        cmPutSUint16(openResponse->fileAttributes, cmHtol16((NQ_UINT16)fileInfo.attributes));
        cmPutSUint32(openResponse->dataSize, cmHtol32(fileInfo.sizeLow));
        {
            NQ_UINT16 smbTime;     /* temporary time in SMB_TIME format */
            NQ_UINT16 smbDate;     /* temporary date in SMB_DATE format */
            cmCifsTimeToSmbTime(fileInfo.creationTime, &smbTime, &smbDate);
            cmPutSUint16(openResponse->creationTime, cmHtol16(smbTime));
            cmPutSUint16(openResponse->creationDate, cmHtol16(smbDate));
        }
    }
    else
    {
        cmPutSUint16(openResponse->fileAttributes, 0);
        cmPutSUint32(openResponse->dataSize, 0);
        cmPutSUint16(openResponse->creationTime, 0L);
        cmPutSUint16(openResponse->creationDate, 0L);
    }

    cmPutSUint16(openResponse->fileType, 0);
    cmPutSUint16(openResponse->deviceState, 0);
    cmPutSUint16(openResponse->fid, 0);
    cmPutSUint32(openResponse->reserved, 0);
    cmPutSUint16(openResponse->eaErrorOffset, 0);
    cmPutSUint32(openResponse->eaLength, 0);

    TRCE();
    return 0;
}

/*
 *====================================================================
 * PURPOSE: Perform CREATE_DIRECTORY 2 subcommand of Transaction2 protocol
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT structure with subcommand parameters:
 *              IN pointer to the CIFS header
 *              IN pointer to the parameter area
 *              OUT pointer to the data area
 *              OUT length of the parameter area
 *              OUT length of the data area
 *
 * RETURNS: SMB error or 0 on success
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT32
csTransaction2CreateDirectory(
    CSTransaction2Descriptor* descriptor
    )
{
    CMCifsCreateDirectory2Request* createRequest;    /* casted request */
    CMCifsCreateDirectory2Response* createResponse;  /* casted response */
    NQ_BOOL unicodeRequired;            /* whether client requires UNICODE */
    CMCifsStatus error;                 /* for composing DOS-style error */
    const CSShare* pShare;              /* pointer to the share */
    NQ_WCHAR* pFileName;                 /* filename to open */
    CSUid uid;                          /* required UID */
    CSTid tid;                          /* required TID */
    CSUser* pUser;                          /* pointer to the user descriptor */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent eventInfo;            /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */

    TRCB();

    /* check unicode flag */

    unicodeRequired = cmLtoh16(cmGetSUint16(descriptor->pHeaderOut->flags2)) & SMB_FLAGS2_UNICODE;

    /* check space in output buffer */

    if (((NQ_UINT)(descriptor->pParams - (NQ_BYTE*)descriptor->pHeaderOut) + sizeof(*createResponse))
        > CS_MAXBUFFERSIZE
       )
    {
        TRCERR("Buffer overflow");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* cast pointers */

    createRequest = (CMCifsCreateDirectory2Request*) (
                                (NQ_BYTE*)descriptor->requestData
                              + cmLtoh16(cmGetSUint16(descriptor->requestData->transHeader.parameterOffset))
                              - sizeof(CMCifsHeader)
                              );
    createResponse = (CMCifsCreateDirectory2Response*) descriptor->pParams;

    /* mark areas */

    descriptor->parameterCount = sizeof(*createResponse);
    descriptor->pData = descriptor->pParams + descriptor->parameterCount;
    descriptor->pData = cmAllignTwo(descriptor->pData);
    descriptor->dataCount = 0;

    /* withdraw UID, TID and PID */

    uid = (CSUid)cmLtoh16(cmGetSUint16(descriptor->pHeaderOut->uid));
    tid = (CSTid)cmLtoh16(cmGetSUint16(descriptor->pHeaderOut->tid));

    pUser = csGetUserByUid(uid);
    if (pUser == NULL)
    {
        TRCERR("Illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvuid);
    }

    pShare = csGetShareByUidTid(uid, tid);
    if (pShare == NULL)
    {
        TRCERR("Illegal TID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvtid);
    }

    /* convert filename to host filename */

    if ((pFileName = cmCifsNtohFilename(
                        fileNameBuff,
                        pShare->map,
                        (NQ_WCHAR*)(createRequest + 1),
                        unicodeRequired,
                        TRUE
                        )
        ) == NULL
       )
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.fileName = NULL;
        eventInfo.access = 0;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Illegal filename");
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_INVALID, DOS_ERRinvalidname);
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.before = TRUE;
    eventInfo.tid = tid;
    eventInfo.rid = csGetUserRid(pUser);
    eventInfo.fileName = pFileName;
    eventInfo.access = 0;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_CREATE,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */

    /* create directory */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
    if (csCheckFile(pShare, pFileName, pUser->preservesCase))
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess),
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Directory already exists");
        TRC1P(" name: %s", cmWDump(pFileName));
        return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        (pUser->preservesCase || (pShare != NULL && syWStrcmp( pShare->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_CREATE,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
    );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
    if (syCreateDirectory(pFileName) == NQ_FAIL)
    {
        error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            pUser->name,
            pUser->ip,
            error,
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Unable to create directory file");
        TRCE();
        return error;
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_CREATE,
        pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
    );
#endif /* UD_NQ_INCLUDEEVENTLOG */

    csNotifyImmediatelly(pFileName, SMB_NOTIFYCHANGE_ADDED, SMB_NOTIFYCHANGE_DIRNAME);

    cmPutSUint16(createResponse->eaErrorOffset, 0);

    TRCE();
    return 0;
}

/*====================================================================
 * PURPOSE: Open a file providing its name and desired access
 *--------------------------------------------------------------------
 * PARAMS:  IN file name
 *          IN tree ID
 *          IN desired access as in the request
 *          OUT pointer to the resulting granted access (may be NULL)
 *          OUT pointer to place a new file descriptor
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   allocates file descriptor and opens the file
 *          on error - file descriptor is deallocated
 *====================================================================
 */

static NQ_UINT32
openFile(
    const NQ_WCHAR* pFileName,
    CSTid tid,
    NQ_UINT16 desiredAccess,
    NQ_UINT16* grantedAccess,
    CSFile** pFile
    )
{
    NQ_BOOL denyRead;       /* deny flag*/
    NQ_BOOL denyWrite;      /* deny flag */
    NQ_BOOL denyExecute;    /* deny flag */
    NQ_UINT16 denyAccess;   /* all deny flags in one */
    CMCifsStatus error = 0; /* for composing DOS-style error */
    const CSTree* pTree;    /* master tree pointer */
    CSName* pName;          /* filename descriptor */
    NQ_UINT32 tempGrantedAccess;    /* temporary value of the granted access */
    NQ_BOOL readOnly;               /* whether file is read-only */
    SYFileInformation fileInfo;     /* for querying file information */
    NQ_BOOL isDirectory = FALSE;
#ifdef UD_NQ_INCLUDEEVENTLOG
    const CSUser* pUser;           /* user structure pointer */
    UDFileAccessEvent eventInfo;   /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */

    TRCB();
    TRC2P("Opening file: %s with mode: %04x", cmWDump(pFileName), desiredAccess);

    /* find tree */

    if ((pTree = csGetTreeByTid(tid)) == NULL)
    {
        TRCERR("illegal TID");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.fileName = pFileName;
    eventInfo.access = desiredAccess;
    eventInfo.tid = pTree->tid;
    pUser = csGetUserByUid(pTree->uid);
    if (NULL == pUser)
    {
        TRCERR("illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }
    eventInfo.rid = csGetUserRid((CSUser *)pUser);
#endif /* UD_NQ_INCLUDEEVENTLOG */

    /* check whether this file is already opened and marked for deletion */

    if (csFileMarkedForDeletion(pFileName))
    {
        TRCERR("File is marked for deletion");
        TRC1P(" file name: %s", cmWDump(pFileName));
        TRCE();
        return csErrorReturn(SMB_STATUS_DELETE_PENDING, DOS_ERRbadaccess);
    }

    /* check if we can delete this file */

    if (0 != (desiredAccess & SMB_ACCESS_A_DELETE) && !csCanDeleteFile(pFileName
#ifdef UD_NQ_INCLUDEEVENTLOG
                                                        ,pUser,
                                                        eventInfo
#endif /* UD_NQ_INCLUDEEVENTLOG */
                                                        ))
    {
        TRCERR("Delete access cannot be granted on the file");
        TRC1P(" file: %s", cmWDump(pFileName));
        TRCE();
        return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
    }

    if ((pName = csGetNameByName(pFileName)) == NULL)
    {
        if ((pName = csGetNewName(pFileName, pTree->uid)) == NULL)
        {
            TRCERR("NID overflow");
            TRCE();
            return csErrorReturn(SMB_STATUS_TOO_MANY_OPENED_FILES, DOS_ERRnofids);
        }
    }
    if (NQ_SUCCESS != csGetFileInformationByName(pTree->share, pName->name, &fileInfo
#ifdef UD_NQ_INCLUDEEVENTLOG
            ,pUser
#endif /* UD_NQ_INCLUDEEVENTLOG */
            ))
    {
        /* sometimes we cannot read when the file is writable and the access is
           not the 1st one */

        readOnly = FALSE;
    }
    else
    {
        readOnly = fileInfo.attributes & SMB_ATTR_READONLY;
        isDirectory = fileInfo.attributes & SMB_ATTR_DIRECTORY;
    }

    if ((tempGrantedAccess = isAccessAllowed(pName, &desiredAccess, pTree->uid, readOnly)) > 0xFFFF)
    {
        TRCERR("Access is not allowed");
        TRC3P(" file: %s, desired: %04x, old: %04x", cmWDump(pFileName), desiredAccess, (pName->first ? pName->first->access : 0));
        TRCE();
        return tempGrantedAccess;
    }
    
    if (grantedAccess != NULL)
    {
        *grantedAccess = (NQ_UINT16)tempGrantedAccess;
    }

    if ((*pFile = csGetNewFile(pTree, pName, desiredAccess)) == NULL)
    {
        TRCERR("FID overflow");
        TRCE();
        return csErrorReturn(SMB_STATUS_TOO_MANY_OPENED_FILES, DOS_ERRnofids);
    }

    syInvalidateFile(&(*pFile)->file);

    /* if file is a pipe - open it as a pipe */
    if (pTree->share->ipcFlag)
    {
#ifdef UD_CS_INCLUDERPC
        if (!csDcerpcOpenPipe(pFileName, *pFile))
        {
            TRCERR("Pipe not found");
            csReleaseFile((*pFile)->fid);
            TRCE();
            return csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
        }
#else
        TRCERR("Pseudo-files (pipes) not supported");
        TRCE();
        return csErrorReturn(SMB_STATUS_NOT_SUPPORTED, (NQ_UINT32)SRV_ERRnosupport);
#endif /* UD_CS_INCLUDERPC */
    }
    else
    {
        NQ_BOOL allowed = TRUE;    /* whether this access is allowed */

        denyAccess = desiredAccess & SMB_ACCESS_S;
        denyRead =    (denyAccess == SMB_ACCESS_S_READWRITEEXECUTE)
                   || (denyAccess == SMB_ACCESS_S_READEXECUTE)
                   || (denyAccess == SMB_ACCESS_S_FCB);
        denyWrite =   (denyAccess == SMB_ACCESS_S_READWRITEEXECUTE)
                   || (denyAccess == SMB_ACCESS_S_WRITE)
                   || (denyAccess == SMB_ACCESS_S_FCB);
        denyExecute = (denyAccess == SMB_ACCESS_S_READWRITEEXECUTE)
                   || (denyAccess == SMB_ACCESS_S_READEXECUTE)
                   || (denyAccess == SMB_ACCESS_S_FCB);

        switch (desiredAccess & SMB_ACCESS_A)
        {
        case SMB_ACCESS_A_NONE:
        case SMB_ACCESS_A_READ:
        case SMB_ACCESS_A_EXECUTE:
            error = csCanReadShare(tid);
            allowed = (error == NQ_SUCCESS);
            break;
        case SMB_ACCESS_A_WRITE:
        case SMB_ACCESS_A_FCB:
        case SMB_ACCESS_A_READWRITE:
            error = csCanWriteShare(tid);
            allowed = (error == NQ_SUCCESS);
            break;      
        default:
            allowed = TRUE;
            desiredAccess &= (NQ_UINT16)(~SMB_ACCESS_A);
            desiredAccess |= SMB_ACCESS_A_READ;
        }

        if (!allowed)
        {
            TRCERR("Access not allowed");
            csReleaseFile((*pFile)->fid);
            TRCE();
            return error;
        }

        /* for directory clear all SMB_ACCESS_A except for SMB_ACCESS_A_READ as directories
           can be open with read access ONLY (at least on Linux) */
        if (isDirectory)
            desiredAccess &= (NQ_UINT16)(~SMB_ACCESS_A | SMB_ACCESS_A_READ);
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.before = TRUE;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_OPEN,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
            );
        eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
        switch (desiredAccess & SMB_ACCESS_A)
        {
        case SMB_ACCESS_A_NONE:
            TRC("SMB_ACCESS_A_NONE");
#ifdef SY_FS_SEPARATEINFOMODE
            (*pFile)->file = syOpenFileForInfo(pFileName, denyRead, denyExecute, denyWrite);
#else
            (*pFile)->file = syOpenFileForRead(pFileName, denyRead, denyExecute, denyWrite);
#endif /* SY_FS_SEPARATEINFOMODE */
            break;
        case SMB_ACCESS_A_READ:
            TRC("SMB_ACCESS_A_READ");
            (*pFile)->file = syOpenFileForRead(pFileName, denyRead, denyExecute, denyWrite);
            break;
        case SMB_ACCESS_A_WRITE:
            TRC("SMB_ACCESS_A_WRITE");
            (*pFile)->file = syOpenFileForWrite(pFileName, denyRead, denyExecute, denyWrite);
            break;
        case SMB_ACCESS_A_FCB:
        case SMB_ACCESS_A_READWRITE:
            TRC("SMB_ACCESS_A_READWRITE");
            (*pFile)->file = syOpenFileForReadWrite(pFileName, denyRead, denyExecute, denyWrite);
            break;
        case SMB_ACCESS_A_EXECUTE:
            TRC("SMB_ACCESS_A_EXECUTE");
            (*pFile)->file = syOpenFileForRead(pFileName, denyRead, denyExecute, denyWrite);
            break;
        default:
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_OPEN,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_ACCESS_VIOLATION, DOS_ERRbadaccess),
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            TRCERR("Illegal open mode");
            TRC1P(" open mode: %08x", desiredAccess);
            csReleaseFile((*pFile)->fid);
            TRCE();
            return csErrorReturn(SMB_STATUS_ACCESS_VIOLATION, DOS_ERRbadaccess);
        }

        if (!syIsValidFile((*pFile)->file))
        {
            error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_OPEN,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                error,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            csReleaseFile((*pFile)->fid);
            TRCERR("Unable to open file");
            return error;
        }

#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_OPEN,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
        );
        (*pFile)->uid = pUser->uid;
        (*pFile)->tid = tid;
#endif /* UD_NQ_INCLUDEEVENTLOG */
        /* set Archive attribute */

        if ((desiredAccess & SMB_ACCESS_A) != SMB_ACCESS_A_READ && (desiredAccess & SMB_ACCESS_A) != SMB_ACCESS_A_NONE)
        {
            SYFileInformation fileInfo;         /* for querying file information */

            if (csGetFileInformation((*pFile), pFileName, &fileInfo) != NQ_SUCCESS)
            {
                error =  csErrorGetLast();

                csReleaseFile((*pFile)->fid);
                TRCERR("Unable to read file information");
                return error;
            }

            fileInfo.attributes |= SMB_ATTR_ARCHIVE;
            if (csSetFileInformation((*pFile), pFileName, &fileInfo) != NQ_SUCCESS)
            {
                error = csErrorGetLast();
                csReleaseFile((*pFile)->fid);
                TRCERR("Unable to set the Archive bit");
                return error;
            }
        }

    }

    (*pFile)->isDirectory = FALSE;
    TRCE();
    return 0;
}

/*====================================================================
 * PURPOSE: Create a file providing its name
 *--------------------------------------------------------------------
 * PARAMS:  IN TRUE for directory
 *          IN file name
 *          IN master tree
 *          OUT pointer to place a new file descriptor
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   allocates file descriptor and creates the file
 *          on error - file descriptor is deallocated
 *====================================================================
 */

static NQ_UINT32
createFile(
    NQ_BOOL directoryRequired,
    const NQ_WCHAR* pFileName,
    CSTid tid,
    NQ_UINT16 desiredAccess,
    CSFile** pFile
    )
{
    NQ_BOOL denyRead;               /* deny flag*/
    NQ_BOOL denyWrite;              /* deny flag */
    NQ_BOOL denyExecute;            /* deny flag */
    NQ_UINT16 denyAccess;           /* all deny flags in one */
    CMCifsStatus error;             /* for composing DOS-style error */
    const CSTree* pTree;            /* master tree pointer */
    CSName* pName;                  /* filename descriptor */
    CSUser* pUser;                  /* pointer to the user descriptor */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent eventInfo;    /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */

    TRCB();
    TRC2P("Creating file: %s with mode: %04x", cmWDump(pFileName), desiredAccess);

    if ((desiredAccess & SMB_ACCESS_A) == SMB_ACCESS_A_NONE)
    {
        desiredAccess = (NQ_UINT16)(desiredAccess & ~(SMB_ACCESS_A | SMB_ACCESS_S));    /* share all */
        desiredAccess |= SMB_ACCESS_A_READ;
    }
    if ((error = csCanWriteShare(tid)) != NQ_SUCCESS)
    {
        TRCERR("Access not allowed");
        TRCE();
        return error;
    }

    /* allocate file descriptor */

    if ((pTree = csGetTreeByTid(tid)) == NULL)
    {
        TRCERR("illegal TID");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }
    pUser = csGetUserByUid(pTree->uid);
    if (pUser == NULL)
    {
        TRCERR("Illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvuid);
    }

#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.fileName = pFileName;
    eventInfo.access = desiredAccess;
    eventInfo.tid = tid;
    eventInfo.rid = csGetUserRid(pUser);
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */

    if ((pName = csGetNameByName(pFileName)) != NULL)
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("File already exists");
        TRC1P(" file: %s", cmWDump(pFileName));
        TRCE();
        return csErrorReturn(SMB_STATUS_OBJECT_NAME_COLLISION, DOS_ERRfileexists);
    }
    if ((pName = csGetNewName(pFileName, pTree->uid)) == NULL)
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            csErrorReturn(SMB_STATUS_TOO_MANY_OPENED_FILES, DOS_ERRnofids),
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("NID overflow");
        TRCE();
        return csErrorReturn(SMB_STATUS_TOO_MANY_OPENED_FILES, DOS_ERRnofids);
    }

    if ((*pFile = csGetNewFile(pTree, pName, desiredAccess)) == NULL)
    {
        TRCERR("FID overflow");
        TRCE();
        return csErrorReturn(SMB_STATUS_TOO_MANY_OPENED_FILES, DOS_ERRnofids);
    }

    (*pFile)->options = 0; /* no options so far */

    if (directoryRequired)
    {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        eventInfo.before = TRUE;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            (pUser->preservesCase || (pTree->share != NULL && syWStrcmp( pTree->share->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
            );
        eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        if (csCheckFile(pTree->share, (NQ_WCHAR*)pFileName, pUser->preservesCase))
        {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                (pUser->preservesCase || (pTree->share != NULL && syWStrcmp( pTree->share->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
                pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess),
                (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
           udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess),
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            TRCERR("Directory already exists");
            TRC1P(" name: %s", cmWDump(pFileName));
            TRCE();
            return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
        }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            (pUser->preservesCase || (pTree->share != NULL && syWStrcmp( pTree->share->map , pFileName) == 0)) ? UD_LOG_FILE_ATTRIBGET : UD_LOG_FILE_QUERYDIRECTORY,
            pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.before = TRUE;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
        );
        eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
        if (syCreateDirectory(pFileName) == NQ_FAIL)
        {
            error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                error,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            csReleaseFile((*pFile)->fid);
            TRCERR("Unable to create directory");
            TRCE();
            return error;
        }
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        csNotifyImmediatelly(pFileName, SMB_NOTIFYCHANGE_ADDED, SMB_NOTIFYCHANGE_DIRNAME);

#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.before = TRUE;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_OPEN,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
        );
        eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
        (*pFile)->directory = syOpenDirectory((NQ_WCHAR*)pFileName);
        if (!syIsValidDirectory((*pFile)->directory))
        {
            error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_OPEN,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                error,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            csReleaseFile((*pFile)->fid);
            TRCERR("Unable to create directory");
            TRCE();
            return error;
        }
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_OPEN,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
    }
    else
    {
        SYFileInformation fileInfo;         /* for querying file information */

        denyAccess = desiredAccess & SMB_ACCESS_S;
        denyRead =    (denyAccess == SMB_ACCESS_S_READWRITEEXECUTE)
                   || (denyAccess == SMB_ACCESS_S_READEXECUTE)
                   || (denyAccess == SMB_ACCESS_S_FCB);
        denyWrite =   (denyAccess == SMB_ACCESS_S_READWRITEEXECUTE)
                   || (denyAccess == SMB_ACCESS_S_WRITE)
                   || (denyAccess == SMB_ACCESS_S_FCB);
        denyExecute = (denyAccess == SMB_ACCESS_S_READWRITEEXECUTE)
                   || (denyAccess == SMB_ACCESS_S_READEXECUTE)
                   || (denyAccess == SMB_ACCESS_S_FCB);
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.before = TRUE;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
        );
        eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
        (*pFile)->file = syCreateFile(pFileName, denyRead, denyExecute, denyWrite);
        if (!syIsValidFile((*pFile)->file))
        {
            error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                error,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            csReleaseFile((*pFile)->fid);
            TRCERR("Unable to create file");
            TRCE();
            return error;
        }
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_CREATE,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        {
            /* close and reopen file with desired access mode */
#ifdef UD_NQ_INCLUDEEVENTLOG
            eventInfo.before = TRUE;
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CLOSE,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                0,
                (const NQ_BYTE*)&eventInfo
            );
            eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
            syCloseFile((*pFile)->file);
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CLOSE,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                (NQ_UINT32)syGetLastError(),
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */

#ifdef UD_NQ_INCLUDEEVENTLOG
            eventInfo.before = TRUE;
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_OPEN,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                0,
                (const NQ_BYTE*)&eventInfo
                );
            eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
            switch (desiredAccess & SMB_ACCESS_A)
            {
            case SMB_ACCESS_A_NONE:
#ifdef SY_FS_SEPARATEINFOMODE
                (*pFile)->file = syOpenFileForInfo(pFileName, denyRead, denyExecute, denyWrite);
#else
                (*pFile)->file = syOpenFileForRead(pFileName, denyRead, denyExecute, denyWrite);
#endif /* SY_FS_SEPARATEINFOMODE */
                break;
            case SMB_ACCESS_A_READ:
                TRC("SMB_ACCESS_A_READ");
                (*pFile)->file = syOpenFileForRead(pFileName, denyRead, denyExecute, denyWrite);
                break;
            case SMB_ACCESS_A_WRITE:
                TRC("SMB_ACCESS_A_WRITE");
                (*pFile)->file = syOpenFileForWrite(pFileName, denyRead, denyExecute, denyWrite);
                break;
            case SMB_ACCESS_A_FCB:
            case SMB_ACCESS_A_READWRITE:
                TRC("SMB_ACCESS_A_READWRITE");
                (*pFile)->file = syOpenFileForReadWrite(pFileName, denyRead, denyExecute, denyWrite);
                break;
            case SMB_ACCESS_A_EXECUTE:
                TRC("SMB_ACCESS_A_EXECUTE");
                (*pFile)->file = syOpenFileForRead(pFileName, denyRead, denyExecute, denyWrite);
                break;
            default:
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    (NQ_WCHAR *)pUser->name,
                    pUser->ip,
                    csErrorReturn(SMB_STATUS_ACCESS_VIOLATION, DOS_ERRbadaccess),
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                TRCERR("Illegal open mode: %08x", desiredAccess);
                csReleaseFile((*pFile)->fid);
                TRCE();
                return csErrorReturn(SMB_STATUS_ACCESS_VIOLATION, DOS_ERRbadaccess);
            }

            if (!syIsValidFile((*pFile)->file))
            {
                error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_OPEN,
                    (NQ_WCHAR *)pUser->name,
                    pUser->ip,
                    error,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                csReleaseFile((*pFile)->fid);
                TRCERR("Unable to open file");
                TRCE();
                return error;
            }
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_OPEN,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                0,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
        }
        TRC("File handle: %d", (*pFile)->file);

        csNotifyImmediatelly(pFileName, SMB_NOTIFYCHANGE_ADDED, SMB_NOTIFYCHANGE_FILENAME);

        /* set Archive attribute */
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.before = TRUE;
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_ATTRIBGET,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
        );
        eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
        if (syGetFileInformationByName(pFileName, &fileInfo) != NQ_SUCCESS)
        {
            error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_ATTRIBGET,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                error,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            csReleaseFile((*pFile)->fid);
            TRCERR("Unable to read file information");
            TRCE();
            return error;
        }
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_ATTRIBGET,
            (NQ_WCHAR *)pUser->name,
            pUser->ip,
            0,
            (const NQ_BYTE*)&eventInfo
        );
        (*pFile)->user = pUser;
        (*pFile)->uid = pUser->uid;
        (*pFile)->tid = tid;
#endif /* UD_NQ_INCLUDEEVENTLOG */

        fileInfo.attributes |= SMB_ATTR_ARCHIVE;
        if (csSetFileInformation((*pFile), pFileName, &fileInfo) != NQ_SUCCESS)
        {
            error = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                error,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEVENTLOG */
            csReleaseFile((*pFile)->fid);
            TRCERR("Unable to set the Archive bit");
            TRCE();
            return error;
        }
    }

    (*pFile)->isDirectory = directoryRequired;
    TRCE();
    return 0;
}

/*====================================================================
 * PURPOSE: convert NT-style access into OPEN-style (POSIX) access
 *--------------------------------------------------------------------
 * PARAMS:  IN desired access in NT style
 *          IN sharing in NT style (deny flags)
 *
 * RETURNS: desired access
 *
 * NOTES:
 *====================================================================
 */

static NQ_UINT16
convertNtAccessToDesiredAccess(
    NQ_UINT32 ntAccess,
    NQ_UINT32 ntShare
    )
{
    NQ_UINT16 desiredAccess = SMB_ACCESS_A_READ;

    TRCB();
    TRC("--------->NT access: 0x%x, share: 0x%x", ntAccess, ntShare); 

    /* concern the requested access mode */

    if ((ntAccess & (NQ_UINT32)(~(SMB_DESIREDACCESS_SYNCHRONISE | SMB_DESIREDACCESS_READATTRIBUTES | SMB_DESIREDACCESS_WRITEATTRIBUTES))) == 0)
    {
        desiredAccess = SMB_ACCESS_A_NONE;
    }
    else
    {
        if (ntAccess & SMB_DESIREDACCESS_GENMASK)
        {
            if (ntAccess & SMB_DESIREDACCESS_GENALL)
                ntAccess |= ( SMB_DESIREDACCESS_WRITEDATA | SMB_DESIREDACCESS_READDATA 
                            | SMB_DESIREDACCESS_EXECUTE | SMB_DESIREDACCESS_DELETE);
            else
            {
                if (ntAccess & SMB_DESIREDACCESS_GENWRITE)
                    ntAccess |= SMB_DESIREDACCESS_WRITEDATA;
                if (ntAccess & SMB_DESIREDACCESS_GENREAD)
                    ntAccess |= SMB_DESIREDACCESS_READDATA;    
                if (ntAccess & SMB_DESIREDACCESS_GENEXECUTE)
                    ntAccess |= SMB_DESIREDACCESS_EXECUTE; 
            }    
        }

        if (0 == (ntAccess & (SMB_DESIREDACCESS_WRITEDATA | SMB_DESIREDACCESS_READDATA)))
        {
            if (ntAccess & SMB_DESIREDACCESS_READATTRIBUTES)
                ntAccess |= SMB_DESIREDACCESS_READDATA;  
            if (ntAccess & SMB_DESIREDACCESS_WRITEATTRIBUTES)
                ntAccess |= SMB_DESIREDACCESS_READDATA;
        }

        
        switch (ntAccess &
                (  SMB_DESIREDACCESS_READDATA
                 | SMB_DESIREDACCESS_WRITEDATA
                 | SMB_DESIREDACCESS_APPENDDATA
                )
               )
        {
            case SMB_DESIREDACCESS_READDATA:
                desiredAccess = SMB_ACCESS_A_READ;
                break;
            case SMB_DESIREDACCESS_WRITEDATA:
            case SMB_DESIREDACCESS_APPENDDATA:
            case SMB_DESIREDACCESS_WRITEDATA|SMB_DESIREDACCESS_APPENDDATA:
                desiredAccess = SMB_ACCESS_A_WRITE;
                break;
            case SMB_DESIREDACCESS_READDATA|SMB_DESIREDACCESS_WRITEDATA:
            case SMB_DESIREDACCESS_READDATA|SMB_DESIREDACCESS_WRITEDATA|SMB_DESIREDACCESS_APPENDDATA:
            case SMB_DESIREDACCESS_READDATA|SMB_DESIREDACCESS_APPENDDATA:
                desiredAccess = SMB_ACCESS_A_READWRITE;
                break;
            default:
                desiredAccess = SMB_ACCESS_A_NONE;
                break;
        }
    }

    if (ntAccess & SMB_DESIREDACCESS_DELETE)
    {
        desiredAccess |= SMB_ACCESS_A_DELETE;
#ifdef SY_FS_SEPARATEINFOMODE
        if ((desiredAccess & SMB_ACCESS_A) == 0)
            desiredAccess |= SMB_ACCESS_A_NONE;
#endif /* SY_FS_SEPARATEINFOMODE */
    }

    /* allow delete sharing when there is no data access */

    if (!(ntAccess & (SMB_DESIREDACCESS_READDATA|SMB_DESIREDACCESS_WRITEDATA|SMB_DESIREDACCESS_APPENDDATA)))
    {
        desiredAccess |= SMB_ACCESS_S_DELETE;
    }

    /* concern the requested share mode */

    switch (ntShare & (SMB_SHAREACCESS_READ|SMB_SHAREACCESS_WRITE))
    {
        case SMB_SHAREACCESS_READ:
            desiredAccess |= SMB_ACCESS_S_WRITE;
            break;
        case SMB_SHAREACCESS_WRITE:
            desiredAccess |= SMB_ACCESS_S_READEXECUTE;
            break;
        case SMB_SHAREACCESS_READ|SMB_SHAREACCESS_WRITE:
            desiredAccess |= SMB_ACCESS_S_NONE;
            break;
        case SMB_SHAREACCESS_NONE:
            desiredAccess |= SMB_ACCESS_S_FCB;
            break;
    }

    if (ntShare & SMB_SHAREACCESS_DELETE)
    {
        desiredAccess |= SMB_ACCESS_S_DELETE;
    }
    
    if ((desiredAccess & SMB_ACCESS_A) ==  SMB_ACCESS_A_NONE)
    {
        desiredAccess &= (NQ_UINT16)(~SMB_ACCESS_S);     /* share all */
        desiredAccess |= SMB_ACCESS_S_NONE;
    }

    TRC("--------->desiredAccess: 0x%x", desiredAccess);    
    TRCE();

    return desiredAccess;
}

/*====================================================================
 * PURPOSE: check if another open may happen
 *--------------------------------------------------------------------
 * PARAMS:  IN name descriptor
 *          IN/OUT pointer to new access bits. This value may be modified
 *          IN new client UID
 *          IN whether the file is readonly
 *
 * RETURNS: error code or 0
 *
 * NOTES:
 *====================================================================
 */

static NQ_UINT32
isAccessAllowed(
    CSName* pName,
    NQ_UINT16* newAccess,
    CSUid newUid,
    NQ_BOOL readOnly
    )
{
    NQ_BOOL isExe;              /* TRUE when this file has a known executable extension */
    NQ_WCHAR* pExtension;        /* file extension pointer */
    NQ_UINT16 denyOld;          /* old share mode */
    NQ_UINT16 denyNew;          /* new share mode */
    NQ_UINT16 openOld;          /* old open mode */
    NQ_UINT16 openNew;          /* new open mode */
    NQ_INT grantedIndex;        /* index in the granted access table */
    NQ_UINT16 grantedAccess;    /* resulting access */

    TRCB();

    /* extract access and deny modes */
    denyNew = *newAccess & SMB_ACCESS_S;
    openNew = *newAccess & SMB_ACCESS_A;
 
    if (SMB_ACCESS_A_NONE == openNew && 
        (0 == (*newAccess & SMB_ACCESS_A_DELETE) || pName->first == NULL)
       )
    {
        *newAccess &= (NQ_UINT16)(~SMB_ACCESS_A);
#ifdef SY_FS_SEPARATEINFOMODE
        *newAccess |= SMB_ACCESS_A_NONE;
#else
        *newAccess |= SMB_ACCESS_A_READ;
#endif /* SY_FS_SEPARATEINFOMODE */        
        TRCE();
        return GRANT_READ;
    }

    pExtension = syWStrrchr(pName->name, cmWChar('.'));

    if (pExtension != NULL && (NQ_UINT)syWStrlen(pExtension) == 4)
    {
        NQ_CHAR extension[10];
        NQ_UINT i;

        syUnicodeToAnsi(extension, pExtension);

        for (i = 0; i < 4; i++)
        {
            extension[i] = syToupper(extension[i]);
        }

        isExe =    syStrcmp(extension, ".EXE") == 0
                || syStrcmp(extension, ".DLL") == 0
                || syStrcmp(extension, ".SYM") == 0
                || syStrcmp(extension, ".COM") == 0;
    }
    else
        isExe = FALSE;

    /* restrict EXECUTABLE to readonly files whose extension is not special */

    if (openNew == SMB_ACCESS_A_EXECUTE)
    {
        if (!isExe && readOnly)
        {
            TRCE();
            return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
        }
    }

    /* immediatelly return success for the first file */

    if (pName->first == NULL)
    {
        /* disable opening read only files for writing */
        if (readOnly)
            switch (openNew)
            {
                case SMB_ACCESS_A_WRITE     :
                case SMB_ACCESS_A_READWRITE : 
                case SMB_ACCESS_A_FCB       :  
                    TRCE();
                    return csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
            }
        TRCE();
        return (NQ_UINT32)((openNew == SMB_ACCESS_A_FCB) ? SMB_ACCESS_A_READWRITE : openNew);
    }

    /* the second open may access for delete only when the 1st one has share for delete */
    if (   (   (pName->first->access & SMB_ACCESS_S_DELETE) ==0
            && (*newAccess & SMB_ACCESS_A_DELETE) != 0
           )
       )
    {
        TRCE();
        return csErrorReturn(SMB_STATUS_SHARING_VIOLATION, DOS_ERRnoaccess);
    }
    
    /* calculate index in the granted access table */

    denyOld = pName->first->access & SMB_ACCESS_S;
    openOld = pName->first->access & SMB_ACCESS_A;

    denyOld = (NQ_UINT16)(denyOld >> 4);
    denyNew = (NQ_UINT16)(denyNew >> 4);
    openOld = accessCodeTable[openOld];
    denyOld = denyCodeTable[denyOld];
    openNew = accessCodeTable[openNew];
    denyNew = denyCodeTable[denyNew];

    grantedIndex = denyOld;
    grantedIndex = grantedIndex * ACCESS_SIZE + openOld;
    grantedIndex = grantedIndex * DENY_SIZE + denyNew;
    grantedIndex = grantedIndex * ACCESS_SIZE + openNew;

    if (pName->uid == newUid)
        if (isExe)
            grantedAccess = grantedAccessTable[grantedIndex].exeSame;
        else
            grantedAccess = grantedAccessTable[grantedIndex].nonSame;
    else
        if (isExe)
            grantedAccess = grantedAccessTable[grantedIndex].exeDiff;
        else
            grantedAccess = grantedAccessTable[grantedIndex].nonDiff;

    /* delegate access for delete from the 1st open to the next */

    if ((grantedAccess != GRANT_NONE) && (pName->first->access & SMB_ACCESS_A_DELETE))
    {
        *newAccess |= SMB_ACCESS_A_DELETE;
    }
    grantedAccess = (NQ_UINT16)(grantedAccess | (*newAccess & SMB_ACCESS_A_DELETE));

    TRCE();
    return grantedAccess == GRANT_NONE ? csErrorReturn(SMB_STATUS_SHARING_VIOLATION, DOS_ERRbadshare) : (NQ_UINT32)grantedAccess;
}

#ifdef UD_CS_INCLUDERPC_SPOOLSS
/*====================================================================
 * PURPOSE: Create a file for printing providing its name and desired access
 *--------------------------------------------------------------------
 * PARAMS:  IN file name
 *          IN share descriptor
 *          IN tree ID
 *          IN desired access as in the request
 *          OUT pointer to place a new file descriptor
 *          IN user context
 *          IN whether to create spooler file (SMB1 not relevant)
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   allocates file descriptor and opens the printer 
 *          on error - file descriptor is deallocated
 *====================================================================
 */
static 
NQ_UINT32
createPrintFile(
    const NQ_WCHAR* pFileName,
    const CSShare* pShare,
    CSTid tid,
    NQ_UINT16 desiredAccess,
    CSFile** pFile,
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
    NQ_BOOL createSpoolerFile,
#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
    CSUser * pUser
    )
{
    const CSTree* pTree;            /* master tree pointer */
    CSName* pName;                  /* filename descriptor */
    CMSdSecurityDescriptor sd;      /* security descriptor */
    CMBlob sdBlob;                  /* security descriptor blob */
    SYPrinterHandle prHandle;       /* printer handle */
    NQ_UINT32 result;               /* operation result */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent   eventInfo;
#endif /*UD_NQ_INCLUDEEVENTLOG*/

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "pFileName:%s pShare:%p tid:%d desiredAccess:0x%x pFile:%p pUser:%p", cmWDump(pFileName), pShare, tid, desiredAccess, pFile, pUser);
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
    LOGMSG(CM_TRC_LEVEL_FUNC_TOOL, "createSpoolerFile:%s", createSpoolerFile ? "yes" : "no");
#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.before = TRUE;
    eventInfo.fileName = pFileName;
    eventInfo.tid = tid;
    eventInfo.rid = csGetUserRid(pUser);
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_CREATE,
        (NQ_WCHAR *)pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
    );
    eventInfo.before = FALSE;
#endif /*UD_NQ_INCLUDEEVENTLOG*/
    if (FALSE == pShare->isPrintQueue)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not a printer share");
        result = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        goto Exit;
    }
  
    if (NULL == (pTree = csGetTreeByTid(tid)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal TID");
        result = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        goto Exit;
    }

    if (NULL == (pName = csGetNewName(pFileName, pTree->uid)))
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_TOO_MANY_OPENED_FILES, DOS_ERRnofids),
                (const NQ_BYTE*)&eventInfo
        );
#endif /*UD_NQ_INCLUDEEVENTLOG*/
        LOGERR(CM_TRC_LEVEL_ERROR, "NID overflow");
        result = csErrorReturn(SMB_STATUS_TOO_MANY_OPENED_FILES, DOS_ERRnofids);
        goto Exit;
    }

    if (NULL == (*pFile = csGetNewFile(pTree, pName, desiredAccess)))
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_TOO_MANY_OPENED_FILES, DOS_ERRnofids),
                (const NQ_BYTE*)&eventInfo
        );
#endif /*UD_NQ_INCLUDEEVENTLOG*/
        LOGERR(CM_TRC_LEVEL_ERROR, "FID overflow");
        result = csErrorReturn(SMB_STATUS_TOO_MANY_OPENED_FILES, DOS_ERRnofids);
        goto Exit;
    }
    (*pFile)->isPrint = TRUE;

    prHandle = syGetPrinterHandle(pShare->map);
    if (FALSE == syIsValidPrinter(prHandle))
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_CREATE,
                (NQ_WCHAR *)pUser->name,
                pUser->ip,
                csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile),
                (const NQ_BYTE*)&eventInfo
        );
#endif /*UD_NQ_INCLUDEEVENTLOG*/
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid printer handle");
        csReleaseFile((*pFile)->fid);
        result =  csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
        goto Exit;
    }
    (*pFile)->printerHandle = prHandle;
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
    if (TRUE == createSpoolerFile)
    {
        /* spooler file is used to gather chunks of data in a proper order before actual sending to printer,
           relevant for SMB2 only where data can arrive not in sequential order,
           not relevant for SMB1, there data arrive sequentially
         */
        (*pFile)->fileSpooler = syCreateFile(pFileName, FALSE, FALSE, FALSE);
        if (FALSE == syIsValidFile((*pFile)->fileSpooler))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to create printer spooler file");
            csReleaseFile((*pFile)->fid);
            result =  csErrorReturn(SMB_STATUS_OBJECT_NAME_NOT_FOUND, DOS_ERRbadfile);
            goto Exit;
        }
    }
#else
    {
        NQ_INT i = 0;

        /* Allocate an array of pointers for the new disordered writes packets */
        (*pFile)->memorySpooler.items = cmMemoryAllocate(UD_CS_SPOOLSS_PRN_MAXDISORDEREDPACKETS * sizeof(CSPrnDisorderedData*));
        if (NULL == (*pFile)->memorySpooler.items)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to allocate disordered writes buffer and start the print job");
            csReleaseFile((*pFile)->fid);
            result = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
            goto Exit;
        }

        for (i = 0 ; i < UD_CS_SPOOLSS_PRN_MAXDISORDEREDPACKETS ; i++)
        {
            (*pFile)->memorySpooler.items[i] = NULL;
        }

        (*pFile)->memorySpooler.context = cmMemoryAllocate(UD_CS_SPOOLSS_PRN_MAXDISORDEREDPACKETS * sizeof(CSPrnDisorderedContext));
        if (NULL == (*pFile)->memorySpooler.context)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to allocate disordered writes context");
            csReleaseFile((*pFile)->fid);
            result = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
            goto Exit;
        }

        (*pFile)->memorySpooler.currentPosition = 0;
        cmU64Zero(&((CSPrnDisorderedContext*)((*pFile)->memorySpooler.context))->currentOffset);
        (*pFile)->memorySpooler.numberOfItems = 0;
        (*pFile)->memorySpooler.divider = 0;
        cmListStart(&(*pFile)->memorySpooler.memoryPool);
    }

#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
    sdBlob.data = sd.data;
    sdBlob.len = sizeof(&sd.length);
    cmSdGetDefaultSecurityDescriptorByToken(&pUser->token, &sdBlob);
    sd.length = sdBlob.len;
    (*pFile)->file = (SYFile)syStartPrintJob(prHandle, pFileName, pFileName, NULL, (const NQ_BYTE *)&sd.data, (NQ_COUNT)sd.length, pUser);
    if (FALSE == syIsValidFile((NQ_INT)(*pFile)->file))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to start print job");
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
        if (TRUE == createSpoolerFile)
        {
            syCloseFile((*pFile)->fileSpooler);
            syInvalidateFile(&((*pFile)->fileSpooler));
            syDeleteFile(pFileName);
        }
#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
        csReleaseFile((*pFile)->fid);
        result = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        goto Exit;
    }
#ifdef UD_NQ_INCLUDEEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_CREATE,
        (NQ_WCHAR *)pUser->name,
        pUser->ip,
        0,
        (const NQ_BYTE*)&eventInfo
    );
#endif /*UD_NQ_INCLUDEEVENTLOG*/

    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result:0x%x", result);
    return result;
}

/*====================================================================
 * PURPOSE: Perform OPEN_PRINT_FILE command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComOpenPrintFile(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    
    CMCifsOpenPrintFileRequest* openRequest;     /* casted request */
    CMCifsOpenPrintFileResponse* openResponse;   /* casted response */
    CSUid uid;                                   /* required UID */
    CSTid tid;                                   /* required TID */
    CSPid pid;                                   /* required PID */
    NQ_UINT32 returnValue;                       /* error code in NT format or 0 for no error */
    const CSShare* pShare;                       /* pointer to the share */
    CSUser* pUser;                               /* pointer to the user descriptor */
    CSFile* pFile;                               /* pointer to file descriptor */
    NQ_STATIC NQ_WCHAR printFileName[CM_BUFFERLENGTH(NQ_WCHAR, 21)];/* print filename */
    NQ_STATIC NQ_CHAR noName[] = "[Name Not Available]";

    TRCB();

    /* check space in output buffer */

    if ((returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, sizeof(*openResponse))
        ) != 0
       )
    {
        TRCE();
        return returnValue;
    }

    /* cast pointers */

    openRequest = (CMCifsOpenPrintFileRequest*) pRequest;
    openResponse = (CMCifsOpenPrintFileResponse*) *pResponse;

    /* check counts */

    if (   openRequest->wordCount != SMB_OPENPRINT_REQUEST_WORDCOUNT
        || cmLtoh16(cmGetSUint16(openRequest->byteCount)) < SMB_OPENRPINT_REQUEST_MINBYTES
        || openRequest->bufferFormat != SMB_OPENRPINT_REQUEST_BUFFERFORMAT)
    {
        TRCERR("Illegal wordCount, byteCount or bufferFormat");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* withdraw UID, TID and PID */

    uid = cmLtoh16(cmGetSUint16(pHeaderOut->uid));
    tid = cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    pid = csGetPidFromHeader(pHeaderOut);

    pUser = csGetUserByUid(uid);
    if (pUser == NULL)
    {
        TRCERR("Illegal UID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvuid);
    }

    pShare = csGetShareByUidTid(uid, tid);
    if (pShare == NULL)
    {
        TRCERR("Illegal TID");
        TRCE();
        return csErrorReturn(SMB_STATUS_INVALID_PARAMETER, SRV_ERRinvtid);
    }

    syAnsiToUnicode(printFileName, noName);
    if (createPrintFile(printFileName, pShare, tid, SMB_ACCESS_A_READWRITE, &pFile,
#ifdef UD_CS_SPOOLSS_PRN_USESPOOLERFILE
            FALSE,
#endif /* UD_CS_SPOOLSS_PRN_USESPOOLERFILE */
            pUser) != NQ_SUCCESS)
    {
        TRCERR("Failed to create a printer file");
        TRCE();
        return csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
    }

    /* store file information */
    pFile->tid = tid;
    pFile->uid = uid;
    pFile->pid = pid;

    /* advance the outgoing response pointer */

    *pResponse += sizeof(*openResponse);

    /* compose the response */

    openResponse->wordCount = SMB_OPENRPINT_RESPONSE_WORDCOUNT;
    cmPutSUint16(openResponse->fid, cmHtol16(pFile->fid));
    cmPutSUint16(openResponse->byteCount, cmHtol16(SMB_OPENRPINT_RESPONSE_BYTECOUNT));

    TRCE();
    return 0;
}

#endif /* UD_CS_INCLUDERPC_SPOOLSS */

#endif /* UD_NQ_INCLUDECIFSSERVER */

