/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : CIFS Server Interface
 *--------------------------------------------------------------------
 * MODULE        : NQ
 * DEPENDENCIES  :
 ********************************************************************/

#ifndef _CSAPI_H_
#define _CSAPI_H_

#include "cmapi.h"

#if defined(UD_NQ_INCLUDECIFSSERVER) || defined(DOXYGEN)

/** @defgroup csapi Server */

/**@defgroup cs_init_shut Initialization and Shutdown
 * @ingroup csapi
 * @{
 */

/**
   This function starts the CIFS Server, must be called after NetBIOS daemon startup.
   See ndStart() in ndapi.h file.
   @return
   This function returns NQ_SUCCESS if the server was started
   successfully or NQ_FAIL otherwise.
   @see csStop()                                               */
NQ_STATUS csStart(void);

/**
   This function stops the CIFS Server.
   @see csStart()                                             */
void csStop(void);

/** @} end of group cs_init_shut Initialization and Shutdown */

/**@defgroup cs_share Shares Management
 * @ingroup csapi
 * @{
 */

/**
   This function adds another share to NQ Server.
   @param name        Name for new share. This name must not
                      contain the '$' character.
   @param path        Local path to the share root.
   @param printQueue  FALSE for file system, TRUE for print
                      queue.
   @param comment     Share description as a free text.
   @param reserved    For future use. This value is currently
                      ignored and may be any, including NULL.
   @return

   value  | meaning
   -------|--------
   0      | Success
   -1     | the DB was not initialized
   -2     | parameter error (string too long)
   -3     | share table full
   -6     | share already exists

   @note
   \* This function should be called from the context of NQ Server thread.
   \* It is recommended to sue it on NQ Server startup only.
   \* This is an ASCII version of csAddShare(). */
NQ_INT csAddShareA(const NQ_CHAR * name, const NQ_CHAR * path, NQ_BOOL printQueue, const NQ_CHAR * comment, const NQ_CHAR * reserved);

/**
   This function adds another share to NQ Server.
   Parameters
   @param name         Name for new share. This name must not contain the '$' character.
   @param path         Local path to the share root.
   @param printQueue   FALSE for file system, TRUE for print queue.
   @param comment      Share description as a free text.
   @param reserved     For future use. This value is currently ignored
                 and may be any, including NULL.

   @return

   value  | meaning
   -------|--------
   0      | Success
   -1     | the DB was not initialized
   -2     | parameter error (string too long)
   -3     | share table full
   -6     | share already exists

   @note
   \* This function should be called from the context of NQ Server thread.
   \* It is recommended to sue it on NQ Server startup only.                                                       */
NQ_INT csAddShare(const NQ_WCHAR * name, const NQ_WCHAR * path, NQ_BOOL printQueue, const NQ_WCHAR * comment, const NQ_CHAR * reserved);

/**
   This function removes an existing share from NQ Server.
   @param name  Name for the share to remove. This name must not
                contain the '$' character.
   @return

   value  | meaning
   -------|--------
   0      | Success
   -1     | NQ Server was not initialized
   -2     | parameter error (share not found)

   @note
   \* This function should be called from the context of NQ Server thread.
   \* It is recommended to sue it on NQ Server startup only.
   \* This is an ASCII version of csRemoveShare(). */
NQ_INT csRemoveShareA(const NQ_CHAR * name);

/**
   This function removes an existing share from NQ Server.
   @param name         Name for the share to remove. This name must not contain the '$' character.
   @returns

   value  | meaning
   -------|--------
   0      | Success
   -1     | NQ Server was not initialized
   -2     | parameter error (share not found)
   @note
   \* This function should be called from the context of NQ Server thread.
   \* It is recommended to sue it on NQ Server startup only.                           */
NQ_INT csRemoveShare(const NQ_WCHAR* name);

#if defined(UD_NQ_INCLUDESMB3) || defined(DOXYGEN)

/**
   This function enables or disables encryption for a particular
   share. An encrypted share may be only accessed over SMB3.0.
   
   This function is applicable to any share regardless of way it
   was created. When called with NULL share name parameter
   global encryption is applied.
   @param name         Share name or NULL.
   @param isEncrypted  TRUE to enable encryption, FALSE to
                       disable encryption. NQ does not consider
                       the previous encryption status.
   @return

   value  | meaning
   -------|--------
   0      | Success
   -1     | NQ Server was not initialized
   -2     | parameter error (share not found)
   @note
     \* To enable or disable global encryption use NULL as share
       name parameter
     \* This function should be called from the context of NQ
       Server thread. It is recommended to issue it on NQ Server
       startup only.
     \* This function is available when NQ was compiled with SMB3
       support as designated by the UD_NQ_INCLUDESMB3 parameter.
     \* This is an ASCII version of csSetShareEncryption() */
NQ_INT csSetShareEncryptionA(const NQ_CHAR * name, NQ_BOOL isEncrypted);

/**
   This function enables or disables encryption for a particular
   share. An encrypted share may be only accessed over SMB3.0.

   This function is applicable to any share regardless of way it
   was created. When called with NULL share name parameter global encryption
   is applied.
   @param name          Share name or NULL.
   @param isEncrypted   TRUE to enable encryption, FALSE
                  to disable encryption. NQ does not consider the
                  previous encryption status.
   @returns

   value  | meaning
   -------|--------
   0      | Success
   -1     | NQ Server was not initialized
   -2     | parameter error (share not found)
   @note
     \* To enable or disable global encryption use NULL as share name
       parameter
     \* This function should be called from the context of NQ
       Server thread. It is recommended to issue it on NQ Server
       startup only.
     \* This function is available when NQ was compiled with SMB3
       support as designated by the UD_NQ_INCLUDESMB3 parameter.          */
NQ_INT csSetShareEncryption(const NQ_WCHAR * name, NQ_BOOL isEncrypted);

#endif /* UD_NQ_INCLUDESMB3 */

/** @} end of group cs_share Shares Management */

/**@defgroup cs_file Open Files Enumeration
 * @ingroup csapi
 * @{
 */

/**
   This structure holds open file or directory info.
   NQ holds one file entry per successful create request. This file info gives
   general info per entry.
   This structure is used when calling csEnumerateOpenFilesA()     */
typedef struct {
#ifdef CM_NQ_STORAGE
    NQ_CHAR   fileNamePath[CM_BUFFERLENGTH(NQ_CHAR, SY_MAXFILENAMELEN)];    /**< File name and path buffer */
#else
    NQ_CHAR   fileNamePath[CM_BUFFERLENGTH(NQ_CHAR, UD_FS_FILENAMELEN)];    /**< File name and path buffer */
#endif
    NQ_CHAR   userName[CM_BUFFERLENGTH(NQ_CHAR, CM_USERNAMELENGTH)];        /**< User name buffer */
    NQ_CHAR   IP [CM_BUFFERLENGTH(NQ_CHAR, CM_IPADDR_MAXLEN)];              /**< IP buffer */
    NQ_BOOL   isDirectory;                                                  /**< Is this a directory or a file */
}FileDataA_t;

/**
   This structure holds open file or directory info.
   NQ holds one file entry per successful create request. This file info gives
   general info per entry.
   This structure is used when calling csEnumerateOpenFiles()     */
typedef struct {
#ifdef CM_NQ_STORAGE
    NQ_WCHAR  fileNamePath[CM_BUFFERLENGTH(NQ_WCHAR, SY_MAXFILENAMELEN)];    /**< File name buffer */
#else
    NQ_WCHAR  fileNamePath[CM_BUFFERLENGTH(NQ_WCHAR, UD_FS_FILENAMELEN)];    /**< File name buffer */
#endif
    NQ_WCHAR  userName[CM_BUFFERLENGTH(NQ_WCHAR, CM_USERNAMELENGTH)];       /**< File name buffer */
    NQ_WCHAR  IP [CM_BUFFERLENGTH(NQ_WCHAR, CM_IPADDR_MAXLEN)];             /**< IP buffer */
    NQ_BOOL   isDirectory;                                                  /**< Is this a directory or a file */
} CSFileData;

/**
   This function is used to retrieve all open files and
   directories on NQ Server. Call this function inside a loop
   and continue till False result is returned. On first call
   index should be 0, advance index by one per call to this
   function.
   
   File data will be filled in function parameter fileData,
   unless returned value is false.
   
   If a file is opened or closed during the loop calling this
   function, the returned values might not exactly reflect the
   open files situation.
   @param index     Index is indicating which open file data to
                    return. Upon 2nd call the 2nd open file in
                    the internal DB will be returned. On call x,
                    the x'th open file will be returned. For NQ
                    Storage zero index is signaling function to
                    start iteration. Next calls must have any
                    index > 0.
   @param fileData  Pointer to a structure, where NQ places the
                    file data for this index. See FileDataA_t structure.
   @return This function returns TRUE if the requested file
   index exists. False - if the requested file index doesn't
   exist.
   
  @note
   \* Be sure to start any loop call to this function with index 0.
   \* This function is non reentrant
   \* This is an ASCII version of csEnumerateOpenFiles() */
NQ_BOOL csEnumerateOpenFilesA(NQ_UINT index, FileDataA_t *fileData);

/**
   This function is used to retrieve all open files and directories on NQ Server.
   Call this function inside a loop and continue till False result is returned.
   On first call index should be 0, advance index by one per call to this function.

   File data will be filled in function parameter fileData, unless
   returned value is false.

   If  a file is opened or closed during the loop calling this function,
   the returned values might not exactly reflect the open files
   situation.

   @param index            Index is indicating which open file data to return.
                           Upon 2nd call the 2nd open file in the internal DB will be returned.
                           On call x, the x'th open file will be returned.
                           For NQ Storage zero index is signaling function to start iteration.
                           Next calls must have any index > 0.
   @param fileData         Pointer to a structure, where NQ places the
                           file data for this index. See CSFileData structure.

   @returns
   This function returns TRUE if the requested file index exists.
   False - if the requested file index doesn't exist.

   @see
   CSFileData structure.

   @note
   \* Be sure to start any loop call to this function with index 0.
   \* This function is non reentrant     */
NQ_BOOL csEnumerateOpenFiles(NQ_UINT index, CSFileData *fileData);

/** @} end of group cs_file Open Files Enumeration */

/**@defgroup cs_clean Cleanup
 * @ingroup csapi
 * @{
 */

/**
   This function closes all connections initiated by a user with
   the given name close user's existing connections to NQ server
   @param name          User name. 
   @param isDomainUser  TRUE for a domain or FALSE for a local
                        user.
   @return
   This function returns NQ_SUCCESS if the function succeed or NQ_FAIL otherwise.
   @note
   This is an ASCII version of csCleanUserServerConnections() */
NQ_STATUS csCleanUserServerConnectionsA(const NQ_CHAR *name, NQ_BOOL isDomainUser);

/**
   This function closes all connections initiated by a user
   with the given name close user's existing connections to NQ
   server

   @param name           User name.
   @param isDomainUser   TRUE for a domain or FALSE for a local user.
   @return
   This function returns NQ_SUCCESS if the function succeed or NQ_FAIL otherwise. */
NQ_STATUS csCleanUserServerConnections(const NQ_WCHAR *name, NQ_BOOL isDomainUser);

/** @} end of group cs_clean Cleanup */

/**@defgroup cs_anonymous Anonymous
 * @ingroup csapi
 * @{
 */

/**
   This function sets Anonymous Access to allowed/restricted

   @param  on           TRUE or FALSE.  */
void csSetAnonymousAllowed(NQ_BOOL on);

/** @} end of group cs_anonymous Anonymous */

/**@defgroup cs_event Event Log
 * @ingroup csapi
 * @{
 */

#if defined(UD_NQ_INCLUDEEVENTLOG) || defined(DOXYGEN)

/**
   This structure designates one share connection entry. A share
   connection entry is created by NQ Server upon a successful TreeConnect
   request.                                                      */
typedef struct {
    NQ_WCHAR userName[256];                 /**< User name buffer. This buffer should be 256 characters long at least. */
    NQ_WCHAR shareName[UD_FS_MAXSHARELEN];  /**< Share name buffer. This buffer should be 256 characters long at least. */
    NQ_IPADDRESS ip;                        /**< Next side (client) IP address. */
    NQ_BOOL ipc;                            /**< IPC flag. This flag is TRUE for the IPC$ share and FALSE for a file share. */
    NQ_BOOL printQueue;                     /**< Printer flag. When TRUE, this entry designates a printer
                                               share. For a file share this flag is FALSE.              */
} EVCSShareConnectionEntry;

/**
   This structure designates one file entry. A file
   entry is created by NQ Server upon a successful file open.    */
typedef struct {
    NQ_WCHAR userName[256];                 /**< User name buffer. */
#ifndef CM_NQ_STORAGE
    NQ_WCHAR fileName[UD_FS_FILENAMELEN];   /**< File name buffer. This buffer will be filled with full file path. should be 300 characters long at least.  */
#else
    NQ_WCHAR fileName[SY_MAXFILENAMELEN];   /**< File name buffer. This buffer will be filled with full file path. should be 300 characters long at least.  */
#endif
    NQ_WCHAR shareName[UD_FS_MAXSHARELEN];  /**< Share name buffer. his buffer should be 256 characters long at least. */
    NQ_IPADDRESS ip;                        /**< Next side (client) IP address. */
    NQ_UINT16 access;                       /**< Access bits as indicated by the open operation. */
} EVCSOpenFileEntry;

/**
   This function reads share connection entries

   @param maxEntries   Buffer capacity.
   @param buffer       Buffer pointer.
   @returns
   Number of entries read.
   @note
     \* This function is only available when NQ was compiled with
       Event Log support as indicated by the UD_NQ_INCLUDEEVENTLOG
       parameter. */
NQ_COUNT evcsEnumerateConnectedShares(NQ_COUNT maxEntries, EVCSShareConnectionEntry* buffer);

/**
   This function reads entries
   Parameters
   @param maxEntries   Buffer capacity.
   @param buffer       Buffer pointer.
   @returns
   Number of entries read.
   @note
     \* This function is only available when NQ was compiled with
       Event Log support as indicated by the UD_NQ_INCLUDEEVENTLOG
       parameter. */
NQ_COUNT evcsEnumerateOpenFiles(NQ_COUNT maxEntries, EVCSOpenFileEntry* buffer);

/** @} end of group cs_event Event Log */

#endif /* UD_NQ_INCLUDEEVENTLOG */
#endif /* UD_NQ_INCLUDECIFSSERVER */
#endif  /* _CSAPI_H_ */
