/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Entire CM - Common library functionality
 *--------------------------------------------------------------------
 * MODULE        : NQ
 * DEPENDENCIES  :
 ********************************************************************/

#ifndef _CMAPI_H_
#define _CMAPI_H_

#include "syapi.h"          /* system-dependent */
#include <cmcommon.h>       /* basic types */
#include <cmparams.h>       /* parameters */
#include <cmutils.h>        /* more common functionality for all modules */
#include <cmlist.h>         /* linked list */
#include <cmmemory.h>       /* dynamic memory */
#include <cmselfip.h>       /* IP configuration */
#include <cmresolver.h>     /* name to IP resolver */
#include <cmthread.h>       /* thread management */
#ifdef UD_NQ_INCLUDECODEPAGE
#include <cmcp.h>           /* Code pages */
#endif /* UD_NQ_INCLUDECODEPAGE */
#include <cmunicod.h>       /* ANSI to UNICODE and v/v */
#include <cmstrparse.h>     /* string parser */
#include <cmstring.h>       /* mapping of string manipulation */
#include <cmnbapi.h>        /* NetBIOS */
#include <cmfsapi.h>        /* CIFS */
#include <cmrpcdef.h>       /* DCERPC */
#include <cmsdescr.h>       /* Security Descriptors */
#include <cmvalida.h>       /* Validation of cross-dependencies */
#include <cmtrace.h>        /* traces */
#include <cmbufman.h>
#ifdef UD_NQ_INCLUDESMBCAPTURE
#include <cmcapture.h>
#endif /* UD_NQ_INCLUDESMBCAPTURE */
#include <udconfig.h>


/** @defgroup cmapi Common */

/**@defgroup cm_init_shut Initialization and Shutdown
 * @ingroup cmapi
 * @{
 */

/**
   This function starts the Common Module
   @param component  One of the following:  #NQ_CLIENT,
                      #NQ_NDDAEMON, #NQ_SERVER
   @return This function return NQ_SUCCESS if module was
   started successfully or NQ_FAIL otherwise.
   @sa cmExit()                                                            */
NQ_STATUS cmInit(NQ_UINT32 component);

/**
   This function performs the shutdown of the Common Module
   @param component  One of the following: #NQ_CLIENT,
                      #NQ_NDDAEMON, #NQ_SERVER
   @sa cmInit()                                                            */
void cmExit(NQ_UINT32 component);

#define NQ_CLIENT                   0x1 /**< Client component */
#define NQ_SERVER                   0x2 /**< Server component */
#define NQ_NDDAEMON                 0x4 /**< NetBIOS daemon component */

/** @} end of group cm_init_shut Initialization and Shutdown */

/** @defgroup cm_res Resolver
 * @ingroup cmapi
 * @{*/

#define NQ_RESOLVER_IPV4            4   /**< The resolved address is IPV4. */
#define NQ_RESOLVER_IPV6            6   /**< The resolved address is IPV6. */
#define NQ_RESOLVER_NONE            0   /**< No address resolved. This value designates a resolution failure.                                                */

#define NQ_RESOLVER_DNS             1   /**< Mechanism type is DNS. */
#define NQ_RESOLVER_NETBIOS         2   /**< Mechanism type is NetBIOS. */
#define NQ_RESOLVER_EXTERNAL_METHOD 5   /**< Mechanism type not known, external method set by the user.*/
#define NQ_RESOLVER_DNS_DC          8   /**< Mechanism type is DNS DC. */
#define NQ_RESOLVER_NETBIOS_DC      10  /**< Mechanism type is NetBIOS DC. */
#define NQ_RESOLVER_WSD             12  /**< Mechanism type is WS-Discovery. */

/** @} end of group cm_res Resolver */

/**@defgroup cm_datastruct Data Structures
 * @ingroup cmapi
 * @{
 */

/**
 * This struct uses 2 data types:
 * 1. CMList    memoryPool  - Each time a new data has to be stored, we check if there are available items
 *                            in this memoryPool, if yes, we use the sorted array from section 2 to point to that data and it is removed from this memoryPool,
 *                            if not, we add a new node to this memoryPool and repeat the scenario from above.
 * 2. void      **items     - This is a cyclic sorted array of pointers. Each time a new data has to be stored, we query the memoryPool from above
 *                            for a new memory block and point to the returned block, when done we return the memory block back to the memoryPool.
 */
typedef struct
{
    void        **items;            /* a cyclic sorted array of pointers to the memoryPool nodes */
    CMList      memoryPool;         /* A list of available unused memory blocks */
    NQ_UINT     currentPosition;    /* holds the current position of the next available memory block in the items array */
    NQ_UINT     numberOfItems;      /* number of current items in the items array */
    NQ_UINT32   divider;            /* when a new memory block has to be stored in the items array we need to use a formula to keep it sorted.
                                       The formula is dependent on a divider, we want to use the same divider for all the items calculations */
    void        *context;           /* a context pointer */
} CMLinkedListArray;

/** @} end of group cm_datastruct Data Structures */

/**@defgroup cm_product Product Information
 * @ingroup cmapi
 * @{
 */

#define CM_PRODUCT_STRING_LEN 20

/* This struct represents the current product information */
typedef struct
{
    NQ_CHAR     productName[CM_PRODUCT_STRING_LEN];    /* The product name */
    NQ_CHAR     productType[CM_PRODUCT_STRING_LEN];    /* Client Standalone / Client Corporate / Server Standalone / Server Corporate */
    NQ_CHAR     version[CM_PRODUCT_STRING_LEN];        /* The current product version */
    NQ_UINT32   checksum;
} CMProductInfo;

/** @} end of group cm_product Product Information */

/** @addtogroup cm_code
 * @{ */

/** This structure defines code page parameters. */
typedef struct
{
    NQ_INT  id;                                                                                 /**<Unique id value for this codepage. Code page is identified
                                                                                                   by this value. It is recommended to use respective Microsoft
                                                                                                   code page numbers (see notes).
                                                                                                   @note
                                                                                                   For Microsoft code page numbers see http://msdn.microsoft.com/en-us/library/dd317756(VS.85).aspx */
    NQ_INT  (*toAnsi)   ( NQ_CHAR* outStr,  NQ_UINT outLength, const NQ_WCHAR* inWStr, NQ_UINT inLength);  /**<Codepage-specific function for Unicode to ANSI conversion */
    NQ_INT  (*toUnicode)( NQ_WCHAR* outWStr, NQ_UINT outLength, const NQ_CHAR* inStr, NQ_UINT inLength);   /**<Codepage-specific function for ANSI to Unicode conversion */
    NQ_INT  (*toUpper)  ( NQ_CHAR* dst,  const NQ_CHAR* src);                                        /**<Codepage-specific function for capitalizing conversion */
    /**<Codepage-specific function for converting ANSI to the encoding used in the local FS                         */
    void    (*ansiToFs) ( NQ_CHAR* str, NQ_INT size, const NQ_BYTE* firstIllegalChar, const NQ_BYTE* anyIllegalChar, NQ_INT numOfFirstIllChars, NQ_INT numOfAnyIllChars);
    /**<Codepage-specific function for converting the encoding used in the local FS into ANSI                                   */
    void    (*fsToAnsi) ( NQ_CHAR* str, NQ_INT size, const NQ_BYTE* firstIllegalChar, const NQ_BYTE* anyIllegalChar, NQ_INT numOfFirstIllChars, NQ_INT numOfAnyIllChars);

    const NQ_WCHAR* a2uTab;                                                                     /**<Fast ANSI to Unicode conversion table. Using this table
                                                                                                   avoids calling the functions defined in this structure. It is
                                                                                                   only used for single-byte encodings like
                                                                                                   Hebrew/Russian/Turkish/Greek/etc. This pointer may be NULL. It
                                                                                                   should be NULL for multiple-byte encodings.                    */  
}
CMCodepage;

/** @} end of group cm_code Codepage */

/** @addtogroup cm_res
 * @{ */

/** A descriptor of one resolver method. Use this structure to
   register a resolution method using the cmResolverRegisterExternalMethod() */
typedef struct
{

    NQ_INT activationPriority;  /**< This parameter may have one of the following values:
                                  1. Activate registered method before all other methods.
                                  2. Activate registered method after the existing NQ unicast
                                     methods and before existing NQ multicast methods. In this
                                     case registered method will be activated only if existing NQ
                                     unicast methods failed to resolve name or IP
                                  3. Activate registered method only if all other methods
                                     failed.                                                      */

    NQ_UINT32 timeout; /**< Timeout in seconds to use with this method */

    NQ_IPADDRESS *serverIP;  /**< IP of the server to perform resolution with. This value is
                               only used by Unicast methods.                              */

    NQ_STATUS (* requestByName)(SYSocketHandle socket, const NQ_WCHAR * name, void * context,
                                                             const NQ_IPADDRESS * serverIp, NQ_COUNT * numOfSentRequests);  /**< Routine for composing and sending a name resolution request
                                                                                                                               @param socket:              Socket handle to use for sending
                                                                                                                               @param name:                Name to resolve
                                                                                                                               @param context:             Pointer to a method-specific context. This value may be NULL on the first call.
                                                                                                                               @param serverIp:            Pointer to the IP of the server to query or NULL for multicast
                                                                                                                               @param numOfSentRequests:   Return the number of requests that were sent.
                                                                                                                               @return
                                                                                                                               NQ_SUCCESS   request sent
                                                                                                                               NQ_ERR_<*>   error
                                                                                                                                                         */

    NQ_STATUS (* responseByName)(SYSocketHandle socket, NQ_IPADDRESS ** pAddressArray, NQ_INT * numIps, void ** pContext);  /**< Routine for receiving and parsing a name resolution response
                                                                                                                               @param socket    Socket handle to use for sending
                                                                                                                               @param pAddressArray  Address of the pointer which this call sets to an array of
                                                                                                                                                resolved IP addresses. It is caller's responsibility to release this array.
                                                                                                                                                On error, this pointer remains untouched.
                                                                                                                               @param numIps    Pointer to the number of resolved IPs.
                                                                                                                               @param pContext  Double pointer to a method-specific context. Method may dispose
                                                                                                                                            context and create a new one.
                                                                                                                               @return
                                                                                                                               NQ_SUCCESS           name successfully resolved
                                                                                                                               NQ_ERR_MOREDATA      more exchange expected
                                                                                                                               NQ_ERR_NOACCESS      more comprehensive method with the same code should be used
                                                                                                                               NQ_ERR_<*>           error
                                                                                                                             */

    NQ_STATUS (* requestByIp)(SYSocketHandle socket, const NQ_IPADDRESS * ip, void * context,
                                                             const NQ_IPADDRESS * serverIp, NQ_COUNT * numOfSentRequests);  /**< Routine for composing and sending an IP resolution request
                                                                                                                               @param socket                Socket handle to use for sending
                                                                                                                               @param ip                    Pointer to the IP address to resolve.
                                                                                                                               @param context               Pointer to a method-specific context. This value may be NULL on the first call.
                                                                                                                               @param serverIp              Pointer to  the IP of the server to query or NULL for multicast
                                                                                                                               @param numOfSentRequests     Return the number of requests that were sent.
                                                                                                                               @return
                                                                                                                               NQ_SUCCESS   request sent
                                                                                                                               NQ_ERR_<*>   error
                                                                                                                             */

    NQ_STATUS (* responseByIp)(SYSocketHandle socket, const NQ_WCHAR ** pName, void ** pContext);   /**< Routine for receiving and parsing a name resolution response
                                                                                                    @param socket    Socket handle to use for sending
                                                                                                    @param pName     Double pointer to the resolved name. On success, this variable will
                                                                                                                point to a newly allocated name. Its is caller's responsibility to release it later.
                                                                                                    @param pContext  Double pointer to a method-specific context. Method may dispose
                                                                                                                context and create a new one.

                                                                                                   @return
                                                                                                   NQ_SUCCESS           name successfully resolved
                                                                                                   NQ_ERR_MOREDATA      more exchange expected
                                                                                                   NQ_ERR_NOACCESS      more comprehensive method with the same code should be used
                                                                                                   NQ_ERR_<*>           error
                                                                                                 */
}
CMResolverRegisteredMethodDescription;

/** @} end of group cm_res Resolver */

/* function prototypes */

/**@defgroup cm_code Codepage
 * @ingroup cmapi
 * @{
 */

/**
   This function installs code page for another language.
   
   Code page is identified by the '<i>id</i>' value of its
   descriptor. Duplicate installation overwrite the previous
   one.
   
   Once successfully installed, the code page becomes available
   by its '<i>id</i>' number through the <i>udGetCodePage()</i>
   function call.
   
   This function may be used for one the following purposes:
     \* Installing new code page.
     \* Replacing one of previously installed code pages.
     \* Replacing one of precompiled code pages.
   @param  codePage Pointer to codepage descriptor
   @return
   NQ_TRUE on success and NQ_FALSE on error. Application may
   analyze error reason by reading the last system error. It may
   be NQ_ERR_NOMEM on codepage table overflow.                                                       */
NQ_BOOL cmCodepageAdd(const CMCodepage * codePage);
 
/**
   This function removes code page.
   
   This call will either remove a code page previously installed
   by calling cmCodepageAdd()
   or it may remove a a pre-compiled page.
   @param codePage Pointer to codepage descriptor as CMCodepage structure.
               NQ uses only the '<i>id'</i> field and ignores
               others.
   @return
   NQ_TRUE on success and NQ_FALSE on error. Application may
   analyze error reason by reading the last system error. It may
   be NQ_ERR_NOTFOUND when this codepage does not exist.                                  */
NQ_BOOL cmCodepageRemove(const CMCodepage * codePage);

/** Conversion of Wide Characters string into Multi Byte.
    @param strMultiByte Pointer to result multi byte string
    @param strWideChar  Pointer to wide char string to convert
                              */
void cmWideCharToMultiByte(NQ_CHAR *strMultiByte, const NQ_WCHAR* strWideChar);

/** Conversion of Multi Byte string into Wide Characters.
    @param strWideChar  Pointer to result wide char string
    @param strMultiByte Pointer to multi byte string to convert
                                  */
void cmMultiByteToWideChar(NQ_WCHAR* strWideChar, const NQ_CHAR *strMultiByte);

/** @} end of group cm_code Codepage */

/**@defgroup cm_crypt Cryptographic
 * @ingroup cmapi
 * @{
 */

/**
   This function prototype designates an abstract cryptographic
   hash algorithm. It is used to replace internal NQ encryption
   with an external one.
   @param dataIn  Pointer to the data to encrypt.
   @param dataOut Pointer to the buffer for encrypted.
   @param length  Length of the incoming data and also the length of
              the encrypted data.
                                                        */
typedef void (* CMAbstractHasher)(const NQ_BYTE * dataIn, NQ_BYTE * dataOut, NQ_COUNT length);

/**
   This function prototype designates an abstract cryptographic
   algorithm using a cryptographic key. It is used to replace
   internal NQ encryption with an external one.
   @param key           Pointer to the encryption key descriptor.
                    Some algorithms do not use this value.
   @param key1          Pointer to the auxiliary key descriptor. This
                    value may be NULL. Also the data pointer may
                    be null as well as the blob length may be
                    zero. In either of those cases the auxiliary
                    key is ignored. Some algorithms may ignore
                    this value anyway.
   @param dataFragments A pointer to the array of data fragment
                    descriptors. For encryption algorithms which
                    allow encrypting multiple fragments, this
                    array may contain more than one element. For
                    other algorithms, only the first element is
                    used. An element may be NULL. Also a data
                    pointer in an element may be NULL as well as
                    blob length may be zero. In either of those
                    cases the respective fragment is ignored.
   @param numFragments  Number of data fragments (see above).
   @param buffer        Place holder for hash result.
   @param bufferSize    The expected length of the encrypted data
                                                         */
typedef void (* CMAbstractCipher)(const CMBlob * key, const CMBlob * key1, const CMIOBlob dataFragments[], NQ_COUNT numFragments, NQ_BYTE * buffer, NQ_COUNT bufferSize);

/**
   This function prototype designates an abstract cryptographic
   algorithm using a cryptographic key. It is used to replace
   internal NQ encryption with an external one.

   @param key           Pointer to the encryption key descriptor.
                    Some algorithms do not use this value.
   @param key1          Pointer to the auxiliary key descriptor. This
                    value may be NULL. Also the data pointer may
                    be null as well as the blob length may be
                    zero. In either of those cases the auxiliary
                    key is ignored. Some algorithms may ignore
                    this value anyway.
   @param dataFragments A pointer to the array of data fragment
                    descriptors. For encryption algorithms which
                    allow encrypting multiple fragments, this
                    array may contain more than one element. For
                    other algorithms, only the first element is
                    used. An element may be NULL. Also a data
                    pointer in an element may be NULL as well as
                    blob length may be zero. In either of those
                    cases the respective fragment is ignored.
   @param numFragments  Number of data fragments (see above).
   @param buffer        Place holder for hash result.
   @param bufferSize    The expected length of the encrypted data
                                                         */
typedef void (* CMAbstractIOCipher)(const CMBlob * key, const CMBlob * key1, const CMIOBlob dataFragments[], NQ_COUNT numFragments, NQ_BYTE * buffer, NQ_COUNT bufferSize);

/**
   This function prototype designates an abstract cryptographic
   algorithm for both encryption and producing authentication
   data (CCM) . It is used to replace internal NQ encryption
   with an external one.
   
   This algorithm assumes that there are two blocks of data:
     1. A prefix that is preserved as is
     2. The messages itself that is to be encrypted
   The authentication is a production of both blocks.
   @param key     Pointer to the encryption key descriptor. Some
              algorithms do not use this value.
   @param key1    Pointer to the auxiliary key descriptor. This value
              may be NULL. Also the data pointer may be null as
              well as the blob length may be zero. In either of
              those cases the auxiliary key is ignored. Some
              algorithms may ignore this value anyway.
   @param prefix  A pointer to the message prefix descriptor. This
              data participate in authentication but remains as
              is, without encryption.
   @param message Pointer to the message descriptor. This message
              will be encrypted in\-place.
   @param auth    Pointer to the authentication data. It will be
              filled as the result of the algorithm.
                                                     */

typedef void (* CMAbstractHasher512)(const CMBlob * key, const CMBlob * key1, const CMIOBlob dataFragments[], NQ_COUNT numFragments, NQ_IOBufPos buffer, NQ_COUNT bufferSize, NQ_BYTE *ctxBuff);

/**
   This function prototype designates an abstract cryptographic
   algorithm for producing authentication data. It is used to 
   replace internal NQ encryption with an external one.
   The authentication (hash result) is a product of all data blocks.
 
   @param key           Pointer to the encryption key descriptor.
                    Some algorithms do not use this value.
   @param key1          Pointer to the auxiliary key descriptor. This
                    value may be NULL. Also the data pointer may
                    be null as well as the blob length may be
                    zero. In either of those cases the auxiliary
                    key is ignored. Some algorithms may ignore
                    this value anyway.
   @param dataFragments A pointer to the array of data fragment
                    descriptors. For encryption algorithms which
                    allow encrypting multiple fragments, this
                    array may contain more than one element. For
                    other algorithms, only the first element is
                    used. An element may be NULL. Also a data
                    pointer in an element may be NULL as well as
                    blob length may be zero. In either of those
                    cases the respective fragment is ignored.
   @param numFragments  Number of data fragments (see above).
   @param buffer        Place holder for hash result.
   @param bufferSize    The expected length of the encrypted data
   @param ctxBuff Buffer for context data. if this buffer is null,
              context buffer will be allocated in function.
                                               */
typedef void (* CMAbstractCcmEncryption)(const CMBlob * key, const CMBlob * key1, const CMIOBlob * prefix, CMIOBlob * message, NQ_BYTE * auth);

/**
   This function prototype designates an abstract cryptographic
   algorithm for both decryption and authentication (CCM) . It
   is used to replace internal NQ encryption with an external
   one.
   
   This algorithm assumes that there are two blocks of data:
     1. A prefix that is preserved as is
     2. The messages itself that is to be encrypted
   The authentication is a production of both blocks.

   @param key     Pointer to the encryption key descriptor. Some
              algorithms do not use this value.
   @param key1    Pointer to the auxiliary key descriptor. This value
              may be NULL. Also the data pointer may be null as
              well as the blob length may be zero. In either of
              those cases the auxiliary key is ignored. Some
              algorithms may ignore this value anyway.
   @param prefix  A pointer to the message prefix descriptor. This
              data participate in authentication but remains as
              is, without encryption.
   @param message Pointer to the message descriptor. This message
              will be encrypted in-place.
   @param auth    Pointer to the authentication data. This value is
              be used to authenticate.
   @return
   TRUE if authenticated, FALSE if not.                           */
typedef NQ_BOOL (* CMAbstractCcmDecryption)(const CMBlob * key, const CMBlob * key1, const CMIOBlob * prefix, CMIOBlob * message, const NQ_BYTE * auth);


/**
   This function prototype designates an abstract cryptographic
   algorithm for both encryption and producing authentication
   data (GCM) . It is used to replace internal NQ encryption
   with an external one.

   This algorithm assumes that there are two blocks of data:
     1. A prefix that is preserved as is
     2. The messages itself that is to be encrypted
   The authentication is a production of both blocks.
   @param key     Pointer to the encryption key descriptor. Some
              algorithms do not use this value.
   @param key1    Pointer to the auxiliary key descriptor. This value
              may be NULL. Also the data pointer may be null as
              well as the blob length may be zero. In either of
              those cases the auxiliary key is ignored. Some
              algorithms may ignore this value anyway.
   @param prefix  A pointer to the message prefix descriptor. This
              data participate in authentication but remains as
              is, without encryption.
   @param message Pointer to the message descriptor. This message
              will be encrypted in-place.
   @param auth    Pointer to the authentication data. It will be
              filled with the result of the algorithm.
   @param keyBuffer: Optional buffer for usage in key calculations.
              If NULL a buffer will be allocated per function call.
              size - AES_PRIV_SIZE
   @param ncMsgBuffer Optional buffer for the message encryption which
              is not done in place. If NULL a buffer will be
              allocated per function call.
              size: message size.
                                                      */
typedef void (* CMAbstractGcmEncryption)(const CMBlob *key, const CMBlob *key1, const CMIOBlob *prefix, CMIOBlob *message, NQ_BYTE *auth, NQ_BYTE *keyBuffer, NQ_BYTE *encMsgBuffer);

/**
   This function prototype designates an abstract cryptographic
   algorithm for both decryption and authentication (GCM). It
   is used to replace internal NQ encryption with an external
   one.

   This algorithm assumes that there are two blocks of data:
     1. A prefix that is preserved as is
     2. The messages itself that is to be encrypted
   The authentication is a production of both blocks.
   @param key     Pointer to the encryption key descriptor. Some
              algorithms do not use this value.
   @param key1    Pointer to the auxiliary key descriptor. This value
              may be NULL. Also the data pointer may be null as
              well as the blob length may be zero. In either of
              those cases the auxiliary key is ignored. Some
              algorithms may ignore this value anyway.
   @param prefix  A pointer to the message prefix descriptor. This
              data participate in authentication but remains as
              is, without encryption.
   @param message Pointer to the message descriptor. This message
              will be encrypted in-place.
   @param auth    Pointer to the authentication data. This value is
              be used to authenticate.
   @param keyBuffer: Optional buffer for usage in key calculations.
              If NULL a buffer will be allocated per function call.
              size - AES_PRIV_SIZE
   @param msgBuffer: Optional buffer for the message decryption which
              is not done in place. If NULL a buffer will be
              allocated per function call.
              Size - message size.
   @return
   TRUE if authenticated, FALSE if not.                           */
typedef NQ_BOOL (* CMAbstractGcmDecryption)(const CMBlob * key, const CMBlob *key1, const CMIOBlob *prefix, CMIOBlob *message, const NQ_BYTE *auth, NQ_BYTE *keyBuffer, NQ_BYTE *msgBuffer);

/**
   This structure designates a list of cryptographic algorithms
   currently used by NQ. It is used to replace existing
   cryptographic algorithms.
   
   If a particular cryptographic algorithm is NULL, it is not
   replaced.                                                    */
typedef struct
{
    CMAbstractHasher md4;           /**<MD4 hasher */
    CMAbstractCipher md5;           /**<MD5 hasher */
    CMAbstractCipher hmacmd5;       /**<HMACMD5 crypter */
    CMAbstractIOCipher sha256;      /**<SHA-256 crypter */
    CMAbstractIOCipher aes128cmac;  /**<AES-CMAC crypter */
    CMAbstractHasher512 sha512;     /**<SHA-512 crypter */
    CMAbstractCcmEncryption aes128ccmEncryption;    /**<AES-CCM encryption algorithm */
    CMAbstractCcmDecryption aes128ccmDecryption;    /**<AES-CCM decryption algorithm */
    CMAbstractGcmEncryption aes128gcmEncryption;    /**<AES-GCM encryption algorithm */
    CMAbstractGcmDecryption aes128gcmDecryption;    /**<AES-GCM decryption algorithm */
} 
CMCrypterList;

/**
   This function replaces internal NQ cryptographic algorithms
   with external ones.
   @param crypters A pointer to the list of internal ciphers. Only
               non-NULL values are applied.
                                                 */
void cmSetExternalCrypters(const CMCrypterList * crypters);

/**
   This function reverts the list of cryptographic algorithms to internal algorithms only.
                                                       */
void cmResetExternalCrypters(void);

/** @} end of group cm_crypt Cryptographic */

/** @addtogroup cm_res
 * @{
 */

/**
   This function prototype designates a callback for resolving
   host IP address by its name. NQ uses this callback for
   external resolution.
   @param name   The name of the host to resolve.
   @param index  Index of the resolved IP address. External
                 resolver may resolve multiple IP addresses for
                 the same name. When external resolution is
                 required, NQ subsequently calls this function
                 with <i>'index'</i> values incrementing,
                 starting from zero. External resolver should
                 return the appropriate IP address or it should
                 should fail the call when an <i>'index'</i>
                 result is not available.
   @return One of the following values:
     \* #NQ_RESOLVER_IPV4. This means that <i>'index'</i>
       designates an IPv4 address.
     \* #NQ_RESOLVER_IPV6. This means that <i>'index'</i>
       designates an IPv6 address.
     \* #NQ_RESOLVER_NONE. This means that either
       resolution failed of the <i>'index'</i> value is out of
       range.
   @note
   This is an ASCII version of CMResolverNameToIp()        */
typedef NQ_INT (*CMResolverNameToIpA)(const NQ_CHAR * name, void * ip, NQ_COUNT index);

/**
   This function prototype designates a callback for resolving
   host IP address by its name. NQ uses this callback for
   external resolution.

   @param name  The name of the host to resolve.
   @param index Index of the resolved IP address. External resolver
            may resolve multiple IP addresses for the same name.
            When external resolution is required, NQ subsequently
            calls this function with <i>'index'</i> values
            incrementing, starting from zero. External resolver
            should return the appropriate IP address or it should
            should fail the call when an <i>'index'</i> result is
            not available.
   @return
   One of the following values:
     \* #NQ_RESOLVER_IPV4 This means that <i>'index'</i>
       designates an IPv4 address.
     \* #NQ_RESOLVER_IPV6 This means that <i>'index'</i>
       designates an IPv6 address.
     \* #NQ_RESOLVER_NONE This means that either
       resolution failed of the <i>'index'</i> value is out of
       range.                                                             */
typedef NQ_INT (*CMResolverNameToIp)(const NQ_WCHAR * name, void * ip,  NQ_COUNT index);

/**
   This function prototype designates a callback for resolving
   host name by its IP address. NQ uses this callback for
   external resolution.
    @param name  Buffer for the resolved name.
    @param ip    Pointer to IP address. This pointer designates
                either NQ_IPADDRESS4, NQ_IPADDRESS4 type
                or NQ_IPADDRESS6, NQ_IPADDRESS6 type
                depending on the <i>'type' </i>argument below.
    @param ipType  IP address type. This value can be either #NQ_RESOLVER_IPV4
                or #NQ_RESOLVER_IPV6. It defines the format
                of the <i>'ip'</i> argument.
   @return TRUE on success, FALSE on failure.
   @note
   This is an ASCII version of CMResolverIpToName()                        */
typedef NQ_BOOL (*CMResolverIpToNameA)(NQ_CHAR * name, const void * ip, NQ_INT ipType);

/**
   This function prototype designates a callback for resolving
   host name by its IP address. NQ uses this callback for
   external resolution.

   @param name Buffer for the resolved name.
   @param ip   Pointer to IP address. This pointer designates either NQ_IPADDRESS4, NQ_IPADDRESS4 type
           or NQ_IPADDRESS6, NQ_IPADDRESS6 type depending
           on the <i>'type' </i>argument below.
   @param ipType IP address type. This value can be either #NQ_RESOLVER_IPV4
           or #NQ_RESOLVER_IPV6. It defines the format of
           the <i>'ip'</i> argument.
   @return
   TRUE on success, FALSE on failure.                                                                     */
typedef NQ_BOOL (*CMResolverIpToName)(NQ_WCHAR * name, const void * ip, NQ_INT ipType);

/**
   This function extends NQ's name-to-IP and IP-to-name
   resolver. NQ attempts to use its internal resolution methods
   first:
     \* NetBIOS Naming Service;
     \* DNS;
     \* LLMNR.
   If none of the above succeeds, NQ attempts an external
   resolver. By calling this function application can enable or
   disable an external resolving mechanism.
   
   NQ uses callback pointers as follows:
     \* First, NQ attempts internal methods. If at least one of
       them succeeded, the external method is not used.
     \* If a Unicode callback was specified, it is used.
     \* If a Unicode callback pointer was not specified or its
       pointer was set to NULL, NQ uses ASCII callback.
     \* If neither Unicode nor ASCII callback were specified or
       both pointers are NULL, NQ skips external resolution.
     \* The approach above applies independently to each of
       name-to-IP and IP-to-name resolution.

   @param nameToIp Pointer to a function that resolves host IP by its
               name. This function signature should conform to
               the CMResolverNameToIp prototype. This
               value can be NULL. In this case, NQ will fail
               external name-to-IP resolution.
   @param ipToName Pointer to a function that resolves host name by
               its IP. This function signature should conform to
               the CMResolverIpToName prototype. This
               value can be NULL. In this case, NQ will fail
               external IP-to-name resolution.
                                                                                */
void cmResolverSetExternalA(CMResolverNameToIpA nameToIp, CMResolverIpToNameA ipToName);

/**
   This function extends NQ's name-to-IP and IP-to-name
   resolver. NQ attempts to use its internal resolution methods
   first:
     \* NetBIOS Naming Service;
     \* DNS;
     \* LLMNR.
   If none of the above succeeds, NQ attempts an external
   resolver. By calling this function application can enable or
   disable an external resolving mechanism.

   NQ uses callback pointers as follows:
     \* First, NQ attempts internal methods. If at least one of
       them succeeded, the external method is not used.
     \* If a Unicode callback was specified, it is used.
     \* If a Unicode callback pointer was not specified or its
       pointer was set to NULL, NQ uses ASCII callback.
     \* If neither Unicode nor ASCII callback were specified or
       both pointers are NULL, NQ skips external resolution.
     \* The approach above applies independently to each of
       name-to-IP and IP-to-name resolution.

   @param nameToIp Pointer to a function that resolves host IP by its
               name. This function signature should conform to
               the CMResolverNameToIp prototype. This
               value can be NULL. In this case, NQ will fail
               external name-to-IP resolution.
   @param ipToName Pointer to a function that resolves host name by
               its IP. This function signature should conform to
               the CMResolverIpToName prototype. This
               value can be NULL. In this case, NQ will fail
               external IP-to-name resolution.
                                                                                           */
void cmResolverSetExternal(CMResolverNameToIp nameToIp, CMResolverIpToName ipToName);

/**
   This function resolves host by its IP and return its name.
   
   Host name is created in allocated memory so that it is
   caller's responsibility to free this memory.
   
   NQ uses several unicast and multicast methods for this
   resolution. First, it attempts all unicast methods
   concurrently. Depending on compilation parameters those
   methods may be:
     \* DNS queries to one or more DNS servers;
     \* NetBIOS query to one or more WINS.
   If none of the above succeeded, NQ concurrently attempts
   multicast methods as:
     \* LLMNR;
     \* NetBIOS broadcasts.
   If those methods did not succeed, NQ calls external method if
   this method has been installed in cmResolverRegisterExternalMethod()
   call.
   @param ip  Pointer to host IP address.
   @return Pointer to newly created host name or NULL on
   failure.
   @note
   It is caller's responsibility to release this name.                                                                                                 */
const NQ_WCHAR * cmResolverGetHostName(const NQ_IPADDRESS * ip);

/**
   This function resolves host by its name and return its IP
   addresses.
   
   The array of IP addresses is created in allocated memory so
   that it is caller's responsibility to free this memory.
   
   NQ uses several unicast and multicast methods for this
   resolution. First, it attempts all unicast methods
   concurrently. Depending on compilation parameters those
   methods may be:
     \* DNS queries to one or more DNS servers;
     \* NetBIOS query to one or more WINS.
   If none of the above succeeded, NQ concurrently attempts
   multicast methods as:
     \* LLMNR;
     \* NetBIOS broadcasts.
   If those methods did not succeed, NQ calls external method if
   this method has been installed in cmResolverRegisterExternalMethod()
   call.
   @param host    Pointer to host name to resolve.
   @param numIps  Pointer to variable that on exit gets the
                  number of resolved IP addresses. 
   @param dnsList DNS ips to register and prepare methods.
   @return Pointer to an array of IP addresses or NULL on error.
   The size of array is placed into the variable pointed by <i>numIps</i>.
   @note
   It is caller's responsibility to release this array. To
   release this array the caller need to call the <i>cmMemoryFree()</i>
   function.                                                                                                                                           */
const NQ_IPADDRESS * cmResolverGetHostIps(const NQ_WCHAR * dnsList, const NQ_WCHAR * host, NQ_INT * numIps);

/**
   This function resolves domain's DC and return its name.
   
   DC name is created in allocated memory so that it is caller's
   responsibility to free this memory.
   
   NQ uses several unicast and multicast methods for this
   resolution. First, it attempts all unicast methods
   concurrently. Depending on compilation parameters those
   methods may be:
     \* DNS queries to one or more DNS servers;
     \* NetBIOS query to one or more WINS.
   If none of the above succeeded, NQ concurrently attempts
   multicast methods as:
     \* NetBIOS broadcasts.
   If those methods did not succeed, NQ calls external method if
   this method has been installed in cmResolverRegisterExternalMethod()
   call.
   @param domain   Pointer to domain name.
   @param dnsList  Semicolon delimited list of DNS servers.
   @param numDCs   Pointer to variable that on exit gets the
                   number of resolved DC names.
   @return Pointer to newly created DC name or NULL on failure.
   @note
   It is caller's responsibility to release this name. To
   release the name call <i>cmMemoryFree()</i>                                                                                                         */
const NQ_WCHAR * cmResolverGetDCName(const NQ_WCHAR * domain, const NQ_WCHAR * dnsList, NQ_INT * numDCs);

/**
   Resolver uses different methods (mechanisms) to resolve host
   IP(s) by name or to resolve host name by IP. Each mechanism
   may use either unicasts or multicasts. NQ attempts all
   possible unicasts first. Then, if unicasts failed, it
   attempts multicasts. Currently, the following mechanisms are
   available:

   Method          | DNS              |  NetBIOS
   --------------- |----------------- | ----------------
   <i>Unicast</i>  |DNS server query  |WINS query
   Multicast       |LLMNR             |local broadcast

   All methods are initially enabled on NQ startup. This
   function allows to enable or disable a particular one in
   run-time.

   @param type       Resolution type. This may be one of the following:
                     #NQ_RESOLVER_DNS, #NQ_RESOLVER_NETBIOS, #NQ_RESOLVER_EXTERNAL_METHOD,
                     #NQ_RESOLVER_DNS_DC, #NQ_RESOLVER_NETBIOS_DC, #NQ_RESOLVER_WSD.
   @param unicast    When this parameter is <i>TRUE</i>, NQ enables
                     unicast method (s) of the given type. When this
                     parameter is <i>FALSE., </i>NQ disables unicast
                     method(s).
   @param multicast  When this parameter is <i>TRUE</i>, NQ enables
                     multicast method(s) of the given type. When this
                     parameter is <i>FALSE., </i>NQ disables multicast
                     method(s).
                                                                  */
void cmResolverEnableMethod(NQ_INT type, NQ_BOOL unicast, NQ_BOOL multicast);

/**
   Resolver uses DNS and NET BIOS to resolve host IP(s) by name
   or to resolve host name by IP. This function enables adding
   another resolution method. A method is defined using the  CMResolverRegisteredMethodDescription Structure.
   See structure description with different members. Not all
   functions have to be defined, but notice following
   guidelines:
     1. Must define at least one request function - request by
        name or request by IP
     2. If a request function is defined, then the corresponding
        response function must be defined as well.
     3. If defined timeout value is too long it might cause a
        delay in setup process.
     4. Resolver methods are not executed all at once.
        Registered method will be executed according to defined
        priority. See the CMResolverRegisteredMethod structure.
   @param pMethod resolution method.
   @return
   TRUE - on registration success. FALSE otherwise.      */
NQ_BOOL cmResolverRegisterExternalMethod(const CMResolverRegisteredMethodDescription * pMethod);

/**
   This function enables run time control on the order of
   resolver methods execution. In resolver point of view three
   groups of methods exist: unicast methods, multicast methods,
   external methods. External methods are all methods that were
   registered by the user. Each group of methods is executed
   separately and if no reply is received the next group is
   executed.

   @param requiredPriority  This parameter takes the following values:
                       1. Execute external methods before all
                          other methods.
                       2. Execute external methods after unicast
                          methods and before multicast methods.
                       3. Execute external methods only if all
                          other resolution methods failed.
   @return
   <i>TRUE</i> - on success. <i>FALSE</i> otherwise.               */
NQ_BOOL cmResolverUpdateExternalMethodsPriority(NQ_INT requiredPriority);

/** @} end of group cm_res Resolver */

/**@defgroup cm_dns DNS and WINS
 * @ingroup cmapi
 * @{
 */

/**
   After startup NQ uses the list of DNS servers as defined in
   the NQ configuration. By calling this function application
   replaces the initial list with a new one.

   @param servers  Pointer to the new list of DNS servers. This
                   list is a string of IP addresses, delimited by
                   a semicolon. Each address may have a form of
                   IPv4 or IPv6 address. This pointer can be
                   NULL. In this case NQ will not query DNS
                   servers.
   @note
   This is an ASCII version of cmDnsSetServers() */
void cmDnsSetServersA(const NQ_CHAR * servers);

/**
   After startup NQ uses the list of DNS servers as defined in
   the NQ configuration. By calling this function application
   replaces the initial list with a new one.

   @param servers Pointer to the new list of DNS servers. This list
              is a string of IP addresses, delimited by a
              semicolon. Each address may have a form of IPv4 or
              IPv6 address. This pointer can be NULL. In this
              case NQ will not query DNS servers.
                                                      */
void cmDnsSetServers(const NQ_WCHAR * servers);

/**After startup NQ uses the DNS servers as defined in the NQ
   configuration or by calling cmDnsSetServersA() or cmDnsSetServers().
   By calling this function with a specific DNS index application returns the specific DNS address.

   @param dnsID  DNS server index.
   @return the pointer of the ip address of the specific DNS server
                    or 0.
   @note
   The pointer should be used as NQ_IPADDRESS4 or NQ_IPADDRESS6 depends on the current system configurations.     */
NQ_IPADDRESS * cmDnsGetServer(NQ_COUNT dnsID);

/**After startup NQ uses the DNS servers as defined in the NQ
   configuration or by calling cmDnsSetServersA() or cmDnsSetServers().
   By calling this function application returns the number of current defined DNS addresses.

   @return Number of defined DNS addresses     */
NQ_COUNT cmDnsGetNumDnsServers(void);

/**
   After startup NQ uses the WINS IP as defined in the NQ
   configuration. By calling this function application replaces
   the initial WINS with a list of new ones.

   @param servers  Pointer to the list of new WINS servers. This
                   list is a string of IP addresses, delimited by
                   a semicolon. Each address should have a form
                   of IPv4 address. This pointer can be NULL, in
                   this case NQ will not query WINS.
   @note
     \* WINS addresses set over this function affect only
       name-to-IP
   and IP-to-name resolution. They do not affect name
   registration, which is always performed against the default
   WINS. We assume that in a case of multiple WINS they
   replicate name registered in just one of them.
     \* This is an ASCII version of cmNetBiosSetWins() */
void cmNetBiosSetWinsA(const NQ_CHAR * servers);

/** After startup NQ uses the WINS IP as defined in the NQ
   configuration. By calling this function application replaces
   the initial WINS with a list of new ones.
   @param servers Pointer to the list of new WINS servers. This list
              is a string of IP addresses, delimited by a
              semicolon. Each address should have a form of IPv4
              address. This pointer can be NULL, in this case NQ
              will not query WINS.
   @note
   WINS addresses set over this function affect only name-to-IP
   and IP-to-name resolution. They do not affect name registration, which is always performed against the default
   WINS. We assume that in a case of multiple WINS they replicate name registered in just one of them.      */
void cmNetBiosSetWins(const NQ_WCHAR * servers);

/**After startup NQ uses the WINS servers as defined in the NQ
   configuration or by calling cmNetBiosSetWinsA() or cmNetBiosSetWins().
   By calling this function with a specific WINS index application returns the specific WINS address.

   @param winsID  WINS server index.
   @return the IPV4 address of the specific WINS server
                    or 0.   */
NQ_IPADDRESS4  cmNetBiosGetWins(NQ_COUNT winsID);

/**After startup NQ uses the WINS servers as defined in the NQ
   configuration or by calling cmNetBiosSetWinsA() or cmNetBiosSetWins().
   By calling this function application returns the number of current defined WINS addresses.

   @return Number of defined WINS addresses     */
NQ_COUNT cmNetBiosGetNumWinsServers(void);

/**After startup NQ uses the domain name as defined in the NQ
   configuration. By calling this function application replaces
   the initial domain name with a new ones.
   This name becomes available for subsequent operations.

   @param domainName  New domain name. It should contain more
                      than one symbol and it should not be longer
                      than UD_NQ_HOSTNAMESIZE
   @return NQ_SUCCESS on success or NQ_FAIL when the domain name
   is NULL or it is too long or too short.
   @note
   This is an ASCII version of cmDnsSetDomain() */
NQ_STATUS cmDnsSetDomainA(const NQ_CHAR * domainName);

/**After startup NQ uses the domain name as defined in the NQ
   configuration. By calling this function application replaces
   the initial domain name with a new ones.

   This name becomes available for subsequent operations.
   @param
   domainName New domain name. It should contain more than one
                 symbol and it should not be longer than
                 UD_NQ_HOSTNAMESIZE
   @return
   NQ_SUCCESS on success or NQ_FAIL when the domain name is NULL
   or it is too long or too short.                                */
NQ_STATUS cmDnsSetDomain(const NQ_WCHAR * domainName);

/** @} end of group cm_dns DNS and WINS */

/**@defgroup cm_aux Auxiliary
 * @ingroup cmapi
 * @{
 */

/** Reload cmnbname.c various name resources in the run-time
    by calling UD methods, can be used to modify hostname or
    domain name in the run-time.
    @return NQ_SUCCESS or NQ_FAIL.               */
NQ_STATUS cmNetBiosNameReload(void);

/** @} end of group cm_aux Auxiliary */

/** @addtogroup cm_product
 * @{
 */

/** This function provides information of the current compiled product.
    @return Pointer to a static CMProductInfo struct filled with the current product information.  */
CMProductInfo * cmGetCurrentProductInformation(void);

/** @} end of group cm_product Product Information */

#endif  /* _CMAPI_H_ */
