/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Processing incoming messages and timeouts for
 *                 Naming Service
 *--------------------------------------------------------------------
 * MODULE        : ND - NetBIOS Daemon
 * DEPENDENCIES  :
 ********************************************************************/

#include "ndnampro.h"
#include "ndinname.h"
#include "ndexname.h"
#include "nssessio.h"

#ifdef UD_ND_INCLUDENBDAEMON

/*
    Static functions & data
    -----------------------
 */

typedef struct
{
    NQ_CHAR scopeId[255];       /* buffer for parsed scope ID */
}
StaticData;

#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

/* determine packet type for incoming message */

static NQ_UINT16                /* packet type code */
getPacketType(
    const CMNetBiosHeader* pHdr /* incoming message */
    );

/* Operation codes

   codes are composed from:
    1) opcode
    2) response flag
    3) RCODE (error) field

   any RCODE other then zero is considered as error

   we compose one value from 1), 2) and 1 in BIT 0 when RCODE is not zero
 */

#define ANY_ERROR           1

#define QUERY_REQUEST           CM_NB_OPCODE_QUERY
#define POSITIVE_QUERY          CM_NB_OPCODE_QUERY | CM_NB_RESPONSE
#define NEGATIVE_QUERY          CM_NB_OPCODE_QUERY | CM_NB_RESPONSE | ANY_ERROR
#define REGISTRATION_REQUEST    CM_NB_OPCODE_REGISTRATION
#define POSITIVE_REGISTRATION   CM_NB_OPCODE_REGISTRATION | CM_NB_RESPONSE
#define NEGATIVE_REGISTRATION   CM_NB_OPCODE_REGISTRATION | CM_NB_RESPONSE | ANY_ERROR
#define RELEASE_REQUEST         CM_NB_OPCODE_RELEASE
#define POSITIVE_RELEASE        CM_NB_OPCODE_RELEASE | CM_NB_RESPONSE
#define NEGATIVE_RELEASE        CM_NB_OPCODE_RELEASE | CM_NB_RESPONSE | ANY_ERROR
#define REFRESH_REQUEST         CM_NB_OPCODE_REFRESH
#define REFRESHHALT_REQUEST     CM_NB_OPCODE_REFRESHALT
#define MULTIHOME_REQUEST       CM_NB_OPCODE_MHREGISTRATION
#define WACK                    CM_NB_OPCODE_WACK | CM_NB_RESPONSE

/*
 *====================================================================
 * PURPOSE: initalize internal data
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *====================================================================
 */

NQ_STATUS
ndNameInit(
    void
    )
{
    NQ_STATUS result = NQ_FAIL;
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocateStartup(sizeof(*staticData));
    if (NULL == staticData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate Naming Service data");
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    if (NQ_FAIL == ndInternalNameInit() || NQ_FAIL == ndExternalNameInit())
    {
        goto Exit;
    }
    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: release internal data
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

void
ndNameStop(
    void
    )
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    ndInternalNameStop();
    ndExternalNameStop();

    /* release memory */
#ifdef SY_FORCEALLOCATION
    if (NULL != staticData)
        cmMemoryFreeShutdown(staticData);
    staticData = NULL;
#endif /* SY_FORCEALLOCATION */

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 *====================================================================
 * PURPOSE: Processing incoming message for Name Service
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT message origin - adapter structure
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL.
 *
 * NOTES:   1. parse message to determine the packet type
 *          2. find existing name request or create a new one
 *          3. check if this packet type is appropriate for the request state
 *          4. call packet processing
 *          5. change state
 *====================================================================
 */

NQ_STATUS
ndNameProcessExternalMessage(
    NDAdapterInfo* adapter
    )
{
    NQ_UINT16 code;                 /* packet code including the response flag */
    CMNetBiosHeader* pHdr;          /* pointer to the header */
    CMNetBiosName name;             /* called name after parsing */
    NQ_BYTE* addData;               /* pointer to the date after the parsed name */
    NQ_STATUS status = NQ_FAIL;  /* return status */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "adapter:%p", adapter);

    pHdr = (CMNetBiosHeader*)adapter->inMsg;

    /* verify opcode */
    code = getPacketType(pHdr);
    switch (code)
    {
        case REGISTRATION_REQUEST:
        case QUERY_REQUEST:
        case POSITIVE_REGISTRATION:
        case NEGATIVE_REGISTRATION:
        case POSITIVE_RELEASE:
        case NEGATIVE_RELEASE:
        case POSITIVE_QUERY:
        case NEGATIVE_QUERY:
        case WACK:
        {
            break;
        }
        default:
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Illegal code: %04x", code);
            goto Exit;
        }
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "external message, type: %d, in port: %d", code, adapter->inPort);

    adapter->inTranId = cmGetSUint16(pHdr->tranID);
    addData = cmNetBiosParseName(
                    adapter->inMsg,
                    adapter->inLen,
                    pHdr + 1,
                    name,
                    staticData->scopeId,
                    sizeof(staticData->scopeId)
                    );
    if (NULL == addData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal message");
        goto Exit;
    }

    /* call packet processing */
    switch (code)
    {
        case REGISTRATION_REQUEST:
        {
            status = ndInternalNameCheckNameConflict(adapter, name);
            break;
        }
        case QUERY_REQUEST:
        {
            status = ndInternalNameWhateverQuery(adapter, name, addData);
            break;
        }
        case POSITIVE_REGISTRATION:
        {
            status = ndInternalNamePositiveRegistration(adapter, name, addData);
            break;
        }
        case NEGATIVE_REGISTRATION:
        {
            status = ndInternalNameNegativeRegistration(adapter, name);
            break;
        }
        case POSITIVE_RELEASE:
        {
            status = ndInternalNamePositiveRelease(adapter, name);
            break;
        }
        case NEGATIVE_RELEASE:
        {
            status = ndInternalNameNegativeRelease(adapter, name);
            break;
        }
        case POSITIVE_QUERY:
        {
            status = ndInternalNamePositiveQuery(adapter, name, addData);
            status = ndExternalNamePositiveQuery(adapter, name, addData);
            break;
        }
        case NEGATIVE_QUERY:
        {
            status = ndInternalNameNegativeQuery(adapter, name);
            status = ndExternalNameNegativeQuery(adapter, name);
            break;
        }
        case WACK:
        {
            status = ndInternalNameWack(adapter, name, addData);
            status = ndExternalNameWack(adapter, name, addData);
            break;
        }
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", status);
    return status;
}

/*
 *====================================================================
 * PURPOSE: Processing internal request for Name Service
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT message origin - adapter structure
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:   the adapter is a "dummy" adapter
 *          1. parse message to determine the packet type
 *          2. find existing name request or create a new one
 *          3. check if this packet type is appropriate for the request state
 *          4. call packet processing
 *          5. change state
 *====================================================================
 */

NQ_STATUS
ndNameProcessInternalMessage (
    NDAdapterInfo* adapter
    )
{
    NQ_UINT16 code;                 /* packet code including the response flag */
    CMNetBiosHeader* pHdr;          /* pointer to the header */
    CMNetBiosNameInfo nameInfo = {"0",FALSE};     /* called name after parsing */
    NQ_BYTE* addData;               /* pointer to the date after the parsed name */
    CMNetBiosAddrEntry* addrEntry;  /* pointer to the date after the parsed name */
    NQ_UINT16 flags;                /* name flags */
    NQ_STATUS result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "adapter:%p, in port: %d", adapter, adapter->inPort);

    pHdr = (CMNetBiosHeader*)adapter->inMsg;
    adapter->inTranId = cmGetSUint16(pHdr->tranID);
    code = getPacketType((CMNetBiosHeader*)adapter->inMsg);
    if (CM_NB_INTERNALREFRESHLIST == code)
    {
        result = ndInternalNameReleaseAllNames(FALSE);
        if (NQ_FAIL == result)
        {
            LOGMSG(CM_TRC_LEVEL_MESS_SOME, "Failed to release all names");
        }

        sySleep(1);
#ifdef UD_NB_INCLUDENAMESERVICE
        ndSetWins((NQ_WCHAR *)(adapter->inMsg + sizeof(CMNetBiosHeader)));
#endif /* UD_NB_INCLUDENAMESERVICE */
        ndNotifyConfigurationChange(adapter);
        result = NQ_SUCCESS;
        goto Exit;
    }

    addData = cmNetBiosParseName(
                    adapter->inMsg,
                    adapter->inLen,
                    pHdr + 1,
                    nameInfo.name,
                    staticData->scopeId,
                    sizeof(staticData->scopeId)
                    );

    LOGMSG(CM_TRC_LEVEL_MESS_SOME, "internal message, type: %d, name: %s, in port: %d", code, nameInfo.name, adapter->inPort);

    if (NULL == addData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal message");
        goto Exit;
    }

    addData += sizeof(CMNetBiosQuestion);

    /* dispatch the operation */

    switch (code)
    {
        case QUERY_REQUEST:
        {
            /* first check the internal name table (do not send any negative response if the name not found */
            if (NQ_SUCCESS != ndInternalProcessNameQuery(adapter, nameInfo.name, FALSE))
            {
                result = ndExternalNameQuery(adapter, nameInfo.name);
                if (NQ_FAIL == result)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Failed to register %s over %p", nameInfo.name, adapter);
                    goto Exit;
                }
            }

            break;
        }
        case REGISTRATION_REQUEST:
        {

            /* continue parsing the message */

            addData = cmNetBiosSkipName(
                            adapter->inMsg,
                            addData
                            );

            addrEntry = (CMNetBiosAddrEntry*)(addData + sizeof(CMNetBiosResourceRecord));
            flags = syNtoh16(cmGetSUint16(addrEntry->flags));
            nameInfo.isGroup = (flags & CM_NB_NAMEFLAGS_G) != 0;

            result = ndInternalNameRegisterAllAdapters(adapter, &nameInfo);
            if (NQ_FAIL == result)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Failed to register %s over all adapters", nameInfo.name);
                goto Exit;
            }
            break;
        }
        case RELEASE_REQUEST:
        {
            result = ndInternalNameReleaseAllAdapters(adapter, nameInfo.name, TRUE);
            if (NQ_FAIL == result)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Failed to release %s from all adapters", nameInfo.name);
                goto Exit;
            }
            break;
        }
        default:
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Illegal code: %04x", code);
            goto Exit;
        }
    }

    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: Timeout processing
 *--------------------------------------------------------------------
 * PARAMS:  IN elapsed time in seconds
 *
 * RETURNS: next timeout interval
 *
 * NOTES:   Scan name requests to determine an expired entry
 *          this may happen for internal name registration or
 *          internal name query only. Call processing and compose an error
 *          response.
 *====================================================================
 */

NQ_COUNT
ndNameProcessTimeout(
    NQ_INT delta
    )
{
    NQ_COUNT internal = ndInternalNameTimeout(delta);
    NQ_COUNT external = ndExternalNameTimeout(delta);

    return internal < external? internal : external;
}

/*
 *====================================================================
 * PURPOSE: Determine incoming packet type
 *--------------------------------------------------------------------
 * PARAMS:  IN packet pointer
 *
 * RETURNS: packet type code
 *
 * NOTES:
 *====================================================================
 */

static NQ_UINT16
getPacketType(
    const CMNetBiosHeader* pHdr
    )
{
    NQ_UINT16 codes;       /* the codes word from the packet */
    codes = syNtoh16(cmGetSUint16(pHdr->packCodes));
    return (NQ_UINT16)((codes & (CM_NB_OPCODE | CM_NB_RESPONSE)) | ((codes & CM_NB_RCODE)? ANY_ERROR:0));
}

#endif /* UD_ND_INCLUDENBDAEMON */

